/* ----------------------------------------------------------------------
   LAMMPS - Large-scale Atomic/Molecular Massively Parallel Simulator
   http://lammps.sandia.gov, Sandia National Laboratories
   Steve Plimpton, sjplimp@sandia.gov

   Copyright (2003) Sandia Corporation.  Under the terms of Contract
   DE-AC04-94AL85000 with Sandia Corporation, the U.S. Government retains
   certain rights in this software.  This software is distributed under
   the GNU General Public License.

   See the README file in the top-level LAMMPS directory.
------------------------------------------------------------------------- */

#ifndef LMP_REGION_H
#define LMP_REGION_H

#include "pointers.h"

namespace LAMMPS_NS {

class Region : protected Pointers {
 public:
  char *id,*style;
  int interior;                     // 1 for interior, 0 for exterior
  int scaleflag;                    // 1 for lattice, 0 for box
  double xscale,yscale,zscale;      // scale factors for box/lattice units
  double extent_xlo,extent_xhi;     // bounding box on region
  double extent_ylo,extent_yhi;
  double extent_zlo,extent_zhi;
  int bboxflag;                     // 1 if bounding box is computable

  // contact = particle near region surface

  struct Contact {
    double r;                 // distance between particle & surf, r > 0.0
    double delx,dely,delz;    // vector from surface pt to particle
  };
  Contact *contact;           // list of contacts
  int cmax;                   // max # of contacts possible with region

  Region(class LAMMPS *, int, char **);
  virtual ~Region();
  void init();
  virtual int dynamic_check();
  int match(double, double, double);
  int surface(double, double, double, double);

  // reset random gen - is called ot of restart by fix that uses region
  void reset_random(int);

  // generates a random point within the region
  virtual void generate_random(double *);

  // generate a point inside region OR within a minimum distance from surface
  virtual void generate_random_expandby_cut(double *,double);

  // generate a point inside region AND further away from surface than cut
  virtual void generate_random_shrinkby_cut(double *,double);

  // inside region AND within a minimum distance from surface
  int match_cut(double *,double);

  // inside region OR within a minimum distance from surface
  int match_expandby_cut(double *,double);

  // inside region OR within a minimum distance from surface
  int match_shrinkby_cut(double *,double);

  // volume calculation based on MC
  virtual double volume_mc(int);

  // flag if region bbox extends outside simulation domain
  virtual int bbox_extends_outside_box();

  virtual int inside(double, double, double) = 0;
  virtual int surface_interior(double *, double) = 0;
  virtual int surface_exterior(double *, double) = 0;

 protected:
  void options(int, char **);
  void add_contact(int, double *, double, double, double);

  int seed;
  class RanPark *random;

 private:
  int time_origin;
  double dt,period,omega_rotate;
  double vx,vy,vz;
  double ax,ay,az;
  double point[3],axis[3],runit[3];

  int dynamic;                      // 1 if region changes over time

  void rotate(double &, double &, double &, double);
};

}

#endif
