/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	Parts of this file are loosely based on an example gcrypt program
	on http://punkroy.drque.net/

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>
#include <gtkmm/stock.h>

#include "helpers.hpp"


using namespace LIFEO;


Error::Error( const Glib::ustring &error_message )
{
	print_error( error_message );
}

// DATE
Date::Date( const Glib::ustring &str_date )
	:	m_date( 0 )
{
	Glib::Date glib_date;

	glib_date.set_parse( str_date );

	if( ! glib_date.valid() )
		throw LIFEO::Error( "Date format not accepted" );

	guint16 year = glib_date.get_year();

	if( year < YEAR_MIN || year > YEAR_MAX )
		throw LIFEO::Error( "Date out of acceptable range" );

	m_date = ( year << 19 ) +
			 ( glib_date.get_month() << 15 ) +
			 ( glib_date.get_day() << 10 );
}

Glib::Date::Month
Date::get_month_glib( void ) const
{
	switch( ( m_date & 0x78000 ) >> 15 )
	{
		case 1:
			return Glib::Date::JANUARY;
		case 2:
			return Glib::Date::FEBRUARY;
		case 3:
			return Glib::Date::MARCH;
		case 4:
			return Glib::Date::APRIL;
		case 5:
			return Glib::Date::MAY;
		case 6:
			return Glib::Date::JUNE;
		case 7:
			return Glib::Date::JULY;
		case 8:
			return Glib::Date::AUGUST;
		case 9:
			return Glib::Date::SEPTEMBER;
		case 10:
			return Glib::Date::OCTOBER;
		case 11:
			return Glib::Date::NOVEMBER;
		case 12:
			return Glib::Date::DECEMBER;
		default:
			return Glib::Date::BAD_MONTH;
	}
}

// TODO: make date string format user configurable
Glib::ustring
Date::format_string( void ) const
{
	if( m_date & ORDINAL_FLAG )
	{
		return Glib::ustring::compose( get_order() ? "%1.%2" : "%1",
				get_ordinal_order() + 1, get_order() );
	}
	else
	{
		return Glib::ustring::compose( "%1.%2.%3",
				Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), get_year() ),
				Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), get_month() ),
				Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), get_day() ) );
	}
}

Glib::ustring
Date::format_string( const time_t *time )
{
	struct tm *timeinfo = localtime( time );
	return Glib::ustring::compose( "%1.%2.%3, %4:%5",
			1900 + timeinfo->tm_year,
			Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), timeinfo->tm_mon + 1 ),
			Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), timeinfo->tm_mday ),
			Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), timeinfo->tm_hour ),
			Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), timeinfo->tm_min ) );
}

// date only
Glib::ustring
Date::format_string_do( const time_t *time )
{
	struct tm *timeinfo = localtime( time );
	return Glib::ustring::compose( "%1.%2.%3",
			1900 + timeinfo->tm_year,
			Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), timeinfo->tm_mon + 1 ),
			Glib::ustring::format( std::setfill( L'0' ), std::setw( 2 ), timeinfo->tm_mday ) );
}

Glib::ustring
Date::get_year_str( void ) const
{
	return Glib::ustring::compose( "%1", get_year() );
}

Glib::ustring
Date::get_month_str( void ) const
{
	return Glib::Date( 20, get_month_glib(), 2000 ).format_string( "%b" );
}

Glib::ustring
Date::get_weekday_str( void ) const
{
	// from wikipedia: http://en.wikipedia.org/wiki/Calculating_the_day_of_the_week
	const unsigned int year = get_year();
	const unsigned int century = ( year - ( year % 100 ) ) / 100;
	int c = 2 * ( 3 - ( century % 4 ) );
	int y = year % 100;
	y = y + floor( y / 4 );

	const static int t_m[] = { 0, 3, 3, 6, 1, 4, 6, 2, 5, 0, 3, 5 };
    struct tm ti;

	int m = get_month() - 1;
	int d = ( c + y + t_m[ m ] + get_day() );

	if( !( get_year() % 4 ) && m < 2 )	// leap year!
		d += 6;

    ti.tm_wday = ( d % 7 );
    char buf[ 32 ];
    strftime( buf, 32, "%A", &ti );
	return Glib::ustring( buf );
}

time_t
Date::get_ctime( void ) const
{
	return( get_ctime( m_date ) );
}

time_t
Date::get_ctime( const Date::date_t d )
{
    time_t t;
    time( &t );
    struct tm *timeinfo = localtime( &t );
    timeinfo->tm_year = get_year( d ) - 1900;
    timeinfo->tm_mon = get_month( d ) - 1;
    timeinfo->tm_mday = get_day( d );
    timeinfo->tm_hour = 0;
    timeinfo->tm_min = 0;
    timeinfo->tm_sec = 0;

    return( mktime( timeinfo ) );
}

void
Date::forward_months( int months )
{
	months += ( ( m_date & 0x78000 ) >> 15 ); // get month
	m_date &= 0x7FF80000;	// isolate year
	const int mod_months = months % 12;
	if( mod_months == 0 )
	{
		m_date += make_year( ( months / 12 ) - 1);
		m_date |= 0x60000;	// make month 12
	}
	else
	{
		m_date += make_year( months / 12 );
		m_date |= make_month( mod_months );
	}
}

void
Date::forward_month( void )
{
	int months = get_month() + 1;
	m_date &= 0x7FF80000;	// isolate year
	const int mod_months = months % 12;
	if( mod_months == 0 )
	{
		m_date += make_year( ( months / 12 ) - 1);
		m_date |= 0x60000;	// make month 12
	}
	else
	{
		m_date += make_year( months / 12 );
		m_date |= make_month( mod_months );
	}
}

void
Date::forward_day( void )
{
	int day = get_day();
	if( day >= Glib::Date::get_days_in_month( get_month_glib(), get_year() ) )
	{
		set_day( 1 );
		forward_month();
	}
	else
		set_day( day + 1 );
}

// CONSOLE MESSAGES
void
LIFEO::print_error( const Glib::ustring &description )
{
	std::cout << "ERROR: " << description << std::endl;
}

void
LIFEO::print_info( const Glib::ustring &description )
{
	std::cout << "INFO: " << description << std::endl;
}

// STRING PROCESSING
long
LIFEO::convert_string( const std::string &str )
{
	//TODO: add negative number support
	long result( 0 );	// result
	for( unsigned int i = 0;
		 i < str.size() && i < 10 && int ( str[ i ] ) >= '0' && int ( str[ i ] ) <= '9';
		 i++ )
	{
		result = ( result * 10 ) + int ( str[ i ] ) - '0';
	}
	return( result );
}

// COLOR PROCESSING
//TODO: this needs lots of love...
Gdk::Color
LIFEO::contrast( const Gdk::Color &bg, const Gdk::Color &fg )
{
	Gdk::Color contrast( fg );
	int dom_index = 0;	// 1,2,3 = R,G,B
	int existingcontrast =	abs( fg.get_red() - bg.get_red() ) +
							abs( fg.get_green() - bg.get_green() ) +
							abs( fg.get_blue() - bg.get_blue() );
	if( existingcontrast > 32767 )
		return contrast;

	if( fg.get_red() > fg.get_green() )
	{
		if( fg.get_red() > fg.get_blue() )
			dom_index = 1;
		else
			dom_index = 3;
	}
	else
	{
		if( fg.get_green() > fg.get_blue() )
			dom_index = 2;
		else
			dom_index = 3;
	}

	switch ( dom_index )
	{
		case 0:
		break;
		case 1:
			if( bg.get_red() > 32767 )
			{
				contrast.set_red( bg.get_red() - 32767 );// + existingcontrast );
//				contrast.set_green( 0 );
//				contrast.set_blue( 0 );
			}
			else
			{
				contrast.set_red( bg.get_red() + 32767 );//- existingcontrast );
//				contrast.set_green( bg.get_green() > fg.get_green() ?
//											fg.get_green()*0.5 :  );
//				contrast.set_blue( 0 );
			}
		break;
		case 2:
			contrast.set_green( bg.get_green() > 32767 ?
					fg.get_green() - 32767 :// + existingcontrast :
					fg.get_green() + 32767 );// - existingcontrast );
		break;
		case 3:
			contrast.set_blue( bg.get_blue() > 32767 ?
					bg.get_blue() - 32767 :// + existingcontrast :
					bg.get_blue() + 32767 );// - existingcontrast );
		break;
	}
	return contrast;
}

Gdk::Color
LIFEO::midtone( const Gdk::Color &c1, const Gdk::Color &c2 )
{
	Gdk::Color midtone;
	midtone.set_red( ( c1.get_red() + c2.get_red() )>>1 );
	midtone.set_green( ( c1.get_green() + c2.get_green() )>>1 );
	midtone.set_blue( ( c1.get_blue() + c2.get_blue() )>>1 );
	return midtone;
}

Gdk::Color
LIFEO::midtone( const Gdk::Color &c1, const Gdk::Color &c2, float ratio )
{
    Gdk::Color midtone;
    midtone.set_red( ( c1.get_red() * ratio ) + ( c2.get_red() * ( 1.0 - ratio ) ) );
    midtone.set_green( ( c1.get_green() * ratio ) + ( c2.get_green() * ( 1.0 - ratio ) ) );
    midtone.set_blue( ( c1.get_blue() * ratio ) + ( c2.get_blue() * ( 1.0 - ratio ) ) );
    return midtone;
}
// FILE FUNCTIONS
std::ios::pos_type
LIFEO::get_file_size( std::ifstream &file )
{
   std::ios::pos_type size;
   const std::ios::pos_type currentPosition = file.tellg();

   file.seekg( 0, std::ios_base::end );
   size = file.tellg();
   file.seekg( currentPosition );

   return size;
}

// URL HANDLERS
void
LIFEO::open_url( Gtk::AboutDialog&, const Glib::ustring &link )
{
	std::string command( "x-www-browser " );
	command += link;
	std::cout << "opening url: " << system( command.c_str() ) << std::endl;
	// TODO: find a better way
}

void
LIFEO::mail_to( Gtk::AboutDialog&, const Glib::ustring &link )
{
	// TODO: find a good way
}

// ENCRYPTION
bool
Cipher::init( void )
{
	// http://www.gnupg.org/documentation/manuals/gcrypt/Initializing-the-library.html

	// initialize subsystems:
	if( ! gcry_check_version( NULL ) )	// TODO: check version
	{
		print_error( "Libgcrypt version mismatch" );
		return false;
	}

	// disable secure memory
	gcry_control( GCRYCTL_DISABLE_SECMEM, 0 );

	// MAYBE LATER:
	/*
	// suppress warnings
	gcry_control( GCRYCTL_SUSPEND_SECMEM_WARN );

	// allocate a pool of 16k secure memory. this makes the secure memory...
	// ...available and also drops privileges where needed
	gcry_control( GCRYCTL_INIT_SECMEM, 16384, 0 );
     
	// resume warnings
	gcry_control( GCRYCTL_RESUME_SECMEM_WARN );
	*/

	// tell Libgcrypt that initialization has completed
	gcry_control( GCRYCTL_INITIALIZATION_FINISHED, 0 );

	return true;
}

void
Cipher::create_iv( unsigned char **iv )
{
	// (Allocate memory for and fill with strong random data)
	*iv = (unsigned char *) gcry_random_bytes( Cipher::cIV_SIZE, GCRY_STRONG_RANDOM );

	if( ! *iv )
		throw Error( "Unable to create IV" );
}

void
Cipher::expand_key(	const char *passphrase,
					const unsigned char *salt,
					unsigned char **key )
{
	gcry_md_hd_t hash;
	gcry_error_t error = 0;
	int hashdigestsize;
	unsigned char *hashresult;

	// OPEN MESSAGE DIGEST ALGORITHM
	error = gcry_md_open( &hash, cHASH_ALGORITHM, 0 );
	if( error )
		throw Error( "Unable to open message digest algorithm: %s" ); //, gpg_strerror( Error ) );

	// RETRIVE DIGEST SIZE
	hashdigestsize = gcry_md_get_algo_dlen( cHASH_ALGORITHM );

	// ADD SALT TO HASH
	gcry_md_write( hash, salt, cSALT_SIZE );

	// ADD PASSPHRASE TO HASH
	gcry_md_write( hash , passphrase , strlen( passphrase ) );

	// FETCH DIGEST (THE EXPANDED KEY)
	hashresult = gcry_md_read( hash , cHASH_ALGORITHM );

	if( ! hashresult )
	{
		gcry_md_close( hash );
		throw Error( "Unable to finalize key" );
	}

	// ALLOCATE MEMORY FOR KEY
	// can't use the 'HashResult' because those resources are freed after the
	// hash is closed
	*key = new unsigned char[ cKEY_SIZE ];
	if( ! key )
	{
		gcry_md_close( hash );
		throw Error( "Unable to allocate memory for key" );
	}

	// DIGEST SIZE SMALLER THEN KEY SIZE?
	if( hashdigestsize < cKEY_SIZE )
	{
		// PAD KEY WITH '0' AT THE END
		memset( *key , 0 , cKEY_SIZE );

		// COPY EVERYTHING WE HAVE
		memcpy( *key , hashresult , hashdigestsize );
	}
	else
		// COPY ALL THE BYTES WE'RE USING
		memcpy( *key , hashresult , hashdigestsize );

	// FINISHED WITH HASH
	gcry_md_close( hash );
}

// create new expanded key
void
Cipher::create_new_key(	char const *passphrase,
						unsigned char **salt,
						unsigned char **key )
{
	// ALLOCATE MEMORY FOR AND FILL WITH STRONG RANDOM DATA
	*salt = (unsigned char *) gcry_random_bytes( cSALT_SIZE, GCRY_STRONG_RANDOM );

	if( ! *salt )
		throw Error( "Unable to create salt value" );

	expand_key( passphrase, *salt, key );
}

void
Cipher::encrypt_buffer (	unsigned char *buffer,
							size_t& size,
							const unsigned char *key,
							const unsigned char *iv )
{
	gcry_cipher_hd_t	cipher;
	gcry_error_t		error = 0;

	error = gcry_cipher_open( &cipher, cCIPHER_ALGORITHM, cCIPHER_MODE, 0 );

	if( error )
		throw Error( "unable to initialize cipher: " ); // + gpg_strerror( Error ) );

	// GET KEY LENGTH
	int cipherKeyLength = gcry_cipher_get_algo_keylen( cCIPHER_ALGORITHM );
	if( ! cipherKeyLength )
		throw Error( "gcry_cipher_get_algo_keylen failed" );

	// SET KEY
	error = gcry_cipher_setkey( cipher, key, cipherKeyLength );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Cipher key setup failed: %s" ); //, gpg_strerror( Error ) );
	}

	// SET INITILIZING VECTOR (IV)
	error = gcry_cipher_setiv( cipher, iv, cIV_SIZE );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Unable to setup cipher IV: %s" );// , gpg_strerror( Error ) );
	}

	// ENCRYPT BUFFER TO SELF
	error = gcry_cipher_encrypt( cipher, buffer, size, NULL, 0 );

	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Encrption failed: %s" ); // , gpg_strerror( Error ) );
	}

	gcry_cipher_close( cipher );
}

void
Cipher::decrypt_buffer (	unsigned char * buffer,
							size_t& size,
							const unsigned char * key,
							const unsigned char * iv )
{
	gcry_cipher_hd_t	cipher;
	gcry_error_t		error = 0;

	error = gcry_cipher_open( &cipher, cCIPHER_ALGORITHM, cCIPHER_MODE, 0 );

	if( error )
		throw Error( "Unable to initilize cipher: " ); // + gpg_strerror( Error ) );

	// GET KEY LENGTH
	int cipherKeyLength = gcry_cipher_get_algo_keylen( cCIPHER_ALGORITHM );
	if( ! cipherKeyLength )
		throw Error( "gcry_cipher_get_algo_keylen failed" );

	// SET KEY
	error = gcry_cipher_setkey( cipher, key, cipherKeyLength );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Cipher key setup failed: %s" ); //, gpg_strerror( Error ) );
	}

	// SET IV
	error = gcry_cipher_setiv( cipher, iv, cIV_SIZE );
	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Unable to setup cipher IV: %s" );// , gpg_strerror( Error ) );
	}

	// DECRYPT BUFFER TO SELF
	error = gcry_cipher_decrypt( cipher, buffer, size, NULL, 0 );

	if( error )
	{
		gcry_cipher_close( cipher );
		throw Error( "Encrption failed: %s" ); // , gpg_strerror( Error ) );
	}

	gcry_cipher_close( cipher );
}

// MARKED UP MENU ITEM
Gtk::MenuItem*
LIFEO::create_menuitem_markup(	const Glib::ustring &str_markup,
								const Glib::SignalProxy0< void >::SlotType &handler )
{
	// thanks to GNote for showing the way
	Gtk::MenuItem *menuitem = Gtk::manage( new Gtk::MenuItem( str_markup ) );
	Gtk::Label *label = dynamic_cast< Gtk::Label* >( menuitem->get_child() );
	if( label )
		label->set_use_markup( true );
	menuitem->signal_activate().connect( handler );
	return menuitem;
}

// ENTRYIDLETEXT ===================================================================================
EntryIdletext::EntryIdletext( BaseObjectType* cobject, const Glib::RefPtr<Gtk::Builder>& )
	:	Gtk::Entry( cobject ), m_flag_empty( true )
{
	signal_icon_release().connect(
			sigc::mem_fun( *this, &EntryIdletext::handle_clear_icon ) );
}

EntryIdletext::EntryIdletext( const Glib::ustring& idletext )
: Gtk::Entry(), m_idletext( idletext ), m_flag_empty( true )
{
	signal_icon_release().connect(
			sigc::mem_fun( *this, &EntryIdletext::handle_clear_icon ) );
}

void
EntryIdletext::handle_clear_icon( Gtk::EntryIconPosition icon_pos, const GdkEventButton *event )
{
	if( icon_pos == Gtk::ENTRY_ICON_SECONDARY && event->button == 1 )
		set_text( "" );
}

void
EntryIdletext::on_map( void )
{
	Gdk::Color color = get_style()->get_text( Gtk::STATE_INSENSITIVE );

	m_idletext.insert( 0, "\">" );
	m_idletext.insert( 0, color.to_string() );
	m_idletext.insert( 0, "<span foreground=\"" );
	m_idletext.append( "</span>" );

	Gtk::Entry::on_map();
}

void
EntryIdletext::on_changed( void )
{
    m_flag_empty = get_text().empty();

    if( m_flag_empty )
    {
        // Fall-back to C API as gtkmm has problems with this:
        gtk_entry_set_icon_from_stock( gobj(), GTK_ENTRY_ICON_SECONDARY, 0 );
        set_icon_activatable( false, Gtk::ENTRY_ICON_SECONDARY );
    }
    else
    {
        set_icon_from_stock( Gtk::Stock::CLEAR, Gtk::ENTRY_ICON_SECONDARY );
        set_icon_activatable( true, Gtk::ENTRY_ICON_SECONDARY );
    }

    Gtk::Entry::on_changed();
}

bool
EntryIdletext::on_expose_event( GdkEventExpose *event )
{
	if( m_flag_empty && has_focus() == false)
		get_layout()->set_markup( m_idletext );

	return( Gtk::Entry::on_expose_event( event ) );
}

bool
EntryIdletext::on_focus_in_event( GdkEventFocus *event )
{
	if( m_flag_empty )
		get_layout()->set_markup( "" );

	return( Gtk::Entry::on_focus_in_event( event ) );
}

bool
EntryIdletext::on_key_release_event( GdkEventKey *event )
{
	if( event->keyval == GDK_Escape )
		set_text( "" );
		
	return Gtk::Entry::on_key_release_event( event );
}

// MENUBUTTON ======================================================================================
Glib::ustring       Menubutton::s_builder_name;

Menubutton::Menubutton(	const Gtk::StockID &stockid,
						const Glib::ustring &label,
						Gtk::ReliefStyle style,
						Gtk::IconSize iconsize,
						Menu2 *menu )
	:	Gtk::ToggleButton(), m_menu( menu )
{
	set_relief( style );

	if( menu == NULL )
		m_menu = new Menu2;
	Gtk::HBox *hbox = Gtk::manage( new Gtk::HBox );
	m_icon = Gtk::manage( new Gtk::Image( stockid, iconsize ) );
	m_label = Gtk::manage( new Gtk::Label( label ) );
	Gtk::Arrow *arrow = Gtk::manage(
			new Gtk::Arrow(Gtk::ARROW_DOWN, Gtk::SHADOW_IN) );

	hbox->pack_start( *m_icon, Gtk::PACK_SHRINK );
	hbox->pack_start( *m_label, Gtk::PACK_EXPAND_WIDGET );
	hbox->pack_start( *arrow, Gtk::PACK_SHRINK );
	add( *hbox );
	m_menu->attach_to_widget( *this );
	m_menu->signal_deactivate().connect( sigc::mem_fun( *this, &Menubutton::release ) );
	add_events( Gdk::BUTTON_PRESS_MASK );
}

//FIXME: code duplication
Menubutton::Menubutton(	const Glib::RefPtr< Gdk::Pixbuf > *pixbuf,
						const Glib::ustring &label,
						Gtk::ReliefStyle style,
						Menu2 *menu )
:   Gtk::ToggleButton(), m_menu( menu )
{
	set_relief( style );
	Gtk::Arrow *arrow;
	Gtk::HBox *hbox;
	try
	{
		if( menu == NULL )
			m_menu = new Menu2;
		hbox = Gtk::manage( new Gtk::HBox );
		if( pixbuf )
			m_icon = Gtk::manage( new Gtk::Image( *pixbuf ) );
		m_label = Gtk::manage( new Gtk::Label( label ) );
		arrow = Gtk::manage( new Gtk::Arrow( Gtk::ARROW_DOWN, Gtk::SHADOW_IN ) );
	}
	catch( ... )
	{
		throw 0;	// TODO...
	}

	if( pixbuf )
		hbox->pack_start( *m_icon, Gtk::PACK_SHRINK );
	hbox->pack_start( *m_label, Gtk::PACK_EXPAND_WIDGET );
	hbox->pack_start( *arrow, Gtk::PACK_SHRINK );
	add( *hbox );
	m_menu->attach_to_widget( *this );
	m_menu->signal_deactivate().connect( sigc::mem_fun( *this, &Menubutton::release ) );
	add_events( Gdk::BUTTON_PRESS_MASK );
}

Menubutton::Menubutton( BaseObjectType* cobject, const Glib::RefPtr<Gtk::Builder> &builder )
:   Gtk::ToggleButton( cobject )
{
    try
    {
        builder->get_widget_derived( s_builder_name + "_menu", m_menu );
        builder->get_widget( s_builder_name + "_icon", m_icon );
        builder->get_widget( s_builder_name + "_label", m_label );
        s_builder_name = "";
    }
    catch( ... )
    {
        throw LIFEO::Error( "Menubutton creation failed" );
    }

    m_menu->attach_to_widget( *this );
    m_menu->signal_deactivate().connect( sigc::mem_fun( this, &Menubutton::release ) );
    add_events( Gdk::BUTTON_PRESS_MASK );
}

Menubutton::~Menubutton( void )
{
	delete m_menu;
}

Menu2*
Menubutton::get_menu( void ) const
{
	return m_menu;
}

bool
Menubutton::clear_menu( void )
{
	if ( m_menu->items().size() > 0 )
	{
		delete m_menu;
		m_menu = new Menu2;
		m_menu->attach_to_widget( *this );
		m_menu->signal_deactivate().connect( sigc::mem_fun( *this, &Menubutton::release ) );

		return true;
	}
	else
		return false;
}

void
Menubutton::release( void )
{
	set_active( false );
}

void
Menubutton::set_label( const Glib::ustring& string )
{
	m_label->set_label( string );
}

void
Menubutton::get_menu_position( int& x, int& y, bool& push_in )
{
	get_window()->get_origin( x, y );
	x += get_allocation().get_x();
	y += get_allocation().get_y() + get_allocation().get_height();
	push_in = true;
}

bool
Menubutton::on_button_press_event( GdkEventButton *l_event )
{
	m_menu->popup( sigc::mem_fun(
			*this, &Menubutton::get_menu_position), l_event->button, l_event->time);

	set_active( true );

	return true;
}

// MENUTOOLBUTTON
/*
 * TO BE REWRITTEN IF NEEDED IN THE FUTURE...
Menutoolbutton::Menutoolbutton( const Gtk::StockID& stockid,
							    const std::string& string,
							    Menu2 *menu,
							    bool flag_important)
:	m_flag_important( flag_important ),
	m_button( stockid, string, Gtk::RELIEF_NONE, Gtk::ICON_SIZE_LARGE_TOOLBAR, menu )
{
	add( m_button );
	m_button.m_icon->show();
	m_button.m_label->show();
}

Menu2*
Menutoolbutton::get_menu (void)
{
	return m_button.m_menu;
}

void
Menutoolbutton::on_toolbar_reconfigured (void)
{
	if( m_button.m_label->get_parent() )
		m_button.m_label->get_parent()->remove( *m_button.m_label );

	if( m_button.m_icon->get_parent() )
		m_button.m_icon->get_parent()->remove( *m_button.m_icon );

	switch( get_toolbar_style() )
	{
		case Gtk::TOOLBAR_BOTH:
			m_button.m_vbox->pack_start( *m_button.m_icon );
			m_button.m_vbox->pack_start( *m_button.m_label );
		break;
		case Gtk::TOOLBAR_BOTH_HORIZ:
			m_button.m_vbox->pack_start( *m_button.m_icon );
			if( m_flag_important )
			{
				m_button.m_hbox->pack_start( *m_button.m_label );
				m_button.m_hbox->reorder_child( *m_button.m_label, 1 );
			}
		break;
		case Gtk::TOOLBAR_ICONS:
			m_button.m_vbox->pack_start( *m_button.m_icon );
		break;
		case Gtk::TOOLBAR_TEXT:
			m_button.m_vbox->pack_start( *m_button.m_label );
		break;
		default:
		break;
	}
}
*/

// MENUITEMRECENT ==================================================================================
MenuitemRecent::MenuitemRecent( const std::string &path )
:	Gtk::MenuItem(), m_icon_remove( Gtk::Stock::DELETE, Gtk::ICON_SIZE_MENU ),
	m_path( path ), m_flag_deletepressed( false )
{
	Gtk::Label		*label = Gtk::manage( new Gtk::Label(
											Glib::filename_display_basename( path ) ) );
	Gtk::HBox		*hbox = Gtk::manage( new Gtk::HBox );

	label->set_justify( Gtk::JUSTIFY_LEFT );
	label->set_alignment( Gtk::ALIGN_LEFT );
	label->set_ellipsize( Pango::ELLIPSIZE_START );
	label->set_tooltip_text( path );
	m_icon_remove.set_tooltip_text( _( "Remove from the list" ) );
	hbox->set_spacing( 5 );

	hbox->pack_start( *label );
	hbox->pack_start( m_icon_remove, Gtk::PACK_SHRINK );
	this->add( *hbox );

	hbox->show();
	label->show();
}

bool
MenuitemRecent::on_motion_notify_event( GdkEventMotion *event )
{
	if( !( event->state &
			( Gdk::BUTTON1_MASK | Gdk::BUTTON2_MASK | Gdk::BUTTON3_MASK ) ) )
		m_icon_remove.show();

	return Gtk::MenuItem::on_motion_notify_event( event );
}

bool
MenuitemRecent::on_leave_notify_event( GdkEventCrossing *event )
{
	m_icon_remove.hide();

	return Gtk::MenuItem::on_leave_notify_event( event );
}

bool
MenuitemRecent::on_button_press_event( GdkEventButton *event )
{
	if(	event->x >= m_icon_remove.get_allocation().get_x() &&
		event->x < m_icon_remove.get_allocation().get_x() +
					m_icon_remove.get_allocation().get_width() )
		m_flag_deletepressed = true;
	else
		m_flag_deletepressed = false;

	return Gtk::MenuItem::on_button_press_event( event );
}

bool
MenuitemRecent::on_button_release_event( GdkEventButton *event )
{
	if( m_flag_deletepressed )
	{
		if(	event->x >= m_icon_remove.get_allocation().get_x() &&
			event->x < m_icon_remove.get_allocation().get_x() +
						m_icon_remove.get_allocation().get_width() )
		{
			m_signal_removerecent.emit( m_path );
			return true;
		}
	}

	return Gtk::MenuItem::on_button_release_event( event );
}


// FILEBUTTONRECENT ================================================================================
Glib::ustring   FilebuttonRecent::fallback_label = _( "Select or Create a Diary" );

/*
FilebuttonRecent::FilebuttonRecent( const Glib::RefPtr< Gdk::Pixbuf > &pixbuf,
                                    ListPaths *list_recent,
                                    const Glib::ustring& label_fallback )
:   Menubutton( &pixbuf, label_fallback, Gtk::RELIEF_NORMAL ),
    m_icon_new( Gtk::Stock::NEW, Gtk::ICON_SIZE_MENU ),
    m_icon_browse( Gtk::Stock::OPEN, Gtk::ICON_SIZE_MENU ),
    m_list_recent( list_recent )
{
    // TODO: m_hbox->set_spacing( 4 );

    m_label->set_ellipsize( Pango::ELLIPSIZE_START );
    m_label->set_alignment( Gtk::ALIGN_LEFT );

    m_menu->items().push_back( Gtk::Menu_Helpers::SeparatorElem() );

    m_menu->items().push_back(
            Gtk::Menu_Helpers::ImageMenuElem( _("_Browse For a Diary..."), m_icon_browse,
                    sigc::mem_fun( *this,
                            &FilebuttonRecent::show_filechooser) ) );
    m_menu->items().push_back(
            Gtk::Menu_Helpers::ImageMenuElem( _("_Create A New Diary..."), m_icon_new,
                    sigc::mem_fun( *this,
                            &FilebuttonRecent::on_create_file) ) );

    m_menu->show_all_children();

    // ACCEPT DROPPED FILES
    //~ std::list< Gtk::TargetEntry > list_targets;
    //~ list_targets.push_back( Gtk::TargetEntry( "STRING" ) );
    //~ list_targets.push_back( Gtk::TargetEntry( "text/plain" ) );
    //~ this->drag_dest_set( list_targets );
    // TODO: i could not figure out how this is supposed to work and replicated
    // almost blindly bmpx' approach.
    drag_dest_set( Gtk::DEST_DEFAULT_ALL, ( Gdk::ACTION_COPY | Gdk::ACTION_MOVE ) );
    drag_dest_add_uri_targets();
}*/

FilebuttonRecent::FilebuttonRecent( BaseObjectType *cobj, const Glib::RefPtr<Gtk::Builder> &bui )
:   Menubutton( cobj, bui ),
    m_icon_new( Gtk::Stock::NEW, Gtk::ICON_SIZE_MENU ),
    m_icon_browse( Gtk::Stock::OPEN, Gtk::ICON_SIZE_MENU ),
    m_list_recent( NULL )
{
    m_label->set_ellipsize( Pango::ELLIPSIZE_START );
    m_label->set_alignment( Gtk::ALIGN_LEFT );
    m_label->set_text( fallback_label );

    m_menu->items().push_back( Gtk::Menu_Helpers::SeparatorElem() );

    m_menu->items().push_back(
            Gtk::Menu_Helpers::ImageMenuElem( _("_Browse For a Diary..."), m_icon_browse,
                    sigc::mem_fun( *this,
                            &FilebuttonRecent::show_filechooser) ) );
    m_menu->items().push_back(
            Gtk::Menu_Helpers::ImageMenuElem( _("_Create A New Diary..."), m_icon_new,
                    sigc::mem_fun( *this,
                            &FilebuttonRecent::on_create_file) ) );

    m_menu->show_all_children();

    // ACCEPT DROPPED FILES
    drag_dest_set( Gtk::DEST_DEFAULT_ALL, ( Gdk::ACTION_COPY | Gdk::ACTION_MOVE ) );
    drag_dest_add_uri_targets();
}

void
FilebuttonRecent::set( ListPaths *list_recent )
{
    m_list_recent = list_recent;
}

std::string
FilebuttonRecent::get_filename( void ) const
{
	return( * m_list_recent->begin() );
}

void
FilebuttonRecent::set_filename( const std::string &path )
{
	if( ! m_list_recent->empty() )
		if( path == * m_list_recent->begin() )
			return;

	m_label->set_label( Glib::filename_display_basename( path ) );
	set_tooltip_text( path );
	add_recent( path );
	m_signal_selectionchanged.emit();
}

void
FilebuttonRecent::update_filenames()
{
	if( ! m_list_recent->empty() )
	{
		for( ListPaths::reverse_iterator iter = m_list_recent->rbegin();
			 iter != m_list_recent->rend(); ++iter )
		{
			MenuitemRecent *menuitem = Gtk::manage( new MenuitemRecent( * iter ) );

			menuitem->signal_activate().connect(
					sigc::bind(	sigc::mem_fun( *this, &FilebuttonRecent::set_filename ),
								* iter ) );
			menuitem->signal_removerecent().connect(
					sigc::mem_fun( *this, &FilebuttonRecent::remove_recent ) );

			m_menu->items().push_front( Gtk::Menu_Helpers::Element( *menuitem ) );

			menuitem->show();
		}

		m_label->set_label( Glib::filename_display_basename( * m_list_recent->begin() ) );
		set_tooltip_text( * m_list_recent->begin() );
	}
}

bool
FilebuttonRecent::add_recent( const std::string &path )
{
	// do not add a file twice:
	Gtk::Menu_Helpers::MenuList::iterator iter_menu = m_menu->items().begin();
	ListPaths::iterator iter_recent = m_list_recent->begin();
	for( ; iter_recent != m_list_recent->end(); ++iter_recent )
	{
		if( ( *iter_recent ) == path )
		{
			m_menu->items().erase( iter_menu );
			m_list_recent->erase( iter_recent );
			break;
		}
		else
			++iter_menu;
	}

	MenuitemRecent *menuitem = Gtk::manage( new MenuitemRecent( path ) );

	menuitem->signal_activate().connect(
			sigc::bind(	sigc::mem_fun( *this, &FilebuttonRecent::set_filename ),
						path ) );
	menuitem->signal_removerecent().connect(
			sigc::mem_fun( *this, &FilebuttonRecent::remove_recent ) );

	m_menu->items().push_front( Gtk::Menu_Helpers::Element( *menuitem ) );

	menuitem->show();

	m_list_recent->push_front( path );

	if( m_list_recent->size() > MAX_RECENT_FILE_COUNT )
	{
		m_menu->items().erase( --iter_menu );
		m_list_recent->erase( --iter_recent );
	}

	return true;	// reserved
}

void
FilebuttonRecent::remove_recent( const std::string &path )
{
    Gtk::Menu_Helpers::MenuList::iterator	iter_menu = m_menu->items().begin();
    std::list< std::string >::iterator		iter_recent = m_list_recent->begin();
    bool flag_update = false;
    for( ; iter_recent != m_list_recent->end(); ++iter_recent )
    {
        if( ( *iter_recent ) == path )
        {
            if( iter_recent == m_list_recent->begin() )
                flag_update = true;
            m_menu->items().erase( iter_menu );
            m_list_recent->erase( iter_recent );
            break;
        }
        else
            ++iter_menu;
    }
    // update current path if first item is deleted
    if( flag_update )
    {
        if( m_list_recent->empty() )
        {
            m_label->set_label( fallback_label );
            set_tooltip_text( "" );
        }
        else
        {
            m_label->set_label( Glib::filename_display_basename( m_list_recent->front() ) );
            set_tooltip_text( m_list_recent->front() );
        }
        m_signal_selectionchanged.emit();
    }
}

void
FilebuttonRecent::show_filechooser( void )
{
	m_filechooserdialog = new Gtk::FileChooserDialog(
	// TRANSLATORS: this is the title of filechooser dialog for selecting a new diary
			_( "Select a Diary" ), Gtk::FILE_CHOOSER_ACTION_OPEN );

	if( m_list_recent->size() > 0 )
		m_filechooserdialog->set_filename( * m_list_recent->begin() );
	else
		m_filechooserdialog->set_current_folder( Glib::get_home_dir() );

	m_filechooserdialog->add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
	m_filechooserdialog->add_button( Gtk::Stock::OPEN, Gtk::RESPONSE_ACCEPT );
	FilefilterAny		filter_any;
	FilefilterDiary		filter_diary;
	m_filechooserdialog->add_filter( filter_any );
	m_filechooserdialog->add_filter( filter_diary );
	m_filechooserdialog->set_filter( filter_diary );
	if( m_filechooserdialog->run() == Gtk::RESPONSE_ACCEPT )
	{
		set_filename( m_filechooserdialog->get_filename() );
	}

	delete m_filechooserdialog;
	m_filechooserdialog= NULL;
}

void
FilebuttonRecent::on_create_file( void )
{
	m_signal_createfile.emit();
}

void
FilebuttonRecent::on_drag_data_received(	const Glib::RefPtr<Gdk::DragContext>& context,
											int, int,
											const Gtk::SelectionData& seldata,
											guint info,
											guint time )
{
	if( seldata.get_length() < 0 )
		return;

	Glib::ustring uri = seldata.get_data_as_string();
	std::string filename = uri.substr( 0, uri.find('\n') - 1 );
	filename = Glib::filename_from_uri( filename );

	if( Glib::file_test( filename, Glib::FILE_TEST_IS_DIR ) )
		return;

	set_filename( filename );
	PRINT_DEBUG( Glib::ustring::compose( "dropped: %1", filename ) );

	context->drag_finish( true, false, time );
}

void
FilebuttonRecent::on_size_allocate( Gtk::Allocation &allocation )
{
	Menubutton::on_size_allocate( allocation );
	m_menu->set_size_request( allocation.get_width(), -1 );
}

SignalVoid
FilebuttonRecent::signal_selection_changed( void )
{
	return m_signal_selectionchanged;
}

SignalVoid
FilebuttonRecent::signal_create_file( void )
{
	return m_signal_createfile;
}

