/***********************************************************************************

	Copyright (C) 2007-2012 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifndef LIFEOGRAPH_VIEWS_HEADER
#define LIFEOGRAPH_VIEWS_HEADER


#include "diarydata.hpp"
#include "entry.hpp"
#include "widget_chart.hpp"
#include "dialog_export.hpp"
#include "dialog_password.hpp"


namespace LIFEO
{

// TAG VIEW ========================================================================================
class TagView : public ElementShower< Tag >
{
	public:
								TagView( void );

        Glib::ustring           get_title_str( void ) const
        { return m_ptr2elem->m_name; }
        Glib::ustring           get_info_str( void ) const
        { return Glib::ustring::compose( _( "Tag with %1 entries" ), m_ptr2elem->get_size() ); }
#ifdef LIFEOGRAPH_DEBUG_BUILD
        Glib::ustring           get_tip_str( void ) const
        { return Glib::ustring::compose( "DEID: %1", m_ptr2elem->get_id() ); }
#endif
        Menu2*                  get_menu( void );
        Gtk::Widget*            get_extra_tools( void )
        { return m_togglebutton_filter; }

        void                    handle_login( void );
        void                    show( Tag &tag );

		void					update_combobox_categories( void );
        void                    update_button_filter_tag( void );

		enum TagComboType { TCT_NONE, TCT_CATEGORY, TCT_NEW, TCT_SEPARATOR };

	protected:
        bool                    is_title_editable( void ) const
        { return true; }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2elem->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

        void                    handle_filter_toggled( void );
        void                    handle_category_changed( void );
        bool                    is_row_separator( const Glib::RefPtr< Gtk::TreeModel >&,
                                                  const Gtk::TreeIter );

        void                    dismiss_tag( void );

        Gtk::ToggleButton       *m_togglebutton_filter;
        Gtk::MenuItem           *m_menuitem_dismiss;
		Gtk::ComboBox			*m_combobox_category;
		WidgetChart				*m_widgetchart;

		class ColrecCategory : public Gtk::TreeModel::ColumnRecord
		{
			public:
				ColrecCategory() { add( name ); add( ptr ); add( type ); }
				Gtk::TreeModelColumn< Glib::ustring >		name;
				Gtk::TreeModelColumn< CategoryTags* >		ptr;
				Gtk::TreeModelColumn< TagComboType >		type;
		};

		Glib::RefPtr< Gtk::ListStore >
								m_liststore_categories;
		ColrecCategory			colrec_category;
};

// TAG CATEGORY VIEW ===============================================================================
class CategoryTagsView : public ElementShower< CategoryTags >
{
	public:
								CategoryTagsView( void );

        Glib::ustring           get_title_str( void ) const
        { return m_ptr2elem->m_name; }
        Glib::ustring           get_info_str( void ) const
        { return Glib::ustring::compose( _( "Tag category with %1 tags" ),
                                         m_ptr2elem->size() ); }
#ifdef LIFEOGRAPH_DEBUG_BUILD
        Glib::ustring           get_tip_str( void ) const
        { return Glib::ustring::compose( "DEID: %1", m_ptr2elem->get_id() ); }
#endif
        Menu2*                  get_menu( void );

        void                    show( CategoryTags& );

	protected:
        bool                    is_title_editable( void ) const
        { return true; }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2elem->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

		void					dismiss_category( void );

        Gtk::MenuItem           *m_menuitem_dismiss;
};

// CHAPTER VIEW ====================================================================================
class ChapterView : public ElementShower< Chapter >
{
	public:
								ChapterView( void );

        Glib::ustring           get_title_str( void ) const
        {
            if( m_ptr2elem->m_date_begin.is_ordinal() )
                return Glib::ustring::compose( "%1: %2",
                                               m_ptr2elem->m_date_begin.format_string(),
                                               m_ptr2elem->m_name );
            else
                return m_ptr2elem->m_name;
        }
        Glib::ustring           get_info_str( void ) const
        {
            if( m_ptr2elem->m_date_begin.is_ordinal() )
                return Glib::ustring::compose( _( "Topic with %1 entries" ),
                                               m_ptr2elem->m_size );
            else
                return Glib::ustring::compose(
                        m_ptr2elem->m_time_span ?
                                _( "Chapter with %1 entries in %2 days" ) :
                                _( "Chapter with %1 entries" ),
                        m_ptr2elem->get_size(), m_ptr2elem->m_time_span );
        }
#ifdef LIFEOGRAPH_DEBUG_BUILD
        Glib::ustring           get_tip_str( void ) const
        { return Glib::ustring::compose( "DEID: %1", m_ptr2elem->get_id() ); }
#endif
        Menu2*                  get_menu( void );

        void                    show( Chapter& );

    protected:
        bool                    is_title_editable( void ) const
        { return true; }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2elem->m_name; }
		bool					check_title_applicable( const Glib::ustring& ) const;
		bool					apply_title( const Glib::ustring& );

		void					handle_date_changed( void );
		void					handle_date_applied( void );

        void                    update_labels( void );

		void					add_new_entry( void );
        void                    dismiss_chapter( void );

        Gtk::MenuItem           *m_menuitem_dismiss;

		Gtk::Entry				*m_entry_begins;
		Gtk::Button				*m_button_date_apply;
		Gtk::Label				*m_label_ends;

		Gtk::Frame				*m_frame_edit;
        Gtk::Frame              *m_frame_actions;
};

// THEME VIEW ======================================================================================
class ThemeView : public ElementShower< Theme >
{
	public:
								ThemeView( void );

        Glib::ustring           get_title_str( void ) const
        { return m_ptr2elem->get_name(); }
        Glib::ustring   get_info_str( void ) const;
#ifdef LIFEOGRAPH_DEBUG_BUILD
        Glib::ustring           get_tip_str( void ) const
        { return Glib::ustring::compose( "DEID: %1", m_ptr2elem->get_id() ); }
#endif
        Menu2*                  get_menu( void );

        void                    show( Theme& );

	protected:
        bool                    is_title_editable( void ) const
        { return( ! m_ptr2elem->is_system() ); }
        Glib::ustring           get_title_edit_str( void ) const
        { return m_ptr2elem->get_name(); }
        bool                    check_title_applicable( const Glib::ustring& ) const;
        bool                    apply_title( const Glib::ustring& );

		void					handle_font_changed( void );
		void					handle_color_changed( void );
		void					update_preview( void );

		void					duplicate( void );
		void					make_default( void );
		void					dismiss_theme( void );

        Gtk::MenuItem           *m_menuitem_duplicate;
        Gtk::MenuItem           *m_menuitem_make_default;
        Gtk::SeparatorMenuItem  *m_menuseparator;
        Gtk::MenuItem           *m_menuitem_dismiss;

        Gtk::FontButton         *m_fontbutton;
        Gtk::ColorButton        *m_colorbutton_bground;
        Gtk::ColorButton        *m_colorbutton_text;
        Gtk::ColorButton        *m_colorbutton_heading;
        Gtk::ColorButton        *m_colorbutton_subheading;
        Gtk::ColorButton        *m_colorbutton_highlight;
        Gtk::EventBox           *m_eventbox_preview;
        Gtk::Label              *m_label_preview;
};

// DIARY VIEW ======================================================================================
class DiaryView : public ElementShower< Diary >
{
	public:
								DiaryView( void );
								~DiaryView( void );

        Glib::ustring           get_list_str( void ) const
        { return Glib::ustring::compose( "<b>%1</b>",
                Glib::Markup::escape_text(
                        Glib::filename_display_basename( m_ptr2elem->m_path ) ) ); }
        Glib::ustring           get_title_str( void ) const
        { return m_ptr2elem->m_name; }
        Glib::ustring           get_info_str( void ) const
        {
            int timespan( 0 );
            // FIXME: does not work for some obscure reason..
            for( Entryciter iter_entry = m_ptr2elem->m_entries.begin();
                    iter_entry != m_ptr2elem->m_entries.end();
                    ++iter_entry )
            {
                if( ! iter_entry->second->get_date().is_ordinal() )
                {
                    timespan = m_ptr2elem->m_entries.rbegin()->second->get_date().
                            calculate_days_between( iter_entry->second->get_date() );
                    break;
                }
            }
            return Glib::ustring::compose( _( "Diary with %1 entries spanning %2 days" ),
                                           m_ptr2elem->m_entries.size(),
                                           timespan );
        }

        void                    handle_login( void );
        void                    show( Diary& );

	protected:
        // DIALOGS
		void					start_dialog_password( void );
		void					start_dialog_import( void );
		void					start_dialog_export( void );
		void					open_diary_folder( void );
        void                    handle_language_changed( void );
		void					handle_startup_type_changed( void );
		void					handle_startup_elem_dropped(
										const Glib::RefPtr< Gdk::DragContext >&,
										int, int,
										const Gtk::SelectionData&,
										uint, guint );
		void					go_to_startup_elem( void );

		void					update_combobox_chapter_ctg( void );
		void					handle_cur_chapter_ctg_changed( void );
		void					rename_cur_chapter_ctg( void );
		void					dismiss_cur_chapter_ctg( void );
		bool					is_row_separator( const Glib::RefPtr< Gtk::TreeModel >&,
												  const Gtk::TreeIter );

        Gtk::MenuItem           *m_menuitem_import;
        Gtk::SeparatorMenuItem  *m_menuseparator;
        Gtk::MenuItem           *m_menuitem_dont_save;

        Gtk::Frame              *m_frame_startup;
        Gtk::Frame              *m_frame_chapters;

		Gtk::Label				*m_label_path;
		Gtk::Label				*m_label_encryption;
        Gtk::ComboBoxText       *m_combobox_spellcheck;
		Gtk::Button				*m_button_password;
        WidgetChart             *m_widgetchart;

		// STARTUP
		Gtk::ComboBox			*m_combobox_startup_type;
		Gtk::Label				*m_label_startup_elem;
		Gtk::Label				*m_label_startup_elem_drop;
		Gtk::Button				*m_button_startup_elem_go;

		class ColrecStartupType : public Gtk::TreeModel::ColumnRecord
		{
			public:
				ColrecStartupType() { add( name ); add( type ); }
				Gtk::TreeModelColumn< Glib::ustring >	name;
				Gtk::TreeModelColumn< DEID >			type;
		};

		Glib::RefPtr< Gtk::ListStore >
								m_liststore_startup_type;
		ColrecStartupType		colrec_startup_type;

		// CHAPTER CATEGORIES & TOPICS
		Gtk::ComboBox           *m_combobox_chapter_ctg;
		Gtk::Button				*m_button_chapter_ctg_dismiss;
		Gtk::Button				*m_button_chapter_ctg_rename;

		enum ChapterCtgComboType { CCCT_CATEGORY, CCCT_NEW, CCCT_SEPARATOR };

		class ColrecChapterCtg : public Gtk::TreeModel::ColumnRecord
		{
			public:
				ColrecChapterCtg() { add( name ); add( ptr ); add( type ); }
				Gtk::TreeModelColumn< Glib::ustring >		name;
				Gtk::TreeModelColumn< CategoryChapters* >	ptr;
				Gtk::TreeModelColumn< ChapterCtgComboType >	type;
		};

		Glib::RefPtr< Gtk::ListStore >
								m_liststore_chapter_ctg;
		ColrecChapterCtg		colrec_chapter_ctg;

		// DIALOGS
		DialogPassword			*m_dialog_password;
		DialogImport			*m_dialog_import;
		DialogExport			*m_dialog_export;
};

} // end of namespace LIFEO

#endif

