/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <iomanip>

#include "lifeobase.hpp"
#include "diary.hpp"
#include "views.hpp"
#include "panel_extra.hpp"


using namespace LIFEO;


// TAG =============================================================================================
TagView *Lifeobase::view_tag = NULL;

TagView::TagView( void )
{
	try
	{
		Lifeobase::builder->get_widget( "vbox_tag", m_vbox );
		Lifeobase::builder->get_widget_derived( "menu_tag_elem", m_menu );
		Lifeobase::builder->get_widget( "menuitem_tag_dismiss", m_menuitem_dismiss );
		Lifeobase::builder->get_widget( "togglebutton_tag_filter", m_togglebutton_filter );
		Lifeobase::builder->get_widget( "combobox_tag_category", m_combobox_category );
		Lifeobase::builder->get_widget_derived( "drawingarea_tag_chart", m_widgetchart );

		m_liststore_categories = Gtk::ListStore::create( colrec_category );
	}
	catch( ... ) {}

	Tag::shower = this;

	m_combobox_category->set_model( m_liststore_categories );
	m_combobox_category->pack_start( colrec_category.name );
	m_combobox_category->set_row_separator_func(
			sigc::mem_fun( this, &TagView::is_row_separator ) );

	// SIGNALS
	m_togglebutton_filter->signal_toggled().connect(
			sigc::mem_fun( this, &TagView::handle_filter_toggled ) );
	m_menuitem_dismiss->signal_activate().connect(
			sigc::mem_fun( this, &TagView::dismiss_tag ) );

	m_combobox_category->signal_changed().connect(
			sigc::mem_fun( this, &TagView::handle_category_changed ) );
}

bool
TagView::check_title_applicable( const Glib::ustring &name ) const
{
	Tag *tag = Diary::d->get_tags()->get_tag( name );
	return( !( tag || name.empty() ) );
}

bool
TagView::apply_title( const Glib::ustring &name )
{
	// a caution against calls by an entry activation event:
	if( ! check_title_applicable( name ) )
		return false;

	Diary::d->get_tags()->rename( m_ptr2elem, name );
	Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2elem );
	row[ PanelExtra::colrec->name ] = m_ptr2elem->get_list_str();

	return true;
}

void
TagView::handle_login( void )
{
    update_combobox_categories();
    m_combobox_category->set_sensitive( ! Diary::d->is_read_only() );
}

void
TagView::handle_filter_toggled( void )
{
    if( Lifeobase::m_internaloperation )
        return;

    // REMOVE EXISTING FILTER IF ANY
    const Tag *tag( Diary::d->get_filter_tag() );
    if( tag != NULL && tag != m_ptr2elem )
    {
        Diary::d->set_filter_tag( NULL );
        Gtk::TreeRow row = * Lifeobase::base->get_element_row( tag );
        row[ PanelExtra::colrec->icon ] = Lifeobase::icons->tag_16;
    }

    // SET THE NEW FILTER
    Diary::d->set_filter_tag( m_togglebutton_filter->get_active() ? m_ptr2elem : NULL );
    update_button_filter_tag();
    Lifeobase::panel_diary->update_button_remove_filters();
    Lifeobase::panel_diary->update_entry_list();
    // UPDATE ICON
    Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2elem );
    row[ PanelExtra::colrec->icon ] = m_ptr2elem->get_icon();
}

void
TagView::handle_category_changed( void )
{
	if( Lifeobase::m_internaloperation )
		return;

	Gtk::TreeIter iter = m_combobox_category->get_active();
	if( iter )
	{
		Gtk::TreeRow row = *iter;
		if( row[ colrec_category.type ]  == TCT_NEW ) // create new
		{
			CategoryTags *category = Diary::d->create_tag_ctg();
			m_ptr2elem->set_category( category );
			update_combobox_categories();
			Lifeobase::panel_extra->populate();
			category->show();
			Lifeobase::panel_main->start_title_edit();
		}
		else	// assign a category or reset
		{
			m_ptr2elem->set_category( row[ colrec_category.ptr ] );
			Lifeobase::panel_extra->populate();
		}
	}
}

void
TagView::update_combobox_categories( void )
{
	m_liststore_categories->clear();

	Gtk::TreeRow row = * m_liststore_categories->append();

	row[ colrec_category.name ] = _( "None" );
	row[ colrec_category.ptr ] = NULL;
	row[ colrec_category.type ] = TCT_NONE;

	// separator 1:
	row = * m_liststore_categories->append();
	row[ colrec_category.type ] = TCT_SEPARATOR;
	row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;

	for( PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.begin();
		 iter != Diary::d->m_tag_categories.end();
		 ++iter )
	{
		row = * m_liststore_categories->append();

		row[ colrec_category.name ] = iter->first;
		row[ colrec_category.ptr ] = iter->second;
		row[ colrec_category.type ] = TCT_CATEGORY;
	}

	// separator 2:
	if( Diary::d->m_tag_categories.size() > 0 )
	{
		row = * m_liststore_categories->append();
		row[ colrec_category.type ] = TCT_SEPARATOR;
		row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
	}

	row = * m_liststore_categories->append();

	row[ colrec_category.name ] = _( "Create New..." );
	// a dummy value to differentiate it from "None":
	row[ colrec_category.ptr ] = ( CategoryTags* ) 0x1;
	row[ colrec_category.type ] = TCT_NEW;
}

void
TagView::update_button_filter_tag( void )
{
    Lifeobase::m_internaloperation++;
    m_togglebutton_filter->set_active( Diary::d->get_filter_tag() == m_ptr2elem );
    Lifeobase::m_internaloperation--;
}

void
TagView::show( Tag &tag )
{
	// do nothing is entry is already the current element:
    if( Lifeobase::panel_main->is_cur_elem( &tag ) )
        return;

	m_ptr2elem = &tag;

    Lifeobase::panel_main->show( this );

	Lifeobase::m_internaloperation++;

	// BODY
	update_button_filter_tag();

	for( Gtk::TreeIter iter = m_liststore_categories->children().begin();
		 iter != m_liststore_categories->children().end();
		 ++iter )
	{
		if( ( *iter )[ colrec_category.ptr ] == tag.get_category() )
			m_combobox_category->set_active( iter );
	}

	// CHART DATA
	ChartPoints *points( new ChartPoints );
	Date yearmonth_last( 0 );
	int value( 0 );
	for( SetDiaryElements::reverse_iterator iter = tag.m_items.rbegin();
		 iter != tag.m_items.rend(); ++iter )
	{
		Entry *entry = dynamic_cast< Entry* >( *iter );
        if( entry->get_date().is_ordinal() )
            continue;
		const Date ym_entry( entry->get_date().get_yearmonth() );
		if( yearmonth_last == 0 )
			yearmonth_last = ym_entry;
		else
		if( yearmonth_last != ym_entry )
		{
			points->add( yearmonth_last, value );

			// fill empty months
			for( Date ym_zero = yearmonth_last; ; )
			{
				ym_zero.forward_month();
				if( ym_zero >= ym_entry )
					break;
				points->add( ym_zero, 0 );
			}

			yearmonth_last = ym_entry;
			value = 0;
		}
		value++;
	}
	points->add( yearmonth_last, value );

	m_widgetchart->set_points( points );

	Lifeobase::m_internaloperation--;
}

Menu2*
TagView::get_menu( void )
{
    return( Diary::d->is_read_only() ? NULL : m_menu );
}

void
TagView::dismiss_tag( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::panel_main->remove_element_from_history( m_ptr2elem );

	// clear filter if necessary:
	if( Diary::d->get_filter_tag() == m_ptr2elem )
	{
		Diary::d->set_filter_tag( NULL );
        Lifeobase::panel_diary->update_button_remove_filters();
        Lifeobase::panel_diary->update_entry_list();
	}
	Diary::d->dismiss_tag( m_ptr2elem );
	Lifeobase::panel_extra->populate();
}

bool
TagView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel > &model,
							 const Gtk::TreeIter iter )
{
	Gtk::TreeRow row = * iter;
	return ( row[ colrec_category.type ] == TCT_SEPARATOR );
}

// TAG CATEGORY ====================================================================================
CategoryTagsView::CategoryTagsView( void )
{
	CategoryTags::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_categorytags", m_vbox );
		Lifeobase::builder->get_widget_derived( "menu_tagctg_elem", m_menu );
		Lifeobase::builder->get_widget( "menuitem_tagctg_dismiss", m_menuitem_dismiss );
	}
	catch( ... ) {}

	m_menuitem_dismiss->signal_activate().connect(
			sigc::mem_fun( this, &CategoryTagsView::dismiss_category ) );
}

void
CategoryTagsView::show( CategoryTags &category )
{
	// do nothing is entry is already the current element:
    if( Lifeobase::panel_main->is_cur_elem( &category ) )
		return;

	m_ptr2elem = &category;

	Lifeobase::panel_main->show( this );
}

Menu2*
CategoryTagsView::get_menu( void )
{
    return( Diary::d->is_read_only() ? NULL : m_menu );
}

bool
CategoryTagsView::check_title_applicable( const Glib::ustring &name ) const
{
	PoolCategoriesTags::const_iterator iter = Diary::d->m_tag_categories.find( name );
	return( iter == Diary::d->m_tag_categories.end() && name.empty() == false );
}

bool
CategoryTagsView::apply_title( const Glib::ustring &name )
{
    if( ! check_title_applicable( name ) )
        return false;

	Diary::d->m_tag_categories.rename_category( m_ptr2elem, name );
	Lifeobase::panel_extra->populate();
	Lifeobase::view_tag->update_combobox_categories();

	return true;
}

void
CategoryTagsView::dismiss_category( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::panel_main->remove_element_from_history( m_ptr2elem );
	// TODO: offer merging with a category
	Diary::d->dismiss_tag_ctg( m_ptr2elem );
	Lifeobase::panel_extra->populate();
	Lifeobase::view_tag->update_combobox_categories();
}

// CHAPTER =========================================================================================
ChapterView::ChapterView( void )
{
	Chapter::shower = this;

	Gtk::Button *button_new_entry( NULL );

	try
	{
		Lifeobase::builder->get_widget( "vbox_chapter", m_vbox );
		Lifeobase::builder->get_widget_derived( "menu_chapter_elem", m_menu );
		Lifeobase::builder->get_widget( "menuitem_chapter_dismiss", m_menuitem_dismiss );

//		Lifeobase::builder->get_widget( "label_chapter_period", m_label_period );
//		Lifeobase::builder->get_widget( "hbox_chapter_period", m_hbox_period );
		Lifeobase::builder->get_widget( "entry_chapter_begins", m_entry_begins );
		Lifeobase::builder->get_widget( "button_chapter_date_apply", m_button_date_apply );
		Lifeobase::builder->get_widget( "label_chapter_ends", m_label_ends );

		Lifeobase::builder->get_widget( "frame_chapter_edit", m_frame_edit );
        Lifeobase::builder->get_widget( "frame_chapter_actions", m_frame_actions );
		Lifeobase::builder->get_widget( "button_chapter_new_entry", button_new_entry );
	}
	catch( ... )
	{
	    throw HELPERS::Error( "Chapter view could not be constructed" );
	}
	
	// SIGNALS
	m_menuitem_dismiss->signal_activate().connect(
			sigc::mem_fun( this, &ChapterView::dismiss_chapter ) );

	m_entry_begins->signal_changed().connect(
			sigc::mem_fun( *this, &ChapterView::handle_date_changed ) );
	m_entry_begins->signal_activate().connect(
			sigc::mem_fun( *m_button_date_apply, &Gtk::Button::clicked ) );
	m_button_date_apply->signal_clicked().connect(
			sigc::mem_fun( *this, &ChapterView::handle_date_applied ) );

	button_new_entry->signal_clicked().connect(
			sigc::mem_fun( *this, &ChapterView::add_new_entry ) );
}

void
ChapterView::dismiss_chapter( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::panel_main->remove_element_from_history( m_ptr2elem );
	Diary::d->dismiss_chapter( m_ptr2elem );
	Lifeobase::panel_diary->update_entry_list();
}

bool
ChapterView::check_title_applicable( const Glib::ustring &str ) const
{
	return( ! str.empty() && str != m_ptr2elem->m_name );
}

bool
ChapterView::apply_title( const Glib::ustring &str )
{
	// a caution against calls by an entry activation event:
	if( ! check_title_applicable( str ) )
		return false;
	m_ptr2elem->set_name( str );
	Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2elem );
	row[ ListData::colrec->info ] = m_ptr2elem->get_list_str();
	return true;
}

void
ChapterView::handle_date_changed( void )
{
	try
	{
		Date date( m_entry_begins->get_text() );
		m_button_date_apply->set_sensitive( 
				! Diary::d->get_current_chapter_ctg()->get_chapter( date.m_date ) );
	}
	catch( LIFEO::Error &er )
	{
		m_button_date_apply->set_sensitive( false );
	}
}

void
ChapterView::handle_date_applied( void )
{
	// a caution against calls by an entry activation event:
	if( ! m_button_date_apply->is_sensitive() )
		return;
	if( Diary::d->get_current_chapter_ctg()->set_chapter_date(
			m_ptr2elem, Date( m_entry_begins->get_text() ) ) )
	{
		m_button_date_apply->set_sensitive( false );
		Lifeobase::panel_diary->update_entry_list();
		update_labels();
	}
}

void
ChapterView::update_labels( void )
{
	if( ! m_ptr2elem->m_date_begin.is_ordinal() )
	{
        Lifeobase::m_internaloperation++;

		m_entry_begins->set_text( m_ptr2elem->m_date_begin.format_string() );

		CategoryChapters::iterator iter( Diary::d->get_current_chapter_ctg()->find(
												m_ptr2elem->get_date().m_date ) );
		if( m_ptr2elem->m_time_span == 0 )
			m_label_ends->set_text( _( "Unlimited" ) );
		else
		{
			--iter;
			m_label_ends->set_text( iter->second->m_date_begin.format_string() );
		}

        Lifeobase::m_internaloperation--;
	}
}

void
ChapterView::add_new_entry( void )
{
	Diary::d->create_entry( m_ptr2elem->get_free_order().m_date );

	Lifeobase::panel_diary->update_entry_list();
	Lifeobase::panel_diary->update_calendar();
}

void
ChapterView::show( Chapter &chapter )
{
    // do nothing if entry is already the current element:
    if( Lifeobase::panel_main->is_cur_elem( &chapter ) )
        return;

	m_ptr2elem = &chapter;

	Lifeobase::panel_main->show( this );

	// BODY
	update_labels();

	bool flag_ordinal( chapter.m_date_begin.is_ordinal() );
	//m_label_period->set_visible( flag_temporal );
	//m_hbox_period->set_visible( flag_temporal );
	m_frame_edit->set_visible( ! flag_ordinal );
	m_frame_actions->set_visible( flag_ordinal );

    // TODO: move to handle_login()
	m_frame_edit->set_sensitive( ! Diary::d->is_read_only() );
    m_frame_actions->set_sensitive( ! Diary::d->is_read_only() );
}

Menu2*
ChapterView::get_menu( void )
{
    return( Diary::d->is_read_only() ? NULL : m_menu );
}

// THEME ===========================================================================================
ThemeView::ThemeView( void )
{
	Theme::shower = this;

	try
	{
		Lifeobase::builder->get_widget( "vbox_theme", m_vbox );
        Lifeobase::builder->get_widget_derived( "menu_theme_elem", m_menu );
        Lifeobase::builder->get_widget( "menuitem_theme_duplicate", m_menuitem_duplicate );
        Lifeobase::builder->get_widget( "menuitem_theme_default", m_menuitem_make_default );
        Lifeobase::builder->get_widget( "menuseparator_theme_1", m_menuseparator );
        Lifeobase::builder->get_widget( "menuitem_theme_dismiss", m_menuitem_dismiss );
		Lifeobase::builder->get_widget( "fontbutton_theme", m_fontbutton );
		Lifeobase::builder->get_widget( "colorbutton_theme_bground", m_colorbutton_bground );
		Lifeobase::builder->get_widget( "colorbutton_theme_text", m_colorbutton_text );
		Lifeobase::builder->get_widget( "colorbutton_theme_heading", m_colorbutton_heading );
		Lifeobase::builder->get_widget( "colorbutton_theme_subheading", m_colorbutton_subheading );
		Lifeobase::builder->get_widget( "colorbutton_theme_highlight", m_colorbutton_highlight );
		Lifeobase::builder->get_widget( "eventbox_theme_preview", m_eventbox_preview );
		Lifeobase::builder->get_widget( "label_theme_preview", m_label_preview );
	}
	catch( ... ) { }

	m_fontbutton->signal_font_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_font_changed ) );
	m_colorbutton_bground->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_text->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_heading->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_subheading->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );
	m_colorbutton_highlight->signal_color_set().connect(
			sigc::mem_fun( *this, &ThemeView::handle_color_changed ) );

	// TOOLBAR
    m_menuitem_duplicate->signal_activate().connect(
            sigc::mem_fun( this, &ThemeView::duplicate ) );
    m_menuitem_make_default->signal_activate().connect(
            sigc::mem_fun( this, &ThemeView::make_default ) );
    m_menuitem_dismiss->signal_activate().connect(
            sigc::mem_fun( this, &ThemeView::dismiss_theme ) );
}

void
ThemeView::show( Theme &theme )
{
	// do nothing is entry is already the current element:
    if( Lifeobase::panel_main->is_cur_elem( &theme ) )
		return;

	m_ptr2elem = &theme;

	Lifeobase::panel_main->show( this );

	// BODY
	m_fontbutton->set_font_name( theme.font.to_string() );
	m_colorbutton_bground->set_rgba( theme.color_base );
	m_colorbutton_text->set_rgba( theme.color_text );
	m_colorbutton_heading->set_rgba( theme.color_heading );
	m_colorbutton_subheading->set_rgba( theme.color_subheading );
	m_colorbutton_highlight->set_rgba( theme.color_highlight );

	update_preview();
	bool flag_sensitive = !( theme.is_system() || Diary::d->is_read_only() );
	m_vbox->set_sensitive( flag_sensitive );
	m_menuitem_make_default->set_visible( &theme != Diary::d->get_default_theme() );
	m_menuseparator->set_visible( flag_sensitive );
    m_menuitem_dismiss->set_visible( flag_sensitive );
}

Glib::ustring
ThemeView::get_info_str( void ) const
{
    Glib::ustring str( m_ptr2elem->is_system() ? _( "System theme" ) : _( "Custom theme" ) );
    if( m_ptr2elem == Diary::d->get_default_theme() )
    {
        str += " (";
        str += _( "Default" );
        str += ")";
    }
    return str;
}

Menu2*
ThemeView::get_menu( void )
{
    return( Diary::d->is_read_only() ? NULL : m_menu );
}

bool
ThemeView::check_title_applicable( const Glib::ustring &name ) const
{
    // system theme added back to m_themes:
//    if( name == STRING::SYSTEM_THEME )
//        return false;
    PoolThemes::const_iterator iter = Diary::d->get_themes().find( name );
	return( !( iter != Diary::d->get_themes().end() || name.empty() ) );
}

bool
ThemeView::apply_title( const Glib::ustring &name )
{
    if( ! check_title_applicable( name ) )
        return false;

    Diary::d->m_themes.rename_theme( m_ptr2elem, name );
    Gtk::TreeRow row = * Lifeobase::base->get_element_row( m_ptr2elem );
    row[ PanelExtra::colrec->name ] = m_ptr2elem->get_list_str();

    return true;
}

void
ThemeView::handle_font_changed( void )
{
	m_ptr2elem->font = Pango::FontDescription( m_fontbutton->get_font_name() );
	update_preview();
}

void
ThemeView::handle_color_changed( void )
{
	m_ptr2elem->color_base = m_colorbutton_bground->get_rgba();
	m_ptr2elem->color_text = m_colorbutton_text->get_rgba();
	m_ptr2elem->color_heading = m_colorbutton_heading->get_rgba();
	m_ptr2elem->color_subheading = m_colorbutton_subheading->get_rgba();
	m_ptr2elem->color_highlight = m_colorbutton_highlight->get_rgba();
	update_preview();
}

void
ThemeView::update_preview( void )
{
    Glib::ustring str_preview( Glib::ustring::compose(
            "<span color='%1' size='xx-large' weight='bold'>Heading</span>\n"
            " <span color='%2'size='larger' weight='bold'>Subheading</span>\n"
            "Body text <span bgcolor='%3'>highlighted text</span>",
            convert_gdkcolor_to_html( m_ptr2elem->color_heading ),
            convert_gdkcolor_to_html( m_ptr2elem->color_subheading ),
            convert_gdkcolor_to_html( m_ptr2elem->color_highlight ) ) );
    m_label_preview->set_markup( str_preview );
    m_label_preview->override_color( m_ptr2elem->color_text, Gtk::STATE_FLAG_NORMAL );
    m_label_preview->override_color( m_ptr2elem->color_text, Gtk::STATE_FLAG_INSENSITIVE );
    m_eventbox_preview->override_background_color( m_ptr2elem->color_base,
                                                   Gtk::STATE_FLAG_NORMAL );
    m_eventbox_preview->override_background_color( m_ptr2elem->color_base,
                                                   Gtk::STATE_FLAG_INSENSITIVE );
    m_label_preview->override_font( m_ptr2elem->font );
}

// creating new themes is done by duplicating existing ones
void
ThemeView::duplicate( void )
{
	Theme *new_theme = Diary::d->duplicate_theme( m_ptr2elem );
	Lifeobase::panel_extra->populate();
	new_theme->show();
	Lifeobase::panel_main->start_title_edit();
}

void
ThemeView::make_default( void )
{
	Diary::d->set_default_theme( m_ptr2elem );
	Lifeobase::panel_extra->populate();
    Lifeobase::panel_main->refresh_extra_info();
	m_menuitem_make_default->set_visible( false );
}

void
ThemeView::dismiss_theme( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	Lifeobase::panel_main->remove_element_from_history( m_ptr2elem );

	Theme *theme = m_ptr2elem; // always be on the safe side
	ThemeSystem::get()->show();
	Diary::d->dismiss_theme( theme );
	Lifeobase::panel_extra->populate();
}

// DIARY ===========================================================================================
DiaryView::DiaryView( void )
:	m_dialog_password( NULL ), m_dialog_import( NULL ), m_dialog_export( NULL )
{
	Diary::shower = this;
	m_ptr2elem = Diary::d;	// for now no other diary is shown in the view

	Gtk::MenuItem   *menuitem_export( NULL );
    Gtk::Box        *hbox_chapter_ctg( NULL );
	try
	{
		Lifeobase::builder->get_widget( "vbox_diary", m_vbox );

        Lifeobase::builder->get_widget_derived( "menu_diary_elem", m_menu );
        Lifeobase::builder->get_widget( "menuitem_import", m_menuitem_import );
        Lifeobase::builder->get_widget( "menuitem_export", menuitem_export );
        Lifeobase::builder->get_widget( "menuseparator_diary_1", m_menuseparator );
        Lifeobase::builder->get_widget( "menuitem_diary_dont_save",
                m_menuitem_dont_save );

		Lifeobase::builder->get_widget( "frame_diary_startup", m_frame_startup );
        Lifeobase::builder->get_widget( "frame_diary_chapters", m_frame_chapters );

		Lifeobase::builder->get_widget( "label_path", m_label_path );
		Lifeobase::builder->get_widget( "label_encryption", m_label_encryption );
		Lifeobase::builder->get_widget( "button_password", m_button_password );
        Lifeobase::builder->get_widget( "combobox_diary_spellcheck", m_combobox_spellcheck );

		Lifeobase::builder->get_widget( "cmb_startup_type", m_combobox_startup_type );
		Lifeobase::builder->get_widget( "label_startup_elem", m_label_startup_elem );
		Lifeobase::builder->get_widget( "label_startup_elem_drop", m_label_startup_elem_drop );
		Lifeobase::builder->get_widget( "button_startup_elem_go", m_button_startup_elem_go );

//		Lifeobase::builder->get_widget( "combobox_chapter_ctg",
//				m_combobox_chapter_ctg );
		Lifeobase::builder->get_widget( "hbox_cur_chapter_ctg",
		        hbox_chapter_ctg );
		Lifeobase::builder->get_widget( "button_chapter_ctg_dismiss",
				m_button_chapter_ctg_dismiss );
		Lifeobase::builder->get_widget( "button_chapter_ctg_rename",
				m_button_chapter_ctg_rename );
        Lifeobase::builder->get_widget( "combo_chapter_ctg", m_combobox_chapter_ctg );

        Lifeobase::builder->get_widget_derived( "drawingarea_diary_chart", m_widgetchart );

		m_liststore_startup_type = Gtk::ListStore::create( colrec_startup_type );
		m_liststore_chapter_ctg = Gtk::ListStore::create( colrec_chapter_ctg );
	}
	catch( ... ) { }

	// STARTUP
	m_combobox_startup_type->set_model( m_liststore_startup_type );
	m_combobox_startup_type->pack_start( colrec_startup_type.name );
	Gtk::TreeRow row( * m_liststore_startup_type->append() );
	row[ colrec_startup_type.name ] = _( "Show Most Current Entry" );
	row[ colrec_startup_type.type ] = HOME_CURRENT_ELEM;
	row = * m_liststore_startup_type->append();
	row[ colrec_startup_type.name ] = _( "Remember Last Item" );
	row[ colrec_startup_type.type ] = HOME_LAST_ELEM;
	row = * m_liststore_startup_type->append();
	row[ colrec_startup_type.name ] = _( "Always Show a Fixed Item" );
	row[ colrec_startup_type.type ] = DEID_MIN;

	m_label_startup_elem_drop->drag_dest_set( Lifeobase::base->drag_targets_entry );

	// CHAPTER CATEGORIES
	m_combobox_chapter_ctg->set_model( m_liststore_chapter_ctg );
	m_combobox_chapter_ctg->set_entry_text_column( colrec_chapter_ctg.name );
	m_combobox_chapter_ctg->set_row_separator_func(
			sigc::mem_fun( this, &DiaryView::is_row_separator ) );

	// SIGNALS
	m_button_password->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_password ) );

    m_combobox_spellcheck->signal_changed().connect(
            sigc::mem_fun( this, &DiaryView::handle_language_changed ) );

	m_combobox_startup_type->signal_changed().connect(
			sigc::mem_fun( this, &DiaryView::handle_startup_type_changed ) );
	m_label_startup_elem_drop->signal_drag_data_received().connect(
			sigc::mem_fun( this, &DiaryView::handle_startup_elem_dropped ) );
	m_button_startup_elem_go->signal_clicked().connect(
			sigc::mem_fun( *this, &DiaryView::go_to_startup_elem ) );

	m_combobox_chapter_ctg->signal_changed().connect(
			sigc::mem_fun( this, &DiaryView::handle_cur_chapter_ctg_changed ) );
	m_combobox_chapter_ctg->get_entry()->signal_activate().connect(
			sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );
	m_button_chapter_ctg_rename->signal_clicked().connect(
			sigc::mem_fun( this, &DiaryView::rename_cur_chapter_ctg ) );
	m_button_chapter_ctg_dismiss->signal_clicked().connect(
			sigc::mem_fun( this, &DiaryView::dismiss_cur_chapter_ctg ) );

	m_menuitem_import->signal_activate().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_import ) );
	menuitem_export->signal_activate().connect(
			sigc::mem_fun( *this, &DiaryView::start_dialog_export ) );
	m_menuitem_dont_save->signal_activate().connect(
			sigc::bind( sigc::mem_fun( *Lifeobase::base, &Lifeobase::logout ), false ) );
}

DiaryView::~DiaryView( void )
{
	Diary::shower = NULL;
}

void
DiaryView::handle_login( void )
{
	Lifeobase::m_internaloperation++;

    // SPELL CHECKING
	m_combobox_spellcheck->remove_all();
    m_combobox_spellcheck->append( _( STRING::OFF ) );
    for( LanguageList::iterator iter = Lifeobase::s_lang_list.begin();
         iter != Lifeobase::s_lang_list.end(); ++iter )
    {
        m_combobox_spellcheck->append( *iter );
    }

    if( Diary::d->m_language.empty() )
        m_combobox_spellcheck->set_active_text( _( STRING::OFF ) );
    else
    {
        if( Lifeobase::s_lang_list.find( Diary::d->m_language ) ==
            Lifeobase::s_lang_list.end() )
            m_combobox_spellcheck->append( Diary::d->m_language );

        m_combobox_spellcheck->set_active_text( Diary::d->m_language );
    }

	update_combobox_chapter_ctg();
	if( Diary::d->m_startup_elem < HOME_FIXED_ELEM )
		m_combobox_startup_type->set_active( Diary::d->m_startup_elem - 1 );
	else
		m_combobox_startup_type->set_active( 2 );

	Lifeobase::m_internaloperation--;

	bool editable( ! Diary::d->is_read_only() );
	m_button_password->set_sensitive( editable );
    m_frame_startup->set_sensitive( editable );
    m_frame_chapters->set_sensitive( editable );
    m_menuitem_import->set_visible( editable );
    m_menuseparator->set_visible( editable );
    m_menuitem_dont_save->set_visible( editable );

    m_label_path->set_markup( Glib::ustring::compose( "<a href=\"file://%1\">%1</a>",
            Glib::path_get_dirname( Diary::d->m_path ) ) );
}

void
DiaryView::show( Diary &diary )
{
	// do nothing is entry is already the current element:
    if( Lifeobase::panel_main->is_cur_elem( &diary ) )
		return;

	// not necessary for now as there is only one diary to show in the view:
	// m_ptr2elem = &diary;

    Lifeobase::panel_main->show( this );

	// BODY
	m_button_password->set_label( diary.m_passphrase.empty() ? _( "Encrypt..." ) :
												_( "Change Password..." ) );
	m_label_encryption->set_label( diary.m_passphrase.empty() ? _( "Unencrypted" ) :
												_( "Encrypted" ) );

    // CHART DATA
    ChartPoints *points( new ChartPoints );
    Date yearmonth_last( 0 );
    int value( 0 );
    for( PoolEntries::reverse_iterator iter = diary.m_entries.rbegin();
         iter != diary.m_entries.rend(); ++iter )
    {
        Entry *entry( iter->second );
        if( entry->get_date().is_ordinal() )
            continue;
        const Date ym_entry( entry->get_date().get_yearmonth() );
        if( yearmonth_last == 0 )
            yearmonth_last = ym_entry;
        else
        if( yearmonth_last != ym_entry )
        {
            points->add( yearmonth_last, value );

            // fill empty months
            for( Date ym_zero = yearmonth_last; ; )
            {
                ym_zero.forward_month();
                if( ym_zero >= ym_entry )
                    break;
                points->add( ym_zero, 0 );
            }

            yearmonth_last = ym_entry;
            value = 0;
        }
        value++;
    }
    points->add( yearmonth_last, value );

    m_widgetchart->set_points( points );
}

void
DiaryView::start_dialog_password( void )
{
	try
	{
		m_dialog_password = new DialogPassword(	m_ptr2elem->m_passphrase,
				m_ptr2elem->m_passphrase.size() > 0 ?
		// TRANSLATORS: this is the title of the password dialog
						_( "Change Password" ) :
		// TRANSLATORS: this is the title of the password dialog
						_( "Add Password" ) );
	}
	catch( ... )
	{
		// TODO: throw
		return;
	}

	m_dialog_password->run();

	if( m_dialog_password->is_passphrase_set() )
	// TODO: if( m_dialog_password->run() )
	{
		if( m_ptr2elem->set_passphrase( m_dialog_password->get_passphrase() ) )
		{
			m_label_encryption->set_label( _( "Encrypted" ) );
			m_button_password->set_label( _( "Change Password..." ) );
		}
	}

	delete m_dialog_password;
	m_dialog_password = NULL;
}

void
DiaryView::start_dialog_import( void )
{
    if( m_dialog_import == NULL )
        Lifeobase::builder->get_widget_derived( "dialog_import", m_dialog_import );

    int response = m_dialog_import->run();
    m_dialog_import->hide();

    if( response == RESPONSE_GO )
    {
        Lifeobase::panel_diary->update_entry_list();
        Lifeobase::panel_extra->populate();
    }
}

void
DiaryView::start_dialog_export( void )
{
    if( m_dialog_export == NULL )
        Lifeobase::builder->get_widget_derived( "dialog_export", m_dialog_export );

    m_dialog_export->run();
    m_dialog_export->hide();
}

//void
//DiaryView::open_diary_folder( void )
//{
//	std::string uri = "file://" + Glib::path_get_dirname( m_path );
//	GError *err = NULL;
//	gtk_show_uri( NULL, uri.c_str(), GDK_CURRENT_TIME, &err);
//}

void
DiaryView::handle_language_changed( void )
{
    if( Lifeobase::m_internaloperation )
        return;
    std::string lang( m_combobox_spellcheck->get_active_text() );
    Diary::d->set_lang( lang == _( STRING::OFF ) ? "" : lang );
}

void
DiaryView::handle_startup_type_changed( void )
{
	if( ! Lifeobase::m_internaloperation )
	{
		Gtk::TreeRow row( * m_combobox_startup_type->get_active() );
		Diary::d->m_startup_elem = row[ colrec_startup_type.type ];
	}
	bool flag_show_fixed_item( Diary::d->m_startup_elem > HOME_FIXED_ELEM );
	m_label_startup_elem->set_visible( flag_show_fixed_item );
	m_label_startup_elem_drop->set_visible( flag_show_fixed_item );
	m_button_startup_elem_go->set_visible( flag_show_fixed_item );
	if( flag_show_fixed_item )
		m_label_startup_elem_drop->set_label( Diary::d->get_startup_elem()->get_name() );
}

void
DiaryView::handle_startup_elem_dropped(
				const Glib::RefPtr< Gdk::DragContext > &context,
				int x, int y,
				const Gtk::SelectionData &selection_data,
				uint info,
				guint time )
{
	context->drag_finish( true, false, time );

	Glib::ustring name = selection_data.get_data_as_string();

	if( name.empty() )	// should never happen
		return;

	switch( info )
	{
		case DRAG_TARGET_ENTRY_INFO:
		{
			DiaryElement *element = Lifeobase::panel_diary->get_dragged_elem();
			if( element != NULL )
			{
				m_label_startup_elem_drop->set_label( element->get_name() );
				Diary::d->m_startup_elem = element->get_id();
			}
			break;
		}
		default:
			break;
	}
}

void
DiaryView::go_to_startup_elem( void )
{
	DiaryElement *element( Diary::d->get_startup_elem() );
	if( element != NULL && element != Diary::d )
		element->show();
}

void
DiaryView::update_combobox_chapter_ctg( void )
{
	m_liststore_chapter_ctg->clear();

	Gtk::TreeRow row;

	for( PoolCategoriesChapters::const_iterator iter =
				m_ptr2elem->m_chapter_categories.begin();
		 iter != m_ptr2elem->m_chapter_categories.end();
		 ++iter )
	{
		row = * m_liststore_chapter_ctg->append();

		CategoryChapters *category = iter->second;

		row[ colrec_chapter_ctg.name ] = category->get_name();
		row[ colrec_chapter_ctg.ptr ] = category;
		row[ colrec_chapter_ctg.type ] = CCCT_CATEGORY;

		// setting current item:
		if( m_ptr2elem->m_ptr2chapter_ctg_cur == category )
			m_combobox_chapter_ctg->set_active( row );
	}

	m_button_chapter_ctg_rename->set_visible( false );
	m_button_chapter_ctg_dismiss->set_visible( true );
	m_button_chapter_ctg_dismiss->set_sensitive( m_ptr2elem->m_chapter_categories.size() > 1 );

	// separator 2:
	if( m_ptr2elem->m_chapter_categories.size() > 0 )
	{
		row = * m_liststore_chapter_ctg->append();
		row[ colrec_chapter_ctg.type ] = CCCT_SEPARATOR;
	}

	row = * m_liststore_chapter_ctg->append();

	row[ colrec_chapter_ctg.name ] = _( "Create New..." );
	row[ colrec_chapter_ctg.ptr ] = NULL;
	row[ colrec_chapter_ctg.type ] = CCCT_NEW;
}

void
DiaryView::handle_cur_chapter_ctg_changed( void )
{
	if( Lifeobase::m_internaloperation ) return;

	Gtk::TreeIter iter = m_combobox_chapter_ctg->get_active();
	bool flag_renamed = ! ( iter ), flag_existing_ctg( false );

	if( flag_renamed )
	{
		Glib::ustring name = m_combobox_chapter_ctg->get_entry_text();
		bool flag_name_usable = m_ptr2elem->m_chapter_categories.count( name ) < 1;
		m_button_chapter_ctg_rename->set_sensitive( flag_name_usable );
	}
	else
	{
		Gtk::TreeRow row = *iter;
		switch( row[ colrec_chapter_ctg.type ] )
		{
			case CCCT_NEW: // create new
				m_ptr2elem->m_ptr2chapter_ctg_cur = m_ptr2elem->create_chapter_ctg();
				update_combobox_chapter_ctg();
				flag_existing_ctg = true;
				break;
			case CCCT_CATEGORY:
				flag_existing_ctg = true;
				m_ptr2elem->m_ptr2chapter_ctg_cur = row[ colrec_chapter_ctg.ptr ];
				Lifeobase::panel_diary->update_entry_list();
				break;
			default:	// for separators
				break;
		}

		m_button_chapter_ctg_dismiss->set_sensitive( flag_existing_ctg );
	}

	m_button_chapter_ctg_rename->set_visible( flag_renamed );
	m_button_chapter_ctg_dismiss->set_visible( ! flag_renamed );
}

void
DiaryView::rename_cur_chapter_ctg( void )
{
	Glib::ustring name = m_combobox_chapter_ctg->get_entry_text();
	if( name == m_ptr2elem->m_ptr2chapter_ctg_cur->get_name() )
		return;
	m_ptr2elem->rename_chapter_ctg( m_ptr2elem->m_ptr2chapter_ctg_cur, name );
	update_combobox_chapter_ctg();
}

void
DiaryView::dismiss_cur_chapter_ctg( void )
{
	if( ! Lifeobase::base->confirm_dismiss_element() )
		return;

    Diary::d->show();

	m_ptr2elem->dismiss_chapter_ctg( m_ptr2elem->m_ptr2chapter_ctg_cur );
	update_combobox_chapter_ctg();
}

bool
DiaryView::is_row_separator( const Glib::RefPtr< Gtk::TreeModel > &model,
							 const Gtk::TreeIter iter )
{
	Gtk::TreeRow row = * iter;
	return ( row[ colrec_chapter_ctg.type ] == CCCT_SEPARATOR );
}
