/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include "lifeobase.hpp"
#include "view_entry.hpp"
#include "panel_extra.hpp"


using namespace LIFEO;


// TAG WIDGET ======================================================================================
TagWidget::TagWidget( BaseObjectType *cobject, const Glib::RefPtr<Gtk::Builder> &b )
:	EntryClear( cobject, b ), m_completion( Gtk::EntryCompletion::create() )
{
	m_completion->set_popup_completion( true );
	m_completion->set_popup_single_match( true );
	m_completion->set_match_func(
			sigc::mem_fun( *this, &TagWidget::compare_text ) );

	m_liststore = Gtk::ListStore::create( *PanelExtra::colrec );
	m_completion->set_model( m_liststore );
	m_completion->set_text_column( PanelExtra::colrec->name );

	set_completion( m_completion );
}

bool
TagWidget::compare_text( const Glib::ustring &text,
						 const Gtk::TreeModel::const_iterator &iter )
{
	Glib::ustring tagname = iter->get_value( PanelExtra::colrec->name );
	if( text.size() == 1 )
		return(	Glib::Unicode::tolower( tagname[ 0 ] ) ==
				Glib::Unicode::tolower( text[ 0 ] ) );
	else
		return( tagname.lowercase().find( text.lowercase() ) != std::string::npos );
}

void
TagWidget::populate( const Tagset *set )
{
	m_liststore->clear();

	Gtk::TreeRow row;

	for( Tagset::const_iterator iter = set->begin();
		 iter != set->end();
		 ++iter )
	{
		row = *( m_liststore->append() );
		row[ PanelExtra::colrec->name ] = ( *iter )->get_name();
	}
}

void
TagWidget::populate( void )
{
	m_liststore->clear();

	Gtk::TreeRow row;

	for( PoolTags::const_iterator iter = Diary::d->get_tags()->begin();
		 iter != Diary::d->get_tags()->end();
		 ++iter )
	{
		row = *( m_liststore->append() );
		row[ PanelExtra::colrec->name ] = ( *iter ).second->get_name();
	}
}

// PSEUDO DIARYELEMENT CLASS FOR HEADERS ===========================================================
class ListHeader : public DiaryElement
{
    public:
                                    ListHeader( DiaryElement::Type type )
        : DiaryElement( NULL ), m_type( type ) {}

        void                        show( void ) {}

        int                         get_size( void ) const
        { return 0; }   // redundant
        DiaryElement::Type          get_type( void ) const
        { return m_type; }

    protected:
        DiaryElement::Type          m_type;
};

static ListHeader s_header( DiaryElement::ET_HEADER ), s_none( DiaryElement::ET_NONE );

// TAG PANEL =======================================================================================
// STATIC MEMBERS
PanelExtra              *Lifeobase::panel_extra = NULL;
PanelExtra::Colrec      *PanelExtra::colrec;

PanelExtra::PanelExtra( BaseObjectType *cobject, const Glib::RefPtr<Gtk::Builder>& )
:   Gtk::TreeView( cobject ), m_treestore_all_tags( NULL ), m_elem_dragged( NULL ),
    m_flag_dragging( false ), m_flag_themes_expanded( false )
{
	Gtk::CellRendererPixbuf *cellrenderer_icon( NULL );
	Gtk::CellRendererText *cellrenderer_name( NULL );
	Gtk::TreeViewColumn *column_name_all( NULL );

	try
	{
		colrec = new Colrec;

		cellrenderer_icon = Gtk::manage( new Gtk::CellRendererPixbuf );
		cellrenderer_name = Gtk::manage( new Gtk::CellRendererText );
		column_name_all = Gtk::manage( new Gtk::TreeViewColumn( "" ) );
	}
	catch( ... ) { }

	cellrenderer_name->property_ellipsize() = Pango::ELLIPSIZE_END;
	//cellrenderer_name->property_scale() = .90;

	column_name_all->pack_start( *cellrenderer_icon, Gtk::PACK_SHRINK );
	column_name_all->pack_start( *cellrenderer_name );
	column_name_all->add_attribute( cellrenderer_icon->property_pixbuf(),
							PanelExtra::colrec->icon );
	column_name_all->add_attribute( cellrenderer_name->property_markup(),
							PanelExtra::colrec->name );

	m_treestore_all_tags = Gtk::TreeStore::create( *PanelExtra::colrec );

	// ALL TAGS
	this->set_model( m_treestore_all_tags );
    column_name_all->set_cell_data_func( *cellrenderer_icon,
                                         sigc::mem_fun( this, &PanelExtra::cell_data_func_icon ) );
    column_name_all->set_cell_data_func( *cellrenderer_name,
                                         sigc::mem_fun( this, &PanelExtra::cell_data_func_text ) );
	this->append_column( *column_name_all );
	this->enable_model_drag_source( Lifeobase::base->drag_targets_tag );
	this->override_background_color(
            Lifeobase::base->get_style_context()->get_background_color( Gtk::STATE_FLAG_NORMAL ),
            Gtk::STATE_FLAG_NORMAL );

	// SIGNALS
	this->signal_row_activated().connect(
			sigc::mem_fun( *this, &PanelExtra::handle_treeview_row_activated ) );

	this->signal_button_release_event().connect_notify(
			sigc::mem_fun( this, &PanelExtra::handle_treeview_button_release ) );

	this->signal_drag_begin().connect_notify(
			sigc::mem_fun( this, &PanelExtra::handle_treeview_drag_begin ) );

	this->signal_drag_data_get().connect_notify(
			sigc::mem_fun( this, &PanelExtra::handle_drag_data_get ) );

	this->signal_row_expanded().connect(
			sigc::mem_fun( this, &PanelExtra::handle_treeview_row_expanded ) );
	this->signal_row_collapsed().connect(
			sigc::mem_fun( this, &PanelExtra::handle_treeview_row_expanded ) );
}

void
PanelExtra::handle_logout( void )
{
    m_treestore_all_tags->clear();
}

void
PanelExtra::cell_data_func_icon( Gtk::CellRenderer *cell, const Gtk::TreeModel::iterator &iter )
{
    DiaryElement *elem( ( * iter )[ colrec->ptr ] );
    cell->set_property( "visible", elem->get_type() != DiaryElement::ET_HEADER );
}

void
PanelExtra::cell_data_func_text( Gtk::CellRenderer *cell, const Gtk::TreeModel::iterator &iter )
{
    DiaryElement *elem( ( * iter )[ colrec->ptr ] );

    switch( elem->get_type() )
    {
        case DiaryElement::ET_HEADER:
            //cell->set_property( "background-gdk", Gdk::Color( "#DDDDDD" ) );
            cell->set_property( "scale", 1.0 );
            cell->set_property( "ypad", 5 );
            break;
        default:
            //cell->set_property( "background-set", false );
            cell->set_property( "scale", 0.9 );
            cell->set_property( "ypad", 0 );
            break;
    }
}

void
PanelExtra::populate( void )
{
	Lifeobase::m_internaloperation++;

	m_treestore_all_tags->clear();

	Gtk::TreeRow row;

	// TAGS GROUP
    row = * m_treestore_all_tags->append();
    row[ colrec->ptr ] = &s_header;
    row[ colrec->name ] = Glib::ustring::compose( "<b>%1</b>", _( "TAGS" ) );

	if( Diary::d->get_tags()->size() < 1 && Diary::d->m_tag_categories.size() < 1 )
	{
		row = * m_treestore_all_tags->append();
		row[ colrec->ptr ] = &s_none;
		row[ colrec->name ] = Glib::ustring::compose( "<i>%1</i>", _( "None" ) );
	}

	// ROOT TAGS
	for( PoolTags::const_iterator iter( Diary::d->get_tags()->begin() );
		 iter != Diary::d->get_tags()->end();
		 ++iter )
	{
		Tag *tag( iter->second );
		if( tag->get_category() == NULL )
		    add_elem_to_list( tag );
	}

	// CATEGORIES
	for( PoolCategoriesTags::const_iterator iter_category( Diary::d->m_tag_categories.begin() );
		 iter_category != Diary::d->m_tag_categories.end();
		 ++iter_category )
	{
		CategoryTags *category = iter_category->second;
	    row = * add_elem_to_list( category );

		for( CategoryTags::const_iterator iter_tag = category->begin();
			 iter_tag != category->end();
			 ++iter_tag )
		{
	        add_elem_to_list( *iter_tag, &row.children() );
		}

		if( category->m_flag_expanded )
            this->expand_row( m_treestore_all_tags->get_path( row ), false );
	}

	// EMPTY SEPARATOR ROW
    row = * m_treestore_all_tags->append();
    row[ colrec->ptr ] = &s_none;

	// THEMES
    row = * m_treestore_all_tags->append();
    row[ colrec->ptr ] = &s_header;
    row[ colrec->name ] = Glib::ustring::compose( "<b>%1</b>", _( "THEMES" ) );

    for( PoolThemes::const_iterator iter = Diary::d->get_themes().begin();
         iter != Diary::d->get_themes().end();
         ++iter )
    {
        add_elem_to_list( iter->second/*, &row.children()*/ );
    }

    if( m_flag_themes_expanded )
        this->expand_row( m_treestore_all_tags->get_path( row ), false );

	Lifeobase::m_internaloperation--;
}

inline Gtk::TreeIter
PanelExtra::add_elem_to_list( DiaryElement *elem, const Gtk::TreeNodeChildren *children )
{
    Gtk::TreeRow row = * ( children != NULL ? m_treestore_all_tags->append( *children ) :
                                              m_treestore_all_tags->append() );

    row[ colrec->ptr ] = elem;
    row[ colrec->name ] = elem->get_list_str();
    row[ colrec->icon ] = elem->get_icon();

    elem->m_list_data->treepath = m_treestore_all_tags->get_path( row );

    return row;
}

Gtk::TreeRow
PanelExtra::get_row( const Gtk::TreePath &path )
{
	return( * m_treestore_all_tags->get_iter( path ) );
	// TODO: check validity
}

void
PanelExtra::handle_treeview_drag_begin( const Glib::RefPtr< Gdk::DragContext >& )
{
	m_flag_dragging = true;
	this->get_selection()->unselect_all();
}

void
PanelExtra::handle_drag_data_get( const Glib::RefPtr<Gdk::DragContext> &context,
								Gtk::SelectionData &data,
								guint info, guint time )
{
	if( m_elem_dragged )
	{
        switch( m_elem_dragged->get_type() )
        {
            case DiaryElement::ET_TAG:
                data.set( TARGET_ENTRY_TAG, m_elem_dragged->get_name() );
                break;
            case DiaryElement::ET_THEME:
                data.set( TARGET_ENTRY_THEME, m_elem_dragged->get_name() );
                break;
            default:
                break;
        }
	}
	else	// should never be the case
	{
		data.set( "N/A", "" );
		PRINT_DEBUG( "Nothing is selected at the drag source" );
	}
}

bool
PanelExtra::on_button_press_event( GdkEventButton *event )
{
	m_flag_dragging = false;
    m_elem_dragged = NULL;

	Gtk::TreePath path;
	if( this->get_path_at_pos( event->x, event->y, path ) )
	{
		Gtk::TreeRow row = * m_treestore_all_tags->get_iter( path );
		m_elem_dragged = row[ colrec->ptr ];
        if( m_elem_dragged == NULL )
            return true;    // ignore
        else
        {
            switch( m_elem_dragged->get_type() )
            {
                case DiaryElement::ET_TAG:
                case DiaryElement::ET_THEME:
                    break;
                case DiaryElement::ET_TAG_CTG:
                    m_elem_dragged = NULL; // dragging not supported...
                    break;
                default:
                    m_elem_dragged = NULL;
                    return true;
            }
        }
    }
    else
    {
        return true;
    }

    return Gtk::TreeView::on_button_press_event( event );
}

void
PanelExtra::handle_treeview_button_release( GdkEventButton *event )
{
	if( m_flag_dragging )
		return;

	Glib::RefPtr< Gtk::TreeSelection > selection = this->get_selection();
	if( selection->count_selected_rows() > 0 )
	{
		Gtk::TreeRow row = ( *selection->get_selected() );

		DiaryElement *elem = row[ colrec->ptr ];
		if( elem )	// if not NULL
			elem->show();
	}
	selection->unselect_all();
}

void
PanelExtra::handle_treeview_row_expanded( const Gtk::TreeIter &iter,
										const Gtk::TreePath &path )
{
	if( Lifeobase::m_internaloperation )
		return;

	DiaryElement *element = ( * iter )[ colrec->ptr ];
    if( element )
    {
        if( element->get_type() == DiaryElement::ET_TAG_CTG )
        {
            CategoryTags *category = dynamic_cast< CategoryTags* >( element );
            category->m_flag_expanded = this->row_expanded( path );
        }
	}
    else
        m_flag_themes_expanded = true;  // TODO: in the future there may be other expandable
                                        // items in the list than THEMES so a check is to be added
}

void
PanelExtra::handle_treeview_row_activated( const Gtk::TreePath &path, Gtk::TreeViewColumn* )
{
	Gtk::TreeRow row( * m_treestore_all_tags->get_iter( path ) );
	DiaryElement *element( row[ colrec->ptr ] );

	if( element == NULL )
	    return;

	if( element->get_type() == DiaryElement::ET_TAG )
	{
		Tag *tag( dynamic_cast< Tag* >( element ) );

		if( tag == Diary::d->get_filter_tag() )	// remove filter
		{
			Diary::d->set_filter_tag( NULL );
			row[ colrec->icon ] = Lifeobase::icons->tag_16;
		}
		else
		{
			if( Diary::d->get_filter_tag() != NULL )
			{
				Gtk::TreeRow row_old( * m_treestore_all_tags->get_iter(
						Diary::d->get_filter_tag()->m_list_data->treepath ) );
				row_old[ colrec->icon ] = Lifeobase::icons->tag_16;
			}
			Diary::d->set_filter_tag( tag );
			row[ colrec->icon ] = Lifeobase::icons->filter_16;
		}
        Lifeobase::view_tag->update_button_filter_tag();
        Lifeobase::panel_diary->update_button_remove_filters();
        Lifeobase::panel_diary->update_entry_list();
	}
}
