/***********************************************************************************

	Copyright (C) 2007-2011 Ahmet Öztürk (aoz_2@yahoo.com)

	This file is part of Lifeograph.

	Lifeograph is free software: you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation, either version 3 of the License, or
	(at your option) any later version.

	Lifeograph is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY; without even the implied warranty of
	MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
	GNU General Public License for more details.

	You should have received a copy of the GNU General Public License
	along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

***********************************************************************************/


#include "lifeobase.hpp"
#include "helpers.hpp"

using namespace LIFEO;

// STATIC MEMBERS
Lifeobase							*Lifeobase::base;
Settings							Lifeobase::settings;
Icons								*Lifeobase::icons = NULL;
Glib::RefPtr< Gtk::Builder >		Lifeobase::builder;
Glib::RefPtr< Gtk::ActionGroup >	Lifeobase::m_actions_read;
Glib::RefPtr< Gtk::ActionGroup >	Lifeobase::m_actions_edit;
int									Lifeobase::m_internaloperation = 0;
int									Lifeobase::m_entrycount = 0;
sigc::connection					Lifeobase::m_connection_timeout;
SignalVoid							Lifeobase::m_signal_logout;
Lifeobase::LoginStatus              Lifeobase::loginstatus = Lifeobase::LOGGED_OUT;
std::set< std::string >             Lifeobase::stock_diaries;
Lifeobase::MapPixbufs               Lifeobase::map_thumbnails;
LanguageList                        Lifeobase::s_lang_list;
EnchantBroker                       *Lifeobase::s_enchant_broker = NULL;


Lifeobase::Lifeobase()
:   m_flag_open_directly( false ), m_flag_read_only( false )
{
    base = this;
    m_actions_read = Gtk::ActionGroup::create( "read actions" );
    m_actions_edit = Gtk::ActionGroup::create( "edit actions" );

    m_actions_read->set_sensitive( false );
    m_actions_edit->set_sensitive( false );

    icons = new Icons;

    drag_targets_tag.push_back( Gtk::TargetEntry( TARGET_ENTRY_TAG,
                                                  Gtk::TARGET_SAME_APP,
                                                  DRAG_TARGET_TAG_INFO ) );
    drag_targets_entry.push_back( Gtk::TargetEntry( TARGET_ENTRY_ENTRY,
                                                    Gtk::TARGET_SAME_APP,
                                                    DRAG_TARGET_ENTRY_INFO ) );
}

bool
Lifeobase::load_gui( const std::string &path )
{
	builder = Gtk::Builder::create();
	try
	{
		builder->add_from_file( path );
	}
	catch( const Glib::FileError& ex )
	{
		std::cerr << "FileError: " << ex.what() << std::endl;
		return false;
	}
	catch( const Gtk::BuilderError& ex )
	{
		std::cerr << "BuilderError: " << ex.what() << std::endl;
		return false;
	}

	return true;
}

void
Lifeobase::create_action( bool                          flag_edit,
                          Glib::RefPtr< Gtk::Action >   &action,
                          const Glib::ustring           &name,
                          const Gtk::StockID            &stockid,
                          const Glib::ustring           &label,
                          const Glib::ustring           &tooltip,
                          const Gtk::AccelKey           &accelkey,
                          const Glib::SignalProxy0< void >::SlotType
                                                        &slot,
                          Gtk::Activatable              *widget_proxy )
{
    action  = Gtk::Action::create( name, stockid, label, tooltip );
    if( flag_edit )
        m_actions_edit->add( action, accelkey );
    else
        m_actions_read->add( action, accelkey );
    action->set_accel_group( base->get_accel_group() );
    action->connect_accelerator();
    if( widget_proxy != NULL )
        widget_proxy->set_related_action( action );
    action->signal_activate().connect( slot );
}

void
Lifeobase::create_action( bool                          flag_edit,
                          Glib::RefPtr< Gtk::Action >   &action,
                          const Glib::ustring           &name,
                          const Glib::ustring           &icon_name,
                          const Glib::ustring           &label,
                          const Glib::ustring           &tooltip,
                          const Gtk::AccelKey           &accelkey,
                          const Glib::SignalProxy0< void >::SlotType
                                                        &slot,
                          Gtk::Activatable              *widget_proxy )
{
    if( icon_name.empty() )
        action = Gtk::Action::create( name, label, tooltip );
    else
        action  = Gtk::Action::create_with_icon_name( name, icon_name, label, tooltip );

    if( flag_edit )
        m_actions_edit->add( action, accelkey );
    else
        m_actions_read->add( action, accelkey );
    action->set_accel_group( base->get_accel_group() );
    action->connect_accelerator();
    if( widget_proxy != NULL )
        widget_proxy->set_related_action( action );
    action->signal_activate().connect( slot );
}

void
Lifeobase::create_toggle_action( bool                              flag_edit,
                                 Glib::RefPtr< Gtk::ToggleAction > &action,
                                 const Glib::ustring               &name,
                                 const Glib::ustring               &icon_name,
                                 const Glib::ustring               &label,
                                 const Glib::ustring               &tooltip,
                                 const Gtk::AccelKey               &accelkey,
                                 const Glib::SignalProxy0< void >::SlotType
                                                                   &slot,
                                 Gtk::Activatable                  *widget_proxy )
{
    if( icon_name.empty() )
        action = Gtk::ToggleAction::create( name, label, tooltip );
    else
        action  = Gtk::ToggleAction::create_with_icon_name( name, icon_name, label, tooltip );

    if( flag_edit )
        m_actions_edit->add( action, accelkey );
    else
        m_actions_read->add( action, accelkey );
    action->set_accel_group( base->get_accel_group() );
    action->connect_accelerator();
    if( widget_proxy != NULL )
        widget_proxy->set_related_action( action );
    action->signal_activate().connect( slot );
}

Glib::RefPtr< Gdk::Pixbuf >
Lifeobase::get_thumbnail( const std::string &path )
{
    MapPixbufs::iterator iter( map_thumbnails.find( path ) );
    if( iter == map_thumbnails.end() )
    {
        Glib::RefPtr< Gdk::Pixbuf > buf( Gdk::Pixbuf::create_from_file( path ) );
        if( buf->get_width() > MAX_THUMBNAIL_WIDTH )
            buf = buf->scale_simple( MAX_THUMBNAIL_WIDTH,
                                     buf->get_height() * MAX_THUMBNAIL_WIDTH / buf->get_width(),
                                     Gdk::INTERP_BILINEAR );
        map_thumbnails[ path ] = buf;
        return buf;
    }
    else
        return iter->second;
}

// DIALOGEVENT =====================================================================================
HELPERS::DialogEvent::DialogEvent( const Glib::ustring &title )
:   Gtk::Dialog( title )
{
    set_transient_for( *Lifeobase::base );
    Lifeobase::signal_logout().connect( sigc::mem_fun( this, &DialogEvent::handle_logout ) );
}

HELPERS::DialogEvent::DialogEvent( BaseObjectType *cobject,
                          const Glib::RefPtr< Gtk::Builder >& )
:   Gtk::Dialog( cobject )
{
    set_transient_for( *Lifeobase::base );
    //Lifeobase::signal_logout().connect( sigc::mem_fun( this, &DialogEvent::handle_logout ) );
}

bool
HELPERS::DialogEvent::on_event( GdkEvent *event )
{
    return Lifeobase::base->on_event( event );
}

