/***********************************************************************************

 Copyright (C) 2007-2012 Ahmet Öztürk (aoz_2@yahoo.com)

 This file is part of Lifeograph.

 Lifeograph is free software: you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation, either version 3 of the License, or
 (at your option) any later version.

 Lifeograph is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with Lifeograph.  If not, see <http://www.gnu.org/licenses/>.

 ***********************************************************************************/

#include <gtkmm/table.h>

#include "lifeobase.hpp"
#include "dialog_export.hpp"
#include "dialog_password.hpp"

using namespace LIFEO;

// DIALOGSAVEDIARY =================================================================================
DialogSaveDiary::DialogSaveDiary( void )
: Gtk::FileChooserDialog(
        // TRANSLATORS: this is the title of file chooser dialog for saving a new diary
        _( "Where to Save The Diary?" ), Gtk::FILE_CHOOSER_ACTION_SAVE )
{
    add_button( Gtk::Stock::CANCEL, Gtk::RESPONSE_CANCEL );
    add_button( Gtk::Stock::OK, Gtk::RESPONSE_ACCEPT );
    set_do_overwrite_confirmation( true );
}

std::string
DialogSaveDiary::get_filename( const std::string &extension )
{
    std::string fname( FileChooserDialog::get_filename() );

    if( ! extension.empty() && extension != "." )
        if( ! str_ends_with( fname, extension ) )
            fname.append( extension );

    return fname;
}

std::string
DialogSaveDiary::get_filename_default( void ) // appends default diary extension
{
    return get_filename( Lifeobase::settings.diary_extension );
}

// DIALOGOPENDIARY =================================================================================
bool DialogOpenDiary::flag_filters_created = false;
Glib::RefPtr< Gtk::FileFilter > DialogOpenDiary::filter_any;
Glib::RefPtr< Gtk::FileFilter > DialogOpenDiary::filter_diary;

DialogOpenDiary::DialogOpenDiary( void )
    // TRANSLATORS: this is the title of file chooser dialog for selecting a new diary
:   Gtk::FileChooserDialog( _( "Select a Diary" ), Gtk::FILE_CHOOSER_ACTION_OPEN )
{
    add_button( Gtk::Stock::CANCEL, RESPONSE_CANCEL );
    add_button( _( "Read" ), RESPONSE_READ );
    add_button( _( "Edit" ), RESPONSE_EDIT );

    init_filters();
    add_filter( filter_any );
    add_filter( filter_diary );
    set_filter( filter_diary );

    signal_selection_changed().connect(
            sigc::mem_fun( this, &DialogOpenDiary::handle_selection_changed ) );

    set_default_response( RESPONSE_EDIT );
}

void
DialogOpenDiary::init_filters( void )
{
    if( flag_filters_created )
        return;

    filter_any = Gtk::FileFilter::create();
    filter_diary = Gtk::FileFilter::create();

    filter_any->set_name( _( "All Files" ) );
    filter_any->add_pattern( "*" );
    filter_diary->set_name( _( "Diary Files" ) );
    filter_diary->add_mime_type( "application/x-lifeographdiary" );

    flag_filters_created = true;
}

void
DialogOpenDiary::handle_selection_changed( void )
{
    bool flag_enable( get_filename().size() > 0 );  // check if anything is selected
    if( flag_enable )
        flag_enable = ( Glib::file_test( get_filename(), Glib::FILE_TEST_IS_DIR ) == false );

    set_response_sensitive( RESPONSE_READ, flag_enable );
    set_response_sensitive( RESPONSE_EDIT, flag_enable );
}

// DIALOG IMPORT ===================================================================================
DialogImport::DialogImport( BaseObjectType* cobject,
                            const Glib::RefPtr< Gtk::Builder > &refbuilder )
:   DialogEvent( cobject, refbuilder ), m_dialog_password( NULL ), m_diary( NULL )
{
    Lifeobase::builder->get_widget( "filebutton_import_diary",
            m_file_chooser_button );
    Lifeobase::builder->get_widget( "label_import_info", m_label_info );
    Lifeobase::builder->get_widget( "label_import_entries", m_label_entries );
    Lifeobase::builder->get_widget( "label_import_tags", m_label_tags );
    Lifeobase::builder->get_widget( "label_import_themes", m_label_themes );
    Lifeobase::builder->get_widget_derived( "entry_import_add_tag",
            m_tag_widget );

    Lifeobase::builder->get_widget( "checkbutton_import_entries",
            m_checkbutton_import_entries );
    Lifeobase::builder->get_widget( "checkbutton_import_tags",
            m_checkbutton_import_tags );
    Lifeobase::builder->get_widget( "checkbutton_import_themes",
            m_checkbutton_import_themes );

    // FILTERING
    DialogOpenDiary::init_filters();
    m_file_chooser_button->add_filter( DialogOpenDiary::filter_any );
    m_file_chooser_button->add_filter( DialogOpenDiary::filter_diary );
    m_file_chooser_button->set_filter( DialogOpenDiary::filter_diary );

    // SIGNALS
    m_file_chooser_button->signal_file_set().connect(
            sigc::mem_fun( this, &DialogImport::handle_diary_changed ) );

    m_checkbutton_import_entries->signal_toggled().connect(
            sigc::mem_fun( this, &DialogImport::handle_parts_changed ) );
    m_checkbutton_import_tags->signal_toggled().connect(
            sigc::mem_fun( this, &DialogImport::handle_parts_changed ) );
    m_checkbutton_import_themes->signal_toggled().connect(
            sigc::mem_fun( this, &DialogImport::handle_parts_changed ) );
}

inline bool
DialogImport::is_options_ready( void )
{
    return( m_option_import_entries || m_option_import_tags || m_option_import_themes );
}

void
DialogImport::on_show( void )
{
    Gtk::Dialog::on_show();
    set_not_ready( _( "Select a diary file for importing" ) );

    m_checkbutton_import_entries->set_active( true );
    m_checkbutton_import_tags->set_active( true );
    m_checkbutton_import_themes->set_active( true );

    m_tag_widget->populate();

    m_file_chooser_button->grab_focus();
}

void
DialogImport::handle_diary_changed( void )
{
    if( m_diary != NULL )
        delete m_diary;

    m_diary = new Diary;

    m_diary->set_path( m_file_chooser_button->get_filename(), false, true );

    if( m_diary->read_header() == SUCCESS )
    {
        if( m_diary->is_encrypted() )
        {
            if( m_dialog_password == NULL )
                Lifeobase::builder->get_widget_derived(
                        "dialog_import_password", m_dialog_password );
            m_dialog_password->set_transient_for( *this );
            m_dialog_password->set_path( m_diary->get_name() );
            if( m_dialog_password->run() == RESPONSE_GO )
                m_diary->set_passphrase(
                        m_dialog_password->m_entry->get_text() );
            m_dialog_password->hide();
        }

        Result result = m_diary->read_body();
        if( result == SUCCESS )
        {
            set_ready();
        }
        else if( result == WRONG_PASSWORD )
        {
            set_not_ready( _( "Entered password is wrong; please try again" ) );
        }
        else
        {
            set_not_ready( _( "Selected file cannot be read" ) );
        }
    }
    else
    {
        set_not_ready( _( "Selected file does not seem to be a valid diary" ) );
    }
}

void
DialogImport::handle_parts_changed( void )
{
    m_option_import_entries = m_checkbutton_import_entries->get_active();
    m_option_import_tags = m_checkbutton_import_tags->get_active();
    m_option_import_themes = m_checkbutton_import_themes->get_active();

    set_response_sensitive( RESPONSE_GO,
            m_flag_diary_is_ready && is_options_ready() );
}

void
DialogImport::set_ready( void )
{
    set_response_sensitive( RESPONSE_GO, is_options_ready() );

    m_label_entries->set_text(
            Glib::ustring::compose( "%1", m_diary->get_size() ) );
    m_label_tags->set_text(
            Glib::ustring::compose( "%1", m_diary->get_tags()->size() ) );
    m_label_themes->set_text(
            Glib::ustring::compose( "%1", m_diary->m_themes.size() - 1 ) );

    m_label_info->set_text( _( "Diary is ready for importing" ) );
    m_flag_diary_is_ready = true;
}

void
DialogImport::set_not_ready( const Glib::ustring &message )
{
    set_response_sensitive( RESPONSE_GO, false );

    m_label_entries->set_text( "-" );
    m_label_tags->set_text( "-" );
    m_label_themes->set_text( "-" );

    if( m_diary )
    {
        delete m_diary;
        m_diary = NULL;
    }

    m_label_info->set_text( message );
    m_flag_diary_is_ready = false;
}

void
DialogImport::on_response( int response )
{
    if( response == RESPONSE_GO )
    {
        if( m_option_import_tags )
        {
            for( PoolTags::const_iterator iter = m_diary->m_tags.begin();
                    iter != m_diary->m_tags.end(); ++iter )
            {
                Diary::d->import_tag( (*iter).second );
            }
        }

        if( m_option_import_themes )
        {
            Diary::d->import_themes( m_diary->m_themes );
        }

        if( m_option_import_entries )
        {
            Diary::d->import_entryset( m_diary->m_entries, m_option_import_tags,
                    m_tag_widget->get_text() );
        }
    }

    delete m_diary;
    m_diary = NULL;

    Gtk::Dialog::on_response( response );
}

// DIALOGEXPORT ====================================================================================
DialogExport::DialogExport( BaseObjectType* cobject,
        const Glib::RefPtr< Gtk::Builder > &refbuilder )
:   DialogEvent( cobject, refbuilder ), m_ptr2widget_options( NULL ), m_ptr2diary( Diary::d )
{
    Lifeobase::builder->get_widget( "combo_export_type", m_combo_type );
    Lifeobase::builder->get_widget( "combo_export_extent", m_combo_extent );
    Lifeobase::builder->get_widget( "frame_export_options", m_frame_options );
    Lifeobase::builder->get_widget( "align_export_options", m_align_options );
    Lifeobase::builder->get_widget( "button_export_go", m_button_export );

    add_saver( new ExporterLifeograph( m_ptr2diary ) );
    add_saver( new ExporterText( m_ptr2diary ) );

    m_combo_extent->append( _( "All Entries" ) );
    m_combo_extent->append( _( "Only Filtered Entries" ) );
    m_combo_extent->set_active( 0 );

    m_combo_type->signal_changed().connect(
            sigc::mem_fun( this, &DialogExport::handle_savetype_changed ) );

    m_combo_type->set_active( 0 );    // defaults to lifeograph diary

    show_all();
}

void
DialogExport::add_saver( Diarysaver *saver )
{
    m_savers.push_back( saver );
    m_combo_type->append( saver->get_name() );
    saver->signal_updated().connect(
            sigc::mem_fun( this, &DialogExport::handle_saver_updated ) );
}

void
DialogExport::handle_saver_updated( bool ready )
{
    m_flag_saverready = ready;
    m_button_export->set_sensitive( ready );
}

void
DialogExport::handle_savetype_changed( void )
{
    m_saver_cur = m_savers[ m_combo_type->get_active_row_number() ];

    if( m_ptr2widget_options )
    {
        m_align_options->remove();
        delete m_ptr2widget_options;
    }

    m_ptr2widget_options = m_saver_cur->draw_options();

    if( m_ptr2widget_options )
    {
        m_align_options->add( *m_ptr2widget_options );
        m_frame_options->show();
    }
    else
        m_frame_options->hide();

    m_flag_saverready = m_saver_cur->is_ready();

    m_button_export->set_sensitive( m_flag_saverready );
}

void
DialogExport::on_response( int response )
{
    if( response == RESPONSE_GO )
    {
        DialogSaveDiary *ds( new DialogSaveDiary );
        ds->set_current_folder( Glib::path_get_dirname( m_ptr2diary->get_path() ) );
        ds->set_current_name( m_ptr2diary->get_name() + ".backup" );

        if( ds->run() == Gtk::RESPONSE_ACCEPT )
            m_saver_cur->save( ds->get_filename(), m_combo_extent->get_active() );

        delete ds;
    }

    Gtk::Dialog::on_response( response );
}

// EXPORTERS =======================================================================================
Diarysaver::Diarysaver( const Glib::ustring& name, Diary *diary )
        : m_name( name ), m_ptr2diary( diary )
{
}

ExporterLifeograph::ExporterLifeograph( Diary *diary )
:   Diarysaver( _( "Lifeograph Diary File" ), diary ), m_passphrase( "" ),
    m_flag_encrypted( false )
{
}

Gtk::Widget*
ExporterLifeograph::draw_options( void )
{
    m_entry_passphrase = Gtk::manage( new Gtk::Entry );
    m_entry_confirm = Gtk::manage( new Gtk::Entry );
    m_check_encrypt = Gtk::manage( new Gtk::CheckButton( _( "Encrypt Diary Content" ) ) );
    Gtk::Table *table = new Gtk::Table;

    m_entry_passphrase->set_visibility( false );
    m_entry_confirm->set_visibility( false );

    table->set_border_width( 5 );
    table->set_row_spacings( 2 );
    table->set_col_spacings( 15 );
    table->attach( *m_check_encrypt, 0, 2, 0, 1, Gtk::FILL, Gtk::SHRINK );
    table->attach( *Gtk::manage( new Gtk::Label( _( "Password:" ) ) ), 0, 1, 1, 2, Gtk::SHRINK );
    table->attach( *m_entry_passphrase, 1, 2, 1, 2, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );
    // TRANSLATORS: confirm here means re-enter password
    table->attach( *Gtk::manage( new Gtk::Label( _( "Confirm:" ) ) ), 0, 1, 2, 3, Gtk::SHRINK );
    table->attach( *m_entry_confirm, 1, 2, 2, 3, Gtk::EXPAND | Gtk::FILL, Gtk::SHRINK );

    // TODO: add a match indicator icon

    m_entry_passphrase->set_sensitive( false );
    m_entry_confirm->set_sensitive( false );

    m_entry_passphrase->signal_changed().connect(
            sigc::mem_fun( this, &ExporterLifeograph::handle_passphrase_changed ) );
    m_entry_confirm->signal_changed().connect(
            sigc::mem_fun( this, &ExporterLifeograph::handle_passphrase_changed ) );
    m_check_encrypt->signal_toggled().connect(
            sigc::mem_fun( this, &ExporterLifeograph::handle_encryption_changed ) );

    table->show_all();

    return table;
}

bool
ExporterLifeograph::is_ready( void )
{
    return( ( ! m_flag_encrypted ) || m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE );
}

void
ExporterLifeograph::handle_passphrase_changed( void )
{
    m_passphrase = m_entry_passphrase->get_text();
    if( m_passphrase != m_entry_confirm->get_text() )
        m_passphrase.clear();
    else
        if( m_passphrase.size() < LIFEO::PASSPHRASE_MIN_SIZE )
            m_passphrase.clear();

    m_signal_updated.emit( m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE );
}

void
ExporterLifeograph::handle_encryption_changed( void )
{
    m_flag_encrypted = m_check_encrypt->get_active();
    m_entry_passphrase->set_sensitive( m_flag_encrypted );
    m_entry_confirm->set_sensitive( m_flag_encrypted );

    m_signal_updated.emit( ( ! m_flag_encrypted ) ||
                           m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE );
}

Result
ExporterLifeograph::save( const std::string &path, bool filtered )
{
    if( m_flag_encrypted && m_passphrase.size() >= LIFEO::PASSPHRASE_MIN_SIZE )
        return m_ptr2diary->write_copy( path, m_passphrase, filtered );
    else
        return m_ptr2diary->write_copy( path, "", filtered );
}

ExporterText::ExporterText( Diary *diary )
        : Diarysaver( _( "Plain Text File" ), diary )
{

}

Result
ExporterText::save( const std::string &path, bool filtered )
{
    return m_ptr2diary->write_txt( path, filtered );
}
