package latexDraw.ui.dialog;

import java.awt.*;
import java.awt.event.ActionEvent;
import java.awt.event.ActionListener;
import java.io.*;
import java.util.Vector;

import javax.swing.*;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.DocumentBuilderFactory;

import latexDraw.lang.LaTeXDrawLang;
import latexDraw.ui.CodePanel;
import latexDraw.ui.DrawContainer;
import latexDraw.ui.LaTeXDrawFrame;
import latexDraw.ui.components.Delimitor;
import latexDraw.ui.components.MagneticGrid;
import latexDraw.ui.components.XScale;
import latexDraw.util.*;

import org.w3c.dom.*;

import com.sun.org.apache.xml.internal.serialize.OutputFormat;
import com.sun.org.apache.xml.internal.serialize.XMLSerializer;


/**
 * This frame allows the user to set his preferences concerning the program and its components.<br>
 * <br>
 * This file is part of LaTeXDraw.<br>
 * Copyright (c) 2005-2008 Arnaud BLOUIN<br>
 *<br>
 *  LaTeXDraw is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.<br>
 *<br>
 *  LaTeXDraw is distributed without any warranty; without even the 
 *  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR 
 *  PURPOSE. See the GNU General Public License for more details.<br>
 *<br>
 * 09/14/06<br>
 * @author Arnaud BLOUIN<br>
 * @version 2.0.0<br>
 */
public class PreferencesFrame extends JFrame implements ActionListener
{
	private static final long serialVersionUID = 1L;

	/** The current drawing */
	private DrawContainer draw;
	
	/** The main frame of the program */ 
	private LaTeXDrawFrame mainFrame;
		
	/** The file chooser of paths selection */
	private JFileChooser fileChooser;
	
	/** The title of the frame */
	public static final String TITLE = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.Pref"); //$NON-NLS-1$
	
	/** The name of the button ok */
	public static final String NAME_BUTTON_OK = "BUTTON_OK";//$NON-NLS-1$
	
	/** The name of the button cancel */
	public static final String NAME_BUTTON_CANCEL = "BUTTON_CANCEL";//$NON-NLS-1$
	
	/** The name of the button default */
	public static final String NAME_BUTTON_DEFAULT = "BUTTON_DEFAULT";//$NON-NLS-1$
	
	/** The label of the X-Scale checkBox */
	public static final String LABEL_CHECKBOX_XSCALE = LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.38"); //$NON-NLS-1$
	
	/** The label of the Y-Scale checkBox */
	public static final String LABEL_CHECKBOX_YSCALE = LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.39"); //$NON-NLS-1$
	
	/** The label of the Grid checkBox */
	public static final String LABEL_CHECKBOX_GRID = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.grid"); //$NON-NLS-1$
	
	/** The label of the codePanel checkBox */
	public static final String LABEL_CHECKBOX_CODEPANEL = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.codePanel"); //$NON-NLS-1$
	
	/** The label of the anti-aliasing checkBox */
	public static final String LABEL_CHECKBOX_ANTIALIASING = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.antiAl"); //$NON-NLS-1$
	
	/** The label of the rendering checkBox */
	public static final String LABEL_CHECKBOX_RENDERING = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.rendQ"); //$NON-NLS-1$
	
	/** The label of the colour rendering checkBox */
	public static final String LABEL_CHECKBOX_COLORRENDER = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.colRendQ"); //$NON-NLS-1$
	
	/** The label of the alpha-interpolation checkBox */
	public static final String LABEL_CHECKBOX_ALPHAINTER = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.AlphaQ"); //$NON-NLS-1$
	
	/** The label of the drawBorders checkBox */
	public static final String LABEL_CHECKBOX_DISPLAY_BORDERS = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.bordersDraw"); //$NON-NLS-1$
	
	/** The name of the button which allows to select a directory for the commands open/save.*/
	public static final String NAME_BUTTON_OPENFIELD = "BUTTON_OPENFIELD";//$NON-NLS-1$
	
	/** The name of the button which allows t select a directory for the exportation of the code. */
	public static final String NAME_BUTTON_EXPORTFIELD = "BUTTON_EXPORTFIELD";//$NON-NLS-1$

	public static final String NAME_BUTTON_TEX_EDITOR = "BUTTON_TEX_EDITOR";//$NON-NLS-1$

	public static final String NAME_BUTTON_LATEX_DISTRIB = "BUTTON_LATEX";//$NON-NLS-1$

	/** The label of the check-box code auto-update */
	public static final String LABEL_AUTOUPDATE = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.codeAuto"); //$NON-NLS-1$
	
	/** The label of the Check-box checkNewVersion */
	public static final String LABEL_CHECK_VERSION = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.newVers"); //$NON-NLS-1$
	
	public static final String LABEL_LANG = LaTeXDrawLang.getDialogFrameString("PreferencesFrame.lge"); //$NON-NLS-1$
	
	public static final int NB_RECENT_FILES = 5;
	
	/** The label of the preferences */
	public static final String LABEL_PREFERENCES = "TEX_DRAW_PREFERENCES";//$NON-NLS-1$
	
	/** This check-box allows to set if antialiasing must be used */
	private JCheckBox antialiasingCheckBox;
	
	/** This check-box allows to set if rendering quality must be used */
	private JCheckBox renderingCheckBox;
	
	/** This check-box allows to set if colour rendering quality must be used */
	private JCheckBox colorRenderCheckBox;
	
	/** This check-box allows to set if the user wants to display the grid */
	private JCheckBox displayGridCB;
	
	private JCheckBox magneticGridCB;
	
	/** This check-box allows to set if the user wants to display the X-scale */
	private JCheckBox displayXScaleCB;
	
	/** This check-box allows to set if the user wants to display the Y-scale */
	private JCheckBox displayYScaleCB;
	
	/** Allows to update or not in real time the PSTricks code */
	private JCheckBox codeAutoUpdateCB;
	
	/** Allows the set if the program must check new version on start up */
	private JCheckBox checkNewVersion;
	
	/** This check-box allows to set if the user wants to display the codepanel */
	private JCheckBox displayCodePanelCB;
	
	/** This check-box allows to set if the user wants to the borders
	 * of the drawing */
	private JCheckBox displayBordersCB;
	
	/** This check-box allows to set if alpha-interpolation must be used */
	private JCheckBox alpaInterCheckBox;
	
	/** This textField allows to set the default directories for open/save
	 * actions */
	private JTextField pathOpenField;
	
	/** This textField allows to set the default directories for exportations
	 * actions */
	private JTextField pathExportField;

	private JTextField pathTexEditorField;
	
	private JTextField pathLatexDistribField;
	
	private JTextArea latexIncludes;
	
	/** Allows to set the unit of length by default */
	private JComboBox unitChoice;
	
	private JComboBox themeList;
	
	private JComboBox langList;
	
	private JRadioButton classicGridRB;

	private JRadioButton persoGridRB;
	
	private JSpinner persoGridGapField;
	
	private JSpinner nbRecentFilesField;

	private Vector<String> recentFilesName;
	
	/** Define if the main frame is maximised or not. @since 2.0.0 */
	private boolean isFrameMaximized;
	
	/** Define the size of the main frame. @since 2.0.0 */
	private Dimension frameSize;
	
	/** Define the position of the main frame. @since 2.0.0 */
	private Point framePosition;
	
	/** Define the position of the divider. @since 2.0.0 */
	private double dividerPosition = 1;
	
	/* These attributes are used to be able to restore the fields value when cancelling modifications. */
	private int curNbRecentFiles;
	private int curGridGap;
	private boolean curClassicGrid;
	private String curLang;
	private String curTheme;
	private String curUnit;
	private String curTexPath;
	private String curExportPath;
	private String curOpenPath;
	private boolean curAlpha;
	private boolean curDisplayBord;
	private boolean curDisplayCode;
	private boolean curCheckVers;
	private boolean curAutoUpdte;
	private boolean curYScale;
	private boolean curXScale;
	private boolean curDisplayGrid;
	private boolean curMagnetic;
	private boolean curColRender;
	private boolean curRendering;
	private boolean curAntiAlias;
	private String curLatexDistrib;
	private String curLatexIncludes;
	

	
	/**
	 * The constructor using a frame.
	 * @param d The current drawing.
	 * @param frame The LaTeXDrawFrame (used to set the preferences).
	 */
	public PreferencesFrame(DrawContainer d, LaTeXDrawFrame frame) 
	{
		try
		{
			draw = d;
			mainFrame = frame;
	  		Dimension dim = Toolkit.getDefaultToolkit().getScreenSize();

	  		setIconImage(LaTeXDrawResources.propertiesIcon.getImage());
		  		
		  	setTitle(TITLE);
	  		setLocation(dim.width*1/3, dim.height*1/4);
	  		
	  		recentFilesName = new Vector<String>();
	  		JTabbedPane tabbedPane = new JTabbedPane();
	  		JPanel pButton  = new JPanel(new BorderLayout());
	  		JPanel pGeneral = new JPanel(new BorderLayout());
	  		
	  		tabbedPane.add(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.general"), createGeneralPanel()); //$NON-NLS-1$
	  		tabbedPane.add(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.folders"), createFoldersPanel()); //$NON-NLS-1$
	  		tabbedPane.add(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.quality"), createQualityPanel()); //$NON-NLS-1$
	  		tabbedPane.add(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.90"), createDisplayPanel()); //$NON-NLS-1$
	  		tabbedPane.add(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.92"), createCodePanel()); //$NON-NLS-1$
	  		tabbedPane.add(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.grid"), createGridPanel()); //$NON-NLS-1$
	  		tabbedPane.add("LaTeX", createLatexPanel()); //$NON-NLS-1$
	  		
	  		JButton buttonOk  = new JButton(AbstractParametersFrame.LABEL_OK), 
				buttonCancel  = new JButton(AbstractParametersFrame.LABEL_CANCEL),
				buttonDefault = new JButton(AbstractParametersFrame.LABEL_DEFAULT);  		
	  		GridLayout  gridButton = new GridLayout(1, 3);
	  		
	  		buttonOk.setActionCommand(NAME_BUTTON_OK);
	  		buttonCancel.setActionCommand(NAME_BUTTON_CANCEL);
	  		buttonDefault.setActionCommand(NAME_BUTTON_DEFAULT);
	  		buttonOk.addActionListener(this);
			buttonCancel.addActionListener(this);
			buttonDefault.addActionListener(this);
			
			pButton.setLayout(gridButton);
			pButton.add(buttonOk);
			pButton.add(buttonCancel);
			pButton.add(buttonDefault);
			pGeneral.add(tabbedPane, BorderLayout.CENTER);
			pGeneral.add(pButton, BorderLayout.SOUTH);
	  		
			getContentPane().add(pGeneral);
	  		setSize(480, 310);//Fixes #1562954
			setVisible(false);
			
			readXMLPreferences();
			
			fileChooser = new JFileChooser();
			fileChooser.setApproveButtonText(LaTeXDrawLang.getStringLaTeXDrawFrame("LaTeXDrawFrame.171"));	//$NON-NLS-1$
			fileChooser.setDialogTitle(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.selectFolder"));	//$NON-NLS-1$
			fileChooser.setMultiSelectionEnabled(false);
			
		}catch(Exception e)
		{
			e.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(e);
		}
	}
	
	
	
	
	/**
	 * Allows to create the panel which allows the user to set the general preferences of latexdraw.
	 * @return The panel.
	 */	
	private JPanel createGeneralPanel()
	{
		JPanel pGeneral = new JPanel();
		JPanel pLang 	= new JPanel();
		JPanel pEditor 	= new JPanel();
  		JPanel pRecent 	= new JPanel();
  		JPanel pTheme	= new JPanel();
	
  		pTheme.setLayout(new BoxLayout(pTheme, BoxLayout.X_AXIS));
		pEditor.setLayout(new BoxLayout(pEditor, BoxLayout.X_AXIS));
		pLang.setLayout(new GridLayout(1, 2));
  		pGeneral.setLayout(new GridLayout(6, 1));
  		pRecent.setLayout(new FlowLayout(FlowLayout.LEFT));
  		
  		JButton bChooseEditor = new JButton(LaTeXDrawResources.openIcon);
  		bChooseEditor.addActionListener(this);
  		bChooseEditor.setActionCommand(NAME_BUTTON_TEX_EDITOR);
  		pathTexEditorField = new JTextField();
  		pathTexEditorField.setMaximumSize(new Dimension(300,30));
  		checkNewVersion = new JCheckBox(LABEL_CHECK_VERSION);
  		langList = new JComboBox();
  		langList.addItem(LaTeXDrawLang.LANG_DE);
  		langList.addItem(LaTeXDrawLang.LANG_EN);
  		langList.addItem(LaTeXDrawLang.LANG_EN_US);
  		langList.addItem(LaTeXDrawLang.LANG_ES);
  		langList.addItem(LaTeXDrawLang.LANG_FR);
  		langList.addItem(LaTeXDrawLang.LANG_IT);
  		langList.addItem(LaTeXDrawLang.LANG_PL);
  		langList.setSelectedItem(LaTeXDrawLang.getlangCurrent());
  		langList.setPreferredSize(new Dimension(75, 30));
  		langList.setSize(new Dimension(75, 30));
  		langList.setMaximumSize(new Dimension(75, 30));
  		langList.setMinimumSize(new Dimension(75, 30));
     	
  		SpinnerModel model = new SpinnerNumberModel(NB_RECENT_FILES,0,20,1);
  		nbRecentFilesField = new JSpinner(model);
  		nbRecentFilesField.setEditor(new JSpinner.NumberEditor(nbRecentFilesField, "0"));//$NON-NLS-1$
  		nbRecentFilesField.setPreferredSize(new Dimension(45,25));
  		nbRecentFilesField.setMaximumSize(new Dimension(45,25));
  		pRecent.add(new JLabel(LaTeXDrawLang.getString1_9("PreferencesFrame.0"))); //$NON-NLS-1$
  		pRecent.add(nbRecentFilesField);
  		
  		UIManager.LookAndFeelInfo[] info = UIManager.getInstalledLookAndFeels();
  		String[] nameThemes = new String[info.length];
  		
  		for(int i=0; i<info.length;i++)
  			nameThemes[i] = info[i].getName();
  		
  		themeList = new JComboBox(nameThemes);
  		themeList.setMaximumSize(new Dimension(120, 30));
  		themeList.setMinimumSize(new Dimension(120, 30));
  		pTheme.add(new JLabel(LaTeXDrawLang.getString1_9("PreferencesFrame.1"))); //$NON-NLS-1$
  		pTheme.add(themeList);
     	
  		pLang.add(new JLabel(LABEL_LANG));
  		pLang.add(langList);
  		pEditor.add(pathTexEditorField);
  		pEditor.add(bChooseEditor);
  		pGeneral.add(pLang);
  		
  		if(LaTeXDrawFrame.WITH_UPDATE)
	  		pGeneral.add(checkNewVersion);
  		
  		pGeneral.add(new JLabel(LaTeXDrawLang.getString1_8("PreferencesFrame.3"))); //$NON-NLS-1$
  		pGeneral.add(pEditor);
  		pGeneral.add(pRecent);
  		pGeneral.add(pTheme);
  		
		return pGeneral;
	}
	
	
	
	
	/**
	 * Allows to create the panel which allows the user to set the code panel preferences.
	 * @return The panel.
	 */	
	private JPanel createCodePanel()
	{
		JPanel pCode = new JPanel(new GridLayout(6, 1));
  		
  		codeAutoUpdateCB = new JCheckBox(LABEL_AUTOUPDATE);
  		
  		pCode.add(codeAutoUpdateCB);
  		
		return pCode;
	}
	
	
	
	
	/**
	 * Creates a JPanel containing elements allowing the set of the grid parameters.
	 * @return The created JPanel.
	 */
	private JPanel createGridPanel()
	{
		JPanel pGrid = new JPanel(new GridLayout(5, 1)), pGap = new JPanel(new FlowLayout(FlowLayout.LEFT));

  		ButtonGroup bg = new ButtonGroup();
  		classicGridRB  = new JRadioButton(LaTeXDrawLang.getString1_8("PreferencesFrame.4")); //$NON-NLS-1$
  		persoGridRB    = new JRadioButton(LaTeXDrawLang.getString1_8("PreferencesFrame.5")); //$NON-NLS-1$
  		bg.add(classicGridRB);
  		bg.add(persoGridRB);
  		classicGridRB.setSelected(true);
  		displayGridCB      = new JCheckBox(LABEL_CHECKBOX_GRID);
  		magneticGridCB	   = new JCheckBox(LaTeXDrawLang.getString1_8("PreferencesFrame.6")); //$NON-NLS-1$
     	SpinnerModel model = new SpinnerNumberModel(MagneticGrid.DEFAULT_PERSONAL_GRID_GAP, 2, 100000, 1);
     	persoGridGapField = new JSpinner(model);
     	persoGridGapField.setEditor(new JSpinner.NumberEditor(persoGridGapField, "0"));//$NON-NLS-1$
     	persoGridGapField.setPreferredSize(new Dimension(45,25));
     	persoGridGapField.setMaximumSize(new Dimension(45,25));
     	pGap.add(new JLabel(LaTeXDrawLang.getString1_8("PreferencesFrame.7"))); //$NON-NLS-1$
     	pGap.add(persoGridGapField);
     	
  		pGrid.add(displayGridCB);
  		pGrid.add(classicGridRB);
  		pGrid.add(persoGridRB);
  		pGrid.add(magneticGridCB);
  		pGrid.add(pGap);
  		
		return pGrid;

	}
	
	
	
	private JPanel createLatexPanel()
	{
		JPanel pLatex 		 = new JPanel();
		JPanel pLatexDistrib = new JPanel();
		JLabel label         =  new JLabel("The path of the latex binaires:");
		
		label.setAlignmentX(Component.RIGHT_ALIGNMENT);
		pLatex.setLayout(new BoxLayout(pLatex, BoxLayout.Y_AXIS));
		pLatexDistrib.setLayout(new BoxLayout(pLatexDistrib, BoxLayout.X_AXIS));
  		pathLatexDistribField = new JTextField();
  		pathLatexDistribField.setMaximumSize(new Dimension(700,30));
  		pathLatexDistribField.setMinimumSize(new Dimension(300,30));
  		
  		JButton bChooseLatex  = new JButton(LaTeXDrawResources.openIcon);
  		bChooseLatex.addActionListener(this);
  		bChooseLatex.setActionCommand(NAME_BUTTON_LATEX_DISTRIB);
  		
  		latexIncludes = new JTextArea();
  		latexIncludes.setToolTipText("<html>Include in this list the latex packages you regularly use in your drawing, e.g. :<br>\\usepackage[frenchb]{babel}<br>\\usepackage[utf8]{inputenc}</html>");
  		JScrollPane scrollPane = new JScrollPane(latexIncludes); 
  		
  		pLatexDistrib.add(pathLatexDistribField);
  		pLatexDistrib.add(bChooseLatex);
  		pLatex.add(label);
  		pLatex.add(pLatexDistrib);
  		pLatex.add(scrollPane);
  		
		return pLatex;
	}
	
	
	
	/**
	 * Allows to create the panel which allows the user to set the preferences about displayable elements.
	 * @return The panel.
	 */	
	private JPanel createDisplayPanel()
	{
  		JPanel pDisplay = new JPanel(new GridLayout(5, 1));
  		 		
   		displayCodePanelCB = new JCheckBox(LABEL_CHECKBOX_CODEPANEL);
  		displayXScaleCB    = new JCheckBox(LABEL_CHECKBOX_XSCALE);
  		displayYScaleCB    = new JCheckBox(LABEL_CHECKBOX_YSCALE);
  		displayBordersCB   = new JCheckBox(LABEL_CHECKBOX_DISPLAY_BORDERS);
  		unitChoice = new JComboBox();
  		unitChoice.addItem(XScale.LABEL_CM);
  		unitChoice.addItem(XScale.LABEL_INCH);
  		
  		pDisplay.add(displayCodePanelCB);
  		pDisplay.add(displayXScaleCB);
  		pDisplay.add(displayYScaleCB);
  		pDisplay.add(displayBordersCB);
  		pDisplay.add(unitChoice);
  		
  		return pDisplay;
	}
	
	
	
	
	
	/**
	 * Allows to create the panel which allows the user to set preferences about paths.
	 * @return The panel.
	 */	
	private JPanel createFoldersPanel()
	{
  		JPanel pFolders  = new JPanel(new GridBagLayout());
  		pathExportField  = new JTextField();
  		pathOpenField    = new JTextField();
  		JButton bExport  = new JButton(LaTeXDrawResources.openIcon);
  		JButton bOpen    = new JButton(LaTeXDrawResources.openIcon);
  		
  		bExport.setActionCommand(NAME_BUTTON_EXPORTFIELD);
  		bOpen.setActionCommand(NAME_BUTTON_OPENFIELD);
  		bExport.addActionListener(this);
  		bOpen.addActionListener(this);
  		
  		GridBagConstraints constraint = new GridBagConstraints();
    	constraint.gridx = 0;
     	constraint.gridy = 0;
     	constraint.gridwidth = 6;
     	constraint.gridheight = 1;
     	constraint.weightx = 0.1;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.HORIZONTAL;
     	constraint.anchor = GridBagConstraints.EAST;
     	pFolders.add(new JLabel(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.defOpenSave")), constraint); //$NON-NLS-1$
     	
  		constraint = new GridBagConstraints();
    	constraint.gridx = 0;
     	constraint.gridy = 1;
     	constraint.gridwidth = 6;
     	constraint.gridheight = 1;
     	constraint.weightx = 10;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.HORIZONTAL;
     	constraint.anchor = GridBagConstraints.EAST;
     	pFolders.add(pathOpenField, constraint);
  		
     	constraint = new GridBagConstraints();
    	constraint.gridx = 6;
     	constraint.gridy = 1;
     	constraint.gridwidth = 1;
     	constraint.gridheight = 1;
     	constraint.weightx = 0.1;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.EAST;
     	pFolders.add(bOpen, constraint);
 
     	constraint = new GridBagConstraints();
    	constraint.gridx = 0;
     	constraint.gridy = 2;
     	constraint.gridwidth = 6;
     	constraint.gridheight = 1;
     	constraint.weightx = 0.1;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.HORIZONTAL;
     	constraint.anchor = GridBagConstraints.EAST;
     	pFolders.add(new JLabel(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.defFold")), constraint); //$NON-NLS-1$
     	
  		constraint = new GridBagConstraints();
    	constraint.gridx = 0;
     	constraint.gridy = 3;
     	constraint.gridwidth = 6;
     	constraint.gridheight = 1;
     	constraint.weightx = 10;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.HORIZONTAL;
     	constraint.anchor = GridBagConstraints.EAST;
     	pFolders.add(pathExportField, constraint);
  		
     	constraint = new GridBagConstraints();
    	constraint.gridx = 6;
     	constraint.gridy = 3;
     	constraint.gridwidth = 1;
     	constraint.gridheight = 1;
     	constraint.weightx = 0.1;
     	constraint.weighty = 0.1;
     	constraint.fill = GridBagConstraints.NONE;
     	constraint.anchor = GridBagConstraints.EAST;
     	pFolders.add(bExport, constraint);
     
  		return pFolders;
	}
	
	
	
	
	/**
	 * Allows to create the panel which allows the user to set preferences about the rendering.
	 * @return The panel.
	 */
	private JPanel createQualityPanel()
	{
  		JPanel pQuality  = new JPanel(new GridLayout(4, 1));

  		antialiasingCheckBox = new JCheckBox(LABEL_CHECKBOX_ANTIALIASING);
  		renderingCheckBox    = new JCheckBox(LABEL_CHECKBOX_RENDERING);
  		colorRenderCheckBox  = new JCheckBox(LABEL_CHECKBOX_COLORRENDER);
  		alpaInterCheckBox    = new JCheckBox(LABEL_CHECKBOX_ALPHAINTER);
  		
  		pQuality.add(antialiasingCheckBox);
  		pQuality.add(renderingCheckBox);
  		pQuality.add(colorRenderCheckBox);
  		pQuality.add(alpaInterCheckBox);
  		
  		return pQuality;
	}
	

	
	public void actionPerformed(ActionEvent e) 
	{
		Object o = e.getSource();
		
		try
		{
			if(o instanceof JButton)
			{
				String label = ((JButton)o).getActionCommand();
				
				if(label==null)
					return ;
				
				
				if(label.equals(NAME_BUTTON_CANCEL))
				{
					super.setVisible(false);
					nbRecentFilesField.setValue(curNbRecentFiles);
					persoGridGapField.setValue(curGridGap);
					classicGridRB.setSelected(curClassicGrid);
					langList.setSelectedItem(curLang);
					themeList.setSelectedItem(curTheme);
					unitChoice.setSelectedItem(curUnit);
					pathTexEditorField.setText(curTexPath);
					pathExportField.setText(curExportPath);
					pathOpenField.setText(curOpenPath);
					alpaInterCheckBox.setSelected(curAlpha);
					antialiasingCheckBox.setSelected(curAntiAlias);
					renderingCheckBox.setSelected(curRendering);
					colorRenderCheckBox.setSelected(curColRender);
					magneticGridCB.setSelected(curMagnetic);
					displayGridCB.setSelected(curDisplayGrid);
					displayXScaleCB.setSelected(curXScale);
					displayYScaleCB.setSelected(curYScale);
					codeAutoUpdateCB.setSelected(curAutoUpdte);
					checkNewVersion.setSelected(curCheckVers);
					displayCodePanelCB.setSelected(curDisplayCode);
					displayBordersCB.setSelected(curDisplayBord);
					latexIncludes.setText(curLatexIncludes);
					pathLatexDistribField.setText(curLatexDistrib);
					
					return;
				}
				
				if(label.equals(NAME_BUTTON_DEFAULT))
				{					
					setDefaultValues();
					return;
				}
				
				if(label.equals(NAME_BUTTON_OK))
				{
					writeXMLPreferences(true);
					mainFrame.setLatexIncludes(latexIncludes.getText());
					mainFrame.setPathDistribLatex(pathLatexDistribField.getText());
					mainFrame.setPathTexEditor(pathTexEditorField.getText());
					mainFrame.setPathExport(pathExportField.getText());
					mainFrame.setPathOpen(pathOpenField.getText());
					draw.setAntiAliasingValue(antialiasingCheckBox.isSelected());
					draw.setRenderingcolorValue(colorRenderCheckBox.isSelected());
					draw.setAlphaInterValue(alpaInterCheckBox.isSelected());
					draw.setRenderingValue(renderingCheckBox.isSelected());
					setVisible(false);
					return;
				}
				
				if(label.equals(NAME_BUTTON_TEX_EDITOR))
				{ 
					fileChooser.setFileSelectionMode(JFileChooser.FILES_ONLY);
					fileChooser.showSaveDialog(this);
					File file = fileChooser.getSelectedFile();
					
					if(file!=null && file.exists() && file.isFile())
							pathTexEditorField.setText(file.getPath());

					return;
				}
				
				if(label.equals(NAME_BUTTON_LATEX_DISTRIB))
				{ 
					fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
					fileChooser.showSaveDialog(this);
					File file = fileChooser.getSelectedFile();
					
					if(file!=null && file.exists() && file.isFile())
						pathLatexDistribField.setText(file.getPath());
					
					return;
				}
				
				if(label.equals(NAME_BUTTON_OPENFIELD))
				{ 
					fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
					fileChooser.setCurrentDirectory(new File(pathOpenField.getText()));
					fileChooser.showSaveDialog(this);
					File file = fileChooser.getSelectedFile();
					
					if(file!=null)
						if(file.isDirectory())
							pathOpenField.setText(file.getPath());
					
					return;
				}
				
				if(label.equals(NAME_BUTTON_EXPORTFIELD))
				{
					fileChooser.setFileSelectionMode(JFileChooser.DIRECTORIES_ONLY);
					fileChooser.setCurrentDirectory(new File(pathExportField.getText()));
					fileChooser.showSaveDialog(this);
					File file = fileChooser.getSelectedFile();
					
					if(file!=null)
						if(file.isDirectory())
							pathExportField.setText(file.getPath());

					return;
				}
			} // if(o instanceof JButton)
		}catch(Exception ex) 
		{ 
			ex.printStackTrace(); 
			ExceptionFrameDialog.showExceptionDialog(ex);
		}
	}

	
	
	
	/**
	 * Reads the preferences of latexdraw defined in XML.
	 * @since 2.0.0
	 */
	public void readXMLPreferences()
	{
        File xml = new File(LaTeXDrawPath.PATH_PREFERENCES_XML_FILE);
    	File prefOld = new File(LaTeXDrawPath.PATH_PREFERENCES_FILE_OLD);
    	
        setDefaultValues();
        
        if(!xml.exists())
        {
        	if(prefOld.exists())
        	{
        		readPreferences();
        		writeXMLPreferences(false);
        	}
        	else
        		updateDrawPreferences();
        	
        	return ;
        }
        
		prefOld.delete();
        
		try
		{
            DocumentBuilderFactory fabrique = DocumentBuilderFactory.newInstance();
            DocumentBuilder constructeur = fabrique.newDocumentBuilder();
            Document document = constructeur.parse(xml);
            Node n = document.getFirstChild(), n2;
            NodeList nl;
            int size, i;
            String name, content;
           
            isFrameMaximized = true;
            
            if(n==null || !n.getNodeName().equals(LaTeXDrawNamespace.XML_ROOT_PREFERENCES))
            	throw new NullPointerException();
            
            nl = n.getChildNodes();
            
            for(i=0, size=nl.getLength(); i<size; i++)
            {
            	n = nl.item(i);
            	name = n.getNodeName();
            	content = n.getTextContent();
            	
            	if(name==null || name.length()==0)
            		continue;
            	
            	if(name.equals(LaTeXDrawNamespace.XML_LATEX_INCLUDES)) latexIncludes.setText(content);
            	else if(name.equals(LaTeXDrawNamespace.XML_PATH_LATEX_DISTRIB)) pathLatexDistribField.setText(content);
            	else if(name.equals(LaTeXDrawNamespace.XML_ALPHA_INTER)) alpaInterCheckBox.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_ANTI_ALIAS)) antialiasingCheckBox.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_AUTO_UPDATE)) codeAutoUpdateCB.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_CHECK_VERSION)) checkNewVersion.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_CLASSIC_GRID))
            	{
            		classicGridRB.setSelected(Boolean.valueOf(content).booleanValue());
            		persoGridRB.setSelected(!Boolean.valueOf(content).booleanValue());
            	}
            	else if(name.equals(LaTeXDrawNamespace.XML_COLOR_RENDERING)) colorRenderCheckBox.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_DRAW_BORDERS)) displayBordersCB.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_DISPLAY_CODE_PANEL)) displayCodePanelCB.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_DISPLAY_GRID)) displayGridCB.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_DISPLAY_X)) displayXScaleCB.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_DISPLAY_Y)) displayYScaleCB.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_GRID_GAP)) persoGridGapField.setValue(Integer.valueOf(content).intValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_LAF))
            	{
    				int count = themeList.getItemCount(), j=0;
    				boolean again = true;
    				
    				while(j<count && again)
    					if(themeList.getItemAt(j).toString().equals(content))
    						again = false;
    					else
    						j++;
    			
    				if(again)
    					 themeList.setSelectedItem(UIManager.getCrossPlatformLookAndFeelClassName());
    				else themeList.setSelectedIndex(j);
            	}
            	else if(name.equals(LaTeXDrawNamespace.XML_LANG)) langList.setSelectedItem(content);
            	else if(name.equals(LaTeXDrawNamespace.XML_MAGNETIC_GRID)) magneticGridCB.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_PATH_EXPORT)) pathExportField.setText(content);
            	else if(name.equals(LaTeXDrawNamespace.XML_PATH_OPEN)) pathOpenField.setText(content);
            	else if(name.equals(LaTeXDrawNamespace.XML_PATH_TEX_EDITOR)) pathTexEditorField.setText(content);
            	else if(name.equals(LaTeXDrawNamespace.XML_RENDERING)) renderingCheckBox.setSelected(Boolean.valueOf(content).booleanValue());
            	else if(name.equals(LaTeXDrawNamespace.XML_UNIT)) unitChoice.setSelectedItem(content);
            	else if(name.equals(LaTeXDrawNamespace.XML_RECENT_FILES))
            	{
            		NodeList nl2 = n.getChildNodes();
            		NamedNodeMap nnm = n.getAttributes();
            		recentFilesName.clear();
            		
            		if(nnm!=null && nnm.getNamedItem(LaTeXDrawNamespace.XML_NB_RECENT_FILES)!=null)
            		{
            			Node attr = nnm.getNamedItem(LaTeXDrawNamespace.XML_NB_RECENT_FILES);
            			
            			if(attr!=null)
            				nbRecentFilesField.setValue(Integer.valueOf(attr.getTextContent()).intValue());
            		}
            		
            		for(int j=0, size2=nl2.getLength(); j<size2; j++)
            		{
            			n2 = nl2.item(j);
            			
            			if(n2.getNodeName().equals(LaTeXDrawNamespace.XML_RECENT_FILE) && n2.getTextContent()!=null)
            				recentFilesName.add(n2.getTextContent());
            		}
            	}
            	else if(name.equals(LaTeXDrawNamespace.XML_MAXIMISED)) isFrameMaximized = Boolean.valueOf(content).booleanValue();
            	else if(name.equals(LaTeXDrawNamespace.XML_SIZE))
            	{
            		NodeList nl2 = n.getChildNodes();
            		frameSize = new Dimension();
            		
            		for(int j=0, size2=nl2.getLength(); j<size2; j++)
            		{
            			n2 = nl2.item(j);
            			
            			if(n2.getNodeName().equals(LaTeXDrawNamespace.XML_WIDTH)) 
            				frameSize.width = Integer.valueOf(n2.getTextContent()).intValue();
            			else 
            				if(n2.getNodeName().equals(LaTeXDrawNamespace.XML_HEIGHT)) 
            					frameSize.height = Integer.valueOf(n2.getTextContent()).intValue();
            		}
            	}
            	else if(name.equals(LaTeXDrawNamespace.XML_POSITION))
            	{
            		NodeList nl2 = n.getChildNodes();
            		framePosition = new Point();
            		
            		for(int j=0, size2=nl2.getLength(); j<size2; j++)
            		{
            			n2 = nl2.item(j);
            			
            			if(n2.getNodeName().equals(LaTeXDrawNamespace.XML_POSITION_X)) 
            				framePosition.x = Math.max(0, Integer.valueOf(n2.getTextContent()).intValue());
            			else 
            				if(n2.getNodeName().equals(LaTeXDrawNamespace.XML_POSITION_Y)) 
            					framePosition.y = Math.max(0, Integer.valueOf(n2.getTextContent()).intValue());
            		}
            	}
            	else if(name.equals(LaTeXDrawNamespace.XML_DIVIDER_POSITION))
            		dividerPosition = Double.valueOf(content).doubleValue();
            	else if(name.equals(LaTeXDrawNamespace.XML_DELIMITOR_OPACITY))
            		Delimitor.setOpacity(Double.valueOf(content).intValue());
            }
		}catch(Exception ex) { ex.printStackTrace(); }
		
		updateDrawPreferences();
		mainFrame.setIsModified(false);
	}
	
	
	
	
	
	/**
	 * Writes the preferences of latexdraw in XML. 
	 * @since 2.0.0
	 */
	public void writeXMLPreferences(boolean withNotification)
	{
		try
		{
			FileOutputStream fos = new FileOutputStream(LaTeXDrawPath.PATH_PREFERENCES_XML_FILE);
			DocumentBuilderFactory factory = DocumentBuilderFactory.newInstance();
            DocumentBuilder builder = factory.newDocumentBuilder();
            Document document = builder.newDocument();
            Element root, elt, elt2;
            OutputFormat of;
            XMLSerializer xmls;
            
            document.setXmlVersion("1.0");//$NON-NLS-1$
            document.setXmlStandalone(true);
            root = document.createElement(LaTeXDrawNamespace.XML_ROOT_PREFERENCES);
            
            Attr attr = document.createAttribute(LaTeXDrawNamespace.XML_VERSION);
            attr.setTextContent(LaTeXDrawFrame.VERSION);
            root.setAttributeNode(attr);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_RENDERING);
            elt.setTextContent(String.valueOf(renderingCheckBox.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_COLOR_RENDERING);
            elt.setTextContent(String.valueOf(colorRenderCheckBox.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_ALPHA_INTER);
            elt.setTextContent(String.valueOf(alpaInterCheckBox.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_ANTI_ALIAS);
            elt.setTextContent(String.valueOf(antialiasingCheckBox.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_PATH_EXPORT);
            elt.setTextContent(pathExportField.getText());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_PATH_OPEN);
            elt.setTextContent(pathOpenField.getText());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_DISPLAY_CODE_PANEL);
            elt.setTextContent(String.valueOf(displayCodePanelCB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_DISPLAY_GRID);
            elt.setTextContent(String.valueOf(displayGridCB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_DISPLAY_X);
            elt.setTextContent(String.valueOf(displayXScaleCB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_DISPLAY_Y);
            elt.setTextContent(String.valueOf(displayYScaleCB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_DRAW_BORDERS);
            elt.setTextContent(String.valueOf(displayBordersCB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_UNIT);
            elt.setTextContent(unitChoice.getSelectedItem().toString());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_AUTO_UPDATE);
            elt.setTextContent(String.valueOf(codeAutoUpdateCB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_CHECK_VERSION);
            elt.setTextContent(String.valueOf(checkNewVersion.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_LANG);
            elt.setTextContent(langList.getSelectedItem().toString());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_MAGNETIC_GRID);
            elt.setTextContent(String.valueOf(magneticGridCB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_CLASSIC_GRID);
            elt.setTextContent(String.valueOf(classicGridRB.isSelected()));
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_GRID_GAP);
            elt.setTextContent(persoGridGapField.getValue().toString());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_PATH_TEX_EDITOR);
            elt.setTextContent(pathTexEditorField.getText());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_PATH_LATEX_DISTRIB);
            elt.setTextContent(pathLatexDistribField.getText());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_LATEX_INCLUDES);
            elt.setTextContent(latexIncludes.getText());
            root.appendChild(elt);
            
            elt = document.createElement(LaTeXDrawNamespace.XML_RECENT_FILES);
            elt.setAttribute(LaTeXDrawNamespace.XML_NB_RECENT_FILES, nbRecentFilesField.getValue().toString());
            root.appendChild(elt);
            
            for(String recentFile : recentFilesName)
            {
	            elt2 = document.createElement(LaTeXDrawNamespace.XML_RECENT_FILE);
	            elt2.setTextContent(recentFile);
	            elt.appendChild(elt2); 
            }
            
            elt = document.createElement(LaTeXDrawNamespace.XML_LAF);
            elt.setTextContent(themeList.getSelectedItem().toString());
            root.appendChild(elt);
            
            if(mainFrame!=null)
            {
	            elt = document.createElement(LaTeXDrawNamespace.XML_MAXIMISED);
	            elt.setTextContent(String.valueOf(mainFrame.getExtendedState()==Frame.MAXIMIZED_BOTH));
	            root.appendChild(elt);
	            
	            elt = document.createElement(LaTeXDrawNamespace.XML_SIZE);
	            root.appendChild(elt);
	            
	            elt2 = document.createElement(LaTeXDrawNamespace.XML_WIDTH);
	            elt2.setTextContent(String.valueOf(mainFrame.getWidth()));
	            elt.appendChild(elt2);
	            
	            elt2 = document.createElement(LaTeXDrawNamespace.XML_HEIGHT);
	            elt2.setTextContent(String.valueOf(mainFrame.getHeight()));
	            elt.appendChild(elt2);
	            
	            elt = document.createElement(LaTeXDrawNamespace.XML_POSITION);
	            root.appendChild(elt);
	            
	            elt2 = document.createElement(LaTeXDrawNamespace.XML_POSITION_X);
	            elt2.setTextContent(String.valueOf(mainFrame.getLocation().x));
	            elt.appendChild(elt2);
	            
	            elt2 = document.createElement(LaTeXDrawNamespace.XML_POSITION_Y);
	            elt2.setTextContent(String.valueOf(mainFrame.getLocation().y));
	            elt.appendChild(elt2);
	            
	            elt = document.createElement(LaTeXDrawNamespace.XML_DIVIDER_POSITION);
	            
	            double divLoc = mainFrame.isCodePanelDisplayed() ? mainFrame.getDividerCurrentLocation() : mainFrame.getFormerDividerLocation();
	            elt.setTextContent(String.valueOf(divLoc<mainFrame.getWidth() && divLoc>=0 ? divLoc/mainFrame.getWidth() : 1.));
	            root.appendChild(elt);
	            
	            elt = document.createElement(LaTeXDrawNamespace.XML_DELIMITOR_OPACITY);
	            elt.setTextContent(String.valueOf(Delimitor.getOpacity()));
	            root.appendChild(elt);
            }
			
            of = new OutputFormat(document);
            of.setIndenting(true);
            xmls = new XMLSerializer(fos, of);
            xmls.serialize(root);
			fos.close();
			
			if(withNotification)
				mainFrame.setStatusBarText(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.prefSaved")); //$NON-NLS-1$
			
		}catch(Exception ex) { ex.printStackTrace(); }
	}
	
	
	
	/**
	 * Allows to read the file of preferences.<br>
	 */
	public void readPreferences()
	{
		try // We save the project in the file
		{
			InputStream is = new FileInputStream(new File(LaTeXDrawPath.PATH_PREFERENCES_FILE_OLD));
			ObjectInputStream in = new ObjectInputStream(is);
			Object o = in.readObject();
			
			if(!(o instanceof String) || !((String)o).equals(LABEL_PREFERENCES))
				throw new LaTeXDrawException(LaTeXDrawException.NOT_TEXDRAW_FILE);
			
			o = in.readObject();

			renderingCheckBox.setSelected(in.readBoolean());
			colorRenderCheckBox.setSelected(in.readBoolean());
			alpaInterCheckBox.setSelected(in.readBoolean());
			antialiasingCheckBox.setSelected(in.readBoolean());
			pathExportField.setText((String)in.readObject());
			pathOpenField.setText((String)in.readObject());
			displayCodePanelCB.setSelected(in.readBoolean());
			displayGridCB.setSelected(in.readBoolean());
			displayXScaleCB.setSelected(in.readBoolean());
			displayYScaleCB.setSelected(in.readBoolean());
			displayBordersCB.setSelected(in.readBoolean());
			unitChoice.setSelectedItem(in.readObject());
			
			codeAutoUpdateCB.setSelected(in.readBoolean());
		
			if(((String)o).compareTo(LaTeXDrawFrame.VERSION)>0)
			{
				checkNewVersion.setSelected(LaTeXDrawFrame.DEFAULT_CHECK_VERSION);
				langList.setSelectedItem(LaTeXDrawLang.getlangCurrent());
			}
			else
			{
				checkNewVersion.setSelected(in.readBoolean());
				langList.setSelectedItem(in.readObject());
			}
			
			if(((String)o).compareTo("1.8")>=0) //$NON-NLS-1$
			{
				magneticGridCB.setSelected(in.readBoolean());
				boolean classic = in.readBoolean();
				classicGridRB.setSelected(classic);
				persoGridRB.setSelected(!classic);
				persoGridGapField.setValue(in.readInt());
				pathTexEditorField.setText((String)in.readObject());
			}
			else
			{
				persoGridRB.setSelected(MagneticGrid.DEFAULT_TYPE==MagneticGrid.TYPE_PERSONAL);
				classicGridRB.setSelected(MagneticGrid.DEFAULT_TYPE!=MagneticGrid.TYPE_PERSONAL);
				persoGridGapField.setValue(MagneticGrid.DEFAULT_PERSONAL_GRID_GAP);
				magneticGridCB.setSelected(false);
			}
			
			if(((String)o).compareTo("1.9")>=0) //$NON-NLS-1$
			{
				nbRecentFilesField.setValue(in.readInt());
				int nb = in.readInt(), i;
				
				recentFilesName.clear();
				for(i=0; i<nb; i++)
					recentFilesName.add((String)in.readObject());
				
				int count = themeList.getItemCount();
				boolean again = true;
				String theme = in.readObject().toString();
				i=0;
				
				while(i<count && again)
					if(themeList.getItemAt(i).toString().equals(theme))
						again = false;
					else
						i++;
			
				if(again)
					 themeList.setSelectedItem(UIManager.getCrossPlatformLookAndFeelClassName());
				else themeList.setSelectedIndex(i);
				
			}
			else
			{
				nbRecentFilesField.setValue(NB_RECENT_FILES);
				themeList.setSelectedItem(UIManager.getCrossPlatformLookAndFeelClassName());
			}
			
			in.close();
			is.close();
			
			updateDrawPreferences();			
		}
		catch (LaTeXDrawException e)
		{
			System.out.println(LaTeXDrawLang.getDialogFrameString("PreferencesFrame.errorReadPref")); //$NON-NLS-1$
			setDefaultValues();	
		}
		catch (Exception e) { setDefaultValues(); }
	}
	
	
	
	
	/**
	 * Allows to set default values to the components of this frame.
	 */
	public void setDefaultValues()
	{
		alpaInterCheckBox.setSelected(
				DrawContainer.DEFAULT_ALPHAINTER_VAL==RenderingHints.VALUE_ALPHA_INTERPOLATION_QUALITY);
			
		renderingCheckBox.setSelected(
				DrawContainer.DEFAULT_RENDERING_VAL==RenderingHints.VALUE_RENDER_QUALITY);
		
		antialiasingCheckBox.setSelected(
				DrawContainer.DEFAULT_ANTIALIASING_VAL==RenderingHints.VALUE_ANTIALIAS_ON);
		
		colorRenderCheckBox.setSelected(
				DrawContainer.DEFAULT_COLORRENDER_VAL==RenderingHints.VALUE_COLOR_RENDER_QUALITY);
		
		pathOpenField.setText(LaTeXDrawFrame.DEFAULT_PATH);
		pathExportField.setText(LaTeXDrawFrame.DEFAULT_PATH);
		
		if(LaTeXDrawFrame.WITH_UPDATE)
			checkNewVersion.setSelected(LaTeXDrawFrame.DEFAULT_CHECK_VERSION);
		
		displayCodePanelCB.setSelected(LaTeXDrawFrame.DEFAULT_DISPLAY_CODEPANEL);
		displayGridCB.setSelected(LaTeXDrawFrame.DEFAULT_DISPLAY_GRID);
		displayXScaleCB.setSelected(LaTeXDrawFrame.DEFAULT_DISPLAY_XSCALE);
		displayYScaleCB.setSelected(LaTeXDrawFrame.DEFAULT_DISPLAY_YSCALE);
		displayBordersCB.setSelected(LaTeXDrawFrame.DEFAULT_DISPLAY_BORDERS);
		unitChoice.setSelectedItem(XScale.DEFAULT_UNIT);
		codeAutoUpdateCB.setSelected(CodePanel.DEFAULT_AUTO_UPDATE);
		classicGridRB.setSelected(MagneticGrid.DEFAULT_TYPE!=MagneticGrid.TYPE_PERSONAL);
		persoGridRB.setSelected(MagneticGrid.DEFAULT_TYPE==MagneticGrid.TYPE_PERSONAL);
		persoGridGapField.setValue(MagneticGrid.DEFAULT_PERSONAL_GRID_GAP);
		nbRecentFilesField.setValue(NB_RECENT_FILES);
		magneticGridCB.setSelected(false);
		
		UIManager.LookAndFeelInfo[] info = UIManager.getInstalledLookAndFeels();
		boolean again = true;
		int i = 0;
		String laf;
		
        if(System.getProperty("os.name").toLowerCase().equals("linux") && System.getProperty("java.version").toLowerCase().compareTo("1.6")>=0)//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$
			laf = "GTK+";//$NON-NLS-1$
		else
			laf = "Windows";//$NON-NLS-1$
		
		while(i < info.length && again)
			if(laf.compareTo(info[i].getName()) == 0)
				again = false;
			else
				i++;
		
		if(again)
		{
			String basic = UIManager.getCrossPlatformLookAndFeelClassName();
			i = 0;
			while(i < info.length && again)
				if(basic.compareTo(info[i].getClassName()) == 0)
					again = false;
				else
					i++;
			
			if(!again)
				themeList.setSelectedItem(info[i].getName());
		}
		else
			themeList.setSelectedItem(laf); 
		
		isFrameMaximized 	= false;
		frameSize 			= new Dimension();
		frameSize.height 	= 3*Toolkit.getDefaultToolkit().getScreenSize().height/2;
		frameSize.width 	= 3*Toolkit.getDefaultToolkit().getScreenSize().width/2;
		framePosition 		= new Point();
		dividerPosition 	= 0.8;
	}
	
	
	
	
	/**
	 * Allows to update the concerned attributes of the drawPanel.
	 */
	public void updateDrawPreferences()
	{
		try 
		{
			boolean isMod = draw.isModified();
			
			draw.setAntiAliasingValue(antialiasingCheckBox.isSelected());
			draw.setRenderingcolorValue(colorRenderCheckBox.isSelected());
			draw.setAlphaInterValue(alpaInterCheckBox.isSelected());
			draw.setRenderingValue(renderingCheckBox.isSelected());
			
			mainFrame.setLatexIncludes(latexIncludes.getText());
			mainFrame.setPathDistribLatex(pathLatexDistribField.getText());
			mainFrame.setMagneticGrid(magneticGridCB.isSelected());
			mainFrame.setPathExport(pathExportField.getText());
			mainFrame.setPathOpen(pathOpenField.getText());
			mainFrame.displayCodePanel(displayCodePanelCB.isSelected());
			mainFrame.setFormerIsInAutoUpdate(codeAutoUpdateCB.isSelected());
			mainFrame.setIsInAutoUpdate(codeAutoUpdateCB.isSelected());
			
			if(LaTeXDrawFrame.WITH_UPDATE)
				
				mainFrame.setCheckNewVersion(checkNewVersion.isSelected());
			mainFrame.displayGrid(displayGridCB.isSelected(), classicGridRB.isSelected());
			mainFrame.setPersoGridGap(Integer.valueOf(persoGridGapField.getValue().toString()).intValue());
			mainFrame.displayXScale(displayXScaleCB.isSelected());
			mainFrame.displayYScale(displayYScaleCB.isSelected());
			mainFrame.displayBorders(displayBordersCB.isSelected());
			mainFrame.setUnitForPixelsPerCm(unitChoice.getSelectedItem().toString());
			mainFrame.setPathTexEditor(pathTexEditorField.getText());
			String lAfName = themeList.getSelectedItem().toString();
	  		UIManager.LookAndFeelInfo[] info = UIManager.getInstalledLookAndFeels();
	  		boolean again = true;
	  		int i=0;
	  		while(i<info.length && again)
	  			if(lAfName.equals(info[i].getName()))
	  				again = false;
	  			else
	  				i++;
	  		
	  		if(again)// Problem with the look & feel
	  			LaTeXDrawFrame.setLookAndFeel(UIManager.getCrossPlatformLookAndFeelClassName());
	  		else
	  			LaTeXDrawFrame.setLookAndFeel(info[i].getClassName());
	  		
			draw.setIsModified(isMod);
			LaTeXDrawLang.setlangCurrent((String)langList.getSelectedItem());
			
			if(framePosition!=null)
				mainFrame.setLocation(framePosition);
			
			if(isFrameMaximized || frameSize==null || frameSize.width==0 || frameSize.height==0)
				mainFrame.setExtendedState(Frame.MAXIMIZED_BOTH);
			
			if(frameSize!=null)
			{
				mainFrame.setSize(frameSize.width, frameSize.height);
				mainFrame.setDividerPosition(dividerPosition);
			}
			
		} catch (Exception e) {e.printStackTrace();}

	}
	
	
	
	
	/**
	 * @return the recentFilesName.
	 */
	public synchronized Vector<String> getRecentFilesName()
	{
		return recentFilesName;
	}




	/**
	 * Adds a recent file.
	 * @since 1.9
	 * @param absolutePath The absolute path of the file to add.
	 */
	public void addRecentFile(String absolutePath)
	{
		int i = recentFilesName.indexOf(absolutePath);
		int max = Float.valueOf(nbRecentFilesField.getValue().toString()).intValue();
	
		if(i!=-1)
			recentFilesName.remove(i);
		
		while(recentFilesName.size()>=max)
			recentFilesName.remove(max-1);
		
		recentFilesName.add(0, absolutePath);
		writeXMLPreferences(false);
	}	
	
	
	
	
	/**
	 * @since 1.9
	 * @return True if the code is in auto update mode.
	 */
	public boolean isCodeInAutoUpdate()
	{
		return codeAutoUpdateCB.isSelected();
	}
	
	
	
	/**
	 * Allows to get the theme of the program.<br>
	 * @since 1.9
	 * @return The class of the theme.
	 */
	public static String readTheme()
	{
		String laf=null;
		
		try 
		{
			File xml = new File(LaTeXDrawPath.PATH_PREFERENCES_XML_FILE);
			
			if(xml.exists())
			{
	            DocumentBuilderFactory fabrique = DocumentBuilderFactory.newInstance();
	            DocumentBuilder constructeur = fabrique.newDocumentBuilder();
	            Document document = constructeur.parse(xml);
	            Node n = document.getFirstChild();
	            NodeList nl;
	            boolean again = true;
	            
	            if(n==null || !n.getNodeName().equals(LaTeXDrawNamespace.XML_ROOT_PREFERENCES))
	            	throw new NullPointerException();
	            
	            nl = n.getChildNodes();
	            
	            for(int i=0, size=nl.getLength() ; i<size && again; i++)
	            	if(nl.item(i).getNodeName().equals(LaTeXDrawNamespace.XML_LAF))
	            	{
	            		laf = nl.item(i).getTextContent();
	            		again = false;
	            	}
			}
			else
			{
				InputStream is = new FileInputStream(new File(LaTeXDrawPath.PATH_PREFERENCES_FILE_OLD));
				ObjectInputStream in = new ObjectInputStream(is);
				Object o = in.readObject();
	
				if(!(o instanceof String) || !((String)o).equals(LABEL_PREFERENCES))
					throw new LaTeXDrawException(LaTeXDrawException.NOT_TEXDRAW_FILE);
				
				o = in.readObject();
				
				if(((String)o).compareTo("1.9")<0) //$NON-NLS-1$
					return UIManager.getCrossPlatformLookAndFeelClassName();
				
				in.readBoolean();
				in.readBoolean();
				in.readBoolean();
				in.readBoolean();
				in.readObject();
				in.readObject();
				in.readBoolean();
				in.readBoolean();
				in.readBoolean();
				in.readBoolean();
				in.readBoolean();
				in.readObject();
				in.readBoolean();
				in.readBoolean();
				in.readObject();
				in.readBoolean();
				in.readBoolean();
				in.readInt();
				in.readObject();
				in.readInt();
				int nb = in.readInt(), i;
				for(i=0; i<nb; i++)
					in.readObject();
				
				laf = (String)in.readObject();
				
				in.close();
				is.close();
			}
		}
		catch(Exception e) { /* Nothing to do. */ }
		
		if(laf==null)
		{
			if(System.getProperty("os.name").toLowerCase().equals("linux") && System.getProperty("java.version").toLowerCase().compareTo("1.6")>=0)//$NON-NLS-1$//$NON-NLS-2$//$NON-NLS-3$//$NON-NLS-4$
				return "com.sun.java.swing.plaf.gtk.GTKLookAndFeel"; //$NON-NLS-1$

			if(System.getProperty("os.name").toLowerCase().contains("win"))//$NON-NLS-1$//$NON-NLS-2$
				return "com.sun.java.swing.plaf.windows.WindowsLookAndFeel";//$NON-NLS-1$
			
  			return UIManager.getCrossPlatformLookAndFeelClassName();  
		}
		
		UIManager.LookAndFeelInfo[] info = UIManager.getInstalledLookAndFeels();
		boolean again = true;
		int i=0;
		
		while(i<info.length && again)
			if(laf.equals(info[i].getName()))
				again = false;
			else
				i++;
		
		if(again)// Problem with the look & feel
			return UIManager.getCrossPlatformLookAndFeelClassName();
		
		return info[i].getClassName();
	}




	@Override
	public void setVisible(boolean b)
	{
		try
		{
			curNbRecentFiles = Double.valueOf(nbRecentFilesField.getValue().toString()).intValue();
			curGridGap		 = Double.valueOf(persoGridGapField.getValue().toString()).intValue();
			curClassicGrid	 = classicGridRB.isSelected();
			curLang			 = langList.getSelectedItem()==null ? null : langList.getSelectedItem().toString();
			curTheme		 = themeList.getSelectedItem()==null ? null : themeList.getSelectedItem().toString();
			curUnit			 = unitChoice.getSelectedItem()==null ? null : unitChoice.getSelectedItem().toString();
			curTexPath		 = pathTexEditorField.getText();
			curExportPath	 = pathExportField.getText();
			curOpenPath		 = pathOpenField.getText();
			curAlpha		 = alpaInterCheckBox.isSelected();
			curDisplayBord	 = displayBordersCB.isSelected();
			curDisplayCode	 = displayCodePanelCB.isSelected();
			curCheckVers	 = checkNewVersion.isSelected();
			curAutoUpdte	 = codeAutoUpdateCB.isSelected();
			curYScale		 = displayYScaleCB.isSelected();
			curXScale		 = displayXScaleCB.isSelected();
			curDisplayGrid	 = displayGridCB.isSelected();
			curMagnetic	 	 = magneticGridCB.isSelected();
			curColRender	 = colorRenderCheckBox.isSelected();
			curRendering	 = renderingCheckBox.isSelected();
			curAntiAlias	 = antialiasingCheckBox.isSelected();
			curLatexDistrib  = pathLatexDistribField.getText();
			curLatexIncludes = latexIncludes.getText();
		}
		catch(Exception e) { e.printStackTrace(); }
		
		super.setVisible(b);
	}
}
