//LabPlot Plot2DSurface.cc

#include <math.h>
#include <fstream>
#include <kdebug.h>
#include <kconfig.h>
#include "Plot2DSurface.h"
#include "defs.h"

using namespace std;

// density and contour 2D Plot class
Plot2DSurface::Plot2DSurface(Worksheet *p)
	: Plot2D(p) {
	kdDebug()<<"Plot2DSurface()"<<endl;

	// Symbol ?
	MainWin *mw=p->getMainWin();

	KConfig *config = mw->Config();
	config->setGroup( "Plot Surface Style" );
	densityenabled=config->readBoolEntry("Density Enabled",true);
	contourenabled=config->readBoolEntry("Contour Enabled",true);
	number=config->readNumEntry("Contour Level",10);
//	palette=config->readNumEntry("Palette",0);
	QString colorscale=config->readEntry("Colorscale",0);
	contour_color=config->readColorEntry("Contour Color",&Qt::black);
	colored_contour=config->readBoolEntry("Colored Contour",false);
	mesh=config->readBoolEntry("Show Mesh",false);
	relative=config->readBoolEntry("Relative Colorscale",true);
	brush=config->readNumEntry("Density Brush",1);
	threshold=config->readDoubleNumEntry("Threshold",-INF);

	// room for legend
	int xmax = (int)(worksheet->width()*(size.X()*p2.X()+position.X()));
	setXMax(xmax-(int)(120*size.X()),worksheet->width());

#ifdef HAVE_GL
	cv.clear();
	Qwt3D::RGBA rgb;

	// use colorscale
	ifstream file((const char*)colorscale.local8Bit());
	
	if(file) {
		while ( file ) {
			file >> rgb.r >> rgb.g >> rgb.b;
			file.ignore(1000,'\n');
			if (!file.good())
				break;
			else {
				rgb.a = 1;
				rgb.r /= 255;
				rgb.g /= 255;
				rgb.b /= 255;
				cv.push_back(rgb);
			}
		}
	}
	else {
		for(int i=0;i<255;i++) {
			rgb.a = 1;
			rgb.r = i/255.0;
			rgb.g = 40.0/255.0;
			rgb.b = 1.0-i/255.0;
			cv.push_back(rgb);
		}
	}
#endif
}

void Plot2DSurface::saveSurfaceXML(QDomDocument doc, QDomElement plottag) {
	QDomElement tag = doc.createElement( "DensityEnabled" );
	plottag.appendChild( tag );
	QDomText t = doc.createTextNode( QString::number(densityenabled) );
	tag.appendChild( t );
	tag = doc.createElement( "ContourEnabled" );
	plottag.appendChild( tag );
	t = doc.createTextNode( QString::number(contourenabled) );
	tag.appendChild( t );
	tag = doc.createElement( "ContourColor" );
	plottag.appendChild( tag );
	t = doc.createTextNode( contour_color.name() );
	tag.appendChild( t );
	tag = doc.createElement( "ColoredContour" );
	plottag.appendChild( tag );
	t = doc.createTextNode( QString::number(colored_contour) );
	tag.appendChild( t );
	tag = doc.createElement( "Mesh" );
	plottag.appendChild( tag );
	t = doc.createTextNode( QString::number(mesh) );
	tag.appendChild( t );
	tag = doc.createElement( "Relative" );
	plottag.appendChild( tag );
	t = doc.createTextNode( QString::number(relative) );
	tag.appendChild( t );
	tag = doc.createElement( "Brush" );
	plottag.appendChild( tag );
	t = doc.createTextNode( QString::number(brush) );
	tag.appendChild( t );
	tag = doc.createElement( "Level" );
	plottag.appendChild( tag );
	t = doc.createTextNode( QString::number(number) );
	tag.appendChild( t );
	tag = doc.createElement( "Threshold" );
	plottag.appendChild( tag );
	t = doc.createTextNode( QString::number(threshold) );
	tag.appendChild( t );

	//color scale
#ifdef HAVE_GL
	QDomElement cvtag = doc.createElement( "ColorVector" );
	plottag.appendChild( cvtag );
	for(unsigned int i=0;i<cv.size();i++) {
		tag = doc.createElement( "Color" );
		tag.setAttribute("r",QString::number(cv[i].r));
		tag.setAttribute("g",QString::number(cv[i].g));
		tag.setAttribute("b",QString::number(cv[i].b));
		tag.setAttribute("a",QString::number(cv[i].a));
		cvtag.appendChild( tag );
	}
#endif
}

void Plot2DSurface::openSurfaceXML(QDomElement e) {
	if(e.tagName() == "DensityEnabled")
		densityenabled = (bool) e.text().toInt();
	else if(e.tagName() == "ContourEnabled")
		contourenabled = (bool) e.text().toInt();
	else if(e.tagName() == "ContourColor")
		contour_color = QColor(e.text());
	else if(e.tagName() == "ColoredContour")
		colored_contour = (bool) e.text().toInt();
	else if(e.tagName() == "Mesh")
		mesh = (bool) e.text().toInt();
	else if(e.tagName() == "Relative")
		relative = (bool) e.text().toInt();
	else if(e.tagName() == "Brush")
		brush = e.text().toInt();
	else if(e.tagName() == "Level")
		number = e.text().toInt();
	else if(e.tagName() == "Threshold")
		threshold = e.text().toDouble();

#ifdef HAVE_GL
	if(e.tagName() == "ColorVector") {
		QDomNode node = e.firstChild();
		cv.clear();
		Qwt3D::RGBA rgb;
		while(!node.isNull()) {
			QDomElement e = node.toElement();
			rgb.r = e.attribute("r").toDouble();
			rgb.g = e.attribute("g").toDouble();
			rgb.b = e.attribute("b").toDouble();
			rgb.a = e.attribute("a").toDouble();
			// kdDebug()<<"RGBA : "<<rgb.r<<' '<<rgb.g<<' '<<rgb.b<<' '<<rgb.a<<endl;
			cv.push_back(rgb);
			node = node.nextSibling();
		}
	}
#endif
}

QStringList Plot2DSurface::Info() {
	kdDebug()<<"Plot2DSurface::Info()"<<endl;
	QStringList s;
	s<<"SURFACE";
	s<<QString::number(position.X())+QString(" , ")+QString::number(position.Y());
	s<<QString::number(size.X())+QString(" X ")+QString::number(size.Y());
	if (transparent)
		s<<QString("yes");
	else
		s<<QString("no");
	s<<bgcolor.name();
	s<<gbgcolor.name();

	return s;
}

void Plot2DSurface::drawLegend(QPainter *p, int x, int y) {
	kdDebug()<<"Plot2DSurface::drawLegend() : x/y "<<x<<' '<<y<<endl;
	int nr_graphs = graphlist->Number();

	if(densityenabled) {
		for (int i=0;i<255;i++) {
			if(legend.getOrientation()) {
				int tmpx=x+(int)((10+i/2)*size.X());
				p->setPen(Color(255-i));
				p->drawLine(tmpx, (int)(y+30*size.Y()*nr_graphs),tmpx,(int)(y+(30+25*nr_graphs)*size.Y()));
			}
			else {
				int tmpy=y+(int)((30*nr_graphs+i/2)*size.Y());
				p->setPen(Color(255-i));
				p->drawLine((int)(x+5*size.X()),tmpy,(int)(x+30*size.X()),tmpy);
			}
		}
	}

	p->setPen(Qt::black);
	int level = Number();
	// draw contour
	if (contourenabled) {
		for (int i=0;i<level;i++) {
			if(legend.getOrientation()) {
				if (colored_contour)
					p->setPen(Color((int)(255-i*255.0/level)));
				int tmpx = (int) (x+size.X()*(10+255/(double)(level-1)*i/2));
				p->drawLine(tmpx,y+(int)(30*size.Y()*nr_graphs),tmpx,y+(int)((30+25*nr_graphs)*size.Y()));
			}
			else {
				if (colored_contour)
					p->setPen(Color((int)(255-i*255.0/level)));
				int tmpy = (int) (y+size.Y()*(30*nr_graphs+255/(double)(level-1)*i/2));
				p->drawLine(x+(int)(5*size.X()),tmpy,x+(int)(30*size.X()),tmpy);
			}
		}
	}

	// draw label
	p->setPen(Qt::black);
	if (level>11)
		level = 11;
	for (int i=0;i<level;i++) {
		LRange r;
		if(graphlist->getStruct(0)==GRAPHM) {
			GraphM *g = graphlist->getGraphM(0);
			r = g->Range(2);
		}
		else if (graphlist->getStruct(0)==GRAPH3D) {
			Graph3D *g = graphlist->getGraph3D(0);
			r = g->Range(2);
		}
		double zmax, zmin;
		if(Relative()) {
			zmin = r.rMin();
			zmax = r.rMax();
		}
		else {
			zmin = - fmax(fabs(r.rMax()),fabs(r.rMin()));
			zmax = fmax(fabs(r.rMax()),fabs(r.rMin()));
		}

		// resize font for contour level in legend with the size
		QFont tmpfont = legend.Font();
		tmpfont.setPointSize((int)(legend.Font().pointSize()*size.X()));
		p->setFont(tmpfont);
		
		double value = zmax-i/(double)(level-1)*(zmax-zmin);
		QString string = QString::number(value,'g',3);
		
		// used for legend border
		QFontMetrics fm = p->fontMetrics();
		int w = fm.width(string);
		if(w > legend.TicLabelLength())
			legend.setTicLabelLength(w);
		
		if(legend.getOrientation())
			p->drawText(x+(int)(size.X()*(6+(255/(level-1))*i/2)),y+(int)((50+25*nr_graphs)*size.Y()),string);
		else
			p->drawText(x+(int)(40*size.X()),y+(int)(size.Y()*(36*nr_graphs+(255/(level-1))*i/2)),string);
	}
}

int Plot2DSurface::graph_segment(QPainter *p,double x1,double y1, double z1, double x2, double y2, double z2,
		double x3, double y3, double z3, double level) {
	double factor;
	double x,y,xa,ya,xb,yb,tmp;

	if(z1>=z2) {
		tmp=x1; x1=x2; x2=tmp;
		tmp=y1; y1=y2; y2=tmp;
		tmp=z1; z1=z2; z2=tmp;
	}
	if(z2>z3) {
		if(z3<z1) {
			tmp=x1; x1=x3; x3=tmp;
			tmp=y1; y1=y3; y3=tmp;
			tmp=z1; z1=z3; z3=tmp;
		}
		tmp=x2; x2=x3; x3=tmp;
		tmp=y2; y2=y3; y3=tmp;
		tmp=z2; z2=z3; z3=tmp;
	}

	if(level<z1 || level > z3) {	// important
		return 0;
	}
	// not needed
	/*if(level == z1 && z1 == z2 && z2 != z3) {
		p.drawLine((int)(x1*FACTOR),(int)(y1*FACTOR),(int)(x2*FACTOR),(int)(y2*FACTOR));
	}
	if(level == z3 && z3 == z2 && z2 != z1) {
		p.drawLine((int)(x3*FACTOR),(int)(y3*FACTOR),(int)(x2*FACTOR),(int)(y2*FACTOR));
	}*/
	if (level == z2 ) {
		if( z3 == z1)
			return 0;
		factor = (level-z1)/(z3-z1);
		x=x1+(x3-x1)*factor;
		y=y1+(y3-y1)*factor;
		p->drawLine((int)(x),(int)(y),(int)(x2),(int)(y2));
	}
	else if (level<z2) {
		factor = (level-z1)/(z2-z1);
		xa=x1+(x2-x1)*factor;
		ya=y1+(y2-y1)*factor;
		factor = (level-z1)/(z3-z1);
		xb=x1+(x3-x1)*factor;
		yb=y1+(y3-y1)*factor;
		p->drawLine((int)(xa),(int)(ya),(int)(xb),(int)(yb));
	} else {
		factor = (level-z1)/(z3-z1);
		xa=x1+(x3-x1)*factor;
		ya=y1+(y3-y1)*factor;
		factor = (level-z2)/(z3-z2);
		xb=x2+(x3-x2)*factor;
		yb=y2+(y3-y2)*factor;
		p->drawLine((int)(xa),(int)(ya),(int)(xb),(int)(yb));
	}
	return 0;
}

void Plot2DSurface::drawCurves(QPainter *p, int w, int h) {
	kdDebug()<<"Plot2DSurface::drawCurves()"<<endl;
	int xmin = (int)(w*(size.X()*p1.X()+position.X()));
	int xmax = (int)(w*(size.X()*p2.X()+position.X()));
	int ymin = (int)(h*(size.Y()*p1.Y()+position.Y()));
	int ymax = (int)(h*(size.Y()*p2.Y()+position.Y()));
	kdDebug()<<"		xmin/xmax ymin/ymax : "<<xmin<<'/'<<xmax<<' '<<ymin<<'/'<<ymax<<endl;

	for (unsigned int i=0; i < graphlist->Number() ; i++) {
		if(graphlist->getGraph(i)->isShown() == false)
			continue;

		LRange r[3];
		int NX, NY;
		double *a;
		if ( graphlist->getStruct(i) == GRAPHIMAGE ) {
			QPixmap pm(graphlist->getGraphIMAGE(i)->Pixmap());
			
			p->save();	// scale painter so that pixmap fits into drawing area
			p->translate(xmin,ymin);
			p->scale((xmax-xmin)/(double)pm.width(),(ymax-ymin)/(double)pm.height());
			p->drawPixmap(0,0,pm);
			p->restore();
			continue;
		}
		else if ( graphlist->getStruct(i) == GRAPH3D ) {
			Graph3D *g = graphlist->getGraph3D(i);
			Point3D *data = g->Data();
			a = new double[g->Number()];
			NX = g->NX();
			NY = g->NY();
			kdDebug()<<"NX/NY = "<<NX<<' '<<NY<<endl;
			kdDebug()<<"Number = "<<g->Number()<<endl;
			r[0]= g->Range(0);
			r[1]= g->Range(1);
			r[2]= g->Range(2);

			double dx=r[0].rMax()-r[0].rMin();
			double dy=r[1].rMax()-r[1].rMin();
			kdDebug()<<"	 dx/dy = "<<dx<<' '<<dy<<endl;
			for(int i=0;i<g->Number();i++) {
				int indexx = (int) round((double)(NX-1)*(data[i].X()-r[0].rMin())/dx);
				int indexy = (int) round((double)(NY-1)*(data[i].Y()-r[1].rMin())/dy);
//				kdDebug()<<"	DATA : X/Y "<<data[i].X()<<' '<<data[i].Y()<<endl;
//				kdDebug()<<"	INDEX X/Y = "<<indexx<<' '<<indexy<<endl;
//				kdDebug()<<"	index="<<g->Number()-1-indexx+indexy*NX<<endl;
				
				int index = indexx+indexy*NX;
				if(index >=0 &&  index<g->Number())
					a[indexx+indexy*NX] = data[i].Z();
				else
					kdDebug()<<"	CONVERSION ERROR in Surface from 3D data!"<<endl;
				// unsorted
				//a[i]=data[i].Z();
			}
		}
		else {
			GraphM *g = graphlist->getGraphM(i);
			kdDebug()<<"		GRAPHM Type = "<<g->Type()<<endl;
			NX = g->NX();
			NY = g->NY();
			a = g->Data();

			r[0]= g->Range(0);
			r[1]= g->Range(1);
			r[2]= g->Range(2);
		}
			
		double minx = actrange[0].rMin();
		double maxx = actrange[0].rMax();
		double miny = actrange[1].rMin();
		double maxy = actrange[1].rMax();

		int xold=xmin, yold=ymin;
		double zmin=r[2].rMin(),zmax=r[2].rMax();

		kdDebug()<<"		minx/maxx = "<<minx<<"/"<<maxx<<endl;
		kdDebug()<<"		miny/maxy = "<<miny<<"/"<<maxy<<endl;

		double rxmin=r[0].rMin(),rxmax=r[0].rMax();
		double rymin=r[1].rMin(),rymax=r[1].rMax();
		
		// density plot
		if(densityenabled) {
			int y=0;
			for (int i=0;i<=NY;i++) {
				if(worksheet->getMainWin()->speedMode()) {
					int mod = (int) (NY/sqrt((double) worksheet->getMainWin()->speedModeValue()));
					if(mod==0) mod=1;
					if(i%mod != 0) continue;	// speed mode
				}
				switch(axis[1].Scale()) {
				case LINEAR:
					// only i = NY*(1-(maxy-r[1].rMin())/(r-rMax()-r.rMin())) ..
					//	NY(1-(miny-r[1].rMin())/(r-rMax()-r.rMin())) needed , see scalesurface.nb
					y = ymax-1 - (int)((rymin-miny+i/(double)(NY)*(rymax-rymin))*(ymax-ymin)/(maxy-miny));
					break;
				case LOG10:
					y = ymax - (int)((rymin-miny+log10((double)i+1)/log10((double)NY+1)*
						(rymax-rymin))*(ymax-ymin)/(maxy-miny));
					break;
				case LOG2:
					y = ymax - (int)((rymin-miny+log2((double)i+1)/log2((double)NY+1)*
						(rymax-rymin))*(ymax-ymin)/(maxy-miny));
					break;
				case LN:
					y = ymax - (int)((rymin-miny+log((double)i+1)/log((double)NY+1)*
						(rymax-rymin))*(ymax-ymin)/(maxy-miny));
					break;
				case SQRT:
					y = ymax - (int)((sqrt(miny)+sqrt((double)i)) * (ymax-ymin)/(sqrt(maxy)-sqrt(miny)));
					break;
				case SX2:
					y = ymax - (int)((miny*miny+i*i) * (ymax-ymin)/(maxy*maxy-miny*miny));
					break;
				}

				if(i==0 || i==NY)
					kdDebug()<<"		y("<<i<<")="<<y<<endl;

				int x=0;
				for (int j=0;j<=NX;j++) {
					if(worksheet->getMainWin()->speedMode()) {
						int mod = (int) (NX/sqrt((double) worksheet->getMainWin()->speedModeValue()));
						if(mod==0) mod=1;
						if(j%mod != 0) continue;	// speed mode
					}
					switch (axis[0].Scale()) {
					case LINEAR:
						x = xmin + (int)((rxmin-minx+j/(double)(NX)*(rxmax-rxmin))
							*(xmax-xmin)/(maxx-minx));
						break;
					case LOG10:
						x = xmin + (int)((rxmin-minx+log10((double)j+1)/log10((double)NX+1)*(rxmax-rxmin))*
							(xmax-xmin)/(maxx-minx));
						break;
					case LOG2:
						x = xmin + (int)((rxmin-minx+log2((double)j+1)/log2((double)NX+1)*(rxmax-rxmin))*
							(xmax-xmin)/(maxx-minx));
						break;
					case LN:
						x = xmin + (int)((rxmin-minx+log((double)j+1)/log((double)NX+1)*(rxmax-rxmin))*
							(xmax-xmin)/(maxx-minx));
						break;
					case SQRT:
						x = xmin + (int)(sqrt(j-minx)*(xmax-xmin)/sqrt(maxx-minx));
						break;
					case SX2:
						x = xmin + (int)((j-minx)*(j-minx)*(xmax-xmin)/((maxx-minx)*(maxx-minx)));
						break;
					}

					if(i==0 && (j==0 || j==NX) )
						kdDebug()<<"		x("<<j<<")="<<x<<endl;
					//kdDebug()<<"	DENSITY : x = "<<x<<" y = "<<y<<" (i="<<i<<",j="<<j<<") "<<endl;

					if(i>0 && j>0) {
						// dont draw empty pixel
						if(a[j-1+NX*(i-1)] < zmin || a[j-1+NX*(i-1)] > zmax)
							continue;
						
						// value = 0 .. 255
						// ??? : i=0,j=0 : a[-1-NX] ???
						int value;
						if (relative)
							value = (int) (255.0*(a[j-1+NX*(i-1)]-zmin)/(zmax-zmin));
						else
							value = (int) (127.0*a[j-1+NX*(i-1)]/fmax(fabs(zmax),fabs(zmin))+127.0);

						
						//kdDebug()<<"	max = "<<zmax<<"/ min = "<<zmin<<endl;
//						kdDebug()<<"	a = "<<a[j-1+NX*(i-1)]<<endl;
//						kdDebug()<<"	color value ("<<i<<"/"<<j<<") = "<<value<<endl;

						QBrush qbrush(Color(value),(Qt::BrushStyle)brush);

						p->setBrush(qbrush);
						if(mesh)
							p->setPen( Qt::black );
						else
							p->setPen( Qt::NoPen );

						int tmpy=y-yold,tmpx=x-xold;
						//if (i==1)
						//	tmpy+=2;
						//if (j==NX)
						//	tmpx--;

						// fill border rects to border
						if (xold<=xmin-1 && xold+tmpx>=xmin-1)
							xold=xmin-1;
						else if (xold+tmpx>=xmax+1 && xold<=xmax+1)
							tmpx=xmax+1-xold;
						if (yold<=ymin-1 && yold+tmpy>=ymin-1)
							yold=ymin-1;
						else if (yold+tmpy>=ymax+1 && yold<=ymax+1)
							tmpy=ymax+1-yold;

						if(xold >= xmin-1  && tmpx+xold <= xmax+1 &&
							yold >= ymin-1 &&  tmpy+yold <= ymax+1) {
//		kdDebug()<<" Draw Rect ("<<xold+1<<","<<yold<<" / size ("<<tmpx<<","<<tmpy<<")"<<endl;
							if (a[j-1+NX*(i-1)]>= threshold)
								p->drawRect(xold+1,yold,tmpx,tmpy);
						}
					}
					xold=(int)x;
				}
				yold=(int)y;
			}
		}

//		kdDebug()<<"DENSITY OK"<<endl;

		// contour plot
		if (contourenabled) {
			double *x,*y;
		 	x = new double[NX];
 			y = new double[NY];

			switch (axis[0].Scale()) {
			case LINEAR:
				for(int i=0;i<NX;i++)
					x[i] = xmin + (rxmin-minx+i/(double)(NX-1)*(rxmax-rxmin))*(xmax-xmin)/(maxx-minx);
				break;
			case LOG10:
				for(int i=0;i<NX;i++)
					x[i] = xmin + (rxmin-minx+log10((double)i+1)/log10((double)NX)*(rxmax-rxmin))*
							(xmax-xmin)/(maxx-minx);
				break;
			case LOG2:
				for(int i=0;i<NX;i++)
					x[i] = xmin + (rxmin-minx+log2((double)i+1)/log2((double)NX)*(rxmax-rxmin))*
							(xmax-xmin)/(maxx-minx);
				break;
			case LN:
				for(int i=0;i<NX;i++)
					x[i] = xmin + (rxmin-minx+log((double)i+1)/log((double)NX)*(rxmax-rxmin))*
							(xmax-xmin)/(maxx-minx);
				break;
			case SQRT:
				for(int i=0;i<NX;i++)
					x[i] = xmin + sqrt(i+1-minx)*(xmax-xmin-1)/sqrt(maxx-minx);
				break;
			case SX2:
				for(int i=0;i<NX;i++)
					x[i] = xmin + (i+1-minx)*(i+1-minx)*(xmax-xmin-1)/((maxx-minx)*(maxx-minx));
				break;
			}

			switch(axis[1].Scale()) {
			case LINEAR:
				for(int j=0;j<NY;j++)
					y[j] = ymax - (rymin-miny+j/(double)(NY-1)*(rymax-rymin))*(ymax-ymin)/(maxy-miny);
				break;
			case LOG10:
				for(int j=0;j<NY;j++)
					y[j] = ymax - (rymin-miny+log10((double)j+1)/log10((double)NY)*
						(rymax-rymin))*(ymax-ymin)/(maxy-miny);
				break;
			case LOG2:
				for(int j=0;j<NY;j++)
					y[j] = ymax - (rymin-miny+log2((double)j+1)/log2((double)NY)*
						(rymax-rymin))*(ymax-ymin)/(maxy-miny);
				break;
			case LN:
				for(int j=0;j<NY;j++)
					y[j] = ymax - (rymin-miny+log((double)j+1)/log((double)NY)*
						(rymax-rymin))*(ymax-ymin)/(maxy-miny);
				break;
			case SQRT:
				for(int j=0;j<NY;j++)
					y[j] = ymin + sqrt(j-1+miny) * (ymax-ymin)/sqrt(maxy-miny);
				break;
			case SX2:
				for(int j=0;j<NY;j++)
					y[j] = ymin + (j-1+miny)*(j-1+miny) * (ymax-ymin)/((maxy-miny)*(maxy-miny));
				break;
			}

			double diff = (zmax-zmin)/number;

			//kdDebug()<<"	diff = "<<diff<<endl;
			//kdDebug()<<"	zmin/zmax = "<<zmin<<' '<<zmax<<endl;

			for (int i=0;i<NY-1;i++) {
				if(worksheet->getMainWin()->speedMode()) {
					int mod = (int) (NY/sqrt((double) worksheet->getMainWin()->speedModeValue()));
					if(mod==0) mod=1;
					if(i%mod != 0) continue;	// speed mode
				}

				double yavg = (y[i]+y[i+1])/2.0;
				for (int j=0;j<NX-1;j++) {
					if(worksheet->getMainWin()->speedMode()) {
						int mod = (int) (NX/sqrt((double) worksheet->getMainWin()->speedModeValue()));
						if(mod==0) mod=1;
						if(j%mod != 0) continue;	// speed mode
					}
					
					double xavg = (x[j]+x[j+1])/2.0;
					//kdDebug()<<" SURFACE : x = "<<x[j]<<" y = "<<y[i]<<" (i="<<i<<",j="<<j<<") "<<endl;

					double zij = a[i*NX+j];
					double zipj = a[i*NX+j+1];
					double zijp = a[(i+1)*NX+j];
					double zipjp = a[(i+1)*NX+j+1];

					//kdDebug()<<zij<<' '<<zipj<<endl;
					//kdDebug()<<zijp<<' '<<zipjp<<endl<<endl;

					double zavg=(zij+zipj+zijp+zipjp)/4.0;

					p->setPen(contour_color);

					for (double level=zmin;level<zmax;level+=diff) {
						if (colored_contour) {
							// value = 0 .. 255
							int value;
							if (relative)
								value = (int)((level-zmin)/(zmax-zmin)*255.0);
							else
								value = (int)(127.0*(level)/fmax(fabs(zmax),fabs(zmin))+127.0);
							p->setPen(Color(value));
						}

						// TODO : fill border rects to border
/*						if (xold<=xmin-1 && xold+tmpx>=xmin-1)
							xold=xmin-1;
						else if (xold+tmpx>=xmax+1 && xold<=xmax+1)
							tmpx=xmax+1-xold;
						if (yold<=ymin-1 && yold+tmpy>=ymin-1)
							yold=ymin-1;
						else if (yold+tmpy>=ymax+1 && yold<=ymax+1)
							tmpy=ymax+1-yold;
*/
						if (x[j]>=xmin && x[j]<=xmax && x[j+1]>= xmin && x[j+1] <= xmax
							&& y[i]>=ymin && y[i]<=ymax && y[i+1]>=ymin && y[i+1]<=ymax) {
							if (zavg> threshold) {
								graph_segment(p,x[j],y[i],zij,x[j],y[i+1],zijp,xavg,yavg,zavg,level);
								graph_segment(p,x[j],y[i],zij,x[j+1],y[i],zipj,xavg,yavg,zavg,level);
								graph_segment(p,x[j+1],y[i],zipj,x[j+1],y[i+1],zipjp,xavg,yavg,zavg,level);
								graph_segment(p,x[j],y[i+1],zijp,x[j+1],y[i+1],zipjp,xavg,yavg,zavg,level);
							}
						}
					}
				}
			}
		}
	}
}

QColor Plot2DSurface::Color(int value) {
	if(value <= 0)
		value=1;
#ifdef HAVE_GL
//	kdDebug()<<"r/g/b "<<255*cv[value-1].r<<' '<<255*cv[value-1].g<<' '<<255*cv[value-1].b<<endl;
	return QColor((int) (255*cv[value-1].r),(int)(255*cv[value-1].g),(int)(255*cv[value-1].b));
#else
	return QColor();
#endif
}
