//LabPlot : ExportDialog.cc

#include <stdlib.h>
#include <stdio.h>
#include <iostream>
#include <math.h>
#include <qlabel.h>
#include <qhbox.h>
#include <qfiledialog.h>
#include <qwizard.h>

#include <klocale.h>
#include <kfilterbase.h>
#include <kfilterdev.h>
#include <kmessagebox.h>
#include <kimageio.h>
#include <ktempfile.h>
#include "ExportDialog.h"
#include "FilterCDF.h"
#include "FilterAUDIOFILE.h"
#include "Plot2DSurface.h"
#include "export.h"

#ifdef HAVE_CDF
#include <cdf.h>
#endif
#include <netcdf.h>
#include <audiofile.h>
#ifdef HAVE_MAGICK
#include <Magick++.h>
using namespace Magick;
#endif

using namespace std;

ExportDialog::ExportDialog(MainWin *mw, const char *name, int item)
	: Dialog(mw, name), item(item)
{
	setCaption(i18n("Export Data")+i18n(" : ")+QString(name));
	KConfig *config = mw->Config();
	config->setGroup( "Export" );
	
	plot=0;
	if(item>=0) {
		Worksheet *w = mw->activeWorksheet();
		if(w)
			plot = w->getPlot(w->API());
	}


	QHBox *hb = new QHBox(vbox);
	new QLabel(i18n("Filename (append .gz or .bz2 for compression) : "),hb);
	hb = new QHBox(vbox);
	filele = new KLineEdit(config->readEntry("Filename",i18n("out.dat")),hb);
	KPushButton *newFile = new KPushButton(i18n("Browse"),hb);
	QObject::connect(newFile,SIGNAL(clicked()),SLOT(selectFile()));

	// TODO : item<0 (spreadsheet) : export header checkbox
	if(item<0) {	// spreadsheet
		QHBox *hb = new QHBox(vbox);
		headercb = new QCheckBox(i18n("Export Header"),hb);
		headercb->setChecked(config->readBoolEntry("Header",false));
	}
	else
			headercb=0;
	
	hb = new QHBox(vbox);
	new QLabel(i18n("Start Row : "),hb);
	startrow = new KLineEdit(config->readEntry("Startrow","1"),hb);
	startrow->setValidator(new QIntValidator(startrow));
	new QLabel(i18n(" End Row : "),hb);
	endrow = new KLineEdit(config->readEntry("Endrow",i18n("END")),hb);
	// no validator here -> make "END" possible

	hb = new QHBox(vbox);
	new QLabel(i18n("Export to : "),hb);
        exportcb = new KComboBox(hb);
//	if ( (plot && plot->Type() == PSURFACE) || item == -1) 
//		elist<<i18n("IMAGE");
	exportcb->insertStrList(exports);
	QObject::connect(exportcb,SIGNAL(activated (int)),SLOT(updateOptions(int)));
	exportcb->setCurrentItem(config->readNumEntry("Format",0));
	
	// ASCII	TODO : use SEPARATOR
	hb = new QHBox(vbox);
	slabel = new QLabel(i18n("Separating character : "),hb);
        sc= new KComboBox(hb);
	QStringList slist;
	slist << i18n("SPACE");
	slist << i18n("TAB");
	slist << i18n(",");
	sc->insertStringList(slist);
	sc->setCurrentItem(config->readNumEntry("SeparatingCharacter",0));

	// IMAGE
	hb = new QHBox(vbox);
	flabel = new QLabel(i18n("Format : "),hb);
        ic = new KComboBox(hb);
	QStringList ilist;
	
	for ( unsigned int i = 0; i < QImageIO::outputFormats().count(); i++ ) {
        	QString str = QString( QImageIO::outputFormats().at( i ) );
		ilist<<str;
	}

	// append ImageMagick formats
#ifdef HAVE_MAGICK
	list<CoderInfo> coderList;
	coderInfoList( &coderList,CoderInfo::AnyMatch,CoderInfo::TrueMatch,CoderInfo::AnyMatch);
	list<CoderInfo>::iterator entry = coderList.begin();
	while( entry != coderList.end() )  {
		ilist<<QString(entry->name().c_str());
		//kdDebug()<<QString(entry->name().c_str())+" : ("+QString(entry->description().c_str())+")"<<endl;
		entry++;
	}
#endif

	ic->insertStringList(ilist);
	QObject::connect(ic,SIGNAL(activated (int)),SLOT(updateImageFormat(int)));
	ic->setCurrentItem(config->readNumEntry("ImageFormat",0));

#ifdef HAVE_CDF
	// cdf
	hb = new QHBox(vbox);
	cdfcomlabel = new QLabel(i18n("Compression : "),hb);
        cdfcomcb = new KComboBox(hb);
	QStringList clist;
	FilterCDF cdf;
	for ( unsigned int i = 0; i < 6; i++ )
        	clist << cdf.Comp(i);
	cdfcomcb->insertStringList(clist);
	cdfcomcb->setCurrentItem(config->readNumEntry("CDFCompression",0));

	hb = new QHBox(vbox);
	cdfenclabel = new QLabel(i18n("Encoding : "),hb);
        cdfenccb = new KComboBox(hb);
	QStringList enclist;
	for ( unsigned int i = 1; i < 17; i++ )
        	enclist << cdf.Enc(i);
	cdfenccb->insertStringList(enclist);
	cdfenccb->setCurrentItem(config->readNumEntry("CDFEncoding",0));
#endif
	
	// netcdf : nothing
	
	// audio
	hb = new QHBox(vbox);
	audioformatlabel = new QLabel(i18n("Format : "),hb);
	audioformatcb= new KComboBox(hb);
	QStringList alist;
	FilterAUDIOFILE auf;
	alist<<QString("RAW")<<QString("AIFFC")<<QString("AIFF")<<QString("SND");
	alist<<QString("WAV")<<QString("BICSF");
	audioformatcb->insertStringList(alist);
	QObject::connect(audioformatcb,SIGNAL(activated (int)),SLOT(updateAudioFormat(int)));
	audioformatcb->setCurrentItem(config->readNumEntry("AudioFormat",0));
	hb = new QHBox(vbox);
	samplelabel = new QLabel(i18n("Sample rate : "),hb);
	samplele = new KLineEdit(config->readEntry("SampleRate","44100"),hb);
	samplele->setValidator(new QIntValidator(samplele));

	// binary
	hb = new QHBox(vbox);
        binarytypecb= new KComboBox(hb);
	QStringList blist;
	blist<<QString("double")<<QString("float")<<QString("long");
	blist<<QString("int (8 Bit)")<<QString("int (16 Bit)")<<QString("int (32 Bit)")<<QString("int (64 Bit)");
	binarytypecb->insertStringList(blist);
	binarytypecb->setCurrentItem(config->readNumEntry("BinaryType",0));

	// TODO : only selected region
	
	updateOptions(0);
	
	QObject::connect(ok,SIGNAL(clicked()),SLOT(ok_clicked()));
	QObject::connect(apply,SIGNAL(clicked()),SLOT(apply_clicked()));
	QObject::connect(save,SIGNAL(clicked()),SLOT(saveSettings()));
	
	setMinimumWidth(vbox->minimumSizeHint().width());
	setMinimumHeight(gbox->minimumSizeHint().height()+vbox->minimumSizeHint().height());
	resize(minimumSize());
}

void ExportDialog::saveSettings() {
	KConfig *config = mw->Config();
	config->setGroup( "Export" );

	config->writeEntry("Filename", filele->text());
	if(headercb != 0)
		config->writeEntry("Header",headercb->isChecked());
	config->writeEntry("Startrow",startrow->text());
	config->writeEntry("Endrow",endrow->text());
	config->writeEntry("Format",exportcb->currentItem());
	config->writeEntry("SeparatingCharacter",sc->currentText());
	config->writeEntry("ImageFormat",ic->currentItem());
	config->writeEntry("CDFCompression",cdfcomcb->currentItem());
	config->writeEntry("CDFEncoding",cdfenccb->currentItem());
	config->writeEntry("AudioFormat",audioformatcb->currentItem());
	config->writeEntry("SampleRate",samplele->text());
	config->writeEntry("BinaryType",binarytypecb->currentItem());
}

//! update options when changing exported format
void ExportDialog::updateOptions(int item) {
	filele->setReadOnly(false);
	startrow->setReadOnly(false);
	endrow->setReadOnly(false);

	slabel->hide();sc->hide();
	flabel->hide();ic->hide();
#ifdef HAVE_CDF
	cdfcomlabel->hide();cdfcomcb->hide();
	cdfenclabel->hide();cdfenccb->hide();
#endif
	audioformatlabel->hide();audioformatcb->hide();
	samplelabel->hide();samplele->hide();
	binarytypecb->hide();
	QString filename =  filele->text();
	
	switch(item) {
	case EASCII: 
		slabel->show();sc->show(); 
		filename.replace( QRegExp(QString("[.]+.*")), ".dat" );	
		break;
#ifdef HAVE_CDF
	case ECDF:
		filename.replace( QRegExp(QString("[.]+.*")), ".cdf" );
		cdfcomlabel->show();cdfcomcb->show();
		cdfenclabel->show();cdfenccb->show();
		break;
#endif
	case ENETCDF:
		filename.replace( QRegExp(QString("[.]+.*")), ".nc" );
		break;
	case EAUDIO:
		filename.replace( QRegExp(QString("[.]+.*")), "." + audioformatcb->currentText().lower() );
		audioformatlabel->show();audioformatcb->show();
		samplelabel->show();samplele->show();
		break;
	case EIMAGE:
		flabel->show();ic->show();
		filename.replace( QRegExp(QString("[.]+.*")), "." + ic->currentText().lower() );
		break;
	case EBINARY:
		filename.replace( QRegExp(QString("[.]+.*")), ".bin" );	
		audioformatlabel->show();
		binarytypecb->show();
		break;
	case EKEXIDB:
		filele->setReadOnly(true);
		startrow->setReadOnly(true);
		endrow->setReadOnly(true);
		break;
	}
        
	filele->setText(filename);
}

void ExportDialog::selectFile() {
	QString f = QFileDialog::getSaveFileName(filele->text(),
			i18n("Data Files (*.dat *.DAT);;Binary Data (*.bin *.BIN)"), this );
	if (! f.isEmpty() )
		filele->setText(QString(f.latin1()));
}

int ExportDialog::apply_clicked() {
	QString filename = filele->text();

	// check if file exists
	if ( QFile::exists(filename) ) {
		int answer = KMessageBox::warningYesNoCancel( this, 
			i18n( "Overwrite\n\'%1\'?" ).arg( filename ), i18n("Export data"));
		if (answer != KMessageBox::Yes)
			return 1;
		else {
			// delete it (needed for CDF)
			QFile::remove(filename);
		}
	}

	// TODO : use SEPARATOR
	QString sep;
	switch (sc->currentItem()) {
	case 0: sep = QString(" "); break;
	case 1: sep = QString("	"); break;
	case 2: sep = QString(","); break;
	}

	QIODevice *file=0;
	QTextStream t;
	QDataStream d;
	int format = exportcb->currentItem();
	if(format == EASCII || format == EBINARY) {
		file = KFilterDev::deviceForFile(filename,QString::null,true);
		if(file == 0) file = new QFile(filename);

		if (! file->open( IO_WriteOnly )) {
			KMessageBox::error(this, i18n("Sorry. Could not open file for writing!"));
			return 0;
		}
	
		if(format == EASCII)
			t.setDevice(file);
		else if (format == EBINARY)
			d.setDevice(file);
	}

	switch (exportcb->currentItem()) {
	case EASCII: 	dumpASCII(&t, sep); break;
	case ECDF: dumpCDF(filename); break;
	case ENETCDF: dumpNETCDF(filename); break;
	case EAUDIO: 	dumpAUDIOFILE(filename); break;
	case EIMAGE: dumpIMAGE(filename); break;
	case EBINARY: dumpBINARY(&d); break;
	case EKEXIDB: dumpKexiDB(); break;	
	}

	if(file) file->close();

	return 0;
}

void ExportDialog::dumpASCII(QTextStream *t, QString sep) {
	int start = startrow->text().toInt();
	int end = endrow->text().toInt();

	if(item<0) {		// exporting data from spreadsheet
		kdDebug()<<"exporting spreadsheet "<<endl;
	 	QTable *table = s->Table();
		// export header
		if(headercb->isChecked()) {
			for(int i=0;i<table->numCols();i++) {
				if(i>0) *t<<sep;
				// OLD : *t<<table->horizontalHeader()->label(i);
				*t<<s->columnTitle(i);
			}
			*t<<endl;
		}
		// export data
		for (int i=start-1;i<table->numRows();i++) {
			if(end>0 && i>=end) return;
			for(int j=0;j<table->numCols();j++) {
				if(j>0) *t<<sep;
				*t<<table->text(i,j);
			}
			*t<<endl;
		}	
	}
	else {			// exporting data from GRAPH
		GraphList *gl = plot->getGraphList();
		GRAPHType st =  gl->getStruct(item);
		switch(st) {
		case GRAPH2D: {
			Graph2D *g = gl->getGraph2D(item);
			Point *ptr=g->Data();
			for (int i=start-1;i<g->Number();i++) {
				if(end>0 && i>end) return;
				
				*t<<ptr[i].X()<<sep<<ptr[i].Y()<<endl;
			}}
			break;
		case GRAPH3D: {
			Graph3D *g = gl->getGraph3D(item);
			Point3D *ptr=g->Data();
			for (int i=start-1;i< g->NX()*g->NY();i++) {
				if(end>0 && i>end) return;
			
				*t<<ptr[i].X()<<sep<<ptr[i].Y()<<sep<<ptr[i].Z()<<endl;
			}}
			break;
		case GRAPH4D: {
			Graph4D *g = gl->getGraph4D(item);
			Point4D *ptr=g->Data();
			for (int i=start-1;i< g->Number();i++) {
				if(end>0 && i>end) return;
			
				*t<<ptr[i].X()<<sep<<ptr[i].Y()<<sep<<ptr[i].Z()<<sep<<ptr[i].T()<<endl;
			}}
			break;
		case GRAPHM: {
			GraphM *g = gl->getGraphM(item);
			double *array = g->Data();
			int nx=g->NX(), ny = g->NY();
		
			for (int i=0;i<ny;i++) {
				for (int j=0;j<nx;j++) {
					if(j>0)
						*t<<sep;
					*t<<array[j+nx*i];
				}
				*t<<endl;
			}}
			break;
		case GRAPHIMAGE: {
			GraphIMAGE *g = gl->getGraphIMAGE(item);
			QPixmap pm = g->Pixmap();
			QImage image = pm.convertToImage();

			for (int i=0;i<pm.width();i++) {
				for (int j=0;j<pm.height();j++) {
					if(j>0)
						*t<<sep;
					*t<<qGray(image.pixel(i,j));
				}
				*t<<endl;
			}}
			break;
		default: break;
		}
	}
}

void ExportDialog::dumpCDF(QString filename) {
	kdDebug()<<"ExportDialog::dumpCDF("<<filename<<")"<<endl;
#ifdef HAVE_CDF
	int start = startrow->text().toInt()-1;
	int end = endrow->text().toInt();

	// TODO	-> FilterCDF ???
	CDFid id;
	long dimsizes[1]={1};
	// extension is automatically added, so we need to delete it first
	filename.replace( QRegExp(QString("[.]+.*")), "" );
	CDFcreate(filename.latin1(),1,dimsizes,cdfenccb->currentItem()+1,ROW_MAJOR,&id);
	// enable compression
	long parms[1];
	switch(cdfcomcb->currentItem()) {
		case 1: parms[0]=RLE_OF_ZEROs;
		case 2: parms[0]=OPTIMAL_ENCODING_TREES;
		case 3: parms[0]=OPTIMAL_ENCODING_TREES;
		case 5: parms[0]=9;
	}
	CDFlib(PUT_,CDF_COMPRESSION_,(long) cdfcomcb->currentItem(),parms);
	
	static long varys[1]={NOVARY};
	long indices[1]={0};
	long varnum1,varnum2,varnum3,varnum4;
	double v;
	
	if(item<0) {		// exporting data from spreadsheet
		kdDebug()<<"exporting spreadsheet "<<endl;
		Spreadsheet *s = mw->activeSpreadsheet();
	 	QTable *table = s->Table();
		for (int j=0;j<table->numCols();j++) {
			long varnum;
			// TODO : format (DOUBLE, INT, ...)
			CDFvarCreate(id,s->columnTitle(j).latin1(),CDF_DOUBLE,1,VARY,varys,&varnum);
			for(int i=0;i<=table->numRows()-start;i++) {
				if(end>0 && i+start>end) break;
				v=table->text(i+start,j).toDouble();
				CDFvarPut(id,varnum,i,indices,&v);
			}
		}	
	}
	else {
		GraphList *gl = plot->getGraphList();
		GRAPHType st =  gl->getStruct(item);
		switch(st) {
		case GRAPH2D: {
			Graph2D *g = gl->getGraph2D(item);
			Point *ptr=g->Data();
			CDFvarCreate(id,"x",CDF_DOUBLE,1,VARY,varys,&varnum1);
			CDFvarCreate(id,"y",CDF_DOUBLE,1,VARY,varys,&varnum2);
			for(int i=0;i<=g->Number()-start;i++) {
				if(end>0 && i+start>end) break;
				v=ptr[i+start].X();
				CDFvarPut(id,varnum1,i,indices,&v);
				v=ptr[i+start].Y();
				CDFvarPut(id,varnum2,i,indices,&v);
			}
			}break;
		case GRAPH3D: {
			Graph3D *g = gl->getGraph3D(item);
			Point3D *ptr=g->Data();
			CDFvarCreate(id,"x",CDF_DOUBLE,1,VARY,varys,&varnum1);
			CDFvarCreate(id,"y",CDF_DOUBLE,1,VARY,varys,&varnum2);
			CDFvarCreate(id,"z",CDF_DOUBLE,1,VARY,varys,&varnum3);
			for(int i=0;i<=g->Number()-start;i++) {
				if(end>0 && i+start>end) break;
				v=ptr[i+start].X();
				CDFvarPut(id,varnum1,i,indices,&v);
				v=ptr[i+start].Y();
				CDFvarPut(id,varnum2,i,indices,&v);
				v=ptr[i+start].Z();
				CDFvarPut(id,varnum3,i,indices,&v);
			}
			} break;
		case GRAPH4D: {
			Graph4D *g = gl->getGraph4D(item);
			Point4D *ptr=g->Data();
			CDFvarCreate(id,"x",CDF_DOUBLE,1,VARY,varys,&varnum1);
			CDFvarCreate(id,"y",CDF_DOUBLE,1,VARY,varys,&varnum2);
			CDFvarCreate(id,"z",CDF_DOUBLE,1,VARY,varys,&varnum3);
			CDFvarCreate(id,"t",CDF_DOUBLE,1,VARY,varys,&varnum4);
			for(int i=0;i<=g->Number()-start;i++) {
				if(end>0 && i+start>end) break;
				v=ptr[i+start].X();
				CDFvarPut(id,varnum1,i,indices,&v);
				v=ptr[i+start].Y();
				CDFvarPut(id,varnum2,i,indices,&v);
				v=ptr[i+start].Z();
				CDFvarPut(id,varnum3,i,indices,&v);
				v=ptr[i+start].T();
				CDFvarPut(id,varnum4,i,indices,&v);
			}
			} break;
		case GRAPHM: {
			GraphM *g = gl->getGraphM(item);
			double *data = g->Data();
			int nx=g->NX(), ny=g->NY();
			// TODO : create 2 dim array nx * ny ?
			CDFvarCreate(id,"i",CDF_DOUBLE,1,VARY,varys,&varnum1);
			CDFvarCreate(id,"j",CDF_DOUBLE,1,VARY,varys,&varnum2);
			CDFvarCreate(id,"v",CDF_DOUBLE,1,VARY,varys,&varnum3);
			for(int i=0;i<=nx*ny-start;i++) {
				if(end>0 && i+start>end) break;
				v=i%nx;
				CDFvarPut(id,varnum1,i,indices,&v);
				v=(int)i/nx;
				CDFvarPut(id,varnum2,i,indices,&v);
				v=data[i];
				CDFvarPut(id,varnum3,i,indices,&v);
			}
			} break;
		case GRAPHIMAGE: {
				// TODO
			} break;
		default:	break;
		}
	}
	
	CDFclose(id);
#else
	KMessageBox::warningContinueCancel(this,i18n("Sorry. Your Installation doesn't support CDF!"));
#endif
}

void ExportDialog::dumpNETCDF(QString filename) {
	int start = startrow->text().toInt()-1;
	int end = endrow->text().toInt();

	int ncid;
	nc_create(filename.latin1(),NC_WRITE, &ncid);

	if(item<0) {		// exporting data from spreadsheet
		kdDebug()<<"exporting spreadsheet "<<endl;
	 	Spreadsheet *s = mw->activeSpreadsheet();
		QTable *table = s->Table();

		if (end==0) end=table->numRows();
		int N = end - start;

		int dimid;
		nc_def_dim(ncid,"vector",N,&dimid);
		int dimids[1]={dimid};
#ifdef HAVE_SOLARIS
#warning Solaris : integer constant expression
		int varid[255];
#else
		int varid[table->numCols()];
#endif
		double *data = new double[N];
		
		for (int j=0;j<table->numCols();j++)
			nc_def_var(ncid,s->columnTitle(j).latin1(),NC_DOUBLE, 1, dimids, &varid[j] );
		nc_enddef(ncid);
		
		for (int j=0;j<table->numCols();j++) {
			for(int i=start;i<end;i++)
				data[i-start]=table->text(i,j).toDouble();
			nc_put_var_double(ncid,varid[j],data);
		}
	}
	else {
		GraphList *gl = plot->getGraphList();
		GRAPHType st =  gl->getStruct(item);
		switch(st) {
		case GRAPH2D: {
			Graph2D *g = gl->getGraph2D(item);
			Point *ptr=g->Data();
			int nr = g->Number();
			if (end==0) end=nr;
			int N = end - start;
		
			int dimid;
			nc_def_dim(ncid,"vector",N,&dimid);
			int xvarid, yvarid;
			int dimids[1]={dimid};
			nc_def_var(ncid,"x",NC_DOUBLE, 1, dimids, &xvarid );
			nc_def_var(ncid,"y",NC_DOUBLE, 1, dimids, &yvarid );
			nc_enddef(ncid);
			double *xdata = new double[N];
			double *ydata = new double[N];
			for(int i=start;i<end;i++) {
				xdata[i-start]=ptr[i].X();
				ydata[i-start]=ptr[i].Y();
			}
			nc_put_var_double(ncid,xvarid,xdata);
			nc_put_var_double(ncid,yvarid,ydata);
			}
			break;
		case GRAPH3D: {
			Graph3D *g = gl->getGraph3D(item);
			Point3D *ptr=g->Data();
			int nr = g->Number();
			if (end==0) end=nr;
			int N = end - start;
		
			int dimid;
			nc_def_dim(ncid,"vector",N,&dimid);
			int xvarid, yvarid, zvarid;
			int dimids[1]={dimid};
			nc_def_var(ncid,"x",NC_DOUBLE, 1, dimids, &xvarid );
			nc_def_var(ncid,"y",NC_DOUBLE, 1, dimids, &yvarid );
			nc_def_var(ncid,"z",NC_DOUBLE, 1, dimids, &zvarid );
			nc_enddef(ncid);
			double *xdata = new double[N];
			double *ydata = new double[N];
			double *zdata = new double[N];
			for(int i=start;i<end;i++) {
				xdata[i-start]=ptr[i].X();
				ydata[i-start]=ptr[i].Y();
				zdata[i-start]=ptr[i].Z();
			}
			nc_put_var_double(ncid,xvarid,xdata);
			nc_put_var_double(ncid,yvarid,ydata);
			nc_put_var_double(ncid,zvarid,zdata);
			}
			break;
		case GRAPH4D: {
			Graph4D *g = gl->getGraph4D(item);
			Point4D *ptr=g->Data();
			int nr = g->Number();
			if (end==0) end=nr;
			int N = end - start;
		
			int dimid;
			nc_def_dim(ncid,"vector",N,&dimid);
			int xvarid, yvarid, zvarid, tvarid;
			int dimids[1]={dimid};
			nc_def_var(ncid,"x",NC_DOUBLE, 1, dimids, &xvarid );
			nc_def_var(ncid,"y",NC_DOUBLE, 1, dimids, &yvarid );
			nc_def_var(ncid,"z",NC_DOUBLE, 1, dimids, &zvarid );
			nc_def_var(ncid,"t",NC_DOUBLE, 1, dimids, &tvarid );
			nc_enddef(ncid);
			double *xdata = new double[N];
			double *ydata = new double[N];
			double *zdata = new double[N];
			double *tdata = new double[N];
			for(int i=start;i<end;i++) {
				xdata[i-start]=ptr[i].X();
				ydata[i-start]=ptr[i].Y();
				zdata[i-start]=ptr[i].Z();
				tdata[i-start]=ptr[i].T();
			}
			nc_put_var_double(ncid,xvarid,xdata);
			nc_put_var_double(ncid,yvarid,ydata);
			nc_put_var_double(ncid,zvarid,zdata);
			nc_put_var_double(ncid,tvarid,tdata);
			}
			break;
		case GRAPHM: {
			GraphM *g = gl->getGraphM(item);
			double *data=g->Data();
			int nx = g->NX(), ny = g->NY();
		
			// TODO : export array
			int xdimid, ydimid;
			nc_def_dim(ncid,"x",nx,&xdimid);
			nc_def_dim(ncid,"y",ny,&ydimid);
			int varid;
			int dimids[2]={xdimid,ydimid};
			nc_def_var(ncid,"value",NC_DOUBLE, 2, dimids, &varid );
			nc_enddef(ncid);
			
			nc_put_var_double(ncid,varid,data);
			} break;
		case GRAPHIMAGE : {
			// TODO
			} break;
		default : break;
		}
	}
		
	nc_close(ncid);
}

void ExportDialog::dumpAUDIOFILE(QString filename) {
#ifdef HAVE_AUDIOFILE
	int start = startrow->text().toInt()-1;
	int end = endrow->text().toInt();

	AFfilehandle af=0;
	AFfilesetup setup = afNewFileSetup();
	afInitFileFormat(setup, audioformatcb->currentItem());
	afInitRate(setup, AF_DEFAULT_TRACK, samplele->text().toInt());
 	afInitSampleFormat(setup, AF_DEFAULT_TRACK, AF_SAMPFMT_TWOSCOMP, 16);

	if(item<0) {
		kdDebug()<<"exporting spreadsheet to audio"<<endl;
	 	Spreadsheet *s = mw->activeSpreadsheet();
		QTable *table = s->Table();

		if (end==0) end=table->numRows();
		int N = end - start;

		if(table->numCols()<2) 
			return;
		else if(table->numCols()==2) {	// mono
			afInitChannels(setup, AF_DEFAULT_TRACK, 1);
			af = afOpenFile(filename.latin1(),"w",setup);

			short *data = new short[N];	// frameCount=1(mono),2(stereo)
		
			for(int i=start;i<end;i++)
				data[i-start] = table->text(i-start,1).toInt();

			afWriteFrames(af,AF_DEFAULT_TRACK,(void *) data, N);
		}
		else {	// stereo
			afInitChannels(setup, AF_DEFAULT_TRACK, 2);
			af = afOpenFile(filename.latin1(),"w",setup);

			short *data = new short[2*N];

			for(int i=start;i<end;i++) {
				data[2*i-start] = table->text(i-start,1).toInt();
				data[2*i+1-start] = table->text(i-start,2).toInt();
			}

			afWriteFrames(af,AF_DEFAULT_TRACK,(void *) data, N);
		}
	}
	else {
		GraphList *gl = plot->getGraphList();
		GRAPHType st =  gl->getStruct(item);
		switch(st) {
		case GRAPH2D: {
			Graph2D *g = gl->getGraph2D(item);
			Point *ptr=g->Data();
			if (end==0) end=g->Number();
			int N = end - start;
	
			afInitChannels(setup, AF_DEFAULT_TRACK, 1);
			af = afOpenFile(filename.latin1(),"w",setup);
	
			short *data = new short[N];	// frameCount=1(mono),2(stereo)
		
			for(int i=start;i<end;i++)
				data[i-start] = (int) ptr[i].Y();
	
			afWriteFrames(af,AF_DEFAULT_TRACK,(void *) data, N);
			} break;
		case GRAPH3D: {
			Graph3D *g = gl->getGraph3D(item);
			Point3D *ptr=g->Data();
			if (end==0) end=g->Number();
			int N = end - start;
	
			afInitChannels(setup, AF_DEFAULT_TRACK, 2);
			af = afOpenFile(filename.latin1(),"w",setup);
	
			short *data = new short[2*N];
			
			for(int i=start;i<end;i++) {
				data[2*i-start] = (int) ptr[i].Y();
				data[2*i+1-start] = (int) ptr[i].Z();
			}
					
			afWriteFrames(af,AF_DEFAULT_TRACK,(void *) data, N);
			} break;
		default: break;
		}
	}
	
	afCloseFile(af);
#endif
}

// this is only available for GRAPHM / GRAPHIMAGE
void ExportDialog::dumpIMAGE(QString filename) {
	kdDebug()<<"ExportDialog::dumpIMAGE("<<filename<<")"<<endl;
	if(item<0) {	// export spreadsheet
		kdDebug()<<"exporting spreadsheet to image"<<endl;
		Spreadsheet *s = mw->activeSpreadsheet();
		QTable *table = s->Table();
		int nx=table->numCols(), ny = table->numRows();

		// depth = 8, numColors = 256
		QImage *image = new QImage(nx,ny,8,256);
		for (int i=0;i<256;i++) {
			QColor c = QColor(i,i,i);
			image->setColor (i, c.rgb() );
		}

		for (int i=0;i<nx;i++) {
			for (int j=0;j<ny;j++) {
				unsigned int value =  table->text(j,i).toInt();
				image->setPixel(i,j,value);
			}
		}
#ifdef HAVE_MAGICK
		if(!image->save(filename,ic->currentText())) {
			// save in tmp file
			KTempFile *tmpfile = new KTempFile(QString::null,".bmp");
			tmpfile->setAutoDelete(true);
			QString tmpname = tmpfile->name();
			image->save(tmpname,"BMP");
				
			// read into Image
			Image im;
			im.read(tmpname.latin1());
			
			im.magick(ic->currentText().latin1());
			im.write(filename.latin1());
		}
#endif
	}
	else {		// export GRAPH
		GraphList *gl = plot->getGraphList();
		GRAPHType st = gl->getStruct(item);
	
		if(st == GRAPHIMAGE) {
			GraphIMAGE *g = gl->getGraphIMAGE(item);
			QPixmap pm = g->Pixmap();
	
#ifdef HAVE_MAGICK
			if(!pm.save(filename,ic->currentText())) {
				// save in tmp file
				KTempFile *tmpfile = new KTempFile(QString::null,".bmp");
				tmpfile->setAutoDelete(true);
				QString tmpname = tmpfile->name();
				pm.save(tmpname,"BMP");
				
				// read into Image
				Image image;
				image.read(tmpname.latin1());
				
				image.magick(ic->currentText().latin1());
				image.write(filename.latin1());
			}
#endif
		}
		else {
			GraphM *g = gl->getGraphM(item);
			double *array = g->Data();
			int nx=g->NX(), ny = g->NY();	
			LRange range = g->Range(2);
			Plot2DSurface *plot2 =(Plot2DSurface *) plot;
		
			// depth = 8, numColors = 256
			QImage *image = new QImage(nx,ny,8,256);
			for (int i=0;i<256;i++) {
				QColor c = plot2->Color(i);
				image->setColor (i, c.rgb() );
			}
	
			for (int i=0;i<nx;i++) {
				for (int j=0;j<ny;j++) {
					unsigned int value =  (int)((array[j+ny*i]-range.rMin())/(range.rMax()-range.rMin())*255.0);
					image->setPixel(i,j,value);
				}
			}
#ifdef HAVE_MAGICK
			if(!image->save(filename,ic->currentText())) {
				// save in tmp file
				KTempFile *tmpfile = new KTempFile(QString::null,".bmp");
				tmpfile->setAutoDelete(true);
				QString tmpname = tmpfile->name();
				image->save(tmpname,"BMP");
				
				// read into Image
				Image im;
				im.read(tmpname.latin1());
				
				im.magick(ic->currentText().latin1());
				im.write(filename.latin1());
			}
#endif
			free(image);
		}
	}
}

void ExportDialog::dumpBINARY(QDataStream *d) {
	int start = startrow->text().toInt();
	int end = endrow->text().toInt();
	
	if(item<0) {		// exporting data from spreadsheet
		kdDebug()<<"exporting spreadsheet "<<endl;
	 	QTable *table = mw->activeSpreadsheet()->Table();
		// export header ???
/*		if(headercb->isChecked()) {
			for(int i=0;i<table->numCols();i++) {
				if(i>0) *t<<sep;
				*t<<table->horizontalHeader()->label(i);
			}
			*t<<endl;
		}
*/
		// export data
		for (int i=start-1;i<table->numRows();i++) {
			if(end>0 && i>=end) return;
			for(int j=0;j<table->numCols();j++) {
				switch(binarytypecb->currentItem()) {
				case 0: *d<<table->text(i,j).toDouble(); break;
				case 1: *d<<table->text(i,j).toFloat(); break;
				case 2: *d<<table->text(i,j).toLong(); break;
				case 3: *d<<(Q_INT8)table->text(i,j).toDouble(); break;
				case 4: *d<<(Q_INT16)table->text(i,j).toDouble(); break;
				case 5: *d<<(Q_INT32)table->text(i,j).toDouble(); break;
				case 6: *d<<(Q_INT64)table->text(i,j).toDouble(); break;
				}
			}
		}	
	}
	else {			// exporting data from GRAPH
		GraphList *gl = plot->getGraphList();
		GRAPHType st =  gl->getStruct(item);
		switch(st) {
		case GRAPH2D: {
			Graph2D *g = gl->getGraph2D(item);
			Point *ptr=g->Data();
			for (int i=start-1;i<g->Number();i++) {
				if(end>0 && i>end) return;
				
				switch(binarytypecb->currentItem()) {
				case 0: *d<<ptr[i].X()<<ptr[i].Y(); break;
				case 1: *d<<(float)ptr[i].X()<<(float)ptr[i].Y(); break;
				case 2: *d<<(Q_LONG)ptr[i].X()<<(Q_LONG)ptr[i].Y(); break;
				case 3: *d<<(Q_INT8)ptr[i].X()<<(Q_INT8)ptr[i].Y(); break;
				case 4: *d<<(Q_INT16)ptr[i].X()<<(Q_INT16)ptr[i].Y(); break;
				case 5: *d<<(Q_INT32)ptr[i].X()<<(Q_INT32)ptr[i].Y(); break;
				case 6: *d<<(Q_INT64)ptr[i].X()<<(Q_INT64)ptr[i].Y(); break;
				}
			}}
			break;
		case GRAPH3D: {
			Graph3D *g = gl->getGraph3D(item);
			Point3D *ptr=g->Data();
			for (int i=start-1;i< g->NX()*g->NY();i++) {
				if(end>0 && i>end) return;
				
				switch(binarytypecb->currentItem()) {
				case 0: *d<<ptr[i].X()<<ptr[i].Y()<<ptr[i].Z(); break;
				case 1: *d<<(float)ptr[i].X()<<(float)ptr[i].Y()<<(float)ptr[i].Z(); break;
				case 2: *d<<(Q_LONG)ptr[i].X()<<(Q_LONG)ptr[i].Y()<<(Q_LONG)ptr[i].Z(); break;
				case 3: *d<<(Q_INT8)ptr[i].X()<<(Q_INT8)ptr[i].Y()<<(Q_INT8)ptr[i].Z(); break;
				case 4: *d<<(Q_INT16)ptr[i].X()<<(Q_INT16)ptr[i].Y()<<(Q_INT16)ptr[i].Z(); break;
				case 5: *d<<(Q_INT32)ptr[i].X()<<(Q_INT32)ptr[i].Y()<<(Q_INT32)ptr[i].Z(); break;
				case 6: *d<<(Q_INT64)ptr[i].X()<<(Q_INT64)ptr[i].Y()<<(Q_INT64)ptr[i].Z(); break;
				}
			}}
			break;
		case GRAPH4D: {
			Graph4D *g = gl->getGraph4D(item);
			Point4D *ptr=g->Data();
			for (int i=start-1;i< g->Number();i++) {
				if(end>0 && i>end) return;
			
				switch(binarytypecb->currentItem()) {
				case 0: *d<<ptr[i].X()<<ptr[i].Y()<<ptr[i].Z()<<ptr[i].T(); break;
				case 1: *d<<(float)ptr[i].X()<<(float)ptr[i].Y()<<(float)ptr[i].Z()<<(float)ptr[i].T(); break;
				case 2: *d<<(Q_LONG)ptr[i].X()<<(Q_LONG)ptr[i].Y()<<(Q_LONG)ptr[i].Z()<<(Q_LONG)ptr[i].T(); break;
				case 3: *d<<(Q_INT8)ptr[i].X()<<(Q_INT8)ptr[i].Y()<<(Q_INT8)ptr[i].Z()<<(Q_INT8)ptr[i].T(); break;
				case 4: *d<<(Q_INT16)ptr[i].X()<<(Q_INT16)ptr[i].Y()<<(Q_INT16)ptr[i].Z()<<(Q_INT16)ptr[i].T(); break;
				case 5: *d<<(Q_INT32)ptr[i].X()<<(Q_INT32)ptr[i].Y()<<(Q_INT32)ptr[i].Z()<<(Q_INT32)ptr[i].T(); break;
				case 6: *d<<(Q_INT64)ptr[i].X()<<(Q_INT64)ptr[i].Y()<<(Q_INT64)ptr[i].Z()<<(Q_INT64)ptr[i].T(); break;
				}
			}}
			break;
		case GRAPHM: {
			GraphM *g = gl->getGraphM(item);
			double *array = g->Data();
			int nx=g->NX(), ny = g->NY();
		
			for (int i=0;i<ny;i++) {
				for (int j=0;j<nx;j++) {
					switch(binarytypecb->currentItem()) {
					case 0: *d<<array[j+nx*i]; break;
					case 1: *d<<(float)array[j+nx*i]; break;
					case 2: *d<<(Q_LONG)array[j+nx*i]; break;
					case 3: *d<<(Q_INT8)array[j+nx*i]; break;
					case 4: *d<<(Q_INT16)array[j+nx*i]; break;
					case 5: *d<<(Q_INT32)array[j+nx*i]; break;
					case 6: *d<<(Q_INT64)array[j+nx*i]; break;
					}
				}
			}}
			break;
		case GRAPHIMAGE: {
			GraphIMAGE *g = gl->getGraphIMAGE(item);
			QPixmap pm = g->Pixmap();
			QImage image = pm.convertToImage();

			for (int i=0;i<pm.width();i++) {
				for (int j=0;j<pm.height();j++) {
					switch(binarytypecb->currentItem()) {
					case 0: *d<<qGray(image.pixel(i,j)); break;
					case 1: *d<<(float)qGray(image.pixel(i,j)); break;
					case 2: *d<<(Q_LONG)qGray(image.pixel(i,j)); break;
					case 3: *d<<(Q_INT8)qGray(image.pixel(i,j)); break;
					case 4: *d<<(Q_INT16)qGray(image.pixel(i,j)); break;
					case 5: *d<<(Q_INT32)qGray(image.pixel(i,j)); break;
					case 6: *d<<(Q_INT64)qGray(image.pixel(i,j)); break;
					}
				}
			}}
			break;
		default: break;
		}
	}	
}

void ExportDialog::wizardKexiDB(const QString &name) {
	//same as ImportDialog::wizardKexiDB()
	kdDebug()<<"ExportDialog::wizardKexiDB()"<<endl;
	
	int res;
	if(name == i18n("Driver")) {	
		kdDebug()<<"	TAB Driver"<<endl;
	}
	else if(name == i18n("Connection")) {
		kdDebug()<<"	TAB Connection : connecting to "<<driver->currentText()<<endl;
		connectionlabel->setText(i18n("Connecting to Driver ")+driver->currentText()+i18n(" ... "));
		res = kexi->connectDriver(driver->currentText());
		if (res) {
			KMessageBox::error(this, i18n("Sorry. Could not connect to driver!"));
			return;
		}
	}
	else if(name == i18n("Database")) {
		kdDebug()<<"	TAB Database"<<endl;
		res = kexi->connect(host->text(),user->text(),password->text());
		if (res) {
			KMessageBox::error(this, i18n("Sorry. Could not connect to database!"));
			return;
		}
		databases->clear();
		kdDebug()<<"	calling Databases()"<<endl;
		if(kexi->Databases().empty()) {
			KMessageBox::error(this, i18n("Sorry. No databases found!"));
			return;
		}
		databases->insertStringList(kexi->Databases());
	}
	else if(name == i18n("Table")) {
		kdDebug()<<"	TAB Database"<<endl;
		tablelabel->setText(i18n("Using Database ")+databases->currentText()+i18n(" ... "));
		res = kexi->connectDatabase(databases->currentText());
		if (res) {
			KMessageBox::error(this, i18n("Sorry. Could not connect to database!"));
			return;
		}
		tables->clear();
		tables->insertStringList(kexi->Tables());
	}
}

void ExportDialog::finishKexiDB() {
	QString tablename = tables->currentText();

	//TODO : use this
	int start = startrow->text().toInt();
	int end = endrow->text().toInt();
	
	kdDebug()<<" CREATE TABLE "<<tablename<<endl;
	kexi->initWrite();
	
	int rows=0, cols=0;
	QString *data=0;
	QStringList fieldlist;
	if(item<0) {		// exporting data from spreadsheet
		kdDebug()<<"exporting spreadsheet "<<endl;
	 	QTable *table = mw->activeSpreadsheet()->Table();
		
		rows=table->numRows();
		cols=table->numCols();
		kdDebug()<<"	Rows/Cols="<<rows<<' '<<cols<<endl;
		
		data = new QString[rows*cols];
		
		// use header label
#if QT_VERSION > 0x030007
		for(int i=0;i<cols;i++) {
			QString tmp = table->horizontalHeader()->label(i);
			tmp.remove(QRegExp(" \\{.+\\]"));
			fieldlist<<tmp;
		}
#endif
		// export data
		for (int i=start-1;i<rows;i++) {
			if(end>0 && i>=end) return;
			for(int j=0;j<cols;j++) {
				data[cols*i+j]=table->text(i,j);
			}
		}
	}
	else {
		GraphList *gl = plot->getGraphList();
		GRAPHType st =  gl->getStruct(item);
		switch(st) {
		case GRAPH2D: {
			Graph2D *g = gl->getGraph2D(item);
			rows = g->Number();
			cols = 2;

			data = new QString[rows*cols];
			Point *ptr=g->Data();
			for (int i=start-1;i<g->Number();i++) {
				if(end>0 && i>end) return;
				data[cols*i]=QString::number(ptr[i].X());
				data[cols*i+1]=QString::number(ptr[i].Y());
			}}
			break;
		case GRAPH3D: {
			Graph3D *g = gl->getGraph3D(item);
			rows = g->Number();
			cols = 3;

			data = new QString[rows*cols];
			Point3D *ptr=g->Data();
			for (int i=start-1;i<g->Number();i++) {
				if(end>0 && i>end) return;
				data[cols*i]=QString::number(ptr[i].X());
				data[cols*i+1]=QString::number(ptr[i].Y());
				data[cols*i+2]=QString::number(ptr[i].Z());
			}}
			break;
		case GRAPH4D: {
			Graph4D *g = gl->getGraph4D(item);
			rows = g->Number();
			cols = 4;

			data = new QString[rows*cols];
			Point4D *ptr=g->Data();
			for (int i=start-1;i<g->Number();i++) {
				if(end>0 && i>end) return;
				data[cols*i]=QString::number(ptr[i].X());
				data[cols*i+1]=QString::number(ptr[i].Y());
				data[cols*i+2]=QString::number(ptr[i].Z());
				data[cols*i+3]=QString::number(ptr[i].T());
			}}
			break;
		case GRAPHM: {
			GraphM *g = gl->getGraphM(item);
			double *array = g->Data();
			cols=g->NX();
			rows = g->NY();

			data = new QString[rows*cols];
			for (int i=0;i<rows;i++) {
				for (int j=0;j<cols;j++) {
					data[cols*i+j]=QString::number(array[j+cols*i]);
				}
			}}
			break;
		case GRAPHIMAGE: {
			GraphIMAGE *g = gl->getGraphIMAGE(item);
			QPixmap pm = g->Pixmap();
			QImage image = pm.convertToImage();
			cols=pm.width();
			rows = pm.height();
			
			data = new QString[rows*cols];
			for (int i=0;i<rows;i++) {
				for (int j=0;j<cols;j++) {
					data[cols*i+j]=QString::number(image.pixel(i,j));
				}
			}}
			break;
		default:
			break;
		}

		// header
		for(int i=0;i<cols;i++) {
			if(i<26)
				fieldlist<<QString(QChar(i+65));
			else
				fieldlist<<QString(QChar(84));
		}
	}

	kexi->createTable(tablename,fieldlist);
	kexi->writeData(data, cols, rows);
}

void ExportDialog::dumpKexiDB() {
	kdDebug()<<"ExportDialog::dumpKexiDB()"<<endl;
#ifdef HAVE_KEXIDB
	kexi = new FilterKexiDB(0);

	QWizard *wizard = new QWizard(this);
	//TODO
	QObject::connect(wizard,SIGNAL(selected(const QString &)),this,SLOT(wizardKexiDB(const QString&))); 
	QObject::connect(wizard->finishButton(),SIGNAL(clicked()),this,SLOT(finishKexiDB())); 
	
	QVBox *page1 = new QVBox(wizard);
	new QLabel(i18n("Please select driver : "),page1);
	driver= new KComboBox(page1);
	driver->insertStringList(kexi->Driver());

	wizard->addPage(page1,i18n("Driver"));
	wizard->setHelpEnabled(page1,false);
	
	QVBox *page2 = new QVBox(wizard);
	
	connectionlabel = new QLabel("",page2);
	new QLabel(i18n("Please provide connection informations  : "),page2);
	QHBox *hb = new QHBox(page2);
	new QLabel(i18n("Hostname : "),hb);
	host = new KLineEdit(i18n("localhost"),hb);
	hb = new QHBox(page2);
	new QLabel(i18n("User : "),hb);
	user = new KLineEdit(i18n("root"),hb);
	hb = new QHBox(page2);
	new QLabel(i18n("Password : "),hb);
	password = new KLineEdit(i18n("mysql"),hb);
	password->setEchoMode(QLineEdit::Password);
	
	wizard->addPage(page2,i18n("Connection"));
	wizard->setHelpEnabled(page2,false);
	
	QVBox *page3 = new QVBox(wizard);
	new QLabel(i18n("Please select database  : "),page3);
	databases = new KComboBox(page3);
	databases->setEditable(true);
	wizard->addPage(page3,i18n("Database"));
	wizard->setHelpEnabled(page3,false);
	
	QVBox *page4 = new QVBox(wizard);
	tablelabel = new QLabel("",page4);
	new QLabel(i18n("Please select table : "),page4);
	tables = new KComboBox(page4);
	tables->setEditable(true);
	wizard->addPage(page4,i18n("Table"));
	wizard->setHelpEnabled(page4,false);
	wizard->setFinishEnabled(page4,true);

	wizard->show();
#endif
}
