/***
* Author: Juergen Heinemann http://www.hjcms.de, (C) 2007-2008
* Copyright: See COPYING file that comes with this distribution
**/

#include "qx11grab.h"
#include "settings.h"
#include "desktopinfo.h"
#include "rubberband.h"
#include "windowgrabber.h"
#include "settingsdialog.h"
#include "ffprocess.h"
#include "kx11grabadaptor.h"
#include "screencombobox.h"

/* QtCore */
#include <QtCore/QDebug>
#include <QtCore/QString>

/* QtGui */
#include <QtGui/QIcon>
#include <QtGui/QPixmap>
#include <QtGui/QPushButton>
#include <QtGui/QDesktopWidget>
#include <QtGui/QRubberBand>

/* KDE */
#include <KDE/KIconLoader>
#include <KDE/KActionCollection>
#include <KDE/KAction>
#include <KDE/KMenu>
#include <KDE/KMessageBox>

/* QtDBus */
#include <QtDBus/QDBusConnection>

QX11Grab::QX11Grab ( Settings *settings )
    : QX11GrabMain ( settings )
    , cfg ( settings )
{

  setDepthBox->hide();
  TimeOutMessages = 5000;

  loadStats();

  /* NOTE ffprocess must init before create Actions ... */
  m_FFProcess = new FFProcess ( this, cfg );

  createActions();
  createEnviroment();
  createSystemTrayIcon();

  rcDBus = new QDBusConnection ( QDBusConnection::sessionBus() );
  rcDBus->registerService ( "de.hjcms.kx11grab" );
  rcDBus->sessionBus().registerObject ( "/kx11grab", this, QDBusConnection::ExportAdaptors );

  m_Kx11grabAdaptor = new Kx11grabAdaptor ( this );
  connect ( m_FFProcess, SIGNAL ( message ( const QString & ) ),
            m_Kx11grabAdaptor, SIGNAL ( message ( const QString & ) ) );

  /* Signals */
  connect ( m_FFProcess, SIGNAL ( message ( const QString & ) ),
            this, SLOT ( pushInfoMessage ( const QString & ) ) );

  connect ( m_FFProcess, SIGNAL ( errmessage ( const QString &, const QString & ) ),
            this, SLOT ( pushErrorMessage ( const QString &, const QString & ) ) );

  connect ( m_FFProcess, SIGNAL ( trigger ( const QString & ) ),
            this, SLOT ( pushToolTip ( const QString & ) ) );

  connect ( screenComboBox, SIGNAL ( screenNameChanged ( const QString & ) ),
            setModeName, SLOT ( setText ( const QString & ) ) );

  connect ( screenComboBox, SIGNAL ( screenWidthChanged ( int ) ),
            setWidthBox, SLOT ( setValue ( int ) ) );

  connect ( screenComboBox, SIGNAL ( screenHeightChanged ( int ) ),
            setHeightBox, SLOT ( setValue ( int ) ) );

  connect ( screenComboBox, SIGNAL ( screenDepthChanged ( int ) ),
            setDepthBox, SLOT ( setValue ( int ) ) );

  connect ( setWidthBox, SIGNAL ( valueChanged ( int ) ),
            this, SLOT ( toRubber ( int ) ) );

  connect ( setHeightBox, SIGNAL ( valueChanged ( int ) ),
            this, SLOT ( toRubber ( int ) ) );

  connect ( setXBox, SIGNAL ( valueChanged ( int ) ),
            this, SLOT ( toRubber ( int ) ) );

  connect ( setYBox, SIGNAL ( valueChanged ( int ) ),
            this, SLOT ( toRubber ( int ) ) );

  connect ( showRubberband, SIGNAL ( clicked ( bool ) ),
            this, SLOT ( showRubber ( bool ) ) );

  connect ( actionGrabbing, SIGNAL ( triggered () ),
            this, SLOT ( grabFromWindow () ) );

  connect ( actionStartRecord, SIGNAL ( triggered () ),
            this, SLOT ( startRecord () ) );

  connect ( actionStopRecord, SIGNAL ( triggered () ),
            m_FFProcess, SLOT ( stop () ) );

  connect ( actionKillRecord, SIGNAL ( triggered () ),
            m_FFProcess, SLOT ( kill () ) );

  connect ( m_FFProcess, SIGNAL ( down () ),
            this, SLOT ( setActionsBack () ) );

  connect ( actionApplication, SIGNAL ( triggered() ),
            this, SLOT ( openSettings() ) );

  connect ( actionMinimize, SIGNAL ( triggered() ),
            this, SLOT ( hide() ) );

  connect ( actionQuit, SIGNAL ( triggered() ),
            qApp, SLOT ( quit() ) );

  setAutoSaveSettings( "kx11grab", true );
}

void QX11Grab::start_record()
{
  if ( ! m_FFProcess )
    return;

  if ( m_FFProcess->isRunning() )
  {
    KMessageBox::sorry ( this, i18n ( "Warning" ), i18n ( "Recorder is running." ) );
    return;
  }
  else
    startRecord ();
}

void QX11Grab::stop_record()
{
  if ( m_FFProcess->isRunning() )
    m_FFProcess->stop ();
}

void QX11Grab::swap_rubberband()
{
  // TODO
}

void QX11Grab::createActions()
{
  grabActionFromWindow = new KAction ( getIcon ( "edit-find" ), i18n ( "Grabbing" ), this );
  connect ( grabActionFromWindow, SIGNAL ( triggered() ), this, SLOT ( grabFromWindow() ) );

  showRubberbandWindow = new KAction ( getIcon ( "select-rectangular" ), i18n ( "Rubberband" ), this );
  connect ( showRubberbandWindow, SIGNAL ( triggered() ), this, SLOT ( swapRubberBand() ) );

  startRecordingWindow = new KAction ( getIcon ( "media-record" ), i18n ( "Recording" ), this );
  connect ( startRecordingWindow, SIGNAL ( triggered() ), this, SLOT ( startRecord() ) );

  stopRecordingWindow = new KAction ( getIcon ( "edit-clear-list" ), i18n ( "Stop" ), this );
  stopRecordingWindow->setEnabled ( false );
  connect ( stopRecordingWindow, SIGNAL ( triggered() ), m_FFProcess, SLOT ( stop () ) );

  minimizeWindowAction = new KAction ( getIcon ( "view-sort-ascending" ), i18n ( "Hide" ), this );
  connect ( minimizeWindowAction, SIGNAL ( triggered() ), this, SLOT ( hide() ) );

  displayWindowAction = new KAction ( getIcon ( "view-sort-descending" ), i18n ( "Show" ), this );
  connect ( displayWindowAction, SIGNAL ( triggered() ), this, SLOT ( showNormal() ) );

  quitWindowAction = new KAction ( getIcon ( "window-close" ), i18n ( "Quit" ), this );
  connect ( quitWindowAction, SIGNAL ( triggered() ), qApp, SLOT ( quit() ) );
}

void QX11Grab::createEnviroment()
{
  m_DesktopInfo = new DesktopInfo ( this );
  FrameMode desktop = m_DesktopInfo->grabScreenGeometry ( centralWidget() );
  setDepthBox->setValue ( desktop.depth );

  m_RubberBand = new RubberBand ( qApp->desktop()->screen() );
  connect ( m_RubberBand, SIGNAL ( error ( const QString &, const QString & ) ),
            this, SLOT ( pushErrorMessage ( const QString &, const QString & ) ) );

  toRubber ( 1 );
  if ( showRubberband->isChecked() )
    m_RubberBand->show();
  else
    m_RubberBand->hide();
}

void QX11Grab::createSystemTrayIcon()
{
  KMenu *systemTrayMenu = new KMenu ( i18n("Grabber Menu"), this );
  systemTrayMenu->addAction ( grabActionFromWindow );
  systemTrayMenu->addAction ( showRubberbandWindow );
  systemTrayMenu->addSeparator();
  systemTrayMenu->addAction ( startRecordingWindow );
  systemTrayMenu->addAction ( stopRecordingWindow );
  systemTrayMenu->addSeparator();
  systemTrayMenu->addAction ( minimizeWindowAction );
  systemTrayMenu->addAction ( displayWindowAction );
  systemTrayMenu->addSeparator();
  systemTrayMenu->addAction ( quitWindowAction );

  systemTrayIcon = new KSystemTrayIcon ( this );
  /* FIXME
  * Currently hide KDE4 KActionCollection to make kx11grab
  * available in KDE3 and don't use init4. kx11grab will crash
  * with KActionCollection KDE4 trigger Actions in KDE3 Enviroments
  */
  systemTrayIcon->actionCollection()->clear();
  systemTrayIcon->setIcon ( getIcon ( "kx11grab", KIconLoader::Small ) );
  systemTrayIcon->setToolTip ( i18n ( "Recording X11 Windows with ffmpeg" ) );
  systemTrayIcon->setContextMenu ( systemTrayMenu );
  connect ( systemTrayIcon, SIGNAL ( quitSelected () ), this, SLOT ( stop_record () ) );
  connect ( systemTrayIcon, SIGNAL ( activated ( QSystemTrayIcon::ActivationReason ) ),
            this, SLOT ( systemTrayWatcher ( QSystemTrayIcon::ActivationReason ) ) );

  systemTrayIcon->show();
}

void QX11Grab::swapRubberBand ()
{
  if ( showRubberband->isChecked() )
  {
    showRubberband->setChecked ( false );
    m_RubberBand->hide();
  }
  else
  {
    showRubberband->setChecked ( true );
    m_RubberBand->show();
  }
}

void QX11Grab::showRubber ( bool b )
{
  if ( b )
  {
    showRubberband->setChecked ( true );
    m_RubberBand->show();
  }
  else
  {
    showRubberband->setChecked ( false );
    m_RubberBand->hide();
  }
}

void QX11Grab::loadStats()
{
  if ( cfg->contains ( "windowPos" ) )
    move ( cfg->value ( "windowPos", pos() ).toPoint() );

  if ( cfg->contains ( "windowSize" ) )
    resize ( cfg->value ( "windowSize", size() ).toSize() );

  if ( ! cfg->contains ( "Version" ) )
  {
    cfg->setValue ( "Version", KX11GRAB_VERSION );
    openSettings();
  }
}

void QX11Grab::saveStats()
{
  cfg->setValue ( "windowState", saveState() );
  cfg->setValue ( "windowPos", pos() );
  cfg->setValue ( "windowSize", size() );
}

void QX11Grab::toRubber ( int i )
{
  if ( i < 1 )
    return;

  if ( ! m_RubberBand )
    return;

  int w = setWidthBox->value();
  int h = setHeightBox->value();
  int x = setXBox->value();
  int y = setYBox->value();
  m_RubberBand->resize ( w, h );
  m_RubberBand->move ( x, y );
}

void QX11Grab::grabFromWindow()
{
  if ( ! m_RubberBand )
    return;

  WindowGrabber *grabber = new WindowGrabber ( this );
  QRect rect = grabber->grabWindowRect();

  if ( rect.isValid() )
  {
    setWidthBox->setValue ( rect.width() );
    setHeightBox->setValue ( rect.height() );
    setXBox->setValue ( rect.x() );
    setYBox->setValue ( rect.y() );
    setModeName->setText ( i18n ( "grabbed Dimension" ) );
    toRubber ( 1 );
  }

  delete grabber;
}

void QX11Grab::systemTrayWatcher ( QSystemTrayIcon::ActivationReason type )
{
  // TODO Trigger Window Gemometry
  switch ( type )
  {
    case QSystemTrayIcon::Unknown:
    case QSystemTrayIcon::Trigger:
    default:
      return;
  }
}

void QX11Grab::openSettings ()
{
  SettingsDialog *confDialog = new SettingsDialog ( centralWidget(), cfg );
  confDialog->exec();
  delete confDialog;
}

void QX11Grab::showEvent ( QShowEvent * )
{
  loadStats();
}

void QX11Grab::hideEvent ( QHideEvent * )
{
  saveStats();
}

void QX11Grab::closeEvent ( QCloseEvent *ev )
{
  if ( m_FFProcess->isRunning() )
  {
    KMessageBox::sorry ( this, i18n ( "Warning" ), i18n ( "Recorder is running." ) );
    ev->ignore();
  }
  saveStats();

  if ( m_Kx11grabAdaptor )
  {
    rcDBus->unregisterObject ( "/kx11grab", QDBusConnection::UnregisterNode );
    delete m_Kx11grabAdaptor;
  }

}

void QX11Grab::pushInfoMessage ( const QString &txt )
{
  if ( systemTrayIcon )
    systemTrayIcon->showMessage ( i18n ( "Info" ), txt,
                                  QSystemTrayIcon::Information, TimeOutMessages );

  if ( ! m_FFProcess->isRunning() )
    systemTrayIcon->setIcon ( getIcon ( "kx11grab" ) );
}

void QX11Grab::pushErrorMessage ( const QString &title, const QString &txt )
{
  if ( systemTrayIcon )
    systemTrayIcon->showMessage ( title, txt, QSystemTrayIcon::Critical, TimeOutMessages );

  if ( ! m_FFProcess->isRunning() )
    systemTrayIcon->setIcon ( getIcon ( "kx11grab" ) );
}

void QX11Grab::pushToolTip ( const QString &txt )
{
  if ( systemTrayIcon )
    systemTrayIcon->setToolTip ( txt );
}

void QX11Grab::startRecord()
{
  if ( ! m_RubberBand->isScalability() )
    return;

  QRect rect ( setXBox->value(), setYBox->value(), setWidthBox->value(), setHeightBox->value() );
  if ( m_FFProcess->create ( rect ) )
  {
    stopRecordingWindow->setEnabled ( true );
    actionStopRecord->setEnabled ( true );
    actionKillRecord->setEnabled ( true );
    startRecordingWindow->setEnabled ( false );
    actionStartRecord->setEnabled ( false );
    showRubber ( false );
    if ( m_FFProcess->start() )
    {
      systemTrayIcon->setIcon ( getIcon ( "media-record", KIconLoader::Small ) );
    }
  }
  else
    KMessageBox::error ( this, i18n ( "Error" ), i18n ( "qx11grap not started" ) );
}

void QX11Grab::setActionsBack()
{
  stopRecordingWindow->setEnabled ( false );
  actionStopRecord->setEnabled ( false );
  actionKillRecord->setEnabled ( false );
  startRecordingWindow->setEnabled ( true );
  actionStartRecord->setEnabled ( true );
  systemTrayIcon->setIcon ( getIcon ( "kx11grab" ) );
}

QX11Grab::~QX11Grab()
{
}
