// =============================================================================
//
//      --- kvi_window.cpp ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#define _KVI_DEBUG_CHECK_RANGE_
#define _KVI_DEBUG_CLASS_NAME_ "KviWindow"

#define _KVI_WINDOW_CPP_

#include <qobjectlist.h>
#include <qtimer.h>

#include "kvi_accel.h"
#include "kvi_app.h"
#include "kvi_console.h"
#include "kvi_debug.h"
#include "kvi_finddialog.h"
#include "kvi_frame.h"
#include "kvi_irc_view.h"
#include "kvi_locale.h"
#include "kvi_malloc.h"
#include "kvi_mdi_child.h"
#include "kvi_options.h"
#include "kvi_settings.h"
#include "kvi_taskbarbutton.h"
#include "kvi_userparser.h"
#include "kvi_window.h"

/**
 * ============ KviWindow ============
 */
KviWindow::KviWindow(const char *name, int type, KviFrame *lpFrm)
	: QWidget(0, name)
{
	m_szCaption         = name ? name : _i18n_("Unnamed");
	m_pFrm              = lpFrm;
	m_type              = type;
	m_pTaskBarButton    = 0;
	m_pView             = 0;
	m_pFindDialog       = 0;
	m_pListBox          = 0;
	m_pInput            = 0;
	m_pSplitter         = 0;
	m_pFocusHandler     = 0;
	m_pLastFocusedChild = 0;
	m_pAccelerators     = 0;
#ifdef COMPILE_PLUGIN_SUPPORT
	m_pluginHandle      = 0;
#endif
	m_bJustGotFocused   = false;
	setMinimumSize(QSize(KVI_WINDOW_MIN_WIDTH, KVI_WINDOW_MIN_HEIGHT));
	setBackgroundMode(NoBackground);
	setFocusPolicy(StrongFocus);
}

/**
 * ============ ~KviWindow ============
 */
KviWindow::~KviWindow()
{
	if( m_pFindDialog ) {
		delete m_pFindDialog;
		m_pFindDialog = 0;
	}
	if( m_pFrm->m_pUserParser ) {
		m_pFrm->m_pUserParser->unregisterTimersFor(this);
	}
}

KviFrame *KviWindow::frame() const
{
	return m_pFrm;
}

void KviWindow::wheelEvent(QWheelEvent *e)
{
	if( m_pView )
		m_pView->wheelEvent(e);
	else
		e->ignore();
}

void KviWindow::installAccelerators()
{
	if( !m_pAccelerators )
		m_pAccelerators = m_pFrm->installAccelerators(this);
}

void KviWindow::destroyAccelerators()
{
	if( m_pAccelerators ) {
		delete m_pAccelerators;
		m_pAccelerators = 0;
	}
}

void KviWindow::getDefaultLogName(KviStr &str)
{
	QDate dt(QDate::currentDate());
	// Keep the semi-alphabetic order...
	KviStr fName = caption();
	if( g_pOptions->m_bAppendDateToLogName ) {
		KviStr theDate(KviStr::Format, "_%d_%d_%d", dt.year(), dt.month(), dt.day());
		fName.append(theDate);
	}
	if( g_pOptions->m_bAppendServerNameToLogName ) {
		if( m_pFrm->m_state == KviFrame::Connected ) {
			fName.append('_');
			fName.append(m_pFrm->m_global.szCurrentServerHost);
		}
	}
	fName.append(".log");
	fName.replaceAll(' ', "_");
	g_pApp->getLocalKVIrcDirectory(str, KviApp::Log, fName.ptr(), true);
}

/**
 * ============== findText ==============
 */
void KviWindow::findText()
{
	if( !m_pView )
		return;

	if( !m_pFindDialog )
		m_pFindDialog = new KviFindDialog(this);
	if( mdiParent() && isMinimized() )
		mdiParent()->setState(KviMdiChild::Normal);
	else
		showNormal();
	setFocus();
	m_pFindDialog->exec();
}

/**
 * ============= setTaskBarButton ==============
 */
void KviWindow::setTaskBarButton(KviTaskBarButton *btn_ptr)
{
	m_pTaskBarButton = btn_ptr;
}

QRect KviWindow::externalGeometry()
{
	return mdiParent() ? mdiParent()->geometry() : geometry();
}

/**
 * ============== minimize ==============
 */
void KviWindow::minimize(bool bAnimate)
{
	if( mdiParent() ) {
		if( !isMinimized() ) {
			mdiParent()->setState(KviMdiChild::Minimized, bAnimate);
		}
	} else showMinimized();
}

void KviWindow::minimize()
{
	minimize(true);
}

/**
 * ============= maximize ==============
 */
void KviWindow::maximize(bool bAnimate)
{
	if( mdiParent() ) {
		if( !isMaximized() ) {
			mdiParent()->setState(KviMdiChild::Maximized, bAnimate);
		}
	} else showMaximized();
}

void KviWindow::maximize()
{
	maximize(true);
}

/**
 * ============== attach ================
 */
void KviWindow::attach()
{
	m_pFrm->attachWindow(this);
}

/**
 * ============== detach =================
 */
void KviWindow::detach()
{
	m_pFrm->detachWindow(this);
}

/**
 * ============== eventFilter =================
 */
bool KviWindow::eventFilter(QObject *o, QEvent *e)
{
	switch( e->type() ) {
		case QEvent::FocusIn:
			if( m_pLastFocusedChild == (QWidget *) o )
				m_bJustGotFocused = true;
			if( m_pFrm->activeWindow() == this )
				m_bJustGotFocused = false;
			m_pLastFocusedChild = (QWidget *) o;
			activateSelf();
			break;
		case QEvent::Enter:
			if( KviApplication::overrideCursor() )
				KviApplication::restoreOverrideCursor();
			break;
		case QEvent::MouseButtonPress:
			if( (((QWidget *) o)->focusPolicy() == NoFocus) ||
				(((QWidget *) o)->focusPolicy() == TabFocus)
			) {
				// This will not focus our window; set the focus to the focus handler
				if( m_pLastFocusedChild ) {
					if( m_pLastFocusedChild->hasFocus() && m_pLastFocusedChild->isVisible() )
						return false;
				}
				if( m_pFocusHandler ) {
					m_pFocusHandler->setFocus();
				} else {
					// We grab the focus
					setFocus();
				}
			}
			break;
		case QEvent::ChildInserted: {
			QChildEvent *ev = (QChildEvent *) e;
			if( ev->child()->isWidgetType() )
				childInserted((QWidget *) ev->child());
			break;
		}
		case QEvent::ChildRemoved: {
			QChildEvent *ev = (QChildEvent *) e;
			if( ev->child()->isWidgetType() )
				childRemoved((QWidget *) ev->child());
			break;
		}
		default:
			break; // Ignore
	}
	return false;
}

void KviWindow::activateSelf()
{
	if( mdiParent() )
		mdiParent()->activate(false);
	m_pFrm->setActiveWindow(this);
}

void KviWindow::focusInEvent(QFocusEvent *)
{
	activateSelf();
	if( m_pFocusHandler ) {
		if( m_pLastFocusedChild ) {
			if( m_pLastFocusedChild->hasFocus() && m_pLastFocusedChild->isVisible() )
				return; // The last focused child still has focus
		}
		m_pFocusHandler->setFocus();
	}
}

void KviWindow::childInserted(QWidget *o)
{
	o->removeEventFilter(this);  // Ensure that we do not filter twice
	o->installEventFilter(this); // We filter its events

	if( o->inherits("KviInput") )
		m_pFocusHandler = o;
	else if( !m_pFocusHandler && (o->focusPolicy() == StrongFocus) )
		m_pFocusHandler = o;

	QObjectList *list = (QObjectList *) (o->children());
	if( list ) {
		for( QObject *c = list->first(); c; c = list->next() ) {
			if( c->isWidgetType() ) {
				childInserted((QWidget *) c);
			}
		}
	}
}

void KviWindow::childRemoved(QWidget *o)
{
	o->removeEventFilter(this);
	if( o == m_pFocusHandler )
		m_pFocusHandler = 0;
	if( o == m_pLastFocusedChild )
		m_pLastFocusedChild = 0;
}

void KviWindow::childEvent(QChildEvent *e)
{
	if( e->child()->isWidgetType() ) {
		if( e->removed() )
			childRemoved((QWidget *) (e->child()));
		else
			childInserted((QWidget *) (e->child()));
	}
	QWidget::childEvent(e);
}

/**
 * =============== isMinimized? =================
 */
bool KviWindow::isMinimized()
{
	if( mdiParent() )
		return (mdiParent()->state() == KviMdiChild::Minimized);
	else
		return QWidget::isMinimized();
}

/**
 * ============== isMaximized? ==================
 */
bool KviWindow::isMaximized()
{
	if( mdiParent() )
		return (mdiParent()->state() == KviMdiChild::Maximized);

	// Eh... how to check it? Empirical check!
	int wdth = (g_pApp->desktop()->width()  * 75) / 100;
	int hght = (g_pApp->desktop()->height() * 75) / 100;

	return ((x() <= 1) && (y() <= 1) && (width() >= wdth) && (height() >= hght));
}

/**
 * ============== restore ================
 */
void KviWindow::restore()
{
	if( mdiParent() ) {
		if( isMinimized() || isMaximized() )
			mdiParent()->setState(KviMdiChild::Normal);
	} else showNormal();
}

/**
 * ============== setFocusHandler ============
 */
void KviWindow::setFocusHandler(QWidget *w)
{
	m_pFocusHandler = w;
}

/**
 * =============== youAreAttached ============
 */
void KviWindow::youAreAttached(KviMdiChild *lpC)
{
	lpC->setCaption(m_szCaption);
	lpC->setIconPointer(myIconPtr()); // It is OK to set a null icon here
}

/**
 * ================ youAreDetached =============
 */
void KviWindow::youAreDetached()
{
	setCaption(m_szCaption);
	if( myIconPtr() )
		setIcon(*(myIconPtr()));
	setFocusPolicy(StrongFocus);
}

/**
 * ================ setWindowCaption ================
 */
void KviWindow::setWindowCaption(const char *szCaption)
{
	m_szCaption = szCaption;
	if( mdiParent() ) {
		mdiParent()->setCaption(m_szCaption);
		if( isMaximized() )
			m_pFrm->updateCaption();
	} else setCaption(m_szCaption);
	if( m_pTaskBarButton )
		m_pTaskBarButton->updateButton();
}

/**
 * ============== closeEvent ================
 */
void KviWindow::closeEvent(QCloseEvent *e)
{
	e->ignore(); // We ignore the event, and then close later if needed.
	m_pFrm->childWindowCloseRequest(this);
}

/**
 * ============= delayedClose ===============
 */
void KviWindow::delayedClose()
{
	QTimer::singleShot(0, this, SLOT(close()));
}

/**
 * ============== output ================
 */
void KviWindow::output(int msg_type, const char *format, ...)
{
	char txt_ptr[512]; // It should be enough for all output...
	va_list list;
	va_start(list, format);
	if( kvi_vsnprintf(txt_ptr, 512, format, list) < 0 ) {
		// Just in case
		va_end(list);
		char *long_txt_ptr = 0;
		int len            = 512;
		int result;
		do {
			len += 512;
			// First time long_txt_ptr == 0 so it is equivalent to malloc
			// At least the man page says that.
			long_txt_ptr = (char *) kvi_realloc((void *) long_txt_ptr, len);
			va_start(list, format);
			result = kvi_vsnprintf(long_txt_ptr, len, format, list);
			va_end(list);
		} while( result < 0 );
		if( m_pView )
			m_pView->appendText(msg_type, long_txt_ptr);
		else {
			// Redirect to the nearest console
			m_pFrm->m_pConsole->outputNoFmt(msg_type, long_txt_ptr);
		}
		kvi_free((void *) long_txt_ptr);
	} else {
		// Successful vsnprintf
		va_end(list);
		if( m_pView )
			m_pView->appendText(msg_type, txt_ptr);
		else {
			// Redirect to the nearest console
			m_pFrm->m_pConsole->outputNoFmt(msg_type, txt_ptr);
		}
	}
	if( m_pTaskBarButton )
		m_pTaskBarButton->highlight((msg_type == KVI_OUT_HIGHLIGHT));
}

/**
 * ================ outputNoFmt ================
 */
void KviWindow::outputNoFmt(int msg_type, const char *text)
{
	__range_valid(text);
	if( m_pView )
		m_pView->appendText(msg_type, text);
	else
		m_pFrm->m_pConsole->outputNoFmt(msg_type, text);
	if( m_pTaskBarButton )
		m_pTaskBarButton->highlight((msg_type == KVI_OUT_HIGHLIGHT));
}

/**
 * ================ myIconPtr =================
 */
QPixmap *KviWindow::myIconPtr()
{
	return 0;
}

/**
 * ================ applyOptions =================
 */
void KviWindow::applyOptions()
{
	// Nothing here
}

/**
 * =============== highlight =================
 */
void KviWindow::highlight(bool bAltColor)
{
	if( m_pTaskBarButton )
		m_pTaskBarButton->highlight(bAltColor);
}

/**
 * ============ setProgress ==============
 */
void KviWindow::setProgress(int progress)
{
	if( m_pTaskBarButton )
		m_pTaskBarButton->setProgress(progress);
}

void KviWindow::setProperties(KviWindowProperty *)
{
	// Nothing here
}

void KviWindow::saveProperties()
{
	// Nothing here
}

void KviWindow::delayedAutoRaise()
{
	QTimer::singleShot(100, this, SLOT(autoRaise()));
}

void KviWindow::autoRaise()
{
	if( isMinimized() )
		restore();
	raise();
	if( m_pFocusHandler )
		m_pFocusHandler->setFocus();
	else
		setFocus();
}

#include "m_kvi_window.moc"
