#ifndef _KVI_MDI_MANAGER_H_INCLUDED_
#define _KVI_MDI_MANAGER_H_INCLUDED_

// =============================================================================
//
//      --- kvi_mdi_manager.h ---
//
//   This file is part of the KVIrc IRC client distribution
//   Copyright (C) 1999-2000 Szymon Stefanek (stefanek@tin.it)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc, 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
// =============================================================================

#include <qframe.h>

class KviMdiChild;
class KviPopupMenu;

/**
 * Manager for KviMdiChild widgets.
 */
class KviMdiManager : public QFrame
{
	friend class KviMdiCaption;
	friend class KviMdiChild;
	Q_OBJECT
public:
	KviMdiManager(QWidget *parent, const char *name = 0);

	/**
	 * Destructor : there should be no child windows anymore...
	 * Howewer it simply deletes all the child widgets :)
	 */
	~KviMdiManager();
public:
	/**
	 * Z Order stack (top == last)
	 * Auto delete enabled.
	 */
	QPtrList<KviMdiChild> *m_pZ;
private:
	KviPopupMenu *m_pWindowMenu;
	QFont         m_captionFont;
	QColor        m_captionActiveBackColor;
	QColor        m_captionActiveForeColor;
	QColor        m_captionInactiveBackColor;
	QColor        m_captionInactiveForeColor;
	int           m_captionFontLineSpacing;
	bool          m_bOpaqueMove;
	bool          m_bOpaqueResize;
public:
	/**
	 * Appends a new KviMdiChild to this manager.<br>
	 * The child is shown, raised and gets focus if this window has it.
	 */
	void manageChild(KviMdiChild *lpC, bool bShow = true, bool bCascade = true);

	/**
	 * Destroys a KviMdiChild managed.<br>
	 * Note that if a client is attached to this child, it is deleted too!
	 */
	void destroyChild(KviMdiChild *lpC, bool bFocusTopChild = true);

	/**
	 * Brings the child lpC to the top of the stack.<br>
	 * The child is focused if bSetFocus is true
	 * otherwise is raised only.
	 */
	void setTopChild(KviMdiChild *lpC, bool bSetFocus = false);

	/**
	 * Returns the window menu... the menu is filled by this manager.<br>
	 * Be sure not to delete it :)
	 */
	KviPopupMenu *getWindowMenu();

	/**
	 * Returns the topmost child (the active one) or 0 if there are no children.
	 * Note that the topmost child may be also hidden, if ALL the windows are minimized.
	 */
	KviMdiChild *topChild();

	/**
	 * Returns the number of visible children.
	 */
	int getVisibleChildCount();

	void setMdiCaptionFont(const QFont &fnt);
	void   setMdiCaptionActiveForeColor(const QColor &clr);
	void   setMdiCaptionActiveBackColor(const QColor &clr);
	void setMdiCaptionInactiveForeColor(const QColor &clr);
	void setMdiCaptionInactiveBackColor(const QColor &clr);
	inline void   setOpaqueMove(bool bOpaque)    {   m_bOpaqueMove = bOpaque; };
	inline void setOpaqueResize(bool bOpaque)    { m_bOpaqueResize = bOpaque; };
public slots:
	/**
	 * Cascades the windows, resizing them to the minimum size.
	 */
	void cascadeWindows();

	/**
	 * Cascades the windows, resizing them to the maximum available size.
	 */
	void cascadeMaximized();
	void expandVertical();
	void expandHorizontal();
	void expand();

	/**
	 * Set focus to the topmost child.<br>
	 * In case it does not get focused automatically...<br>
	 * BTW: that should not happen.
	 */
	void focusTopChild();

	/**
	 * Forces the update of the window menu.
	 */
	void fillWindowMenu();

	/**
	 * Tile Pragma :)
	 */
	void tilePragmaHorizontal();
	void tilePragmaVertical();

	/**
	 * Tile Anodine :)
	 */
	void tileAnodine();

	void minimizeTopChild();
	void restoreTopChild();
	void closeTopChild();
protected:
	void animate(QRect &start, QRect &finish);
	void tileAllInternal(int maxWnds, bool bHorizontal);
	virtual void focusInEvent(QFocusEvent *);
	virtual void resizeEvent(QResizeEvent *);
	QPoint getCascadePoint(int indexOfWindow);
	void mousePressEvent(QMouseEvent *);
	void childMinimized(KviMdiChild *lpC, bool bWasMaximized);
	void childMaximized(KviMdiChild *lpC, bool bWasMinimized);
	void  childRestored(KviMdiChild *lpC, bool bWasMaximized);
protected slots:
	void menuActivated(int id);
signals:
	void enterSDIMode(bool bEnter);
};

#endif // _KVI_MDI_MANAGER_H_INCLUDED_
