//=============================================================================
//
//   File : kvi_uparser.cpp
//   Creation date : Sun Jul 02 2000 14:48:54 by Szymon Stefanek
//
//   This file is part of the KVirc irc client distribution
//   Copyright (C) 1999-2004 Szymon Stefanek (pragma at kvirc dot net)
//
//   This program is FREE software. You can redistribute it and/or
//   modify it under the terms of the GNU General Public License
//   as published by the Free Software Foundation; either version 2
//   of the License, or (at your opinion) any later version.
//
//   This program is distributed in the HOPE that it will be USEFUL,
//   but WITHOUT ANY WARRANTY; without even the implied warranty of
//   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
//   See the GNU General Public License for more details.
//
//   You should have received a copy of the GNU General Public License
//   along with this program. If not, write to the Free Software Foundation,
//   Inc. ,59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
//
//=============================================================================

#define __KVIRC__

#define _KVI_UPARSER_CPP_

#include "kvi_debug.h"

#include "kvi_uparser.h"
#include "kvi_window.h"
#include "kvi_out.h"
#include "kvi_command.h"
#include "kvi_error.h"
#include "kvi_locale.h"
#include "kvi_datacontainer.h"
#include "kvi_mirccntrl.h"
#include "kvi_console.h"
#include "kvi_channel.h"
#include "kvi_query.h"
#include "kvi_app.h"
#include "kvi_modulemanager.h"
//#include "kvi_event.h"
#include "kvi_options.h"
#include "kvi_scriptobject.h"
//#include "kvi_eventhandler.h"
#include "kvi_ircconnection.h"
#include "kvi_ircconnectionuserinfo.h"

#include "kvi_kvs_script.h"
#include "kvi_kvs_aliasmanager.h"
#include "kvi_kvs_eventmanager.h"
#include "kvi_kvs_eventtriggers.h"
#include "kvi_kvs_event.h"

#include <ctype.h>

KVIRC_API KviUserParser * g_pUserParser = 0;





KviUserParser::KviUserParser()
: QObject(0,"user_command_parser")
{
	m_pGlobalDataContainer = new KviDataContainer(true);

	initFunctionDict();
	initCommandDict();
}

KviUserParser::~KviUserParser()
{
	delete m_pFunctionDict;
	delete m_pCommandExecDict;
	delete m_pCommandSkipDict;
	delete m_pGlobalDataContainer;
}

bool KviUserParser::substituteIdentifiers(const char *szCommand,KviWindow * pWnd,KviStr &szRetBuffer)
{
	KviCommand cmd(szCommand,pWnd,0,0);
	return g_pUserParser->parseCmdFinalPart(&cmd,szRetBuffer);
}


void KviUserParser::printError(KviCommand *c,bool bCallStackOnly,int * stackFrames)
{
	// FIXME : Print parameters ?
	// FIXME : Simple/Detailed errors ?

	if(!bCallStackOnly && (c->m_iError != KviError_success))
	{
		QString szErr = KviError::getDescription(c->m_iError);
		c->window()->output(KVI_OUT_PARSERERROR,__tr2qs("Error in command: %c%c%Q"),
			KVI_TEXT_BOLD,KVI_TEXT_UNDERLINE,&szErr);

		if(!c->m_szErrDetail.isEmpty())
		{
			c->window()->output(KVI_OUT_PARSERERROR,__tr2qs("Error token: %c%Q"),
				KVI_TEXT_BOLD,&(c->m_szErrDetail));
		}
		int nLine = 1;
		const char * aux = c->m_szCmdBuffer.ptr();
		const char * linePtr = aux;
		// find the line in that the parsing stopped
		while(aux != c->m_ptr)
		{
			if(*aux == '\n'){
				nLine++;
				linePtr = ++aux;
			} else ++aux;
		}
		int chIdx = c->m_ptr - linePtr;
		KviStr tmp = linePtr;

		tmp.insert(chIdx,KVI_TEXT_REVERSE);

		chIdx++;

		if(chIdx < (tmp.len() - 1))
		{
			tmp.insert(chIdx + 1,KVI_TEXT_RESET);
		} else {
			tmp.append(" "); // need a trailing space to show the char (we were pointing to the null terminator!)
			tmp.append(KVI_TEXT_RESET);
		}

		if(tmp.len() > 50)
		{
			if(chIdx > 30)
			{
				tmp.cutLeft(chIdx - 25);
				tmp.prepend("...");
			}
		}
		bool bAddPoints = false;
		int theLen = tmp.len();
		tmp.cutFromFirst('\n');
		if(theLen < tmp.len())bAddPoints = true;
		if(tmp.len() > 50)
		{
			tmp.setLen(50);
			bAddPoints = true;
		}
		if(bAddPoints)tmp.append("...");

		c->window()->output(KVI_OUT_PARSERERROR,__tr2qs("Parsing stopped at %cline %d , character %d"),
			KVI_TEXT_BOLD,nLine,chIdx);
		c->window()->output(KVI_OUT_PARSERERROR,__tr2qs("   %s"),tmp.ptr());

		if(c->scopeObject())
		{
			if(g_pScriptObjectController->objectExists(c->scopeObject()))
			{
				c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr2qs("Object scope: name (%s), class (%s)"),
					c->scopeObject()->name(),c->scopeObject()->getClass()->name());
			} else {
				c->m_pWnd->outputNoFmt(KVI_OUT_PARSERERROR,__tr2qs("Object scope: deleted object"));
			}
		}
		if(c->thisPointer())
		{
			if(g_pScriptObjectController->objectExists(c->thisPointer()))
			{
				c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr2qs("This pointer: name (%s), class (%s)"),
					c->thisPointer()->name(),c->thisPointer()->getClass()->name());
			} else {
				c->m_pWnd->outputNoFmt(KVI_OUT_PARSERERROR,__tr2qs("This pointer: deleted object"));
			}
		}
		c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr2qs("Internal call stack:"));
	}

	int parentCount = 0;
	KviCommand * par = c->parent();
	while(par)
	{
		parentCount++;
		par = par->parent();
	}

	int i = stackFrames ? *stackFrames : 0;

	if(i <= 25)
	{
		for(const char * ch=c->m_pContextStack->last();ch;ch=c->m_pContextStack->prev())
		{
			i++;
			if(i == 1)c->m_pWnd->output(KVI_OUT_PARSERERROR,"   L%c%d::%d: %s",
				KVI_TEXT_BOLD,parentCount,i,ch);
			else c->m_pWnd->output(KVI_OUT_PARSERERROR,"   L%d::%d: %s",
				parentCount,i,ch);
			if(i > 25)
			{
				c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr2qs("   Current command stack depth: %d frames (stopping output at 25th frame)"),c->m_pContextStack->count());
				break;
			}
		}
	} else {
		c->m_pWnd->output(KVI_OUT_PARSERERROR,__tr2qs("   Parent command stack depth: %d frames"),c->m_pContextStack->count());
	}
		
	if(c->parent()){
		if(stackFrames){
			*stackFrames += c->m_pContextStack->count();
			printError(c->parent(),true,stackFrames);
		} else {
			int iStackCount = c->m_pContextStack->count();
			printError(c->parent(),true,&iStackCount);
		}
	}
}

bool KviUserParser::parseUserCommandFromTextInput(QString &szBuffer,KviWindow * pWnd)
{
	KviStr tmp = szBuffer.utf8().data(); // we always encode in utf8

	// skip leading spaces and newlines
	char * c = tmp.ptr();
	while((*c == ' ') || (*c == '\n'))c++;
	if(!*c)return true; // empty command ?

	// check for the leading \ escape
	if(*c == '\\')
	{
		c++;
		if(*c != '/')c--;
	} else {
		if(*c == '/')
		{
			c++;
			// avoid the c++ comments pasted :D
			if(*c != '/')return parseCommandBuffer(c,pWnd);
			c--;
		}
	}

	if(KVS_TRIGGER_EVENT_1_HALTED(KviEvent_OnTextInput,pWnd,c))return true;

	QString szCmdCopy = szBuffer;
	if(c != tmp.ptr())szCmdCopy.remove(0,c - tmp.ptr());
	parseNonCommand(szCmdCopy,pWnd);
	return true;
}

bool KviUserParser::parseUserCommand(KviStr &buffer,KviWindow *wnd)
{
	// skip leading spaces and newlines
	char * c = buffer.ptr();
	while((*c == ' ') || (*c == '\n'))c++;
	if(!*c)return true; // empty command ?

	// check for the leading \ escape
	if(*c == '\\')
	{
		c++;
		if(*c != '/')c--;
	} else {
		if(*c == '/')
		{
			c++;
			// avoid the c++ comments pasted :D
			if(*c != '/')return parseCommandBuffer(c,wnd);
			c--;
		}
	}

	QString szCmd = c;
	parseNonCommand(szCmd,wnd);
	return true;
}

void KviUserParser::parseNonCommand(const QString &c,KviWindow *wnd)
{
	const QChar * aux = KviQString::nullTerminatedArray(c);
	const QChar * beg = aux;
	if (!aux) return;

	while(aux->unicode())
	{
		while(aux->unicode() && (aux->unicode() != '\n'))aux++;
		QString buf(beg,aux-beg);
		if(aux->unicode() == '\n')aux++;
		beg = aux;

		if(buf.isEmpty())buf = " "; // avoid "No text to send" (d3vah)

		switch(wnd->type())
		{
			case KVI_WINDOW_TYPE_CONSOLE:
				if(wnd->connection())
				{
					QCString data = wnd->connection()->encodeText(buf);
					if(((KviConsole *)wnd)->connection()->sendData(data.data()))
					{
						wnd->output(KVI_OUT_RAW,"[RAW]: %Q",&buf);
						return;
					}
				}
				wnd->output(KVI_OUT_PARSERERROR,__tr2qs("You are not connected to a server"));
			break;
			case KVI_WINDOW_TYPE_CHANNEL:
			case KVI_WINDOW_TYPE_QUERY:
				if(wnd->connection())
				{
					if(KVI_OPTION_BOOL(KviOption_boolExitAwayOnInput)) 
						if(wnd->connection()->userInfo()->isAway())
							parseCommandBuffer("back",wnd->console());
				}
				wnd->ownMessage(buf);
			break;
			case KVI_WINDOW_TYPE_DCCCHAT:
				wnd->ownMessage(buf);
			break;
			default:
				// FIXME: Should pass the message somewhere ?.. a KviWindow handler ?
			break;
		}
	}
}


bool KviUserParser::parseCommandBuffer(char * buffer,KviWindow *wnd,KviParameterList * params)
{
#ifdef COMPILE_NEW_KVS
	#ifndef COMPILE_ONLY_NEW_KVS
	if(*buffer == '-')
	{
		buffer++;
	#endif
		KviKvsVariantList vList;
		vList.setAutoDelete(true);
		if(params)
			for(KviStr * s = params->first();s;s = params->next())
				vList.append(new KviKvsVariant(QString(s->ptr())));
		KviKvsScript kvs("commandline",buffer);
		kvs.run(wnd,&vList,0,KviKvsScript::PreserveParams /*| KviKvsScript::AssumeLocals*/);
		return true;
	#ifndef COMPILE_ONLY_NEW_KVS
	}
	#endif
#endif

#ifndef COMPILE_ONLY_NEW_KVS
	KviCommand cmd(buffer,wnd);
	if(params)cmd.setParams(params);
	if(!parseCommand(&cmd))
	{
		if(cmd.hasError())
		{
			printError(&cmd);
			return false;
		}
	}
	return true;
#endif
}

bool KviUserParser::parseCommand(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"kvirc::parseCommand");

#ifdef COMPILE_NEW_KVS
	#ifdef COMPILE_ONLY_NEW_KVS
		KviKvsVariantList vList;
		vList.setAutoDelete(true);
		if(c->params())
			for(KviStr * s = c->params()->first();s;s = c->params()->next())
				vList.append(new KviKvsVariant(QString(s->ptr())));
		KviKvsScript kvs("old_parser_wrapper",c->m_ptr);
		kvs.run(c->window(),&vList,0,KviKvsScript::PreserveParams /*| KviKvsScript::AssumeLocals*/);
		return c->leaveStackFrame();
	#endif
#endif

	// This one parses a whole command buffer executing it
	// Returns false in case of execution stopped
	// If cmd->err is non-zero in that moment , it
	// means that an error occured , otherwise the execution was simply halted
	c->skipWhiteSpace();
	while(*(c->m_ptr))
	{
		if(*(c->m_ptr) == '{')
		{
			if(!parseCommandBlock(c))return false;
			// Here c->m_ptr points to the char immediately after the closing brace
		} else {
			if(!parseSingleCommand(c))return false;
			// Here c->m_ptr points to the char immediately after the separator...
		}
		c->skipWhiteSpace();
	}
	// All ok.
	return c->leaveStackFrame();
}

bool KviUserParser::parseCommandBlock(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"kvirc::parseCommandBlock");

	__range_valid(*(c->m_ptr) == '{');
	++(c->m_ptr);
	c->skipWhiteSpace();
	while(*(c->m_ptr))
	{
		switch(*(c->m_ptr))
		{
			case '}': ++(c->m_ptr);             return c->leaveStackFrame(); break;
			case '{': if(!parseCommandBlock(c)) return false;             break;
			default : if(!parseSingleCommand(c))return false;             break;
		}
		c->skipWhiteSpace();
	}
	// We should newer reach this point
	return c->error(KviError_missingClosingBrace);
}


// FIXME: #warning "ADD EXAMPLES OF WINDOW REBINDING"

bool KviUserParser::parseSingleCommand(KviCommand *c)
{
	/*

	ENTER_STACK_FRAME(c,"kvirc::parseSingleCommand");

	// Ok , we first check for comments
	if(*(c->m_ptr) == '#')
	{
		skipComment(c);
		return c->leaveStackFrame();
	}

	if(*(c->m_ptr) == '@')
	{
		// $this pointer
		if(c->scopeObject())return c->error(KviError_scopeObjectAlreadyDefined);
		if(!c->thisPointer())return c->error(KviError_noThisObject);

		c->setScopeObject(c->thisPointer());
		// new object scope
		++(c->m_ptr);
		if((*(c->m_ptr) != '%') && (*(c->m_ptr) != '$'))return c->error(KviError_variableOrIdentifierExpected);

		return (parseLValueCommand(c) ? c->leaveStackFrame() : false);
	}

	// not a comment...look for lValue stuff , variables and function calls
	if((*(c->m_ptr) == '%') || (*(c->m_ptr) == '$'))
	{
		return (parseLValueCommand(c) ? c->leaveStackFrame() : false);
	}

	// empty command ?
	if(*(c->m_ptr) == ';'){
		++(c->m_ptr);
		return c->leaveStackFrame();
	}

	// ok...it must be a full command at this point
	char * end = c->m_ptr;
	end++;
	while(isalnum(*end) || (*end == '_'))end++;

	KviStr command;

	// check if it is a module name
	if(*end == '.')
	{
		// a module command
		KviStr module(c->m_ptr,end);

		KviModule * pModule = g_pModuleManager->getModule(module.ptr());
		if(!pModule)return c->error(KviError_errorInLoadingModule,__tr("%s (module %s)"),g_pModuleManager->lastError().ptr(),module.ptr());

		c->m_ptr = ++end;


		while(isalnum(*end) || (*end == '_') || (*end == '.'))end++;
		command.extractFromString(c->m_ptr,end); // FIXME: make the module dict case sensitive and add KviStr::extractStringToUpper() ?
		//command.toUpper();

		c->m_ptr = end;

		KviModuleCommandParseProc * mProc = pModule->findCommandParseProc(command.ptr());
		if(!mProc)
		{
			mProc = pModule->genericCommandParseProc();
			if(!mProc)return c->error(KviError_noSuchModuleCommand,__tr("Module '%s', Command '%s'"),module.ptr(),command.ptr());
		}

		if(!extractSwitches(c))return false;

		KviWindow *pOld = 0;

		if(c->hasSwitch('r'))
		{
			// temp rebinding requested
			KviStr winId;
			if(c->getSwitchValue('r',winId))
			{
				KviWindow * pAux = g_pApp->findWindow(winId.ptr());
				if(pAux){
					pOld = c->window();
					c->rebindToWindow(pAux);
				} else c->warning(__tr2qs("Can't rebind the command: window with id %s not found"),winId.ptr());
			} else c->warning(__tr2qs("Can't rebind the command: missing window identifier after the -r switch"));
		}

		c->setCurrentEntity(command.ptr());
		if(! (*mProc)(pModule,c) )
		{
			if(pOld)c->rebindToWindow(pOld);
			return false;
		} else {
			if(pOld)c->rebindToWindow(pOld);
			return c->leaveStackFrame();
		}
	
	}

	// not a module name... it is a complete command

	command.extractFromString(c->m_ptr,end);
	// use ISO88591 mapping to avoid problems with strange locales (like Turkish that maps i to Ý)
	command.toUpperISO88591();

	// lookup the builtin ones
	KviCommandParseProc * proc = m_pCommandExecDict->find(command.ptr());

	if(proc)
	{
		// builtin command
		c->m_ptr = end;
		if(!extractSwitches(c))return false;

		KviWindow *pOld = 0;

		if(c->hasSwitch('r'))
		{
			// temp rebinding requested
			KviStr winId;
			if(c->getSwitchValue('r',winId))
			{
				KviWindow * pAux = g_pApp->findWindow(winId.ptr());
				if(pAux){
					pOld = c->window();
					c->rebindToWindow(pAux);
				} else c->warning(__tr2qs("Can't rebind the command: window with id %s not found"),winId.ptr());
			} else c->warning(__tr2qs("Can't rebind the command: missing window identifier after the -r switch"));
		}

		c->setCurrentEntity(command.ptr());
		if(! ((this->*(proc->proc))(c)) )
		{
			if(pOld)c->rebindToWindow(pOld);
			return false;
		} else {
			if(pOld)c->rebindToWindow(pOld);
			return c->leaveStackFrame();
		}
	}

	// not a builtin command: alias ?

	const KviKvsScript * a = KviKvsAliasManager::instance()->lookup(command.ptr());
	if(a)
	{
		// Yes...an alias
		// We can easily check for infinite recursion here
		*/
		/*
		if(KVI_OPTION_BOOL(KviOption_boolLimitAliasRecursion))
		{
			if(a->lock() > 1024)
			{
				a->totalUnlock();
				return c->error(KviError_recursionTooDeep);
			}
		} else {
			// In fact the locking mechanism will stop to work if we exceed 2^32 iterations
			// assuming that unsigned long is 32 bits wide. So we still try to
			// avoid such monster recursions. Anyway, we will prolly never get it
			// to be executed: we will get a stack overflow first (2^32 quite large stack frames
			// can't live in our addressing space: the stack pointer would overflow!)
			// So this check is not a real limit (at least on 32 bit systems)
			if(a->lock() > 0xFFFFFFFE)
			{
				a->totalUnlock();
				return c->error(KviError_recursionTooDeep);
			}
		}
		*/
		/*
		c->m_ptr = end;
		if(!extractSwitches(c))return false;

		// extract the parameters...
		KviParameterList * l = new KviParameterList();
		
		c->skipSpace();
		while(*(c->m_ptr) && (*(c->m_ptr) != ';') && (*(c->m_ptr) != '\n'))
		{
			KviStr * pNewParam = new KviStr();
			if(!parseCmdSingleToken(c,*pNewParam))
			{
				delete pNewParam;
				delete l;
				return false;
			}
			l->append(pNewParam);
			c->skipSpace();
		}

		if(*(c->m_ptr))
		{
			c->m_ptr++;
			c->skipWhiteSpace();
		}

		// We want nice output in case of errors...
		ENTER_STACK_FRAME(c,a->name());

		// child command buffer
		KviCommand cmd(QCString(a->code()).data(),c->window(),c);
		cmd.setParams(l,false); // do not transfer param ownership
		cmd.setAliasSwitchList(c->m_pSwitchList);

		// parse it
		if(!parseCommand(&cmd))
		{
			if(cmd.hasError())
			{
				printError(&cmd);
				delete l;
				return false; // this is a "halt" for us
			}
		}

		// propagate the return value
		c->m_szRetBuffer= cmd.m_szRetBuffer;

		delete l;

		c->leaveStackFrame();

		return c->leaveStackFrame();
	}


	// not an alias...
	// check if the user wants to send it as "raw" to server
	if(KVI_OPTION_BOOL(KviOption_boolSendUnknownCommandsAsRaw))
	{
		if(c->window()->console())
		{
			if(c->window()->console()->isConnected())
			{
				KviStr parbuf;
				c->m_ptr = end;
				if(!parseCmdFinalPart(c,parbuf))return false;
				c->window()->connection()->sendFmtData("%s %s",c->window()->connection()->encodeText(command.ptr()).data(),c->window()->connection()->encodeText(parbuf.ptr()).data());
				return c->leaveStackFrame();
			}
		}
	}

	// nope...trigger an error: this is a syntactic one...better to stop
	KviStr tmp(c->m_ptr,end);
	return c->error(KviError_unknownCommand,tmp.ptr());
	*/
	return false;
}


void KviUserParser::completeModuleCommand(const QString &mod,const QString &cmd,KviPtrList<QString> * matches)
{
	KviModule * m = g_pModuleManager->getModule(mod.utf8().data());
	if(!m)return;
	//m->completeCommand(cmd,matches);
}

void KviUserParser::completeModuleFunction(const QString &mod,const QString &cmd,KviPtrList<QString> * matches)
{
	KviModule * m = g_pModuleManager->getModule(mod.utf8().data());
	if(!m)return;
	//m->completeFunction(cmd,matches);
}

void KviUserParser::completeCommand(const QString &cmd,KviPtrList<QString> * matches)
{
	// get a list of commands that match
	if(cmd.find('.') != -1)
	{
		// this is a module name!
		// load it!
		QString szMod = cmd;
		KviQString::cutFromFirst(szMod,QChar('.'));
		QString szCmd = cmd;
		KviQString::cutToFirst(szCmd,QChar('.'));
		completeModuleCommand(szMod,szCmd,matches);
		return;
	}

	QAsciiDictIterator<KviCommandParseProc> it(*m_pCommandExecDict);

	while(it.current())
	{
		if(KviQString::equalCIN(cmd,it.currentKey(),cmd.length()))
		{
			QString * x = new QString(it.currentKey());
			*x = x->lower(); // bleah
			matches->append(x);
		}
		++it;
	}

	// now match the modules
	KviPtrList<QString> tmp;
	tmp.setAutoDelete(true);
	g_pModuleManager->completeModuleNames(cmd,&tmp);
	for(QString * s = tmp.first();s;s = tmp.next())completeModuleCommand(*s,QString::null,matches);

	KviKvsAliasManager::instance()->completeCommand(cmd,matches);
	// FIXME: complete aliases too!
}

void KviUserParser::completeFunction(const QString &cmd,KviPtrList<QString> * matches)
{
	if(cmd.contains('.'))
	{
		// this is a module name!
		// load it!
		QString szMod = cmd;
		KviQString::cutFromFirst(szMod,'.');
		QString szCmd = cmd;
		KviQString::cutToFirst(szCmd,'.');
		completeModuleFunction(szMod,szCmd,matches);
		return;
	}

	QAsciiDictIterator<KviFunctionParseProc> it(*m_pFunctionDict);

	while(it.current())
	{
		if(KviQString::equalCIN(cmd,it.currentKey(),cmd.length()))
		{
			QString * x = new QString(it.currentKey());
			*x = x->lower();
			matches->append(x);
		}
		++it;
	}

	// now match the modules
	KviPtrList<QString> tmp;
	tmp.setAutoDelete(true);
	g_pModuleManager->completeModuleNames(cmd,&tmp);
	for(QString * s = tmp.first();s;s = tmp.next())completeModuleFunction(*s,QString::null,matches);

	KviKvsAliasManager::instance()->completeCommand(cmd,matches);
}

KviPtrList<QString> * KviUserParser::completeCommandAllocateResult(const QString &szInit)
{
KviPtrList<QString> * l = new KviPtrList<QString>;
l->setAutoDelete(true);
completeCommand(szInit,l);
return l;
}
 KviPtrList<QString> * KviUserParser::completeFunctionAllocateResult(const QString &szInit)
{
KviPtrList<QString> * l = new KviPtrList<QString>;
l->setAutoDelete(true);
completeFunction(szInit,l);
return l;
}

void KviUserParser::freeCompletionResult(KviPtrList<QString> * l)
{
delete l;
};

bool KviUserParser::parseCallbackCommand(KviCommand * c,KviParameterList * parms,KviStr * cmd)
{
	ENTER_STACK_FRAME(c,"kvirc::parseCallbackCommand");

	c->skipWhiteSpace();
	if(*(c->m_ptr) != '(')return c->error(KviError_openParenthesisExpected);

	if(!extractFunctionParameters(c,parms))return false;

	c->skipWhiteSpace();

	//if(*(c->m_ptr) != '{')return c->error(KviError_openBraceExpected);
	const char * aux = c->m_ptr;
	if(!skipCommand(c))return false;

	cmd->extractFromString(aux,c->m_ptr);
	// Don't need real formatting here: it's just an internal buffer
	cmd->cutLeft(1);
	cmd->cutRight(1);
	cmd->stripWhiteSpace();

	return c->leaveStackFrame();
}

bool KviUserParser::triggerEvent(int index,KviWindow *wnd,KviParameterList * params,bool bTransferOwnership,bool bIsRaw)
{
	
	bool bHaltCalled = false;
	KviKvsEventHandler * s = 0;

	KviPtrList<KviKvsEventHandler> * l = bIsRaw ? KviKvsEventManager::instance()->rawHandlers(index) : KviKvsEventManager::instance()->appHandlers(index);

	if(!l)goto done_triggering_event;
	if(params)params->init();

	for(s = l->first();s; s = l->next())
	{
		if(s->type() == KviKvsEventHandler::Script)
		{
			if(!((KviKvsScriptEventHandler *)s)->isEnabled())continue;

			KviCommand cmd(((KviKvsScriptEventHandler *)s)->code().utf8().data(),wnd);

			if(params)cmd.setParams(params,false); // do not transfer param ownership
			if(!parseCommand(&cmd))
			{
				if(cmd.hasError())
				{
					printError(&cmd);
					if(bIsRaw)
					{
						KviStr tmp(KviStr::Format,"RAW%d",index);
						wnd->output(KVI_OUT_PARSERERROR,
							__tr2qs("Error triggered from raw event handler %c%s::%Q"),KVI_TEXT_BOLD,
							tmp.ptr(),&(((KviKvsScriptEventHandler *)s)->name()));
					} else {
						wnd->output(KVI_OUT_PARSERERROR,
							__tr2qs("Error triggered from event handler %c%Q::%Q"),KVI_TEXT_BOLD,
							&(KviKvsEventManager::instance()->appEvent(index)->name()),
							&(((KviKvsScriptEventHandler *)s)->name()));
					}
					if(KVI_OPTION_BOOL(KviOption_boolDisableBrokenEventHandlers))
					{
						((KviKvsScriptEventHandler *)s)->setEnabled(false);
						if(bIsRaw)
						{
							KviStr tmp(KviStr::Format,"RAW%d",index);
							wnd->output(KVI_OUT_PARSERERROR,
								__tr2qs("Raw event handler %s::%Q is broken: disabling"),
								tmp.ptr(),&(((KviKvsScriptEventHandler *)s)->name()));
						} else {
							wnd->output(KVI_OUT_PARSERERROR,
								__tr2qs("Event handler %Q::%Q is broken: disabling"),
								&(KviKvsEventManager::instance()->appEvent(index)->name()),
								&(((KviKvsScriptEventHandler *)s)->name()));
						}
					}
				} else {
					if(cmd.haltEncountered())bHaltCalled = true;
				}
			}
		} else if(s->type() == KviKvsEventHandler::OldModule)
		{
			// an old module handler
			if(!((KviKvsOldModuleEventHandler *)s)->proc()(((KviKvsOldModuleEventHandler *)s)->module(),wnd,params))bHaltCalled = true; // halt-like
		} // else s->type() == KviKvsEventHandler::Module
			// but we don't support this interface yet :)
	}

done_triggering_event:

	if(params && bTransferOwnership)delete params;
	return bHaltCalled;

	//__range_valid((index >= 0) && (index < KVI_NUM_SCRIPT_EVENTS));
	
	/*
	
	bool bHaltCalled = false;

	KviPtrList<KviEventHandler> * l = bIsRaw ? g_pEventManager->m_rawNumericEventTable[index] : g_pEventManager->m_eventTable[index].handlers;

	if(!l)goto done_triggering_event;
	if(params)params->init();

	for(KviEventHandler *s = l->first();s; s = l->next())
	{
		if(s->type() == KviEventHandler::Script)
		{
			if(((KviScriptEventHandler *)s)->enabled())
			{
				KviCommand cmd(((KviScriptEventHandler *)s)->code().utf8().data(),wnd);
				if(params)cmd.setParams(params,false); // do not transfer param ownership
				if(!parseCommand(&cmd))
				{
					if(cmd.hasError())
					{
						printError(&cmd);
						if(bIsRaw)
						{
							KviStr tmp(KviStr::Format,"RAW%d",index);
							wnd->output(KVI_OUT_PARSERERROR,
								__tr2qs("Error triggered from raw event handler %c%s::%Q"),KVI_TEXT_BOLD,
								tmp.ptr(),&(((KviScriptEventHandler *)s)->name()));
						} else {
							wnd->output(KVI_OUT_PARSERERROR,
								__tr2qs("Error triggered from event handler %c%Q::%Q"),KVI_TEXT_BOLD,
								&(g_pEventManager->m_eventTable[index].szName),&(((KviScriptEventHandler *)s)->name()));
						}
						if(KVI_OPTION_BOOL(KviOption_boolDisableBrokenEventHandlers))
						{
							((KviScriptEventHandler *)s)->setEnabled(false);
							if(bIsRaw)
							{
								KviStr tmp(KviStr::Format,"RAW%d",index);
								wnd->output(KVI_OUT_PARSERERROR,
									__tr2qs("Raw event handler %s::%Q is broken: disabling"),
									tmp.ptr(),&(((KviScriptEventHandler *)s)->name()));
							} else {
								wnd->output(KVI_OUT_PARSERERROR,
									__tr2qs("Event handler %Q::%Q is broken: disabling"),
									&(g_pEventManager->m_eventTable[index].szName),&(((KviScriptEventHandler *)s)->name()));
							}
						}
					} else {
						if(cmd.haltEncountered())bHaltCalled = true;
					}
				}
			}
		} else {
			// a module handler
			if(!((KviModuleEventHandler *)s)->proc()(((KviModuleEventHandler *)s)->module(),wnd,params))bHaltCalled = true; // halt-like
		}
	}

done_triggering_event:

	if(params && bTransferOwnership)delete params;
	return bHaltCalled;
	*/
}

bool KviUserParser::extractSwitches(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"extractSwitches");
	// This one 
	c->clearSwitchList();
	c->skipSpace();
	while(*(c->m_ptr) == '-'){
		if(!extractSingleSwitch(c))return false;
		c->skipSpace();
	}
	return c->leaveStackFrame();
}


#ifdef JMP
#undef JMP
#endif
#define JMP 1
static char command_parameter_parsing_char_table[256]=
{
	//	000 001 002 003 004 005 006 007   008 009 010 011 012 013 014 015 
	//	NUL SOH STX ETX EOT ENQ ACK BEL   BS  HT  LF  VT  FF  CR  SO  SI
		JMP,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,JMP,JMP,0  ,0  ,0  ,0  ,0  ,
	//	016 017 018 019 020 021 022 023   024 025 026 027 028 029 030 031 
	//	DLE DC1 DC2 DC3 DC4 NAK SYN ETB   CAN EM  SUB ESC FS  GS  RS  US
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	032 033 034 035 036 037 038 039   040 041 042 043 044 045 046 047 
	//	    !   "   #   $   %   &   '     (   )   *   +   ,   -   .   /   
		JMP,0  ,JMP,0  ,JMP,JMP,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	048 049 050 051 052 053 054 055   056 057 058 059 060 061 062 063 
	//	0   1   2   3   4   5   6   7     8   9   :   ;   <   =   >   ?   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,JMP,0  ,0  ,0  ,0  ,
	//	064 065 066 067 068 069 070 071   072 073 074 075 076 077 078 079 
	//	@   A   B   C   D   E   F   G     H   I   J   K   L   M   N   O   
		JMP,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	080 081 082 083 084 085 086 087   088 089 090 091 092 093 094 095 
	//	P   Q   R   S   T   U   V   W     X   Y   Z   [   \   ]   ^   _   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,JMP,0  ,0  ,0  ,
	//	096 097 098 099 100 101 102 103   104 105 106 107 108 109 110 111 
	//	`   a   b   c   d   e   f   g     h   i   j   k   l   m   n   o   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	112 113 114 115 116 117 118 119   120 121 122 123 124 125 126 127 
	//	p   q   r   s   t   u   v   w     x   y   z   {   |   }   ~      
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	128 129 130 131 132 133 134 135   136 137 138 139 140 141 142 143 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	144 145 146 147 148 149 150 151   152 153 154 155 156 157 158 159 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	160 161 162 163 164 165 166 167   168 169 170 171 172 173 174 175 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	176 177 178 179 180 181 182 183   184 185 186 187 188 189 190 191 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	192 193 194 195 196 197 198 199   200 201 202 203 204 205 206 207 
	//	�  �  �  �  �  �  �  �    �  �  �  �  �  �  �  �  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	208 209 210 211 212 213 214 215   216 217 218 219 220 221 222 223 
	//	�  �  �  �  �  �  �  �    �  �  �  �  �  �  �  �  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	224 225 226 227 228 229 230 231   232 233 234 235 236 237 238 239 
	//	�  �  �  �  �  �  �  �    �  �  �  �  �  �  �  �  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	240 241 242 243 244 245 246 247   248 249 250 251 252 253 254 255 
	//	�  �  �  �  �  �  �  �                            
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
};
#undef JMP



bool KviUserParser::extractSingleSwitch(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"kvirc::extractSingleSwitch");

	__range_valid(*(c->m_ptr) == '-');

	++(c->m_ptr); // skip the - char

	// now a LETTER is required
	if( (!(*(c->m_ptr))) || (!isalpha(*(c->m_ptr))) )
	{
		return c->error(KviError_switchDashWithoutSwitchLetter);
	}

	KviStr * sw = new KviStr();
	sw->append(*(c->m_ptr));

	++(c->m_ptr); // skip the switch letter

	c->skipSpace();

	// now if we have a '=' , a token follows the switch

	if(*(c->m_ptr) == '=')
	{
		// a token follows the switch
		++(c->m_ptr);
		c->skipSpace();
		sw->append('=');

		char *aux_ptr = c->m_ptr;

		bool bInString = false;


		for(;;){
			// Now skip all the non interesting chars
			while(!command_parameter_parsing_char_table[*((unsigned char *)aux_ptr)])aux_ptr++;
			// Interesting char
			switch(*aux_ptr){
				case ' ' :
				case '\t':
				case ';' :
					if(!bInString){ // End of the token...append the last block to the buffer and return
						sw->append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
						c->m_pSwitchList->append(sw);
						return c->leaveStackFrame();
					} else while((*aux_ptr == ' ')||(*aux_ptr == '\t')||
								(*aux_ptr == ';'))aux_ptr++; //In string...must skip it
				break;
				case '\n': // end of command
				case '\0': // end of command buffer...append the last block to the buffer and return
					if(bInString){
						// unsecaped newline or end of data
						delete sw;
						return c->error(KviError_unexpectedEndInString);
					} // else it is the end of the command
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					c->m_pSwitchList->append(sw);
					return c->leaveStackFrame();
				break;
				case '\"': // beginning or end of a string
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					bInString = !bInString;
					c->m_ptr = ++aux_ptr;
				break;
				case '@':
					++aux_ptr;
					if(((*aux_ptr == '%') || (*aux_ptr == '$')) && c->thisPointer() && (!c->scopeObject()))
					{
						sw->append(c->m_ptr,(aux_ptr - c->m_ptr - 1));
						c->m_ptr = aux_ptr;
						c->setScopeObject(c->thisPointer());
					}
				break;
				case '%': //variable: append the last block to the buffer and process the var
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseVariable(c,*sw)){
						delete sw;
						return false;
					}
					aux_ptr = c->m_ptr;
				break;
				case '$': //system identifier: append the last block to the buffer and process the ident
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseIdentifier(c,*sw)){
						delete sw;
						return false;
					}
					aux_ptr = c->m_ptr;
				break;
				case '\\': //escape character: append the last block to the processed buffer...
					sw->append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					switch(*aux_ptr){
						case '\0':
							if(bInString){
								delete sw;
								return c->error(KviError_unexpectedEndInString);
							}
							c->m_pSwitchList->append(sw);
							return c->leaveStackFrame();
						break;
						case '\n':  //escaped newline
							c->m_ptr = ++aux_ptr; //skip it
							c->skipSpace();       //skip leading spaces
							aux_ptr = c->m_ptr;   //continue
						break;
					case 'r':
						if(bInString)sw->append('\r');
						else sw->append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(bInString)sw->append('\n');
						else sw->append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
						default: // Must be copied to the buffer without modification
							sw->append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
					}
				break;
			}
		}
	}

	// not '='
	c->m_pSwitchList->append(sw);

	return c->leaveStackFrame();
}

bool KviUserParser::parseCmdSingleToken(KviCommand *c,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"kvirc::parseCmdSingleToken");

	// This one must be called when c->m_ptr points
	// somewhere in a command buffer.
	// It extracts the first token encountered and
	// preprocesses it (subst variables , etc...)
	// then it appends everything to the c->m_buffer.
	// The parsing is stopped by a '\n',' ',';' or '\0'
	// Note that this function does nothing if c->m_ptr
	// points to the end of a command...

	// Skip leading spaces...even the escaped ones
	c->skipSpace();

	char *aux_ptr = c->m_ptr;

	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		while(!command_parameter_parsing_char_table[*((unsigned char *)aux_ptr)])aux_ptr++;

		// Interesting char
		switch(*aux_ptr){
			case ' ' :
			case '\t':
			case ';' :
				if(!bInString){ // End of the token...append the last block to the buffer and return
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					return c->leaveStackFrame();
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t')||
							(*aux_ptr == ';'))aux_ptr++; //In string...must skip it
			break;
			case '\n': // end of command
			case '\0': // end of command buffer...append the last block to the buffer and return
				if(bInString){
					// unsecaped newline or end of data
					return c->error(KviError_unexpectedEndInString);
				} // else it is the end of the command
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				return c->leaveStackFrame();
			break;
			case '\"': // beginning or end of a string
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr;
			break;
			case '@':
				++aux_ptr;
				if(((*aux_ptr == '%') || (*aux_ptr == '$')) && c->thisPointer() && (!c->scopeObject()))
				{
					buffer.append(c->m_ptr,(aux_ptr - c->m_ptr - 1));
					c->m_ptr = aux_ptr;
					c->setScopeObject(c->thisPointer());
				}
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(bInString)return c->error(KviError_unexpectedEndInString);
						return c->leaveStackFrame();
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(bInString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(bInString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}


bool KviUserParser::parseCmdFinalPart(KviCommand *c,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"kvirc::parseCmdFinalPart");

	// Preprocesses the final part of a standard command...
	// It is stopped by a newline , a ';' or the end of the buffer.
	// This one extracts the command parameters , evaluates it , 
	// and appends everything to cmd->buffer.
	// skip spaces...but keep the terminators!!!

	c->skipSpace();
	char *aux_ptr = c->m_ptr;

	bool bInString = false;

	for(;;)
	{
		// Now skip all the non interesting chars
		/*
		while(*aux_ptr && (*aux_ptr!='\\') && (*aux_ptr!='\n') &&
				(*aux_ptr!=' ') && (*aux_ptr!='\t') && (*aux_ptr!=';') &&
				(*aux_ptr!='$') && (*aux_ptr!='\"') &&
				(*aux_ptr!='%'))aux_ptr++;
		*/

		while(!command_parameter_parsing_char_table[*((unsigned char *)aux_ptr)])aux_ptr++;


		// Interesting char
		switch(*aux_ptr)
		{
			case '\0': // end of command buffer...
			case '\n':
				if(bInString){
					// ops....end of buffer while looking for matching '"'
					return c->error(KviError_unexpectedEndInString);
				}
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(*aux_ptr == '\n')aux_ptr++; // point after the semicolon
				c->m_ptr = aux_ptr;
				return c->leaveStackFrame();
			break;
			case '\"': // beginning or end of a string
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				bInString = !bInString;
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!bInString)
				{
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != '\0') && (*aux_ptr != '\n') &&
						(*aux_ptr != ';'))buffer.append(' ');

				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case ';' :
				if(!bInString){ // end of command...
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr; //don't forget to point to next char
					return c->leaveStackFrame();
				} else ++aux_ptr; // just skip it
			break;
			case '@':
				++aux_ptr;
				if(((*aux_ptr == '%') || (*aux_ptr == '$')) && c->thisPointer() && (!c->scopeObject()))
				{
					buffer.append(c->m_ptr,(aux_ptr - c->m_ptr - 1));
					c->m_ptr = aux_ptr;
					c->setScopeObject(c->thisPointer());
				}
			break;
			case '%': //variable
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(bInString)return c->error(KviError_unexpectedEndInString);
						else return c->leaveStackFrame();
					break; //escaped nothing...
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(bInString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(bInString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
			default:
				aux_ptr++;
				debug("ops.. char %c (%d) jumping from table but it's unhandled",*aux_ptr,*aux_ptr);
			break;
		}
	}
	return c->leaveStackFrame();
}


bool KviUserParser::parseCmdUpTo(KviCommand *c,KviStr &buffer,char terminator)
{
	ENTER_STACK_FRAME(c,"kvirc::parseCmdUpTo");

	// Preprocesses the final part of a standard command...
	// It is stopped by a newline , a ';' the end of the buffer or the terminator char.
	// This one extracts the command parameters , evaluates it , 
	// and appends everything to cmd->buffer.
	// skip spaces...but keep the terminators!!!

	c->skipSpace();
	char *aux_ptr = c->m_ptr;
	bool bInString = false;

	for(;;){
		// Now skip all the non interesting chars
		while(!command_parameter_parsing_char_table[*((unsigned char *)aux_ptr)] && (*aux_ptr != terminator))aux_ptr++;

		if(*aux_ptr == terminator)
		{
			if(!bInString)
			{ // end of command...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr; //don't forget to point to next char
				return c->leaveStackFrame();
			} else ++aux_ptr; // just skip it

		} else {
			// Interesting char
			switch(*aux_ptr)
			{
				case '\0': // end of command buffer...
				case '\n':
					if(bInString){
						// ops....end of buffer while looking for matching '"'
						return c->error(KviError_unexpectedEndInString);
					}
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					if(*aux_ptr == '\n')aux_ptr++; // point after the semicolon
					c->m_ptr = aux_ptr;
					return c->leaveStackFrame();
				break;
				case '\"': // beginning or end of a string
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					bInString = !bInString;
					c->m_ptr = ++aux_ptr; //don't forget to point to next char
				break;
				case ' ': // Spaces...need to reduce it to one...
				case '\t':
					if(!bInString)
					{
						// append the last part
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						// skip the other spaces including escaped ones
						c->m_ptr = aux_ptr;
						c->skipSpace();
						aux_ptr = c->m_ptr;
						if((*aux_ptr != '\0') && (*aux_ptr != '\n') &&
							(*aux_ptr != ';'))buffer.append(' ');
	
					} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
				break;
				case ';' :
					if(!bInString){ // end of command...
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = ++aux_ptr; //don't forget to point to next char
						return c->leaveStackFrame();
					} else ++aux_ptr; // just skip it
				break;
				case '@':
					++aux_ptr;
					if(((*aux_ptr == '%') || (*aux_ptr == '$')) && c->thisPointer() && (!c->scopeObject()))
					{
						buffer.append(c->m_ptr,(aux_ptr - c->m_ptr - 1));
						c->m_ptr = aux_ptr;
						c->setScopeObject(c->thisPointer());
					}
				break;
				case '%': //variable
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseVariable(c,buffer))return false;
					aux_ptr = c->m_ptr;
				break;
				case '$': //system identifier
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = aux_ptr;
					if(!parseIdentifier(c,buffer))return false;
					aux_ptr = c->m_ptr;
				break;
				case '\\': //escape character: append the last block to the processed buffer
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					switch(*aux_ptr){
						case '\0':
							if(bInString)return c->error(KviError_unexpectedEndInString);
							else return c->leaveStackFrame();
						break; //escaped nothing...
						case '\n':  //escaped newline
							c->m_ptr = ++aux_ptr; //skip it
							c->skipSpace();       //skip leading spaces
							aux_ptr = c->m_ptr;   //continue
						break;
						case 'r':
							if(bInString)buffer.append('\r');
							else buffer.append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
						case 'n':
							if(bInString)buffer.append('\n');
							else buffer.append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
						default: // Must be copied to the buffer without modification
							buffer.append(*aux_ptr);
							c->m_ptr = ++aux_ptr;
						break;
					}
				break;
			}
		}
	}
	return c->leaveStackFrame();
}



bool KviUserParser::lookupDataType(KviCommand *c,KviDataType *d)
{
	ENTER_STACK_FRAME(c,"kvirc::lookupDataType");
	// %<name> : variable
	// %<name>[<key>] : dictionary
	// %<name>[] : dictionary reference
	// ->%[a-z_.]+ : object scope variable
	// %[A-Z][a-z_.]+ : global variable
	// %[a-z][a-z_.]+ : local variable
	// %:[A-Za-z_.]+ : special scope variable

	__range_valid(*c->m_ptr == '%');
	char * aux = ++(c->m_ptr);

	if(c->scopeObject())d->pDataContainer = c->scopeObject()->dataContainer();
	else {
		if(*aux == ':')
		{
			d->pDataContainer = c->extendedScopeDataContainer();
			if(!d->pDataContainer)d->pDataContainer = c->m_pDataContainer;
			aux = ++(c->m_ptr);
		} else {
			if(isupper(*aux))d->pDataContainer = m_pGlobalDataContainer;
			else d->pDataContainer = c->m_pDataContainer;
		}
	}

	while(isalnum(*aux) || (*aux == '.') || (*aux == '_'))aux++;
	d->szName.extractFromString(c->m_ptr,aux);

	c->m_ptr = aux;

	if(d->szName.isEmpty())
	{
		d->iDataType = KVI_DATA_TYPE_NONE;
		return c->leaveStackFrame();
	}

	if(*(c->m_ptr) == '{')
	{
		++(c->m_ptr);
		// looks like a dictionary
		// no scope object for the key
		KviScriptObject * back = c->scopeObject();
		c->setScopeObject(0);
		if(!parseVariableKey(c,d->szKey))
		{
			c->setScopeObject(back);
			return false;
		}
		c->setScopeObject(back);

		d->pDictionary = d->pDataContainer->lookupDictionary(d->szName.ptr(),true);
		if(d->szKey.isEmpty())
		{
			if(*(c->m_ptr) == '@')
			{
				++(c->m_ptr);	
				d->iDataType = KVI_DATA_TYPE_DICTKEYSREFERENCE;
			} else if(*(c->m_ptr) == '#')
			{
				++(c->m_ptr);
				d->iDataType = KVI_DATA_TYPE_DICTITEMCOUNT;
			} else d->iDataType = KVI_DATA_TYPE_DICTREFERENCE;
		} else {
			d->iDataType = KVI_DATA_TYPE_DICTVARIABLE;
			d->pVariable = d->pDataContainer->lookupDictionaryVariable(d->szName.ptr(),d->szKey.ptr(),true);
		}
	} else if(*(c->m_ptr) == '[')
	{
		++(c->m_ptr);
		// looks like an array
		// no scope object for the index
		KviScriptObject * back = c->scopeObject();
		c->setScopeObject(0);
		if(!parseVariableIndex(c,d->szKey))
		{
			c->setScopeObject(back);
			return false;
		}
		c->setScopeObject(back);

		d->pArray = d->pDataContainer->lookupArray(d->szName.ptr(),true);
		if(d->szKey.isEmpty())
		{
			if(*(c->m_ptr) == '#')
			{
				++(c->m_ptr);
				d->iDataType = KVI_DATA_TYPE_ARRAYITEMCOUNT;
			} else d->iDataType = KVI_DATA_TYPE_ARRAYREFERENCE;
		} else {
			bool bOk;
			d->uIndex = d->szKey.toUInt(&bOk);
			if(!bOk)return c->error(KviError_invalidArrayIndex);
			d->iDataType = KVI_DATA_TYPE_ARRAYVARIABLE;
			d->pVariable = d->pDataContainer->lookupArrayVariable(d->szName.ptr(),d->uIndex,true);
		}
	} else {
		// it is a variable
		d->iDataType = KVI_DATA_TYPE_VARIABLE;
		d->pVariable = d->pDataContainer->lookupVariable(d->szName.ptr(),true);
	}

	return c->leaveStackFrame();
}

bool KviUserParser::parseLValueCommand(KviCommand *c)
{
	ENTER_STACK_FRAME(c,"kvirc::parseLValueCommand");
	__range_valid((*(c->m_ptr)=='%') || (*(c->m_ptr) == '$'));
	//__range_invalid(c->scopeObject());

	bool bLastWasAVariable = false;

	KviDataType d;
	KviStr dummy;

	do {
		if(*(c->m_ptr) == '%')
		{
			bLastWasAVariable = true;
			if(!parseVariable(c,dummy,&d,false))return false;
		} else if(*(c->m_ptr) == '$')
		{
			bLastWasAVariable = false;
			if(!parseIdentifier(c,dummy))return false;
		} else return c->error(KviError_variableOrIdentifierExpected);
	} while(c->scopeObject());
	// check object scope

	if(bLastWasAVariable)
	{
		switch(d.iDataType)
		{
			case KVI_DATA_TYPE_VARIABLE:
				if(!d.pVariable)d.pVariable = d.pDataContainer->lookupVariable(d.szName.ptr(),false); // RW access
			break;
			case KVI_DATA_TYPE_DICTVARIABLE:
				if(!d.pDictionary)d.pDictionary = d.pDataContainer->lookupDictionary(d.szName.ptr(),false);
				if(!d.pVariable)d.pVariable = d.pDataContainer->lookupDictionaryVariable(d.szName.ptr(),d.szKey.ptr(),false);
			break;
			case KVI_DATA_TYPE_DICTREFERENCE:
			case KVI_DATA_TYPE_DICTKEYSREFERENCE:
			case KVI_DATA_TYPE_DICTITEMCOUNT:
				if(!d.pDictionary)d.pDictionary = d.pDataContainer->lookupDictionary(d.szName.ptr(),false);
			break;
			case KVI_DATA_TYPE_ARRAYVARIABLE:
				if(!d.pArray)d.pArray = d.pDataContainer->lookupArray(d.szName.ptr(),false);
				if(!d.pVariable)d.pVariable = d.pDataContainer->lookupArrayVariable(d.szName.ptr(),d.uIndex,false);
			break;
			case KVI_DATA_TYPE_ARRAYREFERENCE:
			case KVI_DATA_TYPE_ARRAYITEMCOUNT:
				if(!d.pArray)d.pArray = d.pDataContainer->lookupArray(d.szName.ptr(),false);
			break;
			case KVI_DATA_TYPE_NONE:
				// this will be error in parseOperator
				return c->error(KviError_missingVariableName);
			break;
		}

		c->skipSpace();
		if(!parseOperator(c,&d))
		{
			// may we have created unnecessary empty variables ?
			// .... yes...this is ugly
			switch(d.iDataType)
			{
				case KVI_DATA_TYPE_VARIABLE:
					if(d.pVariable->isEmpty())d.pDataContainer->removeVariable(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_DICTVARIABLE:
					if(d.pVariable->isEmpty())d.pDataContainer->removeDictionaryVariable(d.szName.ptr(),d.szKey.ptr());
					//if(d.pDictionary->count() == 0)d.pDataContainer->removeDictionary(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_DICTREFERENCE:
				case KVI_DATA_TYPE_DICTKEYSREFERENCE:
				case KVI_DATA_TYPE_DICTITEMCOUNT:
					if(d.pDictionary->count() == 0)d.pDataContainer->removeDictionary(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_ARRAYVARIABLE:
					if(d.pVariable->isEmpty())d.pDataContainer->removeArrayVariable(d.szName.ptr(),d.uIndex);
					//if(d.pArray->size() == 0)d.pDataContainer->removeArray(d.szName.ptr());
				break;
				case KVI_DATA_TYPE_ARRAYREFERENCE:
				case KVI_DATA_TYPE_ARRAYITEMCOUNT:
					if(d.pArray->size() == 0)d.pDataContainer->removeArray(d.szName.ptr());
				break;
			}
			return false;
		}

		return c->leaveStackFrame();
	} else {
		// function call (probably object function)
		return parseCmdFinalPart(c,dummy) ? c->leaveStackFrame() : false;
	}

	// never here

	return c->leaveStackFrame();
}

bool KviUserParser::parseVariable(KviCommand *c,KviStr &buffer,KviDataType * d,bool bIncludeLists)
{
	ENTER_STACK_FRAME(c,"kvirc::parseVariable");
	// This one parses the variable pointed by ptr
	// adds its value to c->m_buffer
	// (or an empty string if the var is not defined)
	// then moves c->m_ptr to the characted immediately 
	// after the variable and returns

	KviDataType dataTypeBuffer;
	if(!d)d = &dataTypeBuffer; // local one

	if(!lookupDataType(c,d))return false;

	KviStr tmpBuffer;

	switch(d->iDataType)
	{
		case KVI_DATA_TYPE_NONE:
			tmpBuffer = '%';
		break;
		case KVI_DATA_TYPE_VARIABLE:
		case KVI_DATA_TYPE_DICTVARIABLE:
		case KVI_DATA_TYPE_ARRAYVARIABLE:
			if(d->pVariable)tmpBuffer = *(d->pVariable);
		break;
		case KVI_DATA_TYPE_DICTITEMCOUNT:
			if(d->pDictionary)tmpBuffer.sprintf("%d",d->pDictionary->count());
			else tmpBuffer = '0';
		break;
		case KVI_DATA_TYPE_DICTKEYSREFERENCE:
			if(d->pDictionary && bIncludeLists)
			{
				KviDictionaryIterator it(*(d->pDictionary));
				bool bFirst = true;
				while(it.current())
				{
					if(bFirst)bFirst = false;
					else tmpBuffer.append(',');

					tmpBuffer.append(it.currentKey());
					++it;
				}
			}
		break;
		case KVI_DATA_TYPE_DICTREFERENCE:
			if(d->pDictionary && bIncludeLists)
			{
				KviDictionaryIterator it(*(d->pDictionary));
				bool bFirst = true;
				while(it.current())
				{
					if(bFirst)bFirst = false;
					else tmpBuffer.append(',');

					tmpBuffer.append(*(it.current()));
					++it;
				}
			}
		break;
		case KVI_DATA_TYPE_ARRAYITEMCOUNT:
			if(d->pArray)tmpBuffer.sprintf("%d",d->pArray->size());
			else tmpBuffer = '0';
		break;
		case KVI_DATA_TYPE_ARRAYREFERENCE:
			if(d->pArray && bIncludeLists)
			{
				for(unsigned int u = 0;u < d->pArray->size();u++)
				{
					if(u > 0)tmpBuffer.append(',');
					KviStr * t = d->pArray->uncheckedAt(u);
					if(t)tmpBuffer.append(*t);
				}
			}
		break;
		default:
			__range_valid(false); //ooops...
		break;
	}

	if(*(c->m_ptr) == '-')
	{
		// scope object ?
		char * aux = c->m_ptr;
		++aux;
		if(*aux == '>')
		{
			// yup...it looks as!
			++aux;
			// ok..we expect a variable or function call now
			if((*aux == '%') || (*aux == '$'))
			{
				c->m_ptr = aux;
				// yup...object scope operator
				KviScriptObject * o = g_pScriptObjectController->lookupObject(tmpBuffer.ptr());
				if(!o)return c->error(KviError_noSuchObject,__tr("Variable evaluated to \"%s\""),tmpBuffer.ptr());
				c->setScopeObject(o);
				return c->leaveStackFrame(); //yup...new object scope
			} // else just a "->"....? :)
		}
	}

	c->setScopeObject(0); // evaluated the variable in the object scope...back to global scope

	buffer.append(tmpBuffer);

	return c->leaveStackFrame();
}


bool KviUserParser::parseVariableKey(KviCommand *c,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"kvirc::parseVariableKey");
	// This one must be called when c->m_ptr points
	// somewhere in a variable parameters block.
	// The parsing is stopped by a '\n', '}' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if '}'
	// is in a string (delimitated by two '"' chars)
	// Skips the ending '}' character
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A key like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces ,

	// Skip leading spaces and tabs (the escaped ones too)
	c->skipSpace();
	register char *aux_ptr = c->m_ptr;

	bool inString      = false;

	for(;;){
		// First skip all the non interesting chars
		while(	*aux_ptr         && (*aux_ptr!='%') &&
				(*aux_ptr!='\n') && (*aux_ptr!='}')  && (*aux_ptr!='$')  && 
				(*aux_ptr!=' ')  && (*aux_ptr!='"')  && (*aux_ptr!='\\') &&
				(*aux_ptr!='\t'))aux_ptr++;
		// Interesting char
		switch(*aux_ptr){
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)return c->error(KviError_unexpectedEndInString);
				else return c->error(KviError_unexpectedEndInDictionaryKey);
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!inString){
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != '}'))buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case '}': // If not in string , exit
				if(!inString){
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					c->m_ptr = ++aux_ptr;
					return c->leaveStackFrame();
				}
				++aux_ptr;
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '@':
				++aux_ptr;
				if(((*aux_ptr == '%') || (*aux_ptr == '$')) && c->thisPointer() && (!c->scopeObject()))
				{
					buffer.append(c->m_ptr,(aux_ptr - c->m_ptr - 1));
					c->m_ptr = aux_ptr;
					c->setScopeObject(c->thisPointer());
				}
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr){
					case '\0':
						if(inString)return c->error(KviError_unexpectedEndInString);
						else return c->error(KviError_unexpectedEndInDictionaryKey);
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(inString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(inString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}


bool KviUserParser::parseVariableIndex(KviCommand *c,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"kvirc::parseVariableIndex");
	// This one must be called when c->m_ptr points
	// somewhere in a variable parameters block.
	// The parsing is stopped by a '\n', ']' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if ']'
	// is in a string (delimitated by two '"' chars)
	// Skips the ending ']' character
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A key like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces ,

	// Skip leading spaces and tabs (the escaped ones too)
	c->skipSpace();

	register char *aux_ptr = c->m_ptr;

	for(;;)
	{
		// First skip all the non interesting chars
		while(	*aux_ptr         && (*aux_ptr!='%') &&
				(*aux_ptr!='\n') && (*aux_ptr!=']')  && (*aux_ptr!='$')  && 
				(*aux_ptr!=' ')  && (*aux_ptr!='\\') &&
				(*aux_ptr!='\t'))aux_ptr++;
		// Interesting char
		switch(*aux_ptr)
		{
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				return c->error(KviError_unexpectedEndInArrayIndex);
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				// append the last part
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				// skip the other spaces including escaped ones
				c->m_ptr = aux_ptr;
				c->skipSpace();
				aux_ptr = c->m_ptr;
				if((*aux_ptr != ']'))return c->error(KviError_unexpectedCharactersInArrayIndex);
			break;
			case ']': // If not in string , exit
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				return c->leaveStackFrame();
			break;
			case '@':
				++aux_ptr;
				if(((*aux_ptr == '%') || (*aux_ptr == '$')) && c->thisPointer() && (!c->scopeObject()))
				{
					buffer.append(c->m_ptr,(aux_ptr - c->m_ptr - 1));
					c->m_ptr = aux_ptr;
					c->setScopeObject(c->thisPointer());
				}
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseVariable(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(!parseIdentifier(c,buffer))return false;
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr)
				{
					case '\0':
						return c->error(KviError_unexpectedEndInArrayIndex);
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
}

bool KviUserParser::parseParameterIdentifier(KviCommand *c,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"kvirc::parseParameterIdentifier");
	// This one expects c->m_ptr to point to a digit or '#'
	__range_valid(isdigit(*(c->m_ptr)) || ((*(c->m_ptr))=='#') || ((*(c->m_ptr)) == '$'));
	if(*(c->m_ptr) == '#')
	{
		c->getParamCount(buffer);
		++(c->m_ptr);
		return c->leaveStackFrame();
	}
	if(*(c->m_ptr) == '$')
	{
		if(c->thisPointer())buffer.append(c->thisPointer()->id());
		else buffer.append('0');
		++(c->m_ptr);
		return c->leaveStackFrame();
	}
	int param = (*(c->m_ptr) - '0');
	++(c->m_ptr);
	while(isdigit(*(c->m_ptr)))
	{
		param = param * 10;
		param += (*(c->m_ptr) - '0');
		++(c->m_ptr);
	}
	if(*(c->m_ptr) == '-'){
		++(c->m_ptr);
		if(isdigit(*(c->m_ptr)))
		{
			int parEnd = (*(c->m_ptr) - '0');
			++(c->m_ptr);
			while(isdigit(*(c->m_ptr)))
			{
				parEnd = parEnd * 10;
				parEnd += (*(c->m_ptr) - '0');
				++(c->m_ptr);
			}
			c->getParam(param,parEnd,buffer);
		} else {
			if(*(c->m_ptr) == '>')
			{
				// ops!...was an object scope operator!
				--(c->m_ptr);
				c->getSingleParam(param,buffer);
			} else c->getParam(param,-1,buffer); //up to the end
		}
	} else c->getSingleParam(param,buffer);
	return c->leaveStackFrame();
}

bool KviUserParser::parseObjectFunction(KviCommand *c,KviStr &buffer)
{
	ENTER_STACK_FRAME(c,"kvirc::parseObjectFunction");

	__range_valid(c->scopeObject());

	KviStr classOverride;

	char * aux = c->m_ptr;

	while(isalnum(*aux) || (*aux == '_'))aux++;

	if(*aux == ':')
	{
		classOverride.extractFromString(c->m_ptr,aux);
		c->m_ptr = ++aux;
		if(*(c->m_ptr) == ':') // allow the C++ syntax class::function
		{
			++(c->m_ptr);
			++aux;
		}
		while(isalnum(*aux) || (*aux == '_'))aux++;
	}

	KviStr funcName(c->m_ptr,aux);
	c->m_ptr = aux;

	KviParameterList * paramList = new KviParameterList;
	paramList->setAutoDelete(true);
	
	if(!extractFunctionParameters(c,paramList))
	{
		delete paramList;
		return false;
	}

	KviScriptObjectFunctionHandler * h = c->scopeObject()->lookupFunctionHandler(funcName.ptr(),
			classOverride.hasData() ? classOverride.ptr() : 0);

	if(!h)
	{
		if(classOverride.hasData())
			return c->error(KviError_noSuchObjectFunction,"%s::%s",classOverride.ptr(),funcName.ptr());
		else
			return c->error(KviError_noSuchObjectFunction,"%s",funcName.ptr());
	}

	if(!callObjectFunction(h,paramList,buffer,c))return false;

	delete paramList;

	return c->leaveStackFrame();
}

bool KviUserParser::callObjectFunction(KviScriptObjectFunctionHandler * h,KviParameterList * paramList,
	KviStr &buffer,KviCommand *c)
{
	ENTER_STACK_FRAME(c,"kvirc::callObjectFunction");
	paramList->init();

	if(h->szScriptHandler.hasData())
	{
		KviCommand cmd(h->szScriptHandler.ptr(),c->window(),c);
		cmd.setThisPointer(c->scopeObject());
		cmd.setParams(paramList,false);
		if(!parseCommand(&cmd))
		{
			if(cmd.hasError())
			{
				printError(&cmd);
				return false; // We could do better ? (just a halt)
			}
		}
		buffer = cmd.m_szRetBuffer;
	} else {
		if(h->proc)
		{
			if(!(((c->scopeObject())->*(h->proc))(c,paramList,buffer)))return false;
		}
	}

	//debug("CALLED OBJECT FUNCTION for object (%s)",c->scopeObject()->id());

	return c->leaveStackFrame();
}


bool KviUserParser::parseIdentifier(KviCommand *c,KviStr &buffer)
{

	//
	// This is a rather huge function that parses an identifier
	// This is something as $identifier , $module.function , 
	// $1 , $2 , $$ , $# , $alias...
	//

	/*

	ENTER_STACK_FRAME(c,"kvirc::parseIdentifier");

	__range_valid(*c->m_ptr == '$');

	++(c->m_ptr);              // skip the $

	// This is our return buffer
	KviStr tmpBuffer;

	if(c->scopeObject())
	{
		// We have a scope object... it must be an object function
		if(!parseObjectFunction(c,tmpBuffer))return false;
		goto check_object_scope;
	}

	if(isdigit(*(c->m_ptr))||(*(c->m_ptr) == '#')||(*(c->m_ptr) == '$'))
	{
		// this must be a parameter identifier
		if(!parseParameterIdentifier(c,tmpBuffer))return false;
		goto check_object_scope;
	}

	if(*(c->m_ptr) == '{')
	{
		// A command call
		// FIXME for KVS : this sucks because it doesn't
		// propagate the local variables :/
		char * aux = c->m_ptr;
		if(!skipCommand(c))return false;
		KviCommand cmd(aux,c->m_ptr,c->window(),c);
		if(!parseCommand(&cmd))
		{
			if(cmd.hasError())
			{
				printError(&cmd);
				return false; // We could do better ? (just a halt)
			}
		}
		tmpBuffer = cmd.m_szRetBuffer;
		goto check_object_scope;
	}

	if(*(c->m_ptr) == '(')
	{
		// an expression to evaluate
		long res;
		++(c->m_ptr);
		if(!evaluateExpression(c,&res))return false;
		tmpBuffer.setNum(res);
		goto check_object_scope;
	}


	{
		// Normal identifier
		KviModule * pModule = 0;
		KviModuleFunctionParseProc * mProc = 0;
		KviFunctionParseProc * proc = 0;
	
		char * aux = c->m_ptr;
	
		while(*aux && (isalnum(*aux)||(*aux == '_')))aux++;
	
		KviStr ident(c->m_ptr,aux);       // and get the ident name
		c->m_ptr = aux;

		if(ident.isEmpty())
		{
			tmpBuffer.append('$');  // single $...nope
			goto processing_done;
		}
	

		if(*(c->m_ptr) == '.')
		{
			// module function
			++(c->m_ptr);

			pModule = g_pModuleManager->getModule(ident.ptr());
			if(!pModule)return c->error(KviError_errorInLoadingModule,__tr("%s (module %s)"),g_pModuleManager->lastError().ptr(),ident.ptr());

			aux = c->m_ptr;

			while(isalnum(*aux) || (*aux == '_') || (*aux == '.'))aux++;
			ident.extractFromString(c->m_ptr,aux); // FIXME: extractFromStringToUpper ???
			//ident.toUpper();
	
			c->m_ptr = aux;
	
			mProc = pModule->findFunctionParseProc(ident.ptr());
			if(!mProc)
			{
				mProc = pModule->genericFunctionParseProc();
				if(!mProc)return c->error(KviError_noSuchModuleFunction,__tr("Module '%s', Function '%s'"),pModule->name(),ident.ptr());
			}

			goto extract_parameters;
		}

	    // have an identifier
		// use ISO88591 mapping to avoid problems with strange locales (like Turkish that maps i to Ý)
		ident.toUpperISO88591(); // this is broken anyway :(

		// FIXME: We could use the $this pointer to allow
		//        calling the member functions without the $$->
		//        We could lookup the object's function here
		//        and check for class overrides too
		//        this has problems tough.. with global names shading (also for wariables!)

		proc = m_pFunctionDict->find(ident.ptr());
	
extract_parameters:
	
		KviParameterList * paramList = new KviParameterList();
		paramList->setAutoDelete(true);
	
		if(!extractFunctionParameters(c,paramList))
		{
			delete paramList;
			return false;
		}
	
	
		paramList->init(); // point to the first item

		c->setCurrentEntity(ident.ptr());

		if(proc)
		{
			// internal function
			if(!(this->*(proc->proc))(c,paramList,tmpBuffer))
			{
				delete paramList;
				return false;
			}
			delete paramList;
			goto check_object_scope;
		}
	
		// was a module function ?
		if(mProc)
		{
			if(!(*mProc)(pModule,c,paramList,tmpBuffer))
			{
				delete paramList;
				return false;
			}
			delete paramList;
			goto check_object_scope;
		}
	
		// check for aliases
	
		const KviKvsScript * a = KviKvsAliasManager::instance()->lookup(ident.ptr());
		if(a)
		{
			// Yes...an alias
			// We can easily check for infinite recursion here
			*/
			/*
			if(KVI_OPTION_BOOL(KviOption_boolLimitAliasRecursion))
			{
				if(a->lock() > 1024)
				{
					a->totalUnlock();
					return c->error(KviError_recursionTooDeep);
				}
			} else {
				// In fact the locking mechanism will stop to work if we exceed 2^32 iterations
				// assuming that unsigned long is 32 bits wide. So we still try to
				// avoid such monster recursions. Anyway, we will prolly never get it
				// to be executed: we will get a stack overflow first (2^32 quite large stack frames
				// can't live in our addressing space: the stack pointer would overflow!)
				// So this check is not a real limit (at least on 32 bit systems)
				if(a->lock() > 0xFFFFFFFE)
				{
					a->totalUnlock();
					return c->error(KviError_recursionTooDeep);
				}
			}
			*/
			/*
			// We want nice output in case of errors...
			ENTER_STACK_FRAME(c,QCString(a->name()).data());
		
			// child command buffer
			QString tmp = a->code();
			KviCommand cmd(QCString(tmp).data(),c->window(),c);
			cmd.setParams(paramList,false); // do not transfer param ownership

			// parse it
			if(!parseCommand(&cmd))
			{
				if(cmd.hasError())
				{
					printError(&cmd);
					delete paramList;
					return false; // this is a "halt" for us
				}
			}
		
			// propagate the return value
			tmpBuffer = cmd.m_szRetBuffer;
		
			delete paramList;
		
			c->leaveStackFrame();
	
			goto check_object_scope;
		}
	
		delete paramList;
		return c->error(KviError_unknownFunction,"%s",ident.ptr());
	}
	
check_object_scope:	

	// check object scope
	if(*(c->m_ptr) == '-')
	{
		// scope object ?
		char * aux = c->m_ptr;
		++aux;
		if(*aux == '>')
		{
			// yup...it looks as!
			++aux;
			// ok..we expect a variable or function call now
			if((*aux == '%') || (*aux == '$'))
			{
				c->m_ptr = aux;
				// yup...object scope operator
				KviScriptObject * o = g_pScriptObjectController->lookupObject(tmpBuffer.ptr());
				if(!o)return c->error(KviError_noSuchObject,__tr("Identifier evaluated to \"%s\""),tmpBuffer.ptr());
				c->setScopeObject(o);
				return c->leaveStackFrame(); //yup...new object scope
			} else {
				return c->error(KviError_variableOrIdentifierExpected);
			}
		}
	}

processing_done:

	c->setScopeObject(0); //evaluated in the object scope...back to global scope now

	buffer.append(tmpBuffer);
	return c->leaveStackFrame();
	*/
	return false;
}

bool KviUserParser::extractParameters(KviCommand *c,KviParameterList *paramList)
{
	/*
	// This one extracts a list of comma separated parameters and puts them in the paramList
	// The parameter list is terminated by a ')' character.

	ENTER_STACK_FRAME(c,"kvirc::extractParameters");
	c->skipSpace();
	if(*(c->m_ptr) != ')')
	{

		KviScriptObject * o = c->scopeObject();
		c->setScopeObject(0);
		// the ident is followed by parameters...
		for(;;)
		{
			if(!parseFncSingleParam(c,paramList))
			{
				c->setScopeObject(o);
				return false;
			} else {
				__range_valid((*(c->m_ptr)==',')||(*(c->m_ptr)==')'));
				if(*(c->m_ptr) == ')')
				{
					++(c->m_ptr);
					break;
				} else ++(c->m_ptr);
			}
		}
		c->setScopeObject(o);
	} else ++(c->m_ptr);
	return c->leaveStackFrame();
	*/
	return false;
}


bool KviUserParser::extractFunctionParameters(KviCommand *c,KviParameterList * paramList)
{
	/*
	// This one extracts the parameters of a function and puts them in the paramList
	// If the first character is not '(' , the function is assumed to have no parameters.
	//	ENTER_STACK_FRAME(c,"extractFunctionParameters"); <-- let's avoid a stack entry

	if(*(c->m_ptr) == '(')
	{
		++(c->m_ptr);
		if(!extractParameters(c,paramList))return false;
	}
	//	return c->leaveStackFrame(); <-- let's avoid such a simple stack entry
	*/
	return true;
}

#ifdef JMP
#undef JMP
#endif
#define JMP 1
static char function_parameter_parsing_char_table[256]=
{
	//	000 001 002 003 004 005 006 007   008 009 010 011 012 013 014 015 
	//	NUL SOH STX ETX EOT ENQ ACK BEL   BS  HT  LF  VT  FF  CR  SO  SI
		JMP,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,JMP,JMP,0  ,0  ,0  ,0  ,0  ,
	//	016 017 018 019 020 021 022 023   024 025 026 027 028 029 030 031 
	//	DLE DC1 DC2 DC3 DC4 NAK SYN ETB   CAN EM  SUB ESC FS  GS  RS  US
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	032 033 034 035 036 037 038 039   040 041 042 043 044 045 046 047 
	//	    !   "   #   $   %   &   '     (   )   *   +   ,   -   .   /   
		JMP,0  ,JMP,0  ,JMP,JMP,0  ,0    ,JMP,JMP,0  ,0  ,JMP,0  ,0  ,0  ,
	//	048 049 050 051 052 053 054 055   056 057 058 059 060 061 062 063 
	//	0   1   2   3   4   5   6   7     8   9   :   ;   <   =   >   ?   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	064 065 066 067 068 069 070 071   072 073 074 075 076 077 078 079 
	//	@   A   B   C   D   E   F   G     H   I   J   K   L   M   N   O   
		JMP,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	080 081 082 083 084 085 086 087   088 089 090 091 092 093 094 095 
	//	P   Q   R   S   T   U   V   W     X   Y   Z   [   \   ]   ^   _   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,JMP,0  ,0  ,0  ,
	//	096 097 098 099 100 101 102 103   104 105 106 107 108 109 110 111 
	//	`   a   b   c   d   e   f   g     h   i   j   k   l   m   n   o   
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	112 113 114 115 116 117 118 119   120 121 122 123 124 125 126 127 
	//	p   q   r   s   t   u   v   w     x   y   z   {   |   }   ~      
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	128 129 130 131 132 133 134 135   136 137 138 139 140 141 142 143 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	144 145 146 147 148 149 150 151   152 153 154 155 156 157 158 159 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	160 161 162 163 164 165 166 167   168 169 170 171 172 173 174 175 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	176 177 178 179 180 181 182 183   184 185 186 187 188 189 190 191 
	//	                                                  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	192 193 194 195 196 197 198 199   200 201 202 203 204 205 206 207 
	//	�  �  �  �  �  �  �  �    �  �  �  �  �  �  �  �  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	208 209 210 211 212 213 214 215   216 217 218 219 220 221 222 223 
	//	�  �  �  �  �  �  �  �    �  �  �  �  �  �  �  �  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	224 225 226 227 228 229 230 231   232 233 234 235 236 237 238 239 
	//	�  �  �  �  �  �  �  �    �  �  �  �  �  �  �  �  
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,
	//	240 241 242 243 244 245 246 247   248 249 250 251 252 253 254 255 
	//	�  �  �  �  �  �  �  �                            
		0  ,0  ,0  ,0  ,0  ,0  ,0  ,0    ,0  ,0  ,0  ,0  ,0  ,0  ,0  ,0
};
#undef JMP

bool KviUserParser::parseFncSingleParam(KviCommand *c,KviParameterList * paramList,bool bSkipIdentAndVars)
{
	/*
	ENTER_STACK_FRAME(c,"kvirc::parseFncSingleParam");

	// This one must be called when c->m_ptr points
	// somewhere in a function parameters block.
	// It extracts the first parameter encountered and
	// preprocesses it (subst variables , etc...)
	// then it appends everything to the buffer
	// The parsing is stopped by a '\n', ',' , ')' or '\0'
	// on '\n' and '\0' an error is returned
	// The parsing is NOT stopped if one of ',' or ')'
	// is in a string (delimitated by two '"' chars)
	// Note that this function does nothing if c->m_ptr
	// points to the end of the param buffer...
	// (eg. c->m_ptr comes from previous calls to processFncSingleParam())
	// It will also skip matching parenthesis :
	// so () is a valid parameter
	// It will return false in case of an error ;
	// If it returns true , c->m_ptr surely points to a ',' or a ')'
	// This function EATS leading and trailing spaces and tabs
	// and reduces the middle ones to one per group.
	// A param like this:
	//          string    "full  of  words"  and  spaces      ,
	// Becomes like this:
	// string full  of  words and spaces,

	// Skip leading spaces and tabs (the escaped ones too)

	c->skipSpace();

	register char *aux_ptr = c->m_ptr;

	int parLevel       = 0;
	bool inString      = false;

	KviStr buffer;



	for(;;){
		// First skip all the non interesting chars
		while(!function_parameter_parsing_char_table[(unsigned char)(*aux_ptr)])aux_ptr++;

		// Interesting char
		switch(*aux_ptr)
		{
			case '\0':
			case '\n': // End of the string ('\n' is not escaped)...fail here
				if(inString)return c->error(KviError_unexpectedEndInString);
				else return c->error(KviError_unexpectedEndInFunctionParams);
			break;
			case ' ': // Spaces...need to reduce it to one...
			case '\t':
				if(!inString)
				{
					// append the last part
					buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
					// skip the other spaces including escaped ones
					c->m_ptr = aux_ptr;
					c->skipSpace();
					aux_ptr = c->m_ptr;
					if((*aux_ptr != ')') && (*aux_ptr != ','))buffer.append(' ');
				} else while((*aux_ptr == ' ')||(*aux_ptr == '\t'))aux_ptr++;
			break;
			case '"': // Not escaped '"'...enter or exit the string (skip the '"')
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				inString = !inString;
			break;
			case '(': // Keep track of matching parenthesis (if not in string)
				++aux_ptr;
				if(!inString)++parLevel;
			break;
			case ')': // If not in string , exit if parenthesis level is 0 , otherwise decrease it
				if(!inString)
				{
					if(parLevel == 0)
					{
						buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
						c->m_ptr = aux_ptr;
						paramList->append(new KviStr(buffer));
						return c->leaveStackFrame();
					} else --parLevel;
				}
				++aux_ptr;
			break;
			case ',': // end of command buffer...append the last block to the buffer and return
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				if(!inString)
				{
					c->m_ptr = aux_ptr;
					paramList->append(new KviStr(buffer));
					return c->leaveStackFrame();
				}
				c->m_ptr = aux_ptr++; //in string...',' must be included
			break;
			case '@':
				++aux_ptr;
				if(((*aux_ptr == '%') || (*aux_ptr == '$')) && c->thisPointer() && (!c->scopeObject()))
				{
					buffer.append(c->m_ptr,(aux_ptr - c->m_ptr - 1));
					c->m_ptr = aux_ptr;
					if(!bSkipIdentAndVars)c->setScopeObject(c->thisPointer());
				}
			break;
			case '%': //variable: append the last block to the buffer and process the var
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;

				if(bSkipIdentAndVars)
				{
					if(!skipVariable(c))return false;
					buffer.append(aux_ptr,c->m_ptr - aux_ptr);
				} else {
					KviDataType d;
					if(!parseVariable(c,buffer,&d,inString))return false;
					if(!inString)
					{
						switch(d.iDataType)
						{
							case KVI_DATA_TYPE_DICTKEYSREFERENCE:
								if(d.pDictionary)
								{
									unsigned int cnt = d.pDictionary->count();
									if(cnt > 0)
									{
										KviDictionaryIterator it(*(d.pDictionary));
										
										buffer.append(it.currentKey());
										if(cnt > 1)
										{
											paramList->append(new KviStr(buffer));
											++it;
											cnt--;
	
											while(cnt > 1)
											{
												paramList->append(new KviStr(it.currentKey()));
												++it;
												cnt--;
											}
											buffer = it.currentKey();
										}
									}
								}
							break;
							case KVI_DATA_TYPE_DICTREFERENCE:
								if(d.pDictionary)
								{
									unsigned int cnt = d.pDictionary->count();
									if(cnt > 0)
									{
										KviDictionaryIterator it(*(d.pDictionary));
										buffer.append(*(it.current()));
										if(cnt > 1)
										{
											paramList->append(new KviStr(buffer));
											++it;
											cnt--;
											while(cnt > 1)
											{
												paramList->append(new KviStr(*(it.current())));
												++it;
												cnt--;
											}
											buffer = *(it.current());
										}
									}
								}
							break;
							case KVI_DATA_TYPE_ARRAYREFERENCE:
								if(d.pArray)
								{
									if(d.pArray->size() > 0)
									{
										unsigned int cnt = d.pArray->size() - 1;
										KviStr * t = d.pArray->uncheckedAt(0);
										if(t)buffer.append(*t);
										if(cnt > 0)
										{
											paramList->append(new KviStr(buffer));
											unsigned int u = 1;
											for(;u < cnt;u++)
											{
												t = d.pArray->uncheckedAt(u);
												if(t)paramList->append(new KviStr(*t));
												else paramList->append(new KviStr());
											}
											t = d.pArray->uncheckedAt(u);
											if(t)buffer = *t;
										}
									}
								}
							break;
						}
					}
				}
				aux_ptr = c->m_ptr;
			break;
			case '$': //system identifier: append the last block to the buffer and process the ident
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = aux_ptr;
				if(bSkipIdentAndVars)
				{
					if(!skipIdentifier(c))return false;
					buffer.append(aux_ptr,c->m_ptr - aux_ptr);
				} else {
					KviParameterList * old = c->returnParameterList();
					c->setReturnParameterList(paramList);
					if(!parseIdentifier(c,buffer))
					{
						c->setReturnParameterList(old); // ensure that it is not deleted
						return false;
					} else c->setReturnParameterList(old);
				}
				aux_ptr = c->m_ptr;
			break;
			case '\\': //escape character: append the last block to the processed buffer...
				buffer.append(c->m_ptr,aux_ptr - c->m_ptr);
				c->m_ptr = ++aux_ptr;
				switch(*aux_ptr)
				{
					case '\0':
						if(inString)return c->error(KviError_unexpectedEndInString);
						else return c->error(KviError_unexpectedEndInFunctionParams);
					break;
					case '\n':  //escaped newline
						c->m_ptr = ++aux_ptr; //skip it
						c->skipSpace();       //skip leading spaces
						aux_ptr = c->m_ptr;   //continue
					break;
					case 'r':
						if(inString)buffer.append('\r');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					case 'n':
						if(inString)buffer.append('\n');
						else buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
					default: // Must be copied to the buffer without modification
						buffer.append(*aux_ptr);
						c->m_ptr = ++aux_ptr;
					break;
				}
			break;
		}
	}
	// Newer arrive here...
	return c->error(KviError_internalError);
	*/
	return false;
}

#include "kvi_uparser.moc"
