/***************************************************************************
                          xplanetwrapper.cpp  -  description
                             -------------------
    begin                : Mon Apr 29 2002
    copyright            : (C) 2002 by Luc Langehegermann
    email                : lx2gt@users.sourceforge.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   Great Arc File calculations taken from earthtrack.c from KD2BD        *
 *                                                                         *
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/


#include <stdio.h>
#include <stdlib.h>
#include <kprocess.h>
#include <qstringlist.h>
#include <kstandarddirs.h>
#include <kglobal.h>
#include <kmessagebox.h>
#include <klocale.h>
#include <qfile.h>
#include <qdatetime.h>
#include <qtextstream.h>
#include <qtimer.h>
#include <qregexp.h>
#include <kapplication.h>
#include <dcopclient.h>
#include <math.h>

#include <X11/Xlib.h>

#include "xplanetwrapper.h"

#define IMAGEFILE locateLocal("appdata", "world.png")
#define MARKERFILE locateLocal("appdata", "markerfile")
#define GREATARCFILE locateLocal("appdata", "greatarcfile")
#define LOGFILE locateLocal("appdata", "xplanet.log")
#define NASAFILE locateLocal("appdata", "nasa")

#define QTHCOLOR "0x00ff00"
#define ACTIVESATCOLOR "0xffff00"
#define VISIBLESATCOLOR "0xffffff"
#define SATCOLOR "0xa0a0a0"

xplanetWrapper::xplanetWrapper(QWidget *parent, const char *name ) : QWidget(parent,name)
{
  openLogFile();
  QTimer::singleShot(0, this, SLOT(callXplanet()));

  // this will make shure, we plot the image when...
  QObject::connect(&proc, SIGNAL(processExited(KProcess *)), this, SLOT(newImage(KProcess*)));
  QObject::connect(&proc, SIGNAL(receivedStderr(KProcess *, char*, int)), this, SLOT(xplanetError(KProcess*, char*, int)));
  QObject::connect(&proc, SIGNAL(receivedStdout(KProcess *, char*, int)), this, SLOT(xplanetError(KProcess*, char*, int)));
  qth=NULL;

  // get the size of the display

  Display* display = XOpenDisplay(NULL);
  int screen_num = DefaultScreen(display);
  screen_width  = DisplayWidth(display, screen_num);
  screen_height = DisplayHeight(display, screen_num);
  XCloseDisplay(display);
  trackingSatellite=NULL;
  rectangle.maxlat=9999;

  KStandardDirs* d = KGlobal::dirs();
  searchdirlist = d->findDirs("data", "ktrack");
}
xplanetWrapper::~xplanetWrapper(){
  closeLogFile();
}

void xplanetWrapper::callXplanet()
{
  // special case - we do not have an map
  if (!parameters->enabled) {
    hide();
    QTimer::singleShot(parameters->updateInterval, this, SLOT(callXplanet()));
    return;
  }

  QFile file;
  QTextStream textstream;
  satellite* sat;
  if (proc.isRunning()) return;
  int x=width();
  int y=height();

  // write the 'nasa' file

  file.setName(NASAFILE);
  if (file.open(IO_WriteOnly)) {
    QTextStream textstream(&file);
    for(sat=satlist.first(); sat!=0; sat=satlist.next()) {
      if (sat->polled()) {
        textstream << QString::number(sat->catnum()) << " altcirc=0 spacing=10 ";
        if (sat==trackingSatellite)
          textstream << "color="ACTIVESATCOLOR"\n";
        else if (sat->elevation() >=0)
            textstream << "color="VISIBLESATCOLOR"\n";
          else
            textstream << "color="SATCOLOR"\n";
      }
    }
    file.close();
  }

  // write the marker file - TODO write it only if we need to write it!

  file.setName(MARKERFILE);
  if ( file.open(IO_WriteOnly) ) {    // file opened successfully
    QTextStream textstream( &file );
    if(qth) {
      textstream << QString::number(qth->latitude()) + " ";
      textstream << QString::number(qth->longitude()) + " ";
      textstream << "\"" + qth->callsign() + "\" color="QTHCOLOR"\n";
    }
    file.close();
  }

  proc.clearArguments();
  proc << "nice" << "-n" << "15";
  proc << "xplanet";

//   QStringList::iterator it;
//   for (it = searchdirlist.end(); it != searchdirlist.begin(); --it ) {
//     proc << "-searchdir" << (*it);
//   }
  proc << "-searchdir" << searchdirlist[1];
  proc << "-searchdir" << searchdirlist[0];

  proc << "-num_times";
  proc << "1";
  proc << "-config";
  proc << "xplanetconfig";
  proc << "-output" << IMAGEFILE;
  proc << "-verbosity" << "1";
  if (!parameters->isBackground)
    proc << "-geometry" << QString::number(x)+"x"+QString::number(y);
  else
    proc << "-geometry" << QString::number(screen_width)+"x"+QString::number(screen_height);

  QString parms=parameters->parameters;
  if (trackingSatellite) {
    QString slon = QString::number(trackingSatellite->longitude());
    QString slat = QString::number(trackingSatellite->latitude());
    int rds=(int)(100.0*(6378.16/trackingSatellite->footprint()));
    if (rds < 50) rds=50;
    QString radius = QString::number(rds);
    parms.replace(QRegExp("%radius"), radius);
    parms.replace(QRegExp("%slon"), slon);
    parms.replace(QRegExp("%slat"), slat);
  }
  QStringList list = QStringList::split(' ', parms);
  for ( QStringList::Iterator it = list.begin(); it != list.end(); ++it ) {
    proc << *it;
  }

  proc.start(KProcess::NotifyOnExit, KProcess::AllOutput);
}
void xplanetWrapper::newImage(KProcess*)
{
  if (parameters->isBackground) {
    static bool i=0;
    hide();
    QByteArray data;
    QDataStream arg(data, IO_WriteOnly);
    if (i)
      arg << QString(IMAGEFILE) << ",5";
    else
      arg << QString("/"+IMAGEFILE) << ",5";
    i=!i;
    DCOPClient* cli = KApplication::kApplication()->dcopClient();
    cli->send("kdesktop", "KBackgroundIface", "setWallpaper(QString,int)", data);
    cli->send("kdesktop", "KBackgroundIface", "configure()", "");
  }
  else {
    show();
    pixmap=QPixmap(IMAGEFILE);
    repaint(false);
  }
  // toggle a new xplanet call
  QTimer::singleShot(parameters->updateInterval, this, SLOT(callXplanet()));
}

void xplanetWrapper::paintEvent(QPaintEvent*)
{
  bitBlt(this, 0, 0, &pixmap);
}

void xplanetWrapper::setSatList(QList<satellite> s)
{
  satlist=s;
}

void xplanetWrapper::writeArcFile()
{
double R0=6378.16, TWOPI=M_PI*2.0, HALFPI=M_PI/2.0, deg2rad=1.74532925199e-02;

  double beta, den, num, rangelat, rangelong, azimuth, ssplat, ssplong, oldrangelong, oldrangelat;
  double firstrangelat=0.0, firstrangelong=0.0;
  int azi;
  QFile file;
  QTextStream textstream;
  satellite* sat;

  file.setName(GREATARCFILE);
  if ( file.open(IO_WriteOnly) ) {    // file opened successfully
    QTextStream textstream( &file );
    for(sat=satlist.first(); sat!=0; sat=satlist.next()) {
      if (sat->polled()) {
        ssplat=sat->latitude()*deg2rad;
        ssplong=sat->longitude()*deg2rad;
        beta=(0.5*sat->footprint())/R0;
        oldrangelong=0.0; oldrangelat=0.0;
        for (azi=0; azi<=360; azi=azi+1) {
          azimuth=deg2rad*(double)azi;
          rangelat=asin(sin(ssplat)*cos(beta)+cos(azimuth)*sin(beta)*cos(ssplat));
          num=cos(beta)-(sin(ssplat)*sin(rangelat));
          den=cos(ssplat)*cos(rangelat);

          if (azi==0 && (beta > HALFPI-ssplat))
            rangelong=ssplong+M_PI;

          else if (azi==180 && (beta > HALFPI+ssplat))
            rangelong=ssplong+M_PI;

          else if (fabs(num/den)>1.0)
            rangelong=ssplong;

          else {
            if ((180-azi)>=0)
              rangelong=ssplong-arccos(num,den);
            else
              rangelong=ssplong+arccos(num,den);
            }

            while (rangelong<0.0)
              rangelong+=TWOPI;

            while (rangelong>TWOPI)
              rangelong-=TWOPI;



            rangelat=rangelat/deg2rad;
            rangelong=rangelong/deg2rad;

            if(sat==trackingSatellite) {
            if (azi==0)
              rectangle.maxlat=rangelat;
            if (azi==90)
              rectangle.maxlon=rangelong;
            if (azi==180)
              rectangle.minlat=rangelat;
            if (azi==270)
              rectangle.minlon=rangelong;
            }
            /* Write range circle data to greatarcfile */

            if(azi!=0) {
              textstream << QString::number(oldrangelat, 'f', 6) + " " + QString::number(oldrangelong, 'f', 6) + " ";
              textstream << QString::number(rangelat, 'f', 6) + " " + QString::number(rangelong, 'f', 6) + " spacing=0.1";
              if(sat==trackingSatellite)
                textstream << " color="ACTIVESATCOLOR"\n";
              else
                if (sat->elevation() < 0.0)
                  textstream << " color="SATCOLOR"\n";
                else
                  textstream << " color="VISIBLESATCOLOR"\n";
            }
            else {
              firstrangelat=rangelat;
              firstrangelong=rangelong;
            }
            oldrangelat=rangelat;
            oldrangelong=rangelong;

          }
          textstream << QString::number(firstrangelat, 'f', 6) + " " + QString::number(firstrangelong, 'f', 6) + " ";
          textstream << QString::number(rangelat, 'f', 6) + " " + QString::number(rangelong, 'f', 6) + " spacing=0.1";

          if(sat==trackingSatellite)
            textstream << " color="ACTIVESATCOLOR"\n";
          else
            if (sat->elevation() < 0.0)
              textstream << " color="SATCOLOR"\n";
            else
              textstream << " color="VISIBLESATCOLOR"\n";

      }
    }
    file.close();
  }
}

double xplanetWrapper::arccos(double x, double y){
  /* This function implements the arccosine function,
     returning a value between 0 and two pi. */

  double result=0.0, fraction;

  fraction=x/y;


  if ((x>0.0) && (y>0.0))
    result=acos(fraction);

  if ((x<0.0) && (y>0.0))
    result=acos(fraction);

  if ((x<0.0) && (y<0.0))
    result=M_PI+acos(fraction);

  if ((x>0.0) && (y<0.0))
    result=M_PI+acos(fraction);

  return result;
}

double xplanetWrapper::convertlong(double longitude) {
  /* This function converts west longitudes (0-360 degrees)
     to a value between -180 and 180 degrees, as required
     by xearth and xplanet. */

  if (longitude<180.0)
    longitude=-longitude;

  else if (longitude>180.0)
    longitude=360.0-longitude;

  return longitude;
}
/** sets the observer qth */
void xplanetWrapper::setObsQth(obsQTH* q)
{
  qth=q;
}
/** No descriptions */
void xplanetWrapper::xplanetError(KProcess*, char* buffer, int buflen){
  char* error = (char*)malloc(buflen+1);
  strncpy (error, buffer, buflen);
  error[buflen]=0;
  logstream << error;
  free (error);
}
/** sets the satellite we highlight (that is being tracked) */
void xplanetWrapper::setTrackingSatellite(satellite* sat)
{
  if(sat) trackingSatellite=sat;
}

/** No descriptions */
void xplanetWrapper::setParameters(s_xplanet* parms){
  parameters=parms;
}
/** Opens an file for xplanet message logging */
void xplanetWrapper::openLogFile(){
  logfile.setName(LOGFILE);
  logfile.open(IO_WriteOnly);
  logstream.setDevice(&logfile);
}
/** closes the log file */
void xplanetWrapper::closeLogFile(){
  logfile.close();
}
