/*
 *  Copyright (c) 2006 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */
#ifndef KIS_GENERIC_COLORSPACE_H_
#define KIS_GENERIC_COLORSPACE_H_

#include "kis_colorspace.h"

#include <qimage.h>

/**
 * This is not a real colorspace, but a fake colorspace you might want to use
 * to be able to use kispaintdevice for big buffer of memory.
 */
template<typename _type, int _nbchannels>
class KisGenericColorspace : public KisColorSpace {


    public:

        KisGenericColorspace() { }
        virtual ~KisGenericColorspace() { }

    public:

        //========== Channels =====================================================//

    /// Return a vector describing all the channels this color model has.
        virtual QValueVector<KisChannelInfo *> channels() const { return QValueVector<KisChannelInfo *>(); }

        virtual Q_UINT32 nChannels() const { return _nbchannels; }

        virtual Q_UINT32 nColorChannels() const { return _nbchannels; };

        virtual Q_UINT32 nSubstanceChannels() const { return 0; };
        
        virtual Q_UINT32 pixelSize() const { return _nbchannels * sizeof(_type); }

        virtual QString channelValueText(const Q_UINT8 */*pixel*/, Q_UINT32 /*channelIndex*/) const { return "invalid"; };

        virtual QString normalisedChannelValueText(const Q_UINT8 */*pixel*/, Q_UINT32 /*channelIndex*/) const { return "invalid"; };

        virtual Q_UINT8 scaleToU8(const Q_UINT8 * /*srcPixel*/, Q_INT32 /*channelPos*/) { return 0; }

        virtual Q_UINT16 scaleToU16(const Q_UINT8 * /*srcPixel*/, Q_INT32 /*channelPos*/) { return 0; }

        virtual void getSingleChannelPixel(Q_UINT8 */*dstPixel*/, const Q_UINT8 */*srcPixel*/, Q_UINT32 /*channelIndex*/) { }

        virtual KisID id() const { return KisID("genericcolorspace",""); }

        virtual Q_UINT32 colorSpaceType() { return 0; }

        virtual icColorSpaceSignature colorSpaceSignature() { return icMaxEnumData; }

        virtual bool willDegrade(ColorSpaceIndependence /*independence*/) { return true; }
    
        virtual KisCompositeOpList userVisiblecompositeOps() const { return KisCompositeOpList(); }

        virtual bool hasHighDynamicRange() const { return false; }

        virtual KisProfile * getProfile() const { return 0; }
        virtual KisProfile * getProfile() { return 0; }


        virtual void fromQColor(const QColor& /*c*/, Q_UINT8 */*dst*/, KisProfile */* profile = 0*/) { }

        virtual void fromQColor(const QColor& /*c*/, Q_UINT8 /*opacity*/, Q_UINT8 */*dst*/, KisProfile * /*profile = 0*/) { }

        virtual void toQColor(const Q_UINT8 */*src*/, QColor */*c*/, KisProfile * /*profile = 0*/) { }
        
        virtual void toQColor(const Q_UINT8 */*src*/, QColor */*c*/, Q_UINT8 */*opacity*/, KisProfile * /*profile = 0*/) { }

        virtual QImage convertToQImage(const Q_UINT8 */*data*/, Q_INT32 /*width*/, Q_INT32 /*height*/,
                                       KisProfile *  /*dstProfile*/, Q_INT32 /*renderingIntent = INTENT_PERCEPTUAL*/,
                                       float /*exposure = 0.0f*/) { return QImage(); }

        virtual void toLabA16(const Q_UINT8 * /*src*/, Q_UINT8 * /*dst*/, const Q_UINT32 /*nPixels*/) const { }

        virtual void fromLabA16(const Q_UINT8 * /*src*/, Q_UINT8 * /*dst*/, const Q_UINT32 /*nPixels*/) const { }

        virtual bool convertPixelsTo(const Q_UINT8 * /*src*/,
                                     Q_UINT8 * /*dst*/, KisColorSpace * /*dstColorSpace*/,
                                     Q_UINT32 /*numPixels*/,
                                     Q_INT32 /*renderingIntent = INTENT_PERCEPTUAL*/) { return false; }

        virtual Q_UINT8 getAlpha(const Q_UINT8 * /*pixel*/) const { return 0; }

        virtual void setAlpha(Q_UINT8 * /*pixels*/, Q_UINT8 /*alpha*/, Q_INT32 /*nPixels*/) const { }
        
        virtual void multiplyAlpha(Q_UINT8 * /*pixels*/, Q_UINT8 /*alpha*/, Q_INT32 /*nPixels*/) { }

        virtual void applyAlphaU8Mask(Q_UINT8 * /*pixels*/, Q_UINT8 * /*alpha*/, Q_INT32 /*nPixels*/) { }

        virtual void applyInverseAlphaU8Mask(Q_UINT8 * /*pixels*/, Q_UINT8 * /*alpha*/, Q_INT32 /*nPixels*/) { }

        virtual KisColorAdjustment *createBrightnessContrastAdjustment(Q_UINT16 */*transferValues*/) { return 0; }

        virtual KisColorAdjustment *createDesaturateAdjustment() { return 0; }
        
        virtual KisColorAdjustment *createPerChannelAdjustment(Q_UINT16 **/*transferValues*/) { return 0; }

        virtual void applyAdjustment(const Q_UINT8 */*src*/, Q_UINT8 */*dst*/, KisColorAdjustment *, Q_INT32 /*nPixels*/) { }

        virtual void invertColor(Q_UINT8 * /*src*/, Q_INT32 /*nPixels*/) { }

        virtual Q_UINT8 difference(const Q_UINT8* /*src1*/, const Q_UINT8* /*src2*/) { return 255; }

        virtual void mixColors(const Q_UINT8 **colors, const Q_UINT8 *weights, Q_UINT32 nColors, Q_UINT8 *dst) const
        {
            const _type** colorsT = reinterpret_cast<const _type**>( colors );
            _type* dstT = reinterpret_cast<_type*>( dst );
            for(uint j = 0; j < _nbchannels; j++)
            {    dstT[j] = 0.0; }
            for(uint i = 0; i <nColors; i++)
            {
                for(int j = 0; j < _nbchannels; j++)
                {
                    dstT[j] += colorsT[i][j] * weights[i];
                }
            }
            for(int j = 0; j < _nbchannels; j++)
            {    dstT[j] /= 255; }
        }

        virtual void convolveColors(Q_UINT8** colors, Q_INT32* kernelValues, KisChannelInfo::enumChannelFlags /*channelFlags*/, Q_UINT8 *dst, Q_INT32 factor, Q_INT32 offset, Q_INT32 nColors) const
        {
            _type totals[ _nbchannels ];
            for(uint i = 0; i < _nbchannels; i++)
            {
                totals[ i ] = 0;
            }

            _type** colorsT = reinterpret_cast<_type**>( colors );
            _type* dstT = reinterpret_cast<_type*>( dst );
            
            while (nColors--)
            {
                Q_INT32 weight = *kernelValues;

                if (weight != 0) {
                    for(uint i = 0; i < _nbchannels; i++)
                    {
                        totals[ i ] += (*colorsT)[ i ] * weight ;
                    }
                }
                colorsT++;
                kernelValues++;
            }
            for(uint i = 0; i < _nbchannels; i++)
            {
                dstT[ i ] = totals[ i ] / factor + offset ;
            }
        }

        virtual void darken(const Q_UINT8 * /*src*/, Q_UINT8 * /*dst*/, Q_INT32 /*shade*/, bool /*compensate*/, double /*compensation*/, Q_INT32 /*nPixels*/) const { }

        virtual Q_UINT8 intensity8(const Q_UINT8 * /*src*/) const { return 0; }
    
        virtual KisID mathToolboxID() const { return KisID("",""); }
    
        virtual void bitBlt(Q_UINT8 */*dst*/,
                            Q_INT32 /*dststride*/,
                            KisColorSpace * /*srcSpace*/,
                            const Q_UINT8 */*src*/,
                            Q_INT32 /*srcRowStride*/,
                            const Q_UINT8 */*srcAlphaMask*/,
                            Q_INT32 /*maskRowStride*/,
                            Q_UINT8 /*opacity*/,
                            Q_INT32 /*rows*/,
                            Q_INT32 /*cols*/,
                            const KisCompositeOp& /*op*/) { }

        virtual QValueList<KisFilter*> createBackgroundFilters()
        { return QValueList<KisFilter*>(); };
};

#endif // KIS_COLORSPACE_H_
