/*
 * This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "PerlinNoise.h"

#include <stdlib.h>
#include <vector>

#include <klocale.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kgenericfactory.h>

#include <kis_colorspace_factory_registry.h>
#include <kis_meta_registry.h>
#include <kis_multi_double_filter_widget.h>
#include <kis_iterators_pixel.h>
#include <kis_progress_display_interface.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_transaction.h>
#include <kis_types.h>
#include <kis_selection.h>

#include <kis_convolution_painter.h>

#include <qimage.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>

#include "PerlinNoiseConfigurationWidget.h"
#include "PerlinNoiseConfigurationBaseWidget.h"

typedef KGenericFactory<KritaPerlinNoise> KritaPerlinNoiseFactory;
K_EXPORT_COMPONENT_FACTORY( kritaPerlinNoise, KritaPerlinNoiseFactory( "krita" ) )

KritaPerlinNoise::KritaPerlinNoise(QObject *parent, const char *name, const QStringList &)
: KParts::Plugin(parent, name)
{
    setInstance(KritaPerlinNoiseFactory::instance());

    kdDebug(41006) << "PerlinNoise filter plugin. Class: "
    << className()
    << ", Parent: "
    << parent -> className()
    << "\n";

    if(parent->inherits("KisFilterRegistry"))
    {
        KisFilterRegistry * manager = dynamic_cast<KisFilterRegistry *>(parent);
        manager->add(new KisPerlinNoiseFilter());
    }
}

KritaPerlinNoise::~KritaPerlinNoise()
{
}

KisPerlinNoiseFilter::KisPerlinNoiseFilter() 
    : KisFilter(id(), "PerlinNoise", i18n("&Perlin noise..."))
{
}

std::list<KisFilterConfiguration*> KisPerlinNoiseFilter::listOfExamplesConfiguration(KisPaintDeviceSP )
{
    std::list<KisFilterConfiguration*> list;
    list.insert(list.begin(), configuration());
    return list;
}

KisFilterConfiguration* KisPerlinNoiseFilter::configuration()
{
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
    config->setProperty("threshold", 0.5);
    return config;
};

KisFilterConfigWidget * KisPerlinNoiseFilter::createConfigurationWidget(QWidget* parent, KisPaintDeviceSP dev)
{
    PerlinNoiseConfigurationWidget* w = new PerlinNoiseConfigurationWidget(parent, "");
    Q_CHECK_PTR(w);
    return w;
}

KisFilterConfiguration* KisPerlinNoiseFilter::configuration(QWidget* nwidget)
{
    PerlinNoiseConfigurationWidget* widget = (PerlinNoiseConfigurationWidget*) nwidget;
    if( widget == 0 )
    {
        return configuration();
    } else {
        KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
        config->setProperty("amount", widget->widget()->amount->value( ));
        config->setProperty("offset", widget->widget()->offset->value( ));
        config->setProperty("factor", widget->widget()->factor->value( ));
        config->setProperty("horizontalScale", widget->widget()->horizontalScale->value( ));
        config->setProperty("verticalScale", widget->widget()->verticalScale->value( ));
        config->setProperty("octaves", widget->widget()->octaves->value( ));
        return config;
    }
}
double fade(double t)
{
    return ((2.0*fabs(t)-3.0)*(t)*(t)+1.0);
}

void KisPerlinNoiseFilter::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, 
                                   KisFilterConfiguration* config, const QRect& rect ) 
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);

    double amount = 0.5;
    double offset = 0.0;
    double factor = 1.0;
    double xscale = 4.0;
    double yscale = 4.0;
    int octaves = 2;
    if(config)
    {
        amount = config->getDouble("amount", 0.5);
        offset = config->getDouble("offset", 0.0);
        factor = config->getDouble("factor", 1.0);
        xscale = config->getDouble("horizontalScale", 4.0);
        yscale = config->getDouble("verticalScale", 4.0);
        octaves = config->getInt("octaves", 1);
    }
    
    double grad_x[64];
    double grad_y[64];
    int permutation[64];
    // Init permutations
    for(int i = 0; i < 64; i++)
    {
        permutation[i] = i;
    }
    for (int i = 0; i < 0xFFF; i++) {
        int j = rand () % 64;
        int k = rand () % 64;
        int t = permutation[j];
        permutation[j] = permutation[k];
        permutation[k] = t;
    }
    // Initialize the gradient table
    for(int i = 0; i < 64; i++)
    {
        grad_x[i] = rand() / (double)RAND_MAX - 0.5;
        grad_y[i] = rand() / (double)RAND_MAX - 0.5;
        double norm = 1.0 / sqrt(grad_x[i] * grad_x[i] + grad_y[i] * grad_y[i]);
        grad_x[i] *= norm;
        grad_y[i] *= norm;
    }
    
    
    KisRectIteratorPixel dstIt = dst->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), true );
    KisRectIteratorPixel srcIt = src->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), false);

    int pixelsProcessed = 0;
    setProgressTotalSteps(rect.width() * rect.height());

    KisColorSpace * cs = src->colorSpace();
    
    KisColorSpace* grayCS = KisMetaRegistry::instance()->csRegistry()->getColorSpace(KisID("GRAYA"),"");
    Q_UINT8 gray[2];
    gray[1] = 0xFF;
    
    Q_UINT8* pixel = new Q_UINT8[ grayCS->pixelSize() ];
    const Q_UINT8 *colors[2];
    colors[0] = pixel;
    Q_UINT8 weights[2];
    weights[0] = amount * 0xff;
    weights[1] = (1.0 - amount) * 0xff;
    while( ! srcIt.isDone() )
    {
        if(srcIt.isSelected())
        {
            double total = 0.0;
            int frequency = 1;
            double x = ((dstIt.x() - rect.x()) ) / (double)rect.width();
            double y = ((dstIt.y() - rect.y()) ) / (double)rect.height();
            x *= xscale;
            y *= yscale;
            for(int oct = 0; oct < octaves; oct++)
            {
                double xs = x * frequency;
                double ys = y * frequency;
                int xs_frac = (int) (xs);
                int ys_frac = (int) (ys);
                double intermtotal = 0.0;
                for(int i = 0; i < 2; i++)
                {
                    for(int j = 0; j < 2; j++)
                    {
                        int n = permutation[(xs_frac + i + permutation[(ys_frac + j) % 64]) % 64];
                        double vx = xs - xs_frac - i;
                        double vy = ys - ys_frac - j;
                        intermtotal += fade(vx) * fade(vy) * (grad_x[n] * vx + grad_y[n] * vy );
                    }
                }
                
                total += intermtotal / frequency;
                
                frequency *= 2;
                
            }
            total += offset;
            total *= 0xff * factor;
            if( total > 0xff) total = 0xff;
            if( total < 0) total = 0;
            gray[0] = (int)total;
            
            // Apply on the pixel
            colors[1] = srcIt.oldRawData();
            grayCS->convertPixelsTo(gray, pixel, cs, 1 );
            char alpha = cs->getAlpha(dstIt.rawData());
            cs->mixColors(colors, weights, 2, dstIt.rawData() );
            cs->setAlpha(dstIt.rawData(), alpha, 1);
        }
        setProgress(++pixelsProcessed);
        ++srcIt;
        ++dstIt;
    }

    delete[]pixel;
    setProgressDone(); // Must be called even if you don't really support progression
}
