/*
 * This file is part of the KDE project
 *
 *  Copyright (c) 2007 Cyrille Berger <cberger@cberger.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 */

#include "Binarize.h"

#include <stdlib.h>
#include <vector>

#include <klocale.h>
#include <kiconloader.h>
#include <kinstance.h>
#include <kmessagebox.h>
#include <kstandarddirs.h>
#include <ktempfile.h>
#include <kdebug.h>
#include <kgenericfactory.h>

#include <kis_multi_double_filter_widget.h>
#include <kis_iterators_pixel.h>
#include <kis_progress_display_interface.h>
#include <kis_filter_registry.h>
#include <kis_global.h>
#include <kis_transaction.h>
#include <kis_types.h>
#include <kis_selection.h>

#include <kis_convolution_painter.h>

#include <qimage.h>
#include <qpixmap.h>
#include <qbitmap.h>
#include <qpainter.h>

typedef KGenericFactory<KritaBinarize> KritaBinarizeFactory;
K_EXPORT_COMPONENT_FACTORY( kritaBinarize, KritaBinarizeFactory( "krita" ) )

KritaBinarize::KritaBinarize(QObject *parent, const char *name, const QStringList &)
: KParts::Plugin(parent, name)
{
    setInstance(KritaBinarizeFactory::instance());

    kdDebug(41006) << "Binarize filter plugin. Class: "
    << className()
    << ", Parent: "
    << parent -> className()
    << "\n";

    if(parent->inherits("KisFilterRegistry"))
    {
        KisFilterRegistry * manager = dynamic_cast<KisFilterRegistry *>(parent);
        manager->add(new KisBinarizeFilter());
    }
}

KritaBinarize::~KritaBinarize()
{
}

KisBinarizeFilter::KisBinarizeFilter() 
    : KisFilter(id(), "Binarize", i18n("&Binarize..."))
{
}

std::list<KisFilterConfiguration*> KisBinarizeFilter::listOfExamplesConfiguration(KisPaintDeviceSP )
{
    std::list<KisFilterConfiguration*> list;
    list.insert(list.begin(), configuration());
    return list;
}

KisFilterConfiguration* KisBinarizeFilter::configuration()
{
    KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
    config->setProperty("threshold", 0.5);
    return config;
};

KisFilterConfigWidget * KisBinarizeFilter::createConfigurationWidget(QWidget* parent, KisPaintDeviceSP dev)
{
    Q_UNUSED(dev);
    vKisDoubleWidgetParam param;
    param.push_back( KisDoubleWidgetParam( 0.0, 1.0, 0.5, i18n("Threshold"), "threshold" ) );
    KisFilterConfigWidget * w = new KisMultiDoubleFilterWidget(parent, id().id().ascii(), id().id().ascii(), param );
    Q_CHECK_PTR(w);
    return w;
}

KisFilterConfiguration* KisBinarizeFilter::configuration(QWidget* nwidget)
{
    KisMultiDoubleFilterWidget* widget = (KisMultiDoubleFilterWidget*) nwidget;
    if( widget == 0 )
    {
        return configuration();
    } else {
        KisFilterConfiguration* config = new KisFilterConfiguration(id().id(),1);
        config->setProperty("threshold", widget->valueAt( 0 ));
        return config;
    }
}

template<typename _TYPE, uint vmax>
void binarize(const Q_UINT8* s, Q_UINT8* d, uint nbpixels, double threshold)
{
    const _TYPE* sT = (_TYPE*)(s);
    _TYPE* dT = (_TYPE*)(d);
    for(uint i = 0; i < nbpixels; i ++)
    {
        if(sT[i] < threshold)
        {
            dT[i] = 0;
        } else
        {
            dT[i] = vmax;
        }
//         kdDebug() << (int)sT[i] << " " << (int)dT[i] << " " << threshold << endl;
    }
}

typedef void (*funcBinarize)(const Q_UINT8* , Q_UINT8* , uint, double );

void KisBinarizeFilter::process(KisPaintDeviceSP src, KisPaintDeviceSP dst, 
                                   KisFilterConfiguration* config, const QRect& rect ) 
{
    Q_ASSERT(src != 0);
    Q_ASSERT(dst != 0);

    double threshold;
    if(config)
    {
        threshold = config->getDouble("threshold", 0.5);
    } else {
        threshold = 0.5;
    }

    
    KisRectIteratorPixel dstIt = dst->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), true );
    KisRectIteratorPixel srcIt = src->createRectIterator(rect.x(), rect.y(), rect.width(), rect.height(), false);

    int pixelsProcessed = 0;
    setProgressTotalSteps(rect.width() * rect.height());

    KisColorSpace * cs = src->colorSpace();

    funcBinarize F;
    KisChannelInfo::enumChannelValueType cT = cs->channels()[0]->channelValueType();

    if( cT == KisChannelInfo::UINT8 || cT == KisChannelInfo::INT8 )
    {
        F = & binarize<Q_UINT8, 255>;
        threshold *= 255;
    } else if( cT == KisChannelInfo::UINT16 || cT == KisChannelInfo::INT16 )
    {
        F = & binarize<Q_UINT16, 65535>;
        threshold *= 65535;
    } else if( cT == KisChannelInfo::FLOAT32 )
    {
        F = & binarize<float, 1>;
    } else {
        return;
    }
    Q_INT32 nC = cs->nColorChannels();

    while( ! srcIt.isDone() )
    {
        if(srcIt.isSelected())
        {
            F( srcIt.oldRawData(), dstIt.rawData(), nC, threshold);
        }
        setProgress(++pixelsProcessed);
        ++srcIt;
        ++dstIt;
    }

    
    setProgressDone(); // Must be called even if you don't really support progression
}
