/*
    Copyright (C) 2008  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#ifndef STATIONLISTWIDGET_H
#define STATIONLISTWIDGET_H

#include <QTableView>
#include <KMenu>
#include "stationlistmodel.h"
#include "customizableheaderview.h"

/** \brief A widget which provides a full-featured table view showing the streams.
*
*   This class inherits \e QTableView. It provides a \e QTableView
*   (with nicer default values than the original) and constructs by
*   default a stationlistModel object that is used as model.
*
*   It provides context menus. This context menus must be set up
*   manually. See #globalContextMenu and #streamContextMenu.
*
*   Furthermore this class provides a number of slots, which can also be
*   used to control the streams. You can, for example, connect KActions
*   to this slots.
*
*   So this class handels mainly everthing that has to do with selection
*   of rows (=streams). For all the rest, it relies on stationlistModel.
*
*   \warning You \e have to call saveAllColumnSizes() when the application is about
*   to close. See the documentation of the function for details.
*   \warning \e Never change the model of this view!
*   \warning Make sure that you have only \e one object of this class at the same time. */
class stationlistWidget : public QTableView
{

     Q_OBJECT

  public:
     /** The constructor.
     *
     * @param parent The parent widget for this widget. (We can't use
     *               <tt>QPointer\<QWidget\></tt> instead of
     *               <tt>QWidget *</tt> because this way we couldn't pass
     *               the argument to <tt>QTableView::QTableView()</tt>.)
     * @param mainWindow Pass the mainwindow of the application here.
     *                   To \e this window, all settings dialogs will
     *                   be centered. */
     explicit stationlistWidget(QWidget *parent = 0, QWidget *mainWindow = 0);
     /** The desctructor. */
     virtual ~stationlistWidget();
     /** The context menu that is displayed when the user makes a click
     *   with the right mouse button where no item is. It is empty by default.
     *   You have to add actions to give him a function.
     *   \sa streamContextMenu */
     KMenu globalContextMenu;
     /** The context menu that is displayed when the user makes a click
     *   with the right mouse button where an item is. It is empty by default.
     *   You have to add actions to give him a function.
     *   \sa #globalContextMenu */
     KMenu streamContextMenu;
     /** Saves some settings before the application closes. */
     virtual bool queryClose();
     /** Reads properties for restoring the previous session
     * (using session management).
     * @param m_configGroup The location where to read the properties from
     * \sa #saveProperties() */
     virtual void readProperties(const KConfigGroup & m_configGroup);
     /** Saves properties for restoring this session later
     * (using session management).
     * @param m_configGroup The location where to save the properties
     * \sa #readProperties() */
     virtual void saveProperties(KConfigGroup & m_configGroup);
     /** Returns a pointer to the model of this view widget. Unlike
     *   model(), this function returns a pointer of type
     *   <tt>QPointer\<stationlistModel\></tt>. */
     QPointer<stationlistModel> stationlistmodel();

  signals:
     /** This signal is emitted when the selection changes somehow
     *   and now more than one of the rows are selected.
     *   @param value is always TRUE.
     *   \sa selectionChanged() */
     void multipleSelected(bool value);
     /** This signal is emitted when the selection changes somehow
     *   and now none of the rows is selected.
     *   @param value is always TRUE.
     *   \sa selectionChanged() */
     void noneSelected(bool value);
     /** This signal is emitted when the selection changes somehow
     *   and now exactly one of the rows is selected.
     *   @param value is always TRUE.
     *   \sa selectionChanged() */
     void oneSelected(bool value);

  public slots:
     /** Adds a new radio station. */
     void addNewStation();
     /** Delete the actually selected stream(s). */
     void deleteStation();
     /** Displays the settings dialog for the actually selected stream.
     *
     * Has only an effect if exactly \e one stream is selected. */
     void displayStreamSettings();
     /** Start recording for the actually selected streams. */
     void record();
     /** This slot saves the column sizes of the widget to the config file.
     *   This is necessary to start the next time with the column sizes with
     *   which we have left the application.
     *
     *   You \e have to call this function when the application is about to be
     *   closed. Call it in <tt>KMainWindow::queryClose()</tt>!
     *
     *   <i>We don't do this task in the destructor, because the session manager
     *   sometimes only requests for saving files (and this way calls
     *   <tt>KMainWindow::queryClose()</tt>), and then kills the applications
     *   after a certain time. So it's better to perform this task not in
     *   the destructor, but earlier.</i> */
     void saveAllColumnSizes();
     /** Stop the selected streams. */
     void stopRecord();

  protected:
     /** Reimplemented from class QAbstractScrollArea.
     *
     * Just displays #globalContextMenu or #streamContextMenu at the appropriate place. */
     virtual void contextMenuEvent(QContextMenuEvent * e);
     /** Reimplemented from base class.
     *
     * Handles middle mouse button (insert from "selection" clipboard") and forwards
     * everthing else to the implementation of the base class.
     *
     * \note This function is called by the event system only when the mouse click was
     * done in the display aerea (not in the scroll bars/headers). */
     virtual void mousePressEvent(QMouseEvent *event);

  protected slots:
     /** This reimplemented virtual function
     * <ul>
     * <li> calls the implementation of the base class </li>
     * <li> emits \e always one of the signals multipleSelected(),
     *      oneSelected(), noneSelected() </li>
     * <li> calls stationlistModel::setPlaying()...
     *      <ul>
     *      <li><i>if exactly \e one row is selected:</i> ...with the index
     *          of the selected row</li>
     *      <li><i>else:</i> ...with the value <i>-1</i></li>
     *      </ul>
     * </li>
     * </ul> */
     virtual void selectionChanged(const QItemSelection & selected,
                                   const QItemSelection & deselected);

  private:
     /** A pointer to the used stationlistModel.
     *
     * We could alternitivly use <tt>QTableView::model()</tt> and
     * than typecast to stationlistModel, but with this pointer,
     * we have the advantage that we don't need to typecast. */
     QPointer<stationlistModel> m_stationlistModel;

  private slots:
     /** Convenience function, that saves the column size of the given
     *   column to the config file. */
     void saveColumnSize(const int column);
};

#endif
