/*
    Copyright (C) 2008-2009  Tim Fechtner < urwald at users dot sourceforge dot net >

    This program is free software; you can redistribute it and/or
    modify it under the terms of the GNU General Public License as
    published by the Free Software Foundation; either version 2 of
    the License or (at your option) version 3 or any later version
    accepted by the membership of KDE e.V. (or its successor approved
    by the membership of KDE e.V.), which shall act as a proxy
    defined in Section 14 of version 3 of the license.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "streamdirectorymodel.h"

#include <QMimeData>
#include <QUrl>
#include <KLocale>
#include "streamdirectoryentry.h"

streamDirectoryModel::streamDirectoryModel(QObject *parent, const QString & genreIconFileName)
  : QAbstractItemModel(parent)
{
  // internals
  rootEntry = new streamDirectoryEntry_root();  // very important!
  genreIcon = KIcon(genreIconFileName);
  setSupportedDragActions(Qt::CopyAction);
};

streamDirectoryModel::~streamDirectoryModel()
{
  delete rootEntry;
}

QModelIndex streamDirectoryModel::parent(const QModelIndex & index) const
{
  if (!index.isValid()) {
    return QModelIndex();
  };

  streamDirectoryEntry *childItem = static_cast<streamDirectoryEntry *>(index.internalPointer());
  streamDirectoryEntry *parentItem = childItem->parent();
  if (parentItem == rootEntry) {
    return QModelIndex();
  } else {
    return createIndex(parentItem->row(), 0, parentItem);
  };
}

QModelIndex streamDirectoryModel::index(int row, int column, const QModelIndex &parent) const
{
  // variables
  streamDirectoryEntry *parentItem;
  streamDirectoryEntry *childItem;

  // code
  if (!hasIndex(row, column, parent)) {
    return QModelIndex();
  };

  if (!parent.isValid()) {
    parentItem = rootEntry;
  } else {
    parentItem = static_cast<streamDirectoryEntry *>(parent.internalPointer());
  };
  childItem = parentItem->child(row);
  if (childItem) {
    return createIndex(row, column, childItem);
  } else {
    return QModelIndex();
  };
}

int streamDirectoryModel::rowCount(const QModelIndex &parent) const
{
  // variables
  streamDirectoryEntry *parentItem;

  // code
  if (parent.column() > 0) {
    return 0;
  };

  if (!parent.isValid()) {
    parentItem = rootEntry;
  } else {
    parentItem = static_cast<streamDirectoryEntry *>(parent.internalPointer());
  };
  return parentItem->childCount();
}

int streamDirectoryModel::columnCount(const QModelIndex &) const
{
  return 5;
}

QVariant streamDirectoryModel::data(const QModelIndex &index, int role) const
{
  if (!index.isValid()) {
    return QVariant();
  };

  if (role != Qt::DisplayRole && role != Qt::UserRole) {
    return QVariant();
  };

  streamDirectoryEntry *item = static_cast<streamDirectoryEntry *>(index.internalPointer());
  if (role == Qt::DisplayRole) {
    return item->data(index.column());
  } else {
    return item->rawData(index.column());
  };
}

QVariant streamDirectoryModel::headerData(int section,
                                          Qt::Orientation orientation,
                                          int role) const
{
  if (orientation == Qt::Horizontal && role == Qt::DisplayRole) {
    switch (section) {
      case 0: return i18nc("@title:column", "stream name");
      case 1: return i18nc("@title:column", "data format");
      case 2: return i18nc("@title:column", "bit rate");
      case 3: return i18nc("@title:column", "track information");
      case 4: return i18nc("@title:column", "URL");
    };
  };
  return QAbstractItemModel::headerData(section, orientation, role);
}

bool streamDirectoryModel::hasChildren(const QModelIndex & parent) const
{
  if (parent.isValid()) { // means: is a genre or a stream
    return (static_cast<streamDirectoryEntry *>(parent.internalPointer()))->hasChildren();
  } else {
    return rootEntry->hasChildren();
  };
}

Qt::ItemFlags streamDirectoryModel::flags(const QModelIndex & index) const
{
  // variables
  Qt::ItemFlags returnValue;

  // code
  returnValue = QAbstractItemModel::flags(index);
  if (index.isValid()) { // add drag support only for valid items!
    returnValue = returnValue | Qt::ItemIsDragEnabled;
  };
  return returnValue;
}

QStringList streamDirectoryModel::mimeTypes() const
{
  // variables
  QStringList types;

  // code
  types << "text/uri-list";
  return types;
}

QMimeData *streamDirectoryModel::mimeData(const QModelIndexList & indexes) const
{
  // TODO Wie werden ausgewählte Genres gehandhabt?
  // variables
  QMimeData *returnValue;
  QList<QUrl> list;

  // code
  if (indexes.size() == 0) {
    returnValue = 0;  // because QAbstractItemModel::mimeData() would do the same!
  } else {
    returnValue = new QMimeData();
    foreach (const QModelIndex & index, indexes) {
      // We get an own index for each column. To make sure to export each row only once,
      // we export only for column 0.
      if (index.isValid() && (index.column() == 0)) {
        list.append(
          QUrl(static_cast<streamDirectoryEntry *>(index.internalPointer())->value()));
      };
    };
    returnValue->setUrls(list);
  };
  return returnValue;
}
