/*
   Copyright (C) 2006 by Stefan Taferner <taferner@kde.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/

/* project.h
 */
#ifndef koverartist_project_h
#define koverartist_project_h

#include <qstring.h>
#include <qcolor.h>
#include <qfont.h>
#include <qvaluevector.h>
#include <qstringlist.h>
#include <qobject.h>
#include <qpixmap.h>

#include <kurl.h>

#include "case.h"
#include "disc.h"
#include "effect.h"
#include "image.h"
#include "textstyle.h"


class QTextStream;
class QXmlAttributes;
class QWidget;


namespace KoverArtist
{

typedef QValueVector<Disc> DiscVector;


/** A KoverArtist project.
 *
 */
class Project
{
public:
   Project();
   Project(const Project&);
   Project& operator=(const Project&);

   virtual ~Project();

   /** Load the project from the given url. The window is used for storing
    * authentication information, @see KIO::NetAccess.
    */
   virtual bool load(const KURL& url, QWidget* window=0);

   /** Save the project to the given url. The window is used for storing
    * authentication information, @see KIO::NetAccess.
    */
   virtual bool save(const KURL& url, QWidget* window=0);

   /** Returns the url of the project. */
   const KURL& url() const { return mUrl; }

   /** Guess from the file extension of fname if the file could be loaded. */
   static bool canLoad(const QString& fname);

   /** Clear the object. */
   virtual void clear();

   /** Test if the project was modified. */
   bool isModified() const { return mModified; }

   /** Mark as modified. */
   virtual void modified();

   /** Clear the modified flag. */
   void clearModified() { mModified = false; }

   /** Returns the title. */
   QString title() const { return mTitle; }

   /** Set the title to t. */
   void setTitle(const QString& t) { mTitle = t; }

   /** Returns the sub-title. */
   QString subTitle() const { return mSubTitle; }

   /** Set the sub-title to ts. */
   void setSubTitle(const QString& ts) { mSubTitle = ts; }

   /** Returns the number of discs. */
   int count() const { return mDiscs.count(); }

   /** Set the number of discs to num. */
   virtual void resize(int num);

   /** Returns the idx-th disc. */
   Disc& disc(int idx) { return mDiscs[idx]; }

   /** Returns the idx-th disc. Const version. */
   const Disc& disc(int idx) const { return mDiscs[idx]; }

   /** Returns the list of discs. */
   DiscVector& discs() { return mDiscs; }
   const DiscVector& discs() const { return mDiscs; }

   /** Returns an iterator that points to the idx-th disc. */
   DiscVector::iterator discIt(int idx);

   /** The text style of the title. */
   TextStyle& titleStyle() { return mStyleTitle; }
   const TextStyle& titleStyle() const { return mStyleTitle; }

   /** The text style of the sub-title. */
   TextStyle& subTitleStyle() { return mStyleSubTitle; }
   const TextStyle& subTitleStyle() const { return mStyleSubTitle; }

   /** The text style of the contents. */
   TextStyle& contentsStyle() { return mStyleContents; }
   const TextStyle& contentsStyle() const { return mStyleContents; }

   /** The text style of the side text. */
   TextStyle& sideStyle() { return mStyleSide; }
   const TextStyle& sideStyle() const { return mStyleSide; }

   /** The effect for title and sub-title. */
   Effect& titleEffect() { return mTitleEffect; }
   const Effect& titleEffect() const { return mTitleEffect; }

   /** The effect for the contents. */
   Effect& contentsEffect() { return mContentsEffect; }
   const Effect& contentsEffect() const { return mContentsEffect; }

   /** The effect for the side text. */
   Effect& sideEffect() { return mSideEffect; }
   const Effect& sideEffect() const { return mSideEffect; }

   /** The cd/dvd case that is used. */
   const Case& box() const { return mCase; }
   Case& box() { return mCase; }
   void setBox(const Case& c) { mCase = c; }

   /** Get/set the background color. */
   const QColor& backgroundColor() const { return mBackgroundColor; }
   void setBackgroundColor(const QColor& c) { mBackgroundColor = c; }

   /** Get/set the case outline color. */
   const QColor& outlineColor() const { return mOutlineColor; }
   void setOutlineColor(const QColor& c) { mOutlineColor = c; }

   /** Get/set the disc title font. */
   const QFont& discTitleFont() const { return mDiscTitleFont; }
   void setDiscTitleFont(const QFont& f) { mDiscTitleFont = f; }

   /** Returns the front image object. */
   Image& imgFront() { return mImgFront; }
   const Image& imgFront() const { return mImgFront; }

   /** Returns the back image object. */
   Image& imgBack() { return mImgBack; }
   const Image& imgBack() const { return mImgBack; }

   /** Get/set if front image wraps around the whole case. Useful e.g. for
    * DVD boxes.
    */
   bool imgFrontWrapAround() const { return mImgFrontWrapAround; }
   void setImgFrontWrapAround(bool f) { mImgFrontWrapAround = f; }

   /** Get/set if front title is hidden. */
   bool frontTitleHide() const { return mFrontTitleHide; }
   void setFrontTitleHide(bool f) { mFrontTitleHide = f; }

   /** Returns extension of file fname, e.g. "koap" for file "test.koap" */
   static QString fileExt(const QString& fname);

protected:
   bool makeImageLocal(Image& img, const QString& name,
                       const QString& projectFileName, QWidget *aWindow);

   virtual bool saveFile(const QString& fname, QWidget *aWindow);
   virtual bool saveFileKoap(const QString& fname, QWidget *aWindow);

   virtual bool loadFile(const QString& fname);
   virtual bool loadFileXml(const QString& fname);
   virtual bool loadFileKmf(const QString& fname, QDomDocument& doc);
   virtual bool loadFileKiap(const QString& fname, QDomDocument& doc);
   virtual bool loadFileKoap(const QString& fname, QDomDocument& doc);
   virtual bool loadFileKoapDiscs(QDomDocument& doc, QDomElement& discs);
   virtual bool loadFileKoapDisc(QDomDocument& doc, QDomElement& discs);

protected:
   QString mTitle, mSubTitle;
   Image mImgFront, mImgBack;
   bool mImgFrontWrapAround, mFrontTitleHide;
   TextStyle mStyleTitle, mStyleSubTitle, mStyleContents, mStyleSide;
   QFont mDiscTitleFont;
   QColor mOutlineColor, mBackgroundColor;
   Effect mTitleEffect, mContentsEffect, mSideEffect;
   Case mCase;
   DiscVector mDiscs;
   KURL mUrl;
   bool mModified;
};


} //namespace

#endif /*koverartist_project_h*/
