/***************************************************************************
 *   Copyright (C) 2005-2008 by Georg Hennig                               *
 *   Email: georg.hennig@web.de                                            *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/

#include <qbuffer.h>
#include <qbuttongroup.h>
#include <qcheckbox.h>
#include <qclipboard.h>
#include <qdir.h>
#include <qeventloop.h>
#include <qheader.h>
#include <qlabel.h>
#include <qobject.h>
#include <qradiobutton.h>
#include <qstyle.h>
#include <qtooltip.h>
#include <qwhatsthis.h>

#include <kaboutapplication.h>
#include <kaction.h>
#include <kcombobox.h>
#include <kdebug.h>
#include <khelpmenu.h>
#include <kiconloader.h>
#include <kio/netaccess.h>
#include <kled.h>
#include <klineedit.h>
#include <klistview.h>
#include <kmessagebox.h>
#include <knuminput.h>
#include <kpassivepopup.h>
#include <kpopupmenu.h>
#include <kprogress.h>
#include <kpropertiesdialog.h>
#include <kpushbutton.h>
#include <krun.h>
#include <ksystemtray.h>
#include <ktabbar.h>
#include <ktabwidget.h>
#include <ktempfile.h>
#include <kurldrag.h>
#include <kurlrequester.h>

#include "kdatecombo.h"
#include "kquery.h"
#include "komparatorfilefilter.h"
#include "komparedialog.h"
#include "komparatorcomparejob.h"
#include "komparatoremptydirectoriesjob.h"
#include "komparatorcopyjob.h"
#include "komparatormovejob.h"
#include "komparatorremovejob.h"
#include "komparatorurllist.h"
#include "kfileitemext.h"
#include "klistviewitemsingle.h"
#include "klistviewitemdups.h"

#include "komparatorwidget.h"

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

KomparatorWidget::KomparatorWidget( const KURL &url1, const KURL &url2, QWidget* parent, const char* name, WFlags fl )
	: KomparatorWidgetBase( parent, name, fl )
{
	// FIXME: this shouldn't be necessary, if qtdesigner + KTabWidget worked as expected. To be removed, if fixed in KDE.
	main_TabWidget->insertTab( duplicateFiles_tab, i18n( "Duplicate files" ) );
	main_TabWidget->insertTab( missingFiles_tab, i18n( "Missing files" ) );
	main_TabWidget->insertTab( newerFiles_tab, i18n( "Newer files" ) );

	m_query = new KQuery( this );
	compare_job = new KomparatorCompareJob( this );

	m_empty_directory_job = new KomparatorEmptyDirectoryJob( this );
	m_copy_job = new KomparatorCopyJob( this );
	m_move_job = new KomparatorMoveJob( this );
	m_remove_job = new KomparatorRemoveJob( this );

	m_files = NULL;
	m_dirs_dupes = NULL;

	m_popup_menu = NULL;

	m_systray = NULL;

	m_requested_download = NULL;
	m_requested_download_job = NULL;

	current_list_view = NULL;

	m_status = CLEAN;
	m_file_remove_action = IDLE;

	m_properties_renamed = false;

	m_unreadable_files.clear();

	m_url1 = KURL( "" );
	m_url2 = KURL( "" );
	m_virtual_subdirectory_current = KURL( "" );

	m_red = QColor( 200, 30, 30 );   // #e61e1e
	m_green = QColor( 5, 155, 10 );  // #059b0a
	m_blue = QColor( 15, 25, 200 );  // #0f19c8
	m_purple = QColor( 145, 40, 190 ); // #9128be

	KHelpMenu *help_menu = new KHelpMenu( this, "" );
	connect( help_KPushButton, SIGNAL( clicked() ), help_menu, SLOT( appHelpActivated() ) );

	connect( m_query, SIGNAL( addFile( const KFileItem*, const QString& ) ),
					SLOT( slotAddFile( const KFileItem*, const QString& ) ) );

	connect( m_query, SIGNAL( result( int, QString ) ), SLOT( slotResult( int, QString ) ) );

	connect( duplicates1_KListView, SIGNAL( selectionChanged() ), SLOT( slotDuplicateSelected() ) );

	connect( this, SIGNAL( signalProgress( QString, int, QString ) ), SLOT( slotProgress( QString, int, QString ) ) );

	connect( this, SIGNAL( signalSaveListViewColumnWidth() ), SLOT( slotSaveListViewColumnWidth() ) );

	slotCount( DUPLICATES, 1, -1 );
	slotCount( DUPLICATES, 2, -1 );
	slotCount( MISSING, 1, -1 );
	slotCount( MISSING, 2, -1 );
	slotCount( NEWER, 1, -1 );
	slotCount( NEWER, 2, -1 );
	connect( this, SIGNAL( signalCount( LIST_VIEW, int, int ) ), SLOT( slotCount( LIST_VIEW, int, int ) ) );

	duplicates1_KomparatorFileFilter->setFilterType( DUPLICATES_TYPE );
	duplicates2_KomparatorFileFilter->setFilterType( MISSING_TYPE ); // This is really "MISSING_TYPE", as it's a list view
	missing1_KomparatorFileFilter->setFilterType( MISSING_TYPE ); // with KListViewItemSingle with no special feats.
	missing2_KomparatorFileFilter->setFilterType( MISSING_TYPE );
	newer1_KomparatorFileFilter->setFilterType( NEWER_TYPE );
	newer2_KomparatorFileFilter->setFilterType( NEWER_TYPE );
	
	connect( duplicates1_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*duplicates_normal*/, const bool &/*duplicates_multiple*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*duplicates_normal*/, const bool &/*duplicates_multiple*/ ) ) );
	connect( duplicates2_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ) );	
	connect( missing1_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ) );
	connect( missing2_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/ ) ) );
	connect( newer1_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ) );
	connect( newer2_KomparatorFileFilter, SIGNAL( signalChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ),
		this, SLOT( slotFilterChanged( KomparatorFileFilter *, const QString &/*name*/,
			const QString &/*path*/, const QString &/*owner*/, const QString &/*group*/, const unsigned long &/*min_size*/,
			const unsigned long &/*max_size*/, const bool &/*consider date*/, const QDate &/*date*/,
			const bool &/*newer_real*/, const bool &/*newer_equal*/, const bool &/*newer_same_time*/ ) ) );

	duplicatesTrash_KPushButton->setIconSet( DesktopIcon( "edittrash" ) );
	duplicatesTrash_KPushButton->setText( "" );

	duplicatesDelete_KPushButton->setIconSet( DesktopIcon( "editdelete" )  );
	duplicatesDelete_KPushButton->setText( "" );

	search_KPushButton->setIconSet( SmallIcon( "komparator" ) );

	about_KPushButton->setIconSet( SmallIcon( "komparator" ) );
	help_KPushButton->setIconSet( SmallIcon( "help" ) );
	whatsThis_KPushButton->setIconSet( SmallIcon( "contexthelp" ) );

	addPreset_KPushButton->setIconSet( SmallIcon( "add" ) );
	removePreset_KPushButton->setIconSet( SmallIcon( "remove" ) );
	updatePreset_KPushButton->setIconSet( SmallIcon( "filesave" ) );

	QSize small_size = QIconSet::iconSize( QIconSet::Small ); // save original icon set size, because we'll change them.

	QImage transparent = transparentImage();

	copyMissing1To2_KPushButton->setIconSet( broadIconSet( SmallIcon( "editcopy" ), transparent, SmallIcon( "2rightarrow" ) ) );
	copyMissing1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyMissing1To2_KPushButton->setText( "" );
	copyMissing2To1_KPushButton->setIconSet( broadIconSet( SmallIcon( "2leftarrow" ), transparent, SmallIcon( "editcopy" ) ) );
	copyMissing2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyMissing2To1_KPushButton->setText( "" );
	moveMissing1To2_KPushButton->setIconSet( broadIconSet( SmallIcon( "goto" ), transparent, SmallIcon( "2rightarrow" ) ) );
	moveMissing1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveMissing1To2_KPushButton->setText( "" );
	moveMissing2To1_KPushButton->setIconSet( broadIconSet( SmallIcon( "2leftarrow" ), transparent, SmallIcon( "goto" ) ) );
	moveMissing2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveMissing2To1_KPushButton->setText( "" );
	removeMissing1_KPushButton->setIconSet( broadIconSet( SmallIcon( "1rightarrow" ), transparent, SmallIcon( "clear_left" ) ) );
	removeMissing1_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeMissing1_KPushButton->setText( "" );
	removeMissing2_KPushButton->setIconSet( broadIconSet( SmallIcon( "clear_left" ), transparent, SmallIcon( "1leftarrow" ) ) );
	removeMissing2_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeMissing2_KPushButton->setText( "" );

	copyNewer1To2_KPushButton->setIconSet( broadIconSet( SmallIcon( "editcopy" ), transparent, SmallIcon( "2rightarrow" ) ) );
	copyNewer1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyNewer1To2_KPushButton->setText( "" );
	copyNewer2To1_KPushButton->setIconSet( broadIconSet( SmallIcon( "2leftarrow" ), transparent, SmallIcon( "editcopy" ) ) );
	copyNewer2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	copyNewer2To1_KPushButton->setText( "" );
	moveNewer1To2_KPushButton->setIconSet( broadIconSet( SmallIcon( "goto" ), transparent, SmallIcon( "2rightarrow" ) ) );
	moveNewer1To2_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveNewer1To2_KPushButton->setText( "" );
	moveNewer2To1_KPushButton->setIconSet( broadIconSet( SmallIcon( "2leftarrow" ), transparent, SmallIcon( "goto" ) ) );
	moveNewer2To1_KPushButton->setMaximumWidth( small_size.width()*3 );
	moveNewer2To1_KPushButton->setText( "" );
	removeNewer1_KPushButton->setIconSet( broadIconSet( SmallIcon( "1rightarrow" ), transparent, SmallIcon( "clear_left" ) ) );
	removeNewer1_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeNewer1_KPushButton->setText( "" );
	removeNewer2_KPushButton->setIconSet( broadIconSet( SmallIcon( "clear_left" ), transparent, SmallIcon( "1leftarrow" ) ) );
	removeNewer2_KPushButton->setMaximumWidth( small_size.width()*3 );
	removeNewer2_KPushButton->setText( "" );

	QIconSet::setIconSize( QIconSet::Small, small_size ); // reset icon set to original size.

	duplicatesReference1_RadioButton->setPaletteForegroundColor( m_green );
	duplicatesReference2_RadioButton->setPaletteForegroundColor( m_blue );

	duplicates1_KListView->setColumnWidthMode( 4, KListView::Manual ); // Don't allow user to resize this column.
	duplicates1_KListView->setColumnWidthMode( 5, KListView::Manual );
	duplicates1_KListView->setColumnWidthMode( 6, KListView::Manual );
	duplicates1_KListView->setColumnAlignment( 2, Qt::AlignRight );

	duplicates2_KListView->setColumnWidthMode( 4, KListView::Manual );
	duplicates2_KListView->setColumnWidthMode( 5, KListView::Manual );
	duplicates2_KListView->setColumnWidthMode( 6, KListView::Manual );
	duplicates2_KListView->setColumnWidthMode( 7, KListView::Manual );
	duplicates2_KListView->setColumnAlignment( 2, Qt::AlignRight );

	missing1_KListView->setColumnWidthMode( 4, KListView::Manual );
	missing1_KListView->setColumnWidthMode( 5, KListView::Manual );
	missing1_KListView->setColumnWidthMode( 6, KListView::Manual );
	missing1_KListView->setColumnAlignment( 2, Qt::AlignRight );

	missing2_KListView->setColumnWidthMode( 4, KListView::Manual );
	missing2_KListView->setColumnWidthMode( 5, KListView::Manual );
	missing2_KListView->setColumnWidthMode( 6, KListView::Manual );
	missing2_KListView->setColumnAlignment( 2, Qt::AlignRight );

	newer1_KListView->setColumnWidthMode( 4, KListView::Manual );
	newer1_KListView->setColumnWidthMode( 5, KListView::Manual );
	newer1_KListView->setColumnWidthMode( 6, KListView::Manual );
	newer1_KListView->setColumnAlignment( 2, Qt::AlignRight );

	newer2_KListView->setColumnWidthMode( 4, KListView::Manual );
	newer2_KListView->setColumnWidthMode( 5, KListView::Manual );
	newer2_KListView->setColumnWidthMode( 6, KListView::Manual );
	newer2_KListView->setColumnAlignment( 2, Qt::AlignRight );

	int i;
	for( i=0; i<main_TabWidget->count(); i++ )
	{
		main_TabWidget->setTabIconSet( main_TabWidget->page( i ),
			KGlobal::iconLoader()->loadIconSet( "ok", KIcon::Toolbar, KIcon::SizeSmall ) );
	}

	// fill preset_KComboBox.
	m_preset_list.setAutoDelete( true );
	KomparatorPreset *new_preset = new KomparatorPreset( this );
	m_preset_list.append( new_preset );
	preset_KComboBox->insertItem( new_preset->title() );
	for ( i=1; i<50; i++ )
	{
		new_preset = new KomparatorPreset( this );
		if ( !new_preset->readConfig( i ) )
		{
			delete new_preset;
			break;
		}
		else
		{
			m_preset_list.append( new_preset );
			preset_KComboBox->insertItem( new_preset->title() );
		}
	}

	m_config_global = new KConfig( "kdeglobals" );
	m_config_global->setGroup( "Notification Messages" );

	// Should be true in the case that the key doesn't exist and the case, that it exists and is true
	// So the only case it is false is: key exists and is false
	bool warning_trash = !m_config_global->hasKey( ":komparator_message_trash") |
		m_config_global->readBoolEntry( ":komparator_message_trash" );
	bool warning_delete = !m_config_global->hasKey( ":komparator_message_delete") |
		m_config_global->readBoolEntry( ":komparator_message_delete" );
	bool warning_overwrite_copy = !m_config_global->hasKey( ":komparator_message_overwrite_copy") |
		m_config_global->readBoolEntry( ":komparator_message_overwrite_copy" );
	bool warning_overwrite_move = !m_config_global->hasKey( ":komparator_message_overwrite_move") |
		m_config_global->readBoolEntry( ":komparator_message_overwrite_move" );
	bool warning_reload = !m_config_global->hasKey( ":komparator_message_reload") |
		m_config_global->readBoolEntry( ":komparator_message_reload" );
	bool warning_remove_preset = !m_config_global->hasKey( ":komparator_message_remove_preset") |
		m_config_global->readBoolEntry( ":komparator_message_remove_preset" );

	warningTrash_CheckBox->setChecked( warning_trash );
	warningDelete_CheckBox->setChecked( warning_delete );
	warningOverwriteCopy_CheckBox->setChecked( warning_overwrite_copy );
	warningOverwriteMove_CheckBox->setChecked( warning_overwrite_move );
	warningReload_CheckBox->setChecked( warning_reload );
	warningRemovePreset_CheckBox->setChecked( warning_remove_preset );

	connect( warningTrash_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningTrash() ) );
	connect( warningDelete_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningDelete() ) );
	connect( warningOverwriteCopy_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningOverwriteCopy() ) );
	connect( warningOverwriteMove_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningOverwriteMove() ) );
	connect( warningReload_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningReload() ) );
	connect( warningRemovePreset_CheckBox, SIGNAL( clicked() ), this, SLOT( slotWarningRemovePreset() ) );

	KConfig *conf = KGlobal::config();
	conf->setGroup( "Global" );

	bool systray = conf->hasKey( "systray" ) &
		conf->readBoolEntry( "systray" );
	bool notification_ready = !conf->hasKey( "notification_ready") |
		conf->readBoolEntry( "notification_ready" );
	bool text_color = conf->hasKey( "text_color" ) &
		conf->readBoolEntry( "text_color" );

	systray_CheckBox->setChecked( systray );
	notification_CheckBox->setChecked( notification_ready );
	textColor_CheckBox->setChecked( text_color );

	QSize total_size = conf->readSizeEntry( "total_size" );
	parent->resize( total_size );

	preset_KComboBox->setCurrentItem( conf->readNumEntry( "preset", 0 ) );
	slotPresetSelected( conf->readNumEntry( "preset", 0 ) ); // select previously used preset; default if not available.

	if ( !url1.isEmpty() ) url1_KURLRequester->setKURL( url1 ); // override preset settings with command line arguments, if there are any.
	if ( !url2.isEmpty() ) url2_KURLRequester->setKURL( url2 );

	url1_KURLRequester->setFocus();

	url1_KURLRequester->setMode( KFile::File | KFile::Directory );
	url1_KURLRequester->setFilter( "application/x-tgz application/x-tbz application/x-tar application/x-zip" );

	url2_KURLRequester->setMode( KFile::File | KFile::Directory );
	url2_KURLRequester->setFilter( "application/x-tgz application/x-tbz application/x-tar application/x-zip" );

	// Still "Global", read sizes of splitter
	QValueList<int> list;
	for ( i=0; i<50; i++ )
	{
		if ( !conf->hasKey( QString( "splitter%1" ).arg( i ) ) ) break;
		list.push_back( conf->readNumEntry( QString( "splitter%1" ).arg( i ) ) );
	}
	duplicates_Splitter->setSizes( list );

	m_systray = new KSystemTray( parent, "m_systray" );
	QToolTip::add( m_systray, kapp->aboutData()->shortDescription() );
	m_systray->setPixmap( m_systray->loadIcon( "komparator" ) );
	m_systray->setAutoResize( true );
	m_systray_movie = new QMovie(
		KGlobal::iconLoader()->loadMovie( QString::fromLatin1( "komparator_working" ), KIcon::MainToolbar/*KIcon::Panel*/ ) );
	connect( systray_CheckBox, SIGNAL( clicked() ), this, SLOT( slotSystray() ) );
	if ( systray_CheckBox->isChecked() )
	{
		m_systray->show();
	}
	connect( this, SIGNAL( signalStartSystrayMovie() ), this, SLOT( slotStartSystrayMovie() ) );
	connect( this, SIGNAL( signalStopSystrayMovie() ), this, SLOT( slotStopSystrayMovie() ) );

	connect( textColor_CheckBox, SIGNAL( clicked() ), this, SLOT( slotTextColor() ) );

	KAction *quit = m_systray->actionCollection()->action( "file_quit" );
	quit->disconnect();
	connect( quit, SIGNAL( activated() ), parent, SLOT( slotQuit() ) );

	addWhatsThisHelp();

	m_progress_timer = new QTimer( this );
	connect( m_progress_timer, SIGNAL( timeout() ), this, SLOT( slotProgressUndefined() ) );
	emit( signalProgress( i18n( "Ready." ), -1, "" ) );

	emit( signalSaveListViewColumnWidth() );
}

void KomparatorWidget::addWhatsThisHelp()
{
	const QString regexContainsWhatsThis // taken from kfind
		= i18n( "<qt>Enter the filename you are looking for. <br>Alternatives may be separated by a semicolon \";\".<br><br>"
			"The filename may contain the following special characters:"
			"<ul>"
			"<li><b>?</b> matches any single character</li>"
			"<li><b>*</b> matches zero or more of any characters</li>"
			"<li><b>[...]</b> matches any of the characters in braces</li>"
			"</ul>"
			"<br>"
			"Example searches:"
			"<ul>"
			"<li><b>*.kwd;*.txt</b> finds all files ending with .kwd or .txt</li>"
			"<li><b>go[dt]</b> finds god and got</li>"
			"<li><b>Hel?o</b> finds all files that start with \"Hel\" and end with \"o\", "
			"having one character in between</li>"
			"<li><b>My Document.kwd</b> finds a file of exactly that name</li>"
			"</ul></qt>" );
	QWhatsThis::add( regex1_TextLabel, regexContainsWhatsThis );
	QWhatsThis::add( regexContains_KLineEdit, regexContainsWhatsThis );
	
	const QString regexContainsNotWhatsThis // taken from kfind
		= i18n( "<qt>Enter the filename you want to exclude from your search. <br>Alternatives may be separated by a semicolon \";\".<br><br>"
			"The filename may not contain the following special characters:"
			"<ul>"
			"<li><b>?</b> matches any single character</li>"
			"<li><b>*</b> matches zero or more of any characters</li>"
			"<li><b>[...]</b> matches any of the characters in braces</li>"
			"</ul>"
			"<br>"
			"Example searches:"
			"<ul>"
			"<li><b>*.kwd;*.txt</b> excludes all files ending with .kwd or .txt</li>"
			"<li><b>go[dt]</b> excludes god and got</li>"
			"<li><b>Hel?o</b> excludes all files that start with \"Hel\" and end with \"o\", "
			"having one character in between</li>"
			"<li><b>My Document.kwd</b> excludes all files with exactly that name</li>"
			"</ul></qt>" );
	QWhatsThis::add( regex2_TextLabel, regexContainsNotWhatsThis );
	QWhatsThis::add( regexContainsNot_KLineEdit, regexContainsNotWhatsThis );

	const QString findSizeWhatsThis
		= i18n( "<qt>Restrict the search result by file size. Only files matching these settings are found "
			"(and considered in duplicates, missing and newer search).<br>"
			"<ul>"
			"<li><i>(none)</i>: No restriction, all files are found. (default)</li>"
			"<li><i>At Least</i>: Files equal to or bigger than the specified size are found.</li>"
			"<li><i>At Most</i>:Files equal to or smaller than the specified size are found.</li>"
			"<li><i>Equal To</i>: Files equal to the specified size are found.</li>"
			"</ul>"
			"The file size selected in the combo box (bytes, KB, MB) are not rounded off.<br>"
			"Therefore a selected size of <i>42 KB</i> will only match files, "
			"that have exactly the size <i>42*1024 = 43008 bytes</i>."
			"</qt>" );
	QWhatsThis::add( size_TextLabel, findSizeWhatsThis );
	QWhatsThis::add( size_KComboBox, findSizeWhatsThis );
	QWhatsThis::add( bytes_KIntNumInput, findSizeWhatsThis );
	QWhatsThis::add( bytes_KComboBox, findSizeWhatsThis );

	const QString modificationTimeWhatsThis
		= i18n( "<qt>Restrict the search result by creation or modification time. On remote / not *nix "
			"file systems, this might not be displayed correctly, as copying can change the modification "
			"time on these systems.<br>"
			"<ul>"
			"<li><i>any</i>: No restriction. (default)</li>"
			"<li><i>between date x and date y</i>: Files that were created or modified between "
			"date x and date y are found (including date x and y).</li>"
			"<li><i>during the last z ...</i>: Files that were created or modified during the last "
			"z minutes, hours, days, months or years are found, including the current minute, hour, day, month, year. "
			"This means, if you are searching on 10th of July for files created or modified "
			"during the last 5 days, files from 6th of July are found but not from 5th.</li>"
			"</ul>"
			"</qt>" );
	QWhatsThis::add( modified_ButtonGroup, modificationTimeWhatsThis );
	QWhatsThis::add( modifiedNoRestriction_RadioButton, modificationTimeWhatsThis );
	QWhatsThis::add( modifiedBetween_RadioButton, modificationTimeWhatsThis );
	QWhatsThis::add( modifiedFrom_KDateCombo, modificationTimeWhatsThis );
	QWhatsThis::add( and_TextLabel, modificationTimeWhatsThis );
	QWhatsThis::add( modifiedTo_KDateCombo, modificationTimeWhatsThis );
	QWhatsThis::add( modifiedLastTime_RadioButton, modificationTimeWhatsThis );
	QWhatsThis::add( modifiedLastTime_KIntNumInput, modificationTimeWhatsThis );
	QWhatsThis::add( modifiedLastTime_KComboBox, modificationTimeWhatsThis );

	const QString presetWhatsThis
		= i18n( "<qt>Presets can simplify your life, if you perform a specific "
			"kind of search regularly. Choosing a preset will set the gui to the preset's settings. "
			"If the preset has empty urls, the current urls aren't changed, if it is selected.<br>"
			"You can"
			"<br><br><ul>"
			"<img src=\"add\">&nbsp;<i>Add a preset</i>: All current settings will be saved as they are. You can choose "
			"a name for the preset (that will be displayed in the combo box), and the urls. "
			"If urls are given, choosing the preset later will set the url fields to these urls. "
			"If the given urls are empty, choosing the preset won't change the currently selected urls."
			"</ul><br><ul>"
			"<img src=\"filesave\">&nbsp;<i>Save to the current preset</i>: Overwrite the currently selected preset by the current "
			"settings of the GUI. You cannot change the name but the urls."
			"</ul><br><ul>"
			"<img src=\"remove\">&nbsp;<i>Remove the current preset</i>: Remove the current preset selecting the preset above."
			"</ul><br>"
			"Please note that no changes can be done to the <i>default</i> preset."
			"</qt>" );
	QMimeSourceFactory::defaultFactory()->setPixmap( "add",
		KGlobal::iconLoader()->loadIconSet( "add", KIcon::Toolbar, KIcon::SizeSmall ).pixmap(
			QIconSet::Small, QIconSet::Normal ) );
	QMimeSourceFactory::defaultFactory()->setPixmap( "filesave",
		KGlobal::iconLoader()->loadIconSet( "filesave", KIcon::Toolbar, KIcon::SizeSmall ).pixmap(
			QIconSet::Small, QIconSet::Normal ) );
	QMimeSourceFactory::defaultFactory()->setPixmap( "remove",
		KGlobal::iconLoader()->loadIconSet( "remove", KIcon::Toolbar, KIcon::SizeSmall ).pixmap(
			QIconSet::Small, QIconSet::Normal ) );
	QWhatsThis::add( preset_GroupBox, presetWhatsThis );

	const QString ignoreEmptyWhatsThis
		= i18n( "<qt>When comparing files by file size, don't find empty files, "
			"because they are equal by file size, checksum and binary comparison anyway."
			"</qt>" );
	QWhatsThis::add( ignoreEmptyFiles_CheckBox, ignoreEmptyWhatsThis );

	const QString sizeWhatsThis
		= i18n( "<qt>Disable search for duplicate files by file size. "
			"In this case, duplicate files are only found by sub-path "
			"in their respective directory."
			"</qt>" );
	QWhatsThis::add( size_CheckBox, sizeWhatsThis );

	const QString checksumWhatsThis
		= i18n( "<qt>Confirm duplicate files matches (found by <i>size comparison</i>) by MD5 checksum. "
			"Remote files must be downloaded for MD5 checksum. Calculating the checksum is slow on large files."
			"</qt>" );
	QWhatsThis::add( calculateChecksum_CheckBox, checksumWhatsThis );

	const QString binaryWhatsThis
		= i18n( "<qt>Confirm duplicate files matches (found by <i>size comparison</i> and <i>MD5 checksum</i>) "
			"binary comparison. "
			"Remote files must be downloaded for binary comparison. Comparing the files is very slow on large files. "
			"Therefore, you can restrict binary comparison to small files."
			"</qt>" );
	QWhatsThis::add( binaryComparison_CheckBox, binaryWhatsThis );
	QWhatsThis::add( binaryComparison_KIntSpinBox, binaryWhatsThis );
	QWhatsThis::add( MB_TextLabel, binaryWhatsThis );
	QWhatsThis::add( filesSmaller_TextLabel, binaryWhatsThis );

	const QString duplicatesOnlyWhatsThis
		= i18n( "<qt>Only find duplicates inside the right URL, disabling the left URL and therefore also "
			"the missing and newer files search."
			"</qt>" );
	QWhatsThis::add( duplicatesOnly_CheckBox, duplicatesOnlyWhatsThis );

	const QString recursiveWhatsThis
		= i18n( "<qt>Search directories including all subdirectories.<br>"
			"Links to directories are only searched, if <i>Follow soft links</i> is enabled."
			"</qt>" );
	QWhatsThis::add( includeSubdirectories_CheckBox, recursiveWhatsThis );

	const QString includeHiddenItemsWhatsThis
		= i18n( "<qt>Find hidden files and search hidden directories, too."
			"</qt>" );
	QWhatsThis::add( includeHiddenItems_CheckBox, includeHiddenItemsWhatsThis );

	const QString fileIndexWhatsThis // also from kfind
		= i18n( "<qt>This lets you use the files' index created by the <i>slocate</i> "
			"package to speed-up the search; remember to update the index from time to time "
			"(using <i>updatedb</i>)."
			"</qt>" );
	QWhatsThis::add( useFilesIndex_CheckBox, fileIndexWhatsThis );

	const QString caseSensitiveWhatsThis
		= i18n( "<qt>This lets you decide whether the regular expression in the <i>Regex</i> "
			"line edit are interpreted strictly or loosely regarding the letters' case.<br>"
			"This also affects the comparison of relative paths and file names, and therefore "
			"the newer / missing files search result."
			"</qt>" );
	QWhatsThis::add( caseSensitiveSearch_CheckBox, caseSensitiveWhatsThis );

	const QString findEmptyWhatsThis
		= i18n( "<qt>Any directories found during the search are checked whether they are empty, "
			"and if they are, insert them into the <i>Duplicate files</i> list."
			"</qt>" );
	QWhatsThis::add( findEmptyDirectories_CheckBox, findEmptyWhatsThis );

	const QString softLinksWhatsThis
		= i18n( "<qt>Treat soft-links as the files that they are linked to.<br>"
			"If the soft-link is a directory, search it, too."
			"</qt>" );
	QWhatsThis::add( followSoftLinks_CheckBox, softLinksWhatsThis );

	const QString referenceWhatsThis
		= i18n( "<qt>Specify reference directory. This changes sorting of "
			"the lower duplicate files list and selects all files from the other directory, "
			"that you can easily delete the files from the other directory."
			"</qt>" );
	QWhatsThis::add( duplicatesReference_ButtonGroup, referenceWhatsThis );

	const QString duplicates1WhatsThis
		= i18n( "<qt>All files that have duplicates are shown in this list, and, if you "
			"want to, also the first empty directory.<br><br>"
			"<i>To view the duplicates</i> of a file and the file itself (or the empty directories), "
			"you need to <i>click on the respective file</i>, and the list of the files / directories "
			"will appear in the list below.<br><br>"
			"It rather by chance which file of all duplicates is shown here, "
			"it depends on the order in which the files were found.<br><br>"
			"If you want to find duplicates by file size (and MD5 checksum and binary "
			"comparison) these files might have different paths but will have the same size "
			"(or are completely equal) and therefore appear in this list.<br><br>"
			"<font color=black><b>Black</b></font> colored items have only one duplicate (that is "
			"in the other searched url and has the same relative path), <font color=blue><b>blue</b></font> "
			"colored ones have duplicates that are no duplicates by relative path."
			"</qt>" );
	QWhatsThis::add( duplicates1_KListView, duplicates1WhatsThis );

	const QString duplicates2WhatsThis
		= i18n( "<qt>All duplicates of one file size (and MD5 checksum and binary comparison) "
			"are shown here (or all empty directories), depending on the selected item in the "
			"list above.<br><br>"
			"They are sorted by reference directory (selected on the right). "
			"Files that do not belong to the reference directory come first. "
			"Files from the reference directory are not selected initially, so "
			"you can easily delete the other ones.<br><br>"
			"For easier identification, files within the left url are colored "
			"<font color=#059b0a><b>green</b></font>, the other ones "
			"<font color=#0f19c8><b>blue</b></font>."
			"</qt>" );
	QWhatsThis::add( duplicates2_KListView, duplicates2WhatsThis );

	const QString missing1WhatsThis
		= i18n( "<qt>This list displays the result of the comparison of relative paths and file names" ) + ".<br><br>" +
			i18n( "All files, that exist in the left url but not in the right one, appear in this list.</qt>" );
	QWhatsThis::add( missing1_KListView, missing1WhatsThis );

	const QString missing2WhatsThis
		= i18n( "<qt>This list displays the result of the comparison of relative paths and file names" ) + ".<br><br>" +
			i18n( "All files that exist in the right url but not in the left one appear in this list.</qt>" );
	QWhatsThis::add( missing2_KListView, missing2WhatsThis );

	const QString newerWhatsThis
		= i18n( "<qt>This list displays the result of the comparison of relative paths and file names" ) +
			i18n( ", combined with a comparison of creation / modification time and file size (and MD5 "
				"checksum and binary comparison), if you want to.<br><br>"
				"There are three different kinds of pairs of files:"
				"<ul>"
				"<li><i>Two different files (by file size etc.), one newer than the other</i>: "
				"The newer file is colored <font color=black><b>black</b></font>, "
				"the older one <font color=#0f19c8><b>blue</b></font>.</li>"
				"<li><i>Two equal files (by file size etc.), one newer than the other</i>: "
				"The newer file is colored <font color=black><b>black</b></font>, "
				"the older one <font color=#9128be><b>purple</b></font>.</li>"
				"<li><i>Two different files with the same time stamp</i>: "
				"Both files are colored <font color=#e61e1e><b>red</b></font>, because "
				"Komparator cannot decide, which file is the desired one.</li>"
				"</ul>"
				"</qt>" );
	QWhatsThis::add( newer1_KListView, newerWhatsThis );
	QWhatsThis::add( newer2_KListView, newerWhatsThis );

	const QString warningTrashWhatsThis
		= i18n( "<qt>Show a warning before trashing files.</qt>" );
	QWhatsThis::add( warningTrash_CheckBox, warningTrashWhatsThis );
	const QString warningDeleteWhatsThis
		= i18n( "<qt>Show a warning before deleting files.</qt>" );
	QWhatsThis::add( warningDelete_CheckBox, warningDeleteWhatsThis );
	const QString warningOverwriteCopyWhatsThis
		= i18n( "<qt>Show a warning before overwriting files while copying files.</qt>" );
	QWhatsThis::add( warningOverwriteCopy_CheckBox, warningOverwriteCopyWhatsThis );
	const QString warningOverwriteMoveWhatsThis
		= i18n( "<qt>Show a warning before overwriting files while moving files.</qt>" );
	QWhatsThis::add( warningOverwriteMove_CheckBox, warningOverwriteMoveWhatsThis );
	const QString warningReloadWhatsThis
		= i18n( "<qt>Show a warning before reloading the search result.</qt>" );
	QWhatsThis::add( warningReload_CheckBox, warningReloadWhatsThis );
	const QString warningRemovePresetWhatsThis
		= i18n( "<qt>Show a warning before removing a preset.</qt>" );
	QWhatsThis::add( warningRemovePreset_CheckBox, warningRemovePresetWhatsThis );
	const QString systrayIconWhatsThis
		= i18n( "<qt>Show a system tray icon.</qt>" );
	QWhatsThis::add( systray_CheckBox, systrayIconWhatsThis );
	const QString notificationWhatsThis
		= i18n( "<qt>Show a passive notification when a user interaction is finished and komparator isn't the active window.</qt>" );
	QWhatsThis::add( notification_CheckBox, notificationWhatsThis );
	const QString textColorWhatsThis
		= i18n( "<qt>Use the komparator text color theme as described in the documentation.</qt>" );
	QWhatsThis::add( textColor_CheckBox, textColorWhatsThis );

	const QString tabWhatsThis
		= i18n( "<qt>Enable or disable tabs by clicking on the icons."
			"<br><br><ul>"
			"<img source=\"ok\">&nbsp;&nbsp;&nbsp;Display search results."
			"</ul><br><ul>"
			"<img source=\"no\">&nbsp;&nbsp;&nbsp;Don't display search result."
			"</ul><br>"
			"If you don't need the result of one tab, you can disable it "
			"to have a slight search speed improvement. Disabling both <i>missing</i> "
			"and <i>newer tab</i> (with enabled <i>Size comparison</i> check box) disables "
			"the relative path comparison, which will lead to a significant search speed improvement."
			"</qt>" );
	QMimeSourceFactory::defaultFactory()->setPixmap( "ok",
		KGlobal::iconLoader()->loadIconSet( "ok", KIcon::Toolbar, KIcon::SizeSmall ).pixmap(
			QIconSet::Small, QIconSet::Normal ) );
	QMimeSourceFactory::defaultFactory()->setPixmap( "no",
		KGlobal::iconLoader()->loadIconSet( "no", KIcon::Toolbar, KIcon::SizeSmall ).pixmap(
			QIconSet::Small, QIconSet::Normal ) );
	QWhatsThis::add( main_TabWidget, tabWhatsThis );

	const QString url1WhatsThis
		= i18n( "<qt>Enter an URL to be searched. You must use an absolute path as "
			"<i>/path/to/directory</i>, <b>not</b> a relative one as <i>../directory</i>.<br>"
			"You can use any KIO-slave protocol, like <i>ftp://</i>, <i>fish://</i> "
			"or <i>media:/</i>." ) + "<br><br>" +
			i18n( "This will be the parent URL of the files displayed on the left side."
				"</qt>" );
	QWhatsThis::add( url1_TextLabel, url1WhatsThis );
	QWhatsThis::add( url1_KURLRequester, url1WhatsThis );

	const QString url2WhatsThis
		= i18n( "<qt>Enter an URL to be searched. You must use an absolute path as "
			"<i>/path/to/directory</i>, <b>not</b> a relative one as <i>../directory</i>.<br>"
			"You can use any KIO-slave protocol, like <i>ftp://</i>, <i>fish://</i> "
				"or <i>media:/</i>." ) + "<br><br>" +
			i18n( "This will be the parent URL of the files displayed on the right side."
					"</qt>" );
	QWhatsThis::add( url2_TextLabel, url2WhatsThis );
	QWhatsThis::add( url2_KURLRequester, url2WhatsThis );
	const QString virtualSubdirectoriesWhatsThis
		= i18n( "<qt>Here you can specify \"virtual subdirectories\" to the right side's URL. "
			"Virtual subdirectories essentially behave as if these directories were linked into the "
			"right side's URL (parent directory).<br>"
			"This might be useful if you want to backup several directories into a single one.<br><br>"
			"You should not use virtual subdirectories, that have the same name as a directory, that "
			"really exists in your parent directory, neither is it a good idea to use "
			"real subdirectories of the parent directory as virtual subdirectories."
			"</qt>" );
	QWhatsThis::add( subdirectories_KomparatorUrlList, virtualSubdirectoriesWhatsThis );

	const QString filterWhatsThis
		= i18n( "<qt>Filter the search results, and hide all entries that don't match "
			"your specifications. <font color=blue><b>Blue</b></font> background color of the "
			"filter button and the counter shows, that some entries are hidden."
			"</qt>" );
	QWhatsThis::add( duplicates1_KomparatorFileFilter, filterWhatsThis );
	QWhatsThis::add( duplicates2_KomparatorFileFilter, filterWhatsThis );
	QWhatsThis::add( missing1_KomparatorFileFilter, filterWhatsThis );
	QWhatsThis::add( missing2_KomparatorFileFilter, filterWhatsThis );
	QWhatsThis::add( newer1_KomparatorFileFilter, filterWhatsThis );
	QWhatsThis::add( newer2_KomparatorFileFilter, filterWhatsThis );

	const QString counterWhatsThis
		= i18n( "<qt>Number of items that are currently shown. <font color=blue><b>Blue</b></font> background color "
		"shows, that some entries are hidden."
		"</qt>" );
	QWhatsThis::add( duplicates1ItemCount_TextLabel, counterWhatsThis );
	QWhatsThis::add( duplicates2ItemCount_TextLabel, counterWhatsThis );
	QWhatsThis::add( missing1ItemCount_TextLabel, counterWhatsThis );
	QWhatsThis::add( missing2ItemCount_TextLabel, counterWhatsThis );
	QWhatsThis::add( newer1ItemCount_TextLabel, counterWhatsThis );
	QWhatsThis::add( newer2ItemCount_TextLabel, counterWhatsThis );
}

KomparatorWidget::~KomparatorWidget()
{
	m_query->kill();

	compare_job->stop();

	m_empty_directory_job->stop();
	m_copy_job->stop();
	m_move_job->stop();
	m_remove_job->stop();

	m_progress_timer->stop();

	if ( m_requested_download ) // Killing current download job and md5sum / byte array operation. Temporary file will be deleted.
	{
		m_requested_download->setDownloadError( KIO::ERR_USER_CANCELED );
	}
	m_requested_download = NULL;

	if ( m_requested_download_job )
	{
		QString del_str = ((KIO::FileCopyJob*)m_requested_download_job)->destURL().url();

		m_requested_download_job->kill();

		KIO::NetAccess::del( del_str, this );
		KIO::NetAccess::del( del_str + ".part", this ); // FIXME: ugly.
	}
	m_requested_download_job = NULL;

	compare_job->wait(); // compare_job uses m_files etc., so we need to wait until kompare has really finished.

	m_empty_directory_job->wait();
	m_copy_job->wait();
	m_move_job->wait();
	m_remove_job->wait();

	m_symlinked_directories.clear();
	m_virtual_subdirectories.clear();

	duplicates1_KListView->clear();
	duplicates2_KListView->clear();

	missing1_KListView->clear();
	missing2_KListView->clear();

	newer1_KListView->clear();
	newer2_KListView->clear();

	// write global config
	KConfig *conf = KGlobal::config();
	conf->deleteGroup( "Global" );
	conf->setGroup( "Global" );

	conf->writeEntry( "systray", systray_CheckBox->isChecked() );
	conf->writeEntry( "notification_ready", notification_CheckBox->isChecked() );
	conf->writeEntry( "text_color", textColor_CheckBox->isChecked() );

	// write widget size
	conf->writeEntry( "total_size", size() );

	// write current preset
	conf->writeEntry( "preset", preset_KComboBox->currentItem() );

	// write sizes of splitter
	int i = 0;
	QValueList<int> list = duplicates_Splitter->sizes();
	QValueList<int>::Iterator it = list.begin();
	while( it != list.end() )
	{
		conf->writeEntry( QString( "splitter%1" ).arg( i ), *it );
		++it;
		i++;
	}

	// write presets to config file
	for ( i=1; i<50; i++ )
	{
		if ( !conf->hasGroup( QString( "Preset%1" ).arg( i ) ) ) break;

		conf->deleteGroup( QString( "Preset%1" ).arg( i ) );
	}

	KomparatorPreset *current_preset;
	for ( i=1; i<50; i++ )
	{
		current_preset = m_preset_list.at( i );
		if ( current_preset )
		{
			current_preset->saveConfig( i );
		}
		else
		{
			break;
		}
	}

	delete m_files;
	m_files = NULL;

	KFileItemExt *to_delete = m_dirs_dupes;
	while ( to_delete )
	{
		to_delete = to_delete->duplicates_size;
		delete m_dirs_dupes;
		m_dirs_dupes = to_delete;
	}

	delete m_popup_menu;
	m_popup_menu = NULL;

	delete m_query;
	m_query = NULL;

	delete compare_job;
	compare_job = NULL;

	delete m_empty_directory_job;
	m_empty_directory_job = NULL;
	delete m_copy_job;
	m_copy_job = NULL;
	delete m_move_job;
	m_move_job = NULL;
	delete m_remove_job;
	m_remove_job = NULL;

	delete m_config_global;
	m_config_global = NULL;
}

void KTabBar::mousePressEvent( QMouseEvent * e ) // This function is responsible to enable / disable icons. Taken from ktabbar.cpp.
{
	QTab *t = selectTab( e->pos() );
	if ( t && t->isEnabled() && t->iconSet() )
	{
		QPixmap pixmap = t->iconSet()->pixmap( QIconSet::Small, QIconSet::Normal );
		QRect rect( 0, 0, pixmap.width() + 4, pixmap.height() +4);

		int xoff = 0, yoff = 0;
		// The additional offsets were found by try and error, TODO: find the rational behind them
		if ( t == tab( currentTab() ) )
		{
			xoff = style().pixelMetric( QStyle::PM_TabBarTabShiftHorizontal, this ) + 3;
			yoff = style().pixelMetric( QStyle::PM_TabBarTabShiftVertical, this ) - 4;
		}
		else
		{
			xoff = 7;
			yoff = 0;
		}
		rect.moveLeft( t->rect().left() + 2 + xoff );
		rect.moveTop( t->rect().center().y()-pixmap.height()/2 + yoff );
		if ( rect.contains( e->pos() ) ) // This lousy part comes from me.
		{
			if ( pixmap.convertToImage() ==
					KGlobal::iconLoader()->loadIconSet( "no", KIcon::Toolbar, KIcon::SizeSmall ).pixmap(
						QIconSet::Small, QIconSet::Normal ).convertToImage() )
				t->setIconSet( KGlobal::iconLoader()->loadIconSet( "ok", KIcon::Toolbar, KIcon::SizeSmall ) );
			else
				t->setIconSet( KGlobal::iconLoader()->loadIconSet( "no", KIcon::Toolbar, KIcon::SizeSmall ) );

			((QTabBar*)this)->update();

			return;
		}
	}

	QTabBar::mousePressEvent( e );
}

/****************************************
* Drag & Drop Functions                *
****************************************/

QDragObject *KListView::dragObject()
{
	KURL::List urls;

	QPixmap *pm = NULL;

	QListViewItemIterator it( this );
	while ( it.current() )
	{
		if ( it.current()->isSelected() )
		{
			urls.append( ((KListViewItemSingle*)it.current())->item->url().url() );
			if ( !pm )
			{
				pm = new QPixmap( ((KListViewItemSingle*)it.current())->item->pixmap( 32 ) );
			}
		}
		it++;
	}

	KURLDrag *ret = new KURLDrag( urls, this );

	if ( pm && !pm->isNull() ) ret->setPixmap( *pm );

	return ret;
}

void KomparatorWidget::slotKListViewDropped( KListView *list_view, QDropEvent *event, QListViewItem * )
{
  event->accept( true );

	if ( list_view == missing1_KListView ) current_list_view = missing2_KListView; // set origin of drag.
	else if ( list_view == missing2_KListView ) current_list_view = missing1_KListView;
	else if ( list_view == newer1_KListView ) current_list_view = newer2_KListView;
	else if ( list_view == newer2_KListView ) current_list_view = newer1_KListView;
	else return;

	if ( KApplication::keyboardMouseState() & Qt::ShiftButton ) slotMoveToOtherSide();
	else if ( KApplication::keyboardMouseState() & Qt::ControlButton ) slotCopyToOtherSide();
	else
	{
		KPopupMenu *moveOrCopyMenu = new KPopupMenu( this );

		QString move_here = i18n( "Move here" ) + "\t" + tr( "Shift" );
		QString copy_here = i18n( "Copy here" ) + "\t" + tr( "Ctrl" );
		QString cancel = i18n( "Cancel" ) + "\t" + tr( "Esc" );

		moveOrCopyMenu->insertItem( SmallIcon( "goto" ), move_here, this, SLOT( slotMoveToOtherSide() ) );
		moveOrCopyMenu->insertItem( SmallIcon( "editcopy" ), copy_here, this, SLOT( slotCopyToOtherSide() ) );
		moveOrCopyMenu->insertSeparator();
		moveOrCopyMenu->insertItem( SmallIcon( "cancel" ), cancel );

		moveOrCopyMenu->popup( list_view->mapToGlobal( list_view->contentsToViewport( event->pos() ) ), 1 );
	}
}

bool KListView::acceptDrag ( QDropEvent *event ) const // acceptDrag only under certain circumstances:
{
	if ( !event->source() || !this ) return false; // no acceptable source (only missing1 & 2 & newer1 & 2).

	if ( ( qstrcmp( event->source()->name(), "missing1_KListView" ) == 0
			&& qstrcmp( this->name(), "missing2_KListView" ) == 0  ) ||
		( qstrcmp( event->source()->name(), "missing2_KListView" ) == 0
			&& qstrcmp( this->name(), "missing1_KListView" ) == 0  ) ||
		( qstrcmp( event->source()->name(), "newer1_KListView" ) == 0
			&& qstrcmp( this->name(), "newer2_KListView" ) == 0  ) ||
		( qstrcmp( event->source()->name(), "newer2_KListView" ) == 0
			&& qstrcmp( this->name(), "newer1_KListView" ) == 0  ) )
	{
		return true;
	}

	return false;
}



/****************************************
* Keyboard Functions                   *
****************************************/

void KListView::keyPressEvent( QKeyEvent *event )
{
	switch( event->key() ) // ignoring events passes them to the parent (=KomparatorWidget).
	{
		case Qt::Key_Delete:
			event->ignore();
			return;
		case Qt::Key_C:
			if ( event->state() == Qt::ControlButton )
			{
				event->ignore();
				return;
			}
			break;
		case Qt::Key_Enter:  // Enter and return are equal events for us.
		case Qt::Key_Return:
			event->ignore();
			return;
		default:
			break;
	}

	event->accept();
	QListView::keyPressEvent( event );
}

void KomparatorWidget::keyPressEvent( QKeyEvent *event )
{
	if ( duplicates1_KListView->hasFocus() ) current_list_view = duplicates1_KListView;
	else if ( duplicates2_KListView->hasFocus() ) current_list_view = duplicates2_KListView;
	else if ( missing1_KListView->hasFocus() ) current_list_view = missing1_KListView;
	else if ( missing2_KListView->hasFocus() ) current_list_view = missing2_KListView;
	else if ( newer1_KListView->hasFocus() ) current_list_view = newer1_KListView;
	else if ( newer2_KListView->hasFocus() ) current_list_view = newer2_KListView;
	else
	{
		event->ignore();
		return;
	}

	switch( event->key() ) // ignoring events passes them to the parent (=KomparatorWidget).
	{
		case Qt::Key_Delete:
			if ( event->state() == Qt::ShiftButton )
			{
				slotDeleteFiles();
			}
			else
			{
				slotTrashFiles();
			}
			event->accept();
			return;
		case Qt::Key_C:
			if ( event->state() == Qt::ControlButton )
			{
				event->accept();
				slotCopySelection();
				return;
			}
			break;
		case Qt::Key_Enter:  // Enter and return are equal events for us.
		case Qt::Key_Return:
			event->accept();
			slotOpenBinding();
			return;
			break;
		default:
			break;
	}

	event->ignore();
}



/****************************************
* Context Menu Functions               *
****************************************/

void KomparatorWidget::slotContextMenu( KListView *list_view, QListViewItem *item, const QPoint &p )
{
	if ( !item ) return;
	if ( !list_view ) return;

	int count = list_view->selectedItems().count();

	current_list_view = list_view;

	if ( count == 0 ) return;

	if ( !m_popup_menu )
	{
		m_popup_menu = new KPopupMenu( this );
	}
	else
	{
		m_popup_menu->clear();
	}

	if ( count == 1 ) // The accelerator keys actually don't do anything but show the shortcut in the menu
	{                 // reason: 1) if no popup menu has been created, accelerators wouldn't work
		QString title;  //         2) the accelerator keys would affect the item, that was clicked with popupmenu last time, not the currently selected item.
		title = ( current_list_view == duplicates1_KListView ) ? item->text( 1 ) : item->text( 0 );
		m_popup_menu->insertTitle( ((KListViewItemSingle *)item)->item->pixmap( 16 ), title );
	}
	else
	{
		m_popup_menu->insertTitle( i18n( "Selected Files" ) );
	}

	m_popup_menu->insertItem( SmallIcon( "clear_left" ), i18n( "Remove from list" ), this, SLOT( slotRemoveFromList() ) );
	if ( list_view == missing1_KListView || list_view == newer1_KListView )
	{
		m_popup_menu->insertItem( SmallIcon( "editcopy" ), i18n( "Copy to right side" ), this, SLOT( slotCopyToOtherSide() ) );
		m_popup_menu->insertItem( SmallIcon( "goto" ), i18n( "Move to right side" ), this, SLOT( slotMoveToOtherSide() ) );
	}
	else if ( list_view == missing2_KListView || list_view == newer2_KListView )
	{
		m_popup_menu->insertItem( SmallIcon( "editcopy" ), i18n( "Copy to left side" ), this, SLOT( slotCopyToOtherSide() ) );
		m_popup_menu->insertItem( SmallIcon( "goto" ), i18n( "Move to left side" ), this, SLOT( slotMoveToOtherSide() ) );
	}
	else if ( list_view == duplicates1_KListView )
	{
		m_popup_menu->insertItem( SmallIcon( "edittrash" ), i18n( "Trash non-reference files" ), this, SLOT( slotTrashNonreferenceFiles() ) );
		m_popup_menu->insertItem( SmallIcon( "editdelete" ), i18n( "Delete non-reference files" ), this, SLOT( slotDeleteNonreferenceFiles() ) );
	}
	m_popup_menu->insertSeparator();

	if ( ( current_list_view == missing1_KListView && missing1_KListView->selectedItems().count() == 1 && missing2_KListView->selectedItems().count() == 1 ) ||
		( current_list_view == missing2_KListView && missing2_KListView->selectedItems().count() == 1 && missing1_KListView->selectedItems().count() == 1 ) ||
		( current_list_view == newer1_KListView && newer1_KListView->selectedItems().count() == 1 && newer2_KListView->selectedItems().count() == 1 ) ||
		( current_list_view == newer2_KListView && newer2_KListView->selectedItems().count() == 1 && newer1_KListView->selectedItems().count() == 1 ) )
	{
		m_popup_menu->insertItem( SmallIcon( "viewmag" ), i18n( "Compare selected files" ), this, SLOT( slotCompareSelected() ) );
	}

	if ( ( current_list_view == newer1_KListView || current_list_view == newer2_KListView ) && current_list_view->selectedItems().count() == 1 )
	{
		m_popup_menu->insertItem( SmallIcon( "viewmag" ), i18n( "Compare newer files pair" ), this, SLOT( slotCompareEqual() ) );
	}

	if ( count == 1 )
	{
		m_popup_menu->insertItem( SmallIcon( "fileopen" ), i18n( "Open" ), this, SLOT( slotOpenBinding() ) );
		m_popup_menu->insertItem( SmallIcon( "window_new" ), i18n( "Open Directory" ), this, SLOT( slotOpenDirectory() ) );
		m_popup_menu->insertSeparator();
	}

	m_popup_menu->insertItem( SmallIcon( "editcopy" ), i18n( "Copy" ), this, SLOT( slotCopySelection() ) );
	m_popup_menu->insertItem( SmallIcon( "edittrash" ), i18n( "Trash" ), this, SLOT( slotTrashFiles() ) );
	m_popup_menu->insertItem( SmallIcon( "editdelete" ), i18n( "Delete" ), this, SLOT( slotDeleteFiles() ) );

	if ( count == 1 )
	{
		m_popup_menu->insertSeparator();
		m_popup_menu->insertItem( i18n( "Open With..." ), this, SLOT( slotOpenWith() ) );
	}
	
	m_popup_menu->insertSeparator();
	m_popup_menu->insertItem( i18n( "Properties" ), this, SLOT( slotFileProperties() ) );

	m_popup_menu->popup( p, 1 );
}



/****************************************
* Interface Functions                  *
****************************************/

void KomparatorWidget::slotSearch()
{
	switch ( m_status )
	{
		case CLEAN:
			m_interaction_cancel = false;

			search1_KLed->setColor( QColor( 255, 0, 0 ) );  // Red
			search2_KLed->setColor( QColor( 255, 0, 0 ) );
			search3_KLed->setColor( QColor( 255, 0, 0 ) );

			enable( false );

			url1_KURLRequester->button()->setEnabled( false );
			url2_KURLRequester->button()->setEnabled( false );
			url1_KURLRequester->lineEdit()->setReadOnly( true );
			url2_KURLRequester->lineEdit()->setReadOnly( true );
			duplicateOptions_GroupBox->setEnabled( false );
			findEmptyDirectories_CheckBox->setEnabled( false );
			preset_GroupBox->setEnabled( false );
			subdirectories_KomparatorUrlList->setReadOnly( true );

			if ( !initializeQuery() ) break;

			m_unreadable_files.clear();
			m_number_of_files = 0;

			m_interaction_empty_dirs.clear();

			m_status = SEARCHING_URL_1;
			search_KPushButton->setText( i18n( "Cancel" ) );
			search_KPushButton->setIconSet( SmallIcon( "cancel" ) );

			emit( signalProgress( i18n( "Searching left URL for files..." ), -1, "" ) );
			m_progress_timer->start( 100, false/*single shot*/ );
			emit( signalStartSystrayMovie() );

			m_symlinked_directories.clear();

			if ( !duplicatesOnly_CheckBox->isChecked() ) m_query->start();
			else slotResult( 0, "" );
			return;
			break;

		case SEARCHING_URL_1:
		case SEARCHING_URL_2:
			m_interaction_cancel = true;
			m_symlinked_directories.clear();
			m_virtual_subdirectories.clear();
			m_query->kill();
			break;

		case SEARCHING_EMPTY_DIRECTORIES:
			m_empty_directory_job->stop();
			m_empty_directory_job->wait();

			m_interaction_cancel = true;
			break;

		case COMPARING:
			compare_job->stop();

			if ( m_requested_download ) // Killing current download job and md5sum /
			{                         // byte array operation. Temporary file will be deleted.
				m_requested_download->setDownloadError( KIO::ERR_USER_CANCELED );
			}
			m_requested_download = NULL;

			if ( m_requested_download_job )
			{
				QString del_str = ((KIO::FileCopyJob*)m_requested_download_job)->destURL().url();

				m_requested_download_job->kill();

				KIO::NetAccess::del( del_str, this );
				KIO::NetAccess::del( del_str + ".part", this ); // FIXME: ugly.
			}
			m_requested_download_job = NULL;

			compare_job->wait(); // kompare uses files_dupes etc., so we need to wait until kompare has really finished.

			break;

		case READY:
			break;

		case INTERACTING:
			m_interaction_cancel = true;
			return;

		default:
			break;
	}

	duplicates1_KListView->clear();
	duplicates2_KListView->clear();

	missing1_KListView->clear();
	missing2_KListView->clear();

	newer1_KListView->clear();
	newer2_KListView->clear();

	delete m_files;
	m_files = NULL;

	KFileItemExt *to_delete = m_dirs_dupes;
	while ( to_delete )
	{
		to_delete = to_delete->duplicates_size;
		delete m_dirs_dupes;
		m_dirs_dupes = to_delete;
	}

	m_symlinked_directories.clear();
	m_virtual_subdirectories.clear();

	search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
	search2_KLed->setColor( QColor( 0, 255, 0 ) );
	search3_KLed->setColor( QColor( 0, 255, 0 ) );

	enable( true );

	url1_KURLRequester->button()->setEnabled( true );
	url2_KURLRequester->button()->setEnabled( true );
	url1_KURLRequester->lineEdit()->setReadOnly( false );
	url2_KURLRequester->lineEdit()->setReadOnly( false );
	duplicateOptions_GroupBox->setEnabled( true );
	findEmptyDirectories_CheckBox->setEnabled( true );
	preset_GroupBox->setEnabled( true );
	subdirectories_KomparatorUrlList->setReadOnly( false );

	m_unreadable_files.clear();
	m_number_of_files = 0;

	emit( signalCount( DUPLICATES, 1, 0 ) );
	emit( signalCount( DUPLICATES, 2, 0 ) );
	emit( signalCount( MISSING, 1, 0 ) );
	emit( signalCount( MISSING, 2, 0 ) );
	emit( signalCount( NEWER, 1, 0 ) );
	emit( signalCount( NEWER, 2, 0 ) );

	m_status = CLEAN;
	search_KPushButton->setText( i18n( "Search" ) );
	search_KPushButton->setIconSet( SmallIcon( "komparator" ) );

	resizeListViewColumns();

	m_progress_timer->stop();
	// process signalProgress events, so the next event will display the current state.
	KApplication::kApplication()->eventLoop()->processEvents( QEventLoop::ExcludeSocketNotifiers, 50 /* Process for maximum of x milliseconds, then continue */ );

	m_url1 = KURL( "" );
	m_url2 = KURL( "" );

	emit( signalProgress( i18n( "Ready." ), -1, "" ) );
	emit( signalStopSystrayMovie() );
}

void KomparatorWidget::slotAbout()
{
	KAboutApplication *about = new KAboutApplication( this );
	about->show();
}

void KomparatorWidget::slotWhatsThis()
{
	QWhatsThis::enterWhatsThisMode();
}

void KomparatorWidget::slotDuplicateSelected()
{
	duplicates2_KListView->clear();

	int i;
	for ( i=0; i<7; i++ )
	{
		duplicates2_KListView->setColumnWidth( i, m_duplicates2_width[i] );
	}
	duplicates2_KListView->setColumnWidth( 7, m_duplicates2_width[7] );

	if ( !duplicates1_KListView->currentItem() ) return;

	KFileItemExt *tmpfile;

	KListViewItemSingle *tmpitem = new KListViewItemSingle( duplicates2_KListView,
		(((KListViewItemDups*)duplicates1_KListView->currentItem())->item), NULL );
	emit( signalCount( DUPLICATES, 2, -1 ) );
	tmpitem->setText( 1, prettyURL( tmpitem->item->url(), URL_DIR_OR_URL_ONLY, KURL( "" ) ) );

	if ( (((KListViewItemDups*)duplicates1_KListView->currentItem())->item)->dir == 0 )
	{
		tmpitem->setText( 7, i18n( "Left" ) );
		tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_green );
	}
	else
	{
		tmpitem->setText( 7, i18n( "Right" ) );
		tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_blue );
	}

	tmpfile = ((KListViewItemDups*)duplicates1_KListView->currentItem())->item->duplicates_size;

	while ( tmpfile )
	{
		tmpitem = new KListViewItemSingle( duplicates2_KListView, tmpfile, NULL );
		emit( signalCount( DUPLICATES, 2, -1 ) );
		tmpitem->setText( 1, prettyURL( tmpitem->item->url(), URL_DIR_OR_URL_ONLY, KURL( "" ) ) );

		if ( tmpfile->dir == 0 )
		{
			tmpitem->setText( 7, i18n( "Left" ) );
			tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_green );
		}
		else
		{
			tmpitem->setText( 7, i18n( "Right" ) );
			tmpitem->setKind( NONE, textColor_CheckBox->isChecked(), m_blue );
		}

		tmpfile = tmpfile->duplicates_size;
	}

	slotReferenceLeft( duplicatesReference1_RadioButton->isChecked() );
	updateFilters();
}

void KomparatorWidget::slotReferenceLeft( bool is_ref )
{
	duplicates2_KListView->setSorting( 7, !is_ref );

	// evil workaround to make grey lines disappear which come from CTRL-unselect style.
	duplicates2_KListView->setSelectionMode( QListView::Single );
	duplicates2_KListView->clearSelection();
	duplicates2_KListView->setSelected( duplicates2_KListView->firstChild(), true ); // select, so all other items are unselected.
	duplicates2_KListView->setSelectionMode( QListView::Extended );

	QListViewItemIterator it( duplicates2_KListView );
	while ( it.current() )
	{
		it.current()->setSelected( ( ( !is_ref && ((KListViewItemSingle*)it.current())->item->dir == 0 ) ||
			( is_ref && ((KListViewItemSingle*)it.current())->item->dir == 1 ) ) ? true : false );
		it++;
	}

//	Don't do that any more; this is unexpected behavior.
// 	duplicates2_KListView->setFocus();
}

void KomparatorWidget::slotRemoveFromList()
{
	if ( !current_list_view ) return;

	emit( signalProgress( ( current_list_view->selectedItems().count() == 1 ) ?
		i18n( "Removing 1 file from list..." ) :
		i18n( "Removing %1 files from list..." ).arg( current_list_view->selectedItems().count() ), -1, "" ) );
	emit( signalStartSystrayMovie() );

	if ( current_list_view == duplicates1_KListView ||
		current_list_view == missing1_KListView ||
		current_list_view == missing2_KListView ||
		current_list_view == newer1_KListView ||
		current_list_view == newer2_KListView )
	{
		current_list_view->selectedItems().setAutoDelete( true );
	}
	else
	{
		QPtrList< QListViewItem > selected_items = duplicates2_KListView->selectedItems();
		QListViewItem *list_item;
		for ( list_item=selected_items.first(); list_item; list_item=selected_items.next() )
		{
			removeDeletedFromLists( ((KListViewItemSingle*)list_item)->item, true, false );
		}
	}

	int visible_items = 0;
	QListViewItemIterator it( current_list_view );
	while( it.current() )
	{
		if ( it.current()->isVisible() ) visible_items++;

		++it;
	}

	if ( current_list_view == duplicates1_KListView ) emit( signalCount( DUPLICATES, 1, visible_items ) );
	else if ( current_list_view == duplicates2_KListView ) emit( signalCount( DUPLICATES, 2, visible_items ) );
	else if ( current_list_view == missing1_KListView ) emit( signalCount( MISSING, 1, visible_items ) );
	else if ( current_list_view == missing2_KListView ) emit( signalCount( MISSING, 2, visible_items ) );
	else if ( current_list_view == newer1_KListView ) emit( signalCount( NEWER, 1, visible_items ) );
	else if ( current_list_view == newer2_KListView ) emit( signalCount( NEWER, 2, visible_items ) );

	current_list_view = NULL;

	emit( signalProgress( "", -1, "" ) );
	emit( signalStopSystrayMovie() );
}

void KomparatorWidget::slotCopyToOtherSide()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_interaction_empty_dirs.clear();

	m_interaction_cancel = false;
	m_status = INTERACTING;
	search_KPushButton->setText( i18n( "Cancel" ) );
	search_KPushButton->setIconSet( SmallIcon( "cancel" ) );

	m_interaction_copy_remaining_items = current_list_view->selectedItems();
	m_interaction_total = m_interaction_copy_remaining_items.count();
	m_interaction_progress = 0;
	m_interaction_progress_str = ( m_interaction_total == 1 ) ?
		i18n( "Copying 1 file from one side to the other..." ) :
		i18n( "Copying %1 files from one side to the other..." ).arg( m_interaction_total );

	enable( false );

	emit( signalProgress( m_interaction_progress_str, -1, "" ) );
	emit( signalStartSystrayMovie() );

	QString destination = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? // destination is right side
		m_url2.url() : m_url1.url();

	bool overwrite = false; // doesn't make sense to overwrite if missing (then the target file shouldn't exist), but should overwrite if newer.

	KListViewItemSingle *i;
	if ( current_list_view == newer1_KListView || current_list_view == newer2_KListView )
	{
		QStringList copy;
		
		for ( i=(KListViewItemSingle *)m_interaction_copy_remaining_items.first(); i; i=(KListViewItemSingle *)m_interaction_copy_remaining_items.next() )
		{
			copy.push_back( prettyURL( i->item->url(), URL_DIR_OR_URL, KURL( "" ) ) + " " + i18n( "to" )
				+ " " + prettyURL( i->brother->item->url(), URL_DIR_OR_URL, KURL( "" ) ) );
		}

		QString msg;
		msg = ( m_interaction_total == 1 ) ?
			i18n( "Do you really want to copy the selected file overwriting the existing one?" ) :
			i18n( "Do you really want to copy the %1 selected files overwriting the existing ones?" ).arg( m_interaction_total );

		KGlobal::config()->reparseConfiguration();
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
		if ( KMessageBox::warningContinueCancelList( parentWidget(), msg, copy, i18n( "Copy files" ),
					KGuiItem( i18n( "Overwrite" ), "editoverwrite"), ":komparator_message_overwrite_copy" ) == KMessageBox::Cancel )
		{
			QApplication::restoreOverrideCursor();
			m_interaction_copy_remaining_items.clear();
			slotMoveNextStat( NULL );
			return;
		}
		QApplication::restoreOverrideCursor();

		m_config_global->reparseConfiguration();
		bool warning_overwrite_copy = !m_config_global->hasKey( ":komparator_message_overwrite_copy" ) |
			m_config_global->readBoolEntry( ":komparator_message_overwrite_copy" );
		warningOverwriteCopy_CheckBox->setChecked( warning_overwrite_copy );

		overwrite = true;
	}

	bool success;
	KURL dest;
	int last_items_dir = ((KListViewItemSingle *)m_interaction_copy_remaining_items.last())->item->dir;
	for ( i=(KListViewItemSingle *)m_interaction_copy_remaining_items.first(); i; i=(KListViewItemSingle *)m_interaction_copy_remaining_items.next() )
	{
		m_virtual_subdirectory_current = KURL( "" );
		if ( last_items_dir == 0 )
		{
			dest = KURL::fromPathOrURL( destination );
			dest.addPath( prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );

			KURL::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KURL::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upURL().prettyURL() + "/" + prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upURL();
					dest.addPath( prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = KURL::fromPathOrURL( destination );
			dest.addPath( prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
		}
		dest.cleanPath();

		KURL dir = dest.upURL();

		while ( !KIO::NetAccess::mkdir( dir, this ) ) // create all destination subdirs if not present
		{
			KURL mkdir = dir.upURL();
			success = true;

			while ( !KIO::NetAccess::mkdir( mkdir, this ) )
			{
				if ( mkdir == mkdir.upURL() )
				{
					success = false;
					break;
				}

				mkdir = mkdir.upURL();
			}

			if ( !success ) break;
		}
	}

	KURL src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
	src.cleanPath();

	KIO::FileCopyJob *job = KIO::file_copy( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
		overwrite, false /*resume*/, false /*progress*/ );
	connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotCopyNext( KIO::Job * ) ) );
}

void KomparatorWidget::slotCopyNext( KIO::Job *finished_job )
{
	if ( finished_job->error() == 0 )
	{
		KIO::StatJob *job = KIO::stat( ((KIO::FileCopyJob *)finished_job)->destURL(), false/*progress*/ );
		connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotCopyNextStat( KIO::Job * ) ) );
	}
	else
	{
		finished_job->showErrorDialog( this );
		slotCopyNextStat( NULL ); // failed to copy item; call stat with NULL means: remove from list but don't do postprocessing
	}
}

void KomparatorWidget::slotCopyNextStat( KIO::Job *finished_job )
{
	if ( finished_job )
	{
		KIO::UDSEntry entry;
		if ( finished_job->error() == 0 )
		{
			// create a new item that holds the file that was just copied. 
			entry = ((KIO::StatJob *)finished_job)->statResult();
		}
		else
		{
			finished_job->showErrorDialog( this );
		}
		KFileItemExt *newitem = new KFileItemExt( entry, ((KIO::StatJob *)finished_job)->url() );
		newitem->dir = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? 1 : 0;
		newitem->virtual_parent_path = newitem->dir ? m_url2 : m_url1;
		newitem->parent_path = m_virtual_subdirectory_current.isEmpty() ? newitem->virtual_parent_path : m_virtual_subdirectory_current.upURL();

		newitem->next = m_files; // Add it to m_files to delete if necessary
		m_files = newitem;

		newitem->isdupe_size = 1;
		newitem->isdupe_path = 1;

		if ( !((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->isdupe_size &&
				!((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->hasdupes_size &&
				( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->size() > 0 ||
					!ignoreEmptyFiles_CheckBox->isChecked() ) ) // no duplicate, copying created a new duplicate
		{
			newitem->duplicates_size = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size;
			if ( newitem->duplicates_size ) newitem->hasdupes_size = 1;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size = newitem;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->hasdupes_size = 1;
			newitem->dup_size_parent = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item;
			if ( isEnabled( DUPLICATES ) )
			{
				(new KListViewItemDups( duplicates1_KListView, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item ))->setUseColor( textColor_CheckBox->isChecked() );
				emit( signalCount( DUPLICATES, 1, -1 ) );
			}
			// We don't need to check whether this is a "real duplicate" as copying created a duplicate that has the same sub-path.
		}
		else // is already a duplicate, append copied file. must delete overwritten file from duplicate chain.
		{
			KFileItemExt *curfile = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size = newitem;
			newitem->dup_size_parent = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->duplicates_size->duplicates_size = curfile;
			if ( curfile ) curfile->dup_size_parent = newitem;
		}

		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother ) // This means, the brother file was overwritten.
		{
			if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->dup_size_parent ) // Not the first in duplicate chain.
			{
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->isdupe_size = 0;
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->hasdupes_size = 0;
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->dup_size_parent->duplicates_size =
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size;
				if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size )
				{
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size->dup_size_parent =
						((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->dup_size_parent;
				}
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size = NULL;
			}
			else                                                                                    // First in duplicate chain.
			{
				if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size &&
						( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->size() > 0 ||
							!ignoreEmptyFiles_CheckBox->isChecked() ) )
				{
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size->dup_size_parent = NULL;
					if ( isEnabled( DUPLICATES ) )
					{
						(new KListViewItemDups( duplicates1_KListView, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() );
						emit( signalCount( DUPLICATES, 1, -1 ) );
					}
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->duplicates_size = NULL;
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->hasdupes_size = 0;
					((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item->isdupe_size = 0;
				}
			}

			QListViewItemIterator it2( duplicates1_KListView );
			while( it2.current() )
			{
				if ( !((KListViewItemDups*)it2.current())->item->duplicates_size )
				{
					delete ((KListViewItemDups*)it2.current());
				}

				((KListViewItemDups*)it2.current())->updateColor();

				it2++;
			}
		}

		if ( findEmptyDirectories_CheckBox->isChecked() ) // search previously empty directories, whether they are still empty.
		{
			// The most simple approach: check, whether any empty directory is a parent directory of our copied file. Delete this directory.
			// There can only be ONE empty directory that's a parent of our file.
			KFileItemExt *tmp_dir = m_dirs_dupes;
			while ( tmp_dir )
			{
				if ( tmp_dir->url().isParentOf( newitem->url() ) )
				{
					if ( tmp_dir->dup_size_parent ) // not first in duplicate chain, simply skip it.
					{
						tmp_dir->dup_size_parent->duplicates_size = tmp_dir->duplicates_size;
						tmp_dir->hasdupes_size = 0;

						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->dup_size_parent->hasdupes_size = 1;
							tmp_dir->duplicates_size->dup_size_parent = tmp_dir->dup_size_parent;
						}

						delete tmp_dir;
					}
					else // first in duplicate chain
					{
						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->duplicates_size->dup_size_parent = NULL;
							if ( isEnabled( DUPLICATES ) )
							{
								(new KListViewItemDups( duplicates1_KListView, tmp_dir->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() );
								emit( signalCount( DUPLICATES, 1, -1 ) );
							}
						}

						QListViewItemIterator it( duplicates1_KListView );
						while( it.current() )
						{
							if ( ((KListViewItemDups*)it.current())->item->url() == tmp_dir->url() )
							{
								delete ((KListViewItemDups*)it.current());
								break;
							}

							it++;
						}
						m_dirs_dupes = tmp_dir->duplicates_size;

						delete tmp_dir;
					}

					break;
				}

				tmp_dir = tmp_dir->duplicates_size;
			}
		}

		delete ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother; // remove the brother from the other list as it was overwritten.

		slotDuplicateSelected(); // We eventually changed something here. Reselect to make sure, new duplicates appear on duplicates2 immediately.

		m_interaction_copy_remaining_items.setAutoDelete( true );
		m_interaction_copy_remaining_items.removeLast();
		m_interaction_copy_remaining_items.setAutoDelete( false );
	}
	else
	{
		m_interaction_copy_remaining_items.removeLast(); // don't delete as item was not copied
	}

	if ( m_interaction_copy_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		bool overwrite = ( current_list_view == newer1_KListView || current_list_view == newer2_KListView ) ? true : false;

		KURL src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
		KURL dest;

		m_virtual_subdirectory_current = KURL( "" );
		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 )
		{
			dest = m_url2;
			dest.addPath( prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );

			KURL::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KURL::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upURL().prettyURL() + "/" + prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upURL();
					dest.addPath( prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) + "/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = m_url1;
			dest.addPath( prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
		}

		src.cleanPath();
		dest.cleanPath();

		KIO::FileCopyJob *job = KIO::file_copy( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
			overwrite, false /*resume*/, false /*progress*/ );
		connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotCopyNext( KIO::Job * ) ) );
	}
	else
	{
		m_interaction_copy_remaining_items.clear();
		updateFilters();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Copying files finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIconSet( SmallIcon( "clear_left" ) );
	}
}

void KomparatorWidget::slotMoveToOtherSide()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_interaction_empty_dirs.clear();

	m_interaction_cancel = false;
	m_status = INTERACTING;
	search_KPushButton->setText( i18n( "Cancel" ) );
	search_KPushButton->setIconSet( SmallIcon( "cancel" ) );

	m_interaction_copy_remaining_items = current_list_view->selectedItems();
	m_interaction_total = m_interaction_copy_remaining_items.count();
	m_interaction_progress = 0;
	m_interaction_progress_str = ( m_interaction_total == 1 ) ?
		i18n( "Moving 1 file from one side to the other..." ) :
		i18n( "Moving %1 files from one side to the other..." ).arg( m_interaction_total );

	enable( false );

	emit( signalProgress( m_interaction_progress_str, -1, "" ) );
	emit( signalStartSystrayMovie() );

	QString destination = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? // destination is right side
		m_url2.url() : m_url1.url();

	bool overwrite = false; // doesn't make sense to overwrite if missing (then the target file shouldn't exist), but should overwrite if newer.
	KListViewItemSingle *i;
	if ( current_list_view == newer1_KListView || current_list_view == newer2_KListView )
	{
		QStringList move;
		for ( i=(KListViewItemSingle *)m_interaction_copy_remaining_items.first(); i; i=(KListViewItemSingle *)m_interaction_copy_remaining_items.next() )
		{
			move.push_back( prettyURL( i->item->url(), URL_DIR_OR_URL, KURL( "" ) ) + " " + i18n( "to" ) + " " + prettyURL( i->brother->item->url(), URL_DIR_OR_URL, KURL( "" ) ) );
		}

		QString msg;
		msg = ( m_interaction_total == 1 ) ?
			i18n( "Do you really want to move the selected file overwriting the existing one?" ) :
			i18n( "Do you really want to move the %1 selected files overwriting the existing ones?" ).arg( m_interaction_total );

		KGlobal::config()->reparseConfiguration();
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
		if ( KMessageBox::warningContinueCancelList( parentWidget(), msg, move, i18n( "Move files" ),
			KGuiItem( i18n( "Overwrite" ), "editoverwrite"), ":komparator_message_overwrite_move" ) == KMessageBox::Cancel )
		{
			QApplication::restoreOverrideCursor();
			m_interaction_copy_remaining_items.clear();
			slotMoveNextStat( NULL );
			return;
		}
		QApplication::restoreOverrideCursor();

		m_config_global->reparseConfiguration();
		bool warning_overwrite_move = !m_config_global->hasKey( ":komparator_message_overwrite_move" ) |
			m_config_global->readBoolEntry( ":komparator_message_overwrite_move" );
		warningOverwriteMove_CheckBox->setChecked( warning_overwrite_move );

		overwrite = true;
	}

	bool success;
	KURL dest;
	int last_items_dir = ((KListViewItemSingle *)m_interaction_copy_remaining_items.last())->item->dir;
	for ( i=(KListViewItemSingle *)m_interaction_copy_remaining_items.first(); i; i=(KListViewItemSingle *)m_interaction_copy_remaining_items.next() )
	{
		m_virtual_subdirectory_current = KURL( "" );
		if ( last_items_dir == 0 )
		{
			dest = KURL::fromPathOrURL( destination );
			dest.addPath( prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );

			KURL::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KURL::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upURL().prettyURL() + "/" + prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upURL();
					dest.addPath( prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = KURL::fromPathOrURL( destination );
			dest.addPath( prettyURL( i->item->url(), URL_REPLACE, i->item->parent_path ) + "/" + i->item->url().fileName() );
		}
		dest.cleanPath();

		KURL dir = dest.upURL();

		while ( !KIO::NetAccess::mkdir( dir, this ) ) // create all destination subdirs if not present
		{
			KURL mkdir = dir.upURL();
			success = true;

			while ( !KIO::NetAccess::mkdir( mkdir, this ) )
			{
				if ( mkdir == mkdir.upURL() )
				{
					success = false;
					break;
				}

				mkdir = mkdir.upURL();
			}

			if ( !success ) break;
		}
	}

	KURL src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
	src.cleanPath();

	KIO::FileCopyJob *job = KIO::file_move( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
		overwrite, false /*resume*/, false/*progress*/ );
	connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotMoveNext( KIO::Job * ) ) );
}

void KomparatorWidget::slotMoveNext( KIO::Job *finished_job )
{
	if ( finished_job->error() == 0 )
	{
		KIO::StatJob *job = KIO::stat( ((KIO::FileCopyJob *)finished_job)->destURL(), false/*progress*/ );
		connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotMoveNextStat( KIO::Job * ) ) );
	}
	else
	{
		finished_job->showErrorDialog( this );
		slotMoveNextStat( NULL ); // failed to copy item; call stat with NULL means: remove from list but don't do postprocessing
	}
}

void KomparatorWidget::slotMoveNextStat( KIO::Job *finished_job )
{
	if ( finished_job )
	{
		KIO::UDSEntry entry;
		if ( finished_job->error() == 0 )
		{
			// create a new item that holds the file that was just copied. 
			entry = ((KIO::StatJob *)finished_job)->statResult();
		}
		else
		{
			finished_job->showErrorDialog( this );
		}

		if ( findEmptyDirectories_CheckBox->isChecked() ) // search previously empty directories, whether they are still empty.
		{
			// The most simple approach: check, whether any empty directory is a parent directory of our copied file. Delete this directory.
			// There can only be ONE empty directory that's a parent of our file.
			KFileItemExt *tmp_dir = m_dirs_dupes;
			while ( tmp_dir )
			{
				if ( tmp_dir->url().isParentOf( ((KIO::StatJob *)finished_job)->url() ) )
				{
					if ( tmp_dir->dup_size_parent ) // not first in duplicate chain, simply skip it.
					{
						tmp_dir->dup_size_parent->duplicates_size = tmp_dir->duplicates_size;
						tmp_dir->hasdupes_size = 0;

						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->dup_size_parent->hasdupes_size = 1;
							tmp_dir->duplicates_size->dup_size_parent = tmp_dir->dup_size_parent;
						}

						delete tmp_dir;
					}
					else // first in duplicate chain
					{
						if ( tmp_dir->duplicates_size )
						{
							tmp_dir->duplicates_size->dup_size_parent = NULL;
							if ( isEnabled( DUPLICATES ) )
							{
								(new KListViewItemDups( duplicates1_KListView, tmp_dir->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() );
								emit( signalCount( DUPLICATES, 1, -1 ) );
							}
						}

						QListViewItemIterator it( duplicates1_KListView );
						while( it.current() )
						{
							if ( ((KListViewItemDups*)it.current())->item->url() == tmp_dir->url() )
							{
								delete ((KListViewItemDups*)it.current());
								break;
							}
							it++;
						}

						m_dirs_dupes = tmp_dir->duplicates_size;

						delete tmp_dir;
					}
				}

				tmp_dir = tmp_dir->duplicates_size;
			}

			m_interaction_empty_dirs.push_back( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url() );
		}

		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother ) // Removes the overwritten from all lists, especially duplicates.
		{
			removeDeletedFromLists( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother->item, true /*duplicates_only*/, false /* add brother to missing tab */ );
			delete ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother;
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->brother = NULL;
		}

		// Update the moved file to have the new path (and new entry, and new dir).
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->setUDSEntry( entry, ((KIO::StatJob *)finished_job)->url() );

		// The dir entry is not yet ok, it represents the old directory (before move).
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir =
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir ? 0 : 1;
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->virtual_parent_path =
			((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir ? m_url2 : m_url1;
		((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path =
			m_virtual_subdirectory_current.isEmpty() ? ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->virtual_parent_path : m_virtual_subdirectory_current.upURL();

		if ( isEnabled( MISSING ) ) // no file was overwritten, the only change must be done to missing list views.
		{
			KListViewItemSingle *new_item_single = new KListViewItemSingle(
				( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 ) ?
				missing1_KListView : missing2_KListView,
				((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item, NULL );
			new_item_single->setText( 1, prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(),
				URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) );
			emit( signalCount( MISSING, ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 ) ? 1 : 2, -1 ) );
		}

		QListViewItemIterator it2( duplicates1_KListView ); // replace path of duplicate1 items by the new path.
		while( it2.current() )
		{
			KURL tmp_url = ((KListViewItemDups*)it2.current())->item->url(); // This is necessary to display a pretty url on non-local file systems.
			tmp_url.setPath( "" );
			((KListViewItemDups*)it2.current())->setText( 0, tmp_url.pathOrURL() + ((KListViewItemDups*)it2.current())->item->url().upURL().path( -1 ) );

			it2++;
		}

		slotDuplicateSelected(); // We eventually changed something here. Reselect to make sure, new duplicates appear on duplicates2 immediately.

		m_interaction_copy_remaining_items.setAutoDelete( true );
		m_interaction_copy_remaining_items.removeLast();
		m_interaction_copy_remaining_items.setAutoDelete( false );
	}
	else
	{
		m_interaction_copy_remaining_items.removeLast(); // don't delete as item was not copied
	}

	if ( m_interaction_copy_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		QString destination = ( current_list_view == missing1_KListView || current_list_view == newer1_KListView ) ? // destination is right side
			m_url2.url() : m_url1.url();

		bool overwrite = ( current_list_view == newer1_KListView || current_list_view == newer2_KListView ) ? true : false;

		KURL src = ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url();
		KURL dest;

		m_virtual_subdirectory_current = KURL( "" );
		if ( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->dir == 0 )
		{
			dest = KURL::fromPathOrURL( destination );
			dest.addPath( prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );

			KURL::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
			KURL::List::iterator it;
			for ( it=virtual_subdirectories.begin(); it!=virtual_subdirectories.end(); ++it )
			{
				if ( (*it).isParentOf( (*it).upURL().prettyURL() + "/" + prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) ) )
				{
					m_virtual_subdirectory_current = *it;
					dest = (*it).upURL();
					dest.addPath( prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) + "/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
				}
			}
		}
		else
		{
			dest = KURL::fromPathOrURL( destination );
			dest.addPath( prettyURL( ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url(), URL_REPLACE, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->parent_path ) +
				"/" + ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->url().fileName() );
		}

		src.cleanPath();
		dest.cleanPath();

		KIO::FileCopyJob *job = KIO::file_move( src, dest, ((KListViewItemSingle*)m_interaction_copy_remaining_items.last())->item->permissions(),
			overwrite, false/*resume*/, false/*progress*/ );
		connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotMoveNext( KIO::Job * ) ) );
	}
	else
	{
		checkEmptyDirs( m_interaction_empty_dirs );

		m_interaction_empty_dirs.clear();
		m_interaction_copy_remaining_items.clear();
		updateFilters();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Moving files finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIconSet( SmallIcon( "clear_left" ) );
	}
}

void KomparatorWidget::slotOpenBinding()
{
	if ( current_list_view )
	{
		((KListViewItemSingle*) current_list_view->currentItem())->item->run();
	}

	current_list_view = NULL;
}

void KomparatorWidget::slotExecute( QListViewItem *item )
{
	((KListViewItemSingle*) item)->item->run();
}

void KomparatorWidget::slotOpenDirectory()
{
	if ( !current_list_view ) return;

	if ( ((KListViewItemSingle*) current_list_view->currentItem())->item->isDir() )
	{
		new KRun( ((KListViewItemSingle*) current_list_view->currentItem())->item->url() );
		return;
	}

	KURL url = ((KListViewItemSingle*) current_list_view->currentItem())->item->url();
	url.setFileName( QString::null );

	new KRun( url );

	current_list_view = NULL;
}

void KomparatorWidget::slotCopySelection()
{
	if ( !current_list_view ) return;

	// Here we do the same as in QDragObject *KListView::dragObject().
	// this function is protected and therefore not usable.
	KURL::List urls;

	QPixmap *pm = NULL;

	QListViewItemIterator it( current_list_view );
	while ( it.current() )
	{
		if ( it.current()->isSelected() )
		{
			urls.append( ((KListViewItemSingle*)it.current())->item->url().url() );
			if ( !pm )
			{
				pm = new QPixmap( ((KListViewItemSingle*)it.current())->item->pixmap(32) );
			}
		}
		it++;
	}

	KURLDrag *ret = new KURLDrag( urls, current_list_view );

	if ( pm && !pm->isNull() ) ret->setPixmap( *pm );

	if ( ret )
	{
		QClipboard *cb = kapp->clipboard();
		cb->setData( ret );
	}

	current_list_view = NULL;
}

void KomparatorWidget::slotTrashFiles()
{
	if ( !current_list_view ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = TRASH;

	m_interaction_empty_dirs.clear();
	m_interaction_remove_remaining_items.clear();

	QPtrList< QListViewItem > tmp_selected = current_list_view->selectedItems();
	QListViewItem *i;
	for ( i=tmp_selected.first(); i; i=tmp_selected.next() )
	{
		m_interaction_remove_remaining_items.append( ( current_list_view == duplicates1_KListView ) ?
			((KListViewItemDups *) i)->item :
			((KListViewItemSingle *) i)->item );
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::slotDeleteFiles()
{
	if ( !current_list_view ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = DELETE;

	m_interaction_empty_dirs.clear();
	m_interaction_remove_remaining_items.clear();

	QPtrList< QListViewItem > tmp_selected = current_list_view->selectedItems();
	QListViewItem *i;
	for ( i=tmp_selected.first(); i; i=tmp_selected.next() )
	{
		m_interaction_remove_remaining_items.append( ( current_list_view == duplicates1_KListView ) ?
			((KListViewItemDups *) i)->item :
			((KListViewItemSingle *) i)->item );
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::slotTrashNonreferenceFiles()
{
	if ( !current_list_view || current_list_view != duplicates1_KListView ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = TRASH_COREF;

	m_interaction_empty_dirs.clear();
	m_interaction_remove_remaining_items.clear();

	KFileItemExt *tmp_item;
	QPtrList< QListViewItem > tmp_selected = current_list_view->selectedItems();
	QListViewItem *i;
	for ( i=tmp_selected.first(); i; i=tmp_selected.next() )
	{
		tmp_item = ((KListViewItemDups*) i)->item;
		if ( tmp_item->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
		{
			m_interaction_remove_remaining_items.append( tmp_item );
		}
		while ( tmp_item->duplicates_size )
		{
			if ( tmp_item->duplicates_size->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
			{
				m_interaction_remove_remaining_items.append( tmp_item->duplicates_size );
			}
			tmp_item = tmp_item->duplicates_size;
		}
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::slotDeleteNonreferenceFiles()
{
	if ( !current_list_view || current_list_view != duplicates1_KListView ) return;

	if ( current_list_view->selectedItems().count() == 0 ) return;

	m_file_remove_action = DELETE_COREF;

	m_interaction_empty_dirs.clear();
	m_interaction_remove_remaining_items.clear();

	KFileItemExt *tmp_item;
	QPtrList< QListViewItem > tmp_selected = current_list_view->selectedItems();
	QListViewItem *i;
	for ( i=tmp_selected.first(); i; i=tmp_selected.next() )
	{
		tmp_item = ((KListViewItemDups*) i)->item;
		if ( tmp_item->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
		{
			m_interaction_remove_remaining_items.append( tmp_item );
		}
		while ( tmp_item->duplicates_size )
		{
			if ( tmp_item->duplicates_size->dir != ( duplicatesReference1_RadioButton->isChecked() ? 0 : 1 ) )
			{
				m_interaction_remove_remaining_items.append( tmp_item->duplicates_size );
			}
			tmp_item = tmp_item->duplicates_size;
		}
	}
	m_interaction_total = m_interaction_remove_remaining_items.count();

	removeFirst();
}

void KomparatorWidget::removeFirst()
{
	if ( m_interaction_total == 0 ) return;

	m_interaction_cancel = false;
	m_status = INTERACTING;
	search_KPushButton->setText( i18n( "Cancel" ) );
	search_KPushButton->setIconSet( SmallIcon( "cancel" ) );

	m_interaction_progress = 0;
	m_interaction_progress_str = ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
		( m_interaction_total == 1 ) ? i18n( "Trashing 1 file..." ) : i18n( "Trashing %1 files..." ).arg( m_interaction_total ) :
		( m_interaction_total == 1 ) ? i18n( "Deleting 1 file..." ) : i18n( "Deleting %1 files..." ).arg( m_interaction_total );

	enable( false );

	emit( signalProgress( m_interaction_progress_str, -1, "" ) );
	emit( signalStartSystrayMovie() );

	QStringList del;
	KFileItemExt *i;
	for ( i=m_interaction_remove_remaining_items.first(); i; i=m_interaction_remove_remaining_items.next() )
	{
		del.push_back( prettyURL( i->url(), URL_DIR_OR_URL, KURL( "" ) ) );
	}

	QString msg = ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
		( m_interaction_total == 1 ) ?
			i18n( "Do you really want to throw the selected file away?" ) :
			i18n( "Do you really want to throw the %1 selected files away?" ).arg( m_interaction_total ) :
		( m_interaction_total == 1 ) ?
			i18n( "Do you really want to delete the selected file?" ) :
			i18n( "Do you really want to delete the %1 selected files?" ).arg( m_interaction_total );

	KGlobal::config()->reparseConfiguration();
	QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
	if ( KMessageBox::warningContinueCancelList( parentWidget(), msg, del,
			( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ? i18n( "Trash files" ) : i18n( "Delete files" ),
			( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
				KGuiItem( i18n( "Trash" ), "edittrash") : KGuiItem( i18n( "Delete" ), "editdelete" ),
				( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ? ":komparator_message_trash" : ":komparator_message_delete" )
					== KMessageBox::Cancel )
	{
		QApplication::restoreOverrideCursor();
		m_interaction_remove_remaining_items.clear();
		slotRemoveNext( NULL );
		return;
	}
	QApplication::restoreOverrideCursor();

	if ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF )
	{
		m_config_global->reparseConfiguration();
		bool warning_trash = !m_config_global->hasKey( ":komparator_message_trash" ) |
			m_config_global->readBoolEntry( ":komparator_message_trash" );
		warningTrash_CheckBox->setChecked( warning_trash );
	}
	else
	{
		m_config_global->reparseConfiguration();
		bool warning_delete = !m_config_global->hasKey( ":komparator_message_delete" ) |
			m_config_global->readBoolEntry( ":komparator_message_delete" );
		warningDelete_CheckBox->setChecked( warning_delete );
	}

	if ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF )
	{
		KIO::CopyJob *job = KIO::trash( m_interaction_remove_remaining_items.last()->url(), false );
		connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotRemoveNext( KIO::Job * ) ) );
	}
	else if ( m_file_remove_action == DELETE || m_file_remove_action == DELETE_COREF )
	{
		KIO::DeleteJob *job = KIO::del( m_interaction_remove_remaining_items.last()->url(), false, false );
		connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotRemoveNext( KIO::Job * ) ) );
	}
	else
	{
		kdDebug() << "Invalid file remove action. This point should never be reached. Please report this to the author." << endl;
	}
}

void KomparatorWidget::slotRemoveNext( KIO::Job *finished_job )
{
	if ( finished_job )
	{
		if ( finished_job->error() == 0 )
		{
			if ( findEmptyDirectories_CheckBox->isChecked() )
			{
				m_interaction_empty_dirs.push_back( ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ) ?
					((KIO::CopyJob *)finished_job)->srcURLs().first() : // append only first item, as we delete ONE
					((KIO::DeleteJob *)finished_job)->urls().first() ); //   item at a time.
			}

			removeDeletedFromLists( m_interaction_remove_remaining_items.last() );
		}
		else
		{
			finished_job->showErrorDialog( this );
		}
	}

	// remove item from list; "removeDeletedFromLists" has already deleted the listviewitem.
	m_interaction_remove_remaining_items.removeLast();

	if ( m_interaction_remove_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		if ( m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF )
		{
			KIO::CopyJob *job = KIO::trash( m_interaction_remove_remaining_items.last()->url(), false );
			connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotRemoveNext( KIO::Job * ) ) );
		}
		else if ( m_file_remove_action == DELETE || m_file_remove_action == DELETE_COREF )
		{
			KIO::DeleteJob *job = KIO::del( m_interaction_remove_remaining_items.last()->url(), false, false );
			connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotRemoveNext( KIO::Job * ) ) );
		}
		else
		{
			kdDebug() << "Invalid file remove action. This point should never be reached. Please report this to the author." << endl;
		}
	}
	else // canceled or no items remaining
	{
		checkEmptyDirs( m_interaction_empty_dirs );
		m_interaction_empty_dirs.clear();
		updateFilters();

		m_interaction_remove_remaining_items.clear();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1,
			m_file_remove_action == TRASH || m_file_remove_action == TRASH_COREF ?
			i18n( "Trashing files finished." ) :
			i18n( "Deleting files finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIconSet( SmallIcon( "clear_left" ) );
	}
}

void KomparatorWidget::removeDeletedFromLists( KFileItemExt *item, bool duplicates_only, bool add_brother_missing )
{
	bool created_missing = false;
	KListViewItemSingle *new_item_single;

	QPtrList< KListViewItemSingle > to_delete_single;
	if ( !duplicates_only )
	{
		if ( !item->isDir() )
		{
			// remove from missing list.
			if ( item->dir == 0 ) // left
			{
				QListViewItemIterator it( missing1_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle*)it.current())->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)it.current() );
					}
					it++;
				}
			}
			else                  // right
			{
				QListViewItemIterator it( missing2_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle*)it.current())->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)it.current() );
					}
					it++;
				}
			}

			// remove from newer list.
			if ( item->dir == 0 ) // left
			{
				QListViewItemIterator it2( newer1_KListView );
				while( it2.current() )
				{
					if ( ((KListViewItemSingle*)it2.current())->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)it2.current() );    // Remove the file from the one side

						if ( isEnabled( MISSING ) )                                        // Append the other file to the other missing side if user wants to.
						{
							new_item_single = new KListViewItemSingle( missing2_KListView, ((KListViewItemSingle*)it2.current())->brother->item, NULL );
							new_item_single->setText( 1, prettyURL( ((KListViewItemSingle*)it2.current())->brother->item->url(), URL_REPLACE, ((KListViewItemSingle*)it2.current())->brother->item->parent_path ) );
							emit( signalCount( MISSING, 2, -1 ) );
						}

						created_missing = true;
						to_delete_single.append( ((KListViewItemSingle*)it2.current())->brother ); // Remove the other file (newer or older).
					}
					it2++;
				}
			}
			else                  // right
			{
				QListViewItemIterator it2( newer2_KListView );
				while( it2.current() )
				{
					if ( ((KListViewItemSingle*)it2.current())->item == item )
					{
						to_delete_single.append( (KListViewItemSingle*)it2.current() );    // Remove the file from the one side

						if ( isEnabled( MISSING ) )                                        // Append the other file to the other missing side if user wants to.
						{
							new_item_single = new KListViewItemSingle( missing1_KListView, ((KListViewItemSingle*)it2.current())->brother->item, NULL );
							new_item_single->setText( 1, prettyURL( ((KListViewItemSingle*)it2.current())->brother->item->url(), URL_REPLACE, ((KListViewItemSingle*)it2.current())->brother->item->parent_path ) );
							emit( signalCount( MISSING, 1, -1 ) );
						}

						created_missing = true;
						to_delete_single.append( ((KListViewItemSingle*)it2.current())->brother ); // Remove the other file (newer or older).
					}
					it2++;
				}
			}
		}
	}

	item->isdupe_size = 0;  // Make sure the item is removed from duplicates1_KListView, if necessary.
	item->hasdupes_size = 0;

	item->isdupe_path = 0;  // Make sure the item's brother is removed from duplicates1_KListView, if necessary.
	item->hasdupes_path = 0;

	if ( !created_missing )
	{
		QString item_str, cmp_str;
		item_str = prettyURL( item->url(), URL_REPLACE, item->parent_path ) + item->url().fileName();

		KFileItemExt *curfile = item;

		while( curfile->dup_size_parent ) curfile = curfile->dup_size_parent; // Go to the first in duplicate chain.

		while ( curfile )
		{
			if ( curfile->url() != item->url() ) // We don't want to add the file that we want to delete.
			{
				cmp_str = prettyURL( curfile->url(), URL_REPLACE, curfile->parent_path ) + curfile->url().fileName();

				if ( !qstrcmp( item_str, cmp_str ) )
				{
					if ( curfile->dir == 0 )
					{
						if ( isEnabled( MISSING ) && add_brother_missing )
						{
							new_item_single = new KListViewItemSingle( missing1_KListView, curfile, NULL );
							new_item_single->setText( 1, prettyURL( curfile->url(), URL_REPLACE, curfile->parent_path ) );
							emit( signalCount( MISSING, 1, -1 ) );
						}
					}
					else
					{
						if ( isEnabled( MISSING ) && add_brother_missing )
						{
							new_item_single = new KListViewItemSingle( missing2_KListView, curfile, NULL );
							new_item_single->setText( 1, prettyURL( curfile->url(), URL_REPLACE, curfile->parent_path ) );
							emit( signalCount( MISSING, 2, -1 ) );
						}
					}
				}
			}

			curfile = curfile->duplicates_size;
		}
	}

	QListViewItemIterator it3( duplicates2_KListView );
	while( it3.current() )
	{
		if ( ((KListViewItemSingle*)it3.current())->item == item )
		{
			to_delete_single.append( (KListViewItemSingle*)it3.current() );
		}
		it3++;
	}

	// skip this file in duplicate chain
	if ( item->dup_size_parent ) // not the first in duplicate chain.
	{
		KFileItemExt *curfile = item->dup_size_parent;
		curfile->duplicates_size = item->duplicates_size;

		if ( !curfile->dup_size_parent && !curfile->duplicates_size ) // a single file. no duplicate.
		{
			curfile->hasdupes_size = 0;
			curfile->isdupe_size = 0;
		}
	}
	else // first in duplicate chain
	{
		if ( item->duplicates_size )
		{
			item->duplicates_size->dup_size_parent = NULL;
			if ( item->duplicates_size->duplicates_size )
			{
				item->duplicates_size->hasdupes_size = 1;
				item->duplicates_size->isdupe_size = 1;
			}
			else
			{
				item->duplicates_size->hasdupes_size = 0;
				item->duplicates_size->isdupe_size = 0;
			}
			if ( isEnabled( DUPLICATES ) )
			{
				(new KListViewItemDups( duplicates1_KListView, item->duplicates_size ))->setUseColor( textColor_CheckBox->isChecked() ); // We might have the same file in two items!
				emit( signalCount( DUPLICATES, 1, -1 ) );
			}
		}
		else // Last item in duplicate chain. only for directories.
		{
			if ( item == m_dirs_dupes )
			{
				delete m_dirs_dupes;
				m_dirs_dupes = NULL;
			}
		}
	}

	QPtrListIterator< KListViewItemSingle > it4( to_delete_single ); // really remove missing / newer / duplicates2
	while( it4.current() )
	{
		delete it4.current();

		++it4;
	}

	QPtrList< KListViewItemDups > to_delete_dups; // Delete items that are invalid.
	QListViewItemIterator it5( duplicates1_KListView );
	while( it5.current() )
	{
		if ( ((KListViewItemDups*)it5.current())->item->isdupe_size == 0 && ((KListViewItemDups*)it5.current())->item->hasdupes_size == 0 && !((KListViewItemDups*)it5.current())->item->isDir() )
		{
			to_delete_dups.append( ((KListViewItemDups*)it5.current()) );
		}
		else if ( !((KListViewItemDups*)it5.current())->item->duplicates_size && !((KListViewItemDups*)it5.current())->item->isDir() )
		{
			to_delete_dups.append( ((KListViewItemDups*)it5.current()) );
		}
		else if ( ((KListViewItemDups*)it5.current())->item->isDir() && ((KListViewItemDups*)it5.current())->item == item )
		{
			to_delete_dups.append( ((KListViewItemDups*)it5.current()) );
		}
		else // we don't want to delete this duplicate list, but check, whether it's a duplicate of two identical sub-paths or not.
		{
			((KListViewItemDups*)it5.current())->updateColor();
		}

		it5++;
	}

	QPtrListIterator< KListViewItemDups > it6( to_delete_dups ); // really remove duplicates1
	while( it6.current() )
	{
		delete it6.current();

		++it6;
	}

	slotDuplicateSelected();
}

void KomparatorWidget::checkEmptyDirs( KURL::List &urls ) // if a file was deleted, the directory might be empty now.
{
	m_empty_directory_job->initialize( &urls );
	m_empty_directory_job->start();
}

void KomparatorWidget::slotOpenWith()
{
	if ( current_list_view )
	{
		KRun::displayOpenWithDialog( KURL::split( ( current_list_view == duplicates1_KListView ) ?
			((KListViewItemDups*) current_list_view->currentItem())->item->url() :
			((KListViewItemSingle*) current_list_view->currentItem())->item->url() ) );
	}

	current_list_view = NULL;
}

void KomparatorWidget::slotFileProperties()
{
	if ( current_list_view )
	{
		KFileItemList tmp_list;
		QListViewItem *i;
		QPtrList< QListViewItem > tmp_selected = current_list_view->selectedItems();
		for ( i=tmp_selected.first(); i; i=tmp_selected.next() )
		{
			tmp_list.append( ( current_list_view == duplicates1_KListView ) ?
				((KListViewItemDups*) i)->item :
				((KListViewItemSingle*) i)->item );
		}
		KPropertiesDialog *prop_dlg = new KPropertiesDialog( tmp_list, this, "propDialog", true /* modal */, false /* auto-show */ );
		connect( prop_dlg, SIGNAL( applied() ), this, SLOT( slotFilePropertiesChanged() ) );
		connect( prop_dlg, SIGNAL( saveAs( const KURL &, KURL & ) ), this, SLOT( slotFilePropertiesRenamed( const KURL &, KURL & ) ) );
		prop_dlg->show();
	}
}

void KomparatorWidget::slotFilePropertiesRenamed( const KURL &old_url, KURL &new_url )
{
	if ( current_list_view )
	{
		if ( old_url != new_url ) m_properties_renamed = true;
		if ( current_list_view == duplicates1_KListView )
		{
			((KListViewItemDups *) current_list_view->selectedItems().first())->item->setURL( new_url );
		}
		else
		{
			((KListViewItemSingle *) current_list_view->selectedItems().first())->item->setURL( new_url );
		}
	}
}

void KomparatorWidget::slotFilePropertiesChanged()
{
	if ( current_list_view )
	{
		QPtrList< KFileItemExt > tmp_list;
		QListViewItem *i;
		QPtrList< QListViewItem > tmp_selected = current_list_view->selectedItems();
		for ( i=tmp_selected.first(); i; i=tmp_selected.next() )
		{
			tmp_list.append( ( current_list_view == duplicates1_KListView ) ?
				((KListViewItemDups *) i)->item :
				((KListViewItemSingle *) i)->item );
		}

		if ( m_properties_renamed == true )
		{
			m_properties_renamed = false;

			QStringList list;
			list.append( prettyURL( tmp_list.first()->url(), URL_DIR_OR_URL, KURL( "" ) ) );
			KGlobal::config()->reparseConfiguration();
			if ( KMessageBox::warningContinueCancelList( this,
				i18n( "You renamed a file. Komparator needs to re-start the search on the whole tree to display correct results. "
							"If you don't want that, keep in mind that all search results concerning this file are invalid." ),
				list, i18n( "Re-start search" ), KGuiItem( i18n( "Re-start" ), "reload"), ":komparator_message_reload" ) != KMessageBox::Cancel )
			{
				m_config_global->reparseConfiguration();
				bool warning_reload = !m_config_global->hasKey( ":komparator_message_reload" ) |
					m_config_global->readBoolEntry( ":komparator_message_reload" );
				warningReload_CheckBox->setChecked( warning_reload );

				slotSearch(); // Reset.
				slotSearch(); // Start search.
			}
		}
		else
		{
			m_interaction_cancel = false;
			m_status = INTERACTING;
			search_KPushButton->setText( i18n( "Cancel" ) );
			search_KPushButton->setIconSet( SmallIcon( "cancel" ) );
		
			m_interaction_remove_remaining_items = tmp_list;
			m_interaction_total = m_interaction_remove_remaining_items.count();
			m_interaction_progress = 0;
			m_interaction_progress_str = ( m_interaction_total == 1 ) ?
			i18n( "Getting file properties of 1 file..." ) :
			i18n( "Getting file properties of %1 files..." ).arg( m_interaction_total );
		
			enable( false );
		
			emit( signalProgress( m_interaction_progress_str, -1, "" ) );
			emit( signalStartSystrayMovie() );

			KIO::StatJob *job = KIO::stat( m_interaction_remove_remaining_items.last()->url(), false /* progress */ );
			connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotFilePropertiesNextStat( KIO::Job * ) ) );
		}
	}
}

void KomparatorWidget::slotFilePropertiesNextStat( KIO::Job *finished_job )
{
	if ( finished_job )
	{
		if ( finished_job->error() == 0 )
		{
			m_interaction_remove_remaining_items.last()->setUDSEntry( ((KIO::StatJob *)finished_job)->statResult(),
				m_interaction_remove_remaining_items.last()->url() );

			QListViewItemIterator it( duplicates1_KListView );
			while( it.current() )
			{
				if ( ((KListViewItemDups *)it.current())->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemDups *)it.current())->setText( 4, ((KListViewItemDups *)it.current())->item->permissionsString() );
					((KListViewItemDups *)it.current())->setText( 5, ((KListViewItemDups *)it.current())->item->user() );
					((KListViewItemDups *)it.current())->setText( 6, ((KListViewItemDups *)it.current())->item->group() );
				}
				it++;
			}

			QListViewItemIterator it2( duplicates2_KListView );
			while( it2.current() )
			{
				if ( ((KListViewItemSingle *)it2.current())->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)it2.current())->setText( 4, ((KListViewItemSingle *)it2.current())->item->permissionsString() );
					((KListViewItemSingle *)it2.current())->setText( 5, ((KListViewItemSingle *)it2.current())->item->user() );
					((KListViewItemSingle *)it2.current())->setText( 6, ((KListViewItemSingle *)it2.current())->item->group() );
				}
				it2++;
			}

			QListViewItemIterator it3( missing1_KListView );
			while( it3.current() )
			{
				if ( ((KListViewItemSingle *)it3.current())->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)it3.current())->setText( 4, ((KListViewItemSingle *)it3.current())->item->permissionsString() );
					((KListViewItemSingle *)it3.current())->setText( 5, ((KListViewItemSingle *)it3.current())->item->user() );
					((KListViewItemSingle *)it3.current())->setText( 6, ((KListViewItemSingle *)it3.current())->item->group() );
				}
				it3++;
			}

			QListViewItemIterator it4( missing2_KListView );
			while( it4.current() )
			{
				if ( ((KListViewItemSingle *)it4.current())->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)it4.current())->setText( 4, ((KListViewItemSingle *)it4.current())->item->permissionsString() );
					((KListViewItemSingle *)it4.current())->setText( 5, ((KListViewItemSingle *)it4.current())->item->user() );
					((KListViewItemSingle *)it4.current())->setText( 6, ((KListViewItemSingle *)it4.current())->item->group() );
				}
				it4++;
			}

			QListViewItemIterator it5( newer1_KListView );
			while( it5.current() )
			{
				if ( ((KListViewItemSingle *)it5.current())->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)it5.current())->setText( 4, ((KListViewItemSingle *)it5.current())->item->permissionsString() );
					((KListViewItemSingle *)it5.current())->setText( 5, ((KListViewItemSingle *)it5.current())->item->user() );
					((KListViewItemSingle *)it5.current())->setText( 6, ((KListViewItemSingle *)it5.current())->item->group() );
				}
				it5++;
			}

			QListViewItemIterator it6( newer2_KListView );
			while( it6.current() )
			{
				if ( ((KListViewItemSingle *)it6.current())->item == m_interaction_remove_remaining_items.last() )
				{
					((KListViewItemSingle *)it6.current())->setText( 4, ((KListViewItemSingle *)it6.current())->item->permissionsString()  );
					((KListViewItemSingle *)it6.current())->setText( 5, ((KListViewItemSingle *)it6.current())->item->user() );
					((KListViewItemSingle *)it6.current())->setText( 6, ((KListViewItemSingle *)it6.current())->item->group() );
				}
				it6++;
			}
		}
		else
		{
			finished_job->showErrorDialog( this );
		}
	}

	m_interaction_remove_remaining_items.removeLast();

	if ( m_interaction_remove_remaining_items.count() && !m_interaction_cancel )
	{
		m_interaction_progress++;
		emit( signalProgress( m_interaction_progress_str, (int)((m_interaction_progress*100)/m_interaction_total), "" ) );

		KIO::StatJob *job = KIO::stat( m_interaction_remove_remaining_items.last()->url(), false /* progress */ );
		connect( job, SIGNAL( result( KIO::Job * ) ), this, SLOT( slotFilePropertiesNextStat( KIO::Job * ) ) );
	}
	else // canceled or no items remaining
	{
		m_interaction_remove_remaining_items.clear();

		current_list_view = NULL;

		emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Getting file properties finished." ) ) );
		emit( signalStopSystrayMovie() );

		enable( true );

		m_status = READY;
		search_KPushButton->setText( i18n( "Clear" ) );
		search_KPushButton->setIconSet( SmallIcon( "clear_left" ) );
	}
}

void KomparatorWidget::slotCompareEqual()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() != 1 ) return;

	KFileItemExt *item1;
	KFileItemExt *item2;

	if ( current_list_view == newer1_KListView )
	{
		item1 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->item;
		item2 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->brother->item;
	}
	else if ( current_list_view == newer2_KListView )
	{
		item2 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->item;
		item1 = ((KListViewItemSingle *)current_list_view->selectedItems().first())->brother->item;
	}
	else return;

	KompareDialog *dlg = new KompareDialog( this, item1, item2, prettyURL( item1->url(), URL_DIR_OR_URL_ONLY, KURL( "" ) ),
		prettyURL( item2->url(), URL_DIR_OR_URL_ONLY, KURL( "" ) ), calculateChecksum_CheckBox->isChecked() );
	dlg->exec();
	int result = dlg->getResult();

	if ( result != CANCEL )
	{
		if ( result & COPY_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QListViewItemIterator it( newer1_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item1 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QListViewItemIterator it( newer1_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item1 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			slotMoveToOtherSide();
		}
		if ( result & COPY_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QListViewItemIterator it( newer2_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item2 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QListViewItemIterator it( newer2_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item2 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			slotMoveToOtherSide();
		}

		m_interaction_empty_dirs.clear();
		m_interaction_remove_remaining_items.clear();
		
		if ( result & TRASH_LEFT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & DELETE_LEFT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & TRASH_RIGHT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item2 );
		}
		if ( result & DELETE_RIGHT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item2 );
		}

		m_interaction_total = m_interaction_remove_remaining_items.count();
		if ( m_interaction_total ) removeFirst();

		if ( result & DIRTY )
		{
			KMessageBox::information( this, i18n( "You modified a file during comparison. "
				"You need to restart the search to make these changes take effect." ),
				i18n( "Re-start search" ) );
		}
	}
}

void KomparatorWidget::slotCompareSelected()
{
	if ( !current_list_view ) return;
	if ( current_list_view->selectedItems().count() != 1 ) return;

	KFileItemExt *item1;
	KFileItemExt *item2;

	if ( current_list_view == missing1_KListView || current_list_view == missing2_KListView )
	{
		item1 = ((KListViewItemSingle *)missing1_KListView->selectedItems().first())->item;
		item2 = ((KListViewItemSingle *)missing2_KListView->selectedItems().first())->item;
	}
	else if ( current_list_view == newer1_KListView || current_list_view == newer2_KListView )
	{
		item1 = ((KListViewItemSingle *)newer1_KListView->selectedItems().first())->item;
		item2 = ((KListViewItemSingle *)newer2_KListView->selectedItems().first())->item;
	}
	else return;

	KompareDialog *dlg = new KompareDialog( this, item1, item2, prettyURL( item1->url(), URL_DIR_OR_URL_ONLY, KURL( "" ) ),
		prettyURL( item2->url(), URL_DIR_OR_URL_ONLY, KURL( "" ) ), calculateChecksum_CheckBox->isChecked() );
	dlg->exec();
	int result = dlg->getResult();

	if ( result != CANCEL )
	{
		if ( result & COPY_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QListViewItemIterator it( newer1_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item1 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			else if ( current_list_view == missing2_KListView )
			{
				missing1_KListView->clearSelection();
				QListViewItemIterator it( missing1_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item1 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing1_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_LEFT_TO_RIGHT )
		{
			if ( current_list_view == newer2_KListView )
			{
				newer1_KListView->clearSelection();
				QListViewItemIterator it( newer1_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item1 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer1_KListView;
			}
			else if ( current_list_view == missing2_KListView )
			{
				missing1_KListView->clearSelection();
				QListViewItemIterator it( missing1_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item1 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing1_KListView;
			}
			slotMoveToOtherSide();
		}
		if ( result & COPY_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QListViewItemIterator it( newer2_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item2 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			else if ( current_list_view == missing1_KListView )
			{
				missing2_KListView->clearSelection();
				QListViewItemIterator it( missing2_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item2 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing2_KListView;
			}
			slotCopyToOtherSide();
		}
		if ( result & MOVE_RIGHT_TO_LEFT )
		{
			if ( current_list_view == newer1_KListView )
			{
				newer2_KListView->clearSelection();
				QListViewItemIterator it( newer2_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item2 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = newer2_KListView;
			}
			else if ( current_list_view == missing1_KListView )
			{
				missing2_KListView->clearSelection();
				QListViewItemIterator it( missing2_KListView );
				while( it.current() )
				{
					if ( ((KListViewItemSingle *)it.current())->item == item2 )
					{
						it.current()->setSelected( true );
						break;
					}
					++it;
				}
				current_list_view = missing2_KListView;
			}
			slotMoveToOtherSide();
		}

		m_interaction_empty_dirs.clear();
		m_interaction_remove_remaining_items.clear();
		
		if ( result & TRASH_LEFT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & DELETE_LEFT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item1 );
		}
		if ( result & TRASH_RIGHT )
		{
			m_file_remove_action = TRASH;
			m_interaction_remove_remaining_items.append( item2 );
		}
		if ( result & DELETE_RIGHT )
		{
			m_file_remove_action = DELETE;
			m_interaction_remove_remaining_items.append( item2 );
		}

		m_interaction_total = m_interaction_remove_remaining_items.count();
		if ( m_interaction_total ) removeFirst();

		if ( result & DIRTY )
		{
			KMessageBox::information( this, i18n( "You modified a file during comparison. "
				"You need to restart the search to make these changes take effect." ),
				i18n( "Re-start search" ) );
		}
	}
}

void KomparatorWidget::slotCopyMissingL2R()
{
	current_list_view = missing1_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotCopyMissingR2L()
{
	current_list_view = missing2_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotCopyNewerL2R()
{
	current_list_view = newer1_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotCopyNewerR2L()
{
	current_list_view = newer2_KListView;
	slotCopyToOtherSide();
}

void KomparatorWidget::slotMoveMissingL2R()
{
	current_list_view = missing1_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotMoveMissingR2L()
{
	current_list_view = missing2_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotMoveNewerL2R()
{
	current_list_view = newer1_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotMoveNewerR2L()
{
	current_list_view = newer2_KListView;
	slotMoveToOtherSide();
}

void KomparatorWidget::slotRemoveSelectionMissing1()
{
	current_list_view = missing1_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotRemoveSelectionMissing2()
{
	current_list_view = missing2_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotRemoveSelectionNewer1()
{
	current_list_view = newer1_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotRemoveSelectionNewer2()
{
	current_list_view = newer2_KListView;
	slotRemoveFromList();
}

void KomparatorWidget::slotTrashDuplicates()
{
	current_list_view = duplicates2_KListView;
	slotTrashFiles();
}

void KomparatorWidget::slotDeleteDuplicates()
{
	current_list_view = duplicates2_KListView;
	slotDeleteFiles();
}

void KomparatorWidget::slotPresetSelected( int sel )
{
	if ( m_status == READY ) slotSearch(); // cancel

	KomparatorPreset *current_preset = m_preset_list.at( sel );
	if ( current_preset )
	{
		current_preset->setToInterface();
	}
}

void KomparatorWidget::slotAddPreset()
{
	KomparatorPreset *new_preset = new KomparatorPreset( this );
	if ( new_preset->readFromInterface() )
	{
		m_preset_list.append( new_preset );
		preset_KComboBox->insertItem( new_preset->title() );
		preset_KComboBox->setCurrentItem( preset_KComboBox->count()-1 );
		new_preset->saveConfig( preset_KComboBox->count()-1 );
		slotPresetSelected( preset_KComboBox->count()-1 );
	}
	else // user canceled
	{
		delete new_preset;
		new_preset = NULL;
	}
}

void KomparatorWidget::slotUpdatePreset()
{
	KomparatorPreset *current_preset = m_preset_list.at( preset_KComboBox->currentItem() );
	if ( preset_KComboBox->currentItem() != 0 && current_preset ) // don't save to default item
	{
		current_preset->readFromInterface( true ); // reread
	}
}

void KomparatorWidget::slotRemovePreset()
{
	KGlobal::config()->reparseConfiguration();
	QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
	if ( KMessageBox::warningContinueCancel( this, i18n( "Do you really want to remove the preset \"%1\"?" ).arg(
				preset_KComboBox->currentText() ), i18n( "Remove preset" ),
			KGuiItem( i18n( "Remove" ), "editdelete" ), ":komparator_message_remove_preset" ) == KMessageBox::Cancel )
	{
		QApplication::restoreOverrideCursor();
		return;
	}
	QApplication::restoreOverrideCursor();

	m_config_global->reparseConfiguration();
	bool warning_remove_preset = !m_config_global->hasKey( ":komparator_message_remove_preset" ) |
		m_config_global->readBoolEntry( ":komparator_message_remove_preset" );
	warningRemovePreset_CheckBox->setChecked( warning_remove_preset );

	int current_item = preset_KComboBox->currentItem();

	if ( current_item == 0 ) return; // don't delete default

	m_preset_list.remove( current_item ); // will delete preset

	preset_KComboBox->removeItem( current_item );

	preset_KComboBox->setCurrentItem( current_item-1 );
	slotPresetSelected( current_item-1 );
}

void KomparatorWidget::slotSystray()
{
	if ( systray_CheckBox->isChecked() )
		m_systray->show();
	else
		m_systray->hide();
}

void KomparatorWidget::slotTextColor()
{
	QListViewItemIterator it_dups1( duplicates1_KListView );
	QListViewItemIterator it_dups2( duplicates2_KListView );
	QListViewItemIterator it_missing1( missing1_KListView );
	QListViewItemIterator it_missing2( missing2_KListView );
	QListViewItemIterator it_newer1( newer1_KListView );
	QListViewItemIterator it_newer2( newer2_KListView );

	while ( it_dups1.current() )
	{
		((KListViewItemDups*)it_dups1.current())->setUseColor( textColor_CheckBox->isChecked() );
		((KListViewItemDups*)it_dups1.current())->repaint();
		++it_dups1;
	}
	while ( it_dups2.current() )
	{
		((KListViewItemSingle*)it_dups2.current())->setKind( (FILTER)((KListViewItemSingle*)it_dups2.current())->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)it_dups2.current())->color() );
		((KListViewItemSingle*)it_dups2.current())->repaint();
		++it_dups2;
	}
	while ( it_missing1.current() )
	{
		((KListViewItemSingle*)it_missing1.current())->setKind( (FILTER)((KListViewItemSingle*)it_missing1.current())->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)it_missing1.current())->color() );
		((KListViewItemSingle*)it_missing1.current())->repaint();
		++it_missing1;
	}
	while ( it_missing2.current() )
	{
		((KListViewItemSingle*)it_missing2.current())->setKind( (FILTER)((KListViewItemSingle*)it_missing2.current())->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)it_missing2.current())->color() );
		((KListViewItemSingle*)it_missing2.current())->repaint();
		++it_missing2;
	}
	while ( it_newer1.current() )
	{
		((KListViewItemSingle*)it_newer1.current())->setKind( (FILTER)((KListViewItemSingle*)it_newer1.current())->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)it_newer1.current())->color() );
		((KListViewItemSingle*)it_newer1.current())->repaint();
		++it_newer1;
	}
	while ( it_newer2.current() )
	{
		((KListViewItemSingle*)it_newer2.current())->setKind( (FILTER)((KListViewItemSingle*)it_newer2.current())->kind(), textColor_CheckBox->isChecked(), ((KListViewItemSingle*)it_newer2.current())->color() );
		((KListViewItemSingle*)it_newer2.current())->repaint();
		++it_newer2;
	}
}

void KomparatorWidget::slotWarningTrash()
{
	bool warning_trash = warningTrash_CheckBox->isChecked();
	m_config_global->writeEntry( ":komparator_message_trash", warning_trash );
	m_config_global->sync();
}

void KomparatorWidget::slotWarningDelete()
{
	bool warning_delete = warningDelete_CheckBox->isChecked();
	m_config_global->writeEntry( ":komparator_message_delete", warning_delete );
	m_config_global->sync();
}

void KomparatorWidget::slotWarningOverwriteCopy()
{
	bool warning_overwrite_copy = warningOverwriteCopy_CheckBox->isChecked();
	m_config_global->writeEntry( ":komparator_message_overwrite_copy", warning_overwrite_copy );
	m_config_global->sync();
}

void KomparatorWidget::slotWarningOverwriteMove()
{
	bool warning_overwrite_move = warningOverwriteMove_CheckBox->isChecked();
	m_config_global->writeEntry( ":komparator_message_overwrite_move", warning_overwrite_move );
	m_config_global->sync();
}

void KomparatorWidget::slotWarningReload()
{
	bool warning_reload = warningReload_CheckBox->isChecked();
	m_config_global->writeEntry( ":komparator_message_reload", warning_reload );
	m_config_global->sync();
}

void KomparatorWidget::slotWarningRemovePreset()
{
	bool warning_remove_preset = warningRemovePreset_CheckBox->isChecked();
	m_config_global->writeEntry( ":komparator_message_remove_preset", warning_remove_preset );
	m_config_global->sync();
}


/****************************************
* File Filter Functions                 *
****************************************/

void KomparatorWidget::slotFilterChanged( KomparatorFileFilter *filter, const QString &name, const QString &path,
	const QString &owner, const QString &group, const unsigned long &min_size, const unsigned long &max_size,
	const bool &consider_date, const QDate &date, const bool &newer_real, const bool &newer_equal, const bool &newer_same_time )
{
	KListView *tmp_list_view;
	if ( filter == duplicates2_KomparatorFileFilter ) tmp_list_view = duplicates2_KListView; // the next three won't be processed here usually
	else if ( filter == missing1_KomparatorFileFilter ) tmp_list_view = missing1_KListView;
	else if ( filter == missing2_KomparatorFileFilter ) tmp_list_view = missing2_KListView;
	else if ( filter == newer1_KomparatorFileFilter ) tmp_list_view = newer1_KListView;
	else if ( filter == newer2_KomparatorFileFilter ) tmp_list_view = newer2_KListView;
	else
	{
		kdDebug() << "Received file filter event with filter " << filter << ", but this filter doesn't exist!" << endl;
		return;
	}

	int visible_items = 0;
	QListViewItemIterator it( tmp_list_view );
	while( it.current() )
	{
		if ( it.current()->text( 0 ).find( name ) != -1 ) // file name contains str, empty filter (=no filter) returns != -1 always.
		{
			((KListViewItemSingle*)it.current())->removeFilter( NAME ); // Don't filter this one out
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( NAME );
		}

		if ( it.current()->text( 1 ).find( path ) != -1 )
		{
			((KListViewItemSingle*)it.current())->removeFilter( PATH );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( PATH );
		}

		if ( it.current()->text( 5 ).find( owner ) != -1 )
		{
			((KListViewItemSingle*)it.current())->removeFilter( OWNER );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( OWNER );
		}

		if ( it.current()->text( 6 ).find( group ) != -1 )
		{
			((KListViewItemSingle*)it.current())->removeFilter( GROUP );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( GROUP );
		}

		if ( ((KListViewItemSingle*)it.current())->item->size() >= min_size &&
			((KListViewItemSingle*)it.current())->item->size() <= max_size )
		{
			((KListViewItemSingle*)it.current())->removeFilter( SIZE );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( SIZE );
		}

		if ( consider_date )
		{
			QDateTime dt;
			dt.setTime_t( ((KListViewItemSingle*)it.current())->item->time( KIO::UDS_MODIFICATION_TIME ) );
			if ( dt.date() == date )
			{
				((KListViewItemSingle*)it.current())->removeFilter( DATE );
			}
			else
			{
				((KListViewItemSingle*)it.current())->addFilter( DATE );
			}
		}
		else
		{
			((KListViewItemSingle*)it.current())->removeFilter( DATE );
		}

		if ( newer_real )
		{
			((KListViewItemSingle*)it.current())->removeFilter( NEWER_REAL );
		}
		else
		{
			if ( ((KListViewItemSingle*)it.current())->kind() == NEWER_REAL )
			{
				((KListViewItemSingle*)it.current())->addFilter( NEWER_REAL );
			}
		}

		if ( newer_equal )
		{
			((KListViewItemSingle*)it.current())->removeFilter( NEWER_EQUAL );
		}
		else
		{
			if ( ((KListViewItemSingle*)it.current())->kind() == NEWER_EQUAL )
			{
				((KListViewItemSingle*)it.current())->addFilter( NEWER_EQUAL );
			}
		}

		if ( newer_same_time )
		{
			((KListViewItemSingle*)it.current())->removeFilter( NEWER_SAME_TIME );
		}
		else
		{
			if ( ((KListViewItemSingle*)it.current())->kind() == NEWER_SAME_TIME )
			{
				((KListViewItemSingle*)it.current())->addFilter( NEWER_SAME_TIME );
			}
		}

		((KListViewItemSingle*)it.current())->setVisibility();

		if ( it.current()->isVisible() ) visible_items++;

		++it;
	}

	if ( filter == duplicates2_KomparatorFileFilter ) emit( signalCount( DUPLICATES, 2, visible_items ) );
	else if ( filter == missing1_KomparatorFileFilter ) emit( signalCount( MISSING, 1, visible_items ) );
	else if ( filter == missing2_KomparatorFileFilter ) emit( signalCount( MISSING, 2, visible_items ) );
	else if ( filter == newer1_KomparatorFileFilter ) emit( signalCount( NEWER, 1, visible_items ) );
	else if ( filter == newer2_KomparatorFileFilter ) emit( signalCount( NEWER, 2, visible_items ) );

	tmp_list_view = NULL;
}

void KomparatorWidget::slotFilterChanged( KomparatorFileFilter *filter, const QString &name, const QString &path,
			const QString &owner, const QString &group, const unsigned long &min_size, const unsigned long &max_size,
			const bool &consider_date, const QDate &date, const bool &duplicates_normal, const bool &duplicates_multiple )
{
	KListView *tmp_list_view;
	if ( filter == duplicates1_KomparatorFileFilter ) tmp_list_view = duplicates1_KListView;
	else
	{
		kdDebug() << "Received file filter event with filter " << filter << ", but this filter doesn't exist!" << endl;
		return;
	}

	int visible_items = 0;
	QListViewItemIterator it( tmp_list_view );
	while( it.current() )
	{
		if ( it.current()->text( 1 ).find( name ) != -1 ) // file name contains str, empty filter (=no filter) returns != -1 always.
		{
			((KListViewItemDups*)it.current())->removeFilter( NAME ); // Don't filter this one out
		}
		else
		{
			((KListViewItemDups*)it.current())->addFilter( NAME );
		}

		if ( it.current()->text( 0 ).find( path ) != -1 )
		{
			((KListViewItemDups*)it.current())->removeFilter( PATH );
		}
		else
		{
			((KListViewItemDups*)it.current())->addFilter( PATH );
		}

		if ( it.current()->text( 5 ).find( owner ) != -1 )
		{
			((KListViewItemDups*)it.current())->removeFilter( OWNER );
		}
		else
		{
			((KListViewItemDups*)it.current())->addFilter( OWNER );
		}

		if ( it.current()->text( 6 ).find( group ) != -1 )
		{
			((KListViewItemDups*)it.current())->removeFilter( GROUP );
		}
		else
		{
			((KListViewItemDups*)it.current())->addFilter( GROUP );
		}

		if ( ((KListViewItemDups*)it.current())->item->size() >= min_size &&
			((KListViewItemDups*)it.current())->item->size() <= max_size )
		{
			((KListViewItemDups*)it.current())->removeFilter( SIZE );
		}
		else
		{
			((KListViewItemDups*)it.current())->addFilter( SIZE );
		}

		if ( consider_date )
		{
			QDateTime dt;
			dt.setTime_t( ((KListViewItemDups*)it.current())->item->time( KIO::UDS_MODIFICATION_TIME ) );
			if ( dt.date() == date )
			{
				((KListViewItemDups*)it.current())->removeFilter( DATE );
			}
			else
			{
				((KListViewItemDups*)it.current())->addFilter( DATE );
			}
		}
		else
		{
			((KListViewItemDups*)it.current())->removeFilter( DATE );
		}
		
		if ( ((KListViewItemDups*)it.current())->isDuplicatesPair() )
		{
			if ( duplicates_normal )
			{
				((KListViewItemDups*)it.current())->removeFilter( DUPLICATES_NORMAL );
			}
			else
			{
				((KListViewItemDups*)it.current())->addFilter( DUPLICATES_NORMAL );
			}
		}
		else
		{
			if ( duplicates_multiple )
			{
				((KListViewItemDups*)it.current())->removeFilter( DUPLICATES_MULTIPLE );
			}
			else
			{
				((KListViewItemDups*)it.current())->addFilter( DUPLICATES_MULTIPLE );
			}
		}

		((KListViewItemDups*)it.current())->setVisibility();

		if ( it.current()->isVisible() ) visible_items++;

		++it;
	}

	emit( signalCount( DUPLICATES, 1, visible_items ) );

	tmp_list_view = NULL;
}

void KomparatorWidget::slotFilterChanged( KomparatorFileFilter *filter, const QString &name, const QString &path,
			const QString &owner, const QString &group, const unsigned long &min_size, const unsigned long &max_size,
			const bool &consider_date, const QDate &date )
{
	KListView *tmp_list_view;
	if ( filter == duplicates2_KomparatorFileFilter ) tmp_list_view = duplicates2_KListView;
	else if ( filter == missing1_KomparatorFileFilter ) tmp_list_view = missing1_KListView;
	else if ( filter == missing2_KomparatorFileFilter ) tmp_list_view = missing2_KListView;
	else if ( filter == newer1_KomparatorFileFilter ) tmp_list_view = newer1_KListView; // these won't be processed here usually; however it would be legal.
	else if ( filter == newer2_KomparatorFileFilter ) tmp_list_view = newer2_KListView;
	else
	{
		kdDebug() << "Received file filter event with filter " << filter << ", but this filter doesn't exist!" << endl;
		return;
	}

	int visible_items = 0;
	QListViewItemIterator it( tmp_list_view );
	while( it.current() )
	{
		if ( it.current()->text( 0 ).find( name ) != -1 ) // file name contains str, empty filter (=no filter) returns != -1 always.
		{
			((KListViewItemSingle*)it.current())->removeFilter( NAME ); // Don't filter this one out
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( NAME );
		}

		if ( it.current()->text( 1 ).find( path ) != -1 )
		{
			((KListViewItemSingle*)it.current())->removeFilter( PATH );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( PATH );
		}

		if ( it.current()->text( 5 ).find( owner ) != -1 )
		{
			((KListViewItemSingle*)it.current())->removeFilter( OWNER );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( OWNER );
		}

		if ( it.current()->text( 6 ).find( group ) != -1 )
		{
			((KListViewItemSingle*)it.current())->removeFilter( GROUP );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( GROUP );
		}

		if ( ((KListViewItemSingle*)it.current())->item->size() >= min_size &&
			((KListViewItemSingle*)it.current())->item->size() <= max_size )
		{
			((KListViewItemSingle*)it.current())->removeFilter( SIZE );
		}
		else
		{
			((KListViewItemSingle*)it.current())->addFilter( SIZE );
		}

		if ( consider_date )
		{
			QDateTime dt;
			dt.setTime_t( ((KListViewItemSingle*)it.current())->item->time( KIO::UDS_MODIFICATION_TIME ) );
			if ( dt.date() == date )
			{
				((KListViewItemSingle*)it.current())->removeFilter( DATE );
			}
			else
			{
				((KListViewItemSingle*)it.current())->addFilter( DATE );
			}
		}
		else
		{
			((KListViewItemSingle*)it.current())->removeFilter( DATE );
		}

		((KListViewItemSingle*)it.current())->setVisibility();

		if ( it.current()->isVisible() ) visible_items++;

		++it;
	}

	if ( filter == duplicates2_KomparatorFileFilter ) emit( signalCount( DUPLICATES, 2, visible_items ) );
	else if ( filter == missing1_KomparatorFileFilter ) emit( signalCount( MISSING, 1, visible_items ) );
	else if ( filter == missing2_KomparatorFileFilter ) emit( signalCount( MISSING, 2, visible_items ) );
	else if ( filter == newer1_KomparatorFileFilter ) emit( signalCount( NEWER, 1, visible_items ) );
	else if ( filter == newer2_KomparatorFileFilter ) emit( signalCount( NEWER, 2, visible_items ) );

	tmp_list_view = NULL;
}


/****************************************
* Internal Functions                   *
****************************************/

void KomparatorWidget::slotSizeTypeChanged( int i )
{
	bytes_KIntNumInput->setEnabled( i );
	bytes_KComboBox->setEnabled( i );
}

void KomparatorWidget::slotSizeEnabled( bool e )
{
	calculateChecksum_CheckBox->setEnabled( e );
	if ( !e )
	{
		calculateChecksum_CheckBox->setChecked( false );
	}
}

void KomparatorWidget::slotMD5ChecksumEnabled( bool e )
{
	binaryComparison_CheckBox->setEnabled( e );
	if ( e )
	{
		binaryComparison_KIntSpinBox->setEnabled( ( binaryComparison_CheckBox->isChecked() ) ? true : false );
	}
	else
	{
		binaryComparison_CheckBox->setChecked( false );
		binaryComparison_KIntSpinBox->setEnabled( false );
	}
}

void KomparatorWidget::slotDuplicatesOnlyEnabled( bool e )
{
	url1_KURLRequester->setEnabled( !e );
}

void KomparatorWidget::slotProgress( QString str, int percent, QString notification )
{
	status_KLineEdit->setText( str );
	main_KProgress->setTextEnabled( ( percent == -1 ) ? false : true );
	main_KProgress->setTotalSteps( 100 );
	main_KProgress->setProgress( ( percent == -1 ) ? 0 : percent );

	if ( notification != "" && notification_CheckBox->isChecked() && !isActiveWindow() )
	{
		KPassivePopup::message( i18n( "Komparator" ),
			notification,
			SmallIcon( "komparator" ), ( systray_CheckBox->isChecked() ? (QWidget*)m_systray : (QWidget*)this ),
			"komparator_notification", 2500 );
	}
}

void KomparatorWidget::slotProgressUndefined()
{
	main_KProgress->setTextEnabled( false );
	if ( main_KProgress->totalSteps() != 0 ) main_KProgress->setTotalSteps( 0 );
	main_KProgress->advance( 2 );
}

void KomparatorWidget::enable( bool e )
{
	settings_TabWidget->setEnabled( e );
	main_TabWidget->setEnabled( e );

	if ( e ) QApplication::restoreOverrideCursor();
	else QApplication::setOverrideCursor( QCursor( Qt::WaitCursor ), true );
}

void KomparatorWidget::resizeListViewColumns()
{
	int i;
	for ( i=0; i<7; i++ )
	{
		duplicates1_KListView->setColumnWidth( i, m_duplicates1_width[i] );
		duplicates2_KListView->setColumnWidth( i, m_duplicates2_width[i] );
		missing1_KListView->setColumnWidth( i, m_missing1_width[i] );
		missing2_KListView->setColumnWidth( i, m_missing2_width[i] );
		newer1_KListView->setColumnWidth( i, m_newer1_width[i] );
		newer2_KListView->setColumnWidth( i, m_newer2_width[i] );
	}
	duplicates2_KListView->setColumnWidth( 7, m_duplicates2_width[7] );

	duplicates1_KListView->setSorting( 0, true );
	duplicates2_KListView->setSorting( 0, true );
	missing1_KListView->setSorting( 0, true );
	missing2_KListView->setSorting( 0, true );
	newer1_KListView->setSorting( 0, true );
	newer2_KListView->setSorting( 0, true );
}

void KomparatorWidget::updateFilters()
{
	duplicates1_KomparatorFileFilter->slotEmitFilterChanged();
	duplicates2_KomparatorFileFilter->slotEmitFilterChanged();
	missing1_KomparatorFileFilter->slotEmitFilterChanged();
	missing2_KomparatorFileFilter->slotEmitFilterChanged();
	newer1_KomparatorFileFilter->slotEmitFilterChanged();
	newer2_KomparatorFileFilter->slotEmitFilterChanged();
}

void KomparatorWidget::slotSaveListViewColumnWidth()
{
	int i;
	for ( i=0; i<7; i++ ) // used for resizing if canceled.
	{
		m_duplicates1_width[i] = duplicates1_KListView->columnWidth( i );
		m_duplicates2_width[i] = duplicates2_KListView->columnWidth( i );
		m_missing1_width[i] = missing1_KListView->columnWidth( i );
		m_missing2_width[i] = missing2_KListView->columnWidth( i );
		m_newer1_width[i] = newer1_KListView->columnWidth( i );
		m_newer2_width[i] = newer2_KListView->columnWidth( i );
	}
	m_duplicates2_width[7] = duplicates2_KListView->columnWidth( 7 );
}

void KomparatorWidget::slotStartSystrayMovie()
{
	m_systray->setMovie( *m_systray_movie );

	m_systray_movie->unpause();
}

void KomparatorWidget::slotStopSystrayMovie()
{
	if ( m_systray_movie && !m_systray_movie->isNull() )
	{
		m_systray_movie->pause();
		m_systray_movie->restart();
	}

	m_systray->setPixmap( m_systray->loadIcon( "komparator" ) );
}

QString KomparatorWidget::prettyURL( const KURL &url, URL_MODE mode, const KURL &rep  )
{
	KURL tmp_url;

	QString ret = "";
	switch( mode )
	{
		case URL_DIR_OR_URL:
			tmp_url = url; // This is necessary to display a pretty url on non-local file systems.
			tmp_url.setPath( "" );
			ret = tmp_url.pathOrURL() + url.path( -1 );
			break;
		case URL_DIR_OR_URL_ONLY:
			tmp_url = url; // This is necessary to display a pretty url on non-local file systems.
			tmp_url.setPath( "" );
			ret = tmp_url.pathOrURL() + url.upURL().path( -1 );
			break;
		case URL_FILE_NAME:
			ret = KIO::decodeFileName( url.fileName() );
			break;
		case URL_REPLACE:
			ret = KURL::relativePath( rep.path( -1 ), url.upURL().path( -1 ) );
			break;
		case URL_URL:
				ret = url.prettyURL();
			break;
		default:
			ret = url.url();
			break;
	}

	return ret;
}

bool KomparatorWidget::isEnabled( LIST_VIEW list_view )
{
	QIconSet tab_icon_set = main_TabWidget->tabIconSet( main_TabWidget->page( (int)list_view ) );
	if ( tab_icon_set.pixmap( QIconSet::Small, QIconSet::Normal ).convertToImage() ==
			KGlobal::iconLoader()->loadIconSet( "no", KIcon::Toolbar, KIcon::SizeSmall ).pixmap(
				QIconSet::Small, QIconSet::Normal ).convertToImage() )
	{
		return false;
	}

	return true;
}

QImage KomparatorWidget::transparentImage()
{
	QImage transparent( SmallIcon( "1rightarrow" ).width(), SmallIcon( "1rightarrow" ).height(), 32 );
	int i, j;
	transparent.setAlphaBuffer( true );
	for ( i=0; i<transparent.width(); i++ ) // FIXME: set complete QImage transparent in an easier way?
		for ( j=0; j<transparent.height(); j++ )
			transparent.setPixel( i, j, qRgba( 0, 0, 0, 0 ) );
	return transparent;
}

QIconSet KomparatorWidget::broadIconSet( const QPixmap pm1, const QPixmap pm2, const QPixmap pm3 )
{
	QPixmap ret_pm( pm1.width() + pm2.width() + pm3.width(), ( pm1.height() + pm2.height() + pm3.height() ) / 3 );

	// void copyBlt ( QPixmap * dst, int dx, int dy, const QPixmap * src, int sx, int sy, int sw, int sh )
	copyBlt( &ret_pm, 0, 0, &pm1, 0, 0, pm1.width(), pm1.height() );
	copyBlt( &ret_pm, pm1.width(), 0, &pm2, 0, 0, pm2.width(), pm2.height() );
	copyBlt( &ret_pm, pm1.width() + pm2.width(), 0, &pm3, 0, 0, pm3.width(), pm3.height() );

	QIconSet::setIconSize( QIconSet::Small, ret_pm.size() );

	return QIconSet( ret_pm );
}

bool KomparatorWidget::isSystrayVisible()
{
	if ( !m_systray ) return false;
	if ( !m_systray->isVisible() ) return false;
	return true;
}

void KomparatorWidget::slotCount( LIST_VIEW list_view, int which, int visible_items )
{
	switch( list_view )
	{
		case DUPLICATES:
			if ( which == 1 )
			{
				duplicates1ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? duplicates1_KListView->childCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == duplicates1_KListView->childCount() )
				{
					duplicates1ItemCount_TextLabel->unsetPalette();
					duplicates1_KomparatorFileFilter->unsetPalette();
				}
				else
				{
					duplicates1ItemCount_TextLabel->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
					duplicates1_KomparatorFileFilter->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
				}
			}
			else if ( which == 2 )
			{
				duplicates2ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? duplicates2_KListView->childCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == duplicates2_KListView->childCount() )
				{
					duplicates2ItemCount_TextLabel->unsetPalette();
					duplicates2_KomparatorFileFilter->unsetPalette();
				}
				else
				{
					duplicates2ItemCount_TextLabel->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
					duplicates2_KomparatorFileFilter->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
				}
			}
			break;
		case MISSING:
			if ( which == 1 )
			{
				missing1ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? missing1_KListView->childCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == missing1_KListView->childCount() )
				{
					missing1ItemCount_TextLabel->unsetPalette();
					missing1_KomparatorFileFilter->unsetPalette();
				}
				else
				{
					missing1ItemCount_TextLabel->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
					missing1_KomparatorFileFilter->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
				}
			}
			else if ( which == 2 )
			{
				missing2ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? missing2_KListView->childCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == missing2_KListView->childCount() )
				{
					missing2ItemCount_TextLabel->unsetPalette();
					missing2_KomparatorFileFilter->unsetPalette();
				}
				else
				{
					missing2ItemCount_TextLabel->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
					missing2_KomparatorFileFilter->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
				}
			}
			break;
		case NEWER:
			if ( which == 1 )
			{
				newer1ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? newer1_KListView->childCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == newer1_KListView->childCount() )
				{
					newer1ItemCount_TextLabel->unsetPalette();
					newer1_KomparatorFileFilter->unsetPalette();
				}
				else
				{
					newer1ItemCount_TextLabel->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
					newer1_KomparatorFileFilter->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
				}
			}
			else
			{
				newer2ItemCount_TextLabel->setText( QString( "%1" ).arg(
					( visible_items == -1 ) ? newer2_KListView->childCount() : visible_items ) );
				if ( visible_items == -1 || visible_items == newer2_KListView->childCount() )
				{
					newer2ItemCount_TextLabel->unsetPalette();
					newer2_KomparatorFileFilter->unsetPalette();
				}
				else
				{
					newer2ItemCount_TextLabel->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
					newer2_KomparatorFileFilter->setPaletteBackgroundColor( QColor( 110, 145, 200 ) );
				}
			}
			break;
		default:
			break;
	}
}



/****************************************
* KQuery Functions                     *
****************************************/

void KomparatorWidget::slotAddFile( const KFileItem* item, const QString & )
{
	KFileItemExt *new_item;
	KIO::UDSEntry entry;
	KURL parent_url = ( m_status == SEARCHING_URL_1 ) ? m_url1 : m_url2;

	if ( item->isFile() && ( followSoftLinks_CheckBox->isChecked() || !item->isLink() ) )
	{ // Real m_files are added always, (soft)links only if user wants to.
		if ( !item->isReadable() )
		{
			// The last string is the filename as it might contain "%2" for example, which makes "arg()" insert its string at the wrong position
			m_unreadable_files.append( QString( "%3  %2  %1" ).arg( item->permissionsString() ).arg(
				item->user() + ":" + item->group() ).arg( item->url().prettyURL() ) );
		}
		else
		{
			m_number_of_files++;

			new_item = new KFileItemExt( *item );

			if ( m_status == SEARCHING_URL_1 )
			{
				new_item->dir = 0;
				new_item->virtual_parent_path = parent_url;
				new_item->parent_path = parent_url;
			}
			else
			{
				new_item->dir = 1;
				if ( !m_virtual_subdirectory_current.isEmpty() )
				{
					new_item->parent_path = m_virtual_subdirectory_current.upURL();
				}
				else
				{
					new_item->parent_path = parent_url;
				}
				new_item->virtual_parent_path = parent_url;
			}

			new_item->next = m_files;
				
			m_files = new_item;
		}
	}
	else if ( item->isDir() && ( followSoftLinks_CheckBox->isChecked() || !item->isLink() )  )
	{
		if ( findEmptyDirectories_CheckBox->isChecked() )
		{
			KURL tmp_url = item->url();
			tmp_url.setPath( item->url().path( 1 ) );
			tmp_url.setFileName( "a" ); // The filename doesn't matter, as "checkEmptyDirs" uses "upURL". But it must be present.
			m_interaction_empty_dirs.push_back( tmp_url );
		}

		if ( item->isLink() && followSoftLinks_CheckBox->isChecked() )
		{
			bool link_is_parent = false;

			KURL recursive_parent( item->url().url() );
			recursive_parent.setFileName( "" );
			recursive_parent.addPath( item->linkDest() );
			recursive_parent.cleanPath();

			KIO::UDSEntry entry;
			KIO::NetAccess::stat( recursive_parent, entry, this );
			KFileItem recursive_parent_item( entry, recursive_parent );

			if ( recursive_parent.isParentOf( item->url() ) ) link_is_parent = true;

			while ( recursive_parent_item.isLink() && !link_is_parent )
			{
				recursive_parent = recursive_parent_item.url().url();
				recursive_parent.setFileName( "" );
				recursive_parent.addPath( recursive_parent_item.linkDest() );
				recursive_parent.cleanPath();

				KIO::NetAccess::stat( recursive_parent, entry, this );
				recursive_parent_item = KFileItem( entry, recursive_parent );

				if ( recursive_parent.isParentOf( item->url() ) ) link_is_parent = true;
			}

			if ( link_is_parent )
			{
				kdDebug() << "Recursive links detected. " << item->url() << " is a link to "
					<< recursive_parent.url() << ", which is a parent of the item." << endl;
				QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
				KMessageBox::information( this, i18n(
					"Not recursing into symlinked directory %1 because it is a link to its parent directory %2." ).arg(
					item->url().prettyURL() ).arg( recursive_parent.prettyURL() ), i18n( "Recursive links detected" ) );
				QApplication::restoreOverrideCursor();
			}
			else
			{
				m_symlinked_directories.append( KURL( item->url() ) );
			}
		}
	}
}

void KomparatorWidget::slotResult( int error_code, QString error_string )
{
	if ( error_code == 0 )
	{
		switch( m_status )
		{
			case SEARCHING_URL_1:
			{
				if ( !m_symlinked_directories.isEmpty() && includeSubdirectories_CheckBox->isChecked() ) // Search symlinked directories before going over to url 2.
				{
					m_query->setPath( m_symlinked_directories.front() );
					m_symlinked_directories.pop_front();
					m_query->start();
					return;
				}

				search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
				search2_KLed->setColor( QColor( 255, 0, 0 ) );  // Red
				search3_KLed->setColor( QColor( 255, 0, 0 ) );

				m_status = SEARCHING_URL_2;

				m_query->setPath( m_url2 );

				emit( signalProgress( i18n( "Searching right URL for files..." ), -1, "" ) );

				m_symlinked_directories.clear();

				m_query->start();
				break;
			}
			case SEARCHING_URL_2:
			{
				if ( !m_symlinked_directories.isEmpty() && includeSubdirectories_CheckBox->isChecked() )
				{
					m_query->setPath( m_symlinked_directories.front() );
					m_symlinked_directories.pop_front();
					m_query->start();
					return;
				}
				if ( !m_virtual_subdirectories.isEmpty() && includeSubdirectories_CheckBox->isChecked() ) // Virtual subdirectories only exist for the right URL.
				{
					m_query->setPath( m_virtual_subdirectories.front() );
					m_virtual_subdirectory_current = m_virtual_subdirectories.front();
					m_virtual_subdirectories.pop_front();
					m_query->start();
					return;
				}
				m_virtual_subdirectory_current = KURL( "" );

				m_progress_timer->stop();

				m_status = SEARCHING_EMPTY_DIRECTORIES;

				emit( signalProgress( i18n( "Searching for empty directories..." ), -1, "" ) );

				if ( m_interaction_empty_dirs.count() > 0 )
				{
					checkEmptyDirs( m_interaction_empty_dirs );
					break;
				}
			}

			case SEARCHING_EMPTY_DIRECTORIES:
			{
				m_interaction_empty_dirs.clear();
				m_virtual_subdirectories.clear();

				search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
				search2_KLed->setColor( QColor( 0, 255, 0 ) );
				search3_KLed->setColor( QColor( 255, 0, 0 ) );  // Red

				emit( signalProgress( "", -1, "" ) );

				compare_job->initialize( m_files, ignoreEmptyFiles_CheckBox->isChecked(), size_CheckBox->isChecked(),
					calculateChecksum_CheckBox->isChecked(), binaryComparison_CheckBox->isChecked(),
					binaryComparison_KIntSpinBox->value(), m_number_of_files, isEnabled( DUPLICATES ),
					isEnabled( MISSING ) && !duplicatesOnly_CheckBox->isChecked(), isEnabled( NEWER ) && !duplicatesOnly_CheckBox->isChecked(),
					caseSensitiveSearch_CheckBox->isChecked() );

				m_status = COMPARING;

				compare_job->start();
				break;
			}

			default:
				break;
		}
	}
	else if ( error_code == KIO::ERR_USER_CANCELED )
	{
		if ( m_status != CLEAN ) slotSearch();
		return; // cancel is handled in slotSearch()
	}
	else if ( error_code == KIO::ERR_MALFORMED_URL )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false/*override already saved cursor*/ );
		if ( m_status == SEARCHING_URL_1 ) KMessageBox::sorry( this, i18n( "Please specify an absolute path in the left URL box." ) );
		else if ( m_status == SEARCHING_URL_2 ) KMessageBox::sorry( this, i18n( "Please specify an absolute path in the right URL box." ) );
		QApplication::restoreOverrideCursor();
		slotSearch(); // Like clicking cancel button.
		return;
	}
	else
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false/*override already saved cursor*/ );
		KMessageBox::sorry( this, error_string );
		QApplication::restoreOverrideCursor();
		slotSearch(); // Like cancel
		return;
	}
}

bool KomparatorWidget::initializeQuery()
{
  KIO::filesize_t size;
  KIO::filesize_t sizeunit;

	// only start if we have valid dates
	if ( !isDateValid() ) return false;

	if ( url1_KURLRequester->url().isEmpty() && !duplicatesOnly_CheckBox->isChecked() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false/*override already saved cursor*/ );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the left URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}
	else if ( url2_KURLRequester->url().isEmpty() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false/*override already saved cursor*/ );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the right URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}

	QString url1_string = url1_KURLRequester->url().stripWhiteSpace();
	QString url2_string = url2_KURLRequester->url().stripWhiteSpace();

	if ( url1_string != "/" )
	{
		while ( url1_string.right( 1 ) == "/" )
		{
			url1_string = url1_string.left( url1_string.length()-1 );
		}
	}

	while ( url1_string.left( 2 ) == "//" )
	{
		url1_string = url1_string.right( url1_string.length()-1 );
	}

	if ( url2_string != "/" )
	{
		while ( url2_string.right( 1 ) == "/" )
		{
			url2_string = url2_string.left( url2_string.length()-1 );
		}
	}

	while ( url2_string.left( 2 ) == "//" )
	{
		url2_string = url2_string.right( url2_string.length()-1 );
	}

	KURL url1 = KURL::fromPathOrURL( url1_string );
	KURL url2 = KURL::fromPathOrURL( url2_string );

	m_virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();

	KIO::UDSEntry entry;
	if ( !duplicatesOnly_CheckBox->isChecked() )
	{
		KIO::NetAccess::stat( url1, entry, this );
	}
	KFileItem url1_item( entry, url1 );

	KIO::NetAccess::stat( url2, entry, this );
	KFileItem url2_item( entry, url2 );

	if ( url1_item.isLocalFile() && url1_item.isReadable() )
	{
		KMimeType::Ptr type1 = url1_item.determineMimeType();

		if      ( type1->is( "application/x-tgz" ) ) url1.setProtocol( "tar" );
		else if ( type1->is( "application/x-tbz" ) ) url1.setProtocol( "tar" );
		else if ( type1->is( "application/x-tar" ) ) url1.setProtocol( "tar" );
		else if ( type1->is( "application/x-zip" ) ) url1.setProtocol( "zip" );
		// else we don't know the correct protocol. this doesn't mean, it cannot be searched by kquery.
		// therefore we don't give an error message here, but let kquery fail if it doesn't work.
	}

	if ( url2_item.isLocalFile() && url2_item.isReadable() )
	{
		KMimeType::Ptr type2 = url2_item.determineMimeType();

		if      ( type2->is( "application/x-tgz" ) ) url2.setProtocol( "tar" );
		else if ( type2->is( "application/x-tbz" ) ) url2.setProtocol( "tar" );
		else if ( type2->is( "application/x-tar" ) ) url2.setProtocol( "tar" );
		else if ( type2->is( "application/x-zip" ) ) url2.setProtocol( "zip" );
		// else we don't know the correct protocol. this doesn't mean, it cannot be searched by kquery.
		// therefore we don't give an error message here, but let kquery fail if it doesn't work.
	}

	url1.cleanPath();
	url2.cleanPath();

	if ( url1_item.localPath() != "" ) url1_KURLRequester->setURL( url1_item.localPath() );
	else url1_KURLRequester->setURL( url1.pathOrURL() );
	if ( url2_item.localPath() != "" ) url2_KURLRequester->setURL( url2_item.localPath() );
	else url2_KURLRequester->setURL( url2.pathOrURL() );

	if ( !url1_item.isReadable() ) // it can be also not a dir to be a valid search path. e.g. media:/sd**/file.zip
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the left URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}
	else if ( !url2_item.isReadable() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
		KMessageBox::sorry( this, i18n( "Please specify an absolute path in the right URL box." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}

	m_url1 = url1;
	m_url2 = url2;

	m_query->setPath( m_url1 );

	QString regexContains = regexContains_KLineEdit->text().isEmpty() ? "*" : regexContains_KLineEdit->text();
	QString regexContainsNot = regexContainsNot_KLineEdit->text();
	m_query->setRegExp( regexContains, regexContainsNot, caseSensitiveSearch_CheckBox->isChecked() );

	m_query->setRecursive( includeSubdirectories_CheckBox->isChecked() );

	switch ( bytes_KComboBox->currentItem() )
	{
		case 0:
			sizeunit = 1; // one byte
			break;
		case 2:
			sizeunit = 1048576; // 1M
			break;
		case 1:
		default:
			sizeunit = 1024; // 1k
			break;
	}
	size = bytes_KIntNumInput->value() * sizeunit;

	m_query->setSizeRange( size_KComboBox->currentItem(), size, 0 );

	// dates
	QDateTime epoch;
	epoch.setTime_t( 0 );

	// Add date predicate
	if ( !modifiedNoRestriction_RadioButton->isChecked() ) // Modified
	{
		if ( modifiedBetween_RadioButton->isChecked() ) // Between dates
		{
			QDate q1, q2;
			modifiedFrom_KDateCombo->getDate( &q1 );
			modifiedTo_KDateCombo->getDate( &q2 );

			// do not generate negative numbers .. find doesn't handle that
			time_t time1 = epoch.secsTo( q1 );
			time_t time2 = epoch.secsTo( q2.addDays( 1 ) ) - 1; // Include the last day

			m_query->setTimeRange( time1, time2 );
		}
		else
		{
			time_t cur = time( NULL );
			time_t minutes = cur;

			switch ( modifiedLastTime_KComboBox->currentItem() )
			{
					case 0: // minutes
						minutes = modifiedLastTime_KIntNumInput->value();
						break;
					case 1: // hours
						minutes = 60 * modifiedLastTime_KIntNumInput->value();
						break;
					case 2: // days
						minutes = 60 * 24 * modifiedLastTime_KIntNumInput->value();
						break;
					case 3: // months
						minutes = 60 * 24 * (time_t)( modifiedLastTime_KIntNumInput->value() * 30.41667 );
						break;
					case 4: // years
						minutes = 12 * 60 * 24 * (time_t)( modifiedLastTime_KIntNumInput->value() * 30.41667 );
						break;
					default:
						break;
			}

			m_query->setTimeRange( cur - minutes * 60, 0 );
		}
	}
	else
		m_query->setTimeRange( 0, 0 );

	m_query->setUsername( "" );
	m_query->setGroupname( "" );

	m_query->setFileType( 0 );

	m_query->setMimeType( QString::null );

	// Use locate to speed-up search ?
	m_query->setUseFileIndex( useFilesIndex_CheckBox->isChecked() );

	m_query->setIncludeHiddenItems( includeHiddenItems_CheckBox->isChecked() );

	return true;
}

bool KomparatorWidget::isDateValid()
{
	// All m_files
	if ( modifiedNoRestriction_RadioButton->isChecked() ) return true;

	if ( modifiedLastTime_RadioButton->isChecked() )
	{
		if (modifiedLastTime_KIntNumInput->value() > 0 ) return true;

		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
		KMessageBox::sorry( this, i18n( "Unable to search within a period which is less than a minute." ) );
		QApplication::restoreOverrideCursor();
		return false;
	}

	// If we can not parse either of the dates or
	// "from" date is bigger than "to" date return false.
	QDate hi1, hi2;

	QString str;
	if ( ! modifiedFrom_KDateCombo->getDate(&hi1).isValid() ||
			! modifiedTo_KDateCombo->getDate(&hi2).isValid() )
		str = i18n( "The date is not valid." );
	else if ( hi1 > hi2 )
		str = i18n( "Invalid date range." );
	else if ( QDate::currentDate() < hi1 )
		str = i18n( "Unable to search dates in the future." );

	if ( !str.isNull() )
	{
		QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
		KMessageBox::sorry( 0, str );
		QApplication::restoreOverrideCursor();
		return false;
	}

	return true;
}



/****************************************
* KompareJob Functions                 *
****************************************/

void KomparatorWidget::customEvent( QCustomEvent *event )
{
	KListViewItemSingle *new_item_single1;
	KListViewItemSingle *new_item_single2;
	QFile file;
	KTempFile *tmp_file;
	int dir;

	switch ( event->type() )
	{
		case RESULT_DUPLICATE:
		{
			(new KListViewItemDups( duplicates1_KListView, ((KFileItemExt *) event->data()) ))->setUseColor( textColor_CheckBox->isChecked() );
			emit( signalCount( DUPLICATES, 1, -1 ) );
			break;
		}

		case RESULT_COMP_MISSING:
		{
			dir = ((KFileItemExt *) event->data())->dir;
			new_item_single1 = new KListViewItemSingle( ( dir == 0 ) ? missing1_KListView : missing2_KListView, ((KFileItemExt *) event->data()), NULL );
			new_item_single1->setText( 1, prettyURL( ((KFileItemExt *) event->data())->url(), URL_REPLACE, ((KFileItemExt *) event->data())->parent_path ) );
			emit( signalCount( MISSING, ( dir == 0 ) ? 1 : 2, -1 ) );
			break;
		}

		case RESULT_COMP_NEWER:
		case RESULT_COMP_NEWER_EQUAL:
		case RESULT_COMP_DIFFERENT:
		{
			dir = ((QPair<KFileItemExt *, KFileItemExt *> *) event->data())->first->dir;

			new_item_single1 = new KListViewItemSingle( ( dir == 0 ) ? newer1_KListView : newer2_KListView,
				((QPair<KFileItemExt *, KFileItemExt *> *) event->data())->first, NULL );
			new_item_single1->setText( 1, prettyURL( ((QPair<KFileItemExt *, KFileItemExt *> *) event->data())->first->url(),
				URL_REPLACE, ((QPair<KFileItemExt *, KFileItemExt *> *) event->data())->first->parent_path ) );

			new_item_single2 = new KListViewItemSingle( ( dir == 0 ) ? newer2_KListView : newer1_KListView,
				((QPair<KFileItemExt *, KFileItemExt *> *) event->data())->second, new_item_single1 );
			new_item_single2->setText( 1, prettyURL( ((QPair<KFileItemExt *, KFileItemExt *> *) event->data())->second->url(),
				URL_REPLACE, ((QPair<KFileItemExt *, KFileItemExt *> *) event->data())->second->parent_path ) );

			new_item_single1->brother = new_item_single2;
			emit( signalCount( NEWER, 1, -1 ) );
			emit( signalCount( NEWER, 2, -1 ) );

			if ( (int)event->type() == (int)RESULT_COMP_NEWER )
			{
				new_item_single1->setKind( NEWER_REAL, textColor_CheckBox->isChecked() );
				new_item_single2->setKind( NEWER_REAL, textColor_CheckBox->isChecked(), m_blue );
			}
			else if ( (int)event->type() == (int)RESULT_COMP_NEWER_EQUAL )
			{
				new_item_single1->setKind( NEWER_EQUAL, textColor_CheckBox->isChecked() );
				new_item_single2->setKind( NEWER_EQUAL, textColor_CheckBox->isChecked(), m_purple );
			}
			else
			{
				new_item_single1->setKind( NEWER_SAME_TIME, textColor_CheckBox->isChecked(), m_red );
				new_item_single2->setKind( NEWER_SAME_TIME, textColor_CheckBox->isChecked(), m_red );
			}

			delete ((QPair<KFileItemExt *, KFileItemExt *> *) event->data());

			break;
		}

		case RESULT_COMP_FINISHED:
		{
			compare_job->wait();
			updateFilters();

			search1_KLed->setColor( QColor( 0, 255, 0 ) );  // Green
			search2_KLed->setColor( QColor( 0, 255, 0 ) );
			search3_KLed->setColor( QColor( 0, 255, 0 ) );

			enable( true );

			if ( m_unreadable_files.count() > 0 )
			{
				QApplication::setOverrideCursor( QCursor( Qt::ArrowCursor ), false /* override already saved cursor */ );
				KMessageBox::informationList( this, ( m_unreadable_files.count() == 1 ) ?
					i18n( "1 file was unreadable, and will not be considered any futher." ) :
					i18n( "%1 files were unreadable, and will not be considered any futher." ).arg( m_unreadable_files.count() )
					+ " " + i18n( "Change the owner and/or permissions to fix that problem." ),
					m_unreadable_files ); // sorryList message box doesn't exist; errorList is probably too much.
				QApplication::restoreOverrideCursor();
			}
			m_unreadable_files = 0;

			m_status = READY;

			emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Search and comparison finished." ) ) );
			emit( signalStopSystrayMovie() );

			search_KPushButton->setText( i18n( "Clear" ) );
			search_KPushButton->setIconSet( SmallIcon( "clear_left" ) );
			break;
		}

		case RESULT_EMPTY_FINISHED:
		{
			KIO::UDSEntry entry;
			//KIO::NetAccess::stat( ((KIO::SimpleJob*)job)->url(), entry, this );
			KURL::List empty_directory_list = m_empty_directory_job->getEmptyDirectoryList();

			KURL::List::Iterator it;
			for ( it = empty_directory_list.begin(); it != empty_directory_list.end(); ++it )
			{
				KFileItemExt *new_item_dupes = new KFileItemExt( entry, (*it) );
				new_item_dupes->dir = 0;
				if ( m_url2.isParentOf( new_item_dupes->url() ) ) new_item_dupes->dir = 1;

				KURL::List virtual_subdirectories = subdirectories_KomparatorUrlList->getUrlList();
				KURL::List::Iterator it2;
				for ( it2 = virtual_subdirectories.begin(); it2 != virtual_subdirectories.end(); ++it2 )
				{
					if ( (*it2).isParentOf( new_item_dupes->url() ) )
					{
						new_item_dupes->dir = 1;
						new_item_dupes->parent_path = new_item_dupes->virtual_parent_path;
						break;
					}
					else
					{
						new_item_dupes->parent_path = new_item_dupes->dir ? m_url2 : m_url1;
					}
				}
				new_item_dupes->virtual_parent_path = new_item_dupes->dir ? m_url2 : m_url1;

				new_item_dupes->isdupe_size = 1;

				if ( m_dirs_dupes )
				{
					m_dirs_dupes->hasdupes_size = 1;
					KFileItemExt *tmp_dupes = m_dirs_dupes->duplicates_size;
					m_dirs_dupes->duplicates_size = new_item_dupes;
					new_item_dupes->dup_size_parent = m_dirs_dupes;
					m_dirs_dupes->duplicates_size->duplicates_size = tmp_dupes;
					if ( m_dirs_dupes->duplicates_size->duplicates_size )
					{
						m_dirs_dupes->duplicates_size->hasdupes_size = 1;
						m_dirs_dupes->duplicates_size->duplicates_size->dup_size_parent = m_dirs_dupes->duplicates_size;
					}
				}
				else
				{
					m_dirs_dupes = new_item_dupes;
					if ( isEnabled( DUPLICATES ) )
					{
						(new KListViewItemDups( duplicates1_KListView, m_dirs_dupes ))->setUseColor( textColor_CheckBox->isChecked() );
						emit( signalCount( DUPLICATES, 1, -1 ) );
					}
				}
			}

			slotDuplicateSelected();
			if ( m_status == SEARCHING_EMPTY_DIRECTORIES ) slotResult( 0, "" );
			else
			{
				updateFilters();

				current_list_view = NULL;

				emit( signalProgress( i18n( "Displaying result." ), -1, i18n( "Searching for empty directories finished." ) ) );
				emit( signalStopSystrayMovie() );

				enable( true );

				m_status = READY;
				search_KPushButton->setText( i18n( "Clear" ) );
				search_KPushButton->setIconSet( SmallIcon( "clear_left" ) );
				break;
			}
		}

		case RESULT_COPY_FINISHED:
		{
			break;
		}

		case RESULT_MOVE_FINISHED:
		{
			break;
		}

		case RESULT_REMOVE_FINISHED:
		{
			break;
		}

		case DOWNLOAD_FILE:
		{
			tmp_file = new KTempFile;
			tmp_file->close();
			m_requested_download = (KFileItemExt *) event->data();
			((KFileItemExt *) event->data())->setTmpFile( tmp_file );

			m_requested_download_job = KIO::file_copy( ((KFileItemExt *) event->data())->url(), tmp_file->name(), -1,
				true /*overwrite*/, false /* resume */, false /* progress */ );
			connect( m_requested_download_job, SIGNAL( result( KIO::Job *) ), this, SLOT( slotFileDownloaded( KIO::Job *) ) );
			break;
		}

		case DELETE_FILE:
		{
// 			KIO::NetAccess::removeTempFile( ((KTempFile *) event->data())->name() );
			((KTempFile *) event->data())->unlink();
			((KTempFile *) event->data())->setAutoDelete( true );
			delete ((KTempFile *) event->data());
			m_requested_download = NULL;
			break;
		}

		case PROGRESS_PROGRESS:
		{
			QString message = *((QPair<std::string *, int *> *) event->data())->first;
			int progress = *((QPair<std::string *, int *> *) event->data())->second;
			emit( signalProgress( message, progress, "" ) );

			delete (std::string*) (((QPair<QString *, int *> *) event->data())->first);
			((QPair<QString *, int *> *) event->data())->first = NULL;
			delete (int*) (((QPair<std::string *, int *> *) event->data())->second);
			((QPair<std::string *, int *> *) event->data())->second = NULL;
			delete (QPair<std::string *, int *> *) (event->data());
			break;
		}
		case MESSAGE_ERROR:
		{
			QString message = *((QPair<QString *, QString *> *) event->data())->first;
			QString title = *((QPair<QString *, QString *> *) event->data())->second;
			KMessageBox::error( this, message, title );

			delete (QString*) (((QPair<QString *, QString *> *) event->data())->first);
			((QPair<QString *, QString *> *) event->data())->first = NULL;
			delete (QString*) (((QPair<QString *, QString *> *) event->data())->second);
			((QPair<QString *, QString *> *) event->data())->second = NULL;
			delete ((QPair<QString *, QString *> *) event->data());
			break;
		}
		default:
		{
			kdDebug() << "Unknown custom event caught: " << event->type() << endl;
			break;
		}
	}
}

void KomparatorWidget::slotFileDownloaded( KIO::Job *job )
{
	if ( m_requested_download )
	{
		if ( job->error() != 0 ) job->showErrorDialog( this );

		m_requested_download->setDownloadError( job->error() );
	}
	else // no m_requested_download ??? that means, something is terribly wrong. might happen on "cancel". trying to delete temp file anyway.
	{
		KIO::NetAccess::del( ((KIO::FileCopyJob*)job)->destURL().url(), this );
		KIO::NetAccess::del( ((KIO::FileCopyJob*)job)->destURL().url() + ".part", this ); // FIXME: ugly.
	}

	m_requested_download = NULL;
	m_requested_download_job = NULL;
}

#include "komparatorwidget.moc"
