/***************************************************************************
                          addemoticondialog.cpp - shows a dialog to add a custom emoticon
                             -------------------
    begin                : Tue April 10 2007
    copyright            : (C) 2007 by Valerio Pilo
    email                : amroth@coldshock.net
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "addemoticondialog.h"

#include <qimage.h>
#include <qlabel.h>
#include <qlineedit.h>
#include <qpixmap.h>
#include <qpushbutton.h>

#include <kdebug.h>
#include <klocale.h>
#include <kfiledialog.h>
#include <kiconloader.h>
#include <kmessagebox.h>

#include "../emoticonmanager.h"
#include "addemoticondialoginterface.h"



#ifdef KMESSDEBUG_EMOTICONS
#define KMESSDEBUG_EMOTICONS_ADDING
#endif



/**
 * Constructor
 *
 * The dialog is instantly shown as non-modal when the class is instantiated.
 *
 * @param theme   Theme to add the new emoticon to
 * @param parent  Parent widget
 * @param name    Qt object name
 */
AddEmoticonDialog::AddEmoticonDialog( EmoticonTheme *theme, QWidget *parent, const char *name )
 : KDialogBase( parent, name,                     // parent, qobject name
                false, i18n("Add New Emoticon"),  // modal, caption
                ( Help | Ok | Cancel ), Ok,       // buttons, defaultbutton
                false )                           // bottom separator
 , theme_(theme)
{
  // Create a new dialog interface
  interface_ = new AddEmoticonDialogInterface( this, "AddEmoticonDialogInterface" );
  setMainWidget( interface_ );

  // Connect the interface's signals to manage its events
  connect( interface_->browseButton_,  SIGNAL(          clicked()               ),
           this,                         SLOT(    choosePicture()               ) );
  connect( interface_->pictureEdit_,   SIGNAL(      textChanged(const QString&) ),
           this,                         SLOT( interfaceChanged()               ) );
  connect( interface_->shortcutEdit_,  SIGNAL(      textChanged(const QString&) ),
           this,                         SLOT( interfaceChanged()               ) );

  // Force an update of the widgets (mainly to show the KMess logo as
  // the preview and to disable the Ok button)
  interfaceChanged();

  // Show the dialog, as small as possible
  setInitialSize( minimumSizeHint(), true );
  show();
}



/**
 * Destructor
 */
AddEmoticonDialog::~AddEmoticonDialog()
{
#ifdef KMESSDEBUG_EMOTICONS_ADDING
  kdDebug() << "AddEmoticonDialog::~AddEmoticonDialog() - DESTROYED AddEmoticonDialog." << endl;
#endif
}



/**
 * Shows a File Selection dialog to choose an image for the new emoticon
 *
 * Gets the emoticon picture location from the user, using a KFile dialog, putting the pictureEdit_ widget contents
 * as a default, and then replaces the contents of said widget with the result of the file selection dialog.
 */
void AddEmoticonDialog::choosePicture()
{
  QString  file;
  QImage   emoticonData;

  // Choose a file, filtering out all files but the preselected image types
  file = KFileDialog::getOpenFileName( interface_->pictureEdit_->text(), "image/png image/mng image/gif image/jpeg", this );

  if( file.isEmpty() )
  {
    return;
  }

  // Update the text widget, so an update will be issued and the preview will be updated.
  interface_->pictureEdit_->setText( file );
}



/**
 * Controls the OK button by checking the dialog's widgets
 *
 * It's called everytime that a change is made to one of the dialog's widgets, and checks if their contents are valid.
 * Then enables or disables the OK button accordingly.
 */
void AddEmoticonDialog::interfaceChanged()
{
  QImage emoticonData;
  emoticonData.load( interface_->pictureEdit_->text() );

  if( ! emoticonData.isNull() )
  {
    // The chosen file is valid, update the preview.

#ifdef KMESSDEBUG_EMOTICONS_ADDING
    kdDebug() << "AddEmoticonDialog::interfaceChanged() - Updated preview." << endl;
#endif

    interface_->emoticonPreview_->setPixmap( emoticonData );
  }
  else if( interface_->emoticonPreview_->pixmap() == 0
        || QPixmap( emoticonData ).serialNumber() != interface_->emoticonPreview_->pixmap()->serialNumber() )
  {
    // The chosen file is not valid, reset the preview to the KMess logo.

#ifdef KMESSDEBUG_EMOTICONS_ADDING
    kdDebug() << "AddEmoticonDialog::interfaceChanged() - Resetting preview." << endl;
#endif

    interface_->emoticonPreview_->setPixmap( KGlobal::iconLoader()->loadIcon( "kmess", KIcon::Desktop, KIcon::SizeHuge ) );
  }

  // Enable the OK button only if both fields have valid contents
  enableButtonOK( ! emoticonData.isNull() && ! interface_->shortcutEdit_->text().isEmpty() );
}



/**
 * Preselects a file name and an emoticon shortcut in the dialog
 *
 * @param pictureName  Full path to the file that will be added
 * @param code         Emoticon shortcut to preselect
 */
void AddEmoticonDialog::preSelect( QString pictureName, QString shortcut )
{
  // Select the image. Disable the field also to avoid messing up with the name.
  interface_->pictureEdit_->setText( pictureName );
  interface_->pictureEdit_->setEnabled( false );
  interface_->browseButton_->setEnabled( false );

  // Select the shortcut, and save it so later we can tell everyone we've added this particular emoticon
  preselectedShortcut_ = shortcut;

  // HTML-Decode the shortcut, more thoroughly as possible, to show it to the user like the original one was.
  // Neither KDE nor Qt have HTML to text decoding methods. Annoying.
  shortcut.replace( "&#34;", "\"" )
          .replace( "&#39;", "'" )
          .replace( "&gt;", ">" )
          .replace( "&lt;", "<" )
          .replace( "&amp;", "&" );

  // Select the shortcut
  interface_->shortcutEdit_->setText( shortcut );
}




/**
 * Adds the new emoticon to the theme
 *
 * First it copies the user-selected picture to the custom theme's folder, then adds it to the theme itself;
 * finally it closes the dialog.
 */
void AddEmoticonDialog::slotOk()
{
  QImage emoticonData;
  QString shortcut, pictureName;

#ifdef KMESSDEBUG_EMOTICONS_ADDING
  kdDebug() << "AddEmoticonDialog::slotOk() - OK button was pressed!" << endl;
#endif

  shortcut = interface_->shortcutEdit_->text();
  pictureName = interface_->pictureEdit_->text();

  emoticonData.load( pictureName );

  if( emoticonData.isNull() || shortcut.isEmpty() )
  {
    kdWarning() << "AddEmoticonDialog::slotOk() - Shortcut or picture is not valid!" << endl;
    return;
  }

  // If the emoticon does already exist in the theme, ask if you want to replace it.
  if( theme_->contains( shortcut ) )
  {
    int result = KMessageBox::questionYesNo( 0,
                                             i18n("The emoticon \"%1\" already exists, do you want to replace it?").arg( shortcut ),
                                             i18n("Add New Emoticon") );
    if ( result == KMessageBox::Yes )
    {
      theme_->removeEmoticon( shortcut );
    }
    else
    {
      return;
    }
  }

  // Get handles for the original picture and the new emoticon
  QFileInfo pictureInfo( pictureName );
  QFile originalPicture( pictureName );
  QFile newThemePicture( theme_->getThemePath() + pictureInfo.fileName() );

  // Open the two images
  if( ! originalPicture.open( IO_ReadOnly ) )
  {
    kdWarning() << "AddEmoticonDialog::slotOk() - Couldn't open source file: " << originalPicture.name() << endl;
    return;
  }

  if( ! newThemePicture.open( IO_WriteOnly ) )
  {
    kdWarning() << "AddEmoticonDialog::slotOk() - Couldn't open destination file: " << newThemePicture.name() << endl;
    return;
  }

  // Copy the file contents
  newThemePicture.writeBlock( originalPicture.readAll() );
  newThemePicture.close();
  originalPicture.close();

  // Quick validity check
  if( newThemePicture.size() != originalPicture.size() )
  {
    kdWarning() << "AddEmoticonDialog::slotOk() - File sizes do not match: "
                <<           originalPicture.name() << " (" << originalPicture.size() << ")"
                << " != " << newThemePicture.name() << " (" << newThemePicture.size() << ")" << endl;
    return;
  }


#ifdef KMESSDEBUG_EMOTICONS_ADDING
   kdDebug() << "AddEmoticonDialog::slotOk() - Adding new emoticon: " << shortcut << "." << endl;
#endif

  // We've just created a file whose name is like "file.name.png", so give to the theme a name "file.name" so it can guess
  // the file type on its own.
  theme_->addEmoticon( pictureInfo.baseName( true ), QStringList( shortcut ) );
  theme_->saveTheme();

  // If a shortcut has been preselected, signal that we've added that emoticon.
  // If it hasn't, just send the new shortcut.
  if( ! preselectedShortcut_.isEmpty() )
  {
    shortcut = preselectedShortcut_;
  }

  // Signal to anyone interested that we've added this emoticon
  emit addedEmoticon( shortcut );

  KDialogBase::slotOk();
}



#include "addemoticondialog.moc"
