/***************************************************************************
                          contactsidebar.cpp  -  description
                             -------------------
    begin                : Thu Jan 16 2003
    copyright            : (C) 2003 by Mike K. Bennett
    email                : mkb137b@hotmail.com
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#include "contactsidebar.h"

#include <qglobal.h>
#include <qlayout.h>
#include <qobjectlist.h>

#include <kdebug.h>

#include "../contact/contactbase.h"
#include "../contact/contactdrag.h"
#include "../currentaccount.h"
#include "../kmessdebug.h"
#include "contactframe.h"

// The constructor
ContactSidebar::ContactSidebar(QWidget *parent, const char *name )
 : QWidget(parent, name)
{
  contactFrames_.setAutoDelete( true );

  setSizePolicy(QSizePolicy::MinimumExpanding, QSizePolicy::MinimumExpanding, false);  // Stretch widget

  // Create a layout within the scrollview so that added widgets are laid out nicely
  viewBox_ = new QVBoxLayout( this, 0, -1, "viewBox" );
//  viewBox_->setResizeMode( QLayout::Minimum );

  // Configure this QScrollView
  setAcceptDrops( true );                       // Allow drag&drop

  // put the viewBoxHolder (and hence the layout holding the frames) into the viewport
  viewBox_->addItem( new QSpacerItem(20, 20, QSizePolicy::Ignored, QSizePolicy::Expanding) );
}



// The destructor
ContactSidebar::~ContactSidebar()
{
}



// A contact joined the chat
void ContactSidebar::contactJoined( QString handle, QString )
{
  ContactBase *contact;
  ContactFrame *contactFrame;
  // See if the contact already has a frame
  contactFrame = getContactFrameByHandle( handle );
  if ( contactFrame != 0 )
  {
    // Reactivate the contact's frame
    contactFrame->reactivate();
    return;
  }

  // Get the first available contact frame
  contactFrame = getFirstInactiveContactFrame();
  if( KMESS_NULL(contactFrame) ) return;

  // Get the given contact
  contact = CurrentAccount::instance()->getContactByHandle( handle );

  if( KMESS_NULL(contact) ) return;

  // Activate the frame
  contactFrame->activate( contact );
}



// A contact left the chat
void ContactSidebar::contactLeft( QString handle, bool /*isChatIdle*/ )
{
  ContactFrame *contactFrame = getContactFrameByHandle( handle );

  if ( contactFrame == 0 )
  {
    return;
  }

  // Delete the frames whenever there is more than one; when only one is left, grey it out instead.
  if( contactFrames_.count() > 1 )
  {
    contactFrames_.remove( contactFrame );
    return;
  }

  // deactivate (grey out) the contact frame
  contactFrame->deactivate();
}



// A contact is typing
void ContactSidebar::contactTyping(QString handle, QString)
{
  ContactFrame *contactFrame = getContactFrameByHandle( handle );
  if ( contactFrame != 0 )
  {
    contactFrame->startTyping();
  }
}



// do we want this drop?
void ContactSidebar::contentsDropEvent( QDropEvent *event )
{
  if ( event->provides( ContactDragInfo::mimeType ) )
  {
    const ContactBase *contact = CurrentAccount::instance()->getContactLastDragged();
    if(contact != 0)
    {
      emit inviteContact( contact->getHandle() );
    }
  }

  event->ignore();
}



// something being dragged has entered our widget
void ContactSidebar::contentsDragEnterEvent( QDragEnterEvent *event )
{
  if ( event->provides( ContactDragInfo::mimeType ) )
  {
    event->accept( true );
  }
}



// something being dragged has been moved in our widget
void ContactSidebar::contentsDragMoveEvent( QDragMoveEvent *event )
{
  if ( event->provides( ContactDragInfo::mimeType ) )
  {
    event->acceptAction( true );
  }
}



// Find the contact frame with the given handle
ContactFrame* ContactSidebar::getContactFrameByHandle(const QString& handle)
{
  for ( ContactFrame *contactFrame = contactFrames_.first(); contactFrame; contactFrame = contactFrames_.next() )
  {
    if ( contactFrame->getHandle() == handle )
    {
      return contactFrame;
    }
  }
  return 0;
}



// Return the first inactive contact frame
ContactFrame* ContactSidebar::getFirstInactiveContactFrame()
{
  ContactFrame *contactFrame;
  for ( ContactFrame *contactFrame = contactFrames_.first(); contactFrame; contactFrame = contactFrames_.next() )
  {
    if ( !contactFrame->isActivated() )
    {
      return contactFrame;
    }
  }

  // Well we don't have any more, so make another.
  contactFrame = new ContactFrame( this, "ContactFrame" );

  connect( contactFrame,   SIGNAL( setContactAllowed( QString )  ),
           this,           SIGNAL( setContactAllowed( QString )  ) );
  connect( contactFrame,   SIGNAL(   setContactAdded( QString, bool )  ),
           this,           SIGNAL(   setContactAdded( QString, bool )  ) );
  connect( contactFrame,   SIGNAL( setContactBlocked( QString, bool )  ),
           this,           SIGNAL( setContactBlocked( QString, bool )  ) );

  // Resize it, and add it to the viewBox so it appears in the sidebar
  contactFrame->setSizePolicy(QSizePolicy::Ignored, QSizePolicy::Preferred, false);
  viewBox_->insertWidget( children()->count() - 2, contactFrame );

  // put it in the list of frames so we can find it again.
  contactFrames_.append( contactFrame );

  return contactFrame;
}



// Initialize the object
bool ContactSidebar::initialize()
{
  // The important frame-creating code has to go in the constructor, so there's less to initialize, here.

  return true;
}



// A message was received from one of the contacts... notify its frame
void ContactSidebar::messageReceived(const QString& handle)
{
  ContactFrame *frame;
  frame = getContactFrameByHandle( handle );
  if ( frame != 0 )
  {
    frame->messageReceived();
  }
}


#include "contactsidebar.moc"
