/***************************************************************************
                          chatmessagestyle.h -  description
                             -------------------
    begin                : Sat Okt 29 2005
    copyright            : (C) 2005 by Diederik van der Boor
    email                : "vdboor" --at-- "codingdomain.com"
 ***************************************************************************/

/***************************************************************************
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 ***************************************************************************/

#ifndef CHATMESSAGESTYLE_H
#define CHATMESSAGESTYLE_H

#include "chatmessage.h"

#include <qobject.h>
#include <qstring.h>
#include <qptrlist.h>
#include <qfont.h>
#include <qstringlist.h>

class XslTransformation;


/**
 * Utility class to convert chat messages to the current style/appearance.
 *
 * @author Diederik van der Boor
 */
class ChatMessageStyle : public QObject
{
  Q_OBJECT

  public:  // public methods

    // The constructor
                         ChatMessageStyle();
    // The destructor
    virtual             ~ChatMessageStyle();

    // Convert a chat message to HTML string
    QString              convertMessage(const ChatMessage &message);
    // Convert a group of chat message to HTML string
    QString              convertMessageList(const QPtrList<ChatMessage> &messageList);
    // Convert the message root.
    QString              convertMessageRoot();
    // Return the base folder of the style.
    const QString &      getBaseFolder() const;
    // Return the font used for contact messages, if forced to.
    const QFont&         getContactFont() const;
    // Return the color of the forced contact font.
    const QString&       getContactFontColor() const;
    // Return the css file attached to the stylesheet. Return null if there is none.
    QString              getCssFile() const;
    // Return the name of the style.
    const QString &      getName() const;
    // Return the ID's of inserted <img> tags for the pending emoticons.
    const QStringList &  getPendingEmoticonTagIds() const;
    // Return whether or not to show contact messages in the stored font.
    bool                 getUseContactFont() const;
    // Return whether an style is loaded.
    bool                 hasStyle() const;

  public slots:
    // The the contact font
    void                 setContactFont(const QFont &font);
    // The the contact font color
    void                 setContactFontColor(const QString &fontColor);
    // Set the show time state
    void                 setShowTime(bool showTime);
    // Set the message style, return false if it failed
    bool                 setStyle(const QString &style);
    // Enable or disable contact font overrides
    void                 setUseContactFont(bool useContactFont);
    // Enable or disable emoticons
    void                 setUseEmoticons(bool useEmoticons);
    // Enable or disable font effects
    void                 setUseFontEffects(bool useFontEffects);


  private:  // private methods

    // Convert the message as HTML, fallback method when XML fails.
    QString              createFallbackMessage(const ChatMessage &message);
    // Convert the message as XML.
    QString              createMessageXml(const ChatMessage &message);
    // Escape the strings for use in XML attributes
    QString              escapeAttribute(const QString &value) const;
    // Return whether the given result is empty
    bool                 isEmptyResult( const QString &parsedMessage );
    // Replace the newline characters
    void                 parseBody(QString &body) const;
    // Do some effects characters (ie, bold, underline and italic specials)
    void                 parseEffects(QString &text) const;
    // Parse the font tags.
    void                 parseFont(const ChatMessage &message, QFont &font, QString &color, QString& fontBefore, QString& fontAfter) const;
    // Replace the Messenger Plus characters with HTML markup
    void                 parseMsnPlusString(QString &text) const;
    // Replace the MSN characters with HTML markup
    void                 parseMsnString(QString &text, const QString &handle = QString::null, bool alwaysShowEmoticons = false);
    // Strip the DOCTYPE tag from the message
    QString              stripDoctype( const QString &parsedMessage );


  private:   // private properties
    // The base folder of the style
    QString              baseFolder_;
    // The contact font
    QFont                contactFont_;
    // The contact font color
    QString              contactFontColor_;
    // The last ID of the pending emoticon placeholders (to generate ID's)
    int                  lastPendingEmoticonId_;
    // The name of the style
    QString              name_;
    // The ID's of the pending emoticons
    QStringList          pendingEmoticonTags_;
    // The filename for the pending placeholder image.
    QString              pendingPlaceholder_;
    // Whether the time should be displayed
    bool                 showTime_;
    // Whether the contact font should be used
    bool                 useContactFont_;
    // Whether emoticons should be used
    bool                 useEmoticons_;
    // Whether font effects should be used
    bool                useFontEffects_;
    // The XSL transformation
    XslTransformation   *xslTransformation_;
};


#endif
