# Copyright (c) 2008 Hideki Ikemoto
#
# Permission is hereby granted, free of charge, to any person obtaining
# a copy of this software and associated documentation files (the
# "Software"), to deal in the Software without restriction, including
# without limitation the rights to use, copy, modify, merge, publish,
# distribute, sublicense, and/or sell copies of the Software, and to
# permit persons to whom the Software is furnished to do so, subject to
# the following conditions:
#
# The above copyright notice and this permission notice shall be
# included in all copies or substantial portions of the Software.
#
# THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
# EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
# MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
# NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE
# LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION
# OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
# WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.

class ThreadListView < Qt::Widget
  slots 'show_thread(QListViewItem*)'
  slots 'search()'
  slots 'context_menu(QListViewItem*,const QPoint&,int)'
  slots 'remove_self()'

  SUBJECT_TXT_REGEXP1 = /((\d+)\.cgi),(.*)\((\d+)\)/
  SUBJECT_TXT_REGEXP2 = /((\d+)\.dat)<>(.*)\((\d+)\)/
  COLUMN_NO = 0
  COLUMN_SEARCH = 1
  COLUMN_TITLE = 2
  COLUMN_RESNUM = 3
  COLUMN_READNUM = 4
  COLUMN_OPENTIME = 5
  COLUMN_SPEED = 6
  COLUMN_URL = 7

  POPUP_COPY_URL = 0
  POPUP_COPY_TITLE_AND_URL = 1

  def initialize(parent)
    super(parent)

    main_layout = Qt::VBoxLayout.new(self)
    tool_layout = Qt::HBoxLayout.new(main_layout)
    @search_line = Qt::LineEdit.new(self)
    tool_layout.addWidget(@search_line)
    @search_button = Qt::ToolButton.new(self)
    @search_button.setPixmap(KDE::SmallIcon("find"))
    @search_button.setEnabled(false)
    tool_layout.addWidget(@search_button)
    connect(@search_button, SIGNAL("clicked()"),
            self, SLOT("search()"))
    connect(@search_line, SIGNAL("returnPressed()"),
            self, SLOT("search()"))

    @close_button = Qt::ToolButton.new(self)
    @close_button.setPixmap(KDE::SmallIcon("fileclose"))
    @close_button.setEnabled(false)
    tool_layout.addWidget(@close_button)
    connect(@close_button, SIGNAL("clicked()"), self, SLOT("remove_self()"))

    @list = KDE::ListView.new(self)
    main_layout.addWidget(@list)

    connect(@list, SIGNAL("clicked(QListViewItem*)"),
            self, SLOT("show_thread(QListViewItem*)"))
    connect(@list, SIGNAL("contextMenuRequested(QListViewItem*,const QPoint&,int)"),
            self, SLOT("context_menu(QListViewItem*,const QPoint&,int)"))

    @list.addColumn("No.")
    @list.addColumn(" ")
    @list.addColumn("Thread name")
    @list.addColumn("Res")
    @list.addColumn("Read")
    @list.addColumn("Open time")
    @list.addColumn("Thread speed")
#    @list.addColumn("Thread url")
    @list.setSorting(-1)
    @list.setAllColumnsShowFocus(true)

    @list.header.setStretchEnabled(true, COLUMN_TITLE)

    # search
    @prev_search_text = nil
    @hit_list = []
    @hit_list_index = 0
  end

  def show_board(url)
    return if url == nil

    @board_url = url
    subject_txt_url = url + "subject.txt"

    thread_list = []
    case Util.board_type(url)
    when Util::TYPE_2CH, Util::TYPE_MACHIBBS, Util::TYPE_UNKNOWN
      str = Downloader.download_subject_txt(subject_txt_url)
    when Util::TYPE_JBBS
      str = Downloader.download_subject_txt_jbbs(subject_txt_url)
    end

    str.each {|line|
      if SUBJECT_TXT_REGEXP1.match(line) || SUBJECT_TXT_REGEXP2.match(line)
        dat_file_name = Regexp.last_match(1)
        open_time = Time::at(Regexp.last_match(2).to_i)
        title = Regexp.last_match(3)
        num = Regexp.last_match(4)
        speed = num.to_i / ((Time::now - open_time) / 86400.0)

        dat_url = Util.make_dat_url(@board_url, dat_file_name)
        thread = ThreadInfo.new(dat_url, title, num, open_time, speed)
        thread_list.push(thread)
      end
    }

    @list.clear

    prev_item = nil
    thread_list.each_with_index {|thread, i|
      item = KDE::ListViewItem.new(@list, prev_item)
      item.setText(COLUMN_NO, (i+1).to_s)
      item.setText(COLUMN_TITLE, thread.title)
      item.setText(COLUMN_RESNUM, thread.res_num)
      item.setText(COLUMN_READNUM, thread.read_num) if thread.read_num
      item.setText(COLUMN_OPENTIME, thread.open_time.strftime("%y/%m/%d %H:%M"))
      item.setText(COLUMN_SPEED, thread.speed.to_s)
      item.setText(COLUMN_URL, thread.url)

      prev_item = item
    }

    ViewMediator.set_url(url)
    enable_buttons
  end

  def show_thread(item)
    return if item == nil

    thread_url = item.text(COLUMN_URL)
    return if thread_url == nil

    ViewMediator.show_thread(thread_url)
  end

  def search
    str = @search_line.text
    if str == @prev_search_text
      search_next
    else
      search_new(str)
      @prev_search_text = str
    end
  end

  def search_clear
    @hit_list.clear
    @hit_list_index = 0
    @prev_search_text = nil

    it = Qt::ListViewItemIterator.new(@list)
    while it.current do
      item = it.current
      item.setText(COLUMN_SEARCH, "")
    end
  end

  def search_new(str)
    @hit_list.clear
    @hit_list_index = 0

    it = Qt::ListViewItemIterator.new(@list)
    regexp = Regexp.new(str, Regexp::IGNORECASE)
    while it.current do
      item = it.current
      item.setText(COLUMN_SEARCH, "") # clear
      title = item.text(COLUMN_TITLE)

      if title.match(regexp) then
        @hit_list.push(item)
      end

      it += 1
    end

    @hit_list.each {|item|
      item.setPixmap(COLUMN_SEARCH, KDE::SmallIcon("find"))
    }
  end

  def search_next
    item = @hit_list[@hit_list_index]
    @list.ensureItemVisible(item)
    @list.setSelected(item, true)

    @hit_list_index += 1
    if @hit_list_index >= @hit_list.size then
      @hit_list_index = 0
    end
  end

  def context_menu(item, pos, col)
    return unless item

    dat_url = item.text(COLUMN_URL)
    thread_url = Util.dat_url_to_thread_url(dat_url)
    thread_title = item.text(COLUMN_TITLE)

    popup = KDE::PopupMenu.new
    popup.insertItem("Copy URL", POPUP_COPY_URL)
    popup.insertItem("Copy title and URL", POPUP_COPY_TITLE_AND_URL)

    clipboard = Qt::Application.clipboard
    case popup.exec(pos)
    when POPUP_COPY_URL
      clipboard.setText(thread_url, Qt::Clipboard::Clipboard)
      clipboard.setText(thread_url, Qt::Clipboard::Selection)
    when POPUP_COPY_TITLE_AND_URL
      cliptext = thread_title + "\n" + thread_url
      clipboard.setText(cliptext, Qt::Clipboard::Clipboard)
      clipboard.setText(cliptext, Qt::Clipboard::Selection)
    end
  end

  def remove_self
    ViewMediator::remove_thread_list_view(self)
  end

  def board_url
    @board_url
  end

  def enable_buttons
    @search_button.setEnabled(true)
    @close_button.setEnabled(true)
  end
end
