#ifndef KP_TRAINING_LOG_
#define KP_TRAINING_LOG_

#include <libxml/tree.h>

#include <stdlib.h>
#include <stdio.h>

#include <glib-object.h>
#include <glib.h>

#include "kputil.h"
#include "kpworkout.h"

typedef enum
{
  KP_TRAINING_LOG_SAVE_HTML_ERROR,
  KP_TRAINING_LOG_SAVE_ERROR,
  KP_TRAINING_LOG_OPEN_ERROR,
  KP_TRAINING_LOG_INTERNAL_ERROR
} KPTrainingLogError;


typedef enum {
  KP_TRAINING_LOG_DATA_SUM,
  KP_TRAINING_LOG_DATA_AVG,
  KP_TRAINING_LOG_DATA_MIN,
  KP_TRAINING_LOG_DATA_MAX
} KPTrainingLogDataMode;

typedef struct KPSportEntryData_
{
  gchar *name;
  guint n;
} KPSportEntryData;

typedef struct KPInfoEntry_
{
  gchar *name;
  gchar *data;
} KPInfoEntry;

typedef struct KPTrainingLog_
{
  GObject         parent_instance;

  const gchar    *title;
  gchar          *file;

  GList          *year_list;
  guint           n_workouts;
  guint           n_entries;
  GHashTable     *id_table;
  
  GSList         *info_list;
  GSList         *sports;

  gboolean        changed;
  gboolean        written_to_disk;  
} KPTrainingLog;

typedef struct KPTrainingLogClass_
{
  GObjectClass    parent_class;

  void            (*saved) (KPTrainingLog *);
  void            (*changed) (KPTrainingLog *);
  void            (*entry_added) (KPTrainingLog *);
  void            (*new_entry_added) (KPTrainingLog *);
  void            (*entry_changed) (KPTrainingLog *);
  void            (*entry_removed) (KPTrainingLog *);
  void            (*sport_list_changed) (KPTrainingLog *);
  void            (*destroy) (KPTrainingLog *);
  
} KPTrainingLogClass;

#define KP_TYPE_TRAINING_LOG (kp_training_log_get_type())
#define KP_TRAINING_LOG(object) (G_TYPE_CHECK_INSTANCE_CAST ((object), KP_TYPE_TRAINING_LOG, KPTrainingLog))
#define KP_TRAINING_LOG_CLASS(klass) (G_TYPE_CHECK_CLASS_CAST ((klass), KP_TYPE_TRAINING_LOG, KPTrainingLogClass))
#define KP_IS_TRAINING_LOG(object) (G_TYPE_CHECK_INSTANCE_TYPE ((object), KP_TYPE_TRAINING_LOG))
#define KP_IS_TRAINING_LOG_CLASS(klass) (G_TYPE_CHECK_CLASS_TYPE ((klass), KP_TYPE_TRAINING_LOG))
#define KP_TRAINING_LOG_GET_CLASS(object) (G_TYPE_INSTANCE_GET_CLASS ((object), KP_TYPE_TRAINING_LOG, KPTrainingLogClass))


typedef struct TYear_
{
  guint           year;

  GList          *workouts[12];
  guint           n_workouts;
  guint           n_entries;
}
TYear;

typedef TYear KPTrainingLogYear;


GQuark          kp_training_log_error_quark             (void);
#define         KP_TRAINING_LOG_ERROR kp_training_log_error_quark()

GType           kp_training_log_get_type                (void);
KPTrainingLog  *kp_training_log_new                     (void);
KPTrainingLog  *kp_training_log_new_from_file           (const gchar *file,
                                                         GError **err);
/* Destroy */
void            kp_training_log_destroy                 (KPTrainingLog *log);

/* Save */
gboolean        kp_training_log_save                    (KPTrainingLog *log,
                                                         const gchar *file,
                                                         GError **error);
gboolean        kp_training_log_create_html_stats       (KPTrainingLog *log,
                                                         const gchar *output_dir,
                                                         GError ** error);
/* Modify */
gboolean        kp_training_log_add                     (KPTrainingLog *log,
                                                         KPCalendarEntry *entry);
gboolean        kp_training_log_add_from_file           (KPTrainingLog *log,
                                                         const gchar *file,
                                                         GError **error);
gboolean        kp_training_log_remove                  (KPTrainingLog *log,
                                                         KPCalendarEntry *entry);
void            kp_training_log_remove_mark             (KPTrainingLog *log,
                                                         guint d, guint m,
                                                         guint y,
                                                         const gchar *mark);
void            kp_training_log_remove_month            (KPTrainingLog *log,
                                                         guint y, guint m);
void            kp_training_log_remove_day              (KPTrainingLog *log,
                                                         guint d, guint m,
                                                         guint y);

void            kp_training_log_remove_all              (KPTrainingLog *log);

/* Get */
GSList          *kp_training_log_get_sports             (KPTrainingLog *log);
GSList          *kp_training_log_get_info               (KPTrainingLog *log);
G_CONST_RETURN gchar *
                 kp_training_log_get_info_entry         (KPTrainingLog *log,
                                                         const gchar *field);
void             kp_training_log_add_info_entry         (KPTrainingLog *log,
                                                         const gchar *field,
                                                         const gchar *data);
KPCalendarEntry *kp_training_log_get_entry              (KPTrainingLog *log,
                                                         guint d, guint m,
                                                         guint y,
                                                         const gchar *mark);
guint           kp_training_log_get_n_years             (KPTrainingLog *log);
TYear          *kp_training_log_get_year                (KPTrainingLog *log,
                                                         guint year);

guint           kp_training_log_get_size                (KPTrainingLog *log);

GList          *kp_training_log_get_all_entries         (KPTrainingLog *log);
GList          *kp_training_log_get_all_entries_between (KPTrainingLog *log,
                                                         GDate *start,
                                                         GDate *end,
                                                         GList **sports);
gdouble       **kp_training_log_get_workout_params_between
                                                        (KPTrainingLog *log,
                                                         KPTrainingLogDataMode
                                                         mode,
                                                         const gchar *sport,
                                                         const gchar *param_name,
                                                         GDate *start,
                                                         GDate *end,
                                                         guint *days_between);
gdouble       **kp_training_log_get_workout_params_year (KPTrainingLog *log,
                                                         const gchar *param,
                                                         guint year,
                                                         guint *days_between);
KPDate         *kp_training_log_get_earliest_date       (KPTrainingLog *log);
KPDate         *kp_training_log_get_latest_date         (KPTrainingLog *log);
GList          *kp_training_log_get_day                 (KPTrainingLog *log,
                                                         guint d, guint m,
                                                         guint y);
GList          *kp_training_log_get_month               (KPTrainingLog *log,
                                                         gint year,
                                                         gint month);
gboolean        kp_training_log_is_modified             (KPTrainingLog *log);
void            kp_training_log_set_filename            (KPTrainingLog *log,
                                                         const gchar *file);
G_CONST_RETURN gchar *
                kp_training_log_get_filename            (KPTrainingLog *log);
gboolean        kp_training_log_is_written_to_disk      (KPTrainingLog *log);
gdouble       **kp_training_log_get_month_as_chart_data (KPTrainingLog *log,
                                                         guint year,
                                                         guint month,
                                                         guint month_len,
                                                         const gchar *param);
gdouble         kp_training_log_get_param_day           (KPTrainingLog *log,
                                                         GDate *date,
                                                         const gchar *param_name);
void            kp_training_log_foreach                 (KPTrainingLog *log,
                                                         GFunc func,
                                                         gpointer data);
/* Debug */
void            kp_training_log_dump                    (KPTrainingLog *log);

#endif
