#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <errno.h>
#include <time.h>

#include <libxml/parser.h>
#include <libxml/tree.h>

#include <glib-object.h>
#include <glib.h>

#include "kpcalendarentry.h"
#include "kpsplitworkout.h"
#include "kpworkoutmodel.h"
#include "kptraininglog.h"
#include "kppresetdata.h"
#include "kpmarshalers.h"
#include "kipina-i18n.h"
#include "kpsettings.h"
#include "kpresults.h"
#include "kpcomment.h"
#include "kpworkout.h"
#include "kparray2d.h"
#include "kputil.h"

enum
{
  SAVED_SIGNAL,
  CHANGED_SIGNAL,
  NEW_ENTRY_ADDED_SIGNAL,
  ENTRY_REMOVED_SIGNAL,
  ENTRY_CHANGED_SIGNAL,
  SPORT_LIST_CHANGED_SIGNAL,
  DESTROY_SIGNAL,
  LAST_SIGNAL
};

static guint      kp_training_log_signals[LAST_SIGNAL] = { 0 };

static
KPTrainingLog    *kp_training_log_parse_from_xml    (const gchar *filename,
                                                     KPTrainingLog *log_,
                                                     GError **error);
xmlDocPtr         kp_training_log_to_xml            (KPTrainingLog *log);
static void       kp_training_log_year_to_xml       (gpointer val,
                                                     gpointer data);
static void       list_add_item_to_passed_list      (gpointer val,
                                                     gpointer data);
static void       kp_training_log_class_init        (GObjectClass *klass,
                                                     gpointer data);
static void       kp_training_log_instance_init     (GObject *object,
                                                     gpointer data);
static void       kp_training_log_instance_finalize (GObject *object);
static void       kp_training_log_entry_changed     (KPCalendarEntry *entry,
                                                     const gchar *old_string,
                                                     KPTrainingLog *log);

/* Handling sports */
GSList           *find_sport                        (KPTrainingLog *log,
                                                     const gchar *sport);

static GString *last_error = NULL;


GType
kp_training_log_get_type ()
{
  static GType kp_training_log_type = 0;

  if (!kp_training_log_type) {
    static const GTypeInfo kp_training_log_info = {
      sizeof (KPTrainingLogClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_training_log_class_init,
      NULL,
      NULL,
      sizeof (KPTrainingLog),
      0,
      (GInstanceInitFunc) kp_training_log_instance_init,
      NULL
    };

    kp_training_log_type = g_type_register_static (G_TYPE_OBJECT, "KPTrainingLog",
                                                &kp_training_log_info, 0);
  }
  return kp_training_log_type;
}


void
kp_training_log_xml_structured_error_func (void *ctx, xmlErrorPtr err)
{
  kp_debug ("XML2 error (%d): %s", err->level, err->message);
  g_string_assign (last_error, err->message);
}


static void
kp_training_log_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_training_log_instance_finalize;

  kp_training_log_signals[SAVED_SIGNAL]
    = g_signal_new ("saved",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, saved),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__VOID,
                    G_TYPE_NONE,
                    0);

  kp_training_log_signals[CHANGED_SIGNAL]
    = g_signal_new ("changed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, changed),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__VOID,
                    G_TYPE_NONE,
                    0);
  kp_training_log_signals[NEW_ENTRY_ADDED_SIGNAL]
    = g_signal_new ("new-entry-added",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, new_entry_added),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__POINTER,
                    G_TYPE_NONE,
                    1,
                    G_TYPE_POINTER);
  kp_training_log_signals[ENTRY_REMOVED_SIGNAL]
    = g_signal_new ("entry-removed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, entry_removed),
                    NULL,
                    NULL,
                    g_cclosure_user_marshal_VOID__UINT_UINT_UINT_STRING,
                    G_TYPE_NONE,
                    4,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_STRING);
  
  kp_training_log_signals[ENTRY_CHANGED_SIGNAL]
    = g_signal_new ("entry-changed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, entry_changed),
                    NULL,
                    NULL,
                    g_cclosure_user_marshal_VOID__UINT_UINT_UINT_STRING_STRING,
                    G_TYPE_NONE,
                    5,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_UINT,
                    G_TYPE_STRING,
                    G_TYPE_STRING);
  
  kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL]
    = g_signal_new ("sport-list-changed",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, sport_list_changed),
                    NULL,
                    NULL,
                    g_cclosure_user_marshal_VOID__BOOLEAN_STRING,
                    G_TYPE_NONE,
                    2,
                    G_TYPE_BOOLEAN,
                    G_TYPE_STRING);

  kp_training_log_signals[DESTROY_SIGNAL]
    = g_signal_new ("destroy",
                    G_OBJECT_CLASS_TYPE (object_class),
                    G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                    G_STRUCT_OFFSET (KPTrainingLogClass, destroy),
                    NULL,
                    NULL,
                    g_cclosure_marshal_VOID__VOID,
                    G_TYPE_NONE,
                    0);

  if (last_error == NULL)
    last_error = g_string_new (NULL);

  xmlSetStructuredErrorFunc (NULL, kp_training_log_xml_structured_error_func);
}

static void
kp_training_log_instance_init (GObject *object, gpointer data)
{
  KPTrainingLog *log;

  log = KP_TRAINING_LOG (object);

  log->info_list = NULL;
  log->sports = NULL;
  log->year_list = NULL;
  log->n_workouts = 0;
  log->n_entries = 0;
  log->id_table = g_hash_table_new (g_direct_hash, g_direct_equal);

  log->changed = FALSE;
  log->written_to_disk = FALSE;
  log->file = NULL;
}


static void
kp_training_log_instance_finalize (GObject *object)
{
  KPTrainingLog *log = KP_TRAINING_LOG (object);
  GObjectClass *parent_class;

  g_signal_emit (object, kp_training_log_signals[DESTROY_SIGNAL], 0);
  kp_debug ("Finalized KPTrainingLog(%p) with %u entries.\n", log,
             kp_training_log_get_size (log));
 
  kp_training_log_remove_all (log);
  
  log->title = NULL;
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}



/**
 * kp_training_log_new:
 * 
 * Create a new instance of #KPTrainingLog.
 *
 * Returns: A #KPTrainingLog
 */
KPTrainingLog *
kp_training_log_new (void)
{
  KPTrainingLog *log;

  log = g_object_new (kp_training_log_get_type (), NULL);

  return log;
}


/**
 * kp_training_log_get_filename:
 * @log: A #KPTrainingLog
 *
 * Get filename that training log lives or is to be saved.
 * 
 * Returns: const string that is owned by log or NULL if the log
 * is unnamed.
 */
G_CONST_RETURN gchar *
kp_training_log_get_filename (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  return log->file;
}



/**
 * kp_training_log_set_filename:
 * @log: A #KPTrainingLog
 * @file: The filename
 *
 * Set the filename that the log will be saved when requested.
 */
void
kp_training_log_set_filename (KPTrainingLog *log, const gchar *file)
{
  if (file == log->file)
    return;
  
  if (log->file != NULL)
    g_free (log->file);
    
  log->file = (file != NULL) ? g_strdup (file) : NULL;
}


/**
 * kp_training_log_is_written_to_disk:
 * @log: A #KPTrainingLog
 * 
 * This function checks if this log is loaded from a file or
 * it has been saved to the disk while running the program.
 * This is mainly intented for use by save routines.
 * 
 * Returns: TRUE if the log is loaded from the file or has
 * been already saved to the file.
 */
gboolean
kp_training_log_is_written_to_disk (KPTrainingLog *log)
{
  return log->written_to_disk;
}


KPTrainingLog *
kp_training_log_new_from_file (const gchar *file, GError **err)
{
  GError *tmp_error = NULL;
  KPTrainingLog *log;

  log = kp_training_log_parse_from_xml (file, NULL, &tmp_error);

  if (!log) {
    g_propagate_error (err, tmp_error);
    return NULL;
  }

  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);
  log->changed = FALSE; 
  
  return log;
}

/**
 * kp_training_log_add_from_file:
 * @log: A #KPTrainingLog
 * @file: The name of the log file
 *
 * Add all the entries from the log specified by the @file to the
 * existing log.
 *
 * Returns: TRUE if successful and FALSE otherwise.
 */
gboolean
kp_training_log_add_from_file (KPTrainingLog *log, const gchar *file,
                               GError **error)
{
  GError *tmp_error;
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), FALSE);
  g_return_val_if_fail (file != NULL, FALSE);
  
  tmp_error = NULL;
  if (!kp_training_log_parse_from_xml (file, log, &tmp_error)) {
    g_propagate_error (error, tmp_error);
    return FALSE;
  }
 
  return TRUE;
}


GQuark
kp_training_log_error_quark (void)
{
  static GQuark quark = 0;

  if (!quark)
    quark = g_quark_from_static_string ("kp-training-log-error-quark");
  return quark;
}


void
kp_training_log_destroy (KPTrainingLog * log)
{
  g_return_if_fail (log != NULL);

  /*g_hash_table_foreach (log->years, destroy_year, NULL);
     g_hash_table_destroy (log->years); */

  g_list_free (log->year_list);
}


gboolean
kp_training_log_is_modified (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), FALSE);
  return log->changed;
}


/**
 * kp_training_log_save:
 * @log: A #KPTrainingLog
 * @file: File to save log into
 *
 * Just save the contents of the log to the file specified by
 * the @file.
 *
 * Returns: TRUE if successful or FALSE otherwise.
 */
gboolean
kp_training_log_save (KPTrainingLog *log, const gchar *file, GError **error)
{
  xmlDocPtr doc;
  gboolean ret_val;
  int val;

  g_return_val_if_fail (log != NULL, FALSE);
  g_return_val_if_fail (file != NULL, FALSE);
  g_return_val_if_fail (error == NULL || *error == NULL, FALSE);

  doc = kp_training_log_to_xml (log);

  g_return_val_if_fail (doc != NULL, FALSE);

  val = xmlSaveFormatFile (file, doc, 1);

  kp_debug ("xmlSaveFormatFile for '%s' returned %d (bytes).", file, val);
  
  if (val >= 0) {  
    ret_val = TRUE;
  }
  else {
    kp_debug ("Setting error!: %s", last_error->str);
   
    g_set_error (error,
                 KP_TRAINING_LOG_ERROR,
                 KP_TRAINING_LOG_SAVE_ERROR,
               _("%s"), last_error->str);

    ret_val = FALSE;
  }

  xmlFreeDoc (doc);

  if (ret_val) {
    kp_debug ("Set filename: %s.", file);
    kp_training_log_set_filename (log, file);
    log->written_to_disk = TRUE;
    log->changed = FALSE;
    g_signal_emit (G_OBJECT (log), kp_training_log_signals[SAVED_SIGNAL], 0);
  }
  
  return ret_val;
}


/**
 * kp_training_log_add:
 * @log: A #KPTrainingLog
 * @entry: A #KPCalendarEntry to add to the log.
 *
 * Just add the calendar entry to the log.
 *
 * Returns: TRUE if the entry was added and FALSE if some error occurred.
 */
gboolean
kp_training_log_add (KPTrainingLog *log, KPCalendarEntry *entry)
{
  KPSportEntryData *s_entry;
  GSList *list;
  KPDate date;
  TYear *ty;
  const gchar *str;
  gint i;

  g_return_val_if_fail (log != NULL, FALSE);
  g_return_val_if_fail (entry != NULL, FALSE);

  kp_calendar_entry_get_date (entry, &date);

  kp_debug ("Adding workout to log, date: %u.%u.%u", date.d, date.m, date.y);
  g_return_val_if_fail (g_date_valid_dmy (date.d, date.m, date.y), FALSE);
  
  ty = kp_training_log_get_year (log, date.y);

  /* If there is no year in list, add it */
  if (ty == NULL) {
    ty = g_new (TYear, 1);
    ty->year = date.y;

    log->year_list = g_list_append (log->year_list, ty);

    for (i = 0; i < 12; i++)
      ty->workouts[i] = NULL;

    ty->n_workouts = 0;
    ty->n_entries = 0;
  }

  g_return_val_if_fail (date.m > 0, FALSE);
  g_return_val_if_fail (date.m <= 12, FALSE);

  if (g_list_find (ty->workouts[date.m - 1], entry)) {
    g_warning ("Trying to add an entry to the log, but the entry is already in the log!");
    g_return_val_if_reached (FALSE);
  }
  
  /* Insert entry to the month list */
  ty->workouts[date.m - 1] = g_list_insert_sorted (ty->workouts[date.m - 1],
                                                   entry,
                                    (GCompareFunc) kp_calendar_entry_cmp);

  log->changed = TRUE;
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[NEW_ENTRY_ADDED_SIGNAL],
                 0, entry);
  
  if (KP_IS_WORKOUT (entry)) {
    ty->n_workouts++;
    log->n_workouts++;

    str = kp_workout_get_sport (KP_WORKOUT (entry));
     
    if (!(list = find_sport (log, str))) {
      s_entry = g_new (KPSportEntryData, 1);
      s_entry->n = 1;
      s_entry->name = g_strdup (str);
        
      log->sports = g_slist_prepend (log->sports, s_entry);
      g_signal_emit (log, kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL], 0,
                     TRUE, str);
    } else {
      ((KPSportEntryData *) list->data)->n++;
    }
  }
  g_hash_table_insert (log->id_table, GINT_TO_POINTER (kp_calendar_entry_get_id (entry)), entry);
  ty->n_entries++;
  log->n_entries++;

  kp_debug ("New entry added to the log with id: %u\n", 
             kp_calendar_entry_get_id (entry));
  
  g_signal_connect (G_OBJECT (entry), "changed",
                    G_CALLBACK (kp_training_log_entry_changed), log);
  
  return TRUE;
}


GSList *
find_sport (KPTrainingLog *log, const gchar *sport)
{
  KPSportEntryData *entry;
  GSList *list;
  gchar *tmp1;
  gchar *tmp2;

  list = log->sports;

  while (list) {
     
    entry = (KPSportEntryData *) list->data;
      
    tmp1 = g_utf8_collate_key (entry->name, -1);
    tmp2 = g_utf8_collate_key (sport, -1);

    if (strcmp (tmp1, tmp2) == 0) {
      g_free (tmp1);
      g_free (tmp2);
      
      return list;
    }
    g_free (tmp1);
    g_free (tmp2);

    list = list->next;
  }

  return NULL;
}


/**
 * kp_training_log_remove:
 * @log: A #KPTrainingLog
 * @entry: A #KPCalendarEntry to remove from the log.
 *
 * Just remove the calendar entry from the log.
 *
 * Returns: TRUE if the entry was removed and FALSE if some error occurred.
 */
gboolean
kp_training_log_remove (KPTrainingLog *log, KPCalendarEntry *entry)
{
  GSList *list;
  const gchar *sport;
  GDate *date;
  TYear *ty;
  const gchar *str;
  guint y;
  guint m;
  guint d;

  g_return_val_if_fail (log != NULL, FALSE);
  g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), FALSE);
  g_return_val_if_fail (entry->datetime != NULL, FALSE);

  date = kp_calendar_time_get_date (entry->datetime);
  y = g_date_get_year (date);
  m = g_date_get_month (date);
  d = g_date_get_day (date);
  ty = kp_training_log_get_year (log, y);
  str = kp_calendar_entry_to_string (entry);
  
  g_return_val_if_fail (ty, FALSE);

  ty->workouts[m-1] = g_list_remove (ty->workouts[m-1], entry);
  
  if (KP_IS_WORKOUT (entry)) {
    ty->n_workouts--;
    log->n_workouts--;

    sport = kp_workout_get_sport (KP_WORKOUT (entry));
    list = find_sport (log, sport);
   
    if (list && ((KPSportEntryData *) list->data)->n-- == 1) {
      g_free (((KPSportEntryData *) list->data)->name);
      log->sports = g_slist_remove (log->sports, list->data);
 
      g_signal_emit (log, kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL], 0,
                     FALSE, sport);
    }
  }
  g_hash_table_remove (log->id_table, GINT_TO_POINTER (kp_calendar_entry_get_id (entry)));
  
  ty->n_entries--;
  log->n_entries--;

  if (ty->n_workouts == 0 && ty->n_entries == 0) {
    kp_debug ("Deleted year from the log!");
    log->year_list = g_list_remove (log->year_list, ty);
    g_free (ty);
    ty = NULL;
  }

  kp_debug ("Removed: %d.%d.%d: %s\n", d, m, y, str);
 
  g_object_set_data (G_OBJECT (log), "entry", entry);
  
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[ENTRY_REMOVED_SIGNAL],
                 0, d, m, y, str);

  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);
  log->changed = TRUE;
  
  g_object_unref (entry);

  return TRUE;
}


static gboolean 
always_true (gpointer key, gpointer val, gpointer data) 
{ 
  return TRUE; 
}


/**
 * kp_training_log_remove_all:
 * @log: A #KPTrainingLog
 *
 * This function removes all the workouts from the log.
 * After running this function, @log will be still ready
 * to use, it just doesn't contain any workouts after that.
 */
void
kp_training_log_remove_all (KPTrainingLog *log)
{
  TYear *ty;
  GSList *s; /* Sport */
  GList *y;  /* Year */
  GList *e;  /* Entry */
  guint m;   /* Month - 1*/
  gchar *sport;

  y = log->year_list;

  while (y)
  {
    ty = (TYear *) y->data;
      
    for (m=0; m < 12; m++) {
      e = ty->workouts[m];

      while (e) {
        g_return_if_fail (KP_IS_CALENDAR_ENTRY (e->data));
        
        if (KP_IS_WORKOUT (e->data)) {
          log->n_workouts--;
          ty->n_workouts--;
        }
        log->n_entries--;
        ty->n_entries--;
        
        g_object_unref (G_OBJECT (e->data));
          
        e = e->next;
      }
      g_list_free (ty->workouts[m]);
    }

    g_return_if_fail (ty->n_workouts == 0);
    g_return_if_fail (ty->n_entries == 0);
    
    g_free (ty);
    
    y = y->next;
  } 

  g_hash_table_foreach_remove (log->id_table, always_true, NULL);
  g_list_free (log->year_list);
  
  
  s = log->sports;

  while (s) {
    sport = ((KPSportEntryData *) s->data)->name;
    g_signal_emit (log, kp_training_log_signals[SPORT_LIST_CHANGED_SIGNAL], 0,
                   FALSE, sport);

    g_free (sport);
    g_free (s->data);
    
    s = s->next;
  }
  g_slist_free (log->sports);
  log->sports = 0;
  
  g_return_if_fail (log->n_workouts == 0);
  g_return_if_fail (log->n_entries == 0);
}




/**
 * kp_training_log_get_sports:
 * @log: A #KPTrainingLog
 * 
 * Get list of all different sports that have entries 
 * in the log.
 * 
 * Returns: List that should NOT be freed. 
 */
GSList *
kp_training_log_get_sports (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);

  return log->sports;
}

/**
 * kp_training_log_get_info:
 * @log: A #KPTrainingLog
 *
 * Returns info that is attached to the @log.
 * 
 * Returns: A list that MUST NOT BE MODIFIED!
 */
GSList *
kp_training_log_get_info (KPTrainingLog *log)
{
  return log->info_list;
}


/**
 * kp_training_log_get_info:
 * @log: A #KPTrainingLog
 * @field: field as a string
 *
 * Returns a string that is attached to field @field.
 * 
 * Returns: A string that MUST NOT BE MODIFIED OR FREED!
 */
G_CONST_RETURN gchar *
kp_training_log_get_info_entry (KPTrainingLog *log, const gchar *field)
{
  GSList *node;
  
  for (node = log->info_list; node; node = node->next)
    if (strcmp (field, ((KPInfoEntry *) node->data)->name) == 0)
      return ((KPInfoEntry *) node->data)->data;

  return NULL;
}


/**
 * kp_training_log_add_info_entry:
 * @log: A #KPTrainingLog
 * @field: The field name
 * @data: The data to attach
 *
 * Adds the information to the list of info entries.
 * The data will be saved to the disk next time the
 * log is saved.
 */
void
kp_training_log_add_info_entry (KPTrainingLog *log, const gchar *field,
                                const gchar *data)
{
  KPInfoEntry *info;
  GSList *node;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (field != NULL);
  g_return_if_fail (data != NULL);
 
  /* Search the list if the entry is there already */
  for (node = log->info_list; node; node = node->next) {
    info = (KPInfoEntry *) node->data;
    
    if (strcmp (info->name, field) == 0) {
      g_free (info->data);
      info->data = g_strdup (data);
      g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL],
                     0);
      return;
    }
  }
  /* Not found, so create a new structure */
  info = g_new0 (KPInfoEntry, 1);
  info->name = g_strdup (field);
  info->data = g_strdup (data);

  log->info_list = g_slist_prepend (log->info_list, info);
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);
  log->changed = TRUE;
}


/**
 * kp_training_log_get_earliest_date:
 * @log: A #KPTrainingLog
 *
 * Get the date of the earliest workout or other entry 
 * that is in the log.
 * 
 * Returns: A newly-allocated KPDate-structure that must
 * be freed.
 */
KPDate *
kp_training_log_get_earliest_date (KPTrainingLog *log)
{
  GList *list;
  GDate *gdate;
  KPDate *date;
  TYear *ty;
  guint i;
  
  if (log->n_entries == 0)
    goto return_today;
  
  if (log->year_list == NULL)
    goto return_today;
  
  list = log->year_list;
  ty = list->data;
  
  while (list) {
    if (((TYear *) list->data)->year < ty->year)
      ty = list->data;

    list = list->next;
  }
  kp_debug ("Found earliest year: %u", ty->year);
  
  for (i=0; i < 12; i++) {
    if (ty->workouts[i]) {
      
      date = g_new0 (KPDate, 1);
      kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (ty->workouts[i]->data), 
                                  date);
      g_return_val_if_fail (kp_date_valid (date), NULL);
      
      return date;
    }
  }

return_today:

  gdate = g_date_new ();
  g_date_set_time (gdate, time (NULL));

  date = kp_date_new_dmy (g_date_get_day (gdate),
                          g_date_get_month (gdate),
                          g_date_get_year (gdate));
  g_date_free (gdate);
  return date;
}


/**
 * kp_training_log_get_latest_date:
 * @log: A #KPTrainingLog
 *
 * Get the date of the latest workout or other entry 
 * that is in the log.
 * 
 * Returns: A newly-allocated KPDate-structure that must
 * be freed.
 */
KPDate *
kp_training_log_get_latest_date (KPTrainingLog *log)
{
  GList *list;
  GDate *gdate;
  KPDate *date;
  TYear *ty;
  guint i;
  
  if (log->n_entries == 0)
    goto return_today;
  
  if (log->year_list == NULL)
    goto return_today;
  
  list = log->year_list;
  ty = list->data;
  
  while (list) {
    if (((TYear *) list->data)->year > ty->year)
      ty = list->data;

    list = list->next;
  }

  for (i=11; i > 0; i--) {
    if ((list = g_list_last (ty->workouts[i]))) {

      date = g_new0 (KPDate, 1);
      kp_calendar_entry_get_date (KP_CALENDAR_ENTRY (list->data), date);
      g_return_val_if_fail (kp_date_valid (date), NULL);
      
      return date;
    }
  }

return_today:

  gdate = g_date_new ();
  g_date_set_time (gdate, time (NULL));

  date = kp_date_new_dmy (g_date_get_day (gdate),
                          g_date_get_month (gdate),
                          g_date_get_year (gdate));
  g_date_free (gdate);
  return date;
}


guint
kp_training_log_get_n_years (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), 0);

  if (log->year_list == NULL)
    return 0;

  return g_list_length (log->year_list);
}



/**
 * kp_training_log_remove_mark:
 * @log: a #KPTrainingLog
 * @d: the day
 * @m: the month
 * @y: the year
 * @mark: the mark to remove
 *
 * Removes the mark from the log. 
 */
void
kp_training_log_remove_mark (KPTrainingLog *log, guint d, guint m, guint y,
                             const gchar *mark)
{
  const gchar *m_str;
  GList *list;

  list = kp_training_log_get_day (log, d, m, y);
  while (list) {
    m_str = kp_calendar_entry_to_string (KP_CALENDAR_ENTRY (list->data));

    if (strcmp (m_str, mark) == 0) {
      kp_training_log_remove (log, KP_CALENDAR_ENTRY (list->data));
      return;
    }
    
    list = list->next;
  }
}

/**
 * kp_training_log_remove_by_date:
 * @log: a #KPTrainingLog
 * @d: Day number or -1
 * @m: Month number or -1
 * @y: Year number or -1
 *
 * Remove one or more calendar entries from the log. If d, m and y are all
 * not -1 and mark is not NULL, only one entry is removed. In more complex cases
 * all entries in the day, month, year or even in the whole log may be removed.
 * 
 * By setting d, m, or y -1 or setting mark NULL mean that all those will be
 * deleted, like if d is -1, all entries in all days in month m will be deleted.
 */
void
kp_training_log_remove_by_date (KPTrainingLog *log, gint d, gint m, gint y,
                                const gchar *mark)
{
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  if (mark == NULL) {
    kp_training_log_remove_day (log, d, m, y);
    return;
  }
  
  if (d < 0) {
    kp_training_log_remove_month (log, y, m);
    return;
  }
  
  if (m < 0) {
    /* TODO:
    kp_training_log_remove_year (log, y);*/
    return;
  }

  kp_training_log_remove_mark (log, d, m, y, mark);
}

/**
 * kp_training_log_remove_month:
 * @log: A #KPTrainingLog
 * @y: the day
 * @m: the month
 *
 * Remove all entries in a month.
 */
void
kp_training_log_remove_month (KPTrainingLog *log, guint y, guint m)
{
  GList *list;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  list = kp_training_log_get_month (log, y, m);
  while (list) {
    kp_training_log_remove (log, KP_CALENDAR_ENTRY (list->data));
    list = list->next;
  }
}

/**
 * kp_training_log_remove_day:
 * @d: the day
 * @m: the month
 * @y: the year
 *
 * Remove all entries of a day.
 */
void
kp_training_log_remove_day (KPTrainingLog *log, guint d, guint m, guint y)
{
  GList *list;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  list = kp_training_log_get_day (log, d, m, y);

  while (list) {
    kp_training_log_remove (log, KP_CALENDAR_ENTRY (list->data));  
    list = list->next;
  }
}

static xmlNodePtr
export_meta_data_xml (KPTrainingLog *log)
{
  GSList *list;
  xmlNodePtr node;
  xmlNodePtr info;

  node = xmlNewNode (NULL, BAD_CAST ("meta"));

  /* Add info fields */
  for (list = log->info_list; list; list = list->next) {
    info = xmlNewNode (NULL, BAD_CAST ("info"));
    xmlSetProp (info, BAD_CAST ("field"), BAD_CAST (((KPInfoEntry *) list->data)->name));
    xmlNodeSetContent (info, BAD_CAST (((KPInfoEntry *) list->data)->data));
    xmlAddChild (node, info);
  }
  return node; 
}


/**
 * kp_training_log_to_xml:
 * @log: a #KPTrainingLog
 * 
 * Convert log to xml. This function just tries to convert data from
 * KPTrainingLog to xmlDocPtr. KPTrainingLog can be empty.
 *
 * Returns: #xmlDocPtr or NULL if some error occurs.
 */
xmlDocPtr
kp_training_log_to_xml (KPTrainingLog *log)
{
  xmlDocPtr doc;
  xmlNodePtr node;
  xmlNodePtr meta;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);

  doc = xmlNewDoc (BAD_CAST ("1.0"));
  node = xmlNewChild ((xmlNodePtr) doc, NULL, BAD_CAST ("traininglog"), NULL);

  meta = export_meta_data_xml (log);
  xmlAddChild (node, meta);
  
  if (log->year_list) 
    g_list_foreach (log->year_list, kp_training_log_year_to_xml, node);

  return doc;
}

static void
kp_training_log_year_to_xml (gpointer val, gpointer data)
{
  xmlDocPtr doc = (xmlDocPtr) data;
  GList *el;
  guint i;
  TYear *ty;

  ty = (TYear *) val;

  for (i = 0; i < 12; i++) {
    if (ty->workouts[i] != NULL) {

      el = ty->workouts[i];

      while (el) {
        g_assert (KP_IS_CALENDAR_ENTRY (el->data));
        
        xmlAddChild ((xmlNodePtr) doc,
                     kp_calendar_entry_to_xml (KP_CALENDAR_ENTRY (el->data)));
        el = el->next;
      }
    }
  }
}

/**
 * kp_training_log_get_entry:
 * @log: A #KPTrainingLog
 * @d: Number of day between 1 and 31
 * @m: Number of month between 1 and 12
 * @y: Number of in format yyyy 
 * @mark:
 *
 * Get a #CalendarEntry. DMY must be valid.
 *
 * Returns: A #KPCalendarEntry or NULL if there is no
 *          such entry in the log.
 */
KPCalendarEntry *
kp_training_log_get_entry (KPTrainingLog *log, guint d, guint m, guint y,
                           const gchar *mark)
{
  const gchar *m_str;
  GList *list;

  list = kp_training_log_get_day (log, d, m, y);
  while (list) {
    m_str = kp_calendar_entry_to_string (KP_CALENDAR_ENTRY (list->data));
    
    if (strcmp (m_str, mark) == 0) {
      return KP_CALENDAR_ENTRY (list->data);
    }
    
    list = list->next;
  }
  return NULL;
}

/**
 * kp_training_log_get_year:
 * @log: A #KPTrainingLog
 * @year: Year in format yyyy
 * 
 * Get a year-structure for any year. If there aren't any kp_workouts
 * with a date in the given year in the log, NULL will be returned.
 *
 * Otherwise, pointer to a TYear will be returned. It won't be copied,
 * so it MUST NOT be freed by the caller of this function.
 *
 * Returns: Year-structure that represents a year.
 */
TYear *
kp_training_log_get_year (KPTrainingLog * log, guint year)
{
  GList *list;
  TYear *ty;

  list = g_list_first (log->year_list);

  while (list) {
    ty = list->data;

    if (ty->year == year)
      return ty;

    list = list->next;
  }
  return NULL;
}

/**
 * kp_training_log_get_size:
 * @log: A #KPTrainingLog
 *
 * Returns the number of entries in the log
 *
 * Returns: Number of entries in the log.
 */
guint
kp_training_log_get_size (KPTrainingLog *log)
{
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), 0);
  return log->n_entries;
}

/**
 * kp_training_loG_get_month:
 * @log: A #KPTrainingLog
 * @year: Year number
 * @month: Month number (1-12)
 * 
 * Get all calendar entries in any month. Return the list
 * of those entries or NULL if some error occurs. Entries
 * are not copied, so they MUST NOT be freed.
 *
 * Returns: List of all entries in the month
 */
GList *
kp_training_log_get_month (KPTrainingLog *log, gint year, gint month)
{
  GList *y_list;
  TYear *ty;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (g_date_valid_month (month), NULL);

  ty = NULL;
  for (y_list = g_list_first (log->year_list); y_list != NULL;
       y_list = y_list->next) {
    ty = y_list->data;
    if (ty->year == (guint) year)
      break;
    else
      ty = NULL;
  }

  if (!ty)
    return NULL;

  return ty->workouts[month - 1];
}

/**
 * kp_training_log_get_day:
 * @log: A #KPTrainingLog
 * @d: Day number
 * @m: Month number (1-12)
 * @y: Year number
 * 
 * Get list of all kp_workouts and other entries for any day in
 * the log. As usual, if some error happens, NULL will be returned.
 * Entries are not copied, so they MUST NOT be freed.
 *
 * Returns: List of all the entries in a day.
 */
GList *
kp_training_log_get_day (KPTrainingLog *log, guint d, guint m, guint y)
{
  KPCalendarEntry *entry;
  GList *list = NULL;
  GList *mon;
  GDate *date;
  TYear *ty;
  
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (g_date_valid_month (m), NULL);

  ty = kp_training_log_get_year (log, y);
  if (!ty)
    return NULL; /* There is no kp_workout in year y */
  
  mon = ty->workouts[m-1];

  while (mon) {
    entry = mon->data;
    g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), NULL);
    
    date = kp_calendar_time_get_date (KP_CALENDAR_ENTRY (entry)->datetime);
    if (g_date_get_day (date) == d)
      list = g_list_prepend (list, entry);
    mon = mon->next;
  }
  return list; 
}


/**
 * kp_training_log_get_all_workouts: 
 * @log: A training log.
 * 
 * Get a list of ALL entries in the log. List items are not copied
 * (maybe they should be?) so they MUST NOT be freed!
 */
GList *
kp_training_log_get_all_entries (KPTrainingLog * log)
{
  GList *wo_list;
  GList *y_list;
  TYear *ty;
  guint i;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);

  y_list = g_list_first (log->year_list);
  wo_list = NULL;

  while (y_list) {
    ty = y_list->data;
    for (i = 0; i < 12; i++) {
      /* Go through all months and add items to wo_list */
      g_list_foreach (ty->workouts[i], list_add_item_to_passed_list, &wo_list);
    }
    y_list = y_list->next;
  }
  
  return wo_list;
}


/**
 * Just private helper-func to use with g_list_foreach ().
 */
static void
list_add_item_to_passed_list (gpointer val, gpointer data)
{
  GList **list = (GList **) data;
  *list = g_list_prepend (*list, val);
}

/**
 * kp_training_log_get_workout_params_between:
 * @log: A #KPTrainingLog
 * @mode: #KPTrainingLogDataMode
 * @sport: Show only params when the sport of the workout is this, NULL = all
 * @param: The name of the param
 * @start: Starting date.
 * @end: Ending date.
 * @days_between: Number of days between start and end will be stored
 *   in @days_between.
 * 
 * Get the distance of every kp_workout between two dates and return
 * data as dynamic two-dimensional array. It must be freed with
 * kp_array_2d_free()-func.
 * 
 * If days_between is not NULL, number of days between two dates will
 * be stored in it.
 *
 * Returns: Two-dimensional gdouble-array of data
 */
gdouble **
kp_training_log_get_workout_params_between (KPTrainingLog *log,
                                            KPTrainingLogDataMode mode,
                                            const gchar *sport,
                                            const gchar *param_name,
                                            GDate *start, GDate *end,
                                            guint *days_between)
{
  KPWorkout *wo;
  GDate *date;
  GList *list;
  guint days;
  gdouble **data;
  gdouble item;
  guint idx;

  /* For avg */
  const gchar *tmp_sport;
  gdouble *avg_sum;
  guint *avg_n;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (start != NULL, NULL);
  g_return_val_if_fail (end != NULL, NULL);
  g_return_val_if_fail (g_date_compare (start, end) <= 0, NULL);
  
  days = g_date_days_between (start, end) + 1;
  data = (gdouble **) kp_array_2d_alloc_init0 (1, days, sizeof (gdouble));
  list = kp_training_log_get_all_entries_between (log, start, end, NULL);

  if (mode == KP_TRAINING_LOG_DATA_AVG) {
    avg_sum = g_malloc0 (days * sizeof (*avg_sum));
    avg_n = g_malloc0 (days * sizeof (*avg_n));
  }
  else {
    /* Prevent a compiler warning */
    avg_sum = NULL;
    avg_n = NULL;
  }
      
  while (list) {
    wo = (KP_IS_WORKOUT (list->data)) ? KP_WORKOUT (list->data) : NULL;
   
    /* If this entry is a workout AND the sport is the same as wanted
     * OR NULL, then add param to the statistics */

    if (wo == NULL)
      goto loop_end;
    
    tmp_sport = kp_workout_get_sport (wo);
    g_return_val_if_fail (tmp_sport != NULL, NULL);
    
    if (sport == NULL || strcasecmp (sport, tmp_sport) == 0) {
      
      date = kp_calendar_time_get_date (
          KP_CALENDAR_ENTRY (list->data)->datetime);
      g_assert (g_date_valid (date));

      idx = g_date_days_between (start, date);  
      g_assert (idx < days);
 
      /* FIXME: quick hack to fix KPChartView */

      item = 0.0;
      if (strcmp (param_name, "pace") == 0)
        item = kp_workout_model_get_pace (KP_WORKOUT_MODEL (wo));
      if (strcmp (param_name, "duration") == 0)
        item = kp_workout_model_get_duration (KP_WORKOUT_MODEL (wo));
      if (strcmp (param_name, "distance") == 0)
        item = kp_workout_model_get_distance (KP_WORKOUT_MODEL (wo));
      else {
        item = kp_param_list_get_double (KP_WORKOUT (list->data)->param_list, param_name);
      }

      switch (mode)
      {
        case KP_TRAINING_LOG_DATA_SUM:      
          data[0][idx] += item;
          break;
          
        case KP_TRAINING_LOG_DATA_MIN:
          if (data[0][idx] > 0.0) {
            if (item < data[0][idx])
              data[0][idx] = item;
          } else
            data[0][idx] = item;
          
          break;

        case KP_TRAINING_LOG_DATA_MAX:
          if (item > data[0][idx])
            data[0][idx] = item;
          break;

        case KP_TRAINING_LOG_DATA_AVG:
          if (item > 0.0) {
            avg_n[idx]++;
            avg_sum[idx] += item;
            data[0][idx] = avg_sum[idx] / (gdouble) avg_n[idx];
          }
          break;

        default:
          g_return_val_if_reached (NULL);
      }
    }
loop_end:
    list = list->next;
  }
  if (days_between)
    *days_between = days;

  return data; 
}

/**
 * kp_training_log_get_workout_params_year:
 * @log:
 * @year:
 * @param:
 * @days_between: 
 *
 * Returns: Two-dimensional gdouble-array of data 
 */
gdouble **
kp_training_log_get_workout_params_year (KPTrainingLog *log,
                                         const gchar *param, guint year,
                                         guint *days_between)
{
  gdouble **ret_data;
  gdouble **data;
  guint days;
  GDate *s;
  GDate *e;
  guint m;
  guint i;
 
  ret_data = kp_array_2dd_alloc_init0 (1, 12, sizeof (gdouble));

  s = g_date_new ();
  e = g_date_new ();

  for (m=1; m <= 12; m++) {
    g_date_set_dmy (s, 1, m, year);
    g_date_set_dmy (e, kp_get_month_len (m, kp_leap (year)), m, year);
    
    data = kp_training_log_get_workout_params_between (log,
        KP_TRAINING_LOG_DATA_SUM, NULL, param, s, e, &days);
 
    for (i=0, ret_data[0][m-1]=0; i < days; i++)
      ret_data[0][m-1] += data[0][i];
    
    kp_array_2dd_free (data, 1, days);
  }
  *days_between = 12;
  return ret_data;
}

/**
 * traiing_log_get_all_entries_between:
 * @log: a training log.
 * @start: starting date.
 * @end: ending date.
 * 
 * Get all CalendarEntries between two dates.
 *
 * Returns: List of all workouts between start and end dates. Note that
 * the list is NOT a copy. It must NOT be freed. If there some error occurs,
 * NULL will be returned.
 */
GList *
kp_training_log_get_all_entries_between (KPTrainingLog *log, GDate *start,
                                         GDate *end, GList **sports)
{
  GList *dlist = NULL;
  GList *list = NULL;
  KPWorkout *wo;
  GDate *date;
  guint days;

  if (sports)
    *sports = NULL;

  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), NULL);
  g_return_val_if_fail (start != NULL, NULL);
  g_return_val_if_fail (end != NULL, NULL);
  g_return_val_if_fail (g_date_compare (start, end) <= 0, NULL);
  
  date = g_date_new_dmy (g_date_get_day (start),
                         g_date_get_month (start),
                         g_date_get_year (start));

  /* +1 is to prevent problem with unsigned int and days >= 0 -thing */
  for (days = g_date_days_between (start, end) + 1; days > 0; days--) {
    dlist = kp_training_log_get_day (log, g_date_get_day (date),
                                          g_date_get_month (date),
                                          g_date_get_year (date));
    list = g_list_concat (list, dlist);

    while (sports && dlist) {
      wo = (KP_IS_WORKOUT (dlist->data)) ? KP_WORKOUT (dlist->data) : NULL;
     
      if (wo && !g_list_find_custom (*sports, kp_workout_get_sport (wo), (GCompareFunc) strcmp))
        *sports = g_list_prepend (*sports, g_strdup (kp_workout_get_sport (wo)));
      
      dlist = dlist->next;
    }
    g_date_add_days (date, 1);
  }
  g_date_free (date);

  return list;
}


static void
print_year (gpointer val, gpointer data)
{
  KPWorkout *wo;
  GList *list;
  TYear *y;
  gint i;
  
  y = (TYear *) val;
  g_print ("\nStatistics for year: %d\n\n", y->year);

  for (i = 0; i < 12; i++) {
    list = y->workouts[i];
    while (list) {
      wo = list->data;
      kp_calendar_entry_to_string (KP_CALENDAR_ENTRY ((wo)));
      list = list->next;
    }
  }
}
/*static void
print_param (gpointer key, gpointer val, gpointer data)
{
  printf ("   - %s=\"%s\"\n", (gchar *) key, (gchar *) val);
}*/

/**
 * kp_training_log_dump:
 * @log: A #KPTrainingLog
 *
 * Dump the contents of the log to stdout.
 * Could be used for debugging.
 */
void
kp_training_log_dump (KPTrainingLog *log)
{
  g_return_if_fail (log != NULL);
  g_print ("KPTrainingLog dump:\n");
  g_list_foreach (log->year_list, print_year, NULL);
}


static void
log_add_info_data (KPTrainingLog *log, xmlNodePtr node)
{
  KPInfoEntry *info;

  info = g_new0 (KPInfoEntry, 1);
  info->name = (gchar *) xmlGetProp (node, BAD_CAST ("field"));
  info->data = (gchar *) xmlNodeGetContent (node);
  
  if (info != NULL && info->name != NULL && info->data != NULL)
    log->info_list = g_slist_append (log->info_list, info); 
  else {
    if (info->name == NULL)
      g_warning ("<info> must have attribute 'field'!");
    if (info->data == NULL)
      g_warning ("<info> must have some content!");
    
    if (info->name)
      g_free (info->name);
    if (info->data)
      g_free (info->data);
    g_free (info);
  }
}

static void
parse_meta_data (KPTrainingLog *log, xmlNodePtr node)
{
  g_return_if_fail (node != NULL);
  
  node = node->children;

  while (node) {
    
    /*** <presetlist> ***/
    if (KP_TAG_MATCH (node, "presetlist")) {
      (void) kp_preset_data_import_tree (node);
    }

    /*** <info> ***/
    if (KP_TAG_MATCH (node, "info")) {
      log_add_info_data (log, node);
    }
    
    node = node->next;
  }
}
 

static KPTrainingLog *
kp_training_log_parse_from_xml (const gchar *filename, KPTrainingLog *log_,
                                GError **error)
{
  KPCalendarEntry *entry;
  KPTrainingLog *log;
  xmlNodePtr child;
  xmlNodePtr node;
  xmlDocPtr doc;
  gboolean is_split;

  g_return_val_if_fail (error == NULL || *error == NULL, NULL);

  if (log_)
    log = log_;
  else
    log = kp_training_log_new ();

  log->file = g_strdup (filename);
  log->written_to_disk = TRUE;
  
  doc = xmlParseFile (filename);
  if (!doc) 
    goto err;

  node = xmlDocGetRootElement (doc);
  if (!node || !node->name || !KP_TAG_MATCH (node, "traininglog"))
    goto err;

  node = node->children;
 
  while (node) {

    /** <meta> **/
    if (KP_TAG_MATCH (node, "meta")) {
      parse_meta_data (log, node);
    }
    
    /** <workout> **/
    else if (KP_TAG_MATCH (node, "workout")) {
      is_split = FALSE;
      for (child = node->children; child; child = child->next) 
        if (KP_TAG_MATCH (child, "split")) {
          is_split = TRUE;
          break;
        }

      if (is_split)
        entry = KP_CALENDAR_ENTRY (kp_split_workout_new ());
      else
        entry = KP_CALENDAR_ENTRY (kp_workout_new ());

      if (!kp_calendar_entry_parse (KP_CALENDAR_ENTRY (entry), node))
        return NULL;
      g_return_val_if_fail (KP_IS_WORKOUT (entry), NULL);
      kp_training_log_add (log, entry);
    }
   
    else if (KP_TAG_MATCH (node, "results")) {
      entry = KP_CALENDAR_ENTRY (kp_results_new (NULL, KP_RESULT_TYPE_TIME));
      
      if (!kp_calendar_entry_parse (KP_CALENDAR_ENTRY (entry), node))
        return NULL;
    
      kp_training_log_add (log, entry);
    }
    
    /** <comment> **/
    else if (KP_TAG_MATCH (node, "comment")) {
      entry = KP_CALENDAR_ENTRY (kp_comment_new (NULL, NULL));

      if (!kp_calendar_entry_parse (KP_CALENDAR_ENTRY (entry), node))
        return NULL;
      g_return_val_if_fail (KP_IS_CALENDAR_ENTRY (entry), NULL);
      kp_training_log_add (log, entry);
    }

    /* some text content.. */
    else if (KP_TAG_MATCH (node, "text")) {
      /* DO NOTHING */
    }
    else {
      /* The log may be made with newer version of Kipin */
    }
    node = node->next;
  }
  return log;

err:

  g_set_error (error,
               KP_TRAINING_LOG_ERROR,
               KP_TRAINING_LOG_OPEN_ERROR,
            _("The log (%s) couldn't be opened: %s!"), 
               filename, last_error->str);

  if (doc)
    xmlFreeDoc (doc);
  
  xmlCleanupParser ();

  /* Don't touch the log if it was an existing one and
   * was passed to the function */
  if (log_) {
    return NULL;
  }
  else
    g_object_unref (log);
    
  return NULL;
}


/**
 * kp_training_log_get_param_day:
 * @log: A #KPTrainingLog.
 * @date: The date of the day as #GDate.
 * @param: Param of which value to get.
 *
 * Return a double value of sum of the entries' params in a day.
 *
 * Returns: a gdouble which can be zero if the requested param is invalid
 * or if its value is zero or if some other error happens.
 */
gdouble
kp_training_log_get_param_day (KPTrainingLog *log, GDate *date,
                               const gchar *param_name)
{
  KPCalendarEntry *entry;
  KPParam *param = NULL;
  gdouble sum = 0.0;
  GList *list;
  GDate *wo_date;
  guint year;
  guint mon;
  TYear *ty;
  gint comp;

  if (log == NULL || log->year_list == NULL)
    return 0.0;

  g_return_val_if_fail (param != NULL, 0.0);
  g_return_val_if_fail (g_date_valid (date), 0.0);

  year = g_date_get_year (date);
  mon = g_date_get_month (date);

  g_return_val_if_fail (mon > 0, 0.0);
  g_return_val_if_fail (mon <= 12, 0.0);

  ty = kp_training_log_get_year (log, year);
  if (!ty)
    return 0.0;

  for (list = ty->workouts[mon - 1]; list != NULL; list = list->next) {
    entry = list->data;

    g_return_val_if_fail (entry != NULL, 0.0);

    wo_date = kp_calendar_time_get_date (entry->datetime);
    g_return_val_if_fail (wo_date != NULL, 0.0);

    comp = g_date_compare (wo_date, date);

    if (comp > 0)
      return sum;

    if (KP_IS_WORKOUT (entry) && comp == 0) {
      param = kp_param_list_get_param (KP_WORKOUT (entry)->param_list, param_name);
      sum += (param) ? kp_param_get_double (param) : 0;
    }
  }

  return sum;
}


/**
 * kp_training_log_foreach:
 * @log: A #KPTrainingLog
 * @func: A func to execute
 * @data: Data to pass to the @func
 *
 * Call some function for each entry in the log.
 *
 * NOTE: This function can not be used to remove all the entries
 * from the log, because kp_training_log_remove () may destroy some
 * structures which are needed to iterate the log! So you must
 * use kp_training_log_remove_all ().
 */
void
kp_training_log_foreach (KPTrainingLog *log, GFunc func, gpointer data)
{
  GList *yl;
  GList *wl;
  TYear *ty;
  guint i;

  g_return_if_fail (KP_IS_TRAINING_LOG (log));
  g_return_if_fail (func != NULL);

  yl = log->year_list;
  
  while (yl) {
    ty = yl->data;

    g_return_if_fail (ty != NULL);
    
    for (i = 0; i < 12; i++) {
      wl = ty->workouts[i];
    
      while (wl) {
        g_return_if_fail (KP_IS_CALENDAR_ENTRY (wl->data));
        func (KP_CALENDAR_ENTRY (wl->data), data);
        wl = wl->next;
      }
    }
    yl = yl->next;
  }
}

static void
kp_training_log_entry_changed (KPCalendarEntry *entry, const gchar *old_string,
                               KPTrainingLog *log)
{
  g_signal_emit (G_OBJECT (log), kp_training_log_signals[CHANGED_SIGNAL], 0);
  log->changed = TRUE;
  kp_debug ("Entry in the log was changed!");
}

