#include <string.h>
#include <unistd.h>

#include <errno.h>
#include <glib.h>
#include <glib-object.h>
#include <sys/stat.h>

#include "../kpcalendarentry.h"
#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kputil.h"

#include "../kpworkoutmodel.h"
#include "../kplogstat.h"
#include "../kplogstatitem.h"
#include "../kparray2d.h"

#include "../../kipina-version.h"

#include "kphtmlutil.h"
#include "kphtmlwriter.h"

#include "page-month.h"
#include "page-year.h"
#include "page-entry.h"
#include "page-index-sport.h"
#include "page-index.h"


static void     kp_html_writer_class_init          (GObjectClass *klass,
                                                    gpointer data);
static void     kp_html_writer_instance_init       (GObject *object,
                                                    gpointer data);
static void     kp_html_writer_instance_finalize   (GObject *object);
  

GQuark
kp_html_writer_error_quark (void)
{
  static GQuark q = 0;
  if (q == 0)
    q = g_quark_from_static_string ("kp-html-writer-error-quark");

  return q;
}


GType
kp_html_writer_get_type ()
{
  static GType kp_html_writer_type = 0;

  if (!kp_html_writer_type) {
    static const GTypeInfo kp_html_writer_info = {
      sizeof (KPHtmlWriterClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_html_writer_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPHtmlWriter),
      0,
      (GInstanceInitFunc) kp_html_writer_instance_init,
      NULL
    };

    kp_html_writer_type = g_type_register_static (G_TYPE_OBJECT,
                                                 "KPHtmlWriter",
                                                 &kp_html_writer_info,
                                                  0);
  }
  
  return kp_html_writer_type;
}


static void
kp_html_writer_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_html_writer_instance_finalize;
}


static void
kp_html_writer_instance_init (GObject *object, gpointer data)
{
  KPHtmlWriter *writer;
  guint i;
  
  writer = KP_HTML_WRITER (object);
  
  for (i=0; i < KP_HTML_WRITER_FLAG_N; i++)
    writer->items[i] = TRUE;

  writer->buf = g_string_new_len (NULL, 2048);
  writer->fnbuf = g_string_new_len (NULL, 128);
  g_string_assign (writer->buf, "");
  writer->dir = NULL;
  writer->theme = g_strdup ("default");
  writer->log = NULL;
}


static void
kp_html_writer_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;
  KPHtmlWriter *writer;
  
  writer = KP_HTML_WRITER (object);
  g_string_free (writer->buf, TRUE);
  g_string_free (writer->fnbuf, TRUE);
  
  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}



/**
 * kp_html_writer_new:
 * @title: The title of the html_writer
 * @text: The html_writer
 * 
 * Create a new instance of #KPHtmlWriter.
 *
 * Returns: A #KPHtmlWriter 
 */
KPHtmlWriter *
kp_html_writer_new (void)
{
  return g_object_new (kp_html_writer_get_type (), NULL);
}


gchar *
kp_html_writer_find_template_filename (KPHtmlWriter *writer, const gchar *template)
{
  gchar *file;
#if 0
  /* First home directory */
  file = g_build_filename (G_DIR_SEPARATOR_S, 
                           g_get_home_dir (),
                          ".kipina",
                          "html_themes",
                           writer->theme,
                           template,
                           NULL);

  if (g_file_test (file, G_FILE_TEST_IS_REGULAR))
    return file;
  else
    g_free (file);

  
  /* Not found in home directory, try install directory */
  file = g_build_filename (G_DIR_SEPARATOR_S,
                           KIPINA_DATA_DIR,
                          "html_themes",
                           writer->theme,
                           template,
                           NULL);
  
  if (g_file_test (file, G_FILE_TEST_IS_REGULAR))
    return file;
  else
    g_free (file);
#endif
  /* Not found, return default */
  file = g_build_filename (G_DIR_SEPARATOR_S,
                           KIPINA_DATA_DIR,
                           template,
                           NULL);
  
  if (g_file_test (file, G_FILE_TEST_IS_REGULAR))
    return file;
  else
    g_free (file);

  return NULL;
}


static gboolean
copy_file (KPHtmlWriter *writer, const gchar *name, GError **err)
{
  FILE *ffp, *tfp;
  gchar *from, *to;
  size_t r, w;
  gchar buf[1024];
  
  g_return_val_if_fail (KP_IS_HTML_WRITER (writer), FALSE);
  g_return_val_if_fail (writer->dir != NULL, FALSE);
  
  from = kp_html_writer_find_template_filename (writer, name);
  if (from == NULL) {
    g_set_error (err,
                 kp_html_writer_error_quark (),
                 0,
                "Can't find file: %s!", name);
    return FALSE;
  }
  
  to = g_strdup_printf ("%s/%s", writer->dir, name);
  
  ffp = fopen (from, "r");
  if (ffp == NULL) {
    g_set_error (err,
                 kp_html_writer_error_quark (),
                 0,
                "Can't open %s for reading: %s", from, strerror (errno));
    
    return FALSE;
  }
    
  tfp = fopen (to, "w+");
  if (tfp == NULL) {
    g_set_error (err,
                 kp_html_writer_error_quark (),
                 0,
                "Can't open %s for writing: %s", to, strerror (errno));

    /* Close from file pointer before leaving */
    fclose (ffp);
    return FALSE;
  }
 
  while (1) {
    r = fread (buf, 1, sizeof (buf), ffp);

    if (r > 0) 
      w = fwrite (buf, 1, r, tfp);
    else if (feof (ffp)) 
      break;
    else {
      /* This is really unlikely to happen.. */
      g_set_error (err,
                   kp_html_writer_error_quark (),
                   0,
                  "Error while reading from %s: %s",
                   from, strerror (errno));
      return FALSE;
      break;
    }
  }

  fclose (ffp);
  fclose (tfp);

  g_free (from);
  g_free (to);

  return TRUE;
}



void
kp_html_writer_set_dir (KPHtmlWriter *writer, const gchar *dir)
{
  g_return_if_fail (KP_IS_HTML_WRITER (writer));
  
  if (writer->dir != NULL)
    g_free (writer->dir);

  writer->dir = g_strdup (dir);
}


FILE *
kp_html_writer_open (KPHtmlWriter *writer, const gchar *file, GError **error)
{
  gchar *filename;
  FILE *fp;

  filename = g_build_filename (G_DIR_SEPARATOR_S, writer->dir, file, NULL);

  fp = fopen (filename, "w");

  if (fp == NULL) {
    g_set_error (error,
                 kp_html_writer_error_quark (),
                 0,
                "Can't open %s for writing: %s\n", 
                 filename, strerror (errno));
    return NULL;
  } 
  g_free (filename);
  
  fputs ("<?xml version=\"1.0\" encoding=\"utf-8\"?>\n"
         "<!DOCTYPE html PUBLIC \"-//W3C//DTD XHTML 1.1//EN\""
         " \"http://www.w3.org/TR/xhtml11/DTD/xhtml11.dtd\">\n"
         "<html xmlns=\"http://www.w3.org/1999/xhtml\">\n"
         "<head>\n"
         "  <title>Kipin\303\244 HTML Report</title>\n"
         "  <link rel=\"stylesheet\" type=\"text/css\" href=\"style.css\"/>\n"
         "  <meta http-equiv=\"Content-type\" content=\"text/html; charset=utf-8\" />\n"
         "</head>\n"
         "<body>\n"
         "<div id=\"container\">\n",
         fp);

  return fp;
}


void
kp_html_writer_close (FILE *fp)
{
  gchar *version;
  
  fputs ("<br/>\n", fp);

  if (KIPINA_DEV_RELEASE)
    version = g_strdup_printf ("<i>[ %s "KIPINA_VERSION_DATE" "KIPINA_VERSION_TIME" ]</i>",
                              _("devel ver; date"));
  else
    version = g_strdup ("<i>"KIPINA_VERSION"</i>");
  
  fprintf (fp, "<p class=\"generated_by\">");
  fprintf (fp, _("Generated by %s version %s"),
                 "<a href=\"http://kipina.sourceforge.net\"><strong>Kipin\303\244"
                 "</strong></a>", version);
  g_free (version);
  
  fputs ("</p>\n</div>\n</body>\n</html>\n", fp);
  fclose (fp);
}


static void
make_pages (KPHtmlWriter *writer, GError **err)
{
  GError *tmp_error;
  TYear *ty;
  GList *node, *wn;
  guint m;
   
  /* Index page */
  tmp_error = NULL;
  page_index (writer, &tmp_error);

  if (tmp_error) {
    g_propagate_error (err, tmp_error);
    return;
  }

  /* Years by sports */
  tmp_error = NULL;
  page_index_sport (writer, &tmp_error);

  if (tmp_error) {
    g_propagate_error (err, tmp_error);
    return;
  }
  
  /* Year pages */
  for (node = writer->log->year_list; node != NULL; node = node->next) { 
    ty = node->data;
    page_year (writer, writer->log, ty->year, NULL);

    /* Month pages */
    for (m=1; m <= 12; m++) {
      page_month (writer, writer->log, ty->year, m, NULL);

      /* Entry pages */
      for (wn = ty->workouts[m-1]; wn; wn = wn->next) {
        tmp_error = NULL;
        page_entry (writer, KP_CALENDAR_ENTRY (wn->data), &tmp_error);

        if (tmp_error) {
          g_propagate_error (err, tmp_error);
          return;
        }
      }
    }
  }
}


gboolean
kp_html_writer_write (KPHtmlWriter *writer, KPTrainingLog *log, 
                      const gchar *dir, GError **err)
{
  gchar *files[] = { "style.css", "back.png", "horiz_1.png", "horiz_2.png" };
  GError *tmp_error;
  KPLogStat *stat;
  guint i;

  g_return_val_if_fail (dir != NULL, FALSE);
  g_return_val_if_fail (KP_IS_TRAINING_LOG (log), FALSE);
  g_return_val_if_fail (KP_IS_HTML_WRITER (writer), FALSE);
  
  if (dir[0] != '/') {
    g_set_error (err,
                 kp_html_writer_error_quark (),
                 0,
                "Path is not absolute (it doesn't start with '/')!");
    return FALSE;
  }
  
  stat = kp_log_stat_new (log);
 
  kp_html_writer_set_dir (writer, dir);
 
  /* writer->dir exists */
  if (g_file_test (writer->dir, G_FILE_TEST_EXISTS)) {
    /* But it's not a directory */
    if (g_file_test (writer->dir, G_FILE_TEST_IS_DIR) == FALSE) {
      g_warning ("File %s exists and it is not a directory!", writer->dir);
      return FALSE;
    }
  }
  else {
    /* Try to create it */
    if (mkdir (writer->dir, S_IRWXU | S_IRWXG | S_IROTH | S_IXOTH)) {
      g_warning ("Can't create directory '%s': %s", writer->dir, strerror (errno));

      g_set_error (err,
                   kp_html_writer_error_quark (),
                   0,
                  "Can't create directory %s: %s", 
                   writer->dir, strerror (errno));
      return FALSE;
    }
  }

 
  for (i=0; i < G_N_ELEMENTS (files); i++) {
    tmp_error = NULL;
    if (copy_file (writer, files[i], &tmp_error) == FALSE) {
      if (tmp_error)
        g_propagate_error (err, tmp_error);
      return FALSE;
    }
  }

  writer->stat = stat;
  writer->log = log;
 
  tmp_error = NULL;
  make_pages (writer, &tmp_error);

  /* Making pages didn't succeed */
  if (tmp_error) {
    g_propagate_error (err, tmp_error);
    return FALSE;
  }
  
  return TRUE;
}

