#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kputil.h"
#include "kpview.h"

static void     kp_view_class_init                    (GObjectClass *klass,
                                                       gpointer data);
static void     kp_view_instance_init                 (GObject *object,
                                                       gpointer data);
static void     kp_view_instance_finalize             (GObject *object);
static void     kp_view_pack_child                    (KPView *view,
                                                       KPViewModel *model);
static gboolean kp_view_unpack_child                  (KPView *view);
KPViewModel    *kp_view_get_current_model             (KPView *view);
static void     kp_view_model_init                    (KPViewModelIface *iface);
static void     kp_view_set_dmy                       (KPViewModel *view,
                                                       guint d,
                                                       guint m,
                                                       guint y);
static void     kp_view_get_dmy                       (KPViewModel *view,
                                                       guint *d,
                                                       guint *m,
                                                       guint *y);
static void     kp_view_set_view_type                 (KPViewModel *view,
                                                       KPViewModelType type);
static
KPViewModelType kp_view_get_view_type                 (KPViewModel *view);

static void     kp_view_set_log                       (KPViewModel *view,
                                                       KPTrainingLog *log);
static void     kp_view_unset_log                     (KPViewModel *view);
static void     kp_view_activate                      (KPViewModel *view);
static void     kp_view_deactivate                    (KPViewModel *view);
static KPViewModelType kp_view_get_view_type          (KPViewModel *view);


enum {
  VIEWER_SET_SIGNAL,
  LAST_SIGNAL
};

guint kp_view_signals[LAST_SIGNAL] = { 0 };

typedef struct KPViewPrivateData_
{
  GtkWidget       *title;
  GtkWidget       *box;
  GtkWidget       *view_menu;
  GtkWidget       *menu;

  KPViewModelType  type;
  
  GDate           *date;  
  GList           *viewers;
} KPViewPrivateData;

#define KP_VIEW_PRIVATE_DATA(widget) ((KPViewPrivateData*) \
        (KP_VIEW (widget)->private_data))


GType
kp_view_get_type ()
{
  static GType kp_view_type = 0;

  if (!kp_view_type) {
    static const GTypeInfo kp_view_info = {
      sizeof (KPViewClass),
      (GBaseInitFunc) NULL,
      (GBaseFinalizeFunc) NULL,
      (GClassInitFunc) kp_view_class_init,
      (GClassFinalizeFunc) NULL,
      NULL,
      sizeof (KPView),
      0,
      (GInstanceInitFunc) kp_view_instance_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_view_model_init,
      NULL,
      NULL
    };
    kp_view_type = g_type_register_static (GTK_TYPE_VBOX,
                                          "KPView",
                                          &kp_view_info,
                                           0);
    g_type_add_interface_static (kp_view_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);
  }
  return kp_view_type;
}

static void
kp_view_model_init (KPViewModelIface *iface)
{
  iface->set_dmy = kp_view_set_dmy;
  iface->get_dmy = kp_view_get_dmy;
  iface->set_log = kp_view_set_log;
  iface->unset_log = kp_view_unset_log;
  iface->set_view_type = kp_view_set_view_type;
  iface->get_view_type = kp_view_get_view_type;
  iface->activate = kp_view_activate;
  iface->deactivate = kp_view_deactivate;
}

static void
kp_view_class_init (GObjectClass *klass, gpointer data)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_view_instance_finalize;

  kp_view_signals[VIEWER_SET_SIGNAL] =
    g_signal_new ("viewer-set", 
                  G_OBJECT_CLASS_TYPE (object_class),
                  G_SIGNAL_RUN_FIRST | G_SIGNAL_ACTION,
                  G_STRUCT_OFFSET (KPViewClass, viewer_set),
                  NULL,
                  NULL,
                  g_cclosure_marshal_VOID__POINTER,
                  G_TYPE_NONE,
                  1,
                  G_TYPE_POINTER);
}

static void
kp_view_instance_init (GObject *object, gpointer data)
{
  KPViewPrivateData *p_data;
  KPView *view;
  GtkWidget *hbox;

  gtk_container_set_border_width (GTK_CONTAINER (object), 0);

  view = KP_VIEW (object);
  view->private_data = g_new0 (KPViewPrivateData, 1);
  p_data = KP_VIEW_PRIVATE_DATA (view);
  p_data->date = g_date_new ();
  p_data->viewers = NULL;
  p_data->box = gtk_vbox_new (FALSE, 0);

  hbox = gtk_hbox_new (FALSE, 12);
  
  p_data->title = gtk_label_new  (NULL);
  gtk_label_set_markup (GTK_LABEL (p_data->title), "<b>Some title</b>");
  p_data->view_menu = gtk_option_menu_new ();
  p_data->menu = gtk_menu_new ();
  gtk_option_menu_set_menu (GTK_OPTION_MENU (p_data->view_menu), p_data->menu);
  
  gtk_widget_show_all (GTK_WIDGET (object));
}


static void
kp_view_instance_finalize (GObject *object)
{
  GObjectClass *parent_class;

  parent_class = g_type_class_peek_parent (G_OBJECT_GET_CLASS (object));
  parent_class->finalize (object);
}


/**
 * kp_view_new:
 * @d: Day number
 * @m: Month number (1-12)
 * @y: Year number 
 * 
 * Create a new instance of #KPView.
 *
 * Returns: A #KPView 
 */
KPView *
kp_view_new (guint d, guint m, guint y)
{
  KPViewPrivateData *p_data;
  GObject *view;
 
  view = g_object_new (kp_view_get_type (), NULL);

  p_data = KP_VIEW_PRIVATE_DATA (view);
  g_return_val_if_fail (p_data != NULL, NULL);
  
  if (d == 0 || m == 0 || y == 0)
    g_date_set_time (p_data->date, time (NULL));
  else 
    kp_view_set_dmy (KP_VIEW_MODEL (view), d, m, y);
  
  return KP_VIEW (view);
}

static void 
kp_view_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  KPViewPrivateData *p_data;
  KPViewModel *model;
 
  g_return_if_fail (g_date_valid_dmy (d, m, y));
  g_return_if_fail (KP_IS_VIEW (view));
  
  model = kp_view_get_current_model (KP_VIEW (view));
  kp_view_model_set_dmy (model, d, m, y);
  
  p_data = KP_VIEW_PRIVATE_DATA (view);
  
  g_date_set_dmy (p_data->date, d, m, y);
}

static void
kp_view_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  KPViewPrivateData *p_data;
  g_return_if_fail (KP_IS_VIEW (view));
  p_data = KP_VIEW_PRIVATE_DATA (view);

  if (d)
    *d = g_date_get_day (p_data->date);
  if (m)
    *m = g_date_get_month (p_data->date);
  if (y)
    *y = g_date_get_year (p_data->date);
}

static void
kp_view_set_view_type (KPViewModel *view, KPViewModelType type)
{
  KPViewModel *model = kp_view_get_current_model (KP_VIEW (view));
  if (model)
    kp_view_model_set_view_type (model, type);
}

static KPViewModelType
kp_view_get_view_type (KPViewModel *view)
{
  KPViewModel *model = kp_view_get_current_model (KP_VIEW (view));
  if (model)
    return kp_view_model_get_view_type (model);
  else
    g_return_val_if_reached (KP_VIEW_MODEL_TYPE_MONTH);
}

static void
kp_view_set_log (KPViewModel *view, KPTrainingLog *log)
{
  GList *list = kp_view_get_viewers (KP_VIEW (view));

  while (list) {
    kp_view_model_set_log (KP_VIEW_MODEL (list->data), log);
    list = list->next;
  }
}


static void
kp_view_unset_log (KPViewModel *view)
{
  GList *list = kp_view_get_viewers (KP_VIEW (view));
  
  while (list) {
    kp_view_model_unset_log (KP_VIEW_MODEL (list->data));
    list = list->next;
  }
}

static void
kp_view_activate (KPViewModel *view)
{
  GList *list = kp_view_get_viewers (KP_VIEW (view));

  g_return_if_fail (KP_IS_VIEW_MODEL (view));

  while (list) {
    g_return_if_fail (KP_IS_VIEW_MODEL (list->data));
    kp_view_model_activate (KP_VIEW_MODEL (list->data));
    list = list->next;
  }
}

static void
kp_view_deactivate (KPViewModel *view)
{
  KPViewModel *model;

  model = kp_view_get_current_model (KP_VIEW (view));
  if (model)
    kp_view_model_activate (model);
}

/**
 * kp_view_add_viewer:
 * @view: A #KPView
 * @name: The name of the view
 * @model: A #KPViewModel to add
 *
 * Adds the viewers to the list of the available viewers. If
 * the list is empty when viewer is added, it will be packed
 * to the view and will be showed.
 */
void
kp_view_add_viewer (KPView *view, const gchar *name, KPViewModel *model)
{
  KPViewPrivateData *p_data;
  gboolean first;
  GtkWidget *mi;

  p_data = KP_VIEW_PRIVATE_DATA (view);
  first = (p_data->viewers == NULL);
  
  g_return_if_fail (GTK_IS_WIDGET (model));

  mi = gtk_menu_item_new_with_label (G_OBJECT_TYPE_NAME (model));
  gtk_menu_shell_append (GTK_MENU_SHELL (p_data->menu), mi);
  /*gtk_option_menu_set_history (GTK_OPTION_MENU (p_data->view_menu), 0);*/
  gtk_widget_show (mi);
  
  p_data->viewers = g_list_append (p_data->viewers, model);
  g_object_ref (G_OBJECT (model));

  if (g_date_valid (p_data->date))
  
  kp_view_model_set_dmy (model, g_date_get_day (p_data->date),
                                g_date_get_month (p_data->date),
                                g_date_get_year (p_data->date));

  g_object_set_data (G_OBJECT (model), "name", g_strdup (name));
                        
  if (first)
    kp_view_pack_child (view, model);
}

/**
 * kp_view_get_viewers:
 * @view: A #KPView
 *
 * Get the list of viewers attached to this #KPView.
 *
 * Returns: A #GList
 */
GList *
kp_view_get_viewers (KPView *view)
{
  KPViewPrivateData *p_data;
  g_return_val_if_fail (KP_IS_VIEW (view), NULL);
  p_data = KP_VIEW_PRIVATE_DATA (view);

  return p_data->viewers;
}


/**
 * kp_view_get_current_viewer:
 * @view: A #KPView
 * 
 * Get viewer that is active at the moment.
 *
 * Returns: KPViewModel or NULL if some error happens.
 */
KPViewModel *
kp_view_get_current_viewer (KPView *view)
{
  return kp_view_get_current_model (view);
}


/**
 * kp_view_unpack_child:
 * @view: A #KPView
 *
 * Unpacks the widget packed into the @view. In case there are
 * more than one widget packed into the @view, they will all be
 * unpacked.
 *
 * Returns: TRUE if everything goes OK and FALSE otherwise.
 */
static gboolean
kp_view_unpack_child (KPView *view)
{
  GList *list;
  GList *ltmp;
  guint n;
  
  g_return_val_if_fail (KP_IS_VIEW (view), FALSE);
  list = gtk_container_get_children (GTK_CONTAINER (view));
  g_return_val_if_fail (list != NULL, FALSE);
  
  n = g_list_length (list);
  
  while (list) {
    g_object_ref (G_OBJECT (list->data));
    gtk_container_remove (GTK_CONTAINER (view),
                          GTK_WIDGET (list->data));
    ltmp = list;
    list = list->next;
    g_list_free_1 (ltmp);
  }
  g_return_val_if_fail (n == 1, FALSE);
  
  return TRUE;
}


/**
 * kp_view_pack_child:
 * @view: A #KPView
 * @model: A #KPViewModel
 *
 * Packs the child to the view. Any other widgets that might
 * be inside the @view must be unpacked first with
 * kp_view_unpack_child ().
 */
static void
kp_view_pack_child (KPView *view, KPViewModel *model)
{
  g_return_if_fail (KP_IS_VIEW (view));
  g_return_if_fail (KP_IS_VIEW_MODEL (model));
  g_return_if_fail (gtk_container_get_children (GTK_CONTAINER (view)) == NULL);

  gtk_box_pack_start (GTK_BOX (view), GTK_WIDGET (model), TRUE, TRUE, 0);
  g_object_unref (model);
  gtk_widget_show (GTK_WIDGET (model));
}

/**
 * kp_view_set_viewer:
 * @view: A #KPView
 * @model: A #KPViewModel
 *
 * @model must be in the viewers list before setting it. So,
 * remember to call kp_view_add_viewer() first!
 */
void
kp_view_set_viewer (KPView *view, KPViewModel *model)
{
  KPViewPrivateData *p_data;
  KPViewModel *old;
  guint d, m, y;
 
  kp_debug ("Set viewer: %s\n", G_OBJECT_TYPE_NAME (model));
  
  g_return_if_fail (KP_IS_VIEW (view));
  g_return_if_fail (KP_IS_VIEW_MODEL (model));

  p_data = KP_VIEW_PRIVATE_DATA (view);
  g_return_if_fail (g_date_valid (p_data->date));
  
  old = kp_view_get_current_model (KP_VIEW (view));
  if (old) {
    kp_view_model_get_dmy (old, &d, &m, &y);
    kp_view_model_deactivate (old);
    p_data->type = kp_view_model_get_view_type (old);
  }
  g_return_if_fail (g_date_valid_dmy (d, m, y));
    
  if (kp_view_unpack_child (view) == TRUE)
    kp_view_pack_child (view, model);
  else
    /* Give a warning */
    g_return_if_reached ();

  g_signal_emit (view, kp_view_signals[VIEWER_SET_SIGNAL], 0, model);

  model = kp_view_get_current_model (view);
  kp_view_model_set_dmy (model,
                         g_date_get_day (p_data->date),
                         g_date_get_month (p_data->date),
                         g_date_get_year (p_data->date));
  kp_view_model_set_view_type (model, p_data->type); 
  kp_view_model_activate (model);
}


/**
 * kp_view_set_viewer_by_object_name:
 * @view: A #KPView
 * @name: the thing that G_OBJECT_TYPE_NAME returns
 *
 * So this sets the viewer, like:
 * 
 * GtkWidget *label;
 * KPView *view;
 *
 * view = get_this_from_somewhere ();
 * kp_view_set_viewer_by_object_name (view, G_OBJECT_TYPE_NAME (object));
 * 
 */
void
kp_view_set_viewer_by_object_name (KPView *view, const gchar *name)
{
  KPViewPrivateData *p_data;
  GList *list;

  p_data = KP_VIEW_PRIVATE_DATA (view);
  
  list = p_data->viewers;
  
  kp_debug ("There are %d viewers available, '%s' wanted.\n", 
            g_list_length (list), name);
  
  while (list) {
    if (strcmp (name, G_OBJECT_TYPE_NAME (list->data)) == 0) {
      kp_view_set_viewer (view, list->data);
      return;
    }
    list = list->next; 
  }
  /* Give a warning */
  g_return_if_reached ();
}

/**
 * kp_view_set_viewer_next:
 * @view: A #KPView
 * 
 * Sets the next view in the list active.
 */
void
kp_view_set_viewer_next (KPView *view)
{
  KPViewPrivateData *p_data;
  KPViewModel *model;
  GList *list;
  const gchar *cur_name;

  p_data = KP_VIEW_PRIVATE_DATA (view);
  model = kp_view_get_current_model (view);
  cur_name = G_OBJECT_TYPE_NAME (model);
  
  list = p_data->viewers;

  while (list)
  {
    if (strcmp (cur_name, G_OBJECT_TYPE_NAME (list->data)) == 0) {
      if (list->next) {
        list = list->next;
      } else {
        list = p_data->viewers;
      }
    
      kp_view_set_viewer (view, list->data);
      return;
    }
    list = list->next;
  }

  /* Give a warning */
  g_return_if_reached ();
}

/**
 * kp_view_set_viewer_prev:
 * @view: A #KPView
 * 
 * Sets the previous view in the list active.
 */
void
kp_view_set_viewer_prev (KPView *view)
{
  KPViewPrivateData *p_data;
  KPViewModel *model;
  GList *list;
  const gchar *cur_name;

  p_data = KP_VIEW_PRIVATE_DATA (view);
  model = kp_view_get_current_model (view);
  cur_name = G_OBJECT_TYPE_NAME (model);
  
  list = p_data->viewers;

  while (list)
  {
    if (strcmp (cur_name, G_OBJECT_TYPE_NAME (list->data)) == 0) {
      if (list->prev) {
        list = list->prev;
      } else {
        list = g_list_last (p_data->viewers);
      }
    
      kp_view_set_viewer (view, list->data);
      return;
    }
    
    list = list->next;
  }

  /* Give a warning */
  g_return_if_reached ();
}



/**
 * kp_view_get_current_model:
 * @view: A #KPView
 * 
 * Get the model that is currently active.
 * 
 * Returns: A #KPViewModel
 */
KPViewModel *
kp_view_get_current_model (KPView *view)
{
  GList *list, *tmp;
  KPViewModel *model = NULL;

  g_return_val_if_fail (GTK_IS_CONTAINER (view), NULL);
  
  list = gtk_container_get_children (GTK_CONTAINER (view));

  while (list) {

    if (model != NULL)
      g_warning ("There is more than one widget packed into KPView"
                 "at the same time!");
    
    if (GTK_WIDGET_VISIBLE (GTK_WIDGET (list->data)))
      model = KP_VIEW_MODEL (list->data);
    
    tmp = list;
    list = list->next; 
    g_list_free_1 (tmp);
  }

  kp_debug ("Current model is %s\n", G_OBJECT_TYPE_NAME (model));
  
  return model;
}


