#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kpresults.h"
#include "../kppresetdata.h"
#include "../kpworkoutmodel.h"
#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kputil.h"
#include "../kpunit.h"

#include "kpnewsplitworkoutdialog.h"
#include "kpnewcommentdialog.h"
#include "kpworkouteditor.h"
#include "kpguiutils.h"
#include "kpresultsview.h"
#include "kpviewmodel.h"
#include "kpcontexts.h"

/* Callbacks */
static void       kp_results_view_class_init       (KPResultsViewClass *klass);
static void       kp_results_view_init             (KPResultsView *lv);
static void       kp_results_view_finalize         (GObject *object);

/* Callback functions */
static void       result_removed                   (KPResults *results,
                                                    guint place,
                                                    KPResult *r,
                                                    KPResultsView *view);
static void       result_added                     (KPResults *results,
                                                    guint place,
                                                    KPResult *r,
                                                    KPResultsView *view);


typedef struct KPResultsViewPrivateData_
{
  KPResults        *results;
  
  GtkWidget        *popup_menu;

  GtkWidget        *popup_mi_properties;
  GtkWidget        *popup_mi_delete;
  GtkWidget        *popup_mi_edit;
 
  GList            *entries;
  GtkListStore     *store;

  KPViewModelType   type;
  KPTrainingLog    *log;
} KPResultsViewPrivateData;

#define KP_RESULTS_VIEW_PRIVATE_DATA(widget) (((KPResultsViewPrivateData*) \
      (KP_RESULTS_VIEW (widget)->private_data)))

enum {
  COLUMN_PLACE,
  COLUMN_NAME,
  COLUMN_ORGANIZATION,
  COLUMN_RESULT,
  COLUMN_DIFFERENCE,
  COLUMN_DATA,
  COLUMN_POINTER,
  COLUMN_N
};
enum {
  SORT_ID_PLACE,
  SORT_ID_NAME,
  SORT_ID_ORGANIZATION,
  SORT_ID_RESULT,
  SORT_ID_DIFFERENCE,
  SORT_ID_DATA
};


GType
kp_results_view_get_type (void)
{
  static GType        kp_results_view_type = 0;

  if (!kp_results_view_type) {
    static const GTypeInfo kp_results_view_info = {
      sizeof (KPResultsViewClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_results_view_class_init,
      NULL,
      NULL,
      sizeof (KPResultsView),
      0,
      (GInstanceInitFunc) kp_results_view_init,
      NULL
    };
    kp_results_view_type = g_type_register_static (GTK_TYPE_TREE_VIEW,
                                                  "KPResultsView",
                                                  &kp_results_view_info, 0);
  }
  return kp_results_view_type;
}

static void
kp_results_view_class_init (KPResultsViewClass *klass)
{
  GObjectClass *object_class;

  object_class = G_OBJECT_CLASS (klass);
  object_class->finalize = kp_results_view_finalize;
}

static void
str_cell_func (GtkTreeViewColumn *column, GtkCellRenderer *renderer,
               GtkTreeModel *model, GtkTreeIter *iter, gpointer data)
{
}
  
static GtkTreeViewColumn *
new_column (KPResultsView *lv, const gchar *title, guint column,
            GtkTreeCellDataFunc func)
{
  GtkTreeViewColumn *col;
  GtkCellRenderer *ren;

  ren = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new_with_attributes (title, ren, "markup", column,
                                                  NULL);
  gtk_tree_view_column_set_sort_column_id (col, column);
  gtk_tree_view_append_column (GTK_TREE_VIEW (lv), col);

  gtk_tree_view_column_set_resizable (col, TRUE);

  return col;
}

static void
kp_results_view_init (KPResultsView *lv)
{
  KPResultsViewPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *tcol;
  
  lv->private_data = g_new (KPResultsViewPrivateData, 1);
  p_data = KP_RESULTS_VIEW_PRIVATE_DATA (lv);
  p_data->log = NULL;
  p_data->type = KP_VIEW_MODEL_TYPE_MONTH;
  p_data->entries = NULL;

  (void) new_column (lv, _("Place"), COLUMN_PLACE, str_cell_func);
  (void) new_column (lv, _("Name"), COLUMN_NAME, str_cell_func);
  (void) new_column (lv, _("Organization"), COLUMN_ORGANIZATION, str_cell_func);
  (void) new_column (lv, _("Result"), COLUMN_RESULT, str_cell_func);
  tcol = new_column (lv, _("Difference"), COLUMN_DIFFERENCE, str_cell_func);
  (void) new_column (lv, _("Comment"), COLUMN_DATA, str_cell_func);
  
  
  p_data->store = gtk_list_store_new (COLUMN_N,
                                      G_TYPE_UINT,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_STRING,
                                      G_TYPE_POINTER);
  
  gtk_tree_view_set_model (GTK_TREE_VIEW (lv), GTK_TREE_MODEL (p_data->store));
  gtk_tree_view_set_rules_hint (GTK_TREE_VIEW (lv), TRUE);
  gtk_tree_view_set_headers_clickable (GTK_TREE_VIEW (lv), TRUE);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (lv));
  gtk_tree_selection_set_mode (selection, GTK_SELECTION_SINGLE);
}


static void
kp_results_view_finalize (GObject *object)
{
  KPResultsViewPrivateData *p_data;

  p_data = KP_RESULTS_VIEW_PRIVATE_DATA (object);

  KP_RESULTS_VIEW (object)->private_data = NULL;
  g_free (KP_RESULTS_VIEW (object)->private_data);  
}


GtkWidget *
kp_results_view_new (KPResults *results)
{
  KPResultsViewPrivateData *p_data;
  GtkWidget *widget;

  widget = g_object_new (kp_results_view_get_type (), NULL);

  p_data = KP_RESULTS_VIEW_PRIVATE_DATA (widget);
  p_data->results = results;
  
  if (KP_IS_RESULTS (results)) { 
    g_signal_connect (G_OBJECT (p_data->results), "result-added",
                      G_CALLBACK (result_added), widget);
    g_signal_connect (G_OBJECT (p_data->results), "result-removed",
                      G_CALLBACK (result_removed), widget);
  
    kp_results_view_set_results (KP_RESULTS_VIEW (widget), p_data->results);
  }
  gtk_widget_show_all (GTK_WIDGET (widget));
  
  return widget;
}

static void
places_update (KPResultsView *view)
{
  KPResultsViewPrivateData *p_data;
  GtkTreeModel *model;
  GtkTreeIter iter;
  guint difference;
  gchar *differ_str;
  GList *list;
  gint place;
  KPResult *res;
  
  p_data = KP_RESULTS_VIEW_PRIVATE_DATA (view);

  list = kp_results_get_list (p_data->results);
  
  model = GTK_TREE_MODEL (p_data->store);

  if (!gtk_tree_model_get_iter_first (model, &iter))
    /* The list is empty */
    return;

  do {
    gtk_tree_model_get (model, &iter, COLUMN_POINTER, &res, -1);
    g_return_if_fail (res != NULL);

    place = g_list_index (list, res) + 1; /* List indexes start from zero */

    difference = res->result - kp_results_get_top_result (p_data->results);
    differ_str = kp_date_mseconds_to_std_string (difference);
    
    gtk_list_store_set (GTK_LIST_STORE (model), &iter, 
                        COLUMN_PLACE, place, 
                        COLUMN_DIFFERENCE, differ_str,
                       -1);

    g_free (differ_str);
    
  } while (gtk_tree_model_iter_next (model, &iter));
}


static void
result_added (KPResults *results, guint place, KPResult *r, KPResultsView *view)
{
  KPResultsViewPrivateData *p_data;
  GtkTreeIter iter;
  gchar *result_str;
  gchar *differ_str;
  guint difference;
  
  g_return_if_fail (KP_IS_RESULTS_VIEW (view));
  g_return_if_fail (KP_IS_RESULTS (results));
  
  p_data = KP_RESULTS_VIEW_PRIVATE_DATA (view);

  result_str = kp_date_mseconds_to_std_string (r->result);
  difference = r->result - kp_results_get_top_result (results);
  differ_str = kp_date_mseconds_to_std_string (difference);

  gtk_list_store_append (p_data->store, &iter);
  gtk_list_store_set (p_data->store, &iter, 
                      COLUMN_PLACE, 32,
                      COLUMN_NAME, r->competitor->name,
                      COLUMN_ORGANIZATION, (r->competitor->organization) ? r->competitor->organization : "",
                      COLUMN_RESULT, result_str, 
                      COLUMN_DIFFERENCE, differ_str, 
                      COLUMN_DATA, (r->comment) ? r->comment : "",
                      COLUMN_POINTER, r, -1);
  
  g_free (result_str);
  g_free (differ_str);
  
  places_update (view);
}

static void
result_removed (KPResults *results, guint place, KPResult *r, 
                KPResultsView *view)
{
  /*TODO: Not implemented */
}


KPResults *
kp_results_view_get_results (KPResultsView *view)
{
  KPResultsViewPrivateData *p_data;
  p_data = KP_RESULTS_VIEW_PRIVATE_DATA (view);
  return p_data->results;
}

void
kp_results_view_set_results (KPResultsView *view, KPResults *results)
{
  KPResultsViewPrivateData *p_data;
  KPResult *res;
  GtkTreeIter iter;  
  GList *node;
  guint n = 1;
  gchar *str_res;
  gchar *str_diff;
  guint difference;
    
  p_data = KP_RESULTS_VIEW_PRIVATE_DATA (view);
 
  g_return_if_fail (KP_IS_RESULTS_VIEW (view));
  g_return_if_fail (KP_IS_RESULTS (results));
  
  if (p_data->results) {
    g_signal_handlers_disconnect_by_func (G_OBJECT (p_data->results), 
                                          G_CALLBACK (result_added), view);
    g_signal_handlers_disconnect_by_func (G_OBJECT (p_data->results), 
                                          G_CALLBACK (result_removed), view);
  } 
  p_data->results = results;
  g_signal_connect (G_OBJECT (p_data->results), "result-added",
                    G_CALLBACK (result_added), view);
  g_signal_connect (G_OBJECT (p_data->results), "result-removed",
                    G_CALLBACK (result_removed), view);


  gtk_list_store_clear (p_data->store);
  
  for (node = kp_results_get_list (results); node; node = node->next)
  {
    res = KP_RESULT (node->data);

    str_res = kp_date_mseconds_to_std_string (res->result);
    difference = res->result - kp_results_get_top_result (results);
    str_diff = kp_date_mseconds_to_std_string (difference);
    
    gtk_list_store_append (p_data->store, &iter);
    gtk_list_store_set (p_data->store, &iter, 
                        COLUMN_PLACE, n,
                        COLUMN_NAME, res->competitor->name,
                        COLUMN_ORGANIZATION, 
                         (res->competitor->organization) ? res->competitor->organization : "",
                        COLUMN_RESULT, str_res, 
                        COLUMN_DIFFERENCE, str_diff, 
                        COLUMN_DATA, (res->comment) ? res->comment : "",
                        COLUMN_POINTER, res, -1);
    g_free (str_diff);

    n++;
  }
}

#if 0
static void
append_to_store (GtkListStore *store, GtkTreeIter *iter, guint n, 
                 KPResults *results, KPResult *res)
{
  gtk_list_store_append (store, iter);
  gtk_list_store_set (store, iter, C
                      COLUMN_PLACE, n)..
    
}
#endif
