#include <gtk/gtk.h>
#include <glade/glade.h>
#include <string.h>
#include <time.h>

#include "kpmainwindow.h"
#include "kpresultseditor.h"
#include "kpresultsview.h"
#include "kpaddcompetitordialog.h"

#include "kpdateentry.h"
#include "kpguiutils.h"

#include "../kptraininglog.h"
#include "../kppresetdata.h"
#include "../kipina-i18n.h"
#include "../kpparamlist.h"
#include "../kpsettings.h"
#include "../kputil.h"

#define SPORT_MENU_TEXT    "Choose a Sport"
#define SPORT_MENU_OTHER   "Other"

typedef enum {
  KP_RESULTS_EDITOR_MODE_EDIT,
  KP_RESULTS_EDITOR_MODE_NEW    
} KPResultsEditorMode;

static void     kp_results_editor_class_init     (KPResultsEditorClass *klass);
static void     kp_results_editor_init           (KPResultsEditor *dialog);
static void     kp_results_editor_finalize       (GObject *object);

/* Callbacks */
static void     kp_results_editor_response_cb    (GtkDialog *dialog,
                                                  int response_id,
                                                  KPResultsEditor *editor);
static void     add_competitor_button_clicked    (GtkButton *button,
                                                  KPResultsEditor *editor);
static void     date_selected_cb                 (KPDateEntry *entry,
                                                  KPDate *Date,
                                                  KPResultsEditor *editor);
static void     time_selected_cb                 (KPDateEntry *entry, 
                                                  KPTime *time, 
                                                  KPResultsEditor *editor);

typedef struct KPResultsEditorPrivateData_
{
  KPTrainingLog       *log;
  KPResultsEditorMode  mode;
  KPResultsView       *view;
  KPResults           *results;

  GtkWidget           *entry_title;
  GtkWidget           *date_entry;
  GtkWidget           *entry_location;
  
  GtkWidget           *button_add_comp;
  GtkWidget           *button_del_comp;
  
  GtkWidget           *frame_view;
  GtkWidget           *closebutton;
} KPResultsEditorPrivateData;

#define KP_RESULTS_EDITOR_PRIVATE_DATA(widget) \
  (((KPResultsEditorPrivateData*) \
  (KP_RESULTS_EDITOR (widget)->private_data)))

static GObjectClass *parent_class = NULL;

GType
kp_results_editor_get_type (void)
{
  static GType kp_results_editor_type = 0;

  if (kp_results_editor_type == 0) {
    static const GTypeInfo our_info = {
      sizeof (KPResultsEditorClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_results_editor_class_init,
      NULL,
      NULL,
      sizeof (KPResultsEditor),
      0,
      (GInstanceInitFunc) kp_results_editor_init,
      NULL,
    };

    kp_results_editor_type = g_type_register_static (GTK_TYPE_DIALOG,
                                                    "KPResultsEditor",
                                                    &our_info, 0);
  }
  return kp_results_editor_type;
}


static void
kp_results_editor_class_init (KPResultsEditorClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS (klass);

  parent_class = g_type_class_peek_parent (klass);
  object_class->finalize = kp_results_editor_finalize;
}


static void
kp_results_editor_init (KPResultsEditor *dialog)
{
  KPResultsEditorPrivateData *p_data;
  GladeXML *xml;
  GtkWidget *cancel;
  
  xml = kp_gui_load ("results", "resultseditor");
  
  dialog->private_data = g_new0 (KPResultsEditorPrivateData, 1);
  p_data = KP_RESULTS_EDITOR_PRIVATE_DATA (dialog);

  g_signal_connect (G_OBJECT (dialog), "response",
                    G_CALLBACK (kp_results_editor_response_cb), dialog);
  
  gtk_container_add (GTK_CONTAINER (GTK_DIALOG (dialog)->vbox),
                     KP_W (xml, "resultseditor"));

  gtk_window_set_title (GTK_WINDOW (dialog), _("The Results Editor"));
  
  cancel = gtk_dialog_add_button (GTK_DIALOG (dialog), GTK_STOCK_CANCEL,
                                  GTK_RESPONSE_CANCEL);
  
  g_signal_connect_swapped (G_OBJECT (cancel), "clicked",
                            G_CALLBACK (gtk_widget_destroy), dialog);

  /* Put widgets to private data */
  p_data->entry_title = KP_W (xml, "title");
  p_data->entry_location = KP_W (xml, "location");
  p_data->button_add_comp = KP_W (xml, "add_competitor");
  p_data->button_del_comp = KP_W (xml, "del_competitor");
  p_data->frame_view = KP_W (xml, "view");

  p_data->date_entry = kp_date_entry_new ();
  gtk_box_pack_start (GTK_BOX (KP_W (xml, "datebox")), p_data->date_entry, 
                      TRUE, TRUE, 0);
  gtk_widget_show (p_data->date_entry);
  
  g_signal_connect (G_OBJECT (p_data->button_add_comp), "clicked",
                    G_CALLBACK (add_competitor_button_clicked), dialog);

  p_data->view = KP_RESULTS_VIEW (kp_results_view_new (NULL));
  gtk_widget_set_size_request (GTK_WIDGET (p_data->view), 400, 150);
 
  gtk_container_add (GTK_CONTAINER (p_data->frame_view),
                     GTK_WIDGET (p_data->view));
  
  g_signal_connect (G_OBJECT (p_data->date_entry), "date_selected",
                    G_CALLBACK (date_selected_cb), dialog);
  g_signal_connect (G_OBJECT (p_data->date_entry), "time_selected",
                    G_CALLBACK (time_selected_cb), dialog);
  
  g_object_unref (G_OBJECT (xml));
}


static void
kp_results_editor_finalize (GObject *object)
{
  KPResultsEditor *dialog;

  g_return_if_fail (object != NULL);
  g_return_if_fail (KP_IS_RESULTS_EDITOR (object));

  dialog = KP_RESULTS_EDITOR (object);

  g_return_if_fail (dialog->private_data != NULL);

  G_OBJECT_CLASS (parent_class)->finalize (object);
}


static void
update_button_states (KPResultsEditor *editor)
{
  KPResultsEditorPrivateData *p_data;
  gboolean val = TRUE; 
  const gchar *title;
  
  p_data = KP_RESULTS_EDITOR_PRIVATE_DATA (editor);
  
  if (!kp_date_entry_get_date (KP_DATE_ENTRY (p_data->date_entry), NULL)) {
    val = 0;
  }
    
  title = gtk_entry_get_text (GTK_ENTRY (p_data->entry_title));
  
  if (title == NULL || strlen (title) == 0)
    val = 0;

  gtk_widget_set_sensitive (p_data->closebutton, val);
}


/**
 * kp_results_editor_new:
 * @date: A #KPDate
 * @log: A #KPTrainingLog
 *
 * Create an instance of #KPResultsEditor.
 *
 * Returns: A #KPResultsEditor
 */
GtkWidget *
kp_results_editor_new (KPDate *date, KPCalendarEntry *results)
{
  KPResultsEditorPrivateData *p_data;
  KPResultsEditor *dialog;
  KPCalendarEntry *entry;
  KPTime t;
  
  dialog = g_object_new (kp_results_editor_get_type (), NULL);
  
  p_data = KP_RESULTS_EDITOR_PRIVATE_DATA (dialog);
  p_data->log = kp_main_window_get_log ();

  if (results != NULL) {
    p_data->mode = KP_RESULTS_EDITOR_MODE_EDIT;
    p_data->results = KP_RESULTS (results);
  
    p_data->closebutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                                 GTK_STOCK_SAVE,
                                                 GTK_RESPONSE_OK);
    gtk_entry_set_text (GTK_ENTRY (p_data->entry_title), 
                        kp_results_get_title (KP_RESULTS (results)));
  } else {
    p_data->mode = KP_RESULTS_EDITOR_MODE_NEW;
    p_data->results = kp_results_new (_("Untitled results"), 
                                      KP_RESULT_TYPE_TIME);
    
    p_data->closebutton = gtk_dialog_add_button (GTK_DIALOG (dialog),
                                                 GTK_STOCK_ADD,
                                                 GTK_RESPONSE_OK);
  }
  g_signal_connect_swapped (G_OBJECT (p_data->closebutton), "clicked",
                            G_CALLBACK (gtk_widget_destroy), dialog);
  
  g_assert (KP_IS_CALENDAR_ENTRY (p_data->results));
    
  entry = KP_CALENDAR_ENTRY (p_data->results);
  kp_calendar_time_set_dmy (entry->datetime, date->d, date->m, date->y);
  kp_date_entry_set_date (KP_DATE_ENTRY (p_data->date_entry), date);
 
  /* Set the time */
  kp_calendar_entry_get_time (KP_CALENDAR_ENTRY (p_data->results), &t);
  kp_date_entry_set_time (KP_DATE_ENTRY (p_data->date_entry), &t);
  
  /* Set the results */
  kp_results_view_set_results (p_data->view, p_data->results);
  
  update_button_states (KP_RESULTS_EDITOR (dialog));
  
  return GTK_WIDGET (dialog);
}

  
static void
kp_results_editor_response_cb (GtkDialog *dialog, int response_id,
                               KPResultsEditor *editor)
{
  KPResultsEditorPrivateData *p_data;
  const gchar *location;
  const gchar *title;
  KPTime t;
  
  p_data = KP_RESULTS_EDITOR_PRIVATE_DATA (editor);

  if (response_id != GTK_RESPONSE_CANCEL
   && response_id != GTK_RESPONSE_OK) {
    return;
  }
 
  if (p_data->mode == KP_RESULTS_EDITOR_MODE_EDIT)
    return;
  
  kp_date_entry_get_time (KP_DATE_ENTRY (p_data->date_entry), &t);
  
  if (response_id == GTK_RESPONSE_CANCEL)
    return;
  
  title = gtk_entry_get_text (GTK_ENTRY (p_data->entry_title));
  if (title && title[0])
    kp_results_set_title (p_data->results, title);
 
  location = gtk_entry_get_text (GTK_ENTRY (p_data->entry_location));
  if (location && location[0])
    kp_results_set_location (p_data->results, location);

  kp_training_log_add (p_data->log, KP_CALENDAR_ENTRY (p_data->results));
}


static void
add_competitor_button_clicked (GtkButton *button, KPResultsEditor *editor)
{
  KPResultsEditorPrivateData *p_data;
  GtkWidget *dialog;
  
  p_data = KP_RESULTS_EDITOR_PRIVATE_DATA (editor);
  
  dialog = kp_add_competitor_dialog_new (p_data->results);
  g_return_if_fail (dialog != NULL);

  gtk_widget_show (dialog);
}

static void
date_selected_cb (KPDateEntry *entry, KPDate *date, KPResultsEditor *editor)
{
  KPResultsEditorPrivateData *p_data;
  KPCalendarEntry *centry;
  
  p_data = KP_RESULTS_EDITOR_PRIVATE_DATA (editor);

  centry = KP_CALENDAR_ENTRY (p_data->results);
  kp_calendar_time_set_dmy (centry->datetime, date->d, date->m, date->y);
  
  update_button_states (editor);
}

static void
time_selected_cb (KPDateEntry *entry, KPTime *time, KPResultsEditor *editor)
{
  KPResultsEditorPrivateData *p_data;
  p_data = KP_RESULTS_EDITOR_PRIVATE_DATA (editor);
  
  kp_calendar_time_set_hmst (KP_CALENDAR_ENTRY (p_data->results)->datetime,
                             time->h, time->m, time->s, 0);
  
  update_button_states (editor);
}
