#include <string.h>
#include <time.h>

#include <glib-object.h>
#include <glib.h>

#include "../kipina-i18n.h"
#include "../kputil.h"

#include "kpnavitreeview.h"
#include "kpmainwindow.h"
#include "kpguiutils.h"
#include "kpview.h"

static void       kp_navi_tree_view_init             (KPNaviTreeView *tv);
static void       make_tree                          (KPNaviTreeView *tv);
static void       make_viewers_tree                  (KPNaviTreeView *tv);

typedef struct KPNaviTreeViewPrivateData_
{
  GtkWidget        *popup_menu_title;
  GtkWidget        *popup_menu;

  GtkWidget        *popup_mi_properties;
  GtkWidget        *popup_mi_add;

  KPTrainingLog    *log;
} KPNaviTreeViewPrivateData;

#define KP_NAVI_TREE_VIEW_PRIVATE_DATA(widget) (((KPNaviTreeViewPrivateData*) \
      (KP_NAVI_TREE_VIEW (widget)->private_data)))

#define FOLDER_VIEWERS    _("Viewers")
#define FOLDER_VIEWS      _("Views")
 
enum {
  COLUMN_ICON,
  COLUMN_TEXT,
  COLUMN_EXTRA,
};

struct KPNaviTreeData_ {
  const gchar *name;
  const gchar *icon;
  const gchar *text;
} navi_data [] = {
  { "node_views",      "folder.png",          N_("Views")      },
  { "view_day",        "dayview.png",         N_("Day View")   },
  { "view_week",       "weekview.png",        N_("Week View")  },
  { "view_month",      "monthview.png",       N_("Month View") },
  { "view_year",       "yearview.png",        N_("Year View")  },
  { "node_viewer",     "folder.png",          N_("Viewers")    },
  { "mode_calendar",   "calendar.png",        N_("Calendar")   },
  { "mode_list",       "list.png",            N_("List View")  },
  { "mode_statistics", "statistics.png",      N_("Stats View") },
  { "mode_chart",      "chart.png",           N_("Chart")      },
};


GType
kp_navi_tree_view_get_type (void)
{
  static GType        kp_navi_tree_view_type = 0;

  if (!kp_navi_tree_view_type) {
    static const GTypeInfo kp_navi_tree_view_info = {
      sizeof (KPNaviTreeViewClass),
      NULL,
      NULL,
      NULL,
      NULL,
      NULL,
      sizeof (KPNaviTreeView),
      0,
      (GInstanceInitFunc) kp_navi_tree_view_init,
      NULL
    };
    kp_navi_tree_view_type = g_type_register_static (GTK_TYPE_TREE_VIEW,
                                                    "KPNaviTreeView",
                                                    &kp_navi_tree_view_info, 0);
  }
  return kp_navi_tree_view_type;
}


static void
selection_changed (GtkTreeSelection *selection, KPNaviTreeView *tv)
{
  GtkTreeModel *model;
  GtkTreeIter iter_parent;
  GtkTreeIter iter;
  KPView *view;
  gchar *f_text;
  gchar *text;
  gdouble id;
 
  if (!gtk_tree_selection_get_selected (selection, &model, &iter))
    return;

  gtk_tree_model_get (model, &iter, COLUMN_EXTRA, &text, -1);
  if (strcmp (text, "folder") == 0)
    return;
 
  if (!gtk_tree_model_iter_parent (model, &iter_parent, &iter))
    return;

  gtk_tree_model_get (model, &iter_parent, COLUMN_EXTRA, &f_text, -1);
  
  view = kp_main_window_get_view ();

  if (strcmp (f_text, FOLDER_VIEWERS) == 0) {
    kp_view_set_viewer_by_object_name (view, text);
  }
  if (strcmp (f_text, FOLDER_VIEWS) == 0) {
    id = kp_number (text);
    if (id >= 0)
      kp_view_model_set_view_type (KP_VIEW_MODEL (view), id);
  }
}


static void
kp_navi_tree_view_init (KPNaviTreeView *tv)
{
  KPNaviTreeViewPrivateData *p_data;
  GtkTreeSelection *selection;
  GtkTreeViewColumn *col;
  GtkCellRenderer *ren;
  
  tv->private_data = g_new (KPNaviTreeViewPrivateData, 1);
  p_data = KP_NAVI_TREE_VIEW_PRIVATE_DATA (tv);

  selection = gtk_tree_view_get_selection (GTK_TREE_VIEW (tv));
  g_signal_connect (G_OBJECT (selection), "changed",
                    G_CALLBACK (selection_changed), tv);

  ren = gtk_cell_renderer_pixbuf_new ();
  col = gtk_tree_view_column_new ();
  gtk_tree_view_column_pack_start (col, ren, FALSE);
  gtk_tree_view_column_set_attributes (col, ren, "pixbuf", COLUMN_ICON, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tv), col);
 
  ren = gtk_cell_renderer_text_new ();
  col = gtk_tree_view_column_new ();
  gtk_tree_view_column_pack_start (col, ren, TRUE);
  gtk_tree_view_column_set_attributes (col, ren, "markup", COLUMN_TEXT, NULL);
  gtk_tree_view_append_column (GTK_TREE_VIEW (tv), col);
  
  make_viewers_tree (tv);
  make_tree (tv);

  gtk_tree_view_expand_all (GTK_TREE_VIEW (tv));
}

GtkWidget *
kp_navi_tree_view_new (void)
{
  GtkWidget *widget;

  widget = g_object_new (kp_navi_tree_view_get_type (), NULL);

  return widget;
}


static void
make_viewers_tree (KPNaviTreeView *tv)
{
  GtkTreeStore *store;
  GtkTreeIter iter_child;
  GtkTreeIter iter;
  GdkPixbuf *icon;
  GObject *obj;
  gchar *tmp;
  KPView *view;
  GList *list;
  gchar *filename;

  /* Third column isn't shown because it's for internal logic only */
  store = gtk_tree_store_new (3, GDK_TYPE_PIXBUF, G_TYPE_STRING, G_TYPE_STRING);
  gtk_tree_view_set_model (GTK_TREE_VIEW (tv), GTK_TREE_MODEL (store));

  view = kp_main_window_get_view ();
 
  icon = kp_get_icon_as_pixbuf ("folder.png");
  tmp = g_strconcat ("<b>", FOLDER_VIEWERS, "</b>", NULL);
  gtk_tree_store_append (store, &iter, NULL);
  gtk_tree_store_set (store, &iter,
                      COLUMN_ICON, icon,
                      COLUMN_TEXT, tmp,
                      COLUMN_EXTRA, FOLDER_VIEWERS,
                     -1);
  g_free (tmp);

  g_return_if_fail (KP_IS_VIEW (view));
  
  for (list = kp_view_get_viewers (view); list; list = list->next) {

    obj = G_OBJECT (list->data);
    filename = kp_view_model_get_icon_name (KP_VIEW_MODEL (list->data));
    icon = kp_get_icon_as_pixbuf (filename);
    g_free (filename);

    gtk_tree_store_append (store, &iter_child, &iter);
    gtk_tree_store_set (store, &iter_child,
                        COLUMN_ICON, icon ? (icon) : NULL,
                        COLUMN_TEXT, g_object_get_data (obj, "name"),
                        COLUMN_EXTRA, G_OBJECT_TYPE_NAME (obj),
                       -1);
    if (icon)
      g_object_unref (icon);
  }
}
 
static void
make_tree (KPNaviTreeView *tv)
{
  GtkTreeStore *store;
  GtkTreeIter iter_child;
  GtkTreeIter iter;
  GdkPixbuf *icon;
  guint i;
  gchar *tmp;
  gchar *name;
  gchar *icon_name;

  g_object_set (G_OBJECT (tv), "headers-visible", FALSE, NULL);
  
  store = GTK_TREE_STORE (gtk_tree_view_get_model (GTK_TREE_VIEW (tv)));
 
  icon = kp_get_icon_as_pixbuf ("folder.png");
  tmp = g_strconcat ("<b>", FOLDER_VIEWS, "</b>", NULL);
    
  gtk_tree_store_append (store, &iter, NULL);
  gtk_tree_store_set (store, &iter,
                      COLUMN_ICON, icon,
                      COLUMN_TEXT, tmp,
                      COLUMN_EXTRA, FOLDER_VIEWS,
                     -1); 
  g_free (tmp);
    
  for (i=0; i < KP_VIEW_MODEL_TYPE_N; i++) {
    name = kp_view_model_type_to_string (i, &icon_name);
    icon = kp_get_icon_as_pixbuf (icon_name);
    tmp = g_strdup_printf ("%u", i);

    gtk_tree_store_append (store, &iter_child, &iter);
    gtk_tree_store_set (store, &iter_child,
                        COLUMN_ICON, icon,
                        COLUMN_TEXT, name,
                        COLUMN_EXTRA, tmp,
                       -1);

    if (icon) 
      g_object_unref (G_OBJECT (icon));
    
    g_free (tmp);
    g_free (name);
  }
}

