/* 
 *  kp_chart.c - An extra widget for GTK+-2.x to show simple charts easily
 *  in any GTK+-2.x-program.
 *            
 *  (C) 2003,2004 Ville Kangas <ville@mirjami.net>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by   
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU Library General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include <gtk/gtk.h>

#include <stdlib.h>
#include <string.h>
#include <stdio.h>
#include <math.h>
#include <time.h>

#include "../kptraininglog.h"
#include "../kipina-i18n.h"
#include "../kputil.h"
#include "kpmainwindow.h"
#include "kpviewmodel.h"
#include "kpguiutils.h"
#include "kpchart.h"

typedef struct DrawData_ {
  gdouble xol;  /* X offset left */
  gdouble xor;  /* X offset right */
  gdouble yot;  /* Y offset top */
  gdouble yob;  /* Y offset bottom */

  gdouble ya;   /* Y active bar-drawing-area height */
  gdouble xa;   /* X active bar-drawing-area width */
  
  gint    w;    /* width of the widget */
  gint    h;    /* height of the widget */

  gdouble max[KP_CHART_AXES_MAX];
  gdouble min[KP_CHART_AXES_MAX];
  gdouble gmax[KP_CHART_AXES_MAX];
  gdouble gmin[KP_CHART_AXES_MAX];
  
  gdouble gsum_p[KP_CHART_AXES_MAX];
  gdouble gsum_n[KP_CHART_AXES_MAX];

  guint   hint_text_width[KP_CHART_AXES_MAX];
} DrawData;



/* Gobject & GtkWidget stuff */
static void       kp_chart_class_init         (KPChartClass *class);
static void       kp_chart_size_request       (GtkWidget *widget,
                                               GtkRequisition *requisition);
static void       kp_chart_realize            (GtkWidget *widget);
static void       kp_chart_size_allocate      (GtkWidget *widget,
                                               GtkAllocation *allocation);
static void       kp_chart_init               (KPChart * chart);
static void       kp_chart_set_items_num      (KPChart *, gint);
static void       kp_chart_set_groups_num     (KPChart *, gint);
static void       menu_grid_clicked           (gpointer data, guint action,
                                               GtkWidget *menu_item);
static void       menu_type_clicked           (gpointer data, guint action,
                                               GtkWidget *menu_item);
static void       save_image_clicked          (KPChart *chart,
                                               guint action,
                                               GtkMenuItem *item);
static gint       kp_chart_button_press       (GtkWidget *widget,
                                               GdkEventButton *event);
static gboolean   kp_chart_expose             (GtkWidget *widget,
                                               GdkEventExpose *expose);
/* KPModelView interface implementation */
static void       kp_chart_view_model_init    (KPViewModelIface *iface);
static void       kp_chart_set_dmy            (KPViewModel *view,
                                               guint d, guint m, guint y);
static void       kp_chart_get_dmy            (KPViewModel *view,
                                               guint *d, guint *m, guint *y);
static void       kp_chart_set_log            (KPViewModel *view,
                                               KPTrainingLog *log);
static void       kp_chart_unset_log          (KPViewModel *view);
static void       kp_chart_set_view_type      (KPViewModel *view,
                                               KPViewModelType type);
static
KPViewModelType   kp_chart_get_view_type      (KPViewModel *view);
static gchar     *kp_chart_get_icon_name      (KPViewModel *view);

/* Helper functions */
static GdkColor  *get_color                   (GtkWidget *widget, GdkGC *gc,
                                               guint8 r, guint8 g, guint8 b);
static void       draw_chart                  (KPChart *chart,
                                               GtkWidget *widget,
                                               GdkGC *gc,
                                               DrawData *d);
static void       update_dates                (KPChart *chart);
static void       update_chart_data           (KPChart *chart);

static void       log_connect_signals         (KPTrainingLog *log,
                                               KPChart *chart);
static void       log_disconnect_signals      (KPTrainingLog *log,
                                               KPChart *chart);
static void       log_changed                 (KPTrainingLog *log,
                                               KPChart *chart);
static void       kp_chart_data_clear         (KPChart *chart);
static void       calc_scale_data             (KPChart *chart, DrawData *d);

enum
{
  LEGEND_MARGIN_LEFT    = 30,
  LEGEND_SPACING_X      = 15,
  LEGEND_SPACING_Y      = 10,
  BAR_WIDTH_MIN         = 10, /* minimum bar width */
  BAR_WIDTH_MAX         = 95, /* maximum bar width */
  BAR_GRID_LINES        = 4,  /* number of horizontal lines when grid is ON */
  GRAPH_X_OFFSET_LEFT   = 10, /* left margin */
  GRAPH_X_OFFSET_RIGHT  = 10, /* right margin */
  GRAPH_Y_OFFSET_TOP    = 10, /* top margin */
  GRAPH_Y_OFFSET_BOTTOM = 30, /* bottom margin */
  GRAPH_COLORS_NUM      = 4,  /* number of different colors to use with bars */
  GRAPH_BAR_MENU_ITEMS  = 3,
  GRAPH_HPAD            = 20, /* space between two bars */
  STICK_HEIGHT          = 5,  /* width of those things on the axes -> __|___|__ */
};


enum
{
  COLOR_1, /* Here must be as many entries as */
  COLOR_2, /* the GRAPH_COLORS_NUM tells. */
  COLOR_3,
  COLOR_4,
  COLOR_BACKGROUND,
  COLOR_CHART_BACKGROUND,
  COLOR_AXES,
  COLOR_GRID,
  COLOR_X_TITLE,
  COLOR_Y_TITLE,
  COLOR_HINTS,
  COLOR_NUM,
};

static const guint8 graph_colors[COLOR_NUM][3] = {
  { 193, 102,  90 }, /* red    */
  { 117, 144, 174 }, /* blue   */
  { 131, 166, 127 }, /* green  */
  { 136, 127, 163 }, /* purple */
  { 238,   0,   0 }, /* wtf? */
  { 230, 230, 230 }, /* gray    -> CHART_BACKGROUND */
  {   0,   0,   0 }, /* black   -> AXES */
  { 238, 238,   0 }, /* yellow */
  {   0,   0,   0 },
  {   0,   0,   0 },
  {   0,   0,   0 },
};

typedef struct Color_
{
  guint8 r;
  guint8 g;
  guint8 b;
}
Color;


/* These are for future development */
#define KP_CHART_Y_AXIS_N   4

typedef struct KPChartYAxis_
{
  gdouble min;
  gdouble max;
  gdouble gmin;
  gdouble gmax;

  KPChartDataUnit unit;
  
} KPChartYAxis;

#define AXIS(object) ((object)->axis_data[(object)->axis])
/* </future> */

typedef struct _KPChartPrivateData KPChartPrivateData;
struct _KPChartPrivateData
{
  KPChartYAxis   axis_data[KP_CHART_Y_AXIS_N];
  
  GtkWidget     *hbox;
  GtkWidget     *vbox;
  GtkWidget     *graph_area;
  GtkWidget     *menu_item[GRAPH_BAR_MENU_ITEMS];
  GtkWidget     *menu;

  gchar        **menu_texts;
  gchar         *title;

  gboolean       chart_show_grid;
  gboolean       chart_show_grid_y;

  /* this tells what axis the group is using */
  guint          axis[KP_CHART_GROUPS_MAX];
  gdouble        graph_data[KP_CHART_GROUPS_MAX][KP_CHART_ITEMS_MAX];

  gchar         *group_titles[KP_CHART_GROUPS_MAX];
  gchar         *item_titles[KP_CHART_ITEMS_MAX];
  
  guint          graph_items;
  guint          graph_groups;
  guint          graph_type[KP_CHART_GROUPS_MAX];


  guint          item_title_max_len;

  GdkColor      *color[COLOR_NUM];
  Color          colors[COLOR_NUM];
  GdkGC         *gc[COLOR_NUM];

  KPTrainingLogDataMode  mode;
  KPChartDataUnit unit;

  GString       *param;
  
  KPViewModelType type;
  KPTrainingLog *log;
  GDate         *date;
  
  GDate         *date_s;
  GDate         *date_e;
};

#define KP_CHART_PRIVATE_DATA(widget) 	(((KPChartPrivateData*) \
			(KP_CHART (widget)->private_data)))

static GtkItemFactoryEntry entries[] = {
{"/Menu",                NULL, NULL,               0, "<Branch>",        NULL},
{"/Menu/Show grid",      NULL, menu_grid_clicked,  1, "<CheckItem>",     NULL},
{"/Menu/sep0",           NULL, NULL,               0, "<Separator>",     NULL},
{"/Menu/Show bars",      NULL, menu_type_clicked,  0, "<RadioItem>",     NULL},
{"/Menu/Show lines",     NULL, menu_type_clicked,  1, "/Menu/Show bars", NULL},
{"/Menu/Show stackbars", NULL, menu_type_clicked,  2, "/Menu/Show bars", NULL},
{"/Menu/Save Image",     NULL, save_image_clicked, 1, "<Item>",          NULL},
};

/*TODO: Add new things to menu
 *
 *  - Show vertical grid
 *  - Show title
 *  
 *  - Make horizontal sticks to show if horizontal grid is not shown
 */

enum
{
  PROP_LINE_SPACING = 0,
  PROP_END,
};

GType
kp_chart_get_type (void)
{
  static GType chart_type = 0;

  if (!chart_type) {
    static const GTypeInfo chart_info = {
      sizeof (KPChartClass),
      NULL,
      NULL,
      (GClassInitFunc) kp_chart_class_init,
      NULL,
      NULL,
      sizeof (KPChart),
      0,
      (GInstanceInitFunc) kp_chart_init,
      NULL
    };
    static const GInterfaceInfo view_model_info = {
      (GInterfaceInitFunc) kp_chart_view_model_init,
      NULL,
      NULL
    };
    chart_type = g_type_register_static (GTK_TYPE_WIDGET,
                                        "KPChart",
                                        &chart_info, 0);
    g_type_add_interface_static (chart_type,
                                 KP_TYPE_VIEW_MODEL,
                                &view_model_info);
  }
  return chart_type;
}

static void
kp_chart_view_model_init (KPViewModelIface *iface)
{
  iface->set_dmy = kp_chart_set_dmy;
  iface->get_dmy = kp_chart_get_dmy;
  iface->set_log = kp_chart_set_log;
  iface->unset_log = kp_chart_unset_log;
  iface->set_view_type = kp_chart_set_view_type;
  iface->get_view_type = kp_chart_get_view_type;
  iface->get_icon_name = kp_chart_get_icon_name;
}

static void
kp_chart_class_init (KPChartClass * class)
{
  GtkWidgetClass *widget_class = GTK_WIDGET_CLASS (class);

/* FIXME
  gtk_object_class->destroy = chart_destroy;
*/

  widget_class->realize = kp_chart_realize;
  widget_class->expose_event = kp_chart_expose;
  widget_class->size_request = kp_chart_size_request;
  widget_class->size_allocate = kp_chart_size_allocate;
  widget_class->button_press_event = kp_chart_button_press;
}

static void
kp_chart_size_request (GtkWidget *widget, GtkRequisition *requisition)
{
  KPChartPrivateData *p_data;

  p_data = KP_CHART_PRIVATE_DATA (KP_CHART (widget));

  requisition->width = p_data->graph_items * BAR_WIDTH_MIN
                    + (p_data->graph_items - 1) * GRAPH_HPAD
                    +  GRAPH_X_OFFSET_LEFT
                    +  GRAPH_X_OFFSET_RIGHT;
         
  requisition->height = GRAPH_Y_OFFSET_TOP
                      + GRAPH_Y_OFFSET_BOTTOM
                      + 100;
}

static void
kp_chart_size_allocate (GtkWidget *widget, GtkAllocation *allocation)
{
  KPChartPrivateData *p_data;
  KPChart *chart = NULL;

  g_return_if_fail (widget != NULL);
  g_return_if_fail (KP_IS_CHART (widget));
  g_return_if_fail (allocation != NULL);

  widget->allocation = *allocation;

  if (GTK_WIDGET_REALIZED (widget)) {
    chart = KP_CHART (widget);
    p_data = KP_CHART_PRIVATE_DATA (chart);

    gdk_window_move_resize (widget->window,
                            allocation->x,
                            allocation->y,
                            allocation->width,
                            allocation->height);
  }
}

static void
kp_chart_realize (GtkWidget *widget)
{
  KPChart *chart;
  GdkWindowAttr attributes;
  gint attributes_mask;

  g_return_if_fail (KP_IS_CHART (widget));

  GTK_WIDGET_SET_FLAGS (widget, GTK_REALIZED);
  chart = KP_CHART (widget);

  attributes.x = widget->allocation.x;
  attributes.y = widget->allocation.y;
  attributes.width = widget->allocation.width;
  attributes.height = widget->allocation.height;
  attributes.wclass = GDK_INPUT_OUTPUT;
  attributes.window_type = GDK_WINDOW_CHILD;
  attributes.event_mask = gtk_widget_get_events (widget) | GDK_EXPOSURE_MASK |
    GDK_BUTTON_PRESS_MASK;
  attributes.visual = gtk_widget_get_visual (widget);
  attributes.colormap = gtk_widget_get_colormap (widget);

  attributes_mask = GDK_WA_X | GDK_WA_Y | GDK_WA_VISUAL | GDK_WA_COLORMAP;
  widget->window =
    gdk_window_new (widget->parent->window, &attributes, attributes_mask);
  widget->style = gtk_style_attach (widget->style, widget->window);

  gdk_window_set_user_data (widget->window, widget);
  gtk_style_set_background (widget->style, widget->window, GTK_STATE_ACTIVE);

  gdk_window_show (widget->window);
}


static void
kp_chart_init (KPChart *chart)
{
  Color *color;
  KPChartPrivateData *p_data;
  GtkItemFactory *ifactory;
  gint n_menu_items;
  gint i;

  chart->private_data = g_malloc (sizeof (*p_data));
  p_data = KP_CHART_PRIVATE_DATA (chart);

  for (i=0; i < KP_CHART_GROUPS_MAX; i++) {
    p_data->group_titles[i] = NULL;
    p_data->axis[i] = 0;
    p_data->graph_type[i] = GRAPH_TYPE_BARS;
  }

  p_data->graph_type[1] = GRAPH_TYPE_LINES;
  
  p_data->chart_show_grid = FALSE;
  p_data->chart_show_grid_y = FALSE;
  p_data->menu_texts = NULL;
  p_data->title = NULL;
  p_data->gc[0] = NULL;
  p_data->log = NULL;
  
  p_data->graph_groups = 0;
  p_data->graph_items = 0;
  kp_chart_data_clear (chart);
  p_data->mode = KP_TRAINING_LOG_DATA_SUM;

  p_data->type = KP_VIEW_MODEL_TYPE_MONTH;
  p_data->date = g_date_new ();
  p_data->date_s = g_date_new ();
  p_data->date_e = g_date_new ();

  for (i = 0; i < COLOR_NUM; i++) {
    color = &p_data->colors[i];

    color->r = graph_colors[i][0];
    color->g = graph_colors[i][1];
    color->b = graph_colors[i][2];
  }

  n_menu_items = sizeof (entries) / sizeof (entries[0]);
  ifactory = gtk_item_factory_new (GTK_TYPE_MENU_BAR, "<main>", NULL);

  gtk_item_factory_create_items (ifactory, n_menu_items, entries, chart);

  p_data->menu = gtk_item_factory_get_widget (ifactory, "/Menu");

  for (i=0; i < KP_CHART_ITEMS_MAX; i++)
    p_data->item_titles[i] = NULL;

  p_data->param = g_string_new ("distance");
}

/**
 * kp_chart_new:
 * @title: 
 * @groups:
 * @items:
 *
 * Returns: A new #GtkWidget
 */
GtkWidget *
kp_chart_new (void)
{
  return GTK_WIDGET (g_object_new (kp_chart_get_type (), NULL));
}

void
kp_chart_set_param_to_chart (KPChart *chart, const gchar *param)
{
  KPChartPrivateData *p_data;
  
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);

  g_string_assign (p_data->param, param);
}


static void
kp_chart_data_clear (KPChart *chart)
{
  KPChartPrivateData *p_data;
  
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);

  memset (p_data->graph_data, 0,
          p_data->graph_items * 
          p_data->graph_groups * sizeof (gdouble));
}

void
kp_chart_set_chart_mode (KPChart *chart, KPTrainingLogDataMode mode)
{
  KPChartPrivateData *p_data;
  
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);
  p_data->mode = mode;
}

void
kp_chart_set_data_unit (KPChart *chart, KPChartDataUnit unit)
{
  KPChartPrivateData *p_data;
  
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);
  p_data->unit = unit;
}

void
kp_chart_set_axis (KPChart *chart, guint group, guint axis)
{
  KPChartPrivateData *p_data;
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);
  g_return_if_fail (group <= KP_CHART_GROUPS_MAX);
  g_return_if_fail (axis <= KP_CHART_AXES_MAX);
  
  p_data->axis[group] = axis;
}

void
kp_chart_update (KPChart *chart)
{
  KPChartPrivateData *p_data;
  static guint n = 0;
  
  g_return_if_fail (KP_IS_CHART (chart));

  n++;
  kp_debug ("Updating, %dth time.", n);
  
  p_data = KP_CHART_PRIVATE_DATA (chart);
  if (!KP_IS_TRAINING_LOG (p_data->log))
    return;
  
  update_dates (chart);
  update_chart_data (chart);
  gtk_widget_queue_draw (GTK_WIDGET (chart));
}

static void
update_dates (KPChart *chart)
{
  KPChartPrivateData *p_data;
  KPDate *d1, *d2;
  guint d, m, y;
  guint n_items;
  guint week;
  
  g_return_if_fail (KP_IS_CHART (chart));

  p_data = KP_CHART_PRIVATE_DATA (chart);

  d = g_date_get_day (p_data->date);
  m = g_date_get_month (p_data->date);
  y = g_date_get_year (p_data->date);
  
  switch (p_data->type)
  {
    case KP_VIEW_MODEL_TYPE_DAY:
      g_date_set_dmy (p_data->date_s, d, m, y);
      g_date_set_dmy (p_data->date_e, d, m, y);
      n_items = 1;
      break;

    case KP_VIEW_MODEL_TYPE_WEEK:
      if (kp_week_of_year (&week, &y, m, d)) {
        g_date_set_dmy (p_data->date_s, d, m, y);
        g_date_subtract_days (p_data->date_s,
                              g_date_get_weekday (p_data->date) - 1);
        g_date_set_julian (p_data->date_e, g_date_get_julian (p_data->date_s));
        g_date_add_days (p_data->date_e, 7-1);
        n_items = 7;
        break;
      } else
        g_return_if_reached ();
    
    case KP_VIEW_MODEL_TYPE_MONTH:
      n_items = kp_get_month_len (m, kp_leap (y));
      g_date_set_dmy (p_data->date_s, 1, m, y);
      g_date_set_dmy (p_data->date_e, n_items, m, y);
      break;
  
    case KP_VIEW_MODEL_TYPE_YEAR:
      g_date_set_dmy (p_data->date_s, 1, 1, y);
      g_date_set_dmy (p_data->date_e, 31, 12, y);
      n_items = 12;
      break;

    case KP_VIEW_MODEL_TYPE_ALL_TIME:

      
      if (!KP_IS_TRAINING_LOG (p_data->log))
        return;

      n_items = kp_training_log_get_n_years (p_data->log);
      if (n_items == 0)
        return;
      
      kp_debug ("All time selected, %p", p_data->log);

      d1 = kp_training_log_get_earliest_date (p_data->log);
      d2 = kp_training_log_get_latest_date (p_data->log);

      g_date_set_dmy (p_data->date_s, d1->d, d1->m, d1->y);
      g_date_set_dmy (p_data->date_e, d2->d, d2->m, d2->y);
      
      kp_date_free (d1);
      kp_date_free (d2);
      
      break;
      
    default:
      g_return_if_reached ();
  }
  
  kp_chart_set_items_num (chart, n_items);
}

void
kp_chart_set_n_items (KPChart *chart, guint n)
{
  KPChartPrivateData *p_data;
  
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);

  p_data->graph_items = n;
}

void
kp_chart_set_n_groups (KPChart *chart, guint n)
{
  KPChartPrivateData *p_data;
  
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);

  p_data->graph_groups = n;
}
 
guint 
kp_chart_get_n_items (KPChart *chart)
{
  KPChartPrivateData *p_data;
  
  g_return_val_if_fail (KP_IS_CHART (chart), 0);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  return p_data->graph_items;
}

guint 
kp_chart_get_n_groups (KPChart *chart)
{
  KPChartPrivateData *p_data;
  
  g_return_val_if_fail (KP_IS_CHART (chart), 0);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  return p_data->graph_groups;
}

static void
update_chart_data (KPChart *chart)
{
  return;
}


static void
kp_chart_set_dmy (KPViewModel *view, guint d, guint m, guint y)
{
  KPChartPrivateData *p_data;
  p_data = KP_CHART_PRIVATE_DATA (view);
  gchar *title;
  
  g_return_if_fail (KP_IS_VIEW_MODEL (view));
  g_return_if_fail (p_data != NULL);
  g_return_if_fail (g_date_valid_dmy (d, m, y));
 
  g_date_set_dmy (p_data->date, d, m, y);

  kp_chart_update (KP_CHART (view));
  title = g_strdup_printf ("%d.%d.%d", d, m, y);
  kp_chart_set_title (KP_CHART (view), title);
  g_free (title);
}

static void
kp_chart_get_dmy (KPViewModel *view, guint *d, guint *m, guint *y)
{
  KPChartPrivateData *p_data;
  p_data = KP_CHART_PRIVATE_DATA (view);
  
  g_return_if_fail (KP_IS_VIEW_MODEL (view));
  g_return_if_fail (p_data != NULL);

  if (d)
    *d = g_date_get_day (p_data->date);
  if (m)
    *m = g_date_get_month (p_data->date);
  if (y)
    *y = g_date_get_year (p_data->date);
}

static void
kp_chart_set_log (KPViewModel *view, KPTrainingLog *log)
{
  KPChartPrivateData *p_data;
  g_return_if_fail (KP_IS_VIEW_MODEL (view));
  g_return_if_fail (KP_IS_TRAINING_LOG (log));

  p_data = KP_CHART_PRIVATE_DATA (view);
  p_data->log = log;

  log_connect_signals (log, KP_CHART (view));
}

static void 
kp_chart_unset_log (KPViewModel *view)
{
  KPChartPrivateData *p_data;
  g_return_if_fail (KP_IS_VIEW_MODEL (view));

  kp_debug ("Unsetting log!");

  p_data = KP_CHART_PRIVATE_DATA (view);

  log_disconnect_signals (p_data->log, KP_CHART (view));
  p_data->log = NULL;
}


static void
log_changed (KPTrainingLog *log, KPChart *chart)
{
  kp_chart_update (KP_CHART (chart));
}
  
static void
log_connect_signals (KPTrainingLog *log, KPChart *chart)
{
  g_signal_connect (G_OBJECT (log), "changed",
                    G_CALLBACK (log_changed), chart);
}

static void
log_disconnect_signals (KPTrainingLog *log, KPChart *chart)
{
  g_signal_handlers_disconnect_by_func (log, log_changed, chart);
}

static void
kp_chart_set_view_type (KPViewModel *view, KPViewModelType type)
{
  KPChartPrivateData *p_data;
  p_data = KP_CHART_PRIVATE_DATA (view);
  guint title_len;
  guint len;
  
  switch (type)
  {
    case KP_VIEW_MODEL_TYPE_DAY:
      title_len = 16;
      len = 1;
      break;

    case KP_VIEW_MODEL_TYPE_WEEK:
      title_len = 16;
      len = 7;
      break;
    
    case KP_VIEW_MODEL_TYPE_MONTH:
      title_len = 16;
      len = 12;
      break;
  
    case KP_VIEW_MODEL_TYPE_YEAR:
      title_len = 16;
      len = 365 + kp_leap (g_date_get_year (p_data->date));
      break;

    case KP_VIEW_MODEL_TYPE_ALL_TIME:
      title_len = 4;
      g_return_if_fail (KP_IS_TRAINING_LOG (p_data->log));
      len = kp_training_log_get_n_years (p_data->log);
      break;

    default:
      g_return_if_reached ();
  }

  kp_chart_set_item_title_max_len (KP_CHART (view), title_len);
  
  kp_chart_set_size (KP_CHART (view), 1, len);
  p_data->type = type;
  
  kp_chart_update (KP_CHART (view));
}

static KPViewModelType
kp_chart_get_view_type (KPViewModel *view)
{
  KPChartPrivateData *p_data;
  g_return_val_if_fail (KP_IS_VIEW_MODEL (view), KP_VIEW_MODEL_TYPE_N);
  p_data = KP_CHART_PRIVATE_DATA (view);
  return p_data->type;
}

static gchar *
kp_chart_get_icon_name (KPViewModel *view)
{
  return g_strdup ("chart.png");
}

void
kp_chart_set_item_title_max_len (KPChart *chart, guint len)
{
  KPChartPrivateData *p_data;

  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);
  p_data->item_title_max_len = len;
}


static G_CONST_RETURN gchar *
kp_chart_get_item_title (KPChart *chart, guint item)
{
  KPChartPrivateData *p_data;

  g_return_val_if_fail (KP_IS_CHART (chart), NULL);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  g_return_val_if_fail (item <= p_data->graph_items, NULL);

  return p_data->item_titles[item];
}


void
kp_chart_set_item_title (KPChart *chart, guint item, const gchar *title)
{
  KPChartPrivateData *p_data;

  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);

  if (p_data->item_titles[item] != NULL)
    g_free (p_data->item_titles[item]);
        
  p_data->item_titles[item] = g_strdup (title);
}

gboolean
kp_chart_set_item_stock_title (KPChart *chart, guint item,
                               KPChartStockTitle title, guint value)
{
  KPChartPrivateData *p_data;

  g_return_val_if_fail (KP_IS_CHART (chart), FALSE);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  return FALSE;
}

gboolean
kp_chart_set_item_title_int (KPChart *chart, guint item, gint value)
{
  KPChartPrivateData *p_data;
  gchar buf[64];

  g_return_val_if_fail (KP_IS_CHART (chart), FALSE);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  g_snprintf (buf, sizeof (buf)-1, "%d", value);
  
  kp_chart_set_item_title (chart, item, buf);

  return TRUE;
}


static void
menu_grid_clicked (gpointer data, guint action, GtkWidget *menu_item)
{
  KPChartPrivateData *p_data;
  p_data = KP_CHART_PRIVATE_DATA (data);

  p_data->chart_show_grid_y = p_data->chart_show_grid =
      GTK_CHECK_MENU_ITEM (menu_item)->active;

  gtk_widget_queue_draw (GTK_WIDGET (data));
}

static void
save_image_clicked (KPChart *chart, guint action, GtkMenuItem *item)
{
  GtkWidget *parent;
  GdkPixbuf *buf;
  gchar *fn;
 
  parent = gtk_widget_get_toplevel (GTK_WIDGET (chart));
  fn = kp_gui_get_file_to_save (GTK_WINDOW (parent));

  if (fn) {
    buf = gdk_pixbuf_get_from_drawable (NULL,
                                        GDK_DRAWABLE (GTK_WIDGET (chart)->window),
                                        NULL, 0, 0, 0, 0,
                                        GTK_WIDGET (chart)->allocation.width,
                                        GTK_WIDGET (chart)->allocation.height);
  
    if (!gdk_pixbuf_save (buf, fn, "png", NULL, NULL))
      g_warning ("Couldn't save image to %s!", fn);
  }
}

static void
menu_type_clicked (gpointer data, guint action, GtkWidget *menu_item)
{
  KPChartPrivateData *p_data;

  p_data = KP_CHART_PRIVATE_DATA (data);

  if (GTK_CHECK_MENU_ITEM (menu_item)->active) {
    p_data->graph_type[0] = action;
    gtk_widget_queue_draw (GTK_WIDGET (data));
  }
}



static gint
kp_chart_button_press (GtkWidget *widget, GdkEventButton *event)
{
  KPChartPrivateData *p_data;
  p_data = KP_CHART_PRIVATE_DATA (KP_CHART (widget));

  if (event->type == GDK_BUTTON_PRESS) {
    GdkEventButton *bevent = event;

    gtk_menu_popup (GTK_MENU (p_data->menu),
                    NULL, NULL, NULL, NULL, bevent->button, bevent->time);
    return TRUE;
  }
  return FALSE;
}

static void
kp_chart_unit_str (gchar *buf, guint len, KPChartDataUnit unit, gdouble val)
{
  gchar *str;
 
  switch (unit)
  {
    case KP_CHART_UNIT_RAW:
      g_snprintf (buf, len, "%.2f", val);
      return;
          
    case KP_CHART_UNIT_MS:

      str = kp_date_mseconds_to_std_string (val);
      g_snprintf (buf, len, "%s", str);
      g_free (str);
      
      return;

    default:
      g_snprintf (buf, len, "%s", "Unknown unit");
      return;
  }
}

static guint
n_axes (KPChart *chart)
{
  KPChartPrivateData *p_data;
  guint max;
  guint i;

  g_return_val_if_fail (KP_IS_CHART (chart), 0);
  
  p_data = KP_CHART_PRIVATE_DATA (chart);
  
  for (i=0, max=0; i < p_data->graph_groups; i++)
    if (p_data->axis[i] > max)
      max = p_data->axis[i];

  return max + 1;
}




static void
calc_scale_data (KPChart *chart, DrawData *d)
{
  KPChartPrivateData *p_data;
  gdouble gsum_p[KP_CHART_AXES_MAX];
  gdouble gsum_n[KP_CHART_AXES_MAX];
  gdouble *d_ptr;
  guint axes;
  guint axis;
  guint i,j,k;

  g_return_if_fail (KP_IS_CHART (chart));
  
  p_data = KP_CHART_PRIVATE_DATA (chart);
 
  for (i=0; i < KP_CHART_AXES_MAX; i++) 
    d->gmax[i] = d->gmin[i] = d->max[i] = d->min[i] = 0;
  
  for (i = 0; i < (guint) p_data->graph_groups; i++) {

    axis = p_data->axis[i];
   
    for (j=0, d_ptr = &p_data->graph_data[i][0]; j < (guint) p_data->graph_items; j++, d_ptr++) {
      if (*d_ptr > d->max[axis])
        d->max[axis] = *d_ptr;

      if (*d_ptr < d->min[axis])
        d->min[axis] = *d_ptr;
    }
  }

  axes = n_axes (chart);
  
  /* Get the biggest group-value */
  for (i = 0; i < p_data->graph_items; i++) {

    for (k=0; k < KP_CHART_AXES_MAX; k++)
      gsum_p[k] = gsum_n[k] = 0;

    for (j = 0; j < p_data->graph_groups; j++) {
      axis = p_data->axis[j];
      if (p_data->graph_data[j][i] > 0)
        gsum_p[axis] += p_data->graph_data[j][i];
      else
        gsum_n[axis] += p_data->graph_data[j][i];
    }
    for (axis=0; axis < axes; axis++) {
      if (gsum_n[axis] > d->gmax[axis])
        d->gmax[axis] = gsum_n[axis];
      if (gsum_p[axis] > d->gmax[axis])
        d->gmax[axis] = gsum_p[axis];
      if (gsum_n[axis] < d->gmin[axis])
        d->gmin[axis] = gsum_n[axis];
      if (gsum_p[axis] < d->gmin[axis])
        d->gmin[axis] = gsum_p[axis];
    }
  }
}


typedef enum {
  KP_CHART_GRID_TYPE_LINE,
  KP_CHART_GRID_TYPE_DASHED,
  KP_CHART_GRID_TYPE_DOTTED
} KPChartGridType;
 

static void
draw_grid_line (KPChart *chart, GdkGC *gc, DrawData *d, guint x1, guint y1,
                guint x2, guint y2)
{
  KPChartPrivateData *p;
  KPChartGridType type;
  GdkWindow *window;
  guint x;
  guint y;

  window = GTK_WIDGET (chart)->window;
  
  g_return_if_fail (x1 == x2 || y1 == y2);
  
  p = KP_CHART_PRIVATE_DATA (chart);

  type = KP_CHART_GRID_TYPE_DASHED;

  switch (type) {
    case KP_CHART_GRID_TYPE_DOTTED:
      if (x1 == x2) { /* Vertical */
        for (x=x1, y=y1; y <= y2; y += 2)
          gdk_draw_point (window, gc, x, y);
      }
      else            /* Horizontal */
        for (x=x1, y=y1; x <= x2; x += 2)
          gdk_draw_point (window, gc, x, y);
      break;

    case KP_CHART_GRID_TYPE_DASHED:
      if (x1 == x2) { /* Vertical */
        for (x=x1, y=y1; y <= y2; y++)
          if (y % 4 != 0)
            gdk_draw_point (window, gc, x, y);
      }
      else {
        for (x=x1, y=y1; x <= x2; x++)
          if (x % 4 != 0)
            gdk_draw_point (window, gc, x, y);
      }
      break;

    case KP_CHART_GRID_TYPE_LINE:
      gdk_draw_line (window, gc, x1, y1, x2, y2);
      break;
  }
}


static gboolean
kp_chart_expose (GtkWidget *widget, GdkEventExpose *event)
{
  PangoLayout *layout;
  PangoRectangle rec;
  DrawData d;
  gchar *format[KP_CHART_AXES_MAX];

  d.xol = GRAPH_X_OFFSET_LEFT;
  d.xor = GRAPH_X_OFFSET_RIGHT;
  d.yot = GRAPH_Y_OFFSET_TOP;
  d.yob = GRAPH_Y_OFFSET_BOTTOM + STICK_HEIGHT;

  gdouble tmp;
  guint axes;

  guint axis;
  guint i;
  guint x;
  gint y;

  GdkGC *gc = widget->style->fg_gc[GTK_WIDGET_STATE (widget)];
  GdkGC *wgc = widget->style->white_gc;

  KPChartPrivateData *p = KP_CHART_PRIVATE_DATA (KP_CHART (widget));

  g_return_val_if_fail (KP_IS_CHART (widget), FALSE);
  
  char buf[127];

  d.w = widget->allocation.width;
  d.h = widget->allocation.height;

  /* Active x and y areas in the middle of the canvas */
  d.xa = (gdouble) d.w - d.xol - d.xor;
  d.ya = (gdouble) d.h - d.yot - d.yob;

  axes = n_axes (KP_CHART (widget));
  for (i=0; i < axes; i++)
    d.max[i] = d.min[i] = d.gmax[i] = d.gmin[i] = 0;

  gdk_window_clear (widget->window);
  
  if (!GTK_WIDGET_DRAWABLE (widget))
    return FALSE;

  /* Init colors */
  if (!p->gc[0]) {
    for (i = 0; i < COLOR_NUM; i++) {
      p->gc[i] = gdk_gc_new (widget->window);
      p->color[i] = get_color (widget, p->gc[i],
                               p->colors[i].r, p->colors[i].g, p->colors[i].b);
    }
  }

  /* Background */
  gdk_draw_rectangle (widget->window, p->gc[COLOR_CHART_BACKGROUND],
                      TRUE, 0, 0, d.w - 1, d.h - 1);
  
  layout = gtk_widget_create_pango_layout (widget, NULL);
 
  g_assert (KP_IS_CHART (widget));
 
  /* Do some axis calculations */
  calc_scale_data (KP_CHART (widget), &d);
 
  axes = n_axes (KP_CHART (widget));
  
  for (i=0; i < axes; i++) {
    kp_autoscale_axis (&d.min[i], &d.max[i]);
    kp_autoscale_axis (&d.gmin[i], &d.gmax[i]);

    format[i] = "%.2f";
    
    /* Calc how much space we need for hints in the
     * left side of y-axis. */

    sprintf (buf, format[i], d.max[i]);
    pango_layout_set_markup (layout, buf, -1);
    pango_layout_get_pixel_extents (layout, NULL, &rec);
  
    /* Add text-width and height to the margins */
    d.xol += rec.width;
    d.xor += rec.width;       
    d.yob += rec.height; /* X-axis labels are drawn with the same font */
 
    d.hint_text_width[i] = rec.width;
    
    /* Make active area smaller because we just made borders bigger */
    d.xa -= 2 * rec.width;
    d.ya -= rec.height;
  }

  /* Make a little space between two axis hints */
  d.xa -= 5;
  d.xol += 5;
  d.hint_text_width[0] += 5;

  /* Title */
  if (p->title) {
    pango_layout_set_markup (layout, p->title, -1);
    pango_layout_get_pixel_extents (layout, NULL, &rec);
    gdk_draw_layout (widget->window, gc,
                     d.xol + (d.xa - rec.width) / 2, /* Center */
                     0, layout);

    /* Add title height to the top margin and remove that from active area */
    tmp = (rec.height > d.yot) ? (rec.height - d.yot) + 5 : rec.height;
    
    d.yot += tmp;
    d.ya  -= tmp;
  }

  /* chart-background */
  gdk_draw_rectangle (widget->window, wgc,
                      TRUE, d.xol, d.yot, d.xa + 1, d.ya + 1);

  /* We draw grid-lines and x-axis' hints */
  for (i = 0, y = d.yob; i <= BAR_GRID_LINES; i++) {

    y = d.h - d.yob - i * (d.ya / (gdouble) BAR_GRID_LINES);

    /* Draw one tick */
    gdk_draw_line (widget->window, gc, d.xol - ((d.xol > 3) * 3), y, d.xol, y);
    
    /* Draw one grid-line. */
    if (p->chart_show_grid) {
      draw_grid_line (KP_CHART (widget), gc, &d, d.xol, y, d.xol + d.xa, y);
    }

    for (axis = 0, x = 0; axis < KP_CHART_AXES_MAX; axis++) {
      /* Y Axis hints */
      if (d.max[axis] - d.min[axis] != 0 || d.gmax[axis] - d.gmin[axis] != 0) {
          /* Unit */
          kp_chart_unit_str (buf, sizeof(buf)-1, p->unit, 
                             d.min[axis] +
                             ((gdouble) i * (d.max[axis] - d.min[axis])) / (gdouble) BAR_GRID_LINES);
        
        pango_layout_set_markup (layout, buf, -1);
        pango_layout_get_pixel_extents (layout, NULL, &rec);

        gdk_draw_layout (widget->window, gc, x,
                         y - (rec.height / 2), layout);
        x += d.hint_text_width[axis];
      }
    }
  }
 
  /* Box around the active area */
  gdk_draw_rectangle (widget->window, p->gc[COLOR_AXES], FALSE, 
                      d.xol, d.yot, d.w - d.xor - d.xol, d.h - d.yot - d.yob);
  
  draw_chart (KP_CHART (widget), NULL, gc, &d);

  return TRUE;
}

 
static void
draw_legends (KPChart *chart, GdkGC *gc, DrawData *d)
{
  KPChartPrivateData *p;
  PangoLayout *layout;
  guint legend_x = 10;
  guint axes;
  guint axis;
  guint g;
  gint lw, lh;
  guint y;
  gchar *str;
  gchar buf[64];
 
  g_return_if_fail (KP_IS_CHART (chart));
  
  p = KP_CHART_PRIVATE_DATA (chart);

  layout = gtk_widget_create_pango_layout (GTK_WIDGET (chart), " ");
  
  y = d->yot + d->ya + (d->yob / 2);

  axis = 0, g = 0;
  axes = n_axes (chart);
  for (axis=0; axis < axes; axis++) {
    g_snprintf (buf, sizeof (buf)-1, _("Axis %u: "), axis);

    pango_layout_set_text (layout, buf, -1);
    gdk_draw_layout (GTK_WIDGET (chart)->window, gc, legend_x + 5, y, layout);
    pango_layout_get_pixel_size (layout, &lw, &lh);
    legend_x += lw + 5;
    
    for (g=0; g < p->graph_groups; g++) {
      if (p->axis[g] == axis) {
         /* Create legends to bottom of the chart-picture */
        if (p->group_titles[g] == NULL) {
          p->group_titles[g] = g_strdup_printf (_("Group %u"), g);
        }
    
        str = p->group_titles[g];
        g_return_if_fail (g_utf8_validate(str, -1, NULL));
        
        pango_layout_set_text (layout, str, -1);
        gdk_draw_layout (GTK_WIDGET (chart)->window, gc, legend_x + 5, y, layout);
        pango_layout_get_pixel_size (layout, &lw, &lh);
     
        /* 10 is width of the color box */
        legend_x += 10 + lw;

        gdk_draw_rectangle (GTK_WIDGET (chart)->window, p->gc[g], TRUE,
                            legend_x, y + lh / 4, 10, 10);

        gdk_draw_rectangle (GTK_WIDGET (chart)->window, gc, FALSE,
                            legend_x, y + lh / 4, 10, 10);

        legend_x += LEGEND_SPACING_X;
      }
    }
  }
}    


static void
draw_chart (KPChart *chart, GtkWidget *widget, GdkGC *gc, DrawData *d)
{
  KPChartPrivateData *p;
  PangoLayout *layout;
  PangoRectangle rec;
  gdouble *d_ptr;
  gdouble gsum_p;               /* Positive group-sum (gsum > 0) */
  gdouble gsum_n;               /* Negative group-sum (gsum < 0) */
  const gchar *cstr;
  gchar buf[32];
  gint px = 1, py = 0;
  guint i, j, k, x, y, h, yan, yap, c_num;
  guint bar_width;
  guint axis;

  gc = GTK_WIDGET (chart)->style->fg_gc[GTK_WIDGET_STATE (GTK_WIDGET (chart))];
  p = KP_CHART_PRIVATE_DATA (chart);

  bar_width = (gdouble)(d->xa - (p->graph_items * GRAPH_HPAD))
            / (gdouble)(p->graph_items)
            / (gdouble)(p->graph_groups);

  if (bar_width < BAR_WIDTH_MIN)
    bar_width = BAR_WIDTH_MIN;
  else if (bar_width > BAR_WIDTH_MAX)
    bar_width = BAR_WIDTH_MAX;
  
  layout = gtk_widget_create_pango_layout (GTK_WIDGET (chart), "");

  /* Draw legends */
  draw_legends (chart, gc, d);
  
  /* This loop draws bars, some hints etc.. */
  for (i = 0, c_num = 0; i < p->graph_groups; i++, c_num++) {
    x = 0;
    y = 0;
    d_ptr = &p->graph_data[i][0];

    axis = p->axis[i];


    /* Variables used in this loop:
     * 
     * x            X-coordinate where something is drawn.
     * y            Y-coordinate where something is drawn.
     * px           X-coordinate in previous round in loop.
     * py           Y-coordinate in previous round in loop.
     * i            GraphGroup we are drawing.
     * j            Current GraphItem in GraphGroup i.
     * c_num        Color number.
     */

    for (j = 0, yap = 0, yan = 0; j < p->graph_items; j++, d_ptr++) {

      /* X-coord */
      /*x = d->xol + GRAPH_HPAD 
         + j * ((gdouble)(d->xa - GRAPH_HPAD) / (gdouble) p->graph_items);*/

      if (p->graph_type[i] == GRAPH_TYPE_LINES)
        x = d->xol + (j) * ((gdouble) d->xa / (gdouble) (p->graph_items-1));
      else
        x = d->xol + j * ((gdouble) d->xa / (gdouble) (p->graph_items));

      /* Positive chart-area's height
       * and negative chart-area's height */
      yap = d->ya * d->max[axis] / fabs (d->max[axis] - d->min[axis]);
      yan = d->ya - yap;

      if ((d->max[axis] - d->min[axis]) == 0)
        return;

      /* Draw X-axis' hint-text only once */
      if (i == 0) {

        if (p->chart_show_grid_y) {
          draw_grid_line (chart, gc, d, x, d->yot, x, d->yot + d->ya);
        }
        
        if ((cstr = kp_chart_get_item_title (chart, j)) != NULL) {
          strncpy (buf, cstr, p->item_title_max_len);
          buf[p->item_title_max_len] = 0;
        } else
          sprintf (buf, "%d", j + 1);

        /* Draw layout in the same place as ticks that are drawn below */
        pango_layout_set_text (layout, buf, -1);
        pango_layout_get_pixel_extents (layout, NULL, &rec);
        
        gdk_draw_layout (GTK_WIDGET (chart)->window, gc,
                         x - rec.width / 3l,
                         d->yot + d->ya + STICK_HEIGHT,
                         layout);
      }
      gdk_draw_line (GTK_WIDGET (chart)->window, gc,
                     x, d->yot + d->ya,
                     x, d->yot + d->ya + STICK_HEIGHT);

      /* Determine what type of diagram we are asked to draw
       * and draw it. */

      switch (p->graph_type[i]) {
      case GRAPH_TYPE_BARS:

        /* Coordinates */
        if (*d_ptr >= 0) {
          y = d->yot + yap - (*d_ptr / d->max[axis]) * yap;
          h = d->yot + yap - y;
        }
        else {
          h = (fabs (*d_ptr) / fabs (d->min[axis])) * yan;
          y = d->yot + yap;
        }
        x += i * bar_width;
      /*y -= GRAPH_AXIS_WIDTH;*/

        /* Bar */
        gdk_draw_rectangle (GTK_WIDGET (chart)->window, p->gc[c_num],
                            TRUE, x+1, y, bar_width-1, h);
        /* border */
        if (h > 0) {
          gdk_draw_rectangle (GTK_WIDGET (chart)->window, gc, FALSE,
                              x, y, bar_width, h);
        }
        break;

      case GRAPH_TYPE_LINES:
        if (*d_ptr >= 0)
          y = d->yot + yap - (*d_ptr / d->max[axis]) * yap;
        else
          y = d->yot + yap + (fabs (*d_ptr) / fabs (d->min[axis])) * yan;
     
        /* Draw circle to same place as ticks */
        
        /* TODO: variables for sizes */
        gdk_draw_arc (GTK_WIDGET (chart)->window, p->gc[c_num],
                      TRUE, x - 3, y - 3, 6, 6, 0, 64 * 360);
        
        if (j > 0)              /* line */
          for (k = 0; k < 2; k++)
            gdk_draw_line (GTK_WIDGET (chart)->window, p->gc[c_num],
                           px, py - k, x, y - k);
        break;

      /* TODO: Fix stack bars */
      case GRAPH_TYPE_STACK_BARS:

        gsum_p = gsum_n = 0;
        py = 0;

        if (i == 0 && d->gmax[axis] != 0) {
          for (k = 0, c_num = 0, gsum_p = 0;
               k < p->graph_groups; k++, c_num++) {

            d_ptr = &p->graph_data[k][j];

            if (*d_ptr >= 0) {
              y = d->yot + yap - yap * (gsum_p + *d_ptr) / fabs (d->gmax[axis]);
              h = yap * ((gdouble)*d_ptr / (gdouble)d->gmax[axis]);
            }
            else {
              y = d->yot + yap + py;
              h = yan * ((gsum_n + fabs (*d_ptr)) / fabs (d->gmin[axis]));
            }
            /* bar */
            gdk_draw_rectangle (GTK_WIDGET (chart)->window, p->gc[c_num],
                                TRUE, x + k * 5, y, bar_width,
                                d->h + (d->h > 0));

            /* border */
            if (h > 0)
              gdk_draw_rectangle (GTK_WIDGET (chart)->window, gc,
                                  FALSE, x-1 + k * 5, y, bar_width + 1, d->h+1);

            if (*d_ptr >= 0)
              gsum_p += *d_ptr;
            else
              gsum_n += fabs (*d_ptr);

            py = d->h;

            if (c_num >= GRAPH_COLORS_NUM)
              c_num = 0;
          }
        }
        break;
      }
      px = x;
      py = y;
    }
    if (c_num >= GRAPH_COLORS_NUM)
      c_num = 0;
  }
}

static GdkColor *
get_color (GtkWidget *widget, GdkGC *gc, guint8 r, guint8 g, guint8 b)
{
  GdkColor *color;

  g_assert (widget != NULL);

  color = g_malloc (sizeof (*color));
  color->red = r * (65535 / 255);
  color->green = g * (65535 / 255);
  color->blue = b * (65535 / 255);
  color->pixel = (gulong) (r * 65536 + g * 256 + b);

  gdk_color_alloc (gtk_widget_get_colormap (widget), color);
  gdk_gc_set_foreground (gc, color);

  return color;
}

void
kp_chart_set_group_title (KPChart *chart, guint group, const gchar *title)
{
  KPChartPrivateData *p_data;
  gchar *ptr;

  g_return_if_fail (title != NULL);
  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);
  g_return_if_fail (group < (guint)p_data->graph_groups);

  if ((ptr = p_data->group_titles[group]) != 0)
    g_free (ptr);
  
  p_data->group_titles[group] = g_strdup (title);
}


void
kp_chart_set_group_titles (KPChart *chart, guint groups, gchar **titles)
{
  KPChartPrivateData *p_data;
  guint i;

  g_return_if_fail (chart != NULL);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  g_return_if_fail (titles != NULL);
  g_return_if_fail (groups <= p_data->graph_groups);

  for (i = 0; i < groups; i++)
    kp_chart_set_group_title (chart, i, titles[i]);
}


void
kp_chart_set (KPChart *chart, gdouble **data)
{
  KPChartPrivateData *p_data;
  guint i, j;

  g_return_if_fail (KP_IS_CHART (chart));
  g_return_if_fail (data != NULL);

  p_data = KP_CHART_PRIVATE_DATA (chart);

  for (i = 0; i < p_data->graph_groups; i++)
    for (j = 0; j < p_data->graph_items; j++) {
      p_data->graph_data[i][j] = data[i][j];
    }

  kp_debug ("Chart set.");
  
  gtk_widget_queue_draw (GTK_WIDGET (chart));
}

void
kp_chart_set_group (KPChart *chart, guint group, gdouble *data)
{
  KPChartPrivateData *p_data;
  guint i;

  g_return_if_fail (chart != NULL);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  g_return_if_fail ((group <= p_data->graph_groups));
  g_return_if_fail (data != NULL);

  for (i = 0; i < p_data->graph_items; i++)
    p_data->graph_data[group][i] = data[i];

  gtk_widget_queue_draw (GTK_WIDGET (chart));
}


void
kp_chart_set_title (KPChart *chart, const gchar *title)
{
  KPChartPrivateData *p_data;

  g_return_if_fail (KP_IS_CHART (chart));
  p_data = KP_CHART_PRIVATE_DATA (chart);
  g_assert (p_data != NULL);

  if (p_data->title)
    g_free (p_data->title);

  if (title)
    p_data->title = g_strdup_printf ("<span foreground=\"#333333\" "
                                     "weight=\"bold\">%s</span>",
                                     title);
  else
    p_data->title = NULL;
}


void
kp_chart_set_size (KPChart *chart, gint groups, gint items)
{
  KPChartPrivateData *p_data;

  p_data = KP_CHART_PRIVATE_DATA (chart);

  g_return_if_fail (chart != NULL);

  if (groups == 0 && items > 0)
    groups = 1;
  
  kp_chart_set_groups_num (chart, groups);
  kp_chart_set_items_num (chart, items);
}


static void
kp_chart_set_items_num (KPChart *chart, gint items)
{
  KPChartPrivateData *p_data;

  g_return_if_fail (chart != NULL);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  p_data->graph_items = items;

  if (p_data->graph_items > 0 && p_data->graph_groups == 0)
    kp_chart_set_groups_num (chart, 1);
}


static void
kp_chart_set_groups_num (KPChart *chart, gint groups)
{
  KPChartPrivateData *p_data;

  g_return_if_fail (chart != NULL);
  p_data = KP_CHART_PRIVATE_DATA (chart);

  p_data->graph_groups = groups;
}



