/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

#ifndef GRUBCOLORPREVIEW_H
#define GRUBCOLORPREVIEW_H

//Qt
#include <qframe.h>

//KGRUBEditor
#include "../core/data.h"

const int space = 10;

/**
 * @short A preview of a GRUB Color using a QVector of GRUB Entries.
 *
 * Shows a preview of a GRUB Color using a QVector of GRUB Entries. 
 * Fully supports the GRUB 'color' command syntax, even blinking.
 */
class GRUBColorPreview : public QFrame
{
Q_OBJECT
public:
	/**
	* Default constructor.
	*
	* Since no color and list of entries are specified, shows a message 
	* that informs the user that the preview could not be drawn.
	*
	* @param parent QWidget parent. Self explanatory.
	*/
	GRUBColorPreview( QWidget *parent = 0 );
	/**
	* Secondary constructor where you can also specify the color to preview
	* and the GRUBEntries that will be drawn.
	*
	* It's equal to calling the default constructor and specifying the color
	* using setColor(), the entries using setEntries() and the highlighted
	* entry using setHighlightedEntry().
	*
	* @param color The color for which a preview will be created.
	* @param entries The list of entries that will be drawn.
	* @param highlightedEntry The # of the entry to be highlighted from the
	*                         items list.
	* @param parent QWidget parent. Self explanatory.
	*/
	GRUBColorPreview( const GRUB::ComplexCommand::Color color, const QVector<GRUB::ConfigFile::Entry> entries, const int highlightedEntry, QWidget *parent = 0 );

	/**
	* Sets the color to preview to the one specified.
	*
	* @param color The color for which a preview will be created.
	*/
	inline void setColor( const GRUB::ComplexCommand::Color color ) { m_color = color; };
	/**
	* Sets the entries to preview to the ones specified.
	*
	* @param entries The list of entries that will be drawn.
	*/
	inline void setEntries( const QVector<GRUB::ConfigFile::Entry> entries ) { m_entries = entries; };
	/**
	* Sets the number of the entry that will be highlighted.
	*
	* @param highlightedEntry The # of the entry to be highlighted from the
	*                         items list.
	*/
	inline void setHighlightedEntry( const int highlightedEntry ) { m_highlightedEntry = highlightedEntry; };

	/**
	* Resets blinking to avoid asynchronous blinking (since both normal
	* and highlighted stop blinking, when they start again, they will be synchronised).
	*/
	inline void resetBlinking() { blinkNormal = false; blinkHighlighted = false; };
private:
	/**
	* Clears all properties. Only used to set initial
	* values to the properties of the widget.
	*/
	inline void clear() { m_color.clear(); m_entries.clear(); m_highlightedEntry = 0; blinkNormal = false; blinkHighlighted = false; };
protected:
	/**
	* Paints the widget. This is where all the work is done.
	*/
	void paintEvent( QPaintEvent * );
private:
	GRUB::ComplexCommand::Color m_color;
	QVector<GRUB::ConfigFile::Entry> m_entries;
	int m_highlightedEntry;

	bool blinkNormal;
	bool blinkHighlighted;
};

#endif
