/***************************************************************************
 *   Copyright (C) 2008 by Konstantinos Smanis                             *
 *   kon.smanis@gmail.com                                                  *
 *                                                                         *
 *   This file is part of KGRUBEditor.                                     *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Street, Fifth Floor                                       *
 *   Boston, MA  02111-1307, USA.                                          *
 ***************************************************************************/

//Own
#include "data.h"

//Qt
#include <qregexp.h>

//KDE
#include <klocale.h>

namespace GRUB
{
	namespace ComplexCommand
	{
		//
		//Map
		//
		Map::Map()
		{
			clear();
		}
		Map::Map( const QString map )
		{
			clear();
			resolve( map );
		}
		void Map::clear()
		{
			m_toDrive = QString();
			m_fromDrive = QString();
		}
		void Map::resolve( const QString map)
		{
			m_toDrive = map.section( QRegExp( "\\s+" ), 0, 0 );
			m_fromDrive = map.section( QRegExp( "\\s+" ), 1 );
		}
		QString Map::result() const
		{
			return i18n( "%1 is mapped to %2", m_fromDrive, m_toDrive );
		}

		//
		//Color
		//
		Color::Color()
		{
			clear();
		}
		Color::Color( const QString color )
		{
			clear();
			resolve( color );
		}
		void Color::clear()
		{
			m_blinkNormal = false;
			m_normalBackground = QString();
			m_normalForeground = QString();
			m_blinkHighlighted = false;
			m_highlightedBackground = QString();
			m_highlightedForeground = QString();
		}
		void Color::resolve( QString color )
		
		{	m_blinkNormal = color.section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 0, 0 ).startsWith( "blink-" );
			m_normalBackground = color.section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 1 );
			m_normalForeground = color.section( QRegExp( "\\s+" ), 0, 0 ).section( "/", 0, 0 ).remove( "blink-" );
			m_blinkHighlighted = color.section( QRegExp( "\\s+" ), 1 ).section( "/", 0, 0 ).startsWith( "blink-" );
			m_highlightedBackground = color.section( QRegExp( "\\s+" ), 1 ).section( "/", 1 );
			m_highlightedForeground = color.section( QRegExp( "\\s+" ), 1 ).section( "/", 0, 0 ).remove( "blink-" );
		}
		QString Color::result() const
		{
			QString result = 
			( m_blinkNormal ? "blink-" : QString() ) + m_normalForeground + ( normalIsEnabled() ? "/" : QString() ) + m_normalBackground 
			+ " " 
			+ ( m_blinkHighlighted ? "blink-" : QString() ) + m_highlightedForeground + ( highlightedIsEnabled() ? "/" : QString() ) + m_highlightedBackground;
			return result;
		}
		QString Color::previewOptimisedColor( const QString color ) const
		{
			if ( color == "magenta" )
				return QString( "darkmagenta" );
			else if ( color == "light-magenta" )
				return QString( "magenta" );
			else if ( color == "light-red" )
				return QString( "orangered" );
			else
				return QString( color ).remove( "-" );
		}
		QString Color::mapOptimisedColor( const QString color ) const
		{
			if ( color == "black" )
				return i18n( "Black" );
			else if ( color == "blue" )
				return i18n( "Blue" );
			else if ( color == "green" )
				return i18n( "Green" );
			else if ( color == "cyan" )
				return i18n( "Cyan" );
			else if ( color == "red" )
				return i18n( "Red" );
			else if ( color == "magenta" )
				return i18n( "Magenta" );
			else if ( color == "brown" )
				return i18n( "Brown" );
			else if ( color == "light-gray" )
				return i18n( "Light-Gray" );
			else if ( color == "dark-gray" )
				return i18n( "Dark-Gray" );
			else if ( color == "light-blue" )
				return i18n( "Light-Blue" );
			else if ( color == "light-green" )
				return i18n( "Light-Green" );
			else if ( color == "light-cyan" )
				return i18n( "Light-Cyan" );
			else if ( color == "light-red" )
				return i18n( "Light-Red" );
			else if ( color == "light-magenta" )
				return i18n( "Light-Magenta" );
			else if ( color == "yellow" )
				return i18n( "Yellow" );
			else if ( color == "white" )
				return i18n( "White" );
			else
				return QString();
		}

		//
		//Password
		//
		Password::Password()
		{
			clear();
		}
		Password::Password( const QString password )
		{
			clear();
			resolve( password );
		}
		void Password::clear()
		{
			m_md5crypted = false;
			m_password = QString();
			m_configFile = QString();
		}
		void Password::resolve( QString password )
		{
			if ( password.startsWith( "--md5" ) )
			{
				m_md5crypted = true;
				m_password = password.section( QRegExp( "\\s+" ), 1, 1 );
				m_configFile = password.section( QRegExp( "\\s+" ), 2, 2 );
			}
			else
			{
				m_md5crypted = false;
				m_password = password.section( QRegExp( "\\s+" ), 0, 0 );
				m_configFile = password.section( QRegExp( "\\s+" ), 1, 1 );
			}
		}
		QString Password::result() const
		{
			return QString( ( m_md5crypted ? "--md5 " : QString() ) + m_password + ( !m_configFile.isEmpty() ? " " + m_configFile : QString() ) );
		}

		//
		//Kernel
		//
		Kernel::Kernel()
		{
			clear();
		}
		Kernel::Kernel( const QString kernel )
		{
			clear();
			resolve( kernel );
		}
		Kernel::Kernel( const QString kernel, const QString arguments )
		{
			m_kernel = kernel;
			m_arguments = arguments;
		}
		void Kernel::clear()
		{
			m_kernel = QString();
			m_arguments = QString();
		}
		void Kernel::resolve( QString kernel )
		{
			m_kernel = kernel.section( QRegExp( "\\s+" ), 0, 0 );
			m_arguments = kernel.section( QRegExp( "\\s+" ), 1 );
		}
		QString Kernel::result() const
		{
			return QString( m_kernel + " " + m_arguments );
		}
	}

	namespace ConfigFile
	{
		//
		//Entry
		//
		Entry::Entry()
		{
			clear();
		}
		void Entry::clear()
		{

			m_title = QString();
			m_lock = false;
			m_password.clear();
			m_root = QString();
			m_kernel.clear();
			m_initrd = QString();
			m_maps.clear();
			m_color.clear();
			m_chainLoader = QString();
			m_saveDefault = false;
			m_makeActive = false;
		}
		
		//
		//Settings
		//
		Settings::Settings()
		{
			clear();
		}
		void Settings::clear()
		{
			m_splashImage = QString();
			m_gfxMenu = QString();
			m_timeout = -1;
			m_default = -1;
			m_fallback = -1;
			m_hiddenMenu = false;
			m_maps.clear();
			m_color.clear();
			m_password.clear();
		}
	}

	namespace Misc
	{
		//
		//Device
		//
		Device::Device()
		{
			clear();
		}
		Device::Device( const QString device, const QString partition, const QString mountpoint )
		{
			clear();
			m_device = device;
			m_partition = partition;
			m_mountPoint = mountpoint;
		}
		void Device::clear()
		{
			m_device = QString();
			m_partition = QString();
			m_grubDevice = QString();
			m_grubPartition = QString();
			m_mountPoint = QString();
			m_uuid = QString();
		}
	}
}
