/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Date  : 05-08-2005
 * Description : Dialog to allow photo properties to be edited
 *
 * Copyright 2005 by M. Asselstine

 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "photoproperties.h"

#include "photo.h"
#include "previewdlg.h"

#include <qlabel.h>
#include <qimage.h>
#include <kglobal.h>
#include <qregexp.h>
#include <klocale.h>
#include <qcursor.h>
#include <qstring.h>
#include <qpixmap.h>
#include <kguiitem.h>
#include <qlistbox.h>
#include <qstrlist.h>
#include <knuminput.h>
#include <qlineedit.h>
#include <qtextedit.h>
#include <qcheckbox.h>
#include <qcombobox.h>
#include <qnamespace.h>
#include <qvalidator.h>
#include <kpushbutton.h>
#include <qradiobutton.h>
#include <kapplication.h>
#include <qkeysequence.h>
#include <kstandarddirs.h>

namespace
{
    /**
     * Adds a blank item to the combobox list and selects it.
     * @param cb Reference to the combobox object to add the blank to
     */
    void addBlankComboItem(QComboBox& cb)
    {
        if( !cb.listBox()->findItem(" ", Qt::ExactMatch) )
        {
            cb.insertItem(" ");
        }
    }

    /**
     * Removes a blank item from the combobox list if one exists.
     * @param cb Reference to the combobox object to add the blank to
     */
    void removeBlankComboItem(QComboBox& cb)
    {
        QListBoxItem* lbi = cb.listBox()->findItem(" ", Qt::ExactMatch);
        if( lbi )
        {
            delete lbi;
        }
    }
}

PhotoProperties::PhotoProperties(QWidget *parent, const char *name)
    :PhotoPropertiesUI(parent, name)
    , m_customWidth(768)
    , m_customHeight(1024)
    , m_batchMode(FALSE)
    , m_activePhoto(0L)
    , m_portraitSizings(TRUE)
{
    // setup widgets
    setPublic(TRUE);
    m_pixmap->setScaledContents(false);

    m_zoom->setGuiItem(KGuiItem("","viewmag"));
    m_rotate->setGuiItem(KGuiItem("","rotate"));

    // setup photo size combobox
    m_sizings[i18n("Square")] = "75x75";
    m_sizings[i18n("Thumb")] = "75x100";
    m_sizings[i18n("Small")] = "180x240";
    m_sizings[i18n("Medium")] = "375x500";
    m_sizings[i18n("Large")] = "768x1024";
    m_sizings[i18n("Original")] = "74x74";
    m_sizings[i18n("Custom")] = "768x1024";
    for(SizeMap::Iterator it(m_sizings.begin()); it != m_sizings.end(); ++it)
    {
        m_sizes->insertItem(it.key());
    }

    // disable photo resize if neither JPEG or PNG write support found
    QStrList fmts = QImage::outputFormats();
    if( !fmts.contains("JPEG") && !fmts.contains("PNG") )
        m_sizes->setEnabled(false);

    // create connections
    connect(m_add, SIGNAL(clicked()), SLOT(addSelectedTag()));
    connect(m_rotate, SIGNAL(clicked()), SLOT(rotatePhoto()));
    connect(m_zoom, SIGNAL(clicked()), SLOT(showLargerPreview()));
    connect(m_remove, SIGNAL(clicked()), SLOT(removeSelectedTags()));
    connect(m_public, SIGNAL(toggled(bool)), SLOT(setPublic(bool)));
    connect(m_private, SIGNAL(toggled(bool)), SLOT(setPrivate(bool)));
    connect(m_tags, SIGNAL(selectionChanged()), SLOT(updateRemoveBtnState()));
    connect(m_availableTags, SIGNAL(activated(const QString&)), SLOT(insertNewTag(const QString&)));
    connect(m_availableTags, SIGNAL(textChanged(const QString&)), SLOT(updateAddBtnState(const QString&)));

    connect(m_title, SIGNAL(textChanged(const QString&)), SLOT(updateTitle(const QString&)));
    connect(m_desc, SIGNAL(textChanged()), SLOT(updateDescription()));
    connect(m_sizes, SIGNAL(activated(const QString&)), SLOT(setSizeSelection(const QString&)));
    connect(m_photosets, SIGNAL(activated(const QString&)), SLOT(updatePhotoset(const QString&)));
    connect(m_photosets, SIGNAL(textChanged(const QString&)), SLOT(updatePhotoset(const QString&)));
    connect(m_licenses, SIGNAL(activated(const QString&)), SLOT(updateLicense(const QString&)));
    connect(m_width, SIGNAL(valueChanged(int)), SLOT(setWidth(int)));
    connect(m_height, SIGNAL(valueChanged(int)), SLOT(setHeight(int)));
    connect(m_public, SIGNAL(toggled(bool)), SLOT(updatePublic(bool)));
    connect(m_private, SIGNAL(toggled(bool)), SLOT(updatePrivate(bool)));
    connect(m_family, SIGNAL(toggled(bool)), SLOT(updateFamily(bool)));
    connect(m_friends, SIGNAL(toggled(bool)), SLOT(updateFriends(bool)));

    clearAndDisable();
}

PhotoProperties::~PhotoProperties()
{
}

void PhotoProperties::setAvailableTags(const QStringList& lst)
{
    m_availableTags->insertStringList( lst );
    m_availableTags->setCurrentText( "" );
}

void PhotoProperties::setPhotosets(const QStringList& titles, const QString& sel)
{
    // update the photosets combobox
    m_photosets->clear();
    m_photosets->insertStringList(titles);
    m_photosets->insertItem(i18n("<photostream only>"), 0);

    if( sel == QString::null )
        m_photosets->setCurrentText(i18n("<photostream only>"));
    else
        m_photosets->setCurrentText(sel);
}

void PhotoProperties::setLicenses(const QStringList& licenses)
{
    // update the licenses combobox
    m_licenses->clear();
    m_licenses->setEnabled(TRUE);
    m_licenses->insertStringList(licenses);
    m_licenses->setCurrentItem(0);
}

void PhotoProperties::rotatePhoto()
{
    Q_ASSERT(!m_batchMode);

    // Rotate the photo being edited, and our preview image
    m_activePhoto->rotate();
    m_pixmap->setPixmap(m_activePhoto->preview());

    // Update the sizings as we have most likely changed our layout
    matchSizingsToPhotoLayout();
}

void PhotoProperties::setNeitherPublicOrPrivate()
{
    // block signals temporarily to prevent the
    // automatic setting of the m_private toggle
    m_public->blockSignals(true);
    m_public->setChecked(false);
    m_public->blockSignals(false);

    m_private->setChecked(false);
}

QStringList PhotoProperties::tags() const
{
    QStringList lst;

    // Add each tag from the tags list
    for( uint i = 0; i < m_tags->count(); ++i )
    {
        // quote any multi word tags
        if( m_tags->text(i).contains(QRegExp("\\s+")) )
            lst += QString("\"" + m_tags->text(i) + "\"");
        else
            lst += m_tags->text(i);
    }
    return lst;
}

void PhotoProperties::setTags(const QStringList &lst)
{
    QRegExp rx("^\".*\"$");
    QStringList::ConstIterator it;

    // clear the tags
    m_tags->clear();

    // Add each tag, remove surounding quotes if found
    for( it = lst.constBegin(); it != lst.constEnd(); ++it )
    {
        if( rx.search(*it) == 0 )
            m_tags->insertItem((*it).mid(1,(*it).length()-2));
        else
            m_tags->insertItem(*it);
    }
}

void PhotoProperties::setPhotoSize(const QString &size)
{
    // Add a blank item, used for batch mode
    if( m_batchMode )
    {
        addBlankComboItem(*m_sizes);
        setSizeSelection(" ", FALSE);
    }
    else
    {
        removeBlankComboItem(*m_sizes);
        setSizeSelection(size.section(' ', 0, 0), FALSE);
    }

    if( m_sizes->currentText() == i18n("Custom") )
    {
        m_width->setValue(size.section(' ', 1, 1).section('x', 0, 0).toInt());
        m_height->setValue(size.section(' ', 1, 1).section('x', 1, 1).toInt());
    }
}

void PhotoProperties::insertNewTag( const QString &str )
{
    m_tags->insertItem( str );
    m_availableTags->setCurrentText( "" );
    updateTags();
}

void PhotoProperties::setPublic(bool pub)
{
    // Toggle both public and private radio buttons
    m_public->setChecked(pub);
    m_private->setChecked(!pub);

    // Enable/disable the family and friends checkboxes accordingly
    m_family->setEnabled( !pub );
    m_friends->setEnabled( !pub );
}

void PhotoProperties::setPrivate(bool priv)
{
    setPublic(!priv);
}

void PhotoProperties::enableUpdates(bool enable)
{
    if( enable )
    {
        m_title->blockSignals(FALSE);
        m_desc->blockSignals(FALSE);
        m_sizes->blockSignals(FALSE);
        m_public->blockSignals(FALSE);
        m_private->blockSignals(FALSE);
        m_family->blockSignals(FALSE);
        m_friends->blockSignals(FALSE);
        m_width->blockSignals(FALSE);
        m_height->blockSignals(FALSE);
        m_licenses->blockSignals(FALSE);
        m_photosets->blockSignals(FALSE);
    }
    else
    {
        m_title->blockSignals(TRUE);
        m_desc->blockSignals(TRUE);
        m_sizes->blockSignals(TRUE);
        m_public->blockSignals(TRUE);
        m_private->blockSignals(TRUE);
        m_family->blockSignals(TRUE);
        m_friends->blockSignals(TRUE);
        m_width->blockSignals(TRUE);
        m_height->blockSignals(TRUE);
        m_licenses->blockSignals(TRUE);
        m_photosets->blockSignals(TRUE);
    }
}

void PhotoProperties::setSizeSelection(const QString& sel ,bool update)
{
    if( m_sizes->currentText() != sel )
    {
        m_sizes->setCurrentText(sel);
    }

    m_width->blockSignals(TRUE);
    m_height->blockSignals(TRUE);
    if( sel != i18n("Custom") )
    {
        m_width->setEnabled(FALSE);
        m_height->setEnabled(FALSE);

        m_width->setValue(m_sizings[sel].section('x', 0, 0).toInt());
        m_height->setValue(m_sizings[sel].section('x', 1, 1).toInt());
    }
    else
    {
        m_width->setEnabled(TRUE);
        m_height->setEnabled(TRUE);

        m_width->setValue(m_customWidth);
        m_height->setValue(m_customHeight);
    }
    m_width->blockSignals(FALSE);
    m_height->blockSignals(FALSE);

    if( update )
    {
        updateSize(QString("%1 %2x%3").arg(sel).arg(m_width->value()).arg(m_height->value()));
    }
}

void PhotoProperties::setWidth(int width)
{
    m_customWidth = width;
    updateSize(QString("%1 %2x%3").arg(m_sizes->currentText()).arg(width).arg(m_height->value()));
}

void PhotoProperties::setHeight(int height)
{
    m_customHeight = height;
    updateSize(QString("%1 %2x%3").arg(m_sizes->currentText()).arg(m_width->value()).arg(height));
}

void PhotoProperties::updateRemoveBtnState( )
{
    QListBoxItem *item = m_tags->firstItem( );

    // check if any items are selected
    while( item != 0L )
    {
        // is this one selected, we only need one
        if( item->isSelected( ) )
            return m_remove->setEnabled( TRUE );

        // check the next item
        item = item->next( );
    }
    // none selected
    return m_remove->setEnabled( FALSE );
}

void PhotoProperties::updateAddBtnState( const QString &str )
{
    // is there any text in the combo
    if( str.length( ) > 0 )
        return m_add->setEnabled( TRUE );   // yes, enable add button
    else
        return m_add->setEnabled( FALSE );  // no, disable it
}

void PhotoProperties::removeSelectedTags( )
{
    QListBoxItem *next;
    QListBoxItem *item = m_tags->firstItem( );

    // remove any selected tags
    while( item != 0L )
    {
        // store next item
        next = item->next( );

        // remove if selected
        if( item->isSelected( ) )
            delete item;

        // move on to next item
        item = next;
    }
    updateTags();
}

void PhotoProperties::addSelectedTag()
{
    bool already;
    QString tagname = m_availableTags->currentText( );

    // add the tag to m_tags
    already = FALSE;
    for( int i = (int)m_tags->count( ) - 1; i >= 0; --i )
    {
        if( tagname == m_tags->text( i ) )
        {
            already = TRUE;
            break;
        }
    }
    if( !already )
    {
        m_tags->insertItem( tagname );
        updateTags();
    }

    // add the tag to the available tags combobox
    already = FALSE;
    for( int i = m_availableTags->count( ) - 1; i >= 0; --i )
    {
        if( tagname == m_availableTags->text( i ) )
        {
            already = TRUE;
            break;
        }
    }
    if( !already )
    {
        m_availableTags->insertItem( tagname );
    }

    // clear out the combobox
    m_availableTags->setCurrentText( "" );
}

void PhotoProperties::clearAndDisable()
{
    m_batchMode = FALSE;
    m_activePhoto = 0L;

    // Do some cleanup
    m_title->clear();
    m_pixmap->clear();
    m_desc->clear();
    m_tags->clear();

    // disable zoom and rotate buttons on the dialog
    m_zoom->setEnabled(FALSE);
    m_rotate->setEnabled(FALSE);

    setDisabled(TRUE);
}

void PhotoProperties::editSinglePhoto(Photo& photo)
{
    m_batchMode = FALSE;
    m_activePhoto = &photo;     // store pointer to active photo item

    // Ensure properties widget is enabled
    setDisabled(FALSE);

    // enable zoom and rotate buttons on the dialog
    m_zoom->setEnabled(TRUE);
    m_rotate->setEnabled(TRUE);

    // set dialog widgets to match the photo's attributes
    enableUpdates(FALSE);
    m_title->setText(photo.title());
    m_pixmap->setPixmap(photo.preview());
    m_desc->setText(photo.description());
    setPublic(photo.exposed());
    m_family->setChecked(photo.family());
    m_friends->setChecked(photo.friends());
    setTags(photo.tags());
    setPhotoSize(photo.size());
    m_photosets->setCurrentText(photo.photoset());
    m_licenses->setCurrentText(photo.license());
    enableUpdates(TRUE);

    // Update the defaults sizings width and height according
    // to the photo's layout (portrait or landscape)
    matchSizingsToPhotoLayout();

    // Remove any blank combo items that might be hanging
    // around from a previous batch edit session.
    removeBlankComboItem(*m_photosets);
    removeBlankComboItem(*m_licenses);
}

void PhotoProperties::editPhotoBatch(QPtrList<Photo> photos)
{
    m_batchMode = TRUE;
    m_activePhoto = 0L;
    m_batchPhotos = photos;     // Keep a local copy of the list of photos being edited

    // Ensure properties widget is enabled
    setDisabled(FALSE);

    // disable zoom and rotate buttons on the dialog
    m_zoom->setEnabled(FALSE);
    m_rotate->setEnabled(FALSE);

    // Display the batch mode image in the preview area.
    KStandardDirs *dirs = KApplication::kApplication()->dirs();
    m_pixmap->setPixmap( QPixmap( dirs->findResource( "data", "kflickr/batchmode.png" ) ) );

    // Clear out all properties values. The idea is that only properties set in batch mode will
    // be applied to all the photos in the batch, all other properties are left untouched.
    enableUpdates(FALSE);
    m_title->clear();
    m_desc->clear();
    m_tags->clear();
    setPhotoSize(" ");
    setNeitherPublicOrPrivate();

    // Update the defaults sizings width and height.
    matchSizingsToPhotoLayout();

    // Add blank items to the comboboxes and select them. Again this is to keep
    // the the theme of only properties set in batch mode propogating to the batch.
    addBlankComboItem(*m_photosets);
    m_photosets->setCurrentText(" ");
    addBlankComboItem(*m_licenses);
    m_licenses->setCurrentText(" ");
    enableUpdates(TRUE);
}

void PhotoProperties::updateTitle(const QString& txt)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->title(txt);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->title(txt);
        }
    }
}

void PhotoProperties::updateDescription()
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->description(m_desc->text());
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        QString desc = m_desc->text();
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->description(desc);
        }
    }
}

void PhotoProperties::updateSize(const QString& newsize)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->size(newsize);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->size(newsize);
        }
        removeBlankComboItem(*m_sizes);
    }
}

void PhotoProperties::updatePublic(bool b)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->exposed(b);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->exposed(b);
        }
    }
}

void PhotoProperties::updatePrivate(bool b)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->exposed(!b);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->exposed(!b);
        }
    }
}

void PhotoProperties::updateFamily(bool b)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->family(b);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->family(b);
        }
    }
}

void PhotoProperties::updateFriends(bool b)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->friends(b);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->friends(b);
        }
    }
}

void PhotoProperties::updateTags()
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->tags(tags());
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        QStringList phototags(tags());
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->tags(phototags);
        }
    }
}

void PhotoProperties::updatePhotoset(const QString& set)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->photoset(set);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->photoset(set);
        }
        removeBlankComboItem(*m_photosets);
    }
}

void PhotoProperties::updateLicense(const QString& lic)
{
    if( !m_batchMode && m_activePhoto )
    {
        m_activePhoto->license(lic);
    }
    else if( m_batchMode && !m_batchPhotos.isEmpty() )
    {
        for( Photo* photo( m_batchPhotos.first()) ; photo; photo = m_batchPhotos.next() )
        {
            photo->license(lic);
        }
        removeBlankComboItem(*m_licenses);
    }
}

void PhotoProperties::showLargerPreview()
{
    // Create large preview dialog
    PreviewDlg *dlg = new PreviewDlg(this);
    dlg->displayPhoto(m_activePhoto->URL(), m_activePhoto->rotation());

    // Give visual feedback
    setCursor(ForbiddenCursor);
    dlg->exec();

    // Cleanup
    setCursor(ArrowCursor);
    delete dlg;
}

void PhotoProperties::matchSizingsToPhotoLayout()
{
    // Only have to swap things if they don't already matchup
    if( !m_batchMode && ((m_activePhoto->isPortrait() && m_portraitSizings) || (!m_activePhoto->isPortrait() && !m_portraitSizings)) )
    {
        return;
    }
    else if(m_batchMode && !m_portraitSizings) // In batch mode we want to use landscape sizings
    {
        return;
    }

    // Update our sizings map to reflect the swap
    for( SizeMap::Iterator it(m_sizings.begin()), end(m_sizings.end()); it != end; ++it )
    {
        QString width = it.data().section('x', 0, 0);
        QString height = it.data().section('x', 1, 1);

        it.data() = QString("%1x%2").arg(height).arg(width);
    }

    // Swap the spinbox width and height values too. Block
    // signals to avoid unnecessary calls to size updates.
    m_width->blockSignals(TRUE);
    m_height->blockSignals(TRUE);

    int tmp = m_width->value();
    m_width->setValue(m_height->value());
    m_height->setValue(tmp);

    m_width->blockSignals(FALSE);
    m_height->blockSignals(FALSE);

    // update our portrait sizings flag accordingly
    m_portraitSizings = !m_portraitSizings;

    // Update the size for the photo being edited. Not relavent for batch mode as we only want to have
    // the size change on explicit user input.
    if(!m_batchMode)
    {
        updateSize(QString("%1 %2x%3").arg(m_sizes->currentText()).arg(m_width->value()).arg(m_height->value()));
    }
}



#include "photoproperties.moc"
