/* ============================================================
 * Author: M. Asselstine <asselsm@gmail.com>
 * Description : Widget to edit a batch of photos.
 *
 * Copyright 2008 by M. Asselstine
 *
 * This program is free software; you can redistribute it
 * and/or modify it under the terms of the GNU General
 * Public License as published by the Free Software Foundation;
 * either version 2, or (at your option)
 * any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * ============================================================ */

#include "batchwidget.h"

#include <QDebug>
#include <QRegExp>
#include <QStringList>
#include <QSqlTableModel>
#include <QModelIndexList>
#include <QStringListModel>

#include "previewmgr.h"
#include "ui_editorwidget.h"

// Private data class
class BatchWidget::PrivateData
{
public:
  PrivateData()
    : editRows(QList<int>())
    , model(0)
    , sizes(QStringList())
  {
    QStringList str;

    str << i18n("Square") << i18n("Thumb") << i18n("Small") << i18n("Medium");
    str << i18n("Large") << i18n("Original") << i18n("Custom");
    sizes.setStringList(str);
  }
  ~PrivateData()
  {
  }

  QModelIndex getIndex(const QString& field)
  {
    Q_ASSERT( model && model->rowCount() > 0 );
    return model->index(0, model->fieldIndex(field));
  }

  void setData(const QString& field, const QVariant& data)
  {
    if( !model || editRows.size() < 1 )
      return;

    foreach( int row, editRows )
    {
      Q_ASSERT( model && model->rowCount() > 0 && row < model->rowCount() );
      model->setData(model->index(row, model->fieldIndex(field)), data);
    }
  }

public:
  Ui::editorwidget ui;              /// The widget UI

  QList<int> editRows;
  QSqlTableModel* model;
  QStringListModel tags;
  QStringListModel sizes;
};

BatchWidget::BatchWidget(QWidget* parent, Qt::WindowFlags f)
  : QFrame(parent, f)
  , d(*(new BatchWidget::PrivateData))
{
  d.ui.setupUi(this);

  d.ui.tags->setModel(&d.tags);
  d.ui.sizes->setModel(&d.sizes);
  d.ui.desc->setCheckSpellingEnabled(true);

  d.ui.rotateCW->setGuiItem(KGuiItem("","rotate_cw"));
  d.ui.rotateCCW->setGuiItem(KGuiItem("","rotate_ccw"));

  connect(d.ui.tags->selectionModel(),
	  SIGNAL(selectionChanged(const QItemSelection&, const QItemSelection&)),
	  SLOT(tagsSelectionChanged(const QItemSelection&, const QItemSelection&)));
}

BatchWidget::~BatchWidget()
{
  delete &d;
}

void BatchWidget::setModel(QSqlTableModel* model)
{
  d.model = model;
}

void BatchWidget::editRows(QList<int>& rows)
{
  // Clear this now so that any changes made throughout this function don't
  // get applied to any photographs, we will store the rows at the end
  // of the function, thus setting which rows are part of the batch.
  d.editRows.clear();

  // Set the title
  d.ui.title->clear();

  // Set the description
  d.ui.desc->clear();

  // Set the size.
  d.ui.sizes->setCurrentIndex(-1);

  // Set the license
  d.ui.licenses->setCurrentIndex(-1);

  // Set the photoset
  d.ui.photosets->clearEditText();

  // Set the correct privacy radio button, we have to disable exclusivity in order
  // to actually get neither of the selected, otherwise when one is deselected, the
  // other is selected automatically.
  d.ui.pub->setAutoExclusive(false);
  d.ui.priv->setAutoExclusive(false);
  d.ui.pub->setChecked(false);
  d.ui.priv->setChecked(false);
  d.ui.family->setEnabled(false);
  d.ui.friends->setEnabled(false);
  d.ui.family->setChecked(false);
  d.ui.friends->setChecked(false);
  d.ui.pub->setAutoExclusive(true);
  d.ui.priv->setAutoExclusive(true);

  // Set the tags
  d.tags.setStringList(QStringList());

  // Remeber this has to happen last, see above.
  d.editRows = rows;
}

void BatchWidget::setTagsModel(QStringListModel& model)
{
  d.ui.availableTags->setModel(&model);
}

void BatchWidget::setLicensesModel(QStringListModel& model)
{
  d.ui.licenses->setModel(&model);
}

void BatchWidget::setPhotosetsModel(QStringListModel& model)
{
  d.ui.photosets->setModel(&model);
}

void BatchWidget::on_title_textEdited(const QString& text)
{
  // Store the new value
  d.setData("title", QVariant(text));
}

void BatchWidget::on_rotateCW_clicked()
{
  int rcurr = d.model->data(d.getIndex("rotation")).toInt();
  if( rcurr == 270 )
  {
    d.setData("rotation", QVariant(0));
  }
  else
  {
    d.setData("rotation", QVariant(rcurr + 90));
  }
}

void BatchWidget::on_rotateCCW_clicked()
{
  int rcurr = d.model->data(d.getIndex("rotation")).toInt();
  if( rcurr == 0 )
  {
    d.setData("rotation", QVariant(270));
  }
  else
  {
    d.setData("rotation", QVariant(rcurr - 90));
  }
}

void BatchWidget::on_desc_textChanged()
{
  // Store the new value
  d.setData("description", QVariant(d.ui.desc->toPlainText()));
}

void BatchWidget::on_priv_clicked(bool b)
{
  // Store the new value
  d.setData("exposed", QVariant(!b));

  // Update widget states
  if( !b )
  {
    d.ui.family->setEnabled(false);
    d.ui.friends->setEnabled(false);
  }
  else
  {
    d.ui.family->setEnabled(true);
    d.ui.friends->setEnabled(true);
  }
}

void BatchWidget::on_pub_clicked(bool b)
{
  // Store the new value
  d.setData("exposed", QVariant(b));

  // Update widget states
  if( b )
  {
    d.ui.family->setEnabled(false);
    d.ui.friends->setEnabled(false);
  }
  else
  {
    d.ui.family->setEnabled(true);
    d.ui.friends->setEnabled(true);
  }
}

void BatchWidget::on_family_clicked(bool b)
{
  // Store the new value
  d.setData("family", QVariant(b));
}

void BatchWidget::on_friends_clicked(bool b)
{
  // Store the new value
  d.setData("friends", QVariant(b));
}

void BatchWidget::on_licenses_activated(int index)
{
  Q_UNUSED(index);

  // Store the new value
  d.setData("license", QVariant(d.ui.licenses->currentText()));
}

void BatchWidget::on_photosets_activated(int index)
{
  Q_UNUSED(index);

  // Store the new value
  d.setData("photoset", QVariant(d.ui.photosets->currentText()));
}

// We use currentIndexChanged() signal since we also want to handle programatic changes
// to the size type selection. We take this time to set the size spinbox values as well
// and ensure they are enabled or disabled accordingly.
void BatchWidget::on_sizes_currentIndexChanged(const QString& text)
{
  // Update the width and height spinboxes
  d.ui.width->setEnabled(false);
  d.ui.height->setEnabled(false);

  if( text != i18n("Custom") )
  {
    d.ui.width->setValue(74);
    d.ui.height->setValue(74);
  }
  else
  {
    // Use the current width and height values as the starting place.
    d.ui.width->setValue(d.model->data(d.getIndex("width")).toInt());
    d.ui.height->setValue(d.model->data(d.getIndex("height")).toInt());
    d.ui.width->setEnabled(true);
    d.ui.height->setEnabled(true);
  }

  // Store the new size type
  d.setData("size", QVariant(text));
}

// Deal with any changes to the width spinbox value.
void BatchWidget::on_width_valueChanged(int value)
{
  // The only time the width and height values are actually used is when the Custom size
  // is selected. We use this to our advantage as it makes the size spinboxes more intuitive.
  if( d.ui.sizes->currentText() == i18n("Custom") )
  { 
    d.setData("width", QVariant(value));
  }
}

// Deal with any changes to the height spinbox value.
void BatchWidget::on_height_valueChanged(int value)
{
  // The only time the width and height values are actually used is when the Custom size
  // is selected. We use this to our advantage as it makes the size spinboxes more intuitive.
  if( d.ui.sizes->currentText() == i18n("Custom") )
  {
    d.setData("height", QVariant(value));
  }
}

void BatchWidget::on_add_clicked(bool b)
{
  Q_UNUSED(b);

  QString str = d.ui.availableTags->currentText();
  QStringList list = d.tags.stringList().filter(QRegExp(".+"));

  if( str != QString::null && !list.contains(str) )
  {
    list << str;
    d.tags.setStringList(list);
    d.setData("tags", QVariant(list.join(",")));
  }
  d.ui.availableTags->clearEditText();
}

void BatchWidget::on_remove_clicked(bool b)
{
  Q_UNUSED(b);

  QModelIndexList list = d.ui.tags->selectionModel()->selection().indexes();
  qSort(list.begin(), list.end(), qGreater<QModelIndex>());

  foreach(QModelIndex index, list)
  {
    Q_ASSERT(index.row() >= 0 && index.row() < d.tags.rowCount());
    d.tags.removeRow(index.row());
  }

  QStringList tagList = d.tags.stringList().filter(QRegExp(".+"));
  d.setData("tags", QVariant(tagList.join(",")));
}

void BatchWidget::on_availableTags_editTextChanged(const QString& text)
{
  if( text.length() > 0 )
  {
    d.ui.add->setEnabled(true);
  }
  else
  {
    d.ui.add->setEnabled(false);
  }
}

void BatchWidget::on_availableTags_activated(const QString &text)
{
  QStringList list = d.tags.stringList().filter(QRegExp(".+"));

  if( text != QString::null && !list.contains(text) )
  {
    list << text;
    d.tags.setStringList(list);
    d.setData("tags", QVariant(list.join(",")));
  }
  d.ui.availableTags->clearEditText();
}

void BatchWidget::tagsSelectionChanged(const QItemSelection& selected,
					const QItemSelection& deselected)
{
  Q_UNUSED(selected);
  Q_UNUSED(deselected);

  if( d.ui.tags->selectionModel()->selection().size() > 0 )
    d.ui.remove->setEnabled(true);
  else
    d.ui.remove->setEnabled(false);

}
