/***************************************************************************
 *                                                                         *
 *   copyright (C) 2003, 2004, 2005  by Michael Buesch                     *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#include <klocale.h>
#include <kmenubar.h>
#include <klistview.h>
#include <ktoolbar.h>
#include <kfiledialog.h>
#include <kstatusbar.h>
#include <kiconloader.h>
#include <kmessagebox.h>
#include <kstatusbar.h>

#include <qpixmap.h>
#include <qcheckbox.h>
#include <qspinbox.h>
#include <qlineedit.h>
#include <qfileinfo.h>
#include <qclipboard.h>

#include <dcopclient.h>

#include <stdio.h>

#include "pwm.h"
#include "pwminit.h"
#include "configwndimpl.h"
#include "pwmprint.h"
#include "addentrywndimpl.h"
#include "globalstuff.h"
#include "findwndimpl.h"
#include "configuration.h"
#include "csv.h"

#ifdef CONFIG_KWALLETIF
# include "kwalletif.h"
# include "kwalletemu.h"
#endif
#ifdef CONFIG_PWMANAGER_SMARTCARD
# include "masterkey/smartkey/smartkey.h"
#endif


#define DEFAULT_SIZE		(QSize(700, 400))

// Button IDs for "file" popup menu
enum {
	BUTTON_POPUP_FILE_NEW = 0,
	BUTTON_POPUP_FILE_OPEN,
	BUTTON_POPUP_FILE_CLOSE,
	BUTTON_POPUP_FILE_SAVE,
	BUTTON_POPUP_FILE_SAVEAS,
	BUTTON_POPUP_FILE_EXPORT,
	BUTTON_POPUP_FILE_IMPORT,
	BUTTON_POPUP_FILE_PRINT,
	BUTTON_POPUP_FILE_QUIT
};
// Button IDs for "manage" popup menu
enum {
	BUTTON_POPUP_MANAGE_ADD = 0,
	BUTTON_POPUP_MANAGE_EDIT,
	BUTTON_POPUP_MANAGE_DEL,
	BUTTON_POPUP_MANAGE_CHANGEMP
};
// Button IDs for "smartcard" popup menu
enum {
	BUTTON_POPUP_SMARTCARD_GETID = 0,
	BUTTON_POPUP_SMARTCARD_GEN,
	BUTTON_POPUP_SMARTCARD_ERASE,
	BUTTON_POPUP_SMARTCARD_TOFILE,
	BUTTON_POPUP_SMARTCARD_FROMFILE
};
// Button IDs for "view" popup menu
enum {
	BUTTON_POPUP_VIEW_FIND = 0,
	BUTTON_POPUP_VIEW_LOCK,
	BUTTON_POPUP_VIEW_DEEPLOCK,
	BUTTON_POPUP_VIEW_UNLOCK
};
// Button IDs for "Tools" popup menu
enum {
	BUTTON_POPUP_TOOLS_PWGEN = 0
};
// Button IDs for "options" popup menu
enum {
	BUTTON_POPUP_OPTIONS_CONFIG = 0
};
// Button IDs for "export" popup menu (in "file" popup menu)
enum {
	BUTTON_POPUP_EXPORT_TEXT = 0,
	BUTTON_POPUP_EXPORT_GPASMAN,
	BUTTON_POPUP_EXPORT_CSV
#ifdef CONFIG_KWALLETIF
	,BUTTON_POPUP_EXPORT_KWALLET
#endif
};
// Button IDs for "import" popup menu (in "file" popup menu)
enum {
	BUTTON_POPUP_IMPORT_TEXT = 0,
	BUTTON_POPUP_IMPORT_GPASMAN,
	BUTTON_POPUP_IMPORT_CSV
#ifdef CONFIG_KWALLETIF
	,BUTTON_POPUP_IMPORT_KWALLET
#endif
};
// Button IDs for toolbar
enum {
	BUTTON_TOOL_NEW = 0,
	BUTTON_TOOL_OPEN,
	BUTTON_TOOL_SAVE,
	BUTTON_TOOL_SAVEAS,
	BUTTON_TOOL_PRINT,
	BUTTON_TOOL_ADD,
	BUTTON_TOOL_EDIT,
	BUTTON_TOOL_DEL,
	BUTTON_TOOL_FIND,
	BUTTON_TOOL_LOCK,
	BUTTON_TOOL_DEEPLOCK,
	BUTTON_TOOL_UNLOCK
};


PwM::PwM(PwMInit *_init, PwMDoc *doc,
	 bool virginity,
	 QWidget *parent, const char *name)
 : KMainWindow(parent, name)
 , view (0)
 , virgin (true)
 , forceQuit (false)
 , forceMinimizeToTray (false)
{
	init = _init;
	connect(doc, SIGNAL(docClosed(PwMDoc *)),
		this, SLOT(docClosed(PwMDoc *)));
	initMenubar();
	initToolbar();
	initMetrics();
	setFocusPolicy(QWidget::WheelFocus);
	statusBar()->show();
	view = makeNewListView(doc);
	setCentralWidget(view);
	setVirgin(virginity);
	showStatMsg(i18n("Ready."));
}

PwM::~PwM()
{
	disconnect(curDoc(), SIGNAL(docClosed(PwMDoc *)),
		   this, SLOT(docClosed(PwMDoc *)));
	conf()->confWndMainWndSize(size());
	emit closed(this);
	delete view;
}

void PwM::initMenubar()
{
	KIconLoader icons;

	filePopup = new KPopupMenu(this);
	importPopup = new KPopupMenu(filePopup);
	exportPopup = new KPopupMenu(filePopup);
	managePopup = new KPopupMenu(this);
#ifdef CONFIG_PWMANAGER_SMARTCARD
	smartcardPopup = new KPopupMenu(this);
#endif
	viewPopup = new KPopupMenu(this);
	toolsPopup = new KPopupMenu(this);
	optionsPopup = new KPopupMenu(this);

// "file" popup menu
	filePopup->insertItem(QIconSet(icons.loadIcon("filenew", KIcon::Small)),
			      i18n("&New Window"), this,
			      SLOT(new_slot()), 0, BUTTON_POPUP_FILE_NEW);
	filePopup->insertItem(QIconSet(icons.loadIcon("fileopen", KIcon::Small)),
			      i18n("&Open..."), this,
			      SLOT(open_slot()), 0, BUTTON_POPUP_FILE_OPEN);
	filePopup->insertItem(QIconSet(icons.loadIcon("fileclose", KIcon::Small)),
			      i18n("&Close"), this,
			      SLOT(close_slot()), 0, BUTTON_POPUP_FILE_CLOSE);
	filePopup->insertSeparator();
	filePopup->insertItem(QIconSet(icons.loadIcon("filesave", KIcon::Small)),
			      i18n("&Save"), this,
			      SLOT(save_slot()), 0, BUTTON_POPUP_FILE_SAVE);
	filePopup->insertItem(QIconSet(icons.loadIcon("filesaveas", KIcon::Small)),
			      i18n("Save &As..."),
			      this, SLOT(saveAs_slot()), 0,
			      BUTTON_POPUP_FILE_SAVEAS);
	filePopup->insertSeparator();
	// "file/export" popup menu
	exportPopup->insertItem(i18n("&Text file..."), this,
				SLOT(exportToText()), 0, BUTTON_POPUP_EXPORT_TEXT);
	exportPopup->insertItem(i18n("&GPasman / KPasman..."), this,
				SLOT(exportToGpasman()), 0, BUTTON_POPUP_EXPORT_GPASMAN);
	exportPopup->insertItem(i18n("&CSV (Comma Separated Value)..."), this,
				SLOT(exportToCsv()), 0, BUTTON_POPUP_EXPORT_CSV);
#ifdef CONFIG_KWALLETIF
	exportPopup->insertItem(i18n("&KWallet..."), this,
				SLOT(exportToKWallet()), 0, BUTTON_POPUP_EXPORT_KWALLET);
#endif
	filePopup->insertItem(QIconSet(icons.loadIcon("fileexport", KIcon::Small)),
			      i18n("E&xport"), exportPopup,
			      BUTTON_POPUP_FILE_EXPORT);
	// "file/import" popup menu
	importPopup->insertItem(i18n("&Text file..."), this,
				SLOT(importFromText()), 0, BUTTON_POPUP_IMPORT_TEXT);
	importPopup->insertItem(i18n("&GPasman / KPasman..."), this,
				SLOT(importFromGpasman()), 0, BUTTON_POPUP_IMPORT_GPASMAN);
	importPopup->insertItem(i18n("&CSV (Comma Separated Value)..."), this,
				SLOT(importCsv()), 0, BUTTON_POPUP_IMPORT_CSV);
#ifdef CONFIG_KWALLETIF
	importPopup->insertItem(i18n("&KWallet..."), this,
				SLOT(importKWallet()), 0, BUTTON_POPUP_IMPORT_KWALLET);
#endif
	filePopup->insertItem(QIconSet(icons.loadIcon("fileimport", KIcon::Small)),
			      i18n("I&mport"), importPopup,
			      BUTTON_POPUP_FILE_IMPORT);
	filePopup->insertSeparator();
	filePopup->insertItem(QIconSet(icons.loadIcon("fileprint", KIcon::Small)),
			      i18n("&Print..."), this,
			      SLOT(print_slot()), 0, BUTTON_POPUP_FILE_PRINT);
	filePopup->insertSeparator();
	filePopup->insertItem(QIconSet(icons.loadIcon("exit", KIcon::Small)),
			      i18n("&Quit"), this,
			      SLOT(quitButton_slot()), 0, BUTTON_POPUP_FILE_QUIT);
	menuBar()->insertItem(i18n("&File"), filePopup);
// "manage" popup menu
	managePopup->insertItem(QIconSet(icons.loadIcon("pencil", KIcon::Small)),
				i18n("&Add Entry..."), this,
				SLOT(addPwd_slot()), 0,
				BUTTON_POPUP_MANAGE_ADD);
	managePopup->insertItem(QIconSet(icons.loadIcon("edit", KIcon::Small)),
				i18n("&Edit Selected Entry..."), this, SLOT(editPwd_slot()), 0,
				BUTTON_POPUP_MANAGE_EDIT);
	managePopup->insertItem(QIconSet(icons.loadIcon("editdelete", KIcon::Small)),
				i18n("&Delete Selected Entry"), this, SLOT(deletePwd_slot()),
				0, BUTTON_POPUP_MANAGE_DEL);
	managePopup->insertSeparator();
	managePopup->insertItem(QIconSet(icons.loadIcon("rotate", KIcon::Small)),
				i18n("Change &Master Key..."), this,
				SLOT(changeMasterPwd_slot()), 0,
				BUTTON_POPUP_MANAGE_CHANGEMP);
	menuBar()->insertItem(i18n("&Edit"), managePopup);
#ifdef CONFIG_PWMANAGER_SMARTCARD
// "smartcard" popup menu
	smartcardPopup->insertItem(QIconSet(icons.loadIcon("identity", KIcon::Small)),
				   i18n("Read SmartKey &ID..."), this, SLOT(readSmartKeyId_slot()),
				   0, BUTTON_POPUP_SMARTCARD_GETID);
	smartcardPopup->insertSeparator();
	smartcardPopup->insertItem(QIconSet(icons.loadIcon("gear", KIcon::Small)),
				   i18n("&Generate New SmartKey..."), this, SLOT(genSmartKey_slot()),
				   0, BUTTON_POPUP_SMARTCARD_GEN);
	smartcardPopup->insertItem(QIconSet(icons.loadIcon("eraser", KIcon::Small)),
				   i18n("&Erase SmartCard..."), this, SLOT(eraseSmartCard_slot()),
				   0, BUTTON_POPUP_SMARTCARD_ERASE);
	smartcardPopup->insertSeparator();
	smartcardPopup->insertItem(QIconSet(icons.loadIcon("fileexport", KIcon::Small)),
				   i18n("E&xport SmartKey To KeyFile..."), this, SLOT(exportSmartKey_slot()),
				   0, BUTTON_POPUP_SMARTCARD_TOFILE);
	smartcardPopup->insertItem(QIconSet(icons.loadIcon("fileimport", KIcon::Small)),
				   i18n("I&mport SmartKey From KeyFile..."), this, SLOT(importSmartKey_slot()),
				   0, BUTTON_POPUP_SMARTCARD_FROMFILE);
	menuBar()->insertItem(i18n("SmartCard"), smartcardPopup);
#endif // CONFIG_PWMANAGER_SMARTCARD
// "view" popup menu
	viewPopup->insertItem(QIconSet(icons.loadIcon("find", KIcon::Small)),
				i18n("&Find..."), this,
				SLOT(find_slot()), 0, BUTTON_POPUP_VIEW_FIND);
	viewPopup->insertSeparator();
	viewPopup->insertItem(QIconSet(icons.loadIcon("halfencrypted", KIcon::Small)),
				i18n("&Lock All Entries"), this,
				SLOT(lockWnd_slot()), 0,
				BUTTON_POPUP_VIEW_LOCK);
	viewPopup->insertItem(QIconSet(icons.loadIcon("encrypted", KIcon::Small)),
				i18n("&Deep Lock All Entries"), this,
				SLOT(deepLockWnd_slot()), 0,
				BUTTON_POPUP_VIEW_DEEPLOCK);
	viewPopup->insertItem(QIconSet(icons.loadIcon("decrypted", KIcon::Small)),
				i18n("&Unlock All Entries"), this,
				SLOT(unlockWnd_slot()), 0,
				BUTTON_POPUP_VIEW_UNLOCK);
	menuBar()->insertItem(i18n("&View"), viewPopup);
// "Tools" popup menu
	toolsPopup->insertItem(QIconSet(icons.loadIcon("math_sqrt", KIcon::Small)),
			       i18n("&Generate Password..."), this,
			       SLOT(genPassword_slot()), 0, BUTTON_POPUP_TOOLS_PWGEN);
	menuBar()->insertItem(i18n("&Tools"), toolsPopup);
// "options" popup menu
	optionsPopup->insertItem(QIconSet(icons.loadIcon("configure", KIcon::Small)),
				 i18n("&Configure..."), this,
				 SLOT(config_slot()),
				 BUTTON_POPUP_OPTIONS_CONFIG);
	menuBar()->insertItem(i18n("&Options"), optionsPopup);
// "help" popup menu
	helpPopup = helpMenu(QString(), false);
	menuBar()->insertItem(i18n("&Help"), helpPopup);
}

void PwM::initToolbar()
{
	KIconLoader icons;

	toolBar()->insertButton(icons.loadIcon("filenew", KIcon::Toolbar),
				BUTTON_TOOL_NEW, SIGNAL(clicked(int)), this,
				SLOT(new_slot()), true, i18n("New Window"));
	toolBar()->insertButton(icons.loadIcon("fileopen", KIcon::Toolbar),
				BUTTON_TOOL_OPEN, SIGNAL(clicked(int)), this,
				SLOT(open_slot()), true, i18n("Open"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(icons.loadIcon("filesave", KIcon::Toolbar),
				BUTTON_TOOL_SAVE, SIGNAL(clicked(int)), this,
				SLOT(save_slot()), true, i18n("Save"));
	toolBar()->insertButton(icons.loadIcon("filesaveas", KIcon::Toolbar),
				BUTTON_TOOL_SAVEAS, SIGNAL(clicked(int)), this,
				SLOT(saveAs_slot()), true, i18n("Save As"));
	toolBar()->insertButton(icons.loadIcon("fileprint", KIcon::Toolbar),
				BUTTON_TOOL_PRINT, SIGNAL(clicked(int)), this,
				SLOT(print_slot()), true, i18n("Print"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(icons.loadIcon("pencil", KIcon::Toolbar),
				BUTTON_TOOL_ADD, SIGNAL(clicked(int)), this,
				SLOT(addPwd_slot()), true,
				i18n("Add Entry"));
	toolBar()->insertButton(icons.loadIcon("edit", KIcon::Toolbar),
				BUTTON_TOOL_EDIT, SIGNAL(clicked(int)), this,
				SLOT(editPwd_slot()), true,
				i18n("Edit Selected Entry"));
	toolBar()->insertButton(icons.loadIcon("editdelete", KIcon::Toolbar),
				BUTTON_TOOL_DEL, SIGNAL(clicked(int)), this,
				SLOT(deletePwd_slot()), true,
				i18n("Delete Selected Entry"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(icons.loadIcon("find", KIcon::Toolbar),
				BUTTON_TOOL_FIND, SIGNAL(clicked(int)), this,
				SLOT(find_slot()), true, i18n("Find"));
	toolBar()->insertSeparator();
	toolBar()->insertButton(icons.loadIcon("halfencrypted", KIcon::Toolbar),
				BUTTON_TOOL_LOCK, SIGNAL(clicked(int)), this,
				SLOT(lockWnd_slot()), true,
				i18n("Lock all entries"));
	toolBar()->insertButton(icons.loadIcon("encrypted", KIcon::Toolbar),
				BUTTON_TOOL_DEEPLOCK, SIGNAL(clicked(int)), this,
				SLOT(deepLockWnd_slot()), true,
				i18n("Deep-Lock all entries"));
	toolBar()->insertButton(icons.loadIcon("decrypted", KIcon::Toolbar),
				BUTTON_TOOL_UNLOCK, SIGNAL(clicked(int)), this,
				SLOT(unlockWnd_slot()), true,
				i18n("Unlock all entries"));
}

void PwM::initMetrics()
{
	QSize s = conf()->confWndMainWndSize();
	if (s.isValid())
		resize(s);
	else
		resize(DEFAULT_SIZE);
}

void PwM::updateCaption()
{
	if (!curView() || !curDoc())
		return;
	setPlainCaption(curDoc()->getTitle()
			+ (curDoc()->isDirty() ? QString(" *") : QString())
			+ " - " PROG_NAME " " PACKAGE_VER
			  " '" PROG_NICKNAME "'");
}

void PwM::hideEvent(QHideEvent *)
{
	if (isMinimized()) {
		if (init->tray()) {
			forceMinimizeToTray = true;
			close();
		}
		int mmlock = conf()->confGlobMinimizeLock();
		switch (mmlock) {
		case 0:		// don't lock anything
			break;
		case 1: {	// normal lock
			curDoc()->lockAll(true);
			break;
		} case 2: {	// deep-lock
			curDoc()->deepLock();
			break;
		} default:
			WARN();
		}
	}
}

void PwM::setVirgin(bool v)
{
	updateCaption();
	if (virgin == v)
		return;
	virgin = v;
	filePopup->setItemEnabled(BUTTON_POPUP_FILE_SAVE, !v);
	filePopup->setItemEnabled(BUTTON_POPUP_FILE_SAVEAS, !v);
	filePopup->setItemEnabled(BUTTON_POPUP_FILE_EXPORT, !v);
	filePopup->setItemEnabled(BUTTON_POPUP_FILE_PRINT, !v);
	managePopup->setItemEnabled(BUTTON_POPUP_MANAGE_EDIT, !v);
	managePopup->setItemEnabled(BUTTON_POPUP_MANAGE_DEL, !v);
	managePopup->setItemEnabled(BUTTON_POPUP_MANAGE_CHANGEMP, !v);
	viewPopup->setItemEnabled(BUTTON_POPUP_VIEW_LOCK, !v);
	viewPopup->setItemEnabled(BUTTON_POPUP_VIEW_DEEPLOCK, !v);
	viewPopup->setItemEnabled(BUTTON_POPUP_VIEW_UNLOCK, !v);
	viewPopup->setItemEnabled(BUTTON_POPUP_VIEW_FIND, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_SAVE, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_SAVEAS, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_PRINT, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_EDIT, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_DEL, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_LOCK, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_DEEPLOCK, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_UNLOCK, !v);
	toolBar()->setItemEnabled(BUTTON_TOOL_FIND, !v);
}

void PwM::new_slot()
{
	init->createMainWnd();
}

void PwM::open_slot(QString fn)
{
	openDoc(fn);
}

PwMDoc * PwM::openDoc(QString filename, bool openDeepLocked)
{
	if (!isVirgin()) {
		// open the document in a new window.
		PwM *newInstance = init->createMainWnd();
		PwMDoc *newDoc = newInstance->openDoc(filename, openDeepLocked);
		if (!newDoc) {
			newInstance->setForceQuit(true);
			delete_and_null(newInstance);
		}
		return newDoc;
	}

	if (!curDoc()->openDocUi(curDoc(), filename, openDeepLocked))
		return 0;
	showStatMsg(i18n("Successfully opened file."));
	updateCaption();
	setVirgin(false);
	return curDoc();
}

PwMView * PwM::makeNewListView(PwMDoc *doc)
{
	PwMView *ret = new PwMView(this, this, doc);
	ret->setFont(conf()->confGlobEntryFont());
	ret->show();
	return ret;
}

void PwM::close_slot()
{
	close();
}

void PwM::quitButton_slot()
{
	init->shutdownApp(0);
}

void PwM::save_slot()
{
	save();
}

bool PwM::save()
{
	if (!curDoc()->saveDocUi(curDoc()))
		return false;
	showStatMsg(i18n("Successfully saved data."));
	updateCaption();
	return true;
}

void PwM::saveAs_slot()
{
	saveAs();
}

bool PwM::saveAs()
{
	if (!curDoc()->saveAsDocUi(curDoc()))
		return false;
	showStatMsg(i18n("Successfully saved data."));
	updateCaption();
	return true;
}

void PwM::addPwd_slot(QString *pw, PwMDoc *_doc)
{
	PwMDoc *doc;
	if (_doc) {
		doc = _doc;
	} else {
		doc = curDoc();
	}
	PWM_ASSERT(doc);
	doc->timer()->getLock(DocTimer::id_autoLockTimer);
	AddEntryWndImpl w;
	vector<string> catList;
	doc->getCategoryList(&catList);
	unsigned i, size = catList.size();
	for (i = 0; i < size; ++i) {
		w.addCategory(catList[i].c_str());
	}
	w.setCurrCategory(view->getCurrentCategory());
	if (pw)
		w.pwLineEdit->setText(*pw);

      tryAgain:
	if (w.exec() == 1) {
		PwMDataItem d;
		d.desc = w.getDescription().latin1();
		d.name = w.getUsername().latin1();
		d.pw = w.getPassword().latin1();
		d.comment = w.getComment().latin1();
		d.url = w.getUrl().latin1();
		d.launcher = w.getLauncher().latin1();
		PwMerror ret = doc->addEntry(w.getCategory(), &d);
		if (ret == e_entryExists) {
			KMessageBox::error(this,
					   i18n
					   ("An entry with this \"Description\", "
					    "already exists;\n"
					    "please select another description."),
					   i18n("Entry Already Exists"));
			goto tryAgain;
		} else if (ret == e_maxAllowedEntr) {
			KMessageBox::error(this, i18n("The maximum possible number of entries "
						      "has been reached: you cannot add more entries."),
						 i18n("Maximum Number of Entries"));
			doc->timer()->putLock(DocTimer::id_autoLockTimer);
			return;
		}
	}
	setVirgin(false);
	doc->timer()->putLock(DocTimer::id_autoLockTimer);
}

void PwM::editPwd_slot(const QString *category, const int *index,
		       PwMDoc *_doc)
{
	PwMDoc *doc;
	if (_doc) {
		doc = _doc;
	} else {
		doc = curDoc();
	}
	PWM_ASSERT(doc);
	if (doc->isDocEmpty())
		return;
	if (doc->isDeepLocked())
		return;
	doc->timer()->getLock(DocTimer::id_autoLockTimer);
	unsigned int curEntryIndex;
	if (index) {
		curEntryIndex = *index;
	} else {
		if (!(view->getCurEntryIndex(&curEntryIndex))) {
			printDebug("couldn't get index. Maybe we have a binary entry here.");
			doc->timer()->putLock(DocTimer::id_autoLockTimer);
			return;
		}
	}
	QString curCategory;
	if (category) {
		curCategory = *category;
	} else {
		curCategory = view->getCurrentCategory();
	}
	PwMDataItem currItem;
	if (!doc->getEntry(curCategory, curEntryIndex, &currItem, true)) {
		doc->timer()->putLock(DocTimer::id_autoLockTimer);
		return;
	}
	BUG_ON(currItem.binary);

	AddEntryWndImpl w;
	vector<string> catList;
	doc->getCategoryList(&catList);
	unsigned i, size = catList.size();
	for (i = 0; i < size; ++i) {
		w.addCategory(catList[i].c_str());
	}
	w.setCurrCategory(curCategory);
	w.setDescription(currItem.desc.c_str());
	w.setUsername(currItem.name.c_str());
	w.setPassword(currItem.pw.c_str());
	w.setUrl(currItem.url.c_str());
	w.setLauncher(currItem.launcher.c_str());
	w.setComment(currItem.comment.c_str());
	if (w.exec() == 1) {
		currItem.desc = w.getDescription().latin1();
		currItem.name = w.getUsername().latin1();
		currItem.pw = w.getPassword().latin1();
		currItem.comment = w.getComment().latin1();
		currItem.url = w.getUrl().latin1();
		currItem.launcher = w.getLauncher().latin1();
		if (!doc->editEntry(curCategory, w.getCategory(),
					 curEntryIndex, &currItem)) {
			KMessageBox::error(this,
					   i18n("Could not edit the entry;\n"
						"maybe you changed the category and "
						"this entry is already present in the new "
						"category?"),
					   i18n("Could Not Edit Entry"));
			doc->timer()->putLock(DocTimer::id_autoLockTimer);
			return;
		}
	}
	doc->timer()->putLock(DocTimer::id_autoLockTimer);
}

void PwM::deletePwd_slot()
{
	PWM_ASSERT(curDoc());
	if (curDoc()->isDocEmpty())
		return;
	if (curDoc()->isDeepLocked())
		return;
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	unsigned int curEntryIndex = 0;
	if (!(view->getCurEntryIndex(&curEntryIndex))) {
		printDebug("couldn't get index");
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		return;
	}

	PwMDataItem currItem;
	QString curCategory = view->getCurrentCategory();
	if (!curDoc()->getEntry(curCategory, curEntryIndex, &currItem)) {
		printDebug("couldn't get entry");
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		return;
	}
	if (KMessageBox::
	    questionYesNo(this,
			  i18n
			  ("Do you really want to delete the selected entry \"%1\"?").arg(QString(currItem.desc.c_str())), i18n("Delete?"))
	    == KMessageBox::Yes) {

		curDoc()->delEntry(curCategory, curEntryIndex);
	}
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
}

void PwM::changeMasterPwd_slot()
{
	PWM_ASSERT(curDoc());
	curDoc()->changeMasterKey();
}

void PwM::lockWnd_slot()
{
	PWM_ASSERT(curDoc());
	curDoc()->lockAll(true);
}

void PwM::deepLockWnd_slot()
{
	PWM_ASSERT(curDoc());
	curDoc()->deepLock();
}

void PwM::unlockWnd_slot()
{
	PWM_ASSERT(curDoc());
	curDoc()->lockAll(false);
}

void PwM::config_slot()
{
	Configuration *conf = Configuration::obj();
	int oldStyle = conf->confWndMainViewStyle();

	// display the configuration window (modal mode)
	if (!conf->showConfWnd(this))
		return;

	int newStyle = conf->confWndMainViewStyle();
	// reinitialize tray
	init->initTray();
	// reinitialize KWallet emulation
	init->initKWalletEmu();

	PwMDocList *_dl = PwMDoc::getOpenDocList();
	const vector<PwMDocList::listItem> *dl = _dl->getList();
	vector<PwMDocList::listItem>::const_iterator i = dl->begin(),
						     end = dl->end();
	PwMDoc *doc;
	while (i != end) {
		doc = (*i).doc;
		// unlock-without-mpw timeout
		doc->timer()->start(DocTimer::id_mpwTimer);
		// auto-lock timeout
		doc->timer()->start(DocTimer::id_autoLockTimer);
		++i;
	}

	const QValueList<PwM *> *ml = init->mainWndList();
	QValueList<PwM *>::const_iterator i2 = ml->begin(),
					  end2 = ml->end();
	PwM *pwm;
	while (i2 != end2) {
		pwm = *i2;
		// reinitialize the window style.
		if (oldStyle != newStyle)
			pwm->curView()->initStyle(newStyle);
		// set the new font
		pwm->curView()->setFont(conf->confGlobEntryFont());
		++i2;
	}
}

void PwM::activateMpButton(bool activate)
{
	managePopup->setItemEnabled(BUTTON_POPUP_MANAGE_CHANGEMP, activate);
}

void PwM::closeEvent(QCloseEvent *e)
{
	e->accept();
}

void PwM::docClosed(PwMDoc *doc)
{
	PARAM_UNUSED(doc);
	PWM_ASSERT(doc == curDoc());
	close();
}

void PwM::find_slot()
{
	PWM_ASSERT(curDoc());
	if (curDoc()->isDocEmpty())
		return;
	if (curDoc()->isDeepLocked())
		return;
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	FindWndImpl findWnd(view);
	findWnd.exec();
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
}

void PwM::exportToText()
{
	PWM_ASSERT(curDoc());
	if (curDoc()->isDocEmpty()) {
		KMessageBox::information(this,
					 i18n
					 ("Sorry, there is nothing to export;\n"
					  "please add some passwords first."),
					 i18n("Nothing to Do"));
		return;
	}
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	QString fn(KFileDialog::getSaveFileName(QString(),
						i18n("*|Plain Text File"),
						this));
	if (fn == "") {
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		return;
	}

	PwMerror ret = curDoc()->exportToText(&fn);
	if (ret != e_success) {
		KMessageBox::error(this,
				   i18n("Error: unable to write to file.\n"
					"Please check if you have permission to write "
					"to the file in that directory."),
				   i18n("Error While Writing"));
	} else
		showStatMsg(i18n("Successfully exported data."));
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
}

bool PwM::importFromText()
{
	if (!isVirgin()) {
		if (KMessageBox::questionYesNo(this,
					       i18n("Do you want to import the data "
					            "into the current document? (If you "
					            "select \"no\", a new document will be "
					            "opened.)"),
					       i18n("Import into This Document?"))
		    == KMessageBox::No) {
			// import the data to a new window.
			PwM *newInstance = init->createMainWnd();
			bool ok = newInstance->importFromText();
			if (!ok) {
				newInstance->setForceQuit(true);
				delete_and_null(newInstance);
			}
			return ok;
		}
	}

	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	PwMerror ret;
	QString path(KFileDialog::getOpenFileName(QString(),
						  i18n("*|PwManager Exported Text File"),
						  this));
	if (path == "")
		goto cancelImport;

	ret = curDoc()->importFromText(&path, 0);
	if (ret == e_fileFormat) {
		KMessageBox::error(this,
				   i18n("Could not read file format.\n"
					"This seems to be _not_ a valid file "
					"exported by PwM."),
				   i18n("Invalid File Format"));
		goto cancelImport;
	} else if (ret == e_invalidArg) {
		BUG();
		goto cancelImport;
	} else if (ret != e_success) {
		KMessageBox::error(this,
				   i18n("Could not import file.\n"
					"Do you have permission to read this file? "
					"Do you have enough free memory?"),
				   i18n("Import Failed"));
		goto cancelImport;
	}
	setVirgin(false);
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
	return true;

cancelImport:
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
	return false;
}

void PwM::exportToGpasman()
{
	PWM_ASSERT(curDoc());
	if (curDoc()->isDocEmpty()) {
		KMessageBox::information(this,
					 i18n
					 ("Sorry, there is nothing to export;\n"
					  "please add some passwords first."),
					 i18n("Nothing to Do"));
		return;
	}
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	QString fn(KFileDialog::getSaveFileName(QString(),
						i18n("*|GPasman or KPasman File"),
						this));
	if (fn == "") {
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		return;
	}

	PwMerror ret = curDoc()->exportToGpasman(&fn);
	if (ret != e_success) {
		if (ret == e_noPw) {
			curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
			return;
		}
		KMessageBox::error(this,
				   i18n("Error: unable to write to file.\n"
					"Please check if you have permission to write "
					"to the file in that directory."),
				   i18n("Error While Writing"));
	} else
		showStatMsg(i18n("Successfully exported data."));
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
}

void PwM::exportToKWallet()
{
#ifdef CONFIG_KWALLETIF
	if (!checkAndAskForKWalletEmu())
		return;
	PWM_ASSERT(curDoc());
	if (curDoc()->isDocEmpty()) {
		KMessageBox::information(this,
					 i18n
					 ("Sorry, there is nothing to export;\n"
					  "please add some passwords first."),
					 i18n("Nothing to Do"));
		init->initKWalletEmu();
		return;
	}
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	KWalletIf walletIf(this);
	if (walletIf.kwalletExport(curDoc())) {
		KMessageBox::information(this,
					 i18n("Successfully exported the data of the current "
					      "document to KWallet."),
					 i18n("Successfully Exported Data"));
		showStatMsg(i18n("Successfully exported data."));
	}
	init->initKWalletEmu();
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
#endif // CONFIG_KWALLETIF
}

bool PwM::importFromGpasman()
{
	if (!isVirgin()) {
		if (KMessageBox::questionYesNo(this,
					       i18n("Do you want to import the data "
					            "into the current document? (If you "
					            "select \"no\", a new document will be "
					            "opened.)"),
					       i18n("Import into This Document?"))
		    == KMessageBox::No) {
			// import the data to a new window.
			PwM *newInstance = init->createMainWnd();
			bool ok = newInstance->importFromGpasman();
			if (!ok) {
				newInstance->setForceQuit(true);
				delete_and_null(newInstance);
			}
			return ok;
		}
	}

	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	PwMerror ret;
	QString path(KFileDialog::getOpenFileName(QString(),
						  i18n("*|GPasman or KPasman File"), this));
	if (path == "")
		goto cancelImport;
	ret = curDoc()->importFromGpasman(&path);
	if (ret == e_wrongPw) {
		if (KMessageBox::questionYesNo(this,
				 	       i18n
					       ("You have probably typed in the wrong master password.\n"
					        "There is no way of accurately determining the "
					        "correctness of the password in the Gpasman "
					        "file format, but it is likely "
					        "that it is wrong.\n"
					        "Do you want to continue regardless?"),
					       i18n("Password Error"))
		    == KMessageBox::No) {
			goto cancelImport;
		}
	} else if (ret != e_success) {
		KMessageBox::error(this,
				   i18n("Could not import file;\n"
					"do you have permission to read it?"),
				   i18n("Import Failed"));
		goto cancelImport;
	}
	setVirgin(false);
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
	return true;

cancelImport:
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
	return false;
}

#ifdef CONFIG_KWALLETIF
bool PwM::checkAndAskForKWalletEmu()
{
	if (init->kwalletEmu()) {
		/* KWallet emulation is enabled. We can't import/export
		 * data from/to it, while emulation is active.
		 */
		if (KMessageBox::questionYesNo(this,
					       i18n("KWallet emulation is enabled.\n"
						    "You cannot import or export data from/to "
						    "the original KWallet, while the emulation "
						    "is active.\n"
						    "Do you want to temporary disable the KWallet emulation?"),
					       i18n("Temporarily Disable KWallet Emulation?"))
		    == KMessageBox::Yes) {
			init->initKWalletEmu(true);
			PWM_ASSERT(!init->kwalletEmu());
			return true;
		}
		return false;
	}
	return true;
}
#endif // CONFIG_KWALLETIF

bool PwM::importKWallet()
{
#ifdef CONFIG_KWALLETIF
	if (!checkAndAskForKWalletEmu())
		return false;
	KWalletIf walletIf(this);
	if (!isVirgin()) {
		if (KMessageBox::questionYesNo(this,
					       i18n("Do you want to import the data "
					            "into the current document? (If you "
					            "select \"no\", a new document will be "
					            "opened.)"),
					       i18n("Import into This Document?"))
		    == KMessageBox::No) {
			// import the data to a new window.
			PwM *newInstance = init->createMainWnd();
			bool ok = newInstance->importKWallet();
			if (!ok) {
				newInstance->setForceQuit(true);
				delete_and_null(newInstance);
				goto exit_fail;
			} else {
				goto exit_ok;
			}
		}
	}
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	if (!walletIf.kwalletImport(curDoc())) {
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		showStatMsg(i18n("KWallet import failed"));
		goto exit_fail;
	}
	KMessageBox::information(this,
				 i18n("Successfully imported the KWallet data "
				      "into the current document."),
				 i18n("Successfully Imported"));
	showStatMsg(i18n("Successfully imported"));
	setVirgin(false);
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);

exit_ok:
	init->initKWalletEmu();
	return true;

exit_fail:
	init->initKWalletEmu();
#endif // CONFIG_KWALLETIF
	return false;
}

void PwM::print_slot()
{
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	PwMPrint p(curDoc(), this);
	p.printNow();
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
}

void PwM::execLauncher_slot()
{
	PWM_ASSERT(curDoc());
	if (curDoc()->isDeepLocked())
		return;
	unsigned int curEntryIndex;
	if (!view->getCurEntryIndex(&curEntryIndex))
		return;
	bool ret = curDoc()->execLauncher(view->getCurrentCategory(),
					  curEntryIndex);
	if (ret)
		showStatMsg(i18n("Executed the \"Launcher\"."));
	else
		showStatMsg(i18n("ERROR: Could not execute the \"Launcher\"."));
}

void PwM::goToURL_slot()
{
	PWM_ASSERT(curDoc());
	if (curDoc()->isDeepLocked())
		return;
	unsigned int curEntryIndex;
	if (!view->getCurEntryIndex(&curEntryIndex))
		return;
	bool ret = curDoc()->goToURL(view->getCurrentCategory(),
				     curEntryIndex);
	if (ret)
		showStatMsg(i18n("Started browser with current URL."));
	else
		showStatMsg(i18n("ERROR: Could not start browser; maybe the URL is invalid?"));
}

void PwM::copyToClipboard(const QString &s)
{
	PwMInit::copyToClipboard(s);
}

void PwM::showStatMsg(const QString &msg)
{
	KStatusBar *statBar = statusBar();
	statBar->message(msg, STATUSBAR_MSG_TIMEOUT * 1000);
}

void PwM::focusInEvent(QFocusEvent *e)
{
	if (e->gotFocus()) {
		emit gotFocus(this);
	} else if (e->lostFocus()) {
		emit lostFocus(this);
	}
}

void PwM::readSmartKeyId_slot()
{
#ifdef CONFIG_PWMANAGER_SMARTCARD
	SmartKey sk(this, this);
	sk.displayKeyId();
#endif
}

void PwM::genSmartKey_slot()
{
#ifdef CONFIG_PWMANAGER_SMARTCARD
	SmartKey sk(this, this);
	sk.genKey();
#endif
}

void PwM::eraseSmartCard_slot()
{
#ifdef CONFIG_PWMANAGER_SMARTCARD
	SmartKey sk(this, this);
	sk.eraseCard();
#endif
}

void PwM::exportSmartKey_slot()
{
#ifdef CONFIG_PWMANAGER_SMARTCARD
	SmartKey sk(this, this);
	sk.exportToKeyFile();
#endif
}

void PwM::importSmartKey_slot()
{
#ifdef CONFIG_PWMANAGER_SMARTCARD
	SmartKey sk(this, this);
	sk.importFromKeyFile();
#endif
}

void PwM::genPassword_slot()
{
	PwMInit::genPassword(this);
}

void PwM::exportToCsv()
{
	PWM_ASSERT(curDoc());
	if (curDoc()->isDocEmpty()) {
		KMessageBox::information(this,
					 i18n
					 ("Sorry, there is nothing to export;\n"
					  "please add some passwords first."),
					 i18n("Nothing to Do"));
		return;
	}
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	QString fn(KFileDialog::getSaveFileName(QString(),
						i18n("*|CSV Text File"),
						this));
	if (fn.isEmpty()) {
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		return;
	}

	Csv csv(this);
	if (!csv.exportData(fn, curDoc())) {
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		showStatMsg(i18n("CSV file export failed."));
		return;
	}
	showStatMsg(i18n("Successfully exported data."));
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
}

bool PwM::importCsv()
{
	Csv csv(this);
	if (!isVirgin()) {
		if (KMessageBox::questionYesNo(this,
					       i18n("Do you want to import the data "
					            "into the current document? (If you "
					            "select \"no\", a new document will be "
					            "opened.)"),
					       i18n("Import into This Document?"))
		    == KMessageBox::No) {
			// import the data to a new window.
			PwM *newInstance = init->createMainWnd();
			bool ok = newInstance->importCsv();
			if (!ok) {
				newInstance->setForceQuit(true);
				delete_and_null(newInstance);
			}
			return ok;
		}
	}
	QString filename = KFileDialog::getOpenFileName(QString(),
							i18n("*|CSV Text File"),
							this);
	if (filename.isEmpty())
		return false;
	curDoc()->timer()->getLock(DocTimer::id_autoLockTimer);
	if (!csv.importData(filename, curDoc())) {
		curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
		showStatMsg(i18n("CSV file import failed."));
		return false;
	}
	curDoc()->timer()->putLock(DocTimer::id_autoLockTimer);
	KMessageBox::information(this,
				 i18n("Successfully imported the CSV data "
				      "into the current document."),
				 i18n("Successfully Imported"));
	showStatMsg(i18n("Successfully imported"));
	setVirgin(false);
	return true;
}

#include "pwm.moc"
