/***************************************************************************
 *                                                                         *
 *   copyright (C) 2004, 2005  by Michael Buesch                           *
 *   email: mbuesch@freenet.de                                             *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License version 2        *
 *   as published by the Free Software Foundation.                         *
 *                                                                         *
 ***************************************************************************/

#ifndef PWMANAGER_MASTERKEY_KEYDATA_H_
#define PWMANAGER_MASTERKEY_KEYDATA_H_

#include "globalstuff.h"

#include <qcstring.h>

#include <stdint.h>


#define PWMCONF_KEYCARD_COMPAT 1 // Do NOT change this. We want compatibility!


/** This class implements the raw key-data-stream.
  * It's used as smartcard-masterkey of keyfile-masterkey.
  *
  * The data stream is as follows:
  *
  * Header:
  *   "PWMANAGERKEY"		(12 byte magic string)
  *   VERSION-NUMBER-MAJOR	(char)
  *   VERSION-NUMBER-MINOR	(char)
  *   KEY-TYPE			(char)
  *   KEY-ID			(char[16]  128-bit ID)
  *   KEY-LENGTH		(uint16_t BIG-ENDIAN)
  *   RESERVED			(char[32])
  * Databody:
  *   KEY-DATA			(size depends on KEY-LENGTH)
  *
  * The header-size is 65 bytes.
  */
class KeyData
{
public:
	enum Type
	{
		/** Undefined type. */
		type_undef	= 0x00,
		/** Key stored on a SmartCard. */
		type_smartcard	= 0x01,
		/** Key stored in a KeyFile. */
		type_keyfile	= 0x02
	};

public:
	KeyData();
	~KeyData();

	/** Returns the current type. */
	Type getType() const;
	/** Returns the key ID. */
	void getId(QByteArray *k) const;
	/** Returns the key data. */
	void getKey(QByteArray *k) const;
	/** Generates a new key and a new key-id.
	  * If len == 0, a key with the default length is generated.
	  */
	void generate(Type t, uint16_t len = 0);
	/** Returns the raw key stream. */
	bool getStream(QByteArray *s) const;
	/** Set the raw key stream. */
	bool setStream(const QByteArray &s);

protected:
#if PWMCONF_KEYCARD_COMPAT != 0
	/** compatibility function for PwManager-1.0 keycards. */
	bool setStream_10compat(const QByteArray &s);
	/** key truncation bug compatibility function. */
	void applyTruncationBug(QByteArray *s);
# ifdef BIG_ENDIAN_HOST
	uint32_t toBigEndian(uint32_t i) const
			{ return i; }
# else // BIG_ENDIAN_HOST
	uint32_t toBigEndian(uint32_t i) const;
# endif // BIG_ENDIAN_HOST
#else // PWMCONF_KEYCARD_COMPAT
	bool setStream_10compat(const QByteArray &)
			{ return false; }
#endif // PWMCONF_KEYCARD_COMPAT
	unsigned int estimateStreamSize() const;
	void writeToStream(QByteArray *s, unsigned int *offset,
			   const char *d, unsigned int size) const;
	bool readFromStream(const QByteArray &s, unsigned int *offset,
			    QByteArray *d, unsigned int size);
#ifdef BIG_ENDIAN_HOST
	uint16_t toBigEndian(uint16_t i) const
			{ return i; }
#else
	uint16_t toBigEndian(uint16_t i) const;
#endif

protected:
	char			verMajor;
	char			verMinor;
	char			type;
	QByteArray		id;
	QByteArray		key;
};

#endif // PWMANAGER_MASTERKEY_KEYDATA_H_
