/***************************************************************************
 *   Copyright (C) 2006 by Rohan McGovern                                  *
 *   rohan.pm@gmail.com                                                    *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/
#include <kdebug.h>
#include <klocale.h>
#include <qdom.h>
#include <qlabel.h>
#include <qfile.h>
#include <qfileinfo.h>
#include <qtextedit.h>
#include <qvbox.h>

#include "dbus/qdbusmessage.h"
#include "dbus/qdbuserror.h"
#include "dbus/qdbusproxy.h"
#include "dbusservice.h"
#include "dbusobject.h"

class DBusService::Private {
public:
    /// DBus proxy to use
    QDBusProxy * proxy;

    /// Try to discover PID.  Returns -1 on failure.
    int discoverPID();

    /// Try to discover process name, given PID.
    QString discoverProcessName( int pid );

    /// Try to discover alias, if any.
    QString discoverAlias();

    /// PID of process owning this service
    int pid;

    /// Name of process owning this service
    QString processName;

    /// If this service is named, unnamed alias
    QString alias;

    DBusService * p;
};

DBusService::DBusService(
  QListView * parent,
  QString const & name,
  QDBusProxy * proxy )
  : DBusItem( parent, name, "Service" ),
    d( new DBusService::Private() )
{
    d->p = this;
    d->proxy = proxy;
    m_service = name;
    d->pid = d->discoverPID();
    d->processName = d->discoverProcessName( d->pid );
    d->alias = d->discoverAlias();
}

void DBusService::populate()
throw( QDBusConnectError, QDBusXmlError, QDBusSendError ) {

    d->proxy->setService( m_service );
    d->proxy->setPath( "/" );
    d->proxy->setInterface( "org.freedesktop.DBus.Introspectable" );

    QDBusMessage reply = d->proxy->sendWithReply(
      "Introspect",
      QValueList<QVariant>()
    );
    if ( reply.type() == QDBusMessage::InvalidMessage )
        throw QDBusSendError(
            QString("%1: %2")
            .arg( d->proxy->lastError().name() )
            .arg( d->proxy->lastError().message() )
        );

    QDomDocument doc;
    if ( !doc.setContent( reply[0].toString() ) )
        throw QDBusXmlError(
          i18n(
            "Do not translate 'Introspect'.",
            "XML parse error in reply to Introspect"
          )
        );

    QDomElement docElem = doc.documentElement();
    ensureElementIsNamed( docElem, "node" );

    try {
        do {
            new DBusObject( this, docElem, d->proxy );
            docElem = docElem.nextSibling().toElement();
        } while ( !docElem.isNull() );
    }
    catch ( std::runtime_error const & e ) {
        m_ok = false;
        m_error = e.what();
    }
}

QWidget * DBusService::widget( QWidget * parent ) const {
    QScrollView * scrollview = new QScrollView( parent );
    QVBox * vbox = new QVBox( scrollview->viewport() );
    scrollview->addChild( vbox );
    scrollview->setMidLineWidth( 0 );
    scrollview->setLineWidth( 0 );
    scrollview->setResizePolicy( QScrollView::AutoOneFit );

    new QLabel( i18n( "Service: %1" ).arg( m_service ), vbox );

    if ( d->pid != -1 ) {
    
        // Try to verify...
        QFileInfo fileinfo( QString("/proc/%1").arg( d->pid ) );
        if ( !fileinfo.exists() )
            new QLabel(
              i18n(
                "Process owning this service: %1"
              ).arg(
                i18n(
                  "unknown (D-BUS reports %1, but process "
                  "with that PID does not seem to exist!)"
                ).arg( d->pid )
              ),
              vbox
            );

        else if ( !d->processName.isNull() )
            new QLabel(
              i18n(
                "Process owning this service: %1"
              ).arg(
                QString(
                  "%1 ( %2 )"
                ).arg( d->pid ).arg( d->processName )
              ),
              vbox
            );
        
        else
            new QLabel(
              i18n(
                "Process owning this service: %1"
              ).arg( d->pid ),
              vbox
            );

    }
    else
        new QLabel(
          i18n( "Process owning this service: %1" )
          .arg( i18n( "unknown" ) ),
          vbox
        );


    if ( !d->alias.isNull() )
        new QLabel(
          i18n(
            "This service is also known as service %1."
          ).arg( d->alias ),
          vbox
        );

    addErrorInfo( vbox );

    return scrollview;
}


int DBusService::Private::discoverPID() {

    // Don't even bother trying with DBus; it won't work.
    if ( "org.freedesktop.DBus" == p->m_service )
        return -1;

    try {
        proxy->setService( "org.freedesktop.DBus" );
        proxy->setPath( "/" );
        proxy->setInterface( "org.freedesktop.DBus" );

        QValueList< QVariant > params;
        params.append( p->m_service );

        QDBusMessage reply = proxy->sendWithReply(
            "GetConnectionUnixProcessID",
            params
        );

        if ( reply.type() == QDBusMessage::InvalidMessage )
            throw QDBusSendError(
                QString("%1: %2")
                .arg( proxy->lastError().name() )
                .arg( proxy->lastError().message() )
            );

        // toInt returns 0 on failure or on success where the value
        // was actually 0; but since 0 isn't a valid PID, we know it
        // means failure here.
        int pid = reply[0].toInt();
        if ( 0 == pid )
            return -1;

        return pid;
    }
    catch ( std::runtime_error const & e ) {
        kdDebug() << i18n(
          "Failed to find out PID of %1: %2"
        ).arg( p->m_service ).arg( e.what() ) << endl;
    }

    return -1;
}


QString DBusService::Private::discoverProcessName( int pid ) {

    if ( -1 == pid )
        return QString::null;

    QString cmdlinePath = QString(
      "/proc/%1/cmdline"
    ).arg( pid );

    QFile cmdlineFile( cmdlinePath );

    if ( !cmdlineFile.open( IO_ReadOnly ) )
        return QString::null;

    QString ret;
    if ( cmdlineFile.readLine( ret, 1024 ) < 1 )
        return QString::null;

    return ret;
}


QString DBusService::Private::discoverAlias() {

    // DBus has no other name.
    if ( "org.freedesktop.DBus" == p->m_service )
        return QString::null;

    // AFAICS the only way to do a lookup from unnamed to named
    // would be to iterate through all services, call their
    // discoverAlias, and see which one matches us.  We're not
    // going to do that just yet.
    if ( p->m_service[0] == ':' )
        return QString::null;

    try {
        proxy->setService( "org.freedesktop.DBus" );
        proxy->setPath( "/" );
        proxy->setInterface( "org.freedesktop.DBus" );

        QValueList< QVariant > params;
        params.append( p->m_service );

        QDBusMessage reply = proxy->sendWithReply(
            "GetNameOwner",
            params
        );

        if ( reply.type() == QDBusMessage::InvalidMessage )
            throw QDBusSendError(
                QString("%1: %2")
                .arg( proxy->lastError().name() )
                .arg( proxy->lastError().message() )
            );

        // Don't need to do error checking here; toString() already
        // returns QString::null on error, which is conveniently
        // what we return on error also.
        return reply[0].toString();
    }
    catch ( std::runtime_error const & e ) {
        kdDebug() << i18n(
          "Failed to find alias of %1: %2"
        ).arg( p->m_service ).arg( e.what() ) << endl;
    }

    return QString::null;
}

DBusService::~DBusService() {
    delete d;
}
