/***************************************************************************
 *   Copyright (C) 2006 by Stephen Leaf                                    *
 *   smileaf@smileaf.org                                                   *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             *
 ***************************************************************************/


#include <qlayout.h>

#include <klocale.h>
#include <kglobal.h>
#include <kparts/genericfactory.h>
#include <ksimpleconfig.h>
#include <kglobalsettings.h>
#include <kstandarddirs.h>
#include <kurlrequester.h>
#include <klistview.h>
#include <kopenwith.h>
#include <kpropertiesdialog.h>
#include <kio/job.h>
#include <qdir.h>
#include <qheader.h>

#include "autostart.h"

class desktop : public KListViewItem {
public:
KService * service;
bool bisDesktop;
KURL fileName;
int iStartOn;
enum { AutoStart, Shutdown, ENV };

desktop( QString service, int startOn, QListView *parent ): KListViewItem( parent ) {
	bisDesktop = false;
	iStartOn = startOn;
	fileName = KURL(service);
	if (service.endsWith(".desktop")) {
		this->service = new KService(service);
		bisDesktop = true;
	}
}
bool isDesktop() { return bisDesktop; }
int startOn() { return iStartOn; }
QString fStartOn() {
	switch (iStartOn) {
		case AutoStart: return i18n("Startup"); break;
		case Shutdown: return i18n("Shutdown"); break;
		case ENV: return i18n("ENV"); break;
		default: return ""; break;
	}
}
void setStartOn(int start) {
	iStartOn = start;
	setText(2, fStartOn() );
	KStandardDirs *ksd = new KStandardDirs();
	KGlobalSettings * kgs = new KGlobalSettings();
	QString path;
	switch (iStartOn) {
		case AutoStart: path = kgs->autostartPath()+"/"; break;
		case Shutdown: path = ksd->localkdedir()+"shutdown/"; break;
		case ENV: path = ksd->localkdedir()+"env/"; break;
	}
	KIO::file_move(fileName, KURL( path + fileName.fileName() ));
	fileName = path + fileName.fileName();
}
void updateService() {
	if (bisDesktop) service = new KService( fileName.path() );
}
~desktop() {
	delete service;
}
};

typedef KGenericFactory<autostart, QWidget> autostartFactory;
K_EXPORT_COMPONENT_FACTORY( kcm_autostart, autostartFactory("kcmautostart"))

autostart::autostart(QWidget *parent, const char *name, const QStringList&)
    : KCModule(parent, name), myAboutData(0)
{
    QGridLayout * AutostartConfigLayout = new QGridLayout( this, 1, 1, 11, 6, "AutostartConfigLayout");

    btnAdd = new KPushButton( this, "btnAdd" );

    AutostartConfigLayout->addWidget( btnAdd, 0, 1 );

    listCMD = new KListView( this, "listCMD" );
    listCMD->addColumn( i18n( "Name" ) );
    listCMD->addColumn( i18n( "Command" ) );
	listCMD->addColumn( i18n( "Run on" ) );
    listCMD->setAllColumnsShowFocus( TRUE );
    listCMD->setShowSortIndicator( TRUE );

    AutostartConfigLayout->addMultiCellWidget( listCMD, 0, 4, 0, 0 );
    QSpacerItem * spacer1 = new QSpacerItem( 71, 170, QSizePolicy::Minimum, QSizePolicy::Expanding );
    AutostartConfigLayout->addItem( spacer1, 4, 1 );

    btnRemove = new KPushButton( this, "btnRemove" );

    AutostartConfigLayout->addWidget( btnRemove, 1, 1 );

    btnProperties = new QPushButton( this, "btnProperties" );

    AutostartConfigLayout->addWidget( btnProperties, 2, 1 );

	cmbStartOn = new QComboBox( this, "cmbStartOn" );

	AutostartConfigLayout->addWidget( cmbStartOn, 3, 1 );

	cmbStartOn->insertItem( i18n("Startup") );
	cmbStartOn->insertItem( i18n("Shutdown") );
	cmbStartOn->insertItem( i18n("ENV") );
	cmbStartOn->setEnabled(false);

    btnAdd->setText( i18n( "&Add" ) );
    btnAdd->setAccel( QKeySequence( i18n( "Alt+A" ) ) );
    btnRemove->setText( i18n( "&Remove" ) );
    btnRemove->setAccel( QKeySequence( i18n( "Alt+R" ) ) );
    btnProperties->setText( i18n( "&Properties" ) );
    btnProperties->setAccel( QKeySequence( i18n( "Alt+P" ) ) );

	connect( btnAdd, SIGNAL(clicked()), SLOT(addCMD()) );
	connect( btnRemove, SIGNAL(clicked()), SLOT(removeCMD()) );
	connect( listCMD, SIGNAL(doubleClicked(QListViewItem*)), SLOT(editCMD(QListViewItem*)) );
	connect( btnProperties, SIGNAL(clicked()), SLOT(editCMD()) );
	connect( cmbStartOn, SIGNAL(activated(int)), SLOT(setStartOn(int)) );
	connect( listCMD, SIGNAL(selectionChanged(QListViewItem*)), SLOT(selectionChanged(QListViewItem*)) );

	listCMD->setFocus();

    load();

	KAboutData* about = new KAboutData("autostart", I18N_NOOP("KDE Autostart Manager"), "0.5",
		I18N_NOOP("KDE Autostart Manager Control Panel Module"),
		KAboutData::License_GPL,
		I18N_NOOP("(c) 2006 Stephen Leaf"), 0, 0);
	about->addAuthor("Stephen Leaf", 0, "smileaf@smileaf.org");
	setAboutData( about );
};


autostart::~autostart()
{}


void autostart::load()
{
	kgs = new KGlobalSettings();
	kdDebug() << "According to KDE your Autostart location is: " << kgs->autostartPath() << endl;
	KStandardDirs *ksd = new KStandardDirs();

	QString path;
	for (int x=0;x<3;x++) {
		if (x==0) path = kgs->autostartPath();
		else if (x==1) path = ksd->localkdedir() + "/shutdown";
		else if (x==2) path = ksd->localkdedir() + "/env";

		if (! KStandardDirs::exists(path)) KStandardDirs::makeDir(path);

		QDir *autostartdir = new QDir( path );
		autostartdir->setFilter( QDir::Files);
		const QFileInfoList *list = autostartdir->entryInfoList();
		QFileInfoListIterator it( *list );
		QFileInfo *fi;
		while ( (fi = it.current()) != 0 ) {
			QString filename = fi->fileName();
			desktop * item = new desktop( fi->absFilePath(), x, listCMD );
			if ( ! item->isDesktop() ) {
				if ( fi->isSymLink() ) {
					QString link = fi->readLink();
					item->setText( 0, filename );
					item->setText( 1, link );
					item->setText( 2, item->fStartOn() );
				} else {
					item->setText( 0, filename );
					item->setText( 1, filename );
					item->setText( 2, item->fStartOn() );
				}
			} else {
				item->setText( 0, item->service->name() );
				item->setText( 1, item->service->exec() );
				item->setText( 2, item->fStartOn() );
			}
			++it;
		}
	}
}

void autostart::addCMD() {
	KService::Ptr service = 0L;
	KOpenWithDlg owdlg( this );
	if (owdlg.exec() != QDialog::Accepted)
		return;
	service = owdlg.service();

	Q_ASSERT(service);
	if (!service)
		return; // Don't crash if KOpenWith wasn't able to create service.


	KURL desktopTemplate;

	if ( service->desktopEntryName().isNull() ) {
		desktopTemplate = KURL( kgs->autostartPath() + service->name() + ".desktop" );
		KSimpleConfig ksc(desktopTemplate.path());
		ksc.setGroup("Desktop Entry");
		ksc.writeEntry("Encoding","UTF-8");
		ksc.writeEntry("Exec",service->exec());
		ksc.writeEntry("Icon","exec");
		ksc.writeEntry("Path","");
		ksc.writeEntry("Terminal",false);
		ksc.writeEntry("Type","Application");
		ksc.sync();

		// FIXME: Make it so you can't give focus to the parent before this goes away.
		// If the parent closes before this does, a crash is generated.
		KPropertiesDialog dlg( desktopTemplate, this, 0, true /*modal*/, false /*no auto-show*/ );
		if ( dlg.exec() != QDialog::Accepted )
			return;
	} else {
		desktopTemplate = KURL( locate("apps", service->desktopEntryPath()) );

		// FIXME: Make it so you can't give focus to the parent before this goes away.
		// If the parent closes before this does, a crash is generated.
		KPropertiesDialog dlg( desktopTemplate, KURL(kgs->autostartPath()), service->name() + ".desktop", this, 0, true /*modal*/, false /*no auto-show*/ );
		if ( dlg.exec() != QDialog::Accepted )
			return;
	}

	desktop * item = new desktop( kgs->autostartPath() + service->name() + ".desktop", desktop::AutoStart, listCMD );
	item->setText( 0, item->service->name() );
	item->setText( 1, item->service->exec() );
	item->setText( 2, item->fStartOn() );
	emit changed(true);
}

void autostart::removeCMD() {
	if (!listCMD->selectedItem()) return;

	KIO::del(((desktop *)listCMD->selectedItem())->fileName);

	listCMD->takeItem( listCMD->selectedItem() );
	kdDebug() << "Deleting file" << endl;

	emit changed(true);
}

void autostart::editCMD(QListViewItem* entry) {
	if (!entry) return;

	((desktop*)entry)->updateService();
	KFileItem kfi = KFileItem( KFileItem::Unknown, KFileItem::Unknown, KURL( ((desktop*)entry)->fileName ), true );
	if (! editCMD( kfi )) return;

	// Remove and recreate
	if (((desktop*)entry)->isDesktop()) {
		listCMD->takeItem( listCMD->selectedItem() );
		desktop * item = new desktop( ((desktop*)entry)->fileName.path(), ((desktop*)entry)->startOn(), listCMD );
		item->setText( 0, ((desktop*)entry)->service->name() );
		item->setText( 1, ((desktop*)entry)->service->exec() );
		item->setText( 2, ((desktop*)entry)->fStartOn() );
	}
}

bool autostart::editCMD( KFileItem item) {
	KPropertiesDialog dlg( &item, this );
	if ( dlg.exec() != QDialog::Accepted )
		return false;

	kdDebug() << "Saving file" << endl;
	emit changed(true);
	return true;
}

void autostart::editCMD() {
	editCMD( listCMD->selectedItem() );
}

void autostart::setStartOn( int index ) {
	((desktop*)listCMD->currentItem())->setStartOn(index);
}

void autostart::selectionChanged(QListViewItem* entry) {
	cmbStartOn->setEnabled( (entry != 0) );
	cmbStartOn->setCurrentItem( ((desktop*)entry)->startOn() );
}

void autostart::defaults(){}


void autostart::save()
{}


int autostart::buttons()
{
    return KCModule::Apply|KCModule::Help;
}


QString autostart::quickHelp() const
{
    return i18n("This module helps you configure which applications KDE runs When starting up and shutting down.");
}
