/***************************************************************************

   Copyright (C) 2007 Antonio Aloisio <gnuton@gnuton.org>

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2 of the License, or
   (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU General Public License
   along with this program; if not, write to the
   Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
   Boston, MA 02110-1301, USA.
 ***************************************************************************/
#include "imagemanager.h"

#include <QImage>
#include <QPixmap>
#include <QLabel>
#include <QLineEdit>
#include <QComboBox>
#include <QFileInfo>
#include <QDir>

#include <kurl.h>
#include <klocale.h>
#include <ktempdir.h>
#include <kdebug.h>
#include <kmessagebox.h>
#include <kvbox.h>

namespace KBlogger
{

KBloggerImageManager::KBloggerImageManager(QDir* tmpDir, QWidget *parent)
        : QWidget(parent), mImage(0), mTempDir(tmpDir)
{
    setupUi(this);
    mImage = new QImage();
}

void KBloggerImageManager::setImage(const QString& filename)
{
    QImage preview128;//preview 128x128
    mFilename = filename;
    if (! mImage->load(filename)) return;

    preview128 = mImage->scaled(128, 128, Qt::KeepAspectRatio, Qt::FastTransformation);
    imagePreviewLabel->setPixmap ( QPixmap(preview128) ); //FIXME  ‘__comp_ctor ’ is deprecated (qpixmap.h:192)

    resizeWidthtLine->setEnabled(false);
    resizeWidthtLine->setText(QString::number(mImage->width()));
    resizeHeightLine->setText(QString::number(mImage->height()));

    //Fill infos
    infoHeightLabel->setText(i18n("Height: %1", mImage->height()));
    infoWidthLabel->setText(i18n("Width: %1", mImage->width()));
    infoDepthLabel->setText(i18n("Depth: %1", mImage->depth()));

    infoGraysLabel->setText(i18n("GreyScale: %1", mImage->isGrayscale()  ));
    infoTransLabel->setText(i18n("Transparency: %1", mImage->hasAlphaChannel() ));
}

void KBloggerImageManager::comboSizeChanged(int size)
{
    if ( mImage->isNull()) return;
    int origWidth, origHeight, blogWidth, modWidth, modHeight;
    origWidth = mImage->width();
    origHeight = mImage->height();
    blogWidth = blogWidthLine->text().toInt();

    switch (size) {
    case 0: { //Original size
        modWidth = origWidth;
        modHeight = origHeight;
    }
    break;
    case 1: { //Big size
        modWidth = blogWidth;
        modHeight = modWidth * origHeight / origWidth;
    }
    break;
    case 2: { //Medium size
        modWidth = blogWidth / 2;
        modHeight = modWidth * origHeight / origWidth;
    }
    break;
    case 3: { //Small size
        modWidth = blogWidth / 3;
        modHeight = modWidth * origHeight / origWidth;
    }
    break;
    case 4: { //Free size
        resizeWidthtLine->setEnabled(true);
        return;
    }
    break;
    break;
    }
    resizeWidthtLine->setEnabled(false);
    resizeWidthtLine->setText(QString::number(modWidth));
    resizeHeightLine->setText(QString::number(modHeight));
}

void KBloggerImageManager::saveResize()
{
    QFileInfo origFile(mFilename);
    KUrl scaledImageFilename;
    QImage scaledImage;
    QString ext = origFile.suffix();
    scaledImage = mImage->scaled(resizeWidthtLine->text().toInt(),
                                 resizeHeightLine->text().toInt(),
                                 Qt::KeepAspectRatio,
                                 Qt::SmoothTransformation);
    scaledImageFilename.setPath( mTempDir->path() );
    scaledImageFilename.addPath(origFile.fileName());
    kDebug() << "KBloggerImageManager::saveResize() file saved:" << scaledImageFilename.path() << endl;
    if (ext.toUpper() == "JPG") ext = "JPEG";
    if (!scaledImage.save(scaledImageFilename.path()))//,ext.upper().toAscii()) )
        KMessageBox::sorry ( 0, i18n ( "File %1 not saved" ).arg ( scaledImageFilename.path() ) );
    else
        imageResized(scaledImageFilename.path());
}

void KBloggerImageManager::recalcSize(const QString& width)
{
    Q_UNUSED(width);
    int modWidth, modHeight, origHeight, origWidth;

    if (mFilename.isNull() ) return;
    origHeight = mImage->height();
    origWidth = mImage->width();
    modWidth = resizeWidthtLine->text().toInt();
    modHeight = modWidth * origHeight / origWidth;
    resizeWidthtLine->setText(QString::number(modWidth));
    resizeHeightLine->setText(QString::number(modHeight));
}

void KBloggerImageManager::widthChanged()
{
    comboSizeChanged(resizeComboBox->currentIndex());
}

}//namespace

#include "imagemanager.moc"
