# -*- coding: utf-8 -*-
#
#       window_region.py
#
#       Copyright 2012 David Klasinc <bigwhale@lubica.net>
#       Copyright 2010 Andrew <andrew@karmic-desktop>
#
#       This program is free software; you can redistribute it and/or modify
#       it under the terms of the GNU General Public License as published by
#       the Free Software Foundation; either version 3 of the License, or
#       (at your option) any later version.
#
#       This program is distributed in the hope that it will be useful,
#       but WITHOUT ANY WARRANTY; without even the implied warranty of
#       MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#       GNU General Public License for more details.
#
#       You should have received a copy of the GNU General Public License
#       along with this program; if not, write to the Free Software
#       Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston,
#       MA 02110-1301, USA.

import cairo
import logging
logger = logging.getLogger("Window Region")

from gettext import gettext as _

from gi.repository import Gtk, GObject, Gdk

class RegionWindow(GObject.GObject):

    __gsignals__ = {
        "region-selected" : (GObject.SIGNAL_RUN_LAST,
                             None,
                               (),
                                ),
        "region-canceled" : (GObject.SIGNAL_RUN_LAST,
                             None,
                               (),
                                ),
    }

    def __init__(self, region = None):
        super(RegionWindow, self).__init__()
        logger.debug("Initializing region window.")
        self.window = Gtk.Window()
        self.window.connect("configure-event", self.cb_configure_event)
        self.window.connect("delete-event", Gtk.main_quit)
        self.window.connect("draw", self.cb_draw)
        self.window.connect("key-press-event", self.cb_keypress_event)
        self.window.connect("button-press-event", self.cb_button_press_event)

        if region:
            logger.debug("Old region defined at: X: {0}, Y: {1}, W: {2}, H: {3}".format(region[0],
                                                                                          region[1],
                                                                                          region[2],
                                                                                          region[3]))
            self.startx = region[0]
            self.starty = region[1]
            self.endx = region[2]
            self.endy = region[3]
            self.window.move(self.startx, self.starty)
        else:
            self.startx = 0
            self.starty = 0
            self.endx = 640
            self.endy = 480
            self.window.set_position(Gtk.WindowPosition.CENTER)

        self.width = self.endx - self.startx
        self.height = self.endy - self.starty
        self.window.set_default_geometry(self.width, self.height)

        self.window.set_border_width(30)
        self.window.set_app_paintable(True)
        self.window.set_has_resize_grip(False)
        self.window.set_resizable(True)
        self.window.add_events(Gdk.EventMask.BUTTON_PRESS_MASK)
        self.window.set_decorated(False)
        self.window.set_property("skip-taskbar-hint", True)
        self.window.set_keep_above(True)
        self.screen = self.window.get_screen()
        self.visual = self.screen.get_rgba_visual()
        self.recording = False

        self.window.set_visual(self.visual)
        if self.visual is not None and self.screen.is_composited():
            self.window.show_all()

    def cb_button_press_event(self, widget, event):
        (op, button) = event.get_button()
        if button == 1:
            if int(event.x) in range(0, 15) and int(event.y) in range(0,15):
                self.window.begin_resize_drag(Gdk.WindowEdge.NORTH_WEST, button,
                                              event.x_root, event.y_root, event.time)

            elif int(event.x) in range(self.width-15, self.width) and int(event.y) in range(0,15):
                self.window.begin_resize_drag(Gdk.WindowEdge.NORTH_EAST, button,
                                              event.x_root, event.y_root, event.time)

            elif int(event.x) in range(self.width-15, self.width) and int(event.y) in range(self.height-15,self.height):
                self.window.begin_resize_drag(Gdk.WindowEdge.SOUTH_EAST, button,
                                              event.x_root, event.y_root, event.time)

            elif int(event.x) in range(0, 15) and int(event.y) in range(self.height-15, self.height):
                self.window.begin_resize_drag(Gdk.WindowEdge.SOUTH_WEST, button,
                                              event.x_root, event.y_root, event.time)

            else:
                self.window.begin_move_drag(button, event.x_root, event.y_root, event.time)

    def cb_keypress_event(self, widget, event):
        (op, keycode) = event.get_keycode()
        if keycode == 36 or keycode == 104: # Enter
            self.window.set_default_geometry(self.width, self.height)
            (self.startx, self.starty) = self.window.get_position()
            self.endx = self.startx + self.width - 1
            self.endy = self.starty + self.height - 1
            self.recording = True
            self.window.input_shape_combine_region(None)
            #
            # When support for masked input is back, remove the hide() call.
            #
            self.window.hide()
            # self.window.queue_draw()
            self.emit("region-selected")
        elif keycode == 9: # ESC
            self.window.hide()
            self.emit("region-canceled")


    def cb_configure_event(self, widget, event):
        self.width = event.width
        self.height = event.height

    def cb_draw(self, widget, cr):
        w = self.width
        h = self.height
        #
        # Drawing a red rectangle around selected area would be extremely nice
        # however, cairo.Region is missing from GIR and from pycairo and
        # it is needed for input_shape_combine_region().
        # See: https://bugs.freedesktop.org/show_bug.cgi?id=44336
        #
        #if self.recording:
        #    cr.set_source_rgba(0.0, 0.0, 0.0, 0.0)
        #    cr.set_operator(cairo.OPERATOR_SOURCE)
        #    cr.paint()
        #    surface = cairo.ImageSurface(cairo.FORMAT_ARGB32, w , h)
        #    surface_ctx = cairo.Context(surface)
        #    surface_ctx.set_source_rgba(1.0, 1.0, 1.0, 0.0)
        #    surface_ctx.set_operator(cairo.OPERATOR_SOURCE)
        #    surface_ctx.paint()
        #    reg = Gdk.cairo_region_create_from_surface(surface)
        #    widget.input_shape_combine_region(reg)
        #    cr.move_to(0, 0)
        #    cr.set_source_rgb(1.0, 0.0, 0.0)
        #    cr.set_line_width(2.0)
        #    cr.rectangle(0, 0, w, h)
        #    cr.stroke()
        #    cr.set_operator(cairo.OPERATOR_OVER)
        #else:
        cr.set_source_rgba(0.0, 0.0, 0.0, 0.4)
        cr.set_operator(cairo.OPERATOR_SOURCE)
        cr.paint()
        cr.set_source_rgba(1.0, 1.0, 1.0, 1.0)
        cr.set_line_width(1.0)
        cr.move_to(0, 0)
        cr.rectangle(0, 0, 15, 15)
        cr.rectangle(w-15, 0, w, 15)
        cr.rectangle(0, h-15, 15, h)
        cr.rectangle(w-15, h-15, w, h)
        cr.fill()
        cr.set_source_rgb(0.0, 0.0, 0.0)
        cr.rectangle(0, 0, w, h)
        cr.stroke()
        cr.set_operator(cairo.OPERATOR_OVER)
        self._outline_text(cr, w, h, 24, _("Select region by resizing the window"))
        self._outline_text(cr, w, h + 50, 24, _("Press ENTER to confirm or ESC to cancel."))
        self._outline_text(cr, w, h + 80, 12, "({0} x {1})".format(w, h))


    def _outline_text(self, cr, w, h, size, text):
        cr.set_font_size(size)
        try:
            cr.select_font_face("Ubuntu", cairo.FONT_SLANT_NORMAL, cairo.FONT_WEIGHT_NORMAL)
        except:
            pass
        te = cr.text_extents(text)
        cr.set_line_width(2.0)
        cx = w/2 - te[2]/2
        cy = h/2 - te[3]/2
        cr.set_source_rgba(0.4, 0.4, 0.4, 1.0)
        cr.move_to(cx, cy)
        cr.text_path(text)
        cr.stroke()
        cr.set_source_rgba(1.0, 1.0, 1.0, 1.0)
        cr.move_to(cx, cy)
        cr.show_text(text)
