/*
 * toolbar.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include "toolbar.hh"
#include <gtkmm/stock.h>
#include "macros.h"
#ifdef HAVE_SPELL
#include "spell.hh"
#endif
#include "utils.hh"
#include <fstream>
#include <iostream>
#include "tempfile.hh"

Toolbar::Toolbar(Conf& conf) :
  _conf(conf),
  _create(Gtk::Stock::NEW),
  _open(Gtk::Stock::OPEN),
  _save(Gtk::Stock::SAVE),
#ifdef ENABLE_PRINT
  _print(Gtk::Stock::PRINT),
#endif
  _close(Gtk::Stock::CLOSE),
  _undo(Gtk::Stock::UNDO),
  _redo(Gtk::Stock::REDO),
  _cut(Gtk::Stock::CUT),
  _copy(Gtk::Stock::COPY),
  _paste(Gtk::Stock::PASTE),
  _erase(Gtk::Stock::DELETE),
  _go_to_l(_("Goto Line")),
  _search_l(_("Search"))
#ifdef HAVE_SPELL
  ,
  _dictionary_l(_("Spelling Dictionary"))
#endif
{
  create_main();
  create_extended();
}

Toolbar::~Toolbar() {

}

Gtk::Toolbar& Toolbar::get_main() {
  return _main;
};

Gtk::VBox& Toolbar::get_extended() {
  return _extended;
}

void Toolbar::create_main() {
  _main.append(_create);
  _main.append(_open);
  _main.append(_save);
#ifdef ENABLE_PRINT
  _main.append(_print);
#endif
  _main.append(s1);
  _main.append(_close);
  _main.append(s2);
  _main.append(_undo);
  _main.append(_redo);
  _main.append(s3);
  _main.append(_cut);
  _main.append(_copy);
  _main.append(_paste);
  _main.append(_erase);

  // Our tooltips.
  // NOTE:
  // http://mail.gnome.org/archives/gtkmm-list/2004-October/msg00010.html
  // http://mail.gnome.org/archives/gtkmm-list/2004-October/msg00011.html
  _main.set_tooltips(true);
  Gtk::Tooltips *tips = _main.get_tooltips_object();
  _create.set_tooltip(*tips, _("Create a new file"));
  _open.set_tooltip(*tips, _("Open a file for editing"));
  _save.set_tooltip(*tips,  _("Save the existing file"));
#ifdef ENABLE_PRINT
  _print.set_tooltip(*tips, _("Print this document"));
#endif
  _close.set_tooltip(*tips, _("Close the active file"));
  _undo.set_tooltip(*tips, _("Undo"));
  _redo.set_tooltip(*tips, _("Redo"));
  _cut.set_tooltip(*tips, _("Cut"));
  _copy.set_tooltip(*tips, _("Copy"));
  _paste.set_tooltip(*tips, _("Paste"));
  _erase.set_tooltip(*tips, _("Delete current selection"));

  // This is for both_horiz style
  // NOTE: http://mail.gnome.org/archives/gtkmm-list/2004-June/msg00112.html
  _create.set_is_important();
  _open.set_is_important();
  _save.set_is_important();
#ifdef ENABLE_PRINT
  _print.set_is_important();
#endif
  _close.set_is_important();
  _undo.set_is_important();
  _redo.set_is_important();
  _cut.set_is_important();
  _copy.set_is_important();
  _paste.set_is_important();
  _erase.set_is_important();

  _main.show_all();

  // Signals.
  _create.signal_clicked().connect(sigc::mem_fun(signal_create_clicked, &sigc::signal<void>::emit));
  _open.signal_clicked().connect(sigc::mem_fun(signal_open_clicked, &sigc::signal<void>::emit));
  _save.signal_clicked().connect(sigc::mem_fun(signal_save_clicked, &sigc::signal<void>::emit));
#ifdef ENABLE_PRINT
  _print.signal_clicked().connect(sigc::mem_fun(signal_print_clicked, &sigc::signal<void>::emit));
#endif
  _close.signal_clicked().connect(sigc::mem_fun(signal_close_clicked, &sigc::signal<void>::emit));
  _undo.signal_clicked().connect(sigc::mem_fun(signal_undo_clicked, &sigc::signal<void>::emit));
  _redo.signal_clicked().connect(sigc::mem_fun(signal_redo_clicked, &sigc::signal<void>::emit));
  _cut.signal_clicked().connect(sigc::mem_fun(signal_cut_clicked, &sigc::signal<void>::emit));
  _copy.signal_clicked().connect(sigc::mem_fun(signal_copy_clicked, &sigc::signal<void>::emit));
  _paste.signal_clicked().connect(sigc::mem_fun(signal_paste_clicked, &sigc::signal<void>::emit));
  _erase.signal_clicked().connect(sigc::mem_fun(signal_erase_clicked, &sigc::signal<void>::emit));
}

void Toolbar::create_extended() {
  _extended.pack_start(box);
  box.pack_start(_search_l, false, false, 10);
  box.pack_start(_search, false, false);
  box.pack_start(_go_to_l, false, false, 10);
  box.pack_start(_go_to, false, false);

#ifdef HAVE_SPELL
  box.pack_start(_dictionary_l, false, false, 10);
  box.pack_start(_dictionary, false, false);

  Gtk::Image *image = Gtk::manage(new Gtk::Image(Gtk::Stock::SPELL_CHECK, Gtk::IconSize(Gtk::ICON_SIZE_MENU)));
  _spell.set_image(*image);
  box.pack_start(_spell, false, false, 10);
  _spell.set_relief(Gtk::RELIEF_NONE);
  _spell.signal_clicked().connect(sigc::mem_fun(signal_spell_clicked, &sigc::signal<void>::emit));

  std::vector<std::string> dicts;
  katoob_spell_list_available(dicts);
  for (unsigned x = 0; x < dicts.size(); x++) {
    _dictionary.append_text(dicts[x]);
  }
  _dictionary.set_active_text(_conf.get("default_dict", "en"));
  // TODO: I want to connect without the callback step.
  signal_dictionary_changed_conn = _dictionary.signal_changed().connect(sigc::mem_fun(*this, &Toolbar::dictionary_changed_cb));
  //  signal_dictionary_changed_conn = _dictionary.signal_changed().connect(sigc::bind<std::string>(sigc::mem_fun(signal_dictionary_changed, &sigc::signal<void, std::string>::emit), _dictionary.get_active_text()));
#endif
  _search.signal_activate().connect(sigc::mem_fun(*this, &Toolbar::search_activate_cb));
  _go_to.signal_activate().connect(sigc::mem_fun(*this, &Toolbar::go_to_activate_cb));

  _extended.pack_start(sep);

  create_extra_buttons();
  box.pack_start(_extra_buttons);
  _extra_buttons.set_toolbar_style(Gtk::ToolbarStyle(Gtk::TOOLBAR_ICONS));

  _extended.show_all();
}

void Toolbar::set_text() {
  _main.set_toolbar_style(Gtk::ToolbarStyle(Gtk::TOOLBAR_TEXT));
  _conf.set("toolbartype","text");
}

void Toolbar::set_icons() {
  _main.set_toolbar_style(Gtk::ToolbarStyle(Gtk::TOOLBAR_ICONS));
  _conf.set("toolbartype","icons");
}

void Toolbar::set_both() {
  _main.set_toolbar_style(Gtk::ToolbarStyle(Gtk::TOOLBAR_BOTH));
  _conf.set("toolbartype","both");
}

void Toolbar::set_beside() {
  _main.set_toolbar_style(Gtk::ToolbarStyle(Gtk::TOOLBAR_BOTH_HORIZ));
  _conf.set("toolbartype","both_horiz");
}

#ifdef HAVE_SPELL
void Toolbar::set_dictionary(std::string& d) {
  signal_dictionary_changed_conn.block();
  _dictionary.set_active_text(d);
  signal_dictionary_changed_conn.unblock();
}

std::string Toolbar::get_dictionary() {
  return _dictionary.get_active_text();
}
#endif

void Toolbar::reset_gui() {
  _conf.get("toolbar", true) == true ? _main.show() : _main.hide();
  _conf.get("extended_toolbar", true) == true ? _extended.show() : _extended.hide();
  _conf.get("extra_buttons", true) == true ? _extra_buttons.show_all() : _extra_buttons.hide();

  const std::string& toolbartype = _conf.get("toolbartype", "both");
  if (toolbartype == "text") {
    set_text();
  }
  else if (toolbartype == "icons") {
    set_icons();
  }
  else if (toolbartype == "both_horiz") {
    set_beside();
  }
  else {
    set_both();
  }
}

void Toolbar::reset_gui(bool enable) {
  _save.set_sensitive(enable);
#ifdef ENABLE_PRINT
  _print.set_sensitive(enable);
#endif
  _close.set_sensitive(enable);
  _undo.set_sensitive(enable);
  _redo.set_sensitive(enable);
  _cut.set_sensitive(enable);
  _copy.set_sensitive(enable);
  _paste.set_sensitive(enable);
  _erase.set_sensitive(enable);
#ifdef HAVE_SPELL
  _dictionary.set_sensitive(enable);
  _spell.set_sensitive(enable);
  _dictionary.set_sensitive(enable);
#endif
  _go_to.set_sensitive(enable);
#ifdef HAVE_SPELL
  _search.set_sensitive(enable);
#endif
  _extra_buttons.set_sensitive(enable);
}

void Toolbar::search_activate_cb() {
  if (_search.get_text().size()) {
    signal_search_activated.emit(_search.get_text());
  }
}

void Toolbar::go_to_activate_cb() {
  if (_go_to.get_text().size()) {
    signal_go_to_activated.emit(atoi(_go_to.get_text().c_str()));
  }
}

void Toolbar::create_extra_buttons() {
  std::string path = Utils::get_data_path("toolbar");
  std::ifstream ifs;
  ifs.open(path.c_str());
  if (ifs.is_open()) {
    std::string buff;
    while (getline(ifs, buff))
      if (!create_extra_button(buff))
	break;
    ifs.close();
  }
}

bool Toolbar::create_extra_button(std::string& label) {
  std::string error;
  TempFile tf;
  if (!tf.ok(error)) {
#if ((defined(_WIN32) && !defined(NDEBUG)) || (!defined(_WIN32)))
    std::cout << "Error creating temp file for icon. " << error << std::endl;
#endif
    return false;
    }

  // TODO: We will do it the C way until we get G++ bindings for cairo and pangocairo.
  cairo_surface_t *s = cairo_image_surface_create(CAIRO_FORMAT_ARGB32, 20, 20);
  cairo_t *t = cairo_create(s);
  PangoLayout *layout = pango_cairo_create_layout(t);
  PangoFontDescription *desc = pango_font_description_from_string("Sans 20");
  pango_layout_set_font_description(layout, desc);
  pango_font_description_free(desc);

  int w, h;
  // Let's clear it.
  cairo_set_source_rgb(t, 1,1,1);
  cairo_paint(t);
  pango_layout_set_text(layout, label.c_str(), -1);
  cairo_set_source_rgb(t, 0, 0, 0);
  pango_cairo_update_layout(t, layout);
  pango_layout_get_size (layout, &w, &h);
  pango_cairo_show_layout(t, layout);
  if (cairo_surface_write_to_png(s, tf.get_name().c_str()) != CAIRO_STATUS_SUCCESS) {
    cairo_destroy (t);
    cairo_surface_destroy(s);
    g_object_unref (layout);
    return false;
  }
  // Now let's create our button.
  Gtk::ToolButton *button = Gtk::manage(new Gtk::ToolButton);

  Gtk::Image *image = Gtk::manage(new Gtk::Image(tf.get_name()));
  button->set_icon_widget(*image);
  button->show_all();
  _extra_buttons.append(*button);
  cairo_destroy(t);
  cairo_surface_destroy(s);
  g_object_unref(layout);
  button->signal_clicked().connect(sigc::bind<std::string>(sigc::mem_fun(*this, &Toolbar::extra_button_clicked), label));
  return true;
}

void Toolbar::enable_undo(bool s) {
  _undo.set_sensitive(s);
}

void Toolbar::enable_redo(bool s) {
  _redo.set_sensitive(s);
}

#ifdef HAVE_SPELL
void Toolbar::enable_dictionary(bool s) {
  _dictionary.set_sensitive(s);
}
#endif

void Toolbar::set_read_only(bool r) {
  _save.set_sensitive(!r);
  _cut.set_sensitive(!r);
  _erase.set_sensitive(!r);
  _paste.set_sensitive(!r);
}

void Toolbar::show_main(bool do_show) {
  do_show ? _main.show() : _main.hide();
  _conf.set("toolbar", do_show);
}

void Toolbar::show_extended(bool do_show) {
  do_show ? _extended.show() : _extended.hide();
  _conf.set("extended_toolbar", do_show);
}
