/*
 * dict.cc
 * This file is part of katoob
 *
 * Copyright (C) 2006 Mohammed Sameer
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330,
 * Boston, MA 02111-1307, USA.
 */

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif /* HAVE_CONFIG_H */

#include <glibmm/thread.h>
#include <iostream>
#include "dict.hh"
#include "dialogs.hh"
#include "macros.h"

// TODO: We need some sort of advanced dialog to display this.

Dict::Dict(std::string& host, int port, int timeout) {
  if (port == 0) {
    port = 2628;
  }
  dc.setHost(const_cast<char *>(host.c_str()), port);
  dc.setTimeout(timeout);
}

bool Dict::ok(std::string& error) {
  try {
    dc.dial();
    return true;
  }
  catch (DICTClient::Exception& ex) {
    error = _("I couldn't connect to the dict server.");
    std::cerr << ex.message() << std::endl;
    return false;
  }
}

Dict::~Dict() {
  // NOTE: Not really needed.
  dc.hangup();
}

bool Dict::define(std::string& word, std::string& book, Definitions& answers, std::string& error) {
  error = _("Couldn't get the definition from the server.");
  try {
    answers = dc.lookup(word.c_str());
    return true;
  }
  catch (DICTClient::Exception& ex) {
    std::cerr << ex.message() << std::endl;
    return false;
  }
}

bool Dict::get_books(Books& books, std::string& error) {
  error = _("Couldn't get the list of dictionaries from the server.");
  try {
    books = dc.ListBooks();
    return true;
  }
  catch (DICTClient::Exception& ex) {
    std::cerr << ex.message() << std::endl;
    return false;
  }
}

void katoob_dict_define_word(Conf *conf, Dispatcher *dispatcher) {
  dispatcher->err = true;
  std::string& error = dispatcher->error;
  std::string& result = dispatcher->out;
  std::string& word = dispatcher->in;

  std::string host = conf->get("dict_host", "dict.arabeyes.org");
  std::string book = conf->get("dict_db", "arabic");
  int port = conf->get("dict_port", 2628);
  int timeout = conf->get("dict_timeout", 4);
  Dict dc(host, port, timeout);
  if (!dc.ok(error)) {
    dispatcher->emit();
    return;
  }

  Definitions answers;

  if (!dc.define(word, book, answers, error)) {
    dispatcher->emit();
    return;
  }

  dispatcher->err = false;

  if (answers.empty()) {
    result = _("No results were found.");
    dispatcher->emit();
    return;
  }

  result = answers[0].definition;

  dispatcher->emit();
  return;
}

bool katoob_dict_get_books(Conf* _conf, Books& books, std::string& error) {
  std::string host = _conf->get("dict_host", "dict.arabeyes.org");
  std::string book = _conf->get("dict_db", "arabic");
  int port = _conf->get("dict_port", 2628);
  int timeout = _conf->get("dict_timeout", 4);
  Dict dc(host, port, timeout);
  if (!dc.ok(error)) {
    return false;
  }

  if (!dc.get_books(books, error)) {
    return false;
  }

  return true;
}
