/***************************************************************************
 *   Copyright (C) 2005 by Roberto Cappuccio and the Kat team              *
 *   Roberto Cappuccio : roberto.cappuccio@gmail.com                       *
 *                                                                         *
 *   This program is free software; you can redistribute it and/or modify  *
 *   it under the terms of the GNU General Public License as published by  *
 *   the Free Software Foundation; either version 2 of the License, or     *
 *   (at your option) any later version.                                   *
 *                                                                         *
 *   This program is distributed in the hope that it will be useful,       *
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of        *
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         *
 *   GNU General Public License for more details.                          *
 *                                                                         *
 *   You should have received a copy of the GNU General Public License     *
 *   along with this program; if not, write to the                         *
 *   Free Software Foundation, Inc.,                                       *
 *   51 Franklin Steet, Fifth Floor, Boston, MA 02110-1301, USA.           *
 ***************************************************************************/

#include <kglobal.h>
#include <kstandarddirs.h>
#include <ctime>
#include <qapplication.h>
#include <klocale.h>
#include <kkeydialog.h>
#include <kmessagebox.h>
#include <krun.h>
#include <kedittoolbar.h>
#include <kmimetype.h>
#include <qfiledialog.h>
#include <qvaluevector.h>
#include <ktrader.h>
#include <kuserprofile.h>
#include <kdebug.h>
#include <ktip.h>
#include <qguardedptr.h>
#include <qpixmap.h>
#include <ksplashscreen.h>
#include <kcmdlineargs.h>
#include <kiconloader.h>

#include "dlgmainwindow.h"
#include "dlgnewcatalog.h"
#include "dlgcreatecatalog.h"
#include "dlgsearch.h"
#include "dlgshowfulltext.h"
#include "dlgshowwordslist.h"
#include "katlistview.h"
#include "katiconview.h"
#include "katiconviewitem.h"
#include "katlistviewitem.h"
#include "dlgproperties.h"
#include "dlgstartupwizard.h"

#include <katlanguagemanager.h>
#include <katengine.h>
#include <katinformation.h>
#include <kwin.h>

#include "katutils.h"

#define FILE_NAME 0
#define FILE_EXTENSION 1
#define FILE_MIMETYPE 2
#define FILE_SIZE 3
#define FILE_DATE 4
#define FILE_TIME 5
#define FILE_THUMBNAIL 6
#define FILE_METADATA 7
#define FILE_FULLTEXT 8
#define FILE_LANGUAGE 9

#define CATALOG_NAME 0
#define CATALOG_BASEPATH 1
#define CATALOG_AUTHOR 2
#define CATALOG_CREATIONDATE 3
#define CATALOG_LASTUPDATE 4
#define CATALOG_FILES 5
#define CATALOG_WORDS 6

KatApplication::KatApplication()
  : KUniqueApplication(), m_mainWindows( 0L )
{
}


KatApplication::~KatApplication()
{
/*
    if ( m_mainWindows != 0L )
        delete m_mainWindows;
*/
}

int KatApplication::newInstance()
{
    if( isRestored() ) {
        // we only have one single mainwindow to restore
        if ( !m_mainWindows )
            m_mainWindows = new DlgMainWindow;
        m_mainWindows->restore(1);
        return 0;
    }
    else
    {
        KCmdLineArgs *args = KCmdLineArgs::parsedArgs();
        if ( m_mainWindows )
        {
            m_mainWindows->show();
            KWin::activateWindow( m_mainWindows->winId() );
            importFile( args );
            return 0;
        }

        m_mainWindows = new DlgMainWindow;
        if ( args->isSet( "onlysystray" ) )
        {
            m_mainWindows->callSystray();
        }
        else
        {
            bool activeMode = args->isSet( "searchmode" );
            config()->setGroup( "General" );
            if( !activeMode && config()->readBoolEntry("ShowSplashScreen", true) && args->isSet( "splash" ) )
            {
                QString appDataPath = locate( "data", "kat/kat_splash.png" );
                QPixmap pixmap( appDataPath );
                splash = new KSplashScreen( pixmap );

                // kill the splash after 2 seconds
                QTimer::singleShot( 2000, this, SLOT( closeSplash() ) );
                splash->show();
                splash->message( i18n( "(C) 2005 - Roberto Cappuccio and the KAT team" ),
                                 QTextEdit::AlignRight, QColor( "white" ) );
            }
            setMainWidget( m_mainWindows );
            m_mainWindows->show();
            if ( args->isSet( "wizard" ) )
                m_mainWindows->activateWizard();
            if ( activeMode )
                m_mainWindows->activateFileFindMode();
            importFile( args );
        }
	args->clear(); //free up memory
    }
    return KUniqueApplication::newInstance();
}

void KatApplication::importFile( KCmdLineArgs *args )
{
    if ( args->isSet( "import" ) )
    {
        KURL urlImportFile = KURL( args->getOption("import") );
        if ( urlImportFile.isValid() )
            m_mainWindows->importCatalog( urlImportFile );
    }
}


void KatApplication::closeSplash()
{
    delete splash;
    splash = 0L;
}


DlgMainWindow::DlgMainWindow() : DockMainWindow ( 0, "KAT" )
{
    m_cat = 0L;
    setCaption( i18n( "Kat" ) );
    m_config = kapp->config();

    initWizard();
    initView();
    initStatusBar();
    initActions();
    createGUI( 0L );

    m_ke = new KatEngine();
    m_catalogs.setAutoDelete( true );
    m_catalogs = m_ke->readCatalogs();

    //now it's not first start we created a db
    m_config->setGroup( "General" );
    m_config->writeEntry( "First Start", false );

    m_config->setDesktopGroup();

    populateViews();

    connectDCOPSignal( 0, "CatalogPage::CatalogAdded", "Catalog_added(int)",
                        "catalogAdded(int)", false );
    connectDCOPSignal( 0, "CatalogPage::CatalogDeleted", "Catalog_deleted(int)",
                        "catalogDeleted(int)", false );
    updateToolbarButton();

    // Start katd kded module
    callSystray();
}

DlgMainWindow::~DlgMainWindow()
{
    slotSaveProperties();
    lvFiles->saveLayout( m_config, "Header-Order" );
    delete m_cat;
    m_cat = 0L;
    delete m_ke;
    m_ke = 0L;
}

void DlgMainWindow::callSystray()
{
    // Start katd kded module
    QCString service = "katd";

    QByteArray data, replyData;
    QCString replyType;
    QDataStream arg( data, IO_WriteOnly );
    arg << service;
    if ( kapp->dcopClient()->call( "kded", "kded", "loadModule(QCString)", data, replyType, replyData ) && replyType=="bool") {
        QDataStream reply( replyData, IO_ReadOnly );
        bool result;
        reply >> result;
        if ( !result )
            KMessageBox::error( this, i18n( "Unable to start kat daemon." ) );
    }
    else {
	 kdDebug() << "loadModule() on kded returned an unexpected type of reply: " << replyType << endl;
    }
}

void DlgMainWindow::activateWizard()
{
    slotLaunchWizard();
}

void DlgMainWindow::activateFileFindMode()
{
    slotFileFind();
    toolbarSearch->setChecked( true );
}

void DlgMainWindow::initWizard()
{
    m_config->setGroup( "General" );

    if ( m_config->readBoolEntry( "First Start", true ) )
    {
        wizard();
    }
}

void DlgMainWindow::slotLaunchWizard()
{
    wizard();
}

void DlgMainWindow::wizard()
{
    //TODO test if we delete it.
    DlgStartupWizard* sw = new DlgStartupWizard();
    sw->exec();
    delete sw;
}

void DlgMainWindow::catalogAdded(int _id)
{
    kdDebug()<<" add new catalog id :"<<_id<<endl;
    m_catalogs.clear();
    m_catalogs = m_ke->readCatalogs();
    populateViews();

    updateToolbarButton();
}

void DlgMainWindow::catalogDeleted(int _id)
{
    kdDebug()<<" delete catalog id :"<<_id<<endl;
    m_catalogs.clear();
    m_catalogs = m_ke->readCatalogs();
    populateViews();

    updateToolbarButton();
}



void DlgMainWindow::updateCatalog()
{
    m_catalogs.clear();

    m_catalogs = m_ke->readCatalogs();
    populateViews();

    updateToolbarButton();
}

void DlgMainWindow::updateToolbarButton()
{
    if ( m_catalogs.count() > 0 )
    {
        toolbarSearch->setEnabled( true );
        toolbarViewDetailed->setEnabled( true );
        toolbarViewIcons->setEnabled( true );
        toolbarViewThumbnails->setEnabled( true );
        fileDeleteAllCatalogsAction->setEnabled( true );
        fileCatalogPropertiesAction->setEnabled( true );
        fileDeleteCatalogAction->setEnabled( true );
        fileExportCatalogAction->setEnabled( true );
    }
    else
    {
        toolbarSearch->setEnabled( false );
        toolbarViewDetailed->setEnabled( false );
        toolbarViewIcons->setEnabled( false );
        toolbarViewThumbnails->setEnabled( false );
        fileDeleteAllCatalogsAction->setEnabled( false );
        fileCatalogPropertiesAction->setEnabled( false );
        fileDeleteCatalogAction->setEnabled( false );
        fileExportCatalogAction->setEnabled( false );

    }

}

void DlgMainWindow::initActions()
{
    // catalog menu
    fileNewAction = new KAction( i18n( "New..." ), "filenew", 0,
        this, SLOT( slotFileNew() ), actionCollection(),"new_catalog" );
    fileNewAction->setToolTip( i18n( "Create a new catalog" ) );

    fileImportCatalogAction = new KAction( i18n( "Import..." ), 0,
        this, SLOT( slotImportCatalog() ), actionCollection(), "import_catalog" );
    fileImportCatalogAction->setToolTip( i18n( "Open a previously saved catalog" ) );
    fileImportCatalogAction->setEnabled( true );

    fileCatalogPropertiesAction = new KAction( i18n( "Properties" ), 0,
        this, SLOT( slotCatalogProperties() ), actionCollection(), "catalog_properties" );
    fileCatalogPropertiesAction->setEnabled( false );
    fileExportCatalogAction = new KAction( i18n( "Export..." ), 0,
        this, SLOT( slotExportCatalog() ), actionCollection(), "export_catalog" );
    fileExportCatalogAction->setEnabled( false );
    fileDeleteCatalogAction = new KAction( i18n( "Delete" ), "editdelete", 0,
        this, SLOT( slotDeleteCatalog() ), actionCollection(), "delete_catalog" );
    fileDeleteCatalogAction->setEnabled( false );
    fileDeleteAllCatalogsAction = new KAction( i18n( "Delete All" ), "editdelete", 0,
        this, SLOT( slotDeleteAllCatalogs() ), actionCollection(), "delete_all_catalogs" );
    if ( m_catalogs.count() > 0 )
        fileDeleteAllCatalogsAction->setEnabled( true );
    else
        fileDeleteAllCatalogsAction->setEnabled( false );

    filePrintAction = KStdAction::print( this, SLOT( slotFilePrint() ), actionCollection(), "print" );
    fileExitAction = KStdAction::quit( this, SLOT( close() ), actionCollection(), "quit" );

    // edit menu
    editUndoAction = KStdAction::undo( this, SLOT( slotEditUndo() ), actionCollection() );
    editRedoAction = KStdAction::redo( this, SLOT( slotEditRedo() ), actionCollection() );
    editCutAction = KStdAction::cut( this, SLOT( slotEditCut() ), actionCollection() );
    editCopyAction = KStdAction::copy( this, SLOT( slotEditCopy() ), actionCollection() );
    editPasteAction = KStdAction::paste( this, SLOT( slotEditPaste() ), actionCollection() );
    editFindAction = KStdAction::find( this, SLOT( slotEditFind() ), actionCollection() );

    (void)KStdAction::keyBindings( this, SLOT( slotConfigureKeys() ), actionCollection() );

    KStdAction::configureToolbars( this, SLOT( slotEditToolbars() ), actionCollection() );

    KStdAction::tipOfDay( this, SLOT( slotTipOfTheDay() ), actionCollection() );

    //FIXME : found an better name
    ( void )new KAction( i18n( "Launch Wizard" ), 0, this, SLOT( slotLaunchWizard() ), actionCollection(), "launch_wizard" );

    toolbarSearch = new KToggleAction( i18n( "Search Files" ), "filefind", 0,
        this, SLOT( slotFileFind() ), actionCollection(), "toolbar_search_files" );
    toolbarSearch->setToolTip( i18n( "Find files in the catalog" ) );

    toolbarViewDetailed = new KToggleAction( i18n( "Show Detailed View" ), "view_detailed", 0,
        this, SLOT(slotViewDetailed() ), actionCollection(), "toolbar_view_detailled" );
    toolbarViewDetailed->setToolTip( i18n( "Detailed view" ) );
    toolbarViewDetailed->setExclusiveGroup( "viewmodes" );
    toolbarViewDetailed->setEnabled( false );
    toolbarViewDetailed->setChecked( true );

    toolbarViewIcons = new KToggleAction( i18n( "Show Detailed View" ), "view_icon", 0,
        this, SLOT( slotViewIcons() ), actionCollection(), "toolbar_view_icons" );
    toolbarViewIcons->setToolTip( i18n( "Icon view" ) );
    toolbarViewIcons->setExclusiveGroup( "viewmodes" );
    toolbarViewIcons->setEnabled( false );

    toolbarViewThumbnails = new KToggleAction(i18n( "Show Detailed View" ), "thumbnail", 0,
        this, SLOT(slotViewThumbnails()), actionCollection(), "toolbar_view_thumbnails" );
    toolbarViewThumbnails->setToolTip( i18n( "Thumbnails view" ) );
    toolbarViewThumbnails->setExclusiveGroup( "viewmodes" );
    toolbarViewThumbnails->setEnabled( false );

    toolbarConfigure = new KAction( i18n( "Configure Kat..." ), "configure", 0,
        this, SLOT( slotConfigureKat() ), actionCollection(), "toolbar_configure_kat" );

    toolbarConfigure->setToolTip( i18n( "Configure Kat" ) );

    connect( lvFolders, SIGNAL( pressed( QListViewItem* ) ),
        this, SLOT( selectFolder( QListViewItem* ) ) );
    connect( lvFolders, SIGNAL( doubleClicked( QListViewItem* ) ),
        this, SLOT( doubleClickedFolder( QListViewItem* ) ) );
    connect( lvFolders, SIGNAL( expanded( QListViewItem* ) ),
        this, SLOT( expandFolder( QListViewItem* ) ) );
    connect( lvFolders, SIGNAL( contextMenu( KListView*, QListViewItem*, const QPoint& ) ),
        this, SLOT( contextFolder( KListView*, QListViewItem*, const QPoint& ) ) );
    connect( lvFolders, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&,int ) ),
        this, SLOT( contextReqFolder( QListViewItem*, const QPoint&, int ) ) );

    connect( lvFiles, SIGNAL( contextMenu( KListView*, QListViewItem*, const QPoint& ) ),
        this, SLOT( contextFile( KListView*, QListViewItem*, const QPoint& ) ) );
    connect( lvFiles, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&,int ) ),
        this, SLOT( contextReqFile( QListViewItem*, const QPoint&, int ) ) );
    connect( lvFiles, SIGNAL( doubleClicked( QListViewItem* ) ),
        this, SLOT( doubleClickedFile( QListViewItem* ) ) );
    connect( lvFiles, SIGNAL( rightButtonClicked( QListViewItem*, const QPoint&, int ) ),
        this, SLOT( rightButtonFile( QListViewItem*, const QPoint&, int ) ) );
    connect( lvFiles, SIGNAL( currentChanged( QListViewItem* ) ),
        this, SLOT( selectFile( QListViewItem* ) ) );

    connect( lvResults, SIGNAL( contextMenu( KListView*, QListViewItem*, const QPoint& ) ),
        this, SLOT( contextFile( KListView*, QListViewItem*, const QPoint& ) ) );
    connect( lvResults, SIGNAL( contextMenuRequested( QListViewItem*, const QPoint&,int ) ),
        this, SLOT( contextReqFile( QListViewItem*, const QPoint&, int ) ) );
    connect( lvResults, SIGNAL( doubleClicked( QListViewItem* ) ),
        this, SLOT( doubleClickedFile( QListViewItem* ) ) );
    connect( lvResults, SIGNAL( rightButtonClicked( QListViewItem*, const QPoint&, int ) ),
        this, SLOT( rightButtonFile( QListViewItem*, const QPoint&, int ) ) );
    connect( lvResults, SIGNAL( currentChanged( QListViewItem* ) ),
        this, SLOT( selectFile( QListViewItem* ) ) );

    connect( ivFiles, SIGNAL( currentChanged( QIconViewItem* ) ),
        this, SLOT( selectFileIcon( QIconViewItem* ) ) );
    connect( ivFiles, SIGNAL( doubleClicked( QIconViewItem* ) ),
        this, SLOT( doubleClickedFileIcon( QIconViewItem* ) ) );
    connect( ivFiles, SIGNAL( contextMenuRequested( QIconViewItem*, const QPoint& ) ),
        this, SLOT( contextFileIcon( QIconViewItem*, const QPoint& ) ) );

    connect( ivResults, SIGNAL( currentChanged( QIconViewItem* ) ),
        this, SLOT( selectFileIcon( QIconViewItem* ) ) );
    connect( ivResults, SIGNAL( doubleClicked( QIconViewItem* ) ),
        this, SLOT( doubleClickedFileIcon( QIconViewItem* ) ) );
    connect( ivResults, SIGNAL( contextMenuRequested( QIconViewItem*, const QPoint& ) ),
        this, SLOT( contextFileIcon( QIconViewItem*, const QPoint& ) ) );

    connect( ivFilesThumbnails, SIGNAL( currentChanged( QIconViewItem* ) ),
        this, SLOT( selectFileThumbnail( QIconViewItem* ) ) );
    connect( ivFilesThumbnails, SIGNAL( doubleClicked( QIconViewItem* ) ),
        this, SLOT( doubleClickedFileThumbnail( QIconViewItem* ) ) );
    connect( ivFilesThumbnails, SIGNAL( contextMenuRequested( QIconViewItem*, const QPoint& ) ),
        this, SLOT( contextFileThumbnail( QIconViewItem*, const QPoint& ) ) );

    connect( ivResultsThumbnails, SIGNAL( currentChanged( QIconViewItem* ) ),
        this, SLOT( selectFileThumbnail( QIconViewItem* ) ) );
    connect( ivResultsThumbnails, SIGNAL( doubleClicked( QIconViewItem* ) ),
        this, SLOT( doubleClickedFileThumbnail( QIconViewItem* ) ) );
    connect( ivResultsThumbnails, SIGNAL( contextMenuRequested( QIconViewItem*, const QPoint& ) ),
        this, SLOT( contextFileThumbnail( QIconViewItem*, const QPoint& ) ) );

    disableAction();
    showTipOnStart();
}

void DlgMainWindow::slotTipOfTheDay()
{
    KTipDialog::showTip( /*0*/this, QString::null, true );
}

void DlgMainWindow::showTipOnStart()
{
    KTipDialog::showTip(this);
}

void DlgMainWindow::disableAction()
{
    filePrintAction->setEnabled( false );
    editUndoAction->setEnabled( false );
    editRedoAction->setEnabled( false );
    editCutAction->setEnabled( false );
    editCopyAction->setEnabled( false );
    editPasteAction->setEnabled( false );
    editFindAction->setEnabled( false );
}

void DlgMainWindow::addCatalogId( int catalogId )
{
    // Send addCatalog
    QByteArray data, replyData;
    QCString replyType;
    QDataStream arg( data, IO_WriteOnly );
    arg << catalogId;

    if (kapp->dcopClient()->call( "kded", "katd", "addCatalog(int)", data, replyType, replyData ) && replyType=="bool" ) {
        QDataStream reply(replyData, IO_ReadOnly);
        bool result;
        reply >> result;
        if ( !result )
            KMessageBox::error(this, i18n("Unable to add catalog."));

    }
    else
    {
            kdDebug() << "addCatalog() on kded returned an unexpected type of reply: " << replyType << endl;
    }

}

void DlgMainWindow::deleteCatalogId( int catalogId )
{
    // Send deleteCatalog
    QByteArray data, replyData;
    QCString replyType;
    QDataStream arg( data, IO_WriteOnly );
    arg << catalogId;
    if (kapp->dcopClient()->call( "kded", "katd", "deleteCatalog(int)", data, replyType, replyData ) && replyType=="bool") {
        QDataStream reply(replyData, IO_ReadOnly);
        bool result;
        reply >> result;
        if ( !result )
            KMessageBox::error(this, i18n("Unable to delete catalog."));
    }
    else
    {
        kdDebug() << "deleteCatalog() on kded returned an unexpected type of reply: " << replyType << endl;
    }

}

void DlgMainWindow::slotDeleteAllCatalogs()
{
    int countItem = lvFolders->childCount();
    if ( countItem == 0 )
        return;
    int result = KMessageBox::warningContinueCancel( this, i18n( "Do you really want to delete all catalog?" ),
                                             i18n( "Delete Catalog" ), KStdGuiItem::del() );
    if ( result == KMessageBox::Continue )
    {
        for ( int i = 0; i < countItem; ++i )
        {
            KatCatalog* cat = m_ke->getCatalog( lvFolders->itemAtIndex(i )->text( 0 ) );
            if ( cat )
            {
                m_ke->deleteCatalog( cat );
                deleteCatalogId( cat->catalogId() );
            }

        }
        m_catalogs.clear();

        m_catalogs = m_ke->readCatalogs();
        populateViews();

        updateToolbarButton();
    }
}

void DlgMainWindow::slotImportCatalog()
{
    importCatalog();
}

void DlgMainWindow::slotOpenURL( const KURL&_url )
{
    importCatalog( _url ); //TODO: This will become File->Import
}

void DlgMainWindow::initStatusBar()
{
    //TODO: Implement the status bar
}

void DlgMainWindow::slotConfigureKat()
{
    QStringList args;
    args << "katcontrol";
    KApplication::kdeinitExec( "kcmshell", args );
}

void DlgMainWindow::slotEditToolbars()
{
    saveMainWindowSettings( m_config, "main_window_settings" );
    KEditToolbar dlg( factory(),this );
    connect( &dlg, SIGNAL( newToolbarConfig() ), SLOT( slotNewToolBarConfig() ) );
    dlg.exec();
}

void DlgMainWindow::slotNewToolBarConfig()
{
    applyMainWindowSettings( m_config, "main_window_settings" );
}

void DlgMainWindow::initView()
{
    //Find the Kat application data path
    QString appDataPath = locate( "data", "kat/kat_bg.png" );

    // File tree
    pmCatalogIcon = loadMenuIconEnabled( "kate" );
    pmDiscIcon = loadMenuIconEnabled( "hdd_unmount" );
    pmFolderIconSmall = loadMenuIconEnabled( "folder" );
    pmFileIconSmall = loadMenuIconEnabled( "unknown" );
    pmCheckIconSmall = loadMenuIconEnabled( "button_ok" );
    pmFolderIconLarge = loadLargeIconEnabled( "folder" );
    pmFileIconLarge = loadLargeIconEnabled( "unknown" );

    // Set up the status bar
    // Laurent: not used for the moment
    //sbStatusBar = statusBar();

    // DOCK MANAGER
    manager()->setSplitterHighResolution( true );
    manager()->setSplitterOpaqueResize( true );
    manager()->setSplitterKeepSize( false );

    // DOCK WIDGETS
    // Folders dock
    dockFolders = createDockWidget( "folders_tree", NULL, 0L, kapp->makeStdCaption( "Folders Tree" ),
                                    "Folders Tree" );
    lvFolders = new KListView( this );
    lvFolders->addColumn( i18n( "Catalogs" ) );
    lvFolders->setColumnAlignment( 0, Qt::AlignLeft );
    lvFolders->setRootIsDecorated( true );
    lvFolders->setSelectionMode( QListView::Single );
    lvFolders->header()->hide(); // TODO: ?????

    dockFolders->setWidget( lvFolders );

    foldersMenu = new QPopupMenu( lvFolders );

    // Files dock
    dockFiles = createDockWidget( "files_view", NULL, 0L,
                                  kapp->makeStdCaption( "Files view" ),
                                  "Files view" );
    lvFiles = new KatListView( this );
    lvFiles->addColumn( i18n( "Filename" ) );
    lvFiles->addColumn( i18n( "Ext" ) );
    lvFiles->addColumn( i18n( "Type" ) );
    lvFiles->addColumn( i18n( "Size" ) );
    lvFiles->addColumn( i18n( "Date" ) );
    lvFiles->addColumn( i18n( "Time" ) );
    lvFiles->addColumn( i18n( "Thumbnail" ) );
    lvFiles->addColumn( i18n( "Metadata" ) );
    lvFiles->addColumn( i18n( "Fulltext" ) );
    lvFiles->addColumn( i18n( "Language" ) );
    lvFiles->setSorting( FILE_NAME, true );
    lvFiles->setAllColumnsShowFocus( true );
    lvFiles->setRootIsDecorated( false );
    lvFiles->setShowSortIndicator( true );
    lvFiles->setColumnAlignment( FILE_NAME, Qt::AlignLeft );
    lvFiles->setColumnAlignment( FILE_EXTENSION, Qt::AlignLeft );
    lvFiles->setColumnAlignment( FILE_MIMETYPE, Qt::AlignLeft );
    lvFiles->setColumnAlignment( FILE_SIZE, Qt::AlignRight );
    lvFiles->setColumnAlignment( FILE_DATE, Qt::AlignRight );
    lvFiles->setColumnAlignment( FILE_TIME, Qt::AlignRight );
    lvFiles->setColumnAlignment( FILE_THUMBNAIL, Qt::AlignCenter );
    lvFiles->setColumnAlignment( FILE_METADATA, Qt::AlignCenter );
    lvFiles->setColumnAlignment( FILE_FULLTEXT, Qt::AlignCenter );
    lvFiles->setColumnAlignment( FILE_LANGUAGE, Qt::AlignLeft );
    lvFiles->header()->setLabel( FILE_NAME, i18n( "Filename" ) );
    lvFiles->header()->setLabel( FILE_EXTENSION, i18n( "Ext" ) );
    lvFiles->header()->setLabel( FILE_MIMETYPE, i18n( "Type" ) );
    lvFiles->header()->setLabel( FILE_SIZE, i18n( "Size" ) );
    lvFiles->header()->setLabel( FILE_DATE, i18n( "Date" ) );
    lvFiles->header()->setLabel( FILE_TIME, i18n( "Time" ) );
    lvFiles->header()->setLabel( FILE_THUMBNAIL, i18n( "Thumbnail" ) );
    lvFiles->header()->setLabel( FILE_METADATA, i18n( "Metadata" ) );
    lvFiles->header()->setLabel( FILE_FULLTEXT, i18n( "Fulltext" ) );
    lvFiles->header()->setLabel( FILE_LANGUAGE, i18n( "Language" ) );
    lvFiles->setSorting( FILE_NAME, true );
    lvFiles->restoreLayout( m_config, "Header-Order" );
    lvFiles->setSelectionMode( QListView::Extended );

    filesMenu = new QPopupMenu( lvFiles );

    ivFiles = new KatIconView( this );
    ivFiles->hide();
    ivFiles->setResizeMode( QIconView::Adjust );
    ivFiles->setItemTextPos( QIconView::Bottom );
    ivFiles->setGridX( 80 );
    ivFiles->setGridY( 80 );
    ivFiles->setMaxItemWidth( 32 );
    ivFiles->setItemsMovable( false );
    ivFiles->setWordWrapIconText( true );
    ivFiles->setSorting( true, true );
    ivFiles->setSelectionMode( QIconView::Extended );

    filesIconsMenu = new QPopupMenu( ivFiles );

    ivFilesThumbnails = new KatIconView( this );
    ivFilesThumbnails->hide();
    ivFilesThumbnails->setResizeMode( QIconView::Adjust );
    ivFilesThumbnails->setItemTextPos( QIconView::Bottom );
    ivFilesThumbnails->setGridX( 80 );
    ivFilesThumbnails->setGridY( 80 );
    ivFilesThumbnails->setMaxItemWidth( 160 );
    ivFilesThumbnails->setItemsMovable( false );
    ivFilesThumbnails->setWordWrapIconText( true );
    ivFilesThumbnails->setSorting( true, true );
    ivFilesThumbnails->setSelectionMode( QIconView::Extended );

    filesThumbnailsMenu = new QPopupMenu( ivFilesThumbnails );

    dockFiles->setWidget( lvFiles );
    dockMain = dockFiles->manualDock( dockFolders, KDockWidget::DockRight, 2000 );

    // Properties dock
    dockProperties = createDockWidget( "properties_view", NULL, 0L,
                                       kapp->makeStdCaption( "Properties view" ),
                                       "Properties view" );
    teProperties = new KTextEdit( this );
    teProperties->setReadOnly( true );
    teProperties->setPaletteBackgroundPixmap( QPixmap( appDataPath ) );

    dockProperties->setWidget( teProperties );
    dockProperties->manualDock( dockFiles, KDockWidget::DockBottom, 6000 );

    // MainDockWidget
    setView( dockMain );
    setMainDockWidget( dockMain );

    // Search dock
    dockSearch = createDockWidget( "search_view", NULL, 0L,
                                   kapp->makeStdCaption( "Search view" ),
                                   "Search view" );
    dsSearch = new DlgSearch( this );
    dockSearch->setWidget( dsSearch );

    // Results dock
    dockResults = createDockWidget( "results_view", NULL, 0L,
                                    kapp->makeStdCaption( "Results view" ),
                                    "Results view" );
    lvResults = new KatListView( this );
    lvResults->hide();
    lvResults->addColumn( i18n( "Filename" ) );
    lvResults->addColumn( i18n( "Ext" ) );
    lvResults->addColumn( i18n( "Type" ) );
    lvResults->addColumn( i18n( "Size" ) );
    lvResults->addColumn( i18n( "Date" ) );
    lvResults->addColumn( i18n( "Time" ) );
    lvResults->addColumn( i18n( "Thumbnail" ) );
    lvResults->addColumn( i18n( "Metadata" ) );
    lvResults->addColumn( i18n( "Fulltext" ) );
    lvResults->addColumn( i18n( "Language" ) );
    lvResults->setSorting( FILE_NAME, true );
    lvResults->setAllColumnsShowFocus( true );
    lvResults->setRootIsDecorated( false );
    lvResults->setShowSortIndicator( true );
    lvResults->setColumnAlignment( FILE_NAME, Qt::AlignLeft );
    lvResults->setColumnAlignment( FILE_EXTENSION, Qt::AlignLeft );
    lvResults->setColumnAlignment( FILE_MIMETYPE, Qt::AlignLeft );
    lvResults->setColumnAlignment( FILE_SIZE, Qt::AlignRight );
    lvResults->setColumnAlignment( FILE_DATE, Qt::AlignRight );
    lvResults->setColumnAlignment( FILE_TIME, Qt::AlignRight );
    lvResults->setColumnAlignment( FILE_THUMBNAIL, Qt::AlignCenter );
    lvResults->setColumnAlignment( FILE_METADATA, Qt::AlignCenter );
    lvResults->setColumnAlignment( FILE_FULLTEXT, Qt::AlignCenter );
    lvResults->setColumnAlignment( FILE_LANGUAGE, Qt::AlignLeft );
    lvResults->header()->setLabel( FILE_NAME, i18n( "Filename" ) );
    lvResults->header()->setLabel( FILE_EXTENSION, i18n( "Ext" ) );
    lvResults->header()->setLabel( FILE_MIMETYPE, i18n( "Type" ) );
    lvResults->header()->setLabel( FILE_SIZE, i18n( "Size" ) );
    lvResults->header()->setLabel( FILE_DATE, i18n( "Date" ) );
    lvResults->header()->setLabel( FILE_TIME, i18n( "Time" ) );
    lvResults->header()->setLabel( FILE_THUMBNAIL, i18n( "Thumbnail" ) );
    lvResults->header()->setLabel( FILE_METADATA, i18n( "Metadata" ) );
    lvResults->header()->setLabel( FILE_FULLTEXT, i18n( "Fulltext" ) );
    lvResults->header()->setLabel( FILE_LANGUAGE, i18n( "Language" ) );
    lvResults->setSorting( FILE_NAME, true );
    lvResults->setSelectionMode( QListView::Extended );

    resultsMenu = new QPopupMenu( lvResults );

    ivResults = new KatIconView( this );
    ivResults->hide();
    ivResults->setResizeMode( QIconView::Adjust );
    ivResults->setItemTextPos( QIconView::Bottom );
    ivResults->setGridX( 80 );
    ivResults->setGridY( 80 );
    ivResults->setMaxItemWidth( 32 );
    ivResults->setItemsMovable( false );
    ivResults->setWordWrapIconText( true );
    ivResults->setSorting( true, true );
    ivResults->setSelectionMode( QIconView::Extended );

    resultsIconsMenu = new QPopupMenu( ivResults );

    ivResultsThumbnails = new KatIconView( this );
    ivResultsThumbnails->hide();
    ivResultsThumbnails->setResizeMode( QIconView::Adjust );
    ivResultsThumbnails->setItemTextPos( QIconView::Bottom );
    ivResultsThumbnails->setGridX( 80 );
    ivResultsThumbnails->setGridY( 80 );
    ivResultsThumbnails->setMaxItemWidth( 160 );
    ivResultsThumbnails->setItemsMovable( false );
    ivResultsThumbnails->setWordWrapIconText( true );
    ivResultsThumbnails->setSorting( true, true );
    ivResultsThumbnails->setSelectionMode( QIconView::Extended );

    resultsThumbnailsMenu = new QPopupMenu( ivResultsThumbnails );

    dockResults->setWidget( lvResults );
    dockMainSearch = dockResults->manualDock( dockSearch, KDockWidget::DockBottom, 5000 );

    // FINISHING
    resize( QSize( 800, 600 ).expandedTo( minimumSizeHint() ) );
    clearWState( WState_Polished );
}

void DlgMainWindow::slotFileNew()
{
    fileNew();
}

void DlgMainWindow::fileNew()
{
    createNewCatalog();
}

void DlgMainWindow::slotFolderItemSelected( QListViewItem* item )
{
}

void DlgMainWindow::createNewCatalog() {
    DlgNewCatalog *nc = new DlgNewCatalog( this );

    if ( nc->exec() == QDialog::Accepted )
    {
        delete m_cat;
        m_cat = nc->newCatalog();
        // Write the catalog to disk
        m_ke->addCatalog( m_cat );

        addCatalogId( m_cat->catalogId() );
        m_catalogs.clear();
        m_catalogs = m_ke->readCatalogs();
        populateViews();
        if ( m_catalogs.count() > 0 )
        {
            toolbarSearch->setEnabled( true );
            toolbarConfigure->setEnabled( true );
            toolbarViewDetailed->setEnabled( true );
            toolbarViewIcons->setEnabled( true );
            toolbarViewThumbnails->setEnabled( true );
        }
        else
        {
            toolbarSearch->setEnabled( false );
            toolbarConfigure->setEnabled( false );
            toolbarViewDetailed->setEnabled( false );
            toolbarViewIcons->setEnabled( false );
            toolbarViewThumbnails->setEnabled( false );
        }
    }
    delete nc;
}

void DlgMainWindow::importCatalog( const KURL &url )
{
    struct stat my_stat;
    if ( !url.isEmpty() )
    {
        QString filename = url.path();

        // Check if the file exists
        int result = stat( filename, &my_stat );
        if ( result == -1 )
        {
            KMessageBox::error( this, i18n( "The specified file does not exist or is not accessible" ) );
            return;
        }

        m_ke->importCatalog( filename );
        m_catalogs.clear();
        m_catalogs = m_ke->readCatalogs();
        populateViews();

        updateToolbarButton();
    }
}

void DlgMainWindow::importCatalog()
{
    openCatalog();
}

void DlgMainWindow::openCatalog()
{
    KURL fn = KFileDialog::getOpenURL( QString::null, "*.kat|" + i18n( "Kat Catalogs (*.kat)" ),
                                       this, i18n( "Select Catalog to Import" ) );

    if ( !fn.isEmpty() )
        importCatalog( fn );
}


void DlgMainWindow::slotFilePrint()
{
}

void DlgMainWindow::slotSaveProperties()
{
    //FIXME ????
}

bool DlgMainWindow::queryClose()
{
    return true;
}

void DlgMainWindow::slotEditUndo()
{
}

void DlgMainWindow::slotEditRedo()
{
}

void DlgMainWindow::slotEditCut()
{
}

void DlgMainWindow::slotEditCopy()
{
}

void DlgMainWindow::slotEditPaste()
{
}

void DlgMainWindow::slotEditFind()
{
}

void DlgMainWindow::slotConfigureKeys()
{
    KKeyDialog::configure( actionCollection(), this );
}

void DlgMainWindow::slotViewDetailed()
{
    if ( toolbarViewDetailed->isChecked() )
    {
    dockFiles->setWidget( lvFiles );
    lvFiles->show();
    ivFiles->hide();
    ivFilesThumbnails->hide();

    dockResults->setWidget( lvResults );
    lvResults->show();
    ivResults->hide();
    ivResultsThumbnails->hide();
    }
    else
        toolbarViewDetailed->setChecked(true);
}

void DlgMainWindow::slotViewIcons()
{
    if ( toolbarViewIcons->isChecked() )
    {
        dockFiles->setWidget( ivFiles );
        ivFiles->show();
        lvFiles->hide();
        ivFilesThumbnails->hide();

        dockResults->setWidget( ivResults );
        ivResults->show();
        lvResults->hide();
        ivResultsThumbnails->hide();
    }
    else
        toolbarViewIcons->setChecked(true);
}

void DlgMainWindow::slotViewThumbnails()
{
    if ( toolbarViewThumbnails->isChecked() )
    {
        dockFiles->setWidget( ivFilesThumbnails );
        ivFilesThumbnails->show();
        lvFiles->hide();
        ivFiles->hide();

        dockResults->setWidget( ivResultsThumbnails );
        ivResultsThumbnails->show();
        lvResults->hide();
        ivResults->hide();
    }
    else
        toolbarViewThumbnails->setChecked(true);
}

void DlgMainWindow::expandFolder( QListViewItem * item )
{
    if ( item )
    {
        KatListViewItem* lvItem = static_cast<KatListViewItem*>(item);
        if ( lvItem->info.fileId != -2)
            DlgMainWindow::insertChildrenFolders( lvItem->info.catalogId, lvItem->info.fileId, item );
    }
}

void DlgMainWindow::selectFolder( QListViewItem* item )
{
    if ( item )
    {
        KatListViewItem* lvItem = static_cast<KatListViewItem*>(item);

        QString fileType = lvItem->info.fileType;
        int fileId = lvItem->info.fileId;
        int catalogId = lvItem->info.catalogId;

        int columns = lvFiles->columns();

        if ( fileType == "KAT_CATALOG" )
        {
            fileCatalogPropertiesAction->setEnabled( true );
            fileDeleteCatalogAction->setEnabled( true );
            fileExportCatalogAction->setEnabled( true );

            delete m_cat;
            m_cat = m_ke->readCatalog( catalogId );

            for ( int i=0; i <= columns; i++ )
                lvFiles->removeColumn( 0 );

            lvFiles->addColumn( i18n( "Catalog Name" ) );
            lvFiles->addColumn( i18n( "Base Path" ) );
            lvFiles->addColumn( i18n( "Author" ) );
            lvFiles->addColumn( i18n( "Creation Date" ) );
            lvFiles->addColumn( i18n( "Last Update" ) );
            lvFiles->addColumn( i18n( "Files" ) );
            lvFiles->addColumn( i18n( "Words" ) );
            lvFiles->setSorting( CATALOG_NAME, true );
            lvFiles->setAllColumnsShowFocus( true );
            lvFiles->setRootIsDecorated( true );
            lvFiles->setShowSortIndicator( true );
            lvFiles->setColumnAlignment( CATALOG_NAME, Qt::AlignLeft );
            lvFiles->setColumnAlignment( CATALOG_BASEPATH, Qt::AlignLeft );
            lvFiles->setColumnAlignment( CATALOG_AUTHOR, Qt::AlignLeft );
            lvFiles->setColumnAlignment( CATALOG_CREATIONDATE, Qt::AlignRight );
            lvFiles->setColumnAlignment( CATALOG_LASTUPDATE, Qt::AlignRight );
            lvFiles->setColumnAlignment( CATALOG_FILES, Qt::AlignRight );
            lvFiles->setColumnAlignment( CATALOG_WORDS, Qt::AlignRight );

            QDateTime cdt;
            cdt.setTime_t( m_cat->creationDate() );
            QDate cdate = cdt.date();
            QTime ctime = cdt.time();

            QString creationdate = cdate.toString( Qt::LocalDate ) + " " + ctime.toString( Qt::TextDate );

            QDateTime ludt;
            ludt.setTime_t( m_cat->lastUpdateDate() );
            QDate ludate = ludt.date();
            QTime lutime = ludt.time();

            QString lastupdatedate = ludate.toString( Qt::LocalDate ) + " " + lutime.toString( Qt::TextDate );

            lvFiles->clear();
            ivFiles->clear();
            ivFilesThumbnails->clear();
            teProperties->setText( QString::null );

            QString files = QString::number( m_cat->files() );
            addSeparators( files );

            QString words = QString::number( m_cat->words() );
            addSeparators( words );

            KatListViewItem* lvItem = new KatListViewItem( lvFiles );
            lvItem->setText( CATALOG_NAME, m_cat->name() );
            lvItem->setText( CATALOG_BASEPATH, m_cat->path() );
            lvItem->setText( CATALOG_AUTHOR, m_cat->author() );
            lvItem->setText( CATALOG_CREATIONDATE, creationdate );
            lvItem->setText( CATALOG_LASTUPDATE, lastupdatedate );
            lvItem->setText( CATALOG_FILES, files );
            lvItem->setText( CATALOG_WORDS, words );

            KatInformation info;
            info.fileType = "KAT_CATALOG";
            info.fileId = -2;
            info.catalogId = catalogId;
            lvItem->info = info;

        }
        else if ( fileType != "KAT_CATALOG" )
        {
            fileCatalogPropertiesAction->setEnabled( false );
            fileDeleteCatalogAction->setEnabled( false );
            fileExportCatalogAction->setEnabled( false );
            delete m_cat;
            m_cat = m_ke->readCatalog( catalogId );

            for ( int i=0; i <= columns; i++ )
                lvFiles->removeColumn( 0 );

            lvFiles->addColumn( i18n( "Filename" ) );
            lvFiles->addColumn( i18n( "Ext" ) );
            lvFiles->addColumn( i18n( "Type" ) );
            lvFiles->addColumn( i18n( "Size" ) );
            lvFiles->addColumn( i18n( "Date" ) );
            lvFiles->addColumn( i18n( "Time" ) );
            lvFiles->addColumn( i18n( "Thumbnail" ) );
            lvFiles->addColumn( i18n( "Metadata" ) );
            lvFiles->addColumn( i18n( "Fulltext" ) );
            lvFiles->addColumn( i18n( "Language" ) );
            lvFiles->setSorting( FILE_NAME, true );
            lvFiles->setAllColumnsShowFocus( true );
            lvFiles->setRootIsDecorated( true );
            lvFiles->setShowSortIndicator( true );
            lvFiles->setColumnAlignment( FILE_NAME, Qt::AlignLeft );
            lvFiles->setColumnAlignment( FILE_EXTENSION, Qt::AlignLeft );
            lvFiles->setColumnAlignment( FILE_MIMETYPE, Qt::AlignLeft );
            lvFiles->setColumnAlignment( FILE_SIZE, Qt::AlignRight );
            lvFiles->setColumnAlignment( FILE_DATE, Qt::AlignRight );
            lvFiles->setColumnAlignment( FILE_TIME, Qt::AlignRight );
            lvFiles->setColumnAlignment( FILE_THUMBNAIL, Qt::AlignCenter );
            lvFiles->setColumnAlignment( FILE_METADATA, Qt::AlignCenter );
            lvFiles->setColumnAlignment( FILE_FULLTEXT, Qt::AlignCenter );
            lvFiles->setColumnAlignment( FILE_LANGUAGE, Qt::AlignLeft );

            insertChildrenFiles( catalogId, fileId );
        }
        else
            insertChildrenFiles( catalogId, fileId );
    }
}

void DlgMainWindow::selectFile( QListViewItem* item )
{
    if ( item )
    {
        KatListViewItem* lvItem = static_cast<KatListViewItem*>(item);
        showFileProperties( lvItem->info );
    }
}

void DlgMainWindow::selectFileIcon(QIconViewItem* item)
{
    if ( item )
    {
        KatIconViewItem* lvItem = static_cast<KatIconViewItem*>(item);
        showFileProperties( lvItem->info );
    }
}

void DlgMainWindow::selectFileThumbnail( QIconViewItem* item )
{
    if ( item )
    {
        KatIconViewItem* lvItem = static_cast<KatIconViewItem*>(item);
        showFileProperties( lvItem->info );
    }
}

void DlgMainWindow::contextFolder( KListView* lv, QListViewItem* item, const QPoint& point )
{
    if ( item )
    {
        KatListViewItem* lvItem = static_cast<KatListViewItem*>(item);
        if ( lvItem->info.fileType == "KAT_CATALOG" )
        {
            foldersMenu->clear();
            foldersMenu->insertItem( i18n( "Properties" ), this, SLOT( slotCatalogProperties() ), 0, 1 );
            foldersMenu->insertItem( SmallIconSet("editdelete"), i18n( "Delete" ), this, SLOT( slotDeleteCatalog() ), 0, 2 );
            foldersMenu->insertItem( i18n( "Export..." ), this, SLOT( slotExportCatalog() ), 0, 3 );

            foldersMenu->popup( point );
        }
    }
}

void DlgMainWindow::slotDeleteCatalog() {
    QPtrList<QListViewItem> lvItems = lvFolders->selectedItems();
    QListViewItem* item = lvItems.first();
    if ( !item )
        return;

    // get the catalog
    KatCatalog* cat = m_ke->getCatalog( item->text( 0 ) );

    // delete it
    if ( cat != 0L )
    {
        int result = KMessageBox::warningContinueCancel( this, i18n( "Do you really want to delete selected catalog?" ),
                                                 i18n( "Delete Catalog" ), KStdGuiItem::del() );
        switch ( result )
        {
            case KMessageBox::Continue:
                m_ke->deleteCatalog( cat );
                deleteCatalogId( cat->catalogId() );
                m_catalogs.clear();
                m_catalogs = m_ke->readCatalogs();
                updateToolbarButton();
                populateViews();
                break;
            case KMessageBox::Cancel:
                break;
        }
    }
}

void DlgMainWindow::slotCatalogProperties()
{
    QPtrList<QListViewItem> lvItems = lvFolders->selectedItems();
    QListViewItem* item = lvItems.first();
    if ( !item )
        return;
    // get the catalog
    KatCatalog* cat = m_ke->getCatalog( item->text( 0 ) );

    if ( cat )
    {
        DlgProperties *dlg = new DlgProperties( this, cat );
        dlg->exec();
        delete dlg;
    }
}

void DlgMainWindow::slotExportCatalog()
{
    QPtrList<QListViewItem> lvItems = lvFolders->selectedItems();
    QListViewItem* item = lvItems.first();
    if ( !item )
        return;
    // get the catalog
    KatCatalog* cat = m_ke->getCatalog( item->text( 0 ) );

    // export it
    if ( cat )
    {
        QString fn = KFileDialog::getSaveFileName( QString::null, "*.kat|" + i18n( "Kat Catalogs (*.kat)" ), this,
                                                   i18n( "Specify Location & Filename for Catalog to Be Exported" ) );
        if ( !fn.isEmpty() )
        {
            if ( m_ke->exportCatalog( cat, fn ) == 0 )
            {
                int result = KMessageBox::warningContinueCancel( this, i18n( "Do you want to delete exported catalog from the main database?" ),
                                                        i18n( "Delete Catalog" ), KStdGuiItem::del() );
                switch ( result )
                {
                    case KMessageBox::Continue:
                        m_ke->deleteCatalog( cat );
                        deleteCatalogId(cat->catalogId() );
                        m_catalogs.clear();
                        m_catalogs = m_ke->readCatalogs();
                        populateViews();
                        break;
                    case KMessageBox::Cancel:
                        break;
                }
            }
            else
                KMessageBox::error( this, i18n( "The catalog has not been exported due to an error." ) );
        }
    }
}

void DlgMainWindow::contextReqFolder( QListViewItem* item, const QPoint& point, int i )
{
}

void DlgMainWindow::contextFile( KListView* lv, QListViewItem* item, const QPoint& point )
{
    QPtrList<QListViewItem> lvItems = lv->selectedItems();

    // check if all lvItems have the same mimeType
    QString mimeType = QString::null;
    bool different = false;
    QPtrList<QListViewItem>::const_iterator end( lvItems.end() );
    for ( QPtrList<QListViewItem>::const_iterator it = lvItems.begin(); it != end; ++it )
    {
        KatListViewItem* lvItem = static_cast<KatListViewItem*>(*it);
        QString fileMimeType = lvItem->info.fileType;
        if ( ( !mimeType.isEmpty() ) && ( mimeType != fileMimeType ) )
        {
            different = true;
            break;
        }
        mimeType = fileMimeType;
    }

    filesMenu->clear();

    //FIXME: test if we mem leak ?
    QPopupMenu* openWithMenu = new QPopupMenu();

    bool addOpenWithMenu = true;
    bool haveItem = false;
    if ( different )
    {
        openWithMenu->setEnabled( false );
    }
    else
    {
        KMimeType::Ptr mime = KMimeType::mimeType( mimeType );

        offers = KTrader::self()->query( mime->name(), "Type == 'Application'" );

        if ( offers.count() > 0 )
        {
            connect( openWithMenu, SIGNAL( activated( int ) ), this, SLOT( slotOpenWith( int ) ) );

            KTrader::OfferList::Iterator iter;
            KService::Ptr ptr;
            int index = 100;
            for( iter = offers.begin(); iter != offers.end(); ++iter )
            {
                ptr = *iter;
                openWithMenu->insertItem( ptr->pixmap( KIcon::Small ), ptr->name(), index++ );
            }
            haveItem = true;
        }
        else
        {
            addOpenWithMenu=false;
        }
    }
    if ( addOpenWithMenu )
        filesMenu->insertItem( i18n( "Open With" ), openWithMenu, 1 );

    // The following menu items are enabled if 1 only file is selected
    if ( lvItems.count() == 1 )
    {
        KatListViewItem* lvItem = static_cast<KatListViewItem*>(lvItems.first());
        if ( lvItem->info.hasFullText )
        {
            if ( addOpenWithMenu )
                filesMenu->insertSeparator();

            filesMenu->insertItem( i18n( "Show Fulltext" ), this, SLOT( slotShowFullText() ), 0, 2 );
            filesMenu->insertItem( i18n( "Show Words List" ), this, SLOT( slotShowWordsList() ), 0, 3 );
            filesMenu->setItemEnabled ( 2, true );
            filesMenu->setItemEnabled ( 3, true );
            haveItem = true;
        }
    }
    else
    {
        if ( addOpenWithMenu )
            filesMenu->insertSeparator();

        filesMenu->insertItem( i18n( "Show Fulltext" ), 2 );
        filesMenu->insertItem( i18n( "Show Words List" ), 3 );
        filesMenu->setItemEnabled ( 2, false );
        filesMenu->setItemEnabled ( 3, false );
        haveItem = true;
    }
    if ( haveItem )
        filesMenu->popup( point );
}

void DlgMainWindow::slotOpenWith( int id )
{
    if ( id >= 100 && id < 1000 )
    {
        QPtrList<QListViewItem> lvItems;
        QPtrList<QIconViewItem> ivItems;

        if ( getMainDockWidget() == dockMain )
        {
            if ( !lvFiles->isHidden() )
                lvItems = lvFiles->selectedItems();
            if ( !ivFiles->isHidden() )
            {
                for ( QIconViewItem *item = ivFiles->firstItem(); item; item = item->nextItem() )
                {
                    if ( item->isSelected() )
                    {
                        ivItems.append( item );
                    }
                }
            }
            if ( !ivFilesThumbnails->isHidden() )
            {
                for ( QIconViewItem *item = ivFilesThumbnails->firstItem(); item; item = item->nextItem() )
                {
                    if ( item->isSelected() )
                        ivItems.append( item );
                }
            }
        }
        else
        {
            if ( !lvResults->isHidden() )
                lvItems = lvResults->selectedItems();
            if ( !ivResults->isHidden() )
            {
                for ( QIconViewItem *item = ivResults->firstItem(); item; item = item->nextItem() )
                {
                    if ( item->isSelected() )
                        ivItems.append( item );
                }
            }
            if ( !ivResultsThumbnails->isHidden() )
            {
                for ( QIconViewItem *item = ivResultsThumbnails->firstItem(); item; item = item->nextItem() )
                {
                    if ( item->isSelected() )
                        ivItems.append( item );
                }
            }
        }

        KTrader::OfferList::Iterator iter;
        KService::Ptr ptr;
        int index = 100;
        for( iter = offers.begin(); iter != offers.end(); ++iter )
        {
            ptr = *iter;
            if ( index == id )
            {
                KURL::List urlList;

                if ( lvItems.count() != 0 )
                {
                    for ( QPtrList<QListViewItem>::iterator it = lvItems.begin(); it != lvItems.end(); ++it )
                    {
                        KatListViewItem* lvItem = static_cast<KatListViewItem*>(*it);
                        urlList.append( KURL( lvItem->info.fullName ) );
                    }
                }
                if ( ivItems.count() != 0 )
                {
                    for ( QPtrList<QIconViewItem>::iterator it = ivItems.begin(); it != ivItems.end(); ++it )
                    {
                        KatIconViewItem* ivItem = static_cast<KatIconViewItem*>(*it);
                        urlList.append( KURL( ivItem->info.fullName ) );
                    }
                }

                if ( urlList.count() )
                    KRun::run( *ptr, urlList );

                break;
            }
            index++;
        }
        offers.clear();
    }
}

void DlgMainWindow::contextFileIcon( QIconViewItem* item, const QPoint& point )
{
    QPtrList<QIconViewItem> ivItems;
    bool haveItem = false;
    if ( getMainDockWidget() == dockMain )
    {
        for ( QIconViewItem *item = ivFiles->firstItem(); item; item = item->nextItem() )
        {
            if ( item->isSelected() )
                ivItems.append( item );
        }
    }
    else
    {
        for ( QIconViewItem *item = ivResults->firstItem(); item; item = item->nextItem() )
        {
            if ( item->isSelected() )
                ivItems.append( item );
        }
    }

    // check if all lvItems have the same mimeType
    QString mimeType = QString::null;
    bool different = false;
    for ( QPtrList<QIconViewItem>::iterator it = ivItems.begin(); it != ivItems.end(); ++it )
    {
        KatIconViewItem* ivItem = static_cast<KatIconViewItem*>(*it);
        QString fileMimeType = ivItem->info.fileType;
        if ( ( !mimeType.isEmpty() ) && ( mimeType != fileMimeType ) )
        {
            different = true;
            break;
        }
        mimeType = fileMimeType;
    }

    filesIconsMenu->clear();

    QPopupMenu* openWithMenu = new QPopupMenu();

    bool addOpenWithMenu = true;
    if ( different )
    {
        openWithMenu->setEnabled( false );
    }
    else
    {
        KMimeType::Ptr mime = KMimeType::mimeType( mimeType );

        offers = KTrader::self()->query( mime->name(), "Type == 'Application'" );

        if ( offers.count() > 0 )
        {
            connect( openWithMenu, SIGNAL( activated( int ) ), this, SLOT( slotOpenWith( int ) )  );

            KTrader::OfferList::Iterator iter;
            KService::Ptr ptr;
            int index = 100;
            for( iter = offers.begin(); iter != offers.end(); ++iter )
            {
                ptr = *iter;
                openWithMenu->insertItem( ptr->pixmap( KIcon::Small ), ptr->name(), index++ );
            }
            haveItem = true;
        }
        else
        {
            addOpenWithMenu = false;
        }
    }
    if ( addOpenWithMenu )
        filesIconsMenu->insertItem( i18n( "Open With" ), openWithMenu, 1 );

    // The following menu items are enabled if 1 only file is selected
    if ( ivItems.count() == 1 )
    {
        KatIconViewItem* ivItem = static_cast<KatIconViewItem*>(ivItems.first());
        if ( ivItem->info.hasFullText )
        {
            if ( addOpenWithMenu )
                filesIconsMenu->insertSeparator();

            filesIconsMenu->insertItem( i18n( "Show Fulltext" ), this, SLOT( slotShowFullText() ), 0, 2 );
            filesIconsMenu->insertItem( i18n( "Show Words List" ), this, SLOT( slotShowWordsList() ), 0, 3 );
            filesIconsMenu->setItemEnabled ( 2, true );
            filesIconsMenu->setItemEnabled ( 3, true );
            haveItem = true;
        }
    }
    else
    {
        if ( addOpenWithMenu )
            filesIconsMenu->insertSeparator();

        filesIconsMenu->insertItem( i18n( "Show Fulltext" ), 2 );
        filesIconsMenu->insertItem( i18n( "Show Words List" ), 3 );
        filesIconsMenu->setItemEnabled ( 2, false );
        filesIconsMenu->setItemEnabled ( 3, false );
        haveItem = true;
    }
    if ( haveItem )
        filesIconsMenu->popup( point );
}

void DlgMainWindow::contextFileThumbnail( QIconViewItem* item, const QPoint& point ) {
    QPtrList<QIconViewItem> ivItems;
    bool haveItem = false;
    if ( getMainDockWidget() == dockMain )
    {
        for ( QIconViewItem *item = ivFilesThumbnails->firstItem(); item; item = item->nextItem() )
        {
            if ( item->isSelected() )
                ivItems.append( item );
        }
    }
    else
    {
        for ( QIconViewItem *item = ivResultsThumbnails->firstItem(); item; item = item->nextItem() )
        {
            if ( item->isSelected() )
                ivItems.append( item );
        }
    }

    // check if all lvItems have the same mimeType
    QString mimeType = QString::null;
    bool different = false;
    for ( QPtrList<QIconViewItem>::iterator it = ivItems.begin(); it != ivItems.end(); ++it )
    {
        KatIconViewItem* ivItem = static_cast<KatIconViewItem*>(*it);
        QString fileMimeType = ivItem->info.fileType;
        if ( ( !mimeType.isEmpty() ) && ( mimeType != fileMimeType ) )
        {
            different = true;
            break;
        }
        mimeType = fileMimeType;
    }

    filesThumbnailsMenu->clear();
    //FIXME: Mem leak ????
    QPopupMenu* openWithMenu = new QPopupMenu();

    bool addOpenWithMenu = true;
    if ( different )
    {
        openWithMenu->setEnabled( false );
    }
    else
    {
        KMimeType::Ptr mime = KMimeType::mimeType( mimeType );

        offers = KTrader::self()->query( mime->name(), "Type == 'Application'" );

        if ( offers.count() > 0 )
        {
            connect( openWithMenu, SIGNAL( activated( int ) ), this, SLOT( slotOpenWith( int ) )  );

            KTrader::OfferList::Iterator iter;
            KService::Ptr ptr;
            int index = 100;
            for( iter = offers.begin(); iter != offers.end(); ++iter )
            {
                ptr = *iter;
                openWithMenu->insertItem( ptr->pixmap( KIcon::Small ), ptr->name(), index++ );
            }
            haveItem = true;
        }
        else
        {
            addOpenWithMenu = false;
        }
    }
    if ( addOpenWithMenu )
        filesThumbnailsMenu->insertItem( i18n( "Open With" ), openWithMenu, 1 );

    // The following menu items are enabled if 1 only file is selected
    if ( ivItems.count() == 1 )
    {
        KatIconViewItem* ivItem = static_cast<KatIconViewItem*>(ivItems.first());
        if ( ivItem->info.hasFullText )
        {
            if ( addOpenWithMenu )
                filesThumbnailsMenu->insertSeparator();

            filesThumbnailsMenu->insertItem( i18n( "Show Fulltext" ), this, SLOT( slotShowFullText() ), 0, 2 );
            filesThumbnailsMenu->insertItem( i18n( "Show Words List" ), this, SLOT( slotShowWordsList() ), 0, 3 );
            filesThumbnailsMenu->setItemEnabled ( 2, true );
            filesThumbnailsMenu->setItemEnabled ( 3, true );
            haveItem = true;
        }
    }
    else
    {
        if ( addOpenWithMenu )
            filesThumbnailsMenu->insertSeparator();

        filesThumbnailsMenu->insertItem( i18n( "Show Fulltext" ), 2 );
        filesThumbnailsMenu->insertItem( i18n( "Show Words List" ), 3 );
        filesThumbnailsMenu->setItemEnabled ( 2, false );
        filesThumbnailsMenu->setItemEnabled ( 3, false );
        haveItem = true;
    }
    if ( haveItem )
        filesThumbnailsMenu->popup( point );
}

void DlgMainWindow::contextReqFile( QListViewItem* item, const QPoint& point, int i )
{
}

void DlgMainWindow::slotShowFullText()
{
    KatListViewItem* lvItem = 0L;
    KatIconViewItem* ivItem = 0L;

    if ( getMainDockWidget() == dockMain )
    {
        if ( !lvFiles->isHidden() )
            lvItem = static_cast<KatListViewItem*>(lvFiles->selectedItems().first());
        if ( !ivFiles->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivFiles->currentItem());
        if ( !ivFilesThumbnails->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivFilesThumbnails->currentItem());
    }
    else
    {
        if ( !lvResults->isHidden() )
            lvItem = static_cast<KatListViewItem*>(lvResults->selectedItems().first());
        if ( !ivResults->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivResults->currentItem());
        if ( !ivResultsThumbnails->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivResultsThumbnails->currentItem());
    }

    if ( lvItem )
    {
        DlgShowFullText sft( this, lvItem->info.fileId );
        sft.exec();
    }
    if ( ivItem )
    {
        DlgShowFullText sft( this, ivItem->info.fileId );
        sft.exec();
    }
}

void DlgMainWindow::slotShowWordsList()
{
    KatListViewItem* lvItem = 0L;
    KatIconViewItem* ivItem = 0L;

    if ( getMainDockWidget() == dockMain )
    {
        if ( !lvFiles->isHidden() )
            lvItem = static_cast<KatListViewItem*>(lvFiles->selectedItems().first());
        if ( !ivFiles->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivFiles->currentItem());
        if ( !ivFilesThumbnails->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivFilesThumbnails->currentItem());
    } else {
        if ( !lvResults->isHidden() )
            lvItem = static_cast<KatListViewItem*>(lvResults->selectedItems().first());
        if ( !ivResults->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivResults->currentItem());
        if ( !ivResultsThumbnails->isHidden() )
            ivItem = static_cast<KatIconViewItem*>(ivResultsThumbnails->currentItem());
    }

    if ( lvItem )
    {
        DlgShowWordsList swl( this, lvItem->info.fileId );
        swl.exec();
    }
    if ( ivItem )
    {
        DlgShowWordsList swl( this, ivItem->info.fileId );
        swl.exec();
    }
}

void DlgMainWindow::openItemSelected( KatListViewItem* item )
{
    if ( item )
    {
        if ( item->info.fileType != "KAT_CATALOG" )
            openFile( item->info );
    }
}

void DlgMainWindow::openItemSelected( KatIconViewItem* item )
{
    if ( item )
    {
        if ( item->info.fileType != "KAT_CATALOG" )
            openFile( item->info );
    }
}

void DlgMainWindow::doubleClickedFile( QListViewItem* item )
{
    KatListViewItem* lvItem = static_cast<KatListViewItem*>(item);
    openItemSelected( lvItem );
}

void DlgMainWindow::doubleClickedFileIcon( QIconViewItem* item )
{
    KatIconViewItem* lvItem = static_cast<KatIconViewItem*>(item);
    openItemSelected( lvItem );
}

void DlgMainWindow::doubleClickedFileThumbnail( QIconViewItem* item )
{
    KatIconViewItem* lvItem = static_cast<KatIconViewItem*>(item);
    openItemSelected( lvItem );
}

void DlgMainWindow::doubleClickedFolder( QListViewItem* item )
{
    if ( item )
    {
        if ( item->isOpen() )
            item->setOpen( false );
        else
            item->setOpen( true );
    }
}

void DlgMainWindow::rightButtonFile( QListViewItem*, const QPoint&, int )
{
}

void DlgMainWindow::rightButtonFolder( QListViewItem*, const QPoint&, int )
{
}

void DlgMainWindow::executeFile( QListViewItem* )
{
}

void DlgMainWindow::executeFolder( QListViewItem* )
{
}

void DlgMainWindow::slotFileFind()
{
    if ( getMainDockWidget() == dockMain )
    {
        makeDockInvisible( dockMain );

        setView( dockMainSearch );
        setMainDockWidget( dockMainSearch );

        makeDockVisible( dockMainSearch );
    }
    else
    {
        makeDockInvisible( dockMainSearch );

        setView( dockMain );
        setMainDockWidget( dockMain );

        makeDockVisible( dockMain );
    }
}

void DlgMainWindow::insertChildrenFolders( int catalogId, int parentId, QListViewItem* parentItem )
{
    QMap<int,QString> children;
    QString fileName;
    QString fileType;
    int fileId;

    // eliminate eventual children
    while( parentItem->firstChild() )
        delete parentItem->firstChild();

    // rebuild the list of children
    children = m_ke->getFileChildren( catalogId, parentId );

    QMap<int,QString>::const_iterator i;
    QMap<int,QString>::const_iterator end( children.end() );
    for ( i = children.begin(); i != end; ++i )
    {
        fileId = i.key();
        fileName = i.data();

        unsigned pos = fileName.findRev( '/', -1 );
        fileName = fileName.right( fileName.length() - pos - 1 );

        KatInformation info = m_ke->readFileInformation( fileId );

        if ( info.fileType == "inode/directory" )
        {
            KatListViewItem* lvItem = new KatListViewItem( parentItem );
            lvItem->setText( 0, fileName );
            lvItem->setPixmap( 0, pmFolderIconSmall );
            lvItem->setOpen( false );
            lvItem->info = info;

            // If the item has children, insert a fake child in order to show the + sign
            if ( m_ke->folderHasChildren( catalogId, fileId ) )
                KatListViewItem* lvFakeItem = new KatListViewItem( lvItem );
        }
    }
}

void DlgMainWindow::insertChildrenFiles( int catalogId, int parentId )
{
    QMap<int,QString> children;
    QString fileName;
    QString fullName;
    QString ext;
    int fileId;
    QString language;

    lvFiles->clear();
    ivFiles->clear();
    ivFilesThumbnails->clear();
    teProperties->setText( QString::null );

    children = m_ke->getFileChildren( catalogId, parentId );

    QMap<int,QString>::const_iterator i;
    QMap<int,QString>::const_iterator end( children.end() );
    for ( i = children.begin(); i != children.end(); ++i )
    {
        fileId = i.key();
        fullName = i.data();

        KatInformation info = m_ke->readFileInformation( fileId );

        language = info.language;

        int pos = fullName.findRev( '/', -1 );
        fileName = fullName.right( fullName.length() - pos - 1 );

        KMimeType::Ptr mime = KMimeType::mimeType( info.fileType );
        QString mimeType = info.fileType;

        pos = fileName.findRev( '.', -1 );
        if ( pos == -1 )
        {
            ext = QString::null;
        }
        else
        {
            ext = fileName.right( fileName.length() - pos - 1 );
            ext = ext.lower();
        }

        children.insert( i.key(), fileName  );

        QDateTime dt;
        dt.setTime_t( info.modificationDate );
        QDate date = dt.date();
        QTime time = dt.time();

        KatListViewItem *lvItem = new KatListViewItem( lvFiles );
        lvItem->setDragEnabled( true );
        lvItem->info = info;
        KatIconViewItem *ivItem = new KatIconViewItem( ivFiles );
        ivItem->info = info;
        KatIconViewItem *ivThmb = new KatIconViewItem( ivFilesThumbnails );
        ivThmb->info = info;

        if ( info.fileType == "inode/directory" )
        {
            const QPixmap& pmFolderIconSmall =
                KMimeType::mimeType( "inode/directory" )->pixmap( KIcon::Small );
            const QPixmap& pmFolderIconLarge =
                KMimeType::mimeType( "inode/directory" )->pixmap( KIcon::FirstGroup );

            lvItem->setText( FILE_NAME, fileName );
            lvItem->setText( FILE_EXTENSION, QString::null );
            lvItem->setText( FILE_MIMETYPE, QString::null );
            lvItem->setText( FILE_SIZE, QString::null );
            lvItem->setText( FILE_DATE, date.toString( Qt::LocalDate ) );
            lvItem->setText( FILE_TIME, time.toString( Qt::TextDate ) );
            lvItem->setText( FILE_THUMBNAIL, QString::null );
            lvItem->setText( FILE_METADATA, QString::null );
            lvItem->setText( FILE_FULLTEXT, QString::null );
            lvItem->setText( FILE_LANGUAGE, QString::null );

            lvItem->setPixmap( 0, pmFolderIconSmall );

            ivItem->setText( fileName );
            ivItem->setPixmap( pmFolderIconLarge );

            ivThmb->setText( fileName );
            ivThmb->setPixmap( pmFolderIconLarge );

        }
        else
        {
            const QPixmap& pmFileIconSmall = mime->pixmap( KIcon::Small );
            const QPixmap& pmFileIconLarge = mime->pixmap( KIcon::FirstGroup );

            QString size = QString::number( info.fileSize );
            addSeparators( size );

            lvItem->setText( FILE_NAME, fileName );
            lvItem->setText( FILE_EXTENSION, ext );
            lvItem->setText( FILE_MIMETYPE, mimeType );
            lvItem->setText( FILE_SIZE, size );
            lvItem->setText( FILE_DATE, date.toString( Qt::LocalDate ) );
            lvItem->setText( FILE_TIME, time.toString( Qt::TextDate ) );
            lvItem->setText( FILE_THUMBNAIL, QString::null );
            lvItem->setText( FILE_METADATA, QString::null );
            lvItem->setText( FILE_FULLTEXT, QString::null );
            lvItem->setText( FILE_LANGUAGE, language );

            lvItem->setPixmap( 0, pmFileIconSmall );

            if ( info.hasThumbnail )
                lvItem->setPixmap( FILE_THUMBNAIL, pmCheckIconSmall );
            if ( info.hasMetaData )
                lvItem->setPixmap( FILE_METADATA, pmCheckIconSmall );
            if ( info.hasFullText )
                lvItem->setPixmap( FILE_FULLTEXT, pmCheckIconSmall );

            ivItem->setText( fileName );
            ivItem->setPixmap( pmFileIconLarge );

            if ( info.thumbnailDataLength == 0 )
            {
                ivThmb->setText( fileName );
                ivThmb->setPixmap( pmFileIconLarge );
            }
            else
            {
                QPixmap qpthumb( info.thumbnailData );

                ivThmb->setText( fileName );
                ivThmb->setPixmap( qpthumb );
            }
        }
    }
}

QString buildPropertyHTML( const QString &field, const QString &value )
{
    QString p = QString::null;
    p = p + "<tr>";
    p = p + " <td><b>" + field + "</b></td>";
    p = p + " <td>" + value + "</td>";
    p = p + "</tr>";
    return p;
}

void DlgMainWindow::showFileProperties( KatInformation info )
{
    if ( info.fileId == -2L )
    { // Catalog
        QString p = QString::null;
        p = p + "<table>";
        p = p + buildPropertyHTML( i18n( "Description:" ), m_cat->description() );
        p = p + buildPropertyHTML( i18n( "Notes:" ), m_cat->notes() );
        p = p + "</table>";

        teProperties->setText( p );
    }
    else
    {
        QDateTime mdt;
        mdt.setTime_t( info.modificationDate );
        QDate mdate = mdt.date();
        QTime mtime = mdt.time();

        QString modificationdate = mdate.toString( Qt::LocalDate ) + " " + mtime.toString( Qt::TextDate );

        QString size = QString::number( info.fileSize );
        addSeparators( size );

        QString p = QString::null;
        p = p + "<table>";

        p = p + buildPropertyHTML( "File type:", info.fileType );
        p = p + buildPropertyHTML( "File size:", size );
        p = p + buildPropertyHTML( "Modification date:", modificationdate );

        if ( info.metaData != QString::null )
        {
            QString name;
            //FIXME: type not used
            QString type;
            QString value;

            QStringList tokens = QStringList::split( "|", info.metaData, false );

            for ( QStringList::Iterator it = tokens.begin(); it != tokens.end(); ++it )
            {
                name = *it;
                ++it;
                type = *it;
                ++it;
                value = *it;

                p = p + buildPropertyHTML( name+':', value );
            }
        }

        p = p + "</table>";

        teProperties->setText( p );
    }
}

QStringList DlgMainWindow::listOfCatalog()
{
    QStringList tmp;

    QPtrList<KatCatalog>::ConstIterator end( m_catalogs.end() );
    for ( QPtrList<KatCatalog>::ConstIterator it = m_catalogs.begin(); it != end; ++it )
    {
        KatCatalog* cat = *it;
        tmp << cat->name();
    }
    return tmp;
}

void DlgMainWindow::populateViews()
{
    lvFolders->clear();
    lvFiles->clear();
    ivFiles->clear();
    ivFilesThumbnails->clear();

    //m_catalogs = m_ke->readCatalogs();
    QPtrList<KatCatalog>::iterator end( m_catalogs.end() );
    for ( QPtrList<KatCatalog>::iterator it = m_catalogs.begin(); it != end; ++it )
    {
        KatCatalog* cat = *it;

        KatInformation info;
        info.fileType = "KAT_CATALOG";
        info.fileId = -2;
        info.parentId = 0;
        info.catalogId = cat->catalogId();

        KatListViewItem* lvItem = new KatListViewItem( lvFolders );
        lvItem->setText( 0, cat->name() );
        lvItem->setPixmap( 0, pmCatalogIcon );
        lvItem->info = info;
        lvItem->setOpen( false );

        KatInformation info2;
        info2.fileType = "inode/directory";
        info2.fileId = -1;
        info2.parentId = cat->catalogId();
        info2.catalogId = cat->catalogId();

        KatListViewItem* rootItem = new KatListViewItem( lvItem );
        rootItem->setText( 0, cat->path() );
        rootItem->setPixmap( 0, pmDiscIcon );
        rootItem->info = info2;
        rootItem->setOpen( false );

        insertChildrenFolders( cat->catalogId(), -1, rootItem );
    }
}

void DlgMainWindow::populateSearchViews( QValueList<int> results )
{
    QString fileName;
    QString fullName;
    QString ext;
    int pos = 0;
    QString language;

    lvResults->clear();

    QValueList<int>::const_iterator i;
    QValueList<int>::const_iterator end( results.end() );
    for ( i = results.begin(); i != end; ++i )
    {
        int fileid = *i;

        KatInformation info = m_ke->readFileInformation( fileid );

        fullName = info.fullName;
        fileName = info.fileName;
        language = info.language;

        KMimeType::Ptr mime = KMimeType::mimeType( info.fileType );
        QString mimeType = info.fileType;

        pos = fileName.findRev( '.', -1 );
        if ( pos == -1 )
        {
            ext = QString::null;
        }
        else
        {
            ext = fileName.right( fileName.length() - pos - 1 );
            ext = ext.lower();
        }

        QDateTime dt;
        dt.setTime_t( info.modificationDate );
        QDate date = dt.date();
        QTime time = dt.time();

        KatListViewItem* lvItem = new KatListViewItem( lvResults );
        lvItem->info = info;
        lvItem->setDragEnabled( true );
        KatIconViewItem* ivItem = new KatIconViewItem( ivResults );
        ivItem->info = info;
        KatIconViewItem* ivThmb = new KatIconViewItem( ivResultsThumbnails );
        ivThmb->info = info;

        if ( info.fileType == "inode/directory" )
        {
            const QPixmap& pmFolderIconSmall =
                KMimeType::mimeType( "inode/directory" )->pixmap( KIcon::Small );
            const QPixmap& pmFolderIconLarge =
                KMimeType::mimeType( "inode/directory" )->pixmap( KIcon::FirstGroup );

            lvItem->setText( FILE_NAME, fileName );
            lvItem->setText( FILE_EXTENSION, QString::null );
            lvItem->setText( FILE_MIMETYPE, QString::null );
            lvItem->setText( FILE_SIZE, QString::null );
            lvItem->setText( FILE_DATE, date.toString( Qt::LocalDate ) );
            lvItem->setText( FILE_TIME, time.toString( Qt::TextDate ) );
            lvItem->setText( FILE_THUMBNAIL, QString::null );
            lvItem->setText( FILE_METADATA, QString::null );
            lvItem->setText( FILE_FULLTEXT, QString::null );
            lvItem->setText( FILE_LANGUAGE, QString::null );

            lvItem->setPixmap( 0, pmFolderIconSmall );

        }
        else
        {
            const QPixmap& pmFileIconSmall = mime->pixmap( KIcon::Small );
            const QPixmap& pmFileIconLarge = mime->pixmap( KIcon::FirstGroup );

            QString size = QString::number( info.fileSize );

            QString thousandsSeparator = KGlobal::locale()->thousandsSeparator();
            int i = size.length()-1;
            for( int j=1; i>0; i--, j++ )
            {
                if ( j%3 == 0 )
                    size.insert(i, thousandsSeparator);
            }

            lvItem->setText( FILE_NAME, fileName );
            lvItem->setText( FILE_EXTENSION, ext );
            lvItem->setText( FILE_MIMETYPE, mimeType );
            lvItem->setText( FILE_SIZE, size );
            lvItem->setText( FILE_DATE, date.toString( Qt::LocalDate ) );
            lvItem->setText( FILE_TIME, time.toString( Qt::TextDate ) );
            lvItem->setText( FILE_THUMBNAIL, QString::null );
            lvItem->setText( FILE_METADATA, QString::null );
            lvItem->setText( FILE_FULLTEXT, QString::null );
            lvItem->setText( FILE_LANGUAGE, language );

            lvItem->setPixmap( 0, pmFileIconSmall );

            if ( info.hasThumbnail )
                lvItem->setPixmap( FILE_THUMBNAIL, pmCheckIconSmall );
            if ( info.hasMetaData )
                lvItem->setPixmap( FILE_METADATA, pmCheckIconSmall );
            if ( info.hasFullText )
                lvItem->setPixmap( FILE_FULLTEXT, pmCheckIconSmall );

            ivItem->setText( fileName );
            ivItem->setPixmap( pmFileIconLarge );

            if ( info.thumbnailDataLength == 0 )
            {
                ivThmb->setText( fileName );
                ivThmb->setPixmap( pmFileIconLarge );

            }
            else
            {
                QPixmap qpthumb( info.thumbnailData );

                ivThmb->setText( fileName );
                ivThmb->setPixmap( qpthumb );
            }
        }
    }
}

void DlgMainWindow::openFile( KatInformation info )
{
    kdDebug()<<" void DlgMainWindow::openFile( KatInformation info )*****************************\n";
    int fileId = info.fileId;
    int catalogId = info.catalogId;
    QString fullname = info.fullName;

    if ( info.fileType == "inode/directory" )
    {
        KatListViewItem* selFolder = static_cast<KatListViewItem*>(lvFolders->selectedItem());
        if ( selFolder )
        {
            selFolder->setOpen( true );

            KatListViewItem* child = static_cast<KatListViewItem*>(selFolder->firstChild());
            while ( child != 0 )
            {
                if ( child->info.fileId == fileId )
                {
                    lvFolders->setSelected( child, true );
                    child->setOpen( true );
                    break;
                }
                child = static_cast<KatListViewItem*>(child->nextSibling());
            }
            insertChildrenFiles( catalogId, fileId );
        }
        else //search mode => selFolder is null
            executeItem( info );
    }
    else
        executeItem( info );
}

void DlgMainWindow::executeItem( KatInformation info )
{
    KURL kurl( info.fullName );
    KMimeType::Ptr mime = KMimeType::mimeType( info.fileType );
    QString mimeType = info.fileType;

    KService::Ptr offer = KServiceTypeProfile::preferredService( mimeType, QString( "Application" ) );

    if ( offer )
    {
        struct stat my_stat;
        if ( stat( info.fullName, &my_stat ) )
            KMessageBox::error( this, i18n( "The file is currently unavailale. "
                                            "Please mount the filesystem in which the file resides and try again." ) );
            else
                KRun::runURL ( kurl, mimeType );
    }
    else
    {
        KMessageBox::error( this, i18n( "Unable to open files of type: %1" ).arg( mimeType ) );
    }
}


#include "dlgmainwindow.moc"
