#ifndef KADU_USERLISTELEMENT_H
#define KADU_USERLISTELEMENT_H

#include <qhostaddress.h>
#include <qobject.h>
#include <qstring.h>
#include <qvariant.h>

#include "status.h"

class ULEPrivate;
class UserGroup;

typedef long int UserListKey;

/**
	Typ opisujcy stan powiadamiania.
	\enum NotifyType
	\brief Typ opisujcy stan powiadamiania.
**/
enum NotifyType {
	NotifyGlobal = 0, /*!< powiadomienia zalene od globalnych ustawie */
	NotifyOff = 1, /*!< powiadomienia wyczone */
	NotifyOwn = 2 /*!< indywidualne ustawienie powiadamiania */
};

/**
	Klasa reprezentujca kontakt (uytkownika na licie kontaktw).
	\class UserListElement
	\brief Klasa reprezentujca kontakt.
	Ze zwgldu na siln hermetyzacj, naley unika dawania dostpu przez wskaniki / referencje
	(nawet const) na zewntrz do wewntrznych danych.

	Klasa zostaa zbudowana w taki sposb, e przekazywanie przez warto jest
	bardzo tanie (adne pola nie s kopiowane). Jednoczenie nie mona wykona
	niezalenej (nie powizanej ze rdem) kopii obiektu, gdy mogoby to
	doprowadzi do stworzenia kilku kontaktw o tych samych danych (czego naley
	si wystrzega!).
**/
class UserListElement : public QObject
{
	Q_OBJECT
	public:
		/**
			\fn UserListElement()
		**/
		UserListElement();

		/**
			\fn UserListElement(const UserListElement &copyMe)
			Konstruktor kopiujcy.
		**/
		UserListElement(const UserListElement &copyMe);

		/**
			\fn virtual ~UserListElement()
			Zwalnia pami zwizan z kontaktem, gdy licznik odwoa == 1
		**/
		virtual ~UserListElement();

		/**
			\fn void operator = (const UserListElement &copyMe)
			Operator kopiujcy. Dane bd dzielone.
			\param copyMe obiekt klasy UserListElement ktry bdzie skopiowany.
		**/
		UserListElement & operator = (const UserListElement &copyMe);

		inline bool operator == (const UserListElement &u) const { return key() == u.key(); }

		inline bool operator != (const UserListElement &u) const { return key() != u.key(); }

		inline bool operator < (const UserListElement &u) const { return key()  < u.key(); }

		/**
			\fn QString ID(const QString &protocolName) const
			\param protocolName identyfikator protokou
			Zwraca identyfikator kontaktu we wskazanym protokole.
		**/
		QString ID(const QString &protocolName) const;

		/**
			\fn const UserStatus &status(const QString &protocolName) const
			\param protocolName identyfikator protokou
			Zwraca status kontaktu w protokole protocolName.
		**/
		const UserStatus &status(const QString &protocolName) const;

		/**
			\fn QVariant data(const QString &name) const
			\param name nazwa wasnoci
			Zwraca wasno (ktra nie jest zwizana z adnym protokoem)
		**/
		QVariant data(const QString &name) const;

		/**
			\fn QVariant protocolData(const QString &protocolName, const QString &name) const
			\param protocolName identyfikator protokou
			\param name nazwa wanoci
			Zwraca wasno name dla protokou protocolName.
		**/
		QVariant protocolData(const QString &protocolName, const QString &name) const;

		/**
			\fn QStringList protocolList() const
			Zwraca list identyfikatorw protokow dla danego kontaktu.
		**/
		QStringList protocolList() const;

		/**
			\fn QStringList protocolDataKeys(const QString &protocolName) const
			Zwraca list identyfikatorw pl dla protokou protocolName.
		**/
		QStringList protocolDataKeys(const QString &protocolName) const;

		/**
			\fn QStringList nonProtocolDataKeys() const
			Zwraca list identyfikatorw pl nie zwizanych z adnym protokoem.
		**/
		QStringList nonProtocolDataKeys() const;

		/**
			\fn QString firstName() const
			Zwraca imi kontaktu.
		**/
		QString firstName() const;

		/**
			\fn QString lastName() const
			Zwraca nazwisko kontaktu.
		**/
		QString lastName() const;

		/**
			\fn QString nickName() const
			Zwraca pseudonim kontaktu.
		**/
		QString nickName() const;

		/**
			\fn QString altNick() const
			Zwraca wywietlany pseudonim kontaktu.
		**/
		QString altNick() const;

		/**
			\fn QString mobile() const
			Zwraca numer telefonu kontaktu.
		**/
		QString mobile() const;

		/**
			\fn QString email() const
			Zwraca adres E-Mail kontaktu.
		**/
		QString email() const;

		/**
			\fn QString homePhone() const
			Zwraca numer telefonu domowego kontaktu.
		**/
		QString homePhone() const;

		/**
			\fn QString aliveSound(NotifyType &type) const
			Zwraca sposb powiadamiania dwikiem o zmianach statusu kontaktu.
		**/
		QString aliveSound(NotifyType &type) const;

		/**
			\fn QString messageSound(NotifyType &type) const
			Zwraca sposb powiadamiania dwikiem o wiadomociach od kontaktu.
		**/
		QString messageSound(NotifyType &type) const;

		/**
			\fn bool isAnonymous() const
			Zwraca informacj, czy kontakt jest anonimowy czy nie.
		**/
		bool isAnonymous() const;

		/**
			\fn bool notify() const
			Zwraca informacj, czy uytkownik jest powiadamiany o kontakcie czy nie.
		**/
		bool notify() const;

		/**
			\fn bool usesProtocol(const QString &name) const
			\param name identyfikator protokou
			Zwraca informacj o tym czy do kontaktu przypisany jest protok o wskazanym identyfikatorze.
		**/
		bool usesProtocol(const QString &name) const;

		/**
			\fn bool hasIPAddress(const QString &protocolName) const
			\param protocolName identyfikator protokou
			Zwraca informacj czy znany jest adres IP we wskazanym protokole.
		**/
		bool hasIP(const QString &protocolName) const;

		/**
			\fn QHostAddress IP(const QString &protocolName) const
			\param protocolName identyfikator protokou
			Zwraca adres IP kontaktu.
		**/
		QHostAddress IP(const QString &protocolName) const;

		/**
			\fn QString DNSName(const QString &protocolName) const
			\param protocolName identyfikator protokou
			Zwraca nazw domeny kontaktu (jeli znaleziona).
		**/
		QString DNSName(const QString &protocolName) const;

		/**
			\fn short port(const QString &protocolName) const
			\param protocolName identyfikator protokou
			Zwraca numer portu kontaktu.
		**/
		short port(const QString &protocolName) const;

	public slots:
		/**
			\fn QVariant setData(const QString &name, const QVariant &val, bool massively = false, bool last = false)
			ustawia wasno kontaktu stowarzyszon z nazw "name"
			zwraca star warto
			\param name nazwa wasnoci
			\param val nowa warto
			\param massively true, gdy jest to cz wikszych zmian
			\param last true, gdy massively == true i jest to ostatnia zmiana

			pola bezprotokoowe:

				QStringList Groups;
				QString FirstName; Imi kontaktu.
				QString LastName; Nazwisko kontaktu.
				QString NickName; Pseudonim kontaktu.
				QString AltNick; Pseudonim kontaktu, ktry jest wywietlany na licie.
				QString Mobile; Numer telefonu kontaktu.
				QString Email; E-Mail kontaktu.
				bool Anonymous; Informuje, czy kontakt jest anonimowy czy nie.
				bool Notify; Informuje czy mamy wczone powiadamianie o kontakcie.

				NotifyType AliveSound; Przechowuje informacj o sposobie powiadamiania
										o zmianie statusu kontaku dwikiem.
				QString OwnAliveSound; Jeli sposb powiadamiania o zmianie statusu kontaktu ma warto OWN,
									to ta zmienna przechowuje nazw pliku dwikowego do odtworzenia.

				NotifyType MessageSound; Przechowuje informacj o sposobie powiadamiania
										o nowej wiadomoci od kontaktu dwikiem.
				QString OwnMessageSound;  Jeli sposb powiadamiania o nowej wiadomoci od kontaktu ma warto OWN,
										to ta zmienna przechowuje nazw pliku dwikowego do odtworzenia.

				QString HomePhone; Numer telefonu domowego kontaktu.
		**/
		QVariant setData(const QString &name, const QVariant &val, bool massively = false, bool last = false);

		/**
			\fn QVariant setProtocolData(const QString &protocolName, const QString &name, const QVariant &val, bool massively = false, bool last = false)
			\param protocolName identyfikator protokou
			\param name nazwa wasnoci
			\param val nowa warto
			\param massively true, gdy jest to cz wikszych zmian
			\param last true, gdy massively == true i jest to ostatnia zmiana
			\return stara warto
			Ustawia wasno name zwizan z protokoem protocolName na val.

			dla protokou "Gadu" dostpne s nastpujce pola:

				int MaxImageSize; Maksymalny rozmiar obrazka, jak moe przyj kontakt.
				QHostAddress IP; Adres IP kontaktu (jeli wykryty).
				QString DNSName; Nazwa domenu kontaktu (jeli znaleziona).
				short Port; Port kontaktu (jeli wykryty).
				int Version; Wersja protokou uywanego przez kontakt.
				bool Blocking; Informuje czy blokujemy kontakt, czy nie.
				bool OfflineTo; Informuje czy mamy wczony tryb "niedostpny dla kontaktu" dla tego kontaktu.
		**/
		QVariant setProtocolData(const QString &protocolName, const QString &name, const QVariant &val, bool massively = false, bool last = false);

		/**
			\fn void addProtocol(const QString &protocolName, const QString &id, bool massively = false, bool last = false)
			\param protocolName identyfikator protokou
			\param id identyfikator w tym protokole w postaci napisu
			\param massively true, gdy jest to cz wikszych zmian
			\param last true, gdy massively == true i jest to ostatnia zmiana
			Dodaje do kontaktu informacj o uywanym protokole.
		**/
		void addProtocol(const QString &protocolName, const QString &id, bool massively = false, bool last = false);

		/**
			\fn void deleteProtocol(const QString &protocolName, bool massively = false, bool last = false)
			\param protocolName identyfikator protokou
			\param massively true, gdy jest to cz wikszych zmian
			\param last true, gdy massively == true i jest to ostatnia zmiana
			Usuwa informacje o protokole.
		**/
		void deleteProtocol(const QString &protocolName, bool massively = false, bool last = false);

		/**
			\fn void setStatus(const QString &protocolName, const UserStatus &status, bool massively = false, bool last = false)
			\param protocolName identyfikator protokou
			\param status nowy status
			\param massively true, gdy jest to cz wikszych zmian
			\param last true, gdy massively == true i jest to ostatnia zmiana
			Zmienia status kontaktu w protokolej protocolName.
		**/
		void setStatus(const QString &protocolName, const UserStatus &status, bool massively = false, bool last = false);

		/**
			\fn void setFirstName(const QString &firstName)
			Ustawia imi dla kontaktu.
			\param firstName imi, ktre zostanie przydzielone kontaktowi.
		**/
		void setFirstName(const QString &firstName);

		/**
			\fn void setLastName(const QString &lastName)
			Ustawia nazwisko dla kontaktu.
			\param lastName nazwisko, ktre zostanie przydzielone kontaktowi.
		**/
		void setLastName(const QString &lastName);

		/**
			\fn void setNickName(const QString &nickName)
			Ustawia pseudonim dla kontaktu.
			\param nickName pseudonim, ktry zostanie przydzielony kontaktowi.
		**/
		void setNickName(const QString &nickName);

		/**
			\fn void setAltNick(const QString &altNick)
			Ustawia wywietlany pseudonim dla kontaktu.
			\param altNick wywietlany pseudonim, ktry zostanie przydzielony kontaktowi.
		**/
		void setAltNick(const QString &altNick);

		/**
			\fn void setMobile(const QString &mobile)
			Ustawia numer telefonu dla kontaktu.
			\param mobile numer telefonu, ktry zostanie przydzielony kontaktowi.
		**/
		void setMobile(const QString &mobile);

		/**
			\fn void setEmail(const QString &email)
			Ustawia adres E-Mail dla kontaktu.
			\param email adres, ktry zostanie przydzielony kontaktowi.
		**/
		void setEmail(const QString &email);

		/**
			\fn void setAnonymous(const bool anonymous)
			Ustawia stan anonimowoci kontaktu.
			\param anonymous warto logiczna informujca, e kontakt jest anonimowy, lub nie.
		**/
		void setAnonymous(const bool &anonymous);

		/**
			\fn void setNotify(const bool notify)
			Ustawia stan powiadamiania o kontakcie.
			\param notify warto logiczna informujca, czy uytkownik ma by powiadamiany o kontakcie.
		**/
		void setNotify(const bool &notify);

		/**
			\fn void setHomePhone(const QString &phone)
			Ustawia numer telefonu domowego dla kontaktu.
			\param phone numer telefonu, ktry zostanie przydzielony kontaktowi.
		**/
		void setHomePhone(const QString &phone);

		/**
			\fn void setAliveSound(NotifyType type, const QString &file = QString::null)
			Ustawia sposb powiadamiania dwikiem o zmianie statusu przez kontakt.
			\param type sposb powiadamiania.
			\arg \c GLOBAL powiadomienia zalene od globalnych ustawie.
			\arg \c OFF powiadomienia wyczone.
			\arg \c OWN indywidualne ustawienie powiadamiania/

			\param file plik dwikowy, wymagany dla indywidualnego ustawienia powiadomie.
		**/
		void setAliveSound(NotifyType type, const QString &file = QString::null);

		/**
			\fn void setMessageSound(NotifyType type, const QString &file = QString::null)
			Ustawia sposb powiadamiania dwikiem o nowej wiadomoci od kontaktu.
			\param type sposb powiadamiania.
			\arg \c GLOBAL powiadomienia zalene od globalnych ustawie.
			\arg \c OFF powiadomienia wyczone.
			\arg \c OWN indywidualne ustawienie powiadamiania/

			\param file plik dwikowy, wymagany dla indywidualnego ustawienia powiadomie.
		**/
		void setMessageSound(NotifyType type, const QString &file = QString::null);

		/**
			\fn void setAddressAndPort(const QString &protocolName, const QHostAddress &ip, short port)
			\param protocolName identyfikator protokou
			\param ip adres IP
			\param port port
			Ustawia adres IP i port dla wskazanego protokou.
		**/
		void setAddressAndPort(const QString &protocolName, const QHostAddress &ip, short port);

		/**
			\fn void setDNSName(const QString &protocolName, const QString &dnsname)
			\param protocolName identyfikator protokou
			\param dnsname nowa domena
			Ustawia domen dla wskazanego protokou.
		**/
		void setDNSName(const QString &protocolName, const QString &dnsname);

		/**
			\fn void refreshDNSName(const QString &protocolName)
			\param protocolName identyfikator protokou
			Wywouje zapytanie o nazw domeny dla kontaktu oraz wypenia odpowiednie pole kontaktu,
			gdy domena zostanie odnaleziona.
		**/
		void refreshDNSName(const QString &protocolName);

	protected:
		/**
			\fn UserListKey key() const
			Zwraca klucz po ktrym jest identyfikowany kontakt
		**/
		UserListKey key() const;

	private:
		ULEPrivate *privateData;
		static unsigned long int used;
		friend class UserGroup;
};

#endif
