// K-3D
// Copyright (c) 1995-2005, Timothy M. Shead
//
// Contact: tshead@k-3d.com
//
// This program is free software; you can redistribute it and/or
// modify it under the terms of the GNU General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
//
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
//
// You should have received a copy of the GNU General Public
// License along with this program; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA

/** \file
		\author Tim Shead (tshead@k-3d.com)
*/

#include "bitmap.h"
#include "color.h"
#include "i3d_2d_mapping.h"
#include "iaqsis.h"
#include "ibitmap_read_format.h"
#include "ibitmap_sink.h"
#include "ibitmap_source.h"
#include "ibitmap_write_format.h"
#include "icamera.h"
#include "icamera_animation_render_engine.h"
#include "icamera_preview_render_engine.h"
#include "icamera_still_render_engine.h"
#include "icolor_source.h"
#include "idocument_read_format.h"
#include "idocument_write_format.h"
#include "ifile_format.h"
#include "igl.h"
#include "ilong_source.h"
#include "imaterial.h"
#include "imesh_sink.h"
#include "imesh_source.h"
#include "imesh_storage.h"
#include "ipreview_render_engine.h"
#include "irenderman.h"
#include "irenderman.h"
#include "iscalar_source.h"
#include "iscript_engine.h"
#include "istill_render_engine.h"
#include "istring_source.h"
#include "itime_sink.h"
#include "itransform_array_1d.h"
#include "itransform_array_2d.h"
#include "itransform_array_3d.h"
#include "itransform_sink.h"
#include "itransform_source.h"
#include "iuri.h"
#include "ivector3_source.h"
#include "iyafray.h"
#include "log.h"
#include "mesh.h"
#include "mesh_selection.h"
#include "types.h"

#include <map>

#if defined(__GNUC__) &&  __GNUC__ >= 3

	#define K3D_HAVE_GCC_DEMANGLE

	// http://lists.debian.org/debian-gcc/2003/09/msg00055.html notes
	// that, in cxxabi.h of gcc-3.x for x < 4, this type is used before it
	// is declared.

	#if __GNUC__ == 3 && __GNUC_MINOR__ < 4
		class __class_type_info;
	#endif

	#include <cxxabi.h>
#endif

namespace k3d
{

namespace detail
{

/// "Wraps" a std::type_info object so it can be used as the key in a sorted container
struct type_info
{
	type_info(const std::type_info& Info) :
		info(Info)
	{
	}

	bool operator<(const type_info& RHS) const
	{
		return info.before(RHS.info);
	}

	const std::type_info& info;
};

/// Defines storage for a mapping of std::type_info objects to their corresponding string representation
typedef std::map<type_info, std::string> type_to_name_map_t;
/// Stores a mapping of std::type_info objects to their corresponding string representation
type_to_name_map_t type_to_name_map;

/// Defines storage for a mapping of string to the corresponding std::type_info
typedef std::map<std::string, type_info> name_to_type_map_t;
/// Stores a mapping of string to the corresponding std::type_info
name_to_type_map_t name_to_type_map;

void register_type(const std::type_info& Info, const std::string& Name)
{
	if(type_to_name_map.count(type_info(Info)) || name_to_type_map.count(Name))
	{
		k3d::log() << error << k3d_file_reference << ": attempt to register [" << Name << "] twice" << std::endl;
		return;
	}

	type_to_name_map.insert(std::make_pair(type_info(Info), Name));
	name_to_type_map.insert(std::make_pair(Name, type_info(Info)));
}

void initialize_types()
{
	static bool initialized = false;
	if(initialized)
		return;

	register_type(typeid(bool), "bool");
	register_type(typeid(double), "double");
	register_type(typeid(float), "float");
	register_type(typeid(int), "int");
	register_type(typeid(k3d::angle_axis), "k3d::angle_axis");
	register_type(typeid(k3d::aqsis::idisplacement_layer), "k3d::aqsis::idisplacement_layer");
	register_type(typeid(k3d::aqsis::isurface_layer), "k3d::aqsis::isurface_layer");
	register_type(typeid(k3d::bitmap), "k3d::bitmap");
	register_type(typeid(k3d::bitmap*), "k3d::bitmap*");
	register_type(typeid(k3d::bounding_box3), "k3d::bounding_box3");
	register_type(typeid(k3d::color), "k3d::color");
	register_type(typeid(k3d::filesystem::path), "k3d::filesystem::path");
	register_type(typeid(k3d::gl::ilight), "k3d::gl::ilight");
	register_type(typeid(k3d::gl::imaterial), "k3d::gl::imaterial");
	register_type(typeid(k3d::i3d_2d_mapping), "k3d::i3d_2d_mapping");
	register_type(typeid(k3d::ibitmap_read_format), "k3d::ibitmap_read_format");
	register_type(typeid(k3d::ibitmap_sink), "k3d::ibitmap_sink");
	register_type(typeid(k3d::ibitmap_source), "k3d::ibitmap_source");
	register_type(typeid(k3d::ibitmap_write_format), "k3d::ibitmap_write_format");
	register_type(typeid(k3d::icamera), "k3d::icamera");
	register_type(typeid(k3d::icamera_animation_render_engine), "k3d::icamera_animation_render_engine");
	register_type(typeid(k3d::icamera_preview_render_engine), "k3d::icamera_preview_render_engine");
	register_type(typeid(k3d::icamera_still_render_engine), "k3d::icamera_still_render_engine");
	register_type(typeid(k3d::icolor_source), "k3d::icolor_source");
	register_type(typeid(k3d::idocument_read_format), "k3d::idocument_read_format");
	register_type(typeid(k3d::idocument_write_format), "k3d::idocument_write_format");
	register_type(typeid(k3d::ifile_format), "k3d::ifile_format");
	register_type(typeid(k3d::ilong_source), "k3d::ilong_source");
	register_type(typeid(k3d::imaterial), "k3d::imaterial");
	register_type(typeid(k3d::imesh_sink), "k3d::imesh_sink");
	register_type(typeid(k3d::imesh_source), "k3d::imesh_source");
	register_type(typeid(k3d::imesh_storage), "k3d::imesh_storage");
	register_type(typeid(k3d::inode), "k3d::inode");
	register_type(typeid(k3d::inode*), "k3d::inode*");
	register_type(typeid(k3d::ipreview_render_engine), "k3d::ipreview_render_engine");
	register_type(typeid(k3d::iscalar_source), "k3d::iscalar_source");
	register_type(typeid(k3d::iscript_engine), "k3d::iscript_engine");
	register_type(typeid(k3d::istill_render_engine), "k3d::istill_render_engine");
	register_type(typeid(k3d::istring_source), "k3d::istring_source");
	register_type(typeid(k3d::itime_sink), "k3d::itime_sink");
	register_type(typeid(k3d::itransform_array_1d), "k3d::itransform_array_1d");
	register_type(typeid(k3d::itransform_array_2d), "k3d::itransform_array_2d");
	register_type(typeid(k3d::itransform_array_3d), "k3d::itransform_array_3d");
	register_type(typeid(k3d::itransform_sink), "k3d::itransform_sink");
	register_type(typeid(k3d::itransform_source), "k3d::itransform_source");
	register_type(typeid(k3d::iuri), "k3d::iuri");
	register_type(typeid(k3d::ivector3_source), "k3d::ivector3_source");
	register_type(typeid(k3d::matrix4), "k3d::matrix4");
	register_type(typeid(k3d::mesh), "k3d::mesh");
	register_type(typeid(k3d::mesh*), "k3d::mesh*");
	register_type(typeid(k3d::mesh_selection), "k3d::mesh_selection");
	register_type(typeid(k3d::normal3), "k3d::normal3");
	register_type(typeid(k3d::point3), "k3d::point3");
	register_type(typeid(k3d::point4), "k3d::point4");
	register_type(typeid(k3d::ri::idisplacement_shader), "k3d::ri::idisplacement_shader");
	register_type(typeid(k3d::ri::iimager_shader), "k3d::ri::iimager_shader");
	register_type(typeid(k3d::ri::ilight), "k3d::ri::ilight");
	register_type(typeid(k3d::ri::ilight_shader), "k3d::ri::ilight_shader");
	register_type(typeid(k3d::ri::imaterial), "k3d::ri::imaterial");
	register_type(typeid(k3d::ri::isurface_shader), "k3d::ri::isurface_shader");
	register_type(typeid(k3d::ri::itexture), "k3d::ri::itexture");
	register_type(typeid(k3d::ri::itexture*), "k3d::ri::itexture*");
	register_type(typeid(k3d::ri::ivolume_shader), "k3d::ri::ivolume_shader");
	register_type(typeid(k3d::vector3), "k3d::vector3");
	register_type(typeid(k3d::yafray::ilight), "k3d::yafray::ilight");
	register_type(typeid(k3d::yafray::imaterial), "k3d::yafray::imaterial");
	register_type(typeid(long), "long");
	register_type(typeid(std::string), "std::string");
	register_type(typeid(unsigned long), "unsigned long");

	/** \todo Come up with a more explicit type for the tweaks filter */
	register_type(typeid(std::vector<k3d::point3>), "std::vector<k3d::point3>");

	initialized = true;
}

const std::string demangle(const std::string& Type)
{
	std::string result = Type;

#ifdef K3D_HAVE_GCC_DEMANGLE

	int status = 0;
	char* const temp = ::abi::__cxa_demangle(Type.c_str(), 0, 0, &status);
	if(temp && (status == 0))
		result = temp;
	if(temp)
		::free(temp);

#endif // K3D_HAVE_GCC_DEMANGLE

	return result;
}

} // namespace detail

void register_type(const std::type_info& Info, const std::string& Name)
{
	detail::initialize_types();
	detail::register_type(Info, Name);
}

const types_t registered_types()
{
	detail::initialize_types();

	types_t results;
	for(detail::type_to_name_map_t::iterator type = detail::type_to_name_map.begin(); type != detail::type_to_name_map.end(); ++type)
		results.push_back(&type->first.info);
	return results;
}

const std::string type_string(const std::type_info& Info)
{
	detail::initialize_types();

	detail::type_to_name_map_t::iterator type = detail::type_to_name_map.find(detail::type_info(Info));
	if(type != detail::type_to_name_map.end())
		return type->second;

	log() << error << k3d_file_reference << ": unknown type: " << Info.name() << std::endl;
	return "";
}

const std::type_info* type_id(const std::string& Name)
{
	detail::initialize_types();

	detail::name_to_type_map_t::iterator type = detail::name_to_type_map.find(Name);
	if(type != detail::name_to_type_map.end())
		return &type->second.info;

	log() << error << k3d_file_reference << ": unknown type: " << Name << std::endl;
	return 0;
}

const std::type_info& type_id_k3d_bitmap_ptr()
{
    return typeid(k3d::bitmap*);
}

const std::string demangle(const std::type_info& Type)
{
	return detail::demangle(Type.name());
}

} // namespace k3d

