"""Testing for the OpenStack provider interface"""
from twisted.internet.defer import inlineCallbacks

from juju import errors
from juju.lib.testing import TestCase
from juju.environment.errors import EnvironmentsConfigError
from juju.machine.constraints import ConstraintSet

from juju.providers.openstack import client as _mod_client
from juju.providers.openstack.tests import OpenStackTestMixin


class ProviderTestMixin(object):

    from juju.providers.openstack.files import FileStorage as FileStorageClass
    from juju.providers.openstack.provider import (
        MachineProvider as ProviderClass)

    environment_name = "testing"

    test_environ = {
        "NOVA_URL": "https://environ.invalid",
        "NOVA_API_KEY": "env-key",
        "EC2_SECRET_KEY": "env-xxxx",
        "NOVA_PROJECT_ID": "env-project",
        }

    def get_config(self):
        return {
            "type": "openstack",
            "auth-mode": "keypair",
            "access-key": "key",
            "secret-key": "xxxxxxxx",
            "auth-url": "https://testing.invalid",
            "project-name": "project",
            "control-bucket": self.environment_name,
            }

    def get_provider(self, config=None):
        if config is None:
            config = self.get_config()
        return self.ProviderClass(self.environment_name, config)

    def get_client(self, provider):
        client = provider.nova._client
        self.assertIs(client, provider.swift._client)
        return client


class ProviderTests(ProviderTestMixin, TestCase):

    def test_empty_config_raises(self):
        """Passing no config raises an exception about lacking credentials"""
        self.change_environment()
        # XXX: Should this raise EnvironmentsConfigError instead?
        self.assertRaises(ValueError, self.get_provider, {})

    def test_client_params(self):
        """Config details get passed through to OpenStack client correctly"""
        provider = self.get_provider()
        creds = provider.credentials
        self.assertEquals("key", creds.access_key)
        self.assertEquals("xxxxxxxx", creds.secret_key)
        self.assertEquals("https://testing.invalid", creds.url)
        self.assertEquals("project", creds.project_name)
        self.assertIs(creds, self.get_client(provider).credentials)

    def test_provider_attributes(self):
        """
        The provider environment name and config should be available as
        parameters in the provider.
        """
        provider = self.get_provider()
        self.assertEqual(provider.environment_name, self.environment_name)
        self.assertEqual(provider.config.get("type"), "openstack")
        self.assertEqual(provider.provider_type, "openstack")

    def test_get_file_storage(self):
        """The file storage is accessible via the machine provider."""
        provider = self.get_provider()
        storage = provider.get_file_storage()
        self.assertTrue(isinstance(storage, self.FileStorageClass))

    def test_config_serialization(self):
        """
        The provider configuration can be serialized to yaml.
        """
        self.change_environment()
        config = self.get_config()
        expected = config.copy()
        config["authorized-keys-path"] = self.makeFile("key contents")
        expected["authorized-keys"] = "key contents"
        provider = self.get_provider(config)
        self.assertEqual(expected, provider.get_serialization_data())

    def test_config_environment_extraction(self):
        """
        The provider serialization loads keys as needed from the environment.

        Variables from the configuration take precendence over those from
        the environment, when serializing.
        """
        self.change_environment(**self.test_environ)
        provider = self.get_provider({
            "auth-mode": "keypair",
            "project-name": "other-project",
            "authorized-keys": "key-data",
            })
        serialized = provider.get_serialization_data()
        expected = {
            "auth-mode": "keypair",
            "access-key": "env-key",
            "secret-key": "env-xxxx",
            "auth-url": "https://environ.invalid",
            "project-name": "other-project",
            "authorized-keys": "key-data",
            }
        self.assertEqual(expected, serialized)

    def test_conflicting_authorized_keys_options(self):
        """
        We can't handle two different authorized keys options, so deny
        constructing an environment that way.
        """
        config = self.get_config()
        config["authorized-keys"] = "File content"
        config["authorized-keys-path"] = "File path"
        error = self.assertRaises(EnvironmentsConfigError,
            self.get_provider, config)
        self.assertEquals(
            str(error),
            "Environment config cannot define both authorized-keys and "
            "authorized-keys-path. Pick one!")


class CheckCertsTests(ProviderTestMixin, TestCase):

    def run_test(self, config_changes, txaws_support=True):
        log = self.capture_logging("juju")
        if txaws_support:
            obj = object()
        else:
            obj = None
        self.patch(_mod_client, "WebVerifyingContextFactory", obj)
        config = self.get_config()
        config.update(config_changes)
        provider = self.get_provider(config)
        return provider, log

    def test_default_true(self):
        provider, log = self.run_test({})
        self.assertNotIn("ssl-hostname-verification", provider.config)
        self.assertEquals(True, provider._check_certs)
        self.assertEquals(True, self.get_client(provider).check_certs)
        self.assertEqual("", log.getvalue())

    def test_false(self):
        provider, log = self.run_test({"ssl-hostname-verification": False})
        self.assertEquals(False, provider._check_certs)
        self.assertEquals(False, self.get_client(provider).check_certs)
        self.assertIn("Set 'ssl-hostname-verification'", log.getvalue())

    def test_true(self):
        provider, log = self.run_test({"ssl-hostname-verification": True})
        self.assertEquals(True, provider._check_certs)
        self.assertEquals(True, self.get_client(provider).check_certs)
        self.assertEqual("", log.getvalue())

    def test_http_auth_url(self):
        provider, log = self.run_test({
            "auth-url": "http://testing.invalid",
            "ssl-hostname-verification": True,
            })
        self.assertEquals(True, self.get_client(provider).check_certs)
        self.assertIn("identity service not using secure", log.getvalue())

    def test_no_txaws_support(self):
        self.assertRaises(errors.SSLVerificationUnsupported, self.run_test,
            {"ssl-hostname-verification": True}, txaws_support=False)


class GetConstraintSetTests(OpenStackTestMixin, TestCase):

    @inlineCallbacks
    def test_get_constraints(self):
        self.expect_nova_get("flavors",
            response={'flavors': self.default_flavors})
        self.mocker.replay()
        provider = self.get_provider()
        cs = yield provider.get_constraint_set()
        self.assertIsInstance(cs, ConstraintSet)
        cs2 = yield provider.get_constraint_set()
        self.assertIsInstance(cs2, ConstraintSet)
        self.assertEqual(cs, cs2)
